/**
 * @file ArkamysTuningHandlerRNAIVI2.cpp
 * @author pau4kor
 * @copyright (c) 2019 RBEI
 * @addtogroup fc_audiomanager
 * @{
 */

#include <stdio.h>
#include <netinet/in.h>


#include "../../../../../ai_osal_linux/components/system/system_types.h"
#include "fc_audiomanager_main.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include <etrace_if.h>

#include "fc_audiomanager_trace.h"
#include "fc_audiomanager_trace_macros.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_TUNING_DATA_SERVICE
#include "trcGenProj/Header/ArkamysTuningHandlerRNAIVI2.cpp.trc.h"
#endif

#include "Arkamys/ArkamysADRInterface.h"
//#include "Arkamys/RNAIVI2/ArkamysFeatureHandlerRNAIVI2.h"

#include "TuningDataService.h"
#include "ArkamysTuningHandlerRNAIVI2.h"

#include "vd_adr3Msg_If.h"


ArkamysTuningHandlerRNAIVI2::ArkamysTuningHandlerRNAIVI2():IF_MessageObserver<PO_MessageConfig::enID>("AIVI2-Arkamys tuning")
{
   m_txHandle = 0;
   m_bHandleValid = false;
   m_bCommExists = false;
   m_pU8Data = NULL;
   m_u32Size = 0;

   m_bARMSetReq = false;
   m_bARMGetReq = false;

   m_bVolGetReq = false;
   m_bMuteGetReq = false;

   m_bMuteSetReq = false;

   // Register for Stream Mute
   m_u8MuteRegId = StreamMute::getInstance() -> registerForStreamMute(StreamMute_IF::INTERNAL_BEH, this);

   //register for PO messages
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this , PO_MessageConfig::ID_NotifyMute);
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this , PO_MessageConfig::ID_NotifyVolume);

}

ArkamysTuningHandlerRNAIVI2::~ArkamysTuningHandlerRNAIVI2()
{
  InternalCommunicationAdapter::getInstance()->POMessages->DeRegisterObserver(this);
  m_pU8Data = NULL;
}

ArkamysTuningHandlerRNAIVI2* ArkamysTuningHandlerRNAIVI2::getInstance()
{
   static ArkamysTuningHandlerRNAIVI2 theInstance;
   return &theInstance;
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::MessageNotification()
 *
 * DESCRIPTION:This method receives the notification for Post Office messages.
 *
 * PARAMETER: PO_MessageConfig::enID MsgId
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
void ArkamysTuningHandlerRNAIVI2::MessageNotification(PO_MessageConfig::enID MsgId)
{
 switch (MsgId)
   {
      case PO_MessageConfig::ID_NotifyMute:
      {
         const ID_NotifyMute* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage<ID_NotifyMute>(MsgId);
         if(m_bMuteGetReq)//only if mutestate get is requested
         {
           PublicParamMessageRx(Tuning_Lib_Mute, (tU8)pMsg->enMuteResult);
           m_bMuteGetReq = false;
         }
      }
      break;

      case PO_MessageConfig::ID_NotifyVolume:
      {
         const ID_NotifyVolume* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage <ID_NotifyVolume> (MsgId);
         if(m_bVolGetReq)//only if volume get is requested
         {
           PublicParamMessageRx(Tuning_Lib_Volume, (tU8)pMsg->u8VolumeStep);
           m_bVolGetReq = false;
         }
      }
      break;

      default:
      {
         ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI2::MessageNotification - unhandled message received: %u", MsgId));
      }
      break;
   }
}


/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::StreamMuteNotify()
 *
 * DESCRIPTION:This method receives the notification for mute/demute.
 *
 * PARAMETER: StreamMuteNotify_IF::tenMuteNotification muteAck, tU8 stream
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
void ArkamysTuningHandlerRNAIVI2::StreamMuteNotify(StreamMuteNotify_IF::tenMuteNotification muteAck, tU8 stream, tU16 sink)
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI2() muteAck=%u,stream = %u and sink %d",(tU8) muteAck, stream, sink));
  if(muteAck == MUTE_ACK_MUTE || muteAck == MUTE_ACK_DEMUTE)
  {
    if(m_bMuteSetReq)//only if set is requested
    {
      vSendAckToClient(TUNING_WriteCmd);//send ack to mute write command
      m_bMuteSetReq = false;
    }
  }

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::Mute()
 *
 * DESCRIPTION:This method sends the mute request to ADR3.
 *
 * PARAMETER: tU8 muteReqId
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::Mute(tU8 muteReqId)
{
   StreamMute::getInstance() -> muteStream(muteReqId, StreamMute_IF::MUTE_ACTION_MUTE, tenFadingReason(EN_AUDIO_FADING_RES_ARKAMYS), tenStream(EN_AUDIO_SOURCE_STREAM_MAIN));
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::Demute()
 *
 * DESCRIPTION:This method sends the demute request to ADR3.
 *
 * PARAMETER: tU8 muteReqId
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::Demute(tU8 muteReqId)
{
   StreamMute::getInstance() -> muteStream(muteReqId, StreamMute_IF::MUTE_ACTION_DEMUTE, tenFadingReason(EN_AUDIO_FADING_RES_ARKAMYS), tenStream(EN_AUDIO_SOURCE_STREAM_MAIN));
}


/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::AdrMessageRx()
 *
 * DESCRIPTION:Receives Data from the TuningDataService:
 *             Command (4 Byte)
 *             Address (4 Bytes)
 *             Length  (4 Bytes)
 *             Value   (<Length> Bytes)
 *             optional: additional [Address, Length, Value] sequences
 * @param size Length of the data that the parameter <data> points to
 * @param data Pointer to the received tuning data. The pointer gets invalid
 *             upon leaving this function. Don't store it!
 * @param txHandle A handle that is needed to send data back to the sender
 *
 * PARAMETER: tU32 size, tVoid* data, tU8 txHandle
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::DataRx(tU32 size, tVoid* pData, tU8 txHandle)
{
   ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI2: Received %i bytes of data. TxHandle: %i", size, txHandle));
   m_txHandle = txHandle;
   m_bHandleValid = true;

   if (size < 4)
   {
      ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI2: Received invalid command."));
      vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
      return;
   }
   m_u32Size = size;
   m_pU8Data = (tU8*)pData;

   if(NULL == m_pU8Data)
       return;

// handle different commands from tuning tool
  switch(m_pU8Data[CMD_ID_INDEX])
  {
  case TUNING_OpenCmd:
  {
    m_bCommExists = true;
    vSendAckToClient(m_pU8Data[CMD_ID_INDEX]);
    break;
  }
  case TUNING_CloseCmd:
  {
    m_bCommExists = false;
    vSendAckToClient(m_pU8Data[CMD_ID_INDEX]);
    break;
  }
  case TUNING_WriteCmd:
  case TUNING_ReadCmd:
  {
    vSendData();
      break;
  }
  default:
    ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI2::DataRx() received invalid command"));
    break;
  }

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::SbrMessageRx()
 *
 * DESCRIPTION:This method receives messages from the sbr function VD_ADR3_FKT_ID_ARKAMYS_CONFIG_DATA (0x0FEB)
 *              and relays them via TCP/IP to the connected Tuning Tool.
 *
 * PARAMETER: tU8 cmd, tU32 len, tS32* data
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::SbrMessageRx(tU8 u8OpType, tU8* u8data, tU16 u16length)
{
   if (((!m_bHandleValid) || (!m_bCommExists)) && ((!m_bARMGetReq) || (!m_bARMSetReq)))
   {
      return;
   }

   ETG_TRACE_USR4(("Got Arkamys Data: Cmd:%x Len:%x", u8OpType, u16length));

   if ((u8data == NULL) || (!u16length))
   {
      ETG_TRACE_ERR(("Invalid data or length"));
      vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
      return;
   }

   tU8 u8Command = ArkamysStaticMessage::ADRToArkamysCommand(u8OpType);

   if ((u8Command != ArkamysStaticMessage::ArkamysOpTypeStatus))
   {
      ETG_TRACE_ERR(("Unknown command %i", u8Command));
      if(u8Command == ArkamysStaticMessage::ArkamysOpTypeError)
      {
        vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
        m_bARMGetReq = false;
        m_bARMSetReq = false;
      }
      return;
   }

   tU8 u8FileID = u8data[0];//1st byte file ID

   //2bytes error code
   tS16 s16ErrorCode;
   memcpy(&s16ErrorCode, &u8data[1], sizeof(tU16));

   ETG_TRACE_USR1(("File ID (%i), Error code (%i)", u8FileID, s16ErrorCode));

   if(s16ErrorCode)//send error code irrespective of read or write commands
   {
       vSendErrToClient(s16ErrorCode);
       m_bARMGetReq = false;
       m_bARMSetReq = false;
       return;
   }

   if((u8FileID == ARKAMYS_SAT_SET_ID) && (m_bARMSetReq))
   {
       m_bARMSetReq = false;
       vSendAckToClient(TUNING_WriteCmd);
   }
   else if((u8FileID == ARKAMYS_SAT_GET_ID) && (m_bARMGetReq))
   {
       m_bARMGetReq = false;
       //4bytes length  starting from 4th byte
       tU32 u32Length;
       memcpy(&u32Length, &u8data[3], sizeof(tU32));
       vSendResponseToClient(u32Length);
   }

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::PublicParamMessageRx()
 *
 * DESCRIPTION:This method receives messages for the mute, volume status
 *              and relays them via TCP/IP to the connected Tuning Tool.
 *
 * PARAMETER: tU32 len, tU8* pU8Data
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::PublicParamMessageRx(tU8 u8PublicParam, tU8 u8Data)
{
   if (!m_bHandleValid || !m_bCommExists)
   {
      return;
   }

   ETG_TRACE_USR4(("status message for %i public parameter", ETG_CENUM( tenTuningProcLibID, u8PublicParam)));

   TuningDataService* tuningService = TuningDataService::getInstance();

   tU32 dataSize =   3 +      //Ack_Cmd, Error status
                 4 +      // data length
           1;       // Data Size
   tU8 *pReplyBuf = new tU8[dataSize];
   if (pReplyBuf == NULL) // lint
   {
      return;
   }

    pReplyBuf[0] = TUNING_AckCmd; //ack_cmd
    pReplyBuf[1] = pReplyBuf[2] = 0x00; // error status
    tU32 tempSize = dataSize - 7; // data length
    memcpy(&pReplyBuf[3], &tempSize, sizeof(tU32));
    pReplyBuf[7] = u8Data;

    tuningService -> vDataTx(TuningDataService::ID_ARKAMYS_RNAIVI, m_txHandle, pReplyBuf, dataSize);

    delete[] pReplyBuf;
}


/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::vAdrOkCallback()
 *
 * DESCRIPTION:This method is called when ADR connection is established
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::vAdrOkCallback()
{
   ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI2::adrOkCallback()"));

}


/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::vSendData()
 *
 * DESCRIPTION:This method receives the data to be sent to ARM or public parameters and relays it respectively.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::vSendData()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI2: vSendData Received %i bytes of data. ", m_u32Size));
  if(m_pU8Data == NULL)
      return;

  if(m_bCommExists)
  {
  //handle different processors (ARM public params)
    switch(m_pU8Data[PROC_ID_INDEX])
    {
    case Tuning_Proc_Public:
      vSendPublicParamData();
      break;
    case Tuning_Proc_ARM:
      vSendARMData();
      break;
    default:
      ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI2::vSendData() received invalid proc id"));
      break;
    }
  }
  else
    ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI2: vSendData communication with the system is : %i (closed)", m_bCommExists));

}


/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::vSendARMData()
 *
 * DESCRIPTION: This method receives the data to be sent to Arkamys library and relays it respectively.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::vSendARMData()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI2: vSendARMData Received %i bytes of data. ", m_u32Size));

  if(m_pU8Data == NULL)
      return;


  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI2: vSendARMData Received %i number of commands. ", m_pU8Data[3]));

  if(m_pU8Data[CMD_ID_INDEX] == TUNING_ReadCmd)
  {
      m_pU8Data[3] = 1;// for read commands explicitly set the number of commands to 1
      vGetARMData();
  }

  if(m_pU8Data[CMD_ID_INDEX] == TUNING_WriteCmd)
  {
      vSetARMData();
  }

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::vSetARMData()
 *
 * DESCRIPTION: This method receives the data to be set to ARMA and relays it respectively.
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::vSetARMData()
{
   if(m_pU8Data == NULL)
          return;
    const char* SAT_SET_FILE = "/tmp/audiomanager/Arkamys_SAT_Set.txt";

    FILE* ofile = NULL;

    ofile = fopen(SAT_SET_FILE, "w");

    if(NULL == ofile)
    {
      ETG_TRACE_ERR(("fail to open the sat set file:  %s", SAT_SET_FILE));
      vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
      return;
    }
    else
    {
        tS32 s32Size = (tS32)fwrite(m_pU8Data, sizeof(tU8), m_u32Size, ofile);
        ETG_TRACE_USR2(("size of arkamys set file %d", s32Size));
        fclose(ofile);

        tU8 u8Data[CONFIG_DATA_FUNC_LEN];
        u8Data[0] = ARKAMYS_SAT_SET_ID;//write cmd
        m_u32Size = htonl(m_u32Size);
        memcpy(&u8Data[1], &m_u32Size, sizeof(tU32));
        vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_ARKAMYS_CONFIG_DATA, VD_ADR3_OPTYPE_SET
                 , CONFIG_DATA_FUNC_LEN, u8Data, ARKAMYS_CONFIG_DATA);//send data to ADR3
        m_bARMSetReq = true;
    }

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::vGetARMData()
 *
 * DESCRIPTION: This method receives the data to be get to ARM and relays it respectively.
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::vGetARMData()
{
    if(m_pU8Data == NULL)
          return;
    const char* SAT_GET_FILE = "/tmp/audiomanager/Arkamys_SAT_Get.txt";

    FILE* ifile = NULL;

    ifile = fopen(SAT_GET_FILE, "w");

    if(NULL == ifile)
    {
      ETG_TRACE_ERR(("fail to open the sat get file:  %s", SAT_GET_FILE));
      vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
      return;
    }
    else
    {
        tS32 s32Size = (tS32)fwrite(m_pU8Data, sizeof(tU8), m_u32Size, ifile);
        ETG_TRACE_USR2(("size of arkamys get file %d", s32Size));
        fclose(ifile);

        tU8 u8Data[CONFIG_DATA_FUNC_LEN];
        u8Data[0] = ARKAMYS_SAT_GET_ID;//read cmd
        m_u32Size = htonl(m_u32Size);
        memcpy(&u8Data[1], &m_u32Size, sizeof(tU32));
        vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_ARKAMYS_CONFIG_DATA, VD_ADR3_OPTYPE_GET
                 , CONFIG_DATA_FUNC_LEN, u8Data, ARKAMYS_CONFIG_DATA);//send data to ADR3
        m_bARMGetReq = true;
    }
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::vSendPublicParamData()
 *
 * DESCRIPTION: This method receives the data to be sent to public parameters and relays it respectively.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::vSendPublicParamData()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI2: vSendPublicParamData Received %i bytes of data. ", m_u32Size));
  if(m_pU8Data == NULL)
      return;

  //handle different parameters (volume, mute)
  switch(m_pU8Data[LIB_ID_INDEX])
  {
  case Tuning_Lib_Volume:
    vSendVolume();
    break;
  case Tuning_Lib_Mute:
    vSendMute();
    break;
  default:
    ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI2::vSendPublicParamData() received invalid Lib id"));
    break;
  }

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::vSendVolume()
 *
 * DESCRIPTION: This method sends the volume message received from tuning tool to sbr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::vSendVolume()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI2: vSendVolume Received %i bytes of data. ", m_u32Size));

  if(m_pU8Data == NULL)
      return;
  if(m_pU8Data[CMD_ID_INDEX] == TUNING_WriteCmd)
    vSetVolume();
  else
    vGetVolume();

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::vSetVolume()
 *
 * DESCRIPTION: This method sends the set volume message received from tuning tool to sbr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::vSetVolume()
{
  if(m_pU8Data == NULL)
          return;
  tU32 u32MsgLen;

  tU8 dataIdx = 4;
  memcpy(&u32MsgLen, &m_pU8Data[dataIdx], sizeof(tU32));

  dataIdx = (tU8) (dataIdx + TUNING_DATA_OFFSET_IVI2);

  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI2: vSetVolume Received %i bytes of volume data. ", u32MsgLen));


  if(u32MsgLen < 2 || u32MsgLen % 2 != 0)
  {
        ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI2:vSetVolume Received invalid Volume Length."));
        vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
        return;
  }

  tU32 u32LenCount = 0;
  while( u32LenCount < u32MsgLen)
  {
        if(TUNING_MSG_FOOTER_VALUE == m_pU8Data[dataIdx + u32LenCount])
          break;// end of msg reached

    // setup outgoing PO message
    VolumeData out;

    out.setVolumeType((midw_fi_tcl_e8_AudioVolumeType::tenType)m_pU8Data[dataIdx + u32LenCount]);
    u32LenCount = u32LenCount + 1;
    out.setVolume(m_pU8Data[dataIdx + u32LenCount]);
    u32LenCount = u32LenCount + 1;

    out.setResource(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS);
    out.m_Channel = midw_fi_tcl_e8_AudioChannel::FI_EN_AUDIO_CHANNEL_NOT_DEF;  // don't have this information here
    //tS16 m_VolumeGain;


    ID_CCA_Start_Volume outMsg(out);
    InternalCommunicationAdapter::POMessages->DeliverMsg(&outMsg);

      ETG_TRACE_USR2(("ArkamysTuningHandlerRNAIVI2 - vSetVolume: - forwarded Volume.Set for Type = %d, Resource = %d, Volume = %d"
           , out.getVolumeType(), out.getResource(), out.getVolume() ));
  }

  //send acknowledgement for the commands received
  vSendAckToClient(m_pU8Data[CMD_ID_INDEX]);
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::vGetVolume()
 *
 * DESCRIPTION: This method sends the get volume message received from tuning tool to sbr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::vGetVolume()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI2: vGetVolume entered "));

  if(m_pU8Data == NULL)
      return;

  m_bVolGetReq = true;
  tU8 Stream = (tU8)EN_AUDIO_SOURCE_STREAM_MAIN;
  vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_VOLUME, VD_ADR3_OPTYPE_GET
         , 1, &Stream, VOLUME_EXC_GET);//send data to ADR3

  //send acknowledgement for the commands received
  //vSendAckToClient(m_pU8Data[0]);
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::vSendMute()
 *
 * DESCRIPTION: This method sends the mute message received from tuning tool to sbr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::vSendMute()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI2: vSendMute Received %i bytes of data. ", m_u32Size));

  if(m_pU8Data == NULL)
      return;
  if(m_pU8Data[CMD_ID_INDEX] == TUNING_WriteCmd)
    vSetMuteDemute();
  else
    vGetMuteDemute();

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::vSetMuteDemute()
 *
 * DESCRIPTION: This method sends the set mute/demute message received from tuning tool to sbr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::vSetMuteDemute()
{

  if(m_pU8Data == NULL)
          return;

  tU32 u32MsgLen;
  tU8 dataIdx = 4;

  memcpy(&u32MsgLen, &m_pU8Data[dataIdx], sizeof(tU32));

  dataIdx = (tU8) (dataIdx + TUNING_DATA_OFFSET_IVI2);

  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI2: vSetMuteDemute Received %i bytes of mute data. ", u32MsgLen));

  if(u32MsgLen != 1)
  {
        ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI2:vSetMuteDemute Received invalid mute Length."));
        vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
        return;
  }

  tU32 u32LenCount = 0;
  while( u32LenCount < u32MsgLen)
  {
        if(TUNING_MSG_FOOTER_VALUE == m_pU8Data[dataIdx + u32LenCount])
          break;// end of msg reached
        tU8 muteState = m_pU8Data[dataIdx + u32LenCount];
        u32LenCount++;

        ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI2: vSetMuteDemute Received mute state %i  . ", muteState));
        m_bMuteSetReq = true;
        if(muteState) //state 0 : mute, state 1: demute
        {
          Demute(m_u8MuteRegId);
        }
        else
        {
          Mute(m_u8MuteRegId);
        }

}

  //send acknowledgement for the commands received
  //vSendAckToClient(m_pU8Data[0]);
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::vGetMuteDemute()
 *
 * DESCRIPTION: This method sends the get mute/demute message received from tuning tool to sbr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::vGetMuteDemute()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI2: vGetMuteDemute entered "));

  if(m_pU8Data == NULL)
      return;

  m_bMuteGetReq = true;
  tU8 Stream = (tU8)EN_AUDIO_SOURCE_STREAM_MAIN;
  vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_MUTE, VD_ADR3_OPTYPE_GET
         , 1, &Stream, MUTE_EXC);//send data to ADR3

  //send acknowledgement for the commands received
  //vSendAckToClient(m_pU8Data[0]);
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::vSendAckToClient()
 *
 * DESCRIPTION: This method sends the acknowledgement to the received commands via TCP/IP to the connected Tuning Tool.
 *
 * PARAMETER: tU8 u8CommandID.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::vSendAckToClient(tU8 u8CommandID)
{
   ETG_TRACE_USR4(("Send Acknowledge to '%i' command", ETG_CENUM(tenTuningCommandID, u8CommandID)));
   tPU8 pu8Message = new tU8 [TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE_IVI];
   if (pu8Message == NULL) // Lint wants a check
   {
       ETG_TRACE_FATAL(("Could not allocate memory."));
       return;
   }
   pu8Message[0] = TUNING_AckCmd;
   memset(&pu8Message[1], 0, TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE_IVI - 1); // error status and data length

   TuningDataService::getInstance() -> vDataTx(TuningDataService::ID_ARKAMYS_RNAIVI, m_txHandle, pu8Message, TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE_IVI);
   delete []pu8Message;
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::vSendErrToClient()
 *
 * DESCRIPTION: This method sends the error codes to the received commands via TCP/IP to the connected Tuning Tool.
 *
 * PARAMETER: tU16 u16ErrorCode.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::vSendErrToClient(tS16 s16ErrorCode)
{
   ETG_TRACE_USR4(("Send Error status '%i' to tuning tool", s16ErrorCode));
   tPU8 pu8Message = new tU8 [TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE_IVI];
   if (pu8Message == NULL) // Lint wants a check
   {
       ETG_TRACE_FATAL(("Could not allocate memory."));
       return;
   }
   pu8Message[0] = TUNING_AckCmd;
   memcpy(&pu8Message[1], &s16ErrorCode, sizeof(tS16));
   memset(&pu8Message[3], 0, TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE_IVI - 3); // data length

   TuningDataService::getInstance() -> vDataTx(TuningDataService::ID_ARKAMYS_RNAIVI, m_txHandle, pu8Message, TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE_IVI);
   delete []pu8Message;
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI2::vSendResponseToClient()
 *
 * DESCRIPTION: This method sends the actual data for the Get commands via TCP/IP to the connected Tuning Tool.
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI2::vSendResponseToClient(tU32 u32FileLength)
{
    ETG_TRACE_USR1(("vSendResponseToClient entered with file size (0x%x)", u32FileLength));

    TuningDataService* tuningService = TuningDataService::getInstance();

    const char* SAT_READ_FILE = "/tmp/audiomanager/Arkamys_SAT_Get.txt";
    FILE* satReadfile = NULL;
    satReadfile = fopen(SAT_READ_FILE, "r");

    if (NULL == satReadfile)
    {
        ETG_TRACE_ERR(("Invalid data or length or failure in file open"));
        vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
        return;
    }

    tS32 s32FileSize;

    fseek(satReadfile, 0L, SEEK_END);//end of file
    s32FileSize = (tS32)ftell(satReadfile);
    fseek(satReadfile, 0L, SEEK_SET); // go back to the beginning of file

    if((s32FileSize < 0) || (u32FileLength != (tU32)s32FileSize))
    {
        ETG_TRACE_ERR(("Invalid data length"));
        vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
        fclose(satReadfile);
        return;
    }

    tU32 dataSize =   3 +      //Ack_Cmd, Error status
                      4 +      //data length variable size
                      (tU32)s32FileSize;  //actual data length to be read from file

    tU8 *pReplyBuf = new tU8[dataSize];
    if (pReplyBuf == NULL) // lint
    {
        fclose(satReadfile);
        return;
    }

    pReplyBuf[0] = TUNING_AckCmd; //ack_cmd
    pReplyBuf[1] = pReplyBuf[2] = 0x00; // error status
    tU32 tempSize = dataSize - 7; // data length
    ETG_TRACE_USR4(("tempSize : %i (%i)", tempSize, dataSize));
    memcpy(&pReplyBuf[3], &tempSize, sizeof(tU32));

    tS32 s32Size = (tS32)fread((void*)&pReplyBuf[7], 1, s32FileSize, satReadfile);
    ETG_TRACE_USR2(("Size of data read from file (%d)", s32Size));

    fclose(satReadfile);

    tuningService -> vDataTx(TuningDataService::ID_ARKAMYS_RNAIVI, m_txHandle, pReplyBuf, dataSize);

    delete[] pReplyBuf;

}
