/*******************************************************************************
* FILE:          aud_tcl_loopback_mgr.cpp
* PROJECT:
* SW-COMPONENT:  fc_audiomanager loop back manager
*-------------------------------------------------------------------------------
*
* DESCRIPTION:   Microphone diagnosis.
*
*-------------------------------------------------------------------------------
*
* COPYRIGHT:    (c) 2016 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author                  | Modification
* 13.04.16  | Ranjit Katuri           | Inital version
*******************************************************************************/

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#include "fc_audiomanager_trace.h"
#include "fc_audiomanager_trace_input.h"
#include "fc_audiomanager_trace_macros.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_TRACE
#include "trcGenProj/Header/aud_tcl_loopbackmgr.cpp.trc.h"
#endif


#include "aud_tcl_loopback_IF.h"
#include "aud_tcl_loopback_mgr.h"
#include "fc_audiomanager_main.h"


using namespace std;

/**
* Constructor
*/
aud_tcl_loopback_mgr::aud_tcl_loopback_mgr(fc_audiomanager_tclApp* pMain)
:m_poMain(pMain)
{
  //Set the mutex attributes to recursive
  pthread_mutexattr_t Attr;
  pthread_mutexattr_init(&Attr);
  pthread_mutexattr_settype(&Attr, PTHREAD_MUTEX_RECURSIVE);
  pthread_mutex_init(&m_lb_queue_lock, &Attr);//Setting the mutex to recursive to get rid of silly deadlock issues
}
/**
* Destructor
*/
aud_tcl_loopback_mgr::~aud_tcl_loopback_mgr()
{
#if 0
  /**
  * This is a memory leak, but chose to retain it as we donot know if the pointer can be consumed or not at this time.
  * If we delete a pointer which is not heap allocated (eg: address of a member variable passed), this would result in a crash
  * But as the loopback manager would only be deleted when the process is about to be terminated, memory associated with the
  * process will also be freed, there by nullifying the impact of this memory leak.
  */

  pthread_mutex_lock(&m_lb_queue_lock);
  m_lb_queue.clear();
  pthread_mutex_unlock(&m_lb_queue_lock);
#endif
  m_poMain = NULL;
}

/**
* Helper function to be triggered by various threads to send a particular message for loopback
*/
bool aud_tcl_loopback_mgr::bSendDataforLoopBack(aud_tcl_loopback_IF* pData)
{

  bool retval = false;
  if(pData && m_poMain)
  {
    pthread_mutex_lock(&m_lb_queue_lock);
    m_lb_queue.push(pData);
    pthread_mutex_unlock(&m_lb_queue_lock);
    //Post an event
    retval = m_poMain->bPostEvent(FC_AUDIOMANAGER_AHL_EVENT_LOOPBACK_MSG_AVAILABLE);
  }
  return retval;
}


/**
* This function is triggered when the event FC_AUDIOMANAGER_AHL_EVENT_LOOPBACK_MSG_AVAILABLE is fired.
* Loop back manager would process the loopback messages upto a max of LOOP_BACK_DISPATCH_BATCH_SIZE,as configured during build
* Dispatch function is executed on every loop back message item.
*/
void aud_tcl_loopback_mgr::vDispatch()
{
  if(m_poMain)
  {
    //Create a temporary queue, copy the items to a temporary queue and work on it
    queue<aud_tcl_loopback_IF*> tmpqueue;
    uint32_t count = 0;

    /**
    * Move work items to temporary queue
    */
    /************************************************************
    * ENTER CRITICAL SECTION
    ************************************************************/
    pthread_mutex_lock(&m_lb_queue_lock);
    //Copy the pointers to a temporary list
    while(!m_lb_queue.empty())
    {
      count++;
      tmpqueue.push(m_lb_queue.front());
      m_lb_queue.pop();//Remove item
      if(count >= LOOP_BACK_DISPATCH_BATCH_SIZE)
        break;
    }
    //Check if there are still some work items present in the queue
    if(!m_lb_queue.empty())
    {
      //There are some work items pending, post an event so that we can process pending items in the next run
      m_poMain->bPostEvent(FC_AUDIOMANAGER_AHL_EVENT_LOOPBACK_MSG_AVAILABLE);
    }
    //UNLOCK the resource m_lb_queue
    pthread_mutex_unlock(&m_lb_queue_lock);
    /************************************************************
    * EXIT CRITICAL SECTION
    ************************************************************/

    /**
    * Process and Dispatch work items
    */
    while(!tmpqueue.empty())
    {
      tmpqueue.front()->vExecute();//Trigger dispatch function
      delete tmpqueue.front();//Free allocated resources
      tmpqueue.pop();//remove the item from queue
    }
  }
}

