/**
 * @file EngineSoundHandlerIVI.cpp
 * @brief Implement the interface to ADR to send the ESE related commands
 * @author prm4kor
 * @copyright (c) 2016 RBEI
 * @addtogroup fc_audiomanager
 * @{
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include <etrace_if.h>
#include "fc_audiomanager_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_ENGINESOUND
#include "trcGenProj/Header/EseGeneratorAdr.cpp.trc.h"
#endif

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_diagdebug_if.h" // For EngineSound EOL Datapool
#include "dp_generic_if.h"
#include "dp_audio_if.h"

#include "EseGeneratorAdr.h"
#include "EseFileContainerIVI.h"
#include "vd_adr3Msg_If.h"
#include "aud_sinkmanager/vd_adr3/vd_adr3Message_Data.h"

#include "InternalComponentCommunication/Messages/Sound/ID_EngineSound_Version.h"
#include "InternalComponentCommunication/Messages/Startup/IDNotifyStartup.h"
#include "InternalComponentCommunication/DataTypes/TypeDefines/ADR3Startup.h"
#include "InternalComponentCommunication/Messages/Sound/ID_EngineSound_LibVersion.h"

static tU8 m_HardCodedData1[ESE_PARAM_SIZE_RSOUND] = {Hardcoded_1};
static tU8 m_HardCodedData2[ESE_PARAM_SIZE_RSOUND] = {Hardcoded_2};

/**
 * EseGeneratorAdr()
 */
EseGeneratorAdr::EseGeneratorAdr():
                     IF_MessageObserver<PO_MessageConfig::enID>("Engine Sound ADR Observer"),
           m_pMsgQueue(NULL),
           m_u8ReqMexSoundType(ENGINESOUND_SOUNDTYPE_NEUTRAL),
                     m_u8ActiveSoundType(ENGINESOUND_SOUNDTYPE_NEUTRAL)
{
   ETG_TRACE_USR4(("EseGeneratorAdr interface created"));

   m_pMsgQueue = new EngineSoundMsgQueue();
   m_u8DatapoolReadStatus = EN_NORESULT;
   m_bESEVersionCheckStatus = false;
   m_s16VolumeOffset = 0;
   vAddObserver();

   //if any new version of ESE lib is introduced in ADR, EOl then the correspoding entry should be made in the version map config. file
   #define ESEADRVERSION_TO_ESEEOLVERSION_MAP
   #define ESEVersionMap(a,b,c,d) ESEVersionMapValuesVector.push_back(tESEVersionMap(a,b,c,d));
   #include "ESEVersionMap.conf"
   #undef ESEVersionMap
   #undef ESEADRVERSION_TO_ESEEOLVERSION_MAP
}

/**
 * ~EseGeneratorAdr()
 */
EseGeneratorAdr::~EseGeneratorAdr()
{
   if (m_pMsgQueue != NULL)
  {
    delete m_pMsgQueue;
    m_pMsgQueue = NULL;
  }
   InternalCommunicationAdapter::getInstance()->POMessages->DeRegisterObserver(this);
}

/**
 * vAddObserver()
 * Observer for the post office messages
 */
void EseGeneratorAdr::vAddObserver()
{
   ETG_TRACE_USR4(("EngineSoundHandlerIVI::vAddObserver"));
   InternalCommunicationAdapter::getInstance() -> POMessages -> AddObserver(this, PO_MessageConfig::ID_EngineSound_Version);
   InternalCommunicationAdapter::getInstance() -> POMessages -> AddObserver(this, PO_MessageConfig::ID_NotifyStartup);
   InternalCommunicationAdapter::getInstance() -> POMessages -> AddObserver(this, PO_MessageConfig::ID_EngineSound_LibVersion);
}

void EseGeneratorAdr::MessageNotification(PO_MessageConfig::enID MsgId)
{
   ETG_TRACE_USR4(("EseGeneratorAdr::MessageNotification() entered. "));
   // Obtaining the Message from PostOffice handler
   switch(MsgId)
   {
      case(PO_MessageConfig::ID_NotifyStartup):
      {
         ETG_TRACE_USR4(("EseGeneratorAdr: NotifyStartup received."));

         const ID_NotifyStartup* pMsg = InternalCommunicationAdapter::getInstance() -> POMessages -> QueryMessage<ID_NotifyStartup>(MsgId);

         if ( pMsg -> enOn == EN_AUDIO_ADR3_AVAILABLE)
         {
            vStartupNotification();
         }
         break;
      }
      case(PO_MessageConfig::ID_EngineSound_Version):
      {
         const ID_EngineSound_Version* pMsg = InternalCommunicationAdapter::getInstance() -> POMessages -> QueryMessage<ID_EngineSound_Version>(MsgId);
         ETG_TRACE_USR4(("EseGeneratorAdr:: Received ESE-Version Message: FileId:%i Version:%x Checksum:%x", pMsg->eseFileId, pMsg->eseVersion, pMsg->eseChecksum));
         vFileNotification((tU8)pMsg -> eseFileId, pMsg -> eseVersion, pMsg -> eseChecksum);
         break;
      }
      case(PO_MessageConfig::ID_EngineSound_LibVersion):
      {
         const ID_EngineSound_LibVersion* pMsg = InternalCommunicationAdapter::getInstance() -> POMessages -> QueryMessage<ID_EngineSound_LibVersion>(MsgId);
         ETG_TRACE_USR4(("EseGeneratorAdr:: Received ESE-LibVersion Message: Lib-Version:%x ", pMsg->eseLibVersion));
         m_bESEVersionCheckStatus = bCompareVersion(pMsg->eseLibVersion);
         break;
      }
      default:
         ETG_TRACE_USR4(("EseGeneratorAdr:: Notification received for unknown message id"));
         break;
   }
}

/**
 * bStartUp(tU8 u8EseSoundType, tU8 u8EseVolume)
 * Function to be called on start-up by EngineSoundHandlerIVI
 */
tBool EseGeneratorAdr::bStartUp(tU8 u8EseSoundType, tU8 u8EseVolume, tU8 u8RSoundOnOff, tU8 u8RSoundType, tU8 u8RSoundVolume)
{
   m_u8ReqMexSoundType = u8EseSoundType;
   /*ToDo:On start-up, version and checksum of data inside ADR to be verified*/
   m_u8ActiveSoundType = (ENGINESOUND_VOLUME_OFF == u8EseVolume) ? ((tU8)(ENGINESOUND_SOUNDTYPE_NEUTRAL)) : u8EseSoundType;
   if(ENGINESOUND_VOLUME_LOW == u8EseVolume) //Set Volume
     m_s16VolumeOffset = -16;//-4 * 4

   if((RSOUND_ONOFF_ON == u8RSoundOnOff))
   {
     m_u8ActiveSoundType = u8RSoundType;
     m_s16VolumeOffset =(tS16)((-28 + (2 * u8RSoundVolume)) * 4);
   }

   dp_tclEngineSoundDPEngineSound_Data oDpEngineSound;

   if(ENGINE_SOUND_DATA_SIZE_BYTES != oDpEngineSound.s32GetData(m_DatapoolData, ENGINE_SOUND_DATA_SIZE_BYTES))
   {
      ETG_TRACE_FATAL(("EseGeneratorAdr : Failed to read EngineSound EOL data"));
      m_u8DatapoolReadStatus = EN_FAILED;
   }
   else
   {
     ETG_TRACE_USR1(("EseGeneratorAdr : EngineSound EOL data read from datapool"));
     m_u8DatapoolReadStatus = EN_PASSED;
   }
   return TRUE;
}

tVoid EseGeneratorAdr::vStartupNotification()
{
   ETG_TRACE_USR4(("EseGeneratorAdr::vStartupNotification() received from ADR"));
   if (m_pMsgQueue != NULL)
   {
      m_pMsgQueue->vEnableQueue();

      //datapool read error check - only if datapool has valid data start ESE sequence
      if(m_u8DatapoolReadStatus == EN_PASSED)
      {
         //request ESE lib version from ADR
         EngineSoundMessage eseMsg(VD_ADR3_OPTYPE_GET,VD_ADR3_FKT_ID_ESE_LIB_VERSION,ENGINE_SOUND_LIB_VERSION,0,NULL);
         m_pMsgQueue->vQueueEngineSoundMessage(eseMsg);
      }
   }
}

tVoid EseGeneratorAdr::vRequestFileInfo()
{
  ETG_TRACE_USR4(("EseGeneratorAdr::vRequestFileInfo() called"));
  // Request checksum and version of all FileIDs from ADR
  for (tU8 iFileId = 1; iFileId <= ENGINE_SOUND_NUMBER_OF_ESE_FILES; iFileId++ )
  {
    tU8 FileID = iFileId;
    EngineSoundMessage eseMsg(VD_ADR3_OPTYPE_GET,VD_ADR3_FKT_ID_ENGINE_SOUND_VERSION,ENGINE_SOUND_VERSION,1,&FileID);
    if (m_pMsgQueue != NULL)
      m_pMsgQueue->vQueueEngineSoundMessage(eseMsg);
  }
}

tVoid EseGeneratorAdr::vFileNotification(tU8 fileId, tU32 fileVersion, tU32 checksum)
{
  ETG_TRACE_USR4(("EseGeneratorAdr::vFileNotification() entered for file ID : %d", fileId));
  //NORMAL_M_ASSERT(1 <= fileId && fileId <= ENGINE_SOUND_NUMBER_OF_ESE_FILES);
  if (fileId < 1 || fileId > ENGINE_SOUND_NUMBER_OF_ESE_FILES)
  {
    ETG_TRACE_ERR(("File-ID Out of range: %i", fileId));
    return;
  }

  EseFileContainerIVI* pEseContainer = NULL;
  tU8* pEseFiledata = NULL;

  pEseContainer = new EseFileContainerIVI();
  pEseFiledata  = new tU8[ESE_FILE_MAX_PAYLOAD];

  if(pEseContainer && pEseFiledata)
  {
    memset(pEseFiledata,0x00,ESE_FILE_MAX_PAYLOAD);

    tU32 u32Size = (m_u8ActiveSoundType < 4)?ESE_PARAM_SIZE_MEX:ESE_PARAM_SIZE_RSOUND;
      ETG_TRACE_USR4(("Active Sound type : %d", m_u8ActiveSoundType));
    tU32 u32EseFileLength = pEseContainer->u32GetEseFileData(getDataPtr(m_u8ActiveSoundType),u32Size,fileId,pEseFiledata);
    EngineSoundMessage eseMsg(ESE_ADR_FILE_VERSION,fileId,pEseFiledata,(tU16)u32EseFileLength);

    // Check if message needs to be sent
    if ((fileVersion != ESE_ADR_FILE_VERSION) ||
        (checksum != eseMsg.GetCrc32()))
    {
         vSendESEOnOff(FALSE); //If there is a download required for any of the file data on start-up, keep ESE OFF in ADR3 until all File IDs are downloaded

      ETG_TRACE_USR4(("EseGeneratorAdr: Parameter Update for File-ID %i needed: Version:%08x vs %08x, Checksum:%08x vs %08x"
          ,fileId , fileVersion, ESE_ADR_FILE_VERSION, checksum, eseMsg.GetCrc32()));
      // Queue Message
      if(m_pMsgQueue)
         m_pMsgQueue->vQueueEngineSoundMessage(eseMsg);
    }
    else
    {
      ETG_TRACE_USR4(("EseGeneratorAdr: Parameter on ADR ok. No update for File-ID %i needed.", fileId));
    }

      if(fileId == ENGINE_SOUND_NUMBER_OF_ESE_FILES) //last Fild ID is also verified. Enable ESE to ON again.
         vSendESEOnOff(TRUE);
  }
  if(pEseContainer)
    delete pEseContainer;

  if(pEseFiledata)
    delete[] pEseFiledata;
}

tBool EseGeneratorAdr::bCompareVersion(tU32 u32ESELibVersion)
{
  ETG_TRACE_USR4(("EseGeneratorAdr::bCompareVersion() ESE Lib version received from ADR 0x%x (%i)", u32ESELibVersion, u32ESELibVersion));

  tBool bVersionMatch = false;//ESE lib version check

  if(!u32ESELibVersion)//nothing to be done if ESE lib version is 0
    return bVersionMatch;

  //extract ESE lib major and minor version number from the received value
    tU8 u8LibMajorVersion = 0; tU8 u8LibMinorVersion = 0; tU16 u16LibPatchVersion = 0;
    u8LibMajorVersion = (tU8)(u32ESELibVersion >> 24);//1st byte - major version
    u8LibMinorVersion = ((u32ESELibVersion >> 16) & 0xFF);//2nd byte - minor version
    u16LibPatchVersion = u32ESELibVersion & 0xFFFF;//3rd-4th byte - patch version
    /*
    if (u32ESELibVersion > 99)
    {
        u8LibMajorVersion  =  u32ESELibVersion / 100;
        u32ESELibVersion -= u8LibMajorVersion * 100;
    }
    if (u32ESELibVersion > 9)
    {
      u8LibMinorVersion =  u32ESELibVersion / 10;
         u32ESELibVersion -= u8LibMinorVersion * 10;
    }
    u8LibPatchVersion  =  u32ESELibVersion;
    */
    ETG_TRACE_USR1(("EseGeneratorAdr::bCompareVersion() ESE ADR Lib major version : (0x%x), minor version: (0x%x), patch version: (0x%x)", u8LibMajorVersion, u8LibMinorVersion, u16LibPatchVersion));

  //extract EOL major and minor version from datapool
    tU8 u8EOLMajorVersion = m_DatapoolData[0];//first byte is version major number
    tU16 u16EOLMinorVersion = 0;
    //EOL minor version is of size - 2 bytes
    u16EOLMinorVersion = m_DatapoolData[1];
    u16EOLMinorVersion = (tU16)(u16EOLMinorVersion << 8);
    u16EOLMinorVersion = (tU16)(u16EOLMinorVersion | m_DatapoolData[2]);

    ETG_TRACE_USR1(("EseGeneratorAdr::bCompareVersion() ESE datapool Lib major version : (0x%x), minor version: (0x%x)", u8EOLMajorVersion, u16EOLMinorVersion));


    //iterate over the configuration values and compare the version numbers
    std::vector<tESEVersionMap>::iterator ESEVersionMapIterator;
  for(ESEVersionMapIterator = ESEVersionMapValuesVector.begin();
      ESEVersionMapIterator != ESEVersionMapValuesVector.end();
      ESEVersionMapIterator++)
  {
    //if lib version is supported proceed to check of EOL major and minor version numbers
    if((u8LibMajorVersion == ESEVersionMapIterator->u8ESELibMajVersion) && ((u8LibMinorVersion == ESEVersionMapIterator->u8ESELibMinVersion)))
    {
      ETG_TRACE_USR1(("EseGeneratorAdr::bCompareVersion() ESE Lib version is supported"));

      //get the EOL major and minor number corresponding to ESE lib major and minor numbers stored in the VersionMap configuration
      tU8 u8MappedEOLMajVersion = ESEVersionMapIterator->u8ESEEOLMajVersion;
      tU16 u16MappedEOLMinVersion = ESEVersionMapIterator->u16ESEEOLMinVersion;

      ETG_TRACE_USR1(("EseGeneratorAdr::bCompareVersion() ESE expected Lib major version : (0x%x), minor version: (0x%x)", u8MappedEOLMajVersion, u16MappedEOLMinVersion));

      //compare with datapool version numbers - both major and minor
      //only if major number is matching, check for minor number
      if(u8EOLMajorVersion == u8MappedEOLMajVersion)
      {
        //if minor version number of EOL data is greater than or equal to config value proceed with loading of EOL data
        if( u16EOLMinorVersion >= u16MappedEOLMinVersion)
        {
            bVersionMatch = true;
          vRequestFileInfo();
        }
        else
          ETG_TRACE_ERR(("EseGeneratorAdr::bCompareVersion() Lib and EOL minor versions are incompatible"));
      }
      else
        ETG_TRACE_ERR(("EseGeneratorAdr::bCompareVersion() Lib and EOL major versions are incompatible"));


      break;//exit iterations once lib version is found in the configuration
    }

  }

  ETG_TRACE_USR4(("EseGeneratorAdr::bCompareVersion() ESE Lib and EOL version check status (%d)", bVersionMatch));

  return bVersionMatch;

}

/**
 * vSendESEOnOff(tBool bOn)
 * Send ESE-OnOff message to ADR3
 */
tVoid EseGeneratorAdr::vSendESEOnOff(tBool bOn)
{
  ETG_TRACE_USR4(("EseGeneratorAdr::vSendESEOnOff() entered. bOn : %d", bOn));
  EngineSoundMessage eseMsg(VD_ADR3_OPTYPE_SET,VD_ADR3_FKT_ID_ENGINE_SOUND_ONOFF,ENGINE_SOUND_SET,1,&bOn);

  if(m_pMsgQueue)
    m_pMsgQueue->vQueueEngineSoundMessage(eseMsg);
}

/**
 * vRequestESEOnOffStatus()
 * Send message to ADR3 to get the ESE On/Off status
 */
tVoid EseGeneratorAdr::vRequestESEOnOffStatus()
{
   ETG_TRACE_USR4(("EseGeneratorAdr::vCheckESEOnOffStatus() entered"));
   EngineSoundMessage eseMsg(VD_ADR3_OPTYPE_GET,VD_ADR3_FKT_ID_ENGINE_SOUND_ONOFF,ENGINE_SOUND_SET,0,NULL);

   if(m_pMsgQueue)
    m_pMsgQueue->vQueueEngineSoundMessage(eseMsg);
}

/**
 * vSendESESoundType(tU8 u8SoundType, tU8 u8Volume)
 * Function to update the MEX-Drive Mode Sound type
 */
tVoid EseGeneratorAdr::vSendESESoundType(tU8 u8SoundType, tU8 u8Volume)
{
   ETG_TRACE_USR4(("EseGeneratorAdr::vSendESESoundType() entered. sound type : %d, volume : %d", u8SoundType,u8Volume));
   m_u8ReqMexSoundType = u8SoundType;

   vUpdateEseSettings(u8SoundType,u8Volume);
}

/**
 * vSendESEVolume(tU8 u8Volume)
 * Function to update the MEX-Drive Mode ESE Volume
 */
tVoid EseGeneratorAdr::vSendESEVolume(tU8 u8Volume)
{
   ETG_TRACE_USR4(("EseGeneratorAdr::vSendESEVolume() entered. volume : %d",u8Volume));

   vUpdateEseSettings(m_u8ReqMexSoundType,u8Volume);
}

/**
 * vUpdateEseSettings(tU8 u8Soundtype, tU8 u8Volume)
 * Function to update the sound type/volume in MEX- Drive Mode ESE
 * Not only a new sound type selection results in downloading soun data to ADR3, but also a volume setting change can trigger this
 * For e.g. Setting volume to OFF means downloading Neutral data.
 */
tVoid EseGeneratorAdr::vUpdateEseSettings(tU8 u8Soundtype, tU8 u8Volume)
{
   tU8 u8NewSoundType = (ENGINESOUND_VOLUME_OFF == u8Volume) ? ((tU8)(ENGINESOUND_SOUNDTYPE_NEUTRAL)) : u8Soundtype;
   vSendESEOnOff(FALSE); //Switch OFF ESE
   if(m_u8ActiveSoundType != u8NewSoundType)
   {
      m_u8ActiveSoundType = u8NewSoundType;

      if(m_bESEVersionCheckStatus)//load ese parameters only for valid version
        vLoadESEParameter(getDataPtr(u8NewSoundType),(tU32)ESE_PARAM_SIZE_MEX); //Load ESE Parameters

      if(ENGINESOUND_VOLUME_LOW == u8Volume) //Set Volume
         vSendVolumeOffset(-4*4);
      else if(ENGINESOUND_VOLUME_HIGH == u8Volume)
         vSendVolumeOffset(0);


   }
   else //if there was a change in sound type, volume messages are already sent along with that
   {
      if(ENGINESOUND_VOLUME_LOW == u8Volume) //Set Volume
         vSendVolumeOffset(-4*4);
      else if(ENGINESOUND_VOLUME_HIGH == u8Volume)
         vSendVolumeOffset(0);
   }
   vSendESEOnOff(TRUE); //Switch ON ESE
}

/**
 * vSendRSoundType(tU8 u8SoundType, tU8 u8Volume)
 * Download the new R-Sound type data, selected from HMI
 */
tVoid EseGeneratorAdr::vSendRSoundType(tU8 u8SoundType, tU8 u8Volume)
{
   ETG_TRACE_USR4(("EseGeneratorAdr::vSendRSoundType() entered. sound type : %d, volume : %d", u8SoundType,u8Volume));
   m_u8ActiveSoundType = u8SoundType;
   vSendESEOnOff(FALSE); //Switch OFF ESE
   if(m_bESEVersionCheckStatus)//load ese parameters only for valid version
     vLoadESEParameter(getDataPtr(u8SoundType),(tU32)ESE_PARAM_SIZE_RSOUND);   //Load ESE Parameters from datapool

   vSendRSoundVolume(u8Volume);
   vSendESEOnOff(TRUE); //Switch ON ESE
}

/**
 * vSendRSoundVolume(tU8 u8Volume)
 * Calculate the volume offset from the parameter (0...14) and send to ADR3
 * R-Sound allows volume level from 0(-28 dB) -> 14 (0dB).
 */
tVoid EseGeneratorAdr::vSendRSoundVolume(tU8 u8Volume)
{
   tS16 s16Offset = (tS16)((-28 + (2 * u8Volume)) * 4);

   if(s16Offset == m_s16VolumeOffset)
     vSendVolumeOffset(s16Offset);

   else if(m_s16VolumeOffset < s16Offset)
   {
      tU8 Iterations = (tU8) (s16Offset - m_s16VolumeOffset);
      Iterations = (tU8)(Iterations/4);//since quarter db should be used
      for(tU8 i = 0; i < Iterations; i ++)
      {
        //send offsets in 1 db steps or 4quarter_db steps
        m_s16VolumeOffset = (tS16) ((m_s16VolumeOffset + 4));
        vSendVolumeOffset(m_s16VolumeOffset);
      }
   }

   else if (m_s16VolumeOffset > s16Offset)
   {
     tU8 Iterations = (tU8)(m_s16VolumeOffset - s16Offset);
     Iterations = (tU8)(Iterations/4);//since quarter db should be used
     for(tU8 i = 0; i < Iterations; i ++)
     {
       //send offsets in 1 db steps or 4quarter_db steps
       m_s16VolumeOffset = (tS16) ((m_s16VolumeOffset - 4));
       vSendVolumeOffset(m_s16VolumeOffset);
     }
   }
}

/**
 * vRequestESEVolumeStatus()
 * Send a message to ADR3 to get the ESE-Volume
 */
tVoid EseGeneratorAdr::vRequestESEVolumeStatus()
{
   EngineSoundMessage eseMsg(VD_ADR3_OPTYPE_GET,VD_ADR3_FKT_ID_ENGINE_SOUND_VOLUME,ENGINE_SOUND_VOLUME,0,NULL);

   if(m_pMsgQueue)
    m_pMsgQueue->vQueueEngineSoundMessage(eseMsg);
}

/**
 * vDataCon()
 * Callback from vd_adr3 module, that the ESE message is transmitted to ADR3
 */
tVoid EseGeneratorAdr::vDataCon()
{
   if(m_pMsgQueue)
    m_pMsgQueue->vAdrReady();
}

/**
 * vLoadESEParameter(tU8* pu8Data, tU32 u32Size)
 * This function split the sound type data to ESE-Files and add to the queue
 */
tVoid EseGeneratorAdr::vLoadESEParameter(tU8* pu8Data, tU32 u32Size)
{
   EseFileContainerIVI* pEseContainer = NULL;
   tU8* pEseFiledata = NULL;
   tU32 u32EseFileLength = 0;

   if((!pu8Data)||(!m_pMsgQueue))
      return;

   pEseContainer = new EseFileContainerIVI();
   pEseFiledata  = new tU8[ESE_FILE_MAX_PAYLOAD];

   if(pEseContainer && pEseFiledata)
   {
      for(tU8 u8fileIdx = 1; u8fileIdx <= ENGINE_SOUND_NUMBER_OF_ESE_FILES; u8fileIdx++)
      {
         memset(pEseFiledata,0x00,ESE_FILE_MAX_PAYLOAD);

         u32EseFileLength = pEseContainer->u32GetEseFileData(pu8Data,u32Size,u8fileIdx,pEseFiledata);

         EngineSoundMessage eseMsg(ESE_ADR_FILE_VERSION,u8fileIdx,pEseFiledata,(tU16)u32EseFileLength);

         m_pMsgQueue->vQueueEngineSoundMessage(eseMsg);
      }
   }

   if(pEseContainer)
      delete pEseContainer;

   if(pEseFiledata)
      delete[] pEseFiledata;
}

/**
 * vSendVolumeOffset(tS16 s16Volume)
 * used to set the Engine sound volume offset
 */
tVoid EseGeneratorAdr::vSendVolumeOffset(tS16 s16Volume)
{
  tU8 u8Data[2];
   u8Data[0] = (tU8)(s16Volume >> 8);
   u8Data[1] = (tU8)(s16Volume & 0xFF);

   EngineSoundMessage eseMsg(VD_ADR3_OPTYPE_SET,VD_ADR3_FKT_ID_ENGINE_SOUND_VOLUME,ENGINE_SOUND_VOLUME,2,u8Data);

   if(m_pMsgQueue)
    m_pMsgQueue->vQueueEngineSoundMessage(eseMsg);
}

/**
 * getDataPtr(tU8 u8SoundType)
 * Get the pointer to the data for the selected sound type
 * MEX (3 numbers) and R-Sound (4 numbers) Types are stored in continuous order in datapool.
 * RSOUND5 and RSOUND6 are hard-coded
 */
tU8* EseGeneratorAdr::getDataPtr(tU8 u8SoundType)
{
   tU16 au16Index[] = {EOL_INDEX_NEUTRAL_DATA,EOL_INDEX_SOUND1_DATA,EOL_INDEX_SOUND2_DATA,EOL_INDEX_RSOUND1_DATA,
                              EOL_INDEX_RSOUND2_DATA,EOL_INDEX_RSOUND3_DATA, EOL_INDEX_RSOUND4_DATA};

   if((u8SoundType > 0) && (u8SoundType <= ENGINESOUND_SOUNDTYPE_RSOUND4))
      return &m_DatapoolData[au16Index[u8SoundType-1]];
   else if(ENGINESOUND_SOUNDTYPE_RSOUND5 == u8SoundType)
      return m_HardCodedData1;
   else if(ENGINESOUND_SOUNDTYPE_RSOUND6 == u8SoundType)
      return m_HardCodedData2;

   return &m_DatapoolData[EOL_INDEX_NEUTRAL_DATA]; //Return Neutral data (also by default)
}
