/**
 * @file EseFileContainer.h
 * @author prm4kor
 * @copyright (c) 2016 RBEI
 * @addtogroup
 * @{
 */

#ifndef ESEFILECONTAINERIVI_H_
#define ESEFILECONTAINERIVI_H_

#include "EngineSoundConfigIVI.h"

/**
 *@brief Size of ESE-Files as required by the ESE-Parameter ADR3<->iMx INC interface
 *
 * EseFileID 1 contains ->  EseDes_aVehAxis[20],EseDes_nEngAxis[142], EseDes_ratAccrPercAxis[20], EseDes_ratAmpAVeh[20], EseDes_ratAmpVVeh[20], EseDes_vVehAxis[20]
 * EseFileID 2 contains ->  EseDes_ratOffsHrmcAccel[50],EseDes_agPhaIninAccel[50],EseDes_ratAmpAccrAccel[20], EseDes_ratAmpNEngAccel[1740]
 * EseFileID 3 contains ->  EseDes_ratAmpNEngAccel[1810],EseDes_ratHrmcAccel[50],
 * EseFileID 4 contains ->  EseDes_ratOffsHrmcDecel[20],EseDes_agPhaIninDecel[20],EseDes_ratAmpAccrDecel[20],EseDes_ratAmpNEngDecel[1420],EseDes_ratHrmcDecel[20]
 * EseFileID 5 contains ->  EseDes_ratAudioGain[8],EseDes_tiAudioDly[8]
 */

#define ESE_FILEID1_SIZE   242
#define ESE_FILEID2_SIZE   1860
#define ESE_FILEID3_SIZE   1860
#define ESE_FILEID4_SIZE   1500
#define ESE_FILEID5_SIZE   16

/**
 *@brief Structures of ESE data layout in EOL
 *ESE EOL data layout is different for MEX and RSound
 */
#pragma pack(push, 1)
typedef struct EseDataMex
{
   tU8 EseDes_nEngAxis[142];              /**< vecRPM */
   tU8 EseDes_ratHrmcDecel[20];           /**< vecRang_i */
   tU8 EseDes_ratAmpNEngDecel[1420];      /**< matAmplitude_i */
   tU8 EseDes_ratHrmcAccel[30];           /**< vecRang_g */
   tU8 EseDes_ratAmpNEngAccel[2130];      /**< matAmplitude_g */
   tU8 EseDes_agPhaIninDecel[20];         /**< vecPhaseInitiale_i */
   tU8 EseDes_agPhaIninAccel[30];         /**< vecPhaseInitiale_g */
   tU8 EseDes_ratOffsHrmcDecel[20];       /**< vecInharmo_i */
   tU8 EseDes_ratOffsHrmcAccel[30];       /**< vecInharmo_g */
   tU8 EseDes_ratAccrPercAxis[20];        /**< vecCharge */
   tU8 EseDes_ratAmpAccrDecel[20];        /**< vecGainCharge_i */
   tU8 EseDes_ratAmpAccrAccel[20];        /**< vecGainCharge_g */
   tU8 EseDes_vVehAxis[20];               /**< vecVitesse */
   tU8 EseDes_ratAmpVVeh[20];             /**< vecGainVitesse */
   tU8 EseDes_aVehAxis[20];               /**< vecAcc */
   tU8 EseDes_ratAmpAVeh[20];             /**< vecGainAcc */
   tU8 EseDes_ratAudioGain[8];            /**< Amplitude table of spatialization*/
   tU8 EseDes_tiAudioDly[8];              /**< Delay table of spatialization */
}EseDataLayout_mex;

typedef struct EseDataRSound
{
   tU8 EseDes_nEngAxis[142];              /**< vecRPM */
   tU8 EseDes_ratHrmcDecel[20];           /**< vecRang_i */
   tU8 EseDes_ratAmpNEngDecel[1420];      /**< matAmplitude_i */
   tU8 EseDes_ratHrmcAccel[50];           /**< vecRang_g */
   tU8 EseDes_ratAmpNEngAccel_1[1740];    /**< matAmplitude_g(1st part to be contained in ESE FileID 2) */
   tU8 EseDes_ratAmpNEngAccel_2[1810];    /**< matAmplitude_g(2nd part to be contained in ESE FileID 3) */
   tU8 EseDes_agPhaIninDecel[20];         /**< vecPhaseInitiale_i */
   tU8 EseDes_agPhaIninAccel[50];         /**< vecPhaseInitiale_g */
   tU8 EseDes_ratOffsHrmcDecel[20];       /**< vecInharmo_i */
   tU8 EseDes_ratOffsHrmcAccel[50];       /**< vecInharmo_g */
   tU8 EseDes_ratAccrPercAxis[20];        /**< vecCharge */
   tU8 EseDes_ratAmpAccrDecel[20];        /**< vecGainCharge_i */
   tU8 EseDes_ratAmpAccrAccel[20];        /**< vecGainCharge_g */
   tU8 EseDes_vVehAxis[20];               /**< vecVitesse */
   tU8 EseDes_ratAmpVVeh[20];             /**< vecGainVitesse */
   tU8 EseDes_aVehAxis[20];               /**< vecAcc */
   tU8 EseDes_ratAmpAVeh[20];             /**< vecGainAcc */
   tU8 EseDes_ratAudioGain[8];            /**< Amplitude table of spatialization*/
   tU8 EseDes_tiAudioDly[8];              /**< Delay table of spatialization */
}EseDataLayout_rsound;
#pragma pack(pop)

class EseFileContainerIVI
{
public:
  EseFileContainerIVI();
  ~EseFileContainerIVI();

  /**
    * @brief Get the data correspnds to a FileID
    *
    * EOL data needs to be mapped to ADR3-BEG ESE File data structures.
    * This function maps the EOL data to the ESE-File layout that can be sent to ADR3.
    *
    * @param[in] pData Pointer to the EOL data (src pointer)
    * @param[in] u32datalen Length of pData
    * @param[in] u8FileId ADR3 ESE-FileID
    * @param[out] pEseFileData Pointer to the array where the ESE-File data needs to copied
    * @return Size of ESE-File
    */
   tU32 u32GetEseFileData(tU8 *pData, tU32 u32datalen, tU8 u8FileId, tU8* pEseFileData);

   /**
    * @brief Get the size of a ESE-File
    *
    * @param[in] fileId ADR3 ESE-FileID
    * @return Size of ESE-File
    */
  tU32 u32GetDataSize(tU8 fileId) const;

private:
   /**
    * @brief Inflate a Two Dimensional matrix columns
    *
    * @param[in] pData Pointer to the data to be inflated
    * @param[out] pExpandedData Pointer to an array where the inflated data is copied
    * @param[in] u8InRows Number of rows in input data
    * @param[in] u8InCols Number of columns in input data
    * @param[in] u8OutCols Number of columns expected in inflated/output data
    */
   tVoid vInflateBoxDataColumns(const tU8 *pData, tU8* pExpandedData, tU8 u8InRows, tU8 u8InCols, tU8 u8OutCols);
   /**
    * @brief Internal function to get ESE-File data for Mex sound
    *
    * This function internall maps the EOL data to ESE-File data elements
    * @param[in] pData Pointer to the EOL data (src pointer)
    * @param[in] u8FileId ADR3 ESE-FileID
    * @param[out] pEseFileData Pointer to the array where the ESE-File data needs to copied
    */
   tU32 u32GetEseFileDataForMex(tU8 *pData, tU8 u8FileId, tU8* pEseFileData);
   /**
    * @brief Internal function to get ESE-File data for RSound
    *
    * This function internall maps the EOL data to ESE-File data elements
    * @param[in] pData Pointer to the EOL data (src pointer)
    * @param[in] u8FileId ADR3 ESE-FileID
    * @param[out] pEseFileData Pointer to the array where the ESE-File data needs to copied
    */
   tU32 u32GetEseFileDataForRSound(tU8 *pData, tU8 u8FileId, tU8* pEseFileData);
};

#endif /* ESEFILECONTAINERIVI_H_ */
