/**
 * @file EngineSoundHandlerIVI.cpp
 * @author pau4kor
 * @copyright (c) 2016 RBEI
 * @addtogroup fc_audiomanager
 * @{
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include <etrace_if.h>
#include "fc_audiomanager_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_ENGINESOUND
#include "trcGenProj/Header/EngineSoundHandlerIVI.cpp.trc.h"
#endif

/* Include FI interface */
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_ERRORCODES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_SERVICEINFO
#define MIDW_FI_S_IMPORT_INTERFACE_FI_TYPES
#include <midw_fi_if.h>

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_diagdebug_if.h" // For EngineSound EOL Datapool
#include "dp_generic_if.h"
#include "dp_audio_if.h"

#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include "vd_diaglog_if.h" // For Dialog support

#include "util/Macro.h"

#include "EngineSoundHandlerIVI.h"
#include "EseGeneratorAdr.h"
#include "EseGeneratorExtAmp.h"

#include "vd_adr3Msg_If.h" //for PREMIUM_AMP2

#include "InternalComponentCommunication/Messages/settings/IDProfileDataChanged.h"
#include "InternalComponentCommunication/Messages/Diag/IDDiagDefSet.h"

//Configurations for AllianceESEActivationType
#define ESE_DISABLED                      0x00
#define ESE_ENABLED_IN_HU                 0x01
#define ESE_ENABLED_IN_EXTAMP             0x02

//Configurations for AllianceESEHMIType
#define ESE_NO_HMI                        0x00
//#define ESE_HMI_DRIVEMODE               0x01
#define ESE_HMI_RSOUND                    0x02

#define IS_ESE_ACTIVE_IN_HU()             ((ESE_ENABLED_IN_HU == m_u8EseActivation) && (m_u8OutputInformation != PREMIUM_AMP1 && m_u8OutputInformation != PREMIUM_AMP2) && m_u8EseSupport)
#define IS_ESE_ACTIVE_IN_EXTAMP()         ((ESE_ENABLED_IN_EXTAMP == m_u8EseActivation) && (m_u8OutputInformation == PREMIUM_AMP1 || m_u8OutputInformation == PREMIUM_AMP2) && m_u8EseSupport)
#define IS_ESE_ACTIVE()                   (IS_ESE_ACTIVE_IN_HU() || IS_ESE_ACTIVE_IN_EXTAMP())
#define IS_ESE_ACTIVE_WITH_HMI()          (IS_ESE_ACTIVE() && (ESE_NO_HMI != m_u8EseHMI))
#define IS_ESE_ACTIVE_WITH_RSOUND()       (IS_ESE_ACTIVE() && (ESE_HMI_RSOUND == m_u8EseHMI))
// Only for g3g view build
#ifndef ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_08
#define ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_08 0x8608
#endif

EngineSoundHandlerIVI* EngineSoundHandlerIVI::getInstance()
{
   static EngineSoundHandlerIVI theInstance;
   return &theInstance;
}

EngineSoundHandlerIVI::EngineSoundHandlerIVI():
      IF_MessageObserver<PO_MessageConfig::enID>("Engine Sound Handler IVI Observer"),
      m_poMainAppl(NULL),
      m_u8EseActivation(ESE_DISABLED),
      m_u8OutputInformation(0x00),
      m_u8EseHMI(ESE_NO_HMI),
      m_u8EseSupport(0),
      m_pEseGeneratorIf(NULL),
      m_poResultHandler(NULL)
{
   vReadKDSConfigurations();

   if(IS_ESE_ACTIVE_IN_HU())
      m_pEseGeneratorIf = new EseGeneratorAdr(); //ESE Active in HU
   else if(IS_ESE_ACTIVE_IN_EXTAMP())
      m_pEseGeneratorIf = new EseGeneratorExtAmp(); //ESE Active in External Amplifier
   else
      ETG_TRACE_USR4(("EngineSoundHandlerIVI::ESE not active in HU or External Amplifier"));

   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this, PO_MessageConfig::ID_CCA_GetDiagResult);
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this, PO_MessageConfig::ID_ProfileDataChanged);
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this, PO_MessageConfig::ID_DiagDefSet);
}

/**
 * vReadKDSConfigurations()
 * Read KDS configurations related to ESE
 */
tVoid EngineSoundHandlerIVI::vReadKDSConfigurations()
{
   if(DP_S32_NO_ERR != DP_s32GetConfigItem("SystemConfiguration1","ESEActivation", &m_u8EseActivation, 1))
   {
      m_u8EseActivation = ESE_DISABLED;
      ETG_TRACE_ERR(("EngineSoundHandlerIVI : Failed to read KDS configuration for ESEActivation"));
   }

   if(DP_S32_NO_ERR != DP_s32GetConfigItem("SystemConfiguration1","OutputInformation", &m_u8OutputInformation, 1))
   {
      m_u8EseActivation = ESE_DISABLED; //To disable ESE incase KDS config is not valid
      ETG_TRACE_ERR(("EngineSoundHandlerIVI : Failed to read KDS configuration for OutputInformation"));
   }

   if(DP_S32_NO_ERR != DP_s32GetConfigItem("SystemConfiguration1","ESEHMI", &m_u8EseHMI, 1))
   {
      m_u8EseActivation = ESE_DISABLED; //To disable ESE incase KDS config is not valid
      ETG_TRACE_ERR(("EngineSoundHandlerIVI : Failed to read KDS configuration for ESEHMI"));
   }

   if(DP_S32_NO_ERR != DP_s32GetConfigItem("CMVariantCoding","ESESupport", &m_u8EseSupport, 1))
   {
      m_u8EseSupport = ESE_DISABLED; //To disable ESE incase KDS config is not valid
      ETG_TRACE_ERR(("EngineSoundHandlerIVI : Failed to read KDS PD configuration for ESESupport"));
   }
}

/**
 * vOnLoadSettings()
 * LOad the settings, after defset.
 */
tVoid EngineSoundHandlerIVI::vOnLoadSettings()
{
  ETG_TRACE_USR4(("EngineSoundHandlerIVI::vOnLoadSettings() entered."));

   m_oEseModel.vReload(); //Reload the data- model

   if(IS_ESE_ACTIVE_WITH_HMI())
   {
      if((ESE_HMI_RSOUND == m_u8EseHMI) && (RSOUND_ONOFF_ON == m_oEseModel.u8GetActiveRSoundStatus()) )
         m_pEseGeneratorIf->vSendRSoundType(m_oEseModel.u8GetActiveRSoundType(),m_oEseModel.u8GetActiveRSoundVolumeLevel());
      else
         m_pEseGeneratorIf->vSendESESoundType(m_oEseModel.u8GetActiveESESoundType(),m_oEseModel.u8GetActiveESEVolumeLevel());

      m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_RSOUNDONOFF);
      m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_RSOUNDSOUNDTYPE);
      m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_RSOUNDVOLUME);
      m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_ESEVOLUME);
      m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_ESESOUNDTYPE);
   }
}

EngineSoundHandlerIVI::~EngineSoundHandlerIVI()
{
  InternalCommunicationAdapter::getInstance()->POMessages->DeRegisterObserver(this);

  if (m_pEseGeneratorIf != NULL)
  {
    delete m_pEseGeneratorIf;
    m_pEseGeneratorIf = NULL;
  }

  m_poMainAppl = OSAL_NULL;
  m_poResultHandler = OSAL_NULL;
}

/**
 * bStartup()
 * Perform startup checks and corresponding actions. To be called only once at the beginning of the lifecycle
 */
tBool EngineSoundHandlerIVI::bStartup(fc_audiomanager_tclApp* poMainAppl)
{
  ETG_TRACE_USR4(("EngineSoundHandlerIVI::bStartup() entered."));

   tBool bRet = TRUE;
  m_poMainAppl = poMainAppl;

   if(IS_ESE_ACTIVE())
   {
      if(m_pEseGeneratorIf != NULL)
      {
         tU8 u8EseSoundType = (ESE_NO_HMI == m_u8EseHMI)?((tU8)(ENGINESOUND_SOUNDTYPE_NEUTRAL)):m_oEseModel.u8GetActiveESESoundType();

         //Special handling! when KDS configuration is changed from RSOUND enabled to disabled, let us set the status RSOUND to OFF
         if((ESE_HMI_RSOUND != m_u8EseHMI) && (RSOUND_ONOFF_ON == m_oEseModel.u8GetActiveRSoundStatus()))
            m_oEseModel.vSetActiveRSoundStatus(RSOUND_ONOFF_OFF);

         bRet = m_pEseGeneratorIf->bStartUp(u8EseSoundType, m_oEseModel.u8GetActiveESEVolumeLevel(),
                                             m_oEseModel.u8GetActiveRSoundStatus(),m_oEseModel.u8GetActiveRSoundType(),m_oEseModel.u8GetActiveRSoundVolumeLevel());
      }
      else
         bRet = FALSE;
   }

   //Special handling! when ESE configuration is changed from Enabled in HU to "Disabled" or "Enabled in External Amplifier", let us set the status in ADR3 to ESE OFF.
   if(m_u8EseSupport && (ESE_ENABLED_IN_HU != m_u8EseActivation))
   {
      tBool bOnOff = FALSE;
      vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_ENGINE_SOUND_ONOFF, VD_ADR3_OPTYPE_SET, 1, &bOnOff, ENGINE_SOUND_SET);
   }

  return bRet;
}

void EngineSoundHandlerIVI::vHandleTTFisCmd(tU32 size, tPCUChar pcu8Data)
{
  ETG_TRACE_USR4(("EngineSoundHandlerIVI::vHandleTTFisCmd() entered size : %d, data : %i", size, pcu8Data[2]));
   if(size < 4)
   {
    ETG_TRACE_FATAL(("EngineSoundHandlerIVI:not enough parameter"));
    return;
   }
  switch(pcu8Data[2])
  {
  case 0x00: // SET_ESE
      if(IS_ESE_ACTIVE())
      {
         if (m_pEseGeneratorIf != NULL)
            return m_pEseGeneratorIf->vSendESEOnOff((tBool)pcu8Data[3]);
      }
    break;
  default :
    break;
  }
}

void EngineSoundHandlerIVI::vLoadESEParameter(tU8* pu8Data, tU32 u32Size)
{
   if(IS_ESE_ACTIVE())
   {
      if(m_pEseGeneratorIf)
         m_pEseGeneratorIf->vLoadESEParameter(pu8Data,u32Size);
   }
}

void EngineSoundHandlerIVI::MessageNotification(PO_MessageConfig::enID MsgId)
{
  ETG_TRACE_USR4(("EngineSoundHandlerIVI::MessageNotification() entered. "));

   switch(MsgId)
   {
      case PO_MessageConfig::ID_CCA_GetDiagResult:
         {
            ETG_TRACE_USR4(("EngineSoundHandlerIVI::MessageNotification: ID_CCA_GetDiagResult"));
            // request
            const ID_CCA_GetDiagResult* poMsg = InternalCommunicationAdapter::POMessages->QueryMessage<ID_CCA_GetDiagResult>(MsgId);
            if (poMsg)
            {
               m_poResultHandler = &(poMsg->roResultHandler);
               for (tU16 i=0; i < poMsg -> roRequestedITCs.size(); i++)
                  if(poMsg -> roRequestedITCs[i] == ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_08)
                     vSendTestResultToDiaglog(); // Send result only for ESE ITC
            }
         }
         break;

      case PO_MessageConfig::ID_ProfileDataChanged:
         {
            const MSG_ProfileDataChanged* pMsgProfileData = InternalCommunicationAdapter::POMessages->QueryMessage<MSG_ProfileDataChanged>(MsgId);
            if(pMsgProfileData)
            {
               if(MSG_ProfileDataChanged::DATA_CHANGE_PROFILE_CHANGED == pMsgProfileData->m_enDataChangedType)
                  vOnLoadSettings();
            }
         }
         break;

      case PO_MessageConfig::ID_DiagDefSet:
         {
            const ID_DiagDefSet* pMsg =  InternalCommunicationAdapter::POMessages->QueryMessage<ID_DiagDefSet>(MsgId);
            if(pMsg)
            {
               tenDiagDefSetPhase enPhase = pMsg->enDiagDefSetPhase;
               tenDiagDefSetType enType = pMsg->enDiagDefSetType;

               if (EN_DEFSET_FINISH == enPhase && ( enType == EN_DEFSET_TEF || enType == EN_DEFSET_HMI ))
                  vOnLoadSettings ();
            }
         }
         break;

      default:
         break;
   }
}

void EngineSoundHandlerIVI::vAdrReady(enMsgType msgType)
{
   (void)msgType;

   if(IS_ESE_ACTIVE_IN_HU())
   {
      if(m_pEseGeneratorIf)
         m_pEseGeneratorIf->vDataCon();
   }
}

/**
 * bGetStatusMessage()
 * updates the requester for the property offered
 */
tBool EngineSoundHandlerIVI::bGetStatusMessage(tU16 u16FunctionId, amt_tclServiceData& roOutMsg, amt_tclServiceData* poInMsg) const
{
   switch (u16FunctionId)
   {
      case MIDW_MASCFFI_C_U16_ESESOUNDTYPE:
         return bSendESESoundTypeStatus(roOutMsg, poInMsg);

      case MIDW_MASCFFI_C_U16_ESEVOLUME:
         return bSendESEVolumeStatus(roOutMsg, poInMsg);

      case MIDW_MASCFFI_C_U16_ESEDRIVEMODE:
         return bSendESEDriveModeStatus(roOutMsg, poInMsg);

      case MIDW_MASCFFI_C_U16_RSOUNDONOFF:
         return bSendRSoundOnOffStatus(roOutMsg, poInMsg);

      case MIDW_MASCFFI_C_U16_RSOUNDSOUNDTYPE:
         return bSendRSoundTypeStatus(roOutMsg, poInMsg);

      case MIDW_MASCFFI_C_U16_RSOUNDVOLUME:
         return bSendRSoundVolumeStatus(roOutMsg, poInMsg);

    case MIDW_MASCFFI_C_U16_RSOUNDACTIVATION:
        return bSendRSoundActivationStatus(roOutMsg, poInMsg);

      default:
         return FALSE;
   }
}

/**
 * bProcessSet()
 * receives the CCA message set request for the property offered
 */
tBool EngineSoundHandlerIVI::bProcessSet(tU16 u16FunctionID, fi_tclVisitorMessage& roMessage, tBool& bPropertyChanged, tU16& u16ErrorCode)
{
   ETG_TRACE_USR4(("EngineSoundHandlerIVI::bProcessSet() entered. FID = 0x%4x.", u16FunctionID));

   // Handle property setting for this function ID here ...

   // Set [OUT] parameter 'bPropertyChanged' to TRUE if the value of the just
   // set property has changed.

   switch (u16FunctionID)
   {
      case MIDW_MASCFFI_C_U16_ESESOUNDTYPE:
         return bSetESESoundType(roMessage, bPropertyChanged, u16ErrorCode);

      case MIDW_MASCFFI_C_U16_ESEVOLUME:
         return bSetESEVolumeLevel(roMessage, bPropertyChanged, u16ErrorCode);

      case MIDW_MASCFFI_C_U16_ESEDRIVEMODE:
         return bSetESEDriveMode(roMessage, bPropertyChanged, u16ErrorCode);

      case MIDW_MASCFFI_C_U16_RSOUNDONOFF:
         return bSetRSoundOnOff(roMessage, bPropertyChanged, u16ErrorCode);

      case MIDW_MASCFFI_C_U16_RSOUNDSOUNDTYPE:
         return bSetRSoundType(roMessage, bPropertyChanged, u16ErrorCode);

      case MIDW_MASCFFI_C_U16_RSOUNDVOLUME:
         return bSetRSoundVolume(roMessage, bPropertyChanged, u16ErrorCode);

      case MIDW_MASCFFI_C_U16_RSOUNDACTIVATION:
       return bSetRSoundActivation(roMessage, bPropertyChanged, u16ErrorCode);

      default:
         u16ErrorCode = MIDW_MASCFFI_C_U16_ERROR_OPCODENOTSUPPORTED;
         return FALSE;
   }
}


/**
 * bSetESESoundType()
 * used to set the ESE sound type and send the corresponding ESE parameters to ADR3
 */
tBool EngineSoundHandlerIVI::bSetESESoundType(fi_tclVisitorMessage& roMessage, tBool& bPropertyChanged, tU16& u16ErrorCode)
{
  midw_mascffi_tclMsgESESoundTypeSet inmsg;

   if(roMessage.s32GetData(inmsg, FC_AUDIOMANAGER_SERVICE_AUDIO_FUNCTION_FI_MAJOR_VERSION) != OSAL_ERROR)
  {
    ETG_TRACE_USR4(("EngineSoundHandlerIVI : bSetESESoundType() Type=0x%x", inmsg.SoundType));

      switch(inmsg.SoundType)
      {
         case ENGINESOUND_SOUNDTYPE_NEUTRAL:
         case ENGINESOUND_SOUNDTYPE_SOUND1:
         case ENGINESOUND_SOUNDTYPE_SOUND2:
            if(IS_ESE_ACTIVE_WITH_HMI())
            {
               if(!m_pEseGeneratorIf || !m_poMainAppl)
                  return FALSE;

               if(m_oEseModel.u8GetActiveESESoundType() != inmsg.SoundType)
               {
                  bPropertyChanged = TRUE;
                  m_oEseModel.vSetActiveESESoundType(inmsg.SoundType); //Update the model
                  m_pEseGeneratorIf->vSendESESoundType(inmsg.SoundType,m_oEseModel.u8GetActiveESEVolumeLevel()); //Inform ADR3/Ext Amp
               }

               m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_ESEVOLUME);
               m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_ESESOUNDTYPE);
            }
            return TRUE;

         default:
            ETG_TRACE_ERR(("EngineSoundHandlerIVI : bSetESESoundType() Type=0x%x is unknown", inmsg.SoundType));
            u16ErrorCode = MIDW_MASCFFI_C_U16_ERROR_PARAMETEROUTOFRANGE;
            return FALSE;
      }
   }
  return FALSE;
}

/**
 * bSetESEVolumeLevel()
 * used to set the volume level for ESE sound type and send the corresponding ESE parameters to ADR3
 */
tBool EngineSoundHandlerIVI::bSetESEVolumeLevel(fi_tclVisitorMessage& roMessage, tBool& bPropertyChanged, tU16& u16ErrorCode)
{
  midw_mascffi_tclMsgESEVolumeSet inmsg;

  if(roMessage.s32GetData(inmsg, FC_AUDIOMANAGER_SERVICE_AUDIO_FUNCTION_FI_MAJOR_VERSION) != OSAL_ERROR)
  {
    ETG_TRACE_USR4(("bSetESEVolumeLevel() Level=0x%x", inmsg.Volume));

      switch(inmsg.Volume)
      {
         case ENGINESOUND_VOLUME_OFF:
         case ENGINESOUND_VOLUME_LOW:
         case ENGINESOUND_VOLUME_HIGH:
            if(IS_ESE_ACTIVE_WITH_HMI())
            {
               if(!m_pEseGeneratorIf || !m_poMainAppl)
                  return FALSE;

               if(m_oEseModel.u8GetActiveESEVolumeLevel() != inmsg.Volume)
               {
                  bPropertyChanged = TRUE;
                  m_oEseModel.vSetActiveESEVolumeLevel(inmsg.Volume); //Update the model
                  m_pEseGeneratorIf->vSendESEVolume(inmsg.Volume); //Inform ADR3/Ext Amp
               }

               m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_ESEVOLUME);
            }
            return TRUE;

         default:
            ETG_TRACE_ERR(("EngineSoundHandlerIVI : bSetESEVolumeLevel()Level=0x%x is unknown",inmsg.Volume));
            u16ErrorCode = MIDW_MASCFFI_C_U16_ERROR_PARAMETEROUTOFRANGE;
            return FALSE;
      }
  }

  return FALSE;
}

tBool EngineSoundHandlerIVI::bSetESEDriveMode(fi_tclVisitorMessage& roMessage, tBool& bPropertyChanged, tU16& u16ErrorCode)
{
  midw_mascffi_tclMsgESEDriveModeSet inmsg;

  if(roMessage.s32GetData(inmsg, FC_AUDIOMANAGER_SERVICE_AUDIO_FUNCTION_FI_MAJOR_VERSION) != OSAL_ERROR)
  {
    ETG_TRACE_USR4(("EngineSoundHandlerIVI : bSetESEDriveMode() Mode=0x%x", inmsg.DriveMode));
    if(inmsg.DriveMode > ENGINESOUND_DRIVEMODE_PERSO)
    {
      ETG_TRACE_ERR(("bSetESEDriveMode() unknown Drive Mode"));
      u16ErrorCode = MIDW_MASCFFI_C_U16_ERROR_PARAMETEROUTOFRANGE;
      return FALSE;
    }
    else if(IS_ESE_ACTIVE_WITH_HMI()) //TODO : Ext Amp requirements to be clear
    {
         if(!m_pEseGeneratorIf || !m_poMainAppl)
            return FALSE;

         if(m_oEseModel.u8GetActiveESEDriveMode() != inmsg.DriveMode)
         {
            bPropertyChanged = TRUE;
            m_oEseModel.vSetActiveESEDriveMode(inmsg.DriveMode); //Update the model

            if(RSOUND_ONOFF_OFF == m_oEseModel.u8GetActiveRSoundStatus())
               m_pEseGeneratorIf->vSendESESoundType(m_oEseModel.u8GetActiveESESoundType(),m_oEseModel.u8GetActiveESEVolumeLevel()); //Inform ADR3/Ext Amp
         }
         m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_ESEVOLUME);
         m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_ESESOUNDTYPE);
    }
    return TRUE;
  }
  return FALSE;
}

/**
 * bSetRSoundOnOff()
 * Used to set R-Sound On/Off
 */
tBool EngineSoundHandlerIVI::bSetRSoundOnOff(fi_tclVisitorMessage& roMessage, tBool& bPropertyChanged, tU16& u16ErrorCode)
{
   midw_mascffi_tclMsgRSoundOnOffSet inmsg;

   if(roMessage.s32GetData(inmsg, FC_AUDIOMANAGER_SERVICE_AUDIO_FUNCTION_FI_MAJOR_VERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR4(("EngineSoundHandlerIVI : bSetRSoundOnOff() Req =0x%x", inmsg.RSound));
      if(IS_ESE_ACTIVE_WITH_RSOUND())
      {
         if(!m_pEseGeneratorIf || !m_poMainAppl)
            return FALSE;

         if(m_oEseModel.u8GetActiveRSoundStatus() != inmsg.RSound)
         {
            bPropertyChanged = TRUE;
            m_oEseModel.vSetActiveRSoundStatus(inmsg.RSound); //Update the model

            if(RSOUND_ONOFF_ON == inmsg.RSound)
            {
               m_pEseGeneratorIf->vSendRSoundType(m_oEseModel.u8GetActiveRSoundType(),m_oEseModel.u8GetActiveRSoundVolumeLevel()); //Inform ADR3/Ext Amp

               //Update the clients
               m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_RSOUNDONOFF);
               m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_RSOUNDSOUNDTYPE);
               m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_RSOUNDVOLUME);

            }
            else if (RSOUND_ONOFF_OFF == inmsg.RSound)
            {
               m_pEseGeneratorIf->vSendESESoundType(m_oEseModel.u8GetActiveESESoundType(),m_oEseModel.u8GetActiveESEVolumeLevel()); //Update the drive mode ESE settings when R-Sound is OFF

               //Update the clients
               m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_RSOUNDONOFF);
               m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_ESEVOLUME);
               m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_ESESOUNDTYPE);
            }
            else
            {
               u16ErrorCode = MIDW_MASCFFI_C_U16_ERROR_PARAMETEROUTOFRANGE;
               return FALSE;
            }
         }

      }
      return TRUE;
   }
   return FALSE;
}

/**
 * bSetRSoundType()
 * Used to set the R-Sound sound type selected
 */
tBool EngineSoundHandlerIVI::bSetRSoundType(fi_tclVisitorMessage& roMessage, tBool& bPropertyChanged, tU16& u16ErrorCode)
{
   midw_mascffi_tclMsgRSoundSoundTypeSet inmsg;

   if(roMessage.s32GetData(inmsg, FC_AUDIOMANAGER_SERVICE_AUDIO_FUNCTION_FI_MAJOR_VERSION) != OSAL_ERROR)
   {
      //HMI uses indexes 1 -> 6 for RSOUND1 -> RSOUND6
      //Since the Mex and R-Sound types are continuously stored in EOL, internally Audio uses indexs 4 -> 9 for RSOUND types.
      tU8 u8SoundType = (tU8)(inmsg.RSoundType + NUM_OF_MEXSOUND);

      ETG_TRACE_USR4(("EngineSoundHandlerIVI : bSetRSoundType() SoundType = 0x%x (defined internally)", u8SoundType));
      if((u8SoundType < ENGINESOUND_SOUNDTYPE_RSOUND1) || (u8SoundType > ENGINESOUND_SOUNDTYPE_RSOUND6))
      {
         ETG_TRACE_ERR(("bSetRSoundType() unknown R-Sound type parameter"));
      u16ErrorCode = MIDW_MASCFFI_C_U16_ERROR_PARAMETEROUTOFRANGE;
      return FALSE;
      }
      else if(IS_ESE_ACTIVE_WITH_RSOUND())
      {
         if(!m_pEseGeneratorIf || !m_poMainAppl)
            return FALSE;

         if(m_oEseModel.u8GetActiveRSoundType() != u8SoundType)
         {
            bPropertyChanged = TRUE;
            m_oEseModel.vSetActiveRSoundType(u8SoundType); //Update the model
            m_pEseGeneratorIf->vSendRSoundType(u8SoundType,m_oEseModel.u8GetActiveRSoundVolumeLevel()); //Inform ADR3/Ext Amp
         }
         m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_RSOUNDSOUNDTYPE); //update clients
      }
      return TRUE;
   }
   return FALSE;
}

/**
 * bSetRSoundVolume()
 * Used to set R-Sound volume
 */
tBool EngineSoundHandlerIVI::bSetRSoundVolume(fi_tclVisitorMessage& roMessage, tBool& bPropertyChanged, tU16& u16ErrorCode)
{
   midw_mascffi_tclMsgRSoundVolumeSet inmsg;

   if(roMessage.s32GetData(inmsg, FC_AUDIOMANAGER_SERVICE_AUDIO_FUNCTION_FI_MAJOR_VERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR4(("EngineSoundHandlerIVI : bSetRSoundVolume() Volume = 0x%x", inmsg.Volume));
      if(inmsg.Volume > MAX_RSOUND_VOL_STEP)
      {
         ETG_TRACE_ERR(("bSetRSoundVolume() unknown R-Sound volume parameter"));
      u16ErrorCode = MIDW_MASCFFI_C_U16_ERROR_PARAMETEROUTOFRANGE;
      return FALSE;
      }
      if(IS_ESE_ACTIVE_WITH_RSOUND())
      {
         if(!m_pEseGeneratorIf || !m_poMainAppl)
            return FALSE;

         if(m_oEseModel.u8GetActiveRSoundVolumeLevel() != inmsg.Volume)
         {
            bPropertyChanged = TRUE;
            m_oEseModel.vSetActiveRSoundVolumeLevel(inmsg.Volume); //Update the model
            m_pEseGeneratorIf->vSendRSoundVolume(inmsg.Volume); //Inform ADR3/Ext Amp
         }
         m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_RSOUNDVOLUME); //update clients
      }
      return TRUE;
   }
   return FALSE;
}

tBool EngineSoundHandlerIVI::bSetRSoundActivation(fi_tclVisitorMessage& roVisitorMsg, tBool& bPropertyChanged, tU16& u16ErrorCode)
{
  midw_mascffi_tclMsgRSoundActivationSet inmsg;

  if(roVisitorMsg.s32GetData(inmsg, FC_AUDIOMANAGER_SERVICE_AUDIO_FUNCTION_FI_MAJOR_VERSION) != OSAL_ERROR)
  {
    ETG_TRACE_USR4(("EngineSoundHandlerIVI : bSetRSoundActivation() State=0x%x", inmsg.RSoundActive));

    if(inmsg.RSoundActive > RSOUND_ACTIVATION_ON)
    {
      ETG_TRACE_ERR(("bSetRSoundActivation() unknown RSound Activation State"));
      u16ErrorCode = MIDW_MASCFFI_C_U16_ERROR_PARAMETEROUTOFRANGE;
      return FALSE;
    }
    else if(m_oEseModel.u8GetActiveRSoundActivationStatus() != inmsg.RSoundActive)
    {
      if(!m_poMainAppl)
          return FALSE;

      bPropertyChanged = TRUE;
      m_oEseModel.vSetActiveRSoundActivationStatus(inmsg.RSoundActive); //Update the model

      m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_RSOUNDACTIVATION);
    }
    return TRUE;
  }
  return FALSE;
}

/**
 * bSendESESoundTypeStatus()
 * used to send the Sound type information to the requester(HMI)
 */
tBool EngineSoundHandlerIVI::bSendESESoundTypeStatus(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)const
{
  midw_mascffi_tclMsgESESoundTypeStatus oESESoundTypeStatus;

  oESESoundTypeStatus.SoundType = m_oEseModel.u8GetActiveESESoundType();

  fi_tclVisitorMessage oCCaMsg(oESESoundTypeStatus);
  // Always destroy the FI data object before leaving its creation scope
  oESESoundTypeStatus.vDestroy();

  return oCCaMsg.bHandOver(&roOutMsg);
}

/**
 * bSendESEVolumeStatus()
 * used to send the volume level information to the requester(HMI)
 */
tBool EngineSoundHandlerIVI::bSendESEVolumeStatus(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)const
{
  midw_mascffi_tclMsgESEVolumeStatus oESEVolumeStatus;

  oESEVolumeStatus.Volume = m_oEseModel.u8GetActiveESEVolumeLevel();

  fi_tclVisitorMessage oCCaMsg(oESEVolumeStatus);
  // Always destroy the FI data object before leaving its creation scope
  oESEVolumeStatus.vDestroy();

  return oCCaMsg.bHandOver(&roOutMsg);
}

/**
 * bSendESEDriveModeStatus()
 * used to send the drive mode information to the requester(HMI)
 */
tBool EngineSoundHandlerIVI::bSendESEDriveModeStatus(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)const
{
   midw_mascffi_tclMsgESEDriveModeStatus oESEDriveModeStatus;

   oESEDriveModeStatus.DriveMode = m_oEseModel.u8GetActiveESEDriveMode();

   fi_tclVisitorMessage oCCaMsg(oESEDriveModeStatus);
   // Always destroy the FI data object before leaving its creation scope
   oESEDriveModeStatus.vDestroy();

   return oCCaMsg.bHandOver(&roOutMsg);
}

/**
 * bSendRSoundOnOffStatus()
 * used to send the RSound On/Off information to the requester(HMI)
 */
tBool EngineSoundHandlerIVI::bSendRSoundOnOffStatus(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)const
{
   midw_mascffi_tclMsgRSoundOnOffStatus oStatus;

   oStatus.RSound = m_oEseModel.u8GetActiveRSoundStatus();

   fi_tclVisitorMessage oCCaMsg(oStatus);
   // Always destroy the FI data object before leaving its creation scope
   oStatus.vDestroy();

   return oCCaMsg.bHandOver(&roOutMsg);
}

/**
 * bSendESEDriveModeStatus()
 * used to send the RSound sound type information to the requester(HMI)
 */
tBool EngineSoundHandlerIVI::bSendRSoundTypeStatus(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)const
{
   midw_mascffi_tclMsgRSoundSoundTypeStatus oStatus;

   //HMI uses indexes 1 -> 6 for RSOUND1 -> RSOUND6
   //Since the Mex and R-Sound types are continuously stored in EOL, internally Audio uses indexs 4 -> 9 for RSOUND types (after 3 Mex sounds)
   //Here adjusting the value as expected by HMI.
   oStatus.RSoundType = (tU8) ((m_oEseModel.u8GetActiveRSoundType())- NUM_OF_MEXSOUND);

   fi_tclVisitorMessage oCCaMsg(oStatus);
   // Always destroy the FI data object before leaving its creation scope
   oStatus.vDestroy();

   return oCCaMsg.bHandOver(&roOutMsg);
}

/**
 * bSendESEDriveModeStatus()
 * used to send the RSound volume information to the requester(HMI)
 */
tBool EngineSoundHandlerIVI::bSendRSoundVolumeStatus(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)const
{
   midw_mascffi_tclMsgRSoundVolumeStatus oStatus;

   oStatus.Volume = m_oEseModel.u8GetActiveRSoundVolumeLevel();

   fi_tclVisitorMessage oCCaMsg(oStatus);
   // Always destroy the FI data object before leaving its creation scope
   oStatus.vDestroy();

   return oCCaMsg.bHandOver(&roOutMsg);
}

/**
 * bSendRSoundActivationStatus()
 * used to send the RSound Activation information to the requester(HMI)
 */
tBool EngineSoundHandlerIVI::bSendRSoundActivationStatus(amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)const
{
  midw_mascffi_tclMsgRSoundActivationStatus oRSoundActivationStatus;

  oRSoundActivationStatus.RSoundActive = m_oEseModel.u8GetActiveRSoundActivationStatus();

   fi_tclVisitorMessage oCCaMsg(oRSoundActivationStatus);
   // Always destroy the FI data object before leaving its creation scope
   oRSoundActivationStatus.vDestroy();

   return oCCaMsg.bHandOver(&roOutMsg);
}

tVoid EngineSoundHandlerIVI::vSendESEOnOff(tBool bOn)
{
   if(IS_ESE_ACTIVE())
   {
      if(m_pEseGeneratorIf)
         m_pEseGeneratorIf->vSendESEOnOff(bOn);
   }
}

tVoid EngineSoundHandlerIVI::vRequestESEOnOffStatus()
{
   if(IS_ESE_ACTIVE())
   {
      if(m_pEseGeneratorIf)
         m_pEseGeneratorIf->vRequestESEOnOffStatus();
   }
}

tVoid EngineSoundHandlerIVI::vSendRSoundVolume(tU8 u8Volume)
{
   if(IS_ESE_ACTIVE())
   {
      if(m_pEseGeneratorIf)
         m_pEseGeneratorIf->vSendRSoundVolume(u8Volume);
   }
}

tVoid EngineSoundHandlerIVI::vRequestESEVolumeStatus()
{
   if(IS_ESE_ACTIVE())
   {
      if(m_pEseGeneratorIf)
         m_pEseGeneratorIf->vRequestESEVolumeStatus();
   }
}

tVoid EngineSoundHandlerIVI::vSendTestResultToDiaglog()
{
  ETG_TRACE_USR4(("EngineSoundHandlerIVI : vSendTestResultToDiaglog() entered"));

  tTestResultList oFiTestResultObject;//test result list

  tTestResult oTestData;// Set the parameters of the testresult data object

  oTestData.u16ID = ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_08;
  //oTestData.enResult = EN_NORESULT;//default no test result for ESE datapool read

  tU8 au8DatapoolData[ENGINE_SOUND_DATA_SIZE_BYTES];
  dp_tclEngineSoundDPEngineSound_Data oDpEngineSound;
  oDpEngineSound.bReloadDpElement();//reload the dp contents if already read once

  tS32 s32Ret = oDpEngineSound.s32GetData(au8DatapoolData, ENGINE_SOUND_DATA_SIZE_BYTES);

  oTestData.enResult = (ENGINE_SOUND_DATA_SIZE_BYTES == s32Ret) ? EN_PASSED : EN_FAILED; //TODO: to be extended with CRC check as well


  ETG_TRACE_USR3(("EngineSoundHandlerIVI::vSendTestResultToDiaglog - storing result to result list : %i", ETG_CENUM(tenTestResult, oTestData.enResult)));
  //add the testData to the TestResultList
  oFiTestResultObject.push_back(oTestData);

  if(m_poResultHandler != NULL)
  {
      ETG_TRACE_USR3(("EngineSoundHandlerIVI::vSendTestResultToDiaglog - sending result to result handler"));
      m_poResultHandler->vSaveDiagResult(oFiTestResultObject);
  }

  //clear the result list
  oFiTestResultObject.clear();
}
