/*******************************************************************************
* FILE:          fc_audiomanager_DiaglibHandler_Buzzer.cpp
* PROJECT:
* SW-COMPONENT:  fc_audiomanager application
*-------------------------------------------------------------------------------
*
* DESCRIPTION:   Implementation of a IOControl listener.
* This handler implements the Buzzer Control via the Diagnosis
*
*-------------------------------------------------------------------------------
*
* COPYRIGHT:    (c) 2018 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author                  | Modification
* 02.03.18  | STC2HI                  | Inital version
*******************************************************************************/


// +++ Specify what part of diaglib you need in this file +++
#define DIAGLIB_INCLUDE_IO_CONTROL
#include "diaglib_if.h"

#include "fc_audiomanager_DiaglibHandler_Buzzer.h"
#include "fc_audiomanager_trace.h"
#include "../InternalComponentCommunication/InternalCommunicationAdapter.h"
#include "../InternalComponentCommunication/Messages/Diag/ID_DiagBuzzer.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_AUDIO_DIAGNOSIS
#include "trcGenProj/Header/fc_audiomanager_DiaglibHandler_Buzzer.cpp.trc.h"
#endif



/*******************************************************************************
**  FUNCTION:
*   fc_audiomanager_tclDiaglibHandler_Buzzer(
*                                    diaglib::tclServiceDiaglib* pDiaglibService)
*                                    : mpDiaglibService(pDiaglibService)
*   Constructor
*******************************************************************************/

fc_audiomanager_tclDiaglibHandler_Buzzer::fc_audiomanager_tclDiaglibHandler_Buzzer(
                                      diaglib::tclServiceDiaglib* pDiaglibService)
                                        : mpDiaglibService(pDiaglibService)
{
#ifdef IOCONTROLID_BUZZER_TEST_TONE
   if(mpDiaglibService != NULL)
   {
      mpDiaglibService->poGetIoCtrl()->vRegisterListener(IOCONTROLID_BUZZER_TEST_TONE, this);
      ETG_TRACE_USR3(("fc_audiomanager_tclDiaglibHandler_Buzzer => vRegisterListener(IOCONTROLID_BUZZER_TEST_TONE) "));
   }
   else
   {
      ETG_TRACE_FATAL(( "fc_audiomanager_tclDiaglibHandler_Buzzer constructor: diaglib pointer is NULL" ));
      FATAL_M_ASSERT_ALWAYS();
   }

#endif
}



/*******************************************************************************
**  FUNCTION:
*   ~fc_audiomanager_tclDiaglibHandler_Buzzer()
*   Destructor
*******************************************************************************/
fc_audiomanager_tclDiaglibHandler_Buzzer::~fc_audiomanager_tclDiaglibHandler_Buzzer()
{
   mpDiaglibService = NULL;
}



/*******************************************************************************
**  FUNCTION:
*   vOnIoControl(tU32 u32IoControlId, diaglib::tenIoControlAction enActionId,
*                 const diaglib::tclParameterVector& oParameterVector,diaglib::tContext MsgContext)
*   DESCRIPTION: Command from Diagnosis for set Phantom Voltage request
*******************************************************************************/
tU32 fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControl (
                                                             tU32 u32IoControlId,
                                                             diaglib::tenIoControlAction enActionId,
                                                             const diaglib::tclParameterVector& oParameterVector,
                                                             diaglib::tContext MsgContext
                                                           )
{
   ETG_TRACE_USR3(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControl entered"));

   tU32 u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
   diaglib::tenIoCtrlResult oCtrlResult  = diaglib::EN_IO_CONTROL_NOT_OK;

   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( enActionId );
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( MsgContext );
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( oParameterVector )
   // check for expected Routine ID

#ifdef IOCONTROLID_BUZZER_TEST_TONE
   switch(u32IoControlId)
   {

   case IOCONTROLID_BUZZER_TEST_TONE:
      {
         ETG_TRACE_USR3(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControl => IOCONTROLID_BUZZER_TEST_TONE"));

         // create object for response
         diaglib::tclParameterVector oTmpVec;

         if(diaglib::bCheckParameterSignature(oParameterVector,
                                              IOCONTROLID_BUZZER_TEST_TONE_SET_LEN,
                                              IOCONTROLID_BUZZER_TEST_TONE_SET_PARAM1,
                                              IOCONTROLID_BUZZER_TEST_TONE_SET_PARAM2) == FALSE)
         {
            ETG_TRACE_ERR(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControl => ERROR: bCheckParameterSignature FAILED"));
            oCtrlResult = diaglib::EN_IO_CONTROL_NOT_OK; // Control failed

            diaglib::trParameter paramaError;
            paramaError.enType         = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
            paramaError.u8Value        = diaglib::EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE;
            oTmpVec.push_back(paramaError);
         }// if(diaglib::bCheckParameterSignature(oParamVector, ...
         else
         {
             // get Data
            tU8 u8BuzzerId = oParameterVector[IOCONTROLID_BUZZER_TEST_TONE_SET_u8Buzzer].u8Value; //0=Buzzer1, 1=Buzzer2...
            tBool bControlState = oParameterVector[IOCONTROLID_BUZZER_TEST_TONE_SET_bControlState].u8Value; //0=disable, 1=enable

            ETG_TRACE_USR3(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControl => Diag req sent for Buzzer ID=%d, ControlState=%d", u8BuzzerId, bControlState));

             if (u8BuzzerId)
             {
                 ID_DiagBuzzer DiagBuzzerMsg(OSAL_EN_FASCIA_CTRL_BZR_4K, bControlState);
                 InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&DiagBuzzerMsg);
             }
             else
             {
                 ID_DiagBuzzer DiagBuzzerMsg(OSAL_EN_FASCIA_CTRL_BZR_4K5, bControlState);
                 InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&DiagBuzzerMsg);
             }

             oCtrlResult = diaglib::EN_IO_CONTROL_OK; // data valid
             diaglib::trParameter paramBuzzer;
             paramBuzzer.enType         = static_cast<diaglib::tenParameterType>(IOCONTROLID_BUZZER_TEST_TONE_RESULT_PARAM1);
             paramBuzzer.u8Value        = u8BuzzerId;
             oTmpVec.push_back(paramBuzzer);

             diaglib::trParameter paramControlState;
             paramControlState.enType         = static_cast<diaglib::tenParameterType>(IOCONTROLID_BUZZER_TEST_TONE_RESULT_PARAM2);
             paramControlState.u8Value        = bControlState;
             oTmpVec.push_back(paramControlState);
         }



         // send message back to diagnosis
         if(mpDiaglibService != OSAL_NULL)
         {
            if(TRUE == mpDiaglibService->poGetIoCtrl()->bSendIoCtrlMethodResult( oCtrlResult,
                                                                                 enActionId,
                                                                                 oTmpVec,
                                                                                 MsgContext) )
            {
               u32Return = diaglib::U32_DIAGLIB_RETURN_OK;
               ETG_TRACE_USR3(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControl => bSendIoCtrlMethodResult successfull"));
            }// if(TRUE == mpDiaglibService->poGetIoCtrl()
            else
            {
               u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
               ETG_TRACE_ERR(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControl => ERROR : bSendRoutineCtrlMethodResult failed"));
            }// else // if(TRUE == mpDiaglibService->poGetIoCtrl()
         }// if(mpDiaglibService != OSAL_NULL)
         else
         {
            u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
            ETG_TRACE_ERR(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControl => ERROR : mpDiaglibService == OSAL_NULL"));
         }// else // if(mpDiaglibService != OSAL_NULL)
         break;

      }// case IOCONTROLID_BUZZER_TEST_TONE:

      default:
      {
         break;
      }// default:
   }// switch(u32IoControlId)
#endif
   ETG_TRACE_USR3(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControl(%x)",u32Return));
   ETG_TRACE_USR3(("---oCtrlResult = %d ",oCtrlResult));
   ETG_TRACE_USR3(("---u32IoControlId = %d ",u32IoControlId));

   return u32Return;
}



/*******************************************************************************
**  FUNCTION:
*   vOnIoControlFreeze ( tU32 u32IoControlId, diaglib::tContext MsgContext)
*   DESCRIPTION: Command from Diagnosis for Freeze the State before the Diagnosis Request
*                (not implemented in the diagnosis)
*******************************************************************************/
tU32 fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControlFreeze (
                                              tU32 u32IoControlId,
                                              diaglib::tContext MsgContext
                                            )
{
  (void)u32IoControlId;
   ETG_TRACE_USR3(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControlFreeze"));

   tU32 u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
   diaglib::tenIoCtrlResult oCtrlResult  = diaglib::EN_IO_CONTROL_OK;

   //////////////////////////////////////////////////////
   //// HANDLE FREEZE for Buzzer
/*   if (u32IoControlId == IOCONTROLID_BUZZER_TEST_TONE)
   {
      //Normally nothing to do
   }*/
   //////////////////////////////////////////////////////


   if(mpDiaglibService != OSAL_NULL)
   {
      if(TRUE == mpDiaglibService->poGetIoCtrl()->bSendIoCtrlFreezeResult( oCtrlResult,
                                                                           MsgContext) )
      {

         u32Return = diaglib::U32_DIAGLIB_RETURN_OK;
         ETG_TRACE_USR3(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControlFreeze => bSendIoCtrlFreezeResult successfull"));

      }// if(TRUE == mpDiaglibService->poGetIoCtrl()
      else
      {
         u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
         ETG_TRACE_ERR(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControlFreeze => ERROR : bSendRoutineCtrlMethodResult failed"));
      }// else // if(TRUE == mpDiaglibService->poGetIoCtrl()
   }// if(mpDiaglibService != OSAL_NULL)
   else
   {
      u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
      ETG_TRACE_ERR(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControlFreeze => ERROR : mpDiaglibService == OSAL_NULL"));
   }// else // if(mpDiaglibService != OSAL_NULL)

   ETG_TRACE_USR3(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControlFreeze(%x)",u32Return));

   return u32Return;
}



/*******************************************************************************
**  FUNCTION:
*   vOnIoControlUnfreeze ( tU32 u32IoControlId, diaglib::tContext MsgContext)
*   DESCRIPTION: Command from Diagnosis for Unfreeze the State before the Diagnosis Request
*
*******************************************************************************/
tU32 fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControlUnfreeze (
                                              tU32 u32IoControlId,
                                              diaglib::tContext MsgContext
                                            )
{
  (void)u32IoControlId;
   ETG_TRACE_USR3(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControlUnfreeze"));

   tU32 u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
   diaglib::tenIoCtrlResult oCtrlResult  = diaglib::EN_IO_CONTROL_OK;

   //////////////////////////////////////////////////////
   //// HANDLE UN-FREEZE for Buzzer
#ifdef IOCONTROLID_BUZZER_TEST_TONE
   if (u32IoControlId == IOCONTROLID_BUZZER_TEST_TONE)
   {
      //Switch all Buzzer off - two buzzers supported by audio
       ID_DiagBuzzer DiagBuzzer1Msg(OSAL_EN_FASCIA_CTRL_BZR_4K5, 0);
       InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&DiagBuzzer1Msg);

       ID_DiagBuzzer DiagBuzzer2Msg(OSAL_EN_FASCIA_CTRL_BZR_4K, 0);
       InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&DiagBuzzer2Msg);
   }
#endif
   //////////////////////////////////////////////////////

   if(mpDiaglibService != OSAL_NULL)
   {
      if(TRUE == mpDiaglibService->poGetIoCtrl()->bSendIoCtrlUnfreezeResult( oCtrlResult,
                                                                             MsgContext) )
      {
         u32Return = diaglib::U32_DIAGLIB_RETURN_OK;
         ETG_TRACE_USR3(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControlUnfreeze => bSendIoCtrlFreezeResult successfull"));

      }// if(TRUE == mpDiaglibService->poGetIoCtrl()
      else
      {
         u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
         ETG_TRACE_ERR(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControlUnfreeze => ERROR : bSendRoutineCtrlMethodResult failed"));
      }// else // if(TRUE == mpDiaglibService->poGetIoCtrl()
   }// if(mpDiaglibService != OSAL_NULL)
   else
   {
      u32Return = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
      ETG_TRACE_ERR(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControlUnfreeze => ERROR : mpDiaglibService == OSAL_NULL"));
   }// else // if(mpDiaglibService != OSAL_NULL)

   ETG_TRACE_USR3(("fc_audiomanager_tclDiaglibHandler_Buzzer::vOnIoControlUnfreeze(%x)",u32Return));
   return u32Return;

}


