/*******************************************************************************
* FILE:          fc_audiomanager_Microphone_Diagnosis_main.cpp
* PROJECT:
* SW-COMPONENT:  Microphone Diagnosis in the FC_Audiomanager application
*-------------------------------------------------------------------------------
*
* DESCRIPTION:   Microphone diagnosis main class
*
*-------------------------------------------------------------------------------
*
* COPYRIGHT:    (c) 2015 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author                            | Modification
* 09.02.15  | SoftTec Gaidatzi-Lken            | Inital version
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/


#include "../../fc_audiomanager_main.h"
//#include "../../fc_audiomanager_service_Audio_Function.h"

//#include "fc_audiomanager_Microphone.h"
#include "fc_audiomanager_Microphone_Diagnosis_main.h"
#include "fc_audiomanager_DiaglogClientHandler_Microphone.h"
//#include "fc_audiomanager_DiaglibClientHandler_SetMicVoltage.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#include "dp_audio_if.h"

// already included --- #include "../InternalComponentCommunication/Messages/power/ID_ApplicationStatus.h"
// already included --- #include "../InternalComponentCommunication/Messages/mute/IDMuteStateHandleVoltage.h"


#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_MIC_DIAGNOSIS
#include "trcGenProj/Header/fc_audiomanager_Microphone_Diagnosis_main.cpp.trc.h"
#endif

#define MIC_NONE 0x00
#if defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SMART)
#define MIC_1_RENAULT 0x01
#define BFM2 0x02
#define BFM3 0x03
#define MIC_2_RENAULT 0x04
#endif

//new PSA requirement
#if defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX
    #define FC_AUDIOMANAGER_TIMER_MIC_DIAG_FIRST_TICK        4000
    #define FC_AUDIOMANAGER_TIMER_MIC_DIAG_TICK_INTERVAL     2000

#else
    #define FC_AUDIOMANAGER_TIMER_MIC_DIAG_FIRST_TICK        2000
    #define FC_AUDIOMANAGER_TIMER_MIC_DIAG_TICK_INTERVAL     1000
#endif
tenMicState fc_audiomanager_tclMicrophone_Diagnosis::m_tMicState[2] = {MIC_STATE_UNKNOWN,MIC_STATE_UNKNOWN};
/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::fc_audiomanager_tclMicrophone_Diagnosis
 *
 * DESCRIPTION:   Constructor
 *
 * PARAMETERS:    poMain:        (->I) main
 *                poService:     (->I) service
 *
 * RETURN TYPE:   None
 *
 *******************************************************************************/
fc_audiomanager_tclMicrophone_Diagnosis::fc_audiomanager_tclMicrophone_Diagnosis(fc_audiomanager_tclApp * poMain, fc_audiomanager_tclService_Audio_Function * poService)
: IF_MessageObserver<PO_MessageConfig::enID>("Microphone_Diagnosis")
{

  if (poMain == NULL || poService == NULL)
  {
      ETG_TRACE_FATAL(( "fc_audiomanager_tclMicrophone_Diagnosis: Pointer poMain or poService is NULL" ));
      FATAL_M_ASSERT_ALWAYS(); //coverity warning fix
  }

   m_poMainApp = poMain;
   m_poAudioService = poService;
   m_bNormalAppState = FALSE;
   m_bNormalVoltageState = TRUE;
   m_bTimerStarted = FALSE;
   bExistTestDataToSend = FALSE;
   m_u8MicVal = 0x01;

   // Initialize the MicDataList for all the microphones
   for (tU8 i=0; i<MAX_MIC+1; i++)
   {
        MicDataList[i].MicValid = FALSE;
        MicDataList[i].MicState = MIC_STATE_UNKNOWN;
        MicDataList[i].MicCurrent = 0;
        MicDataList[i].MicDetection = MIC_DET_UNKNOWN;
   }

   vGetMicConfig(); //Get microphone configuration from KDS

   /* Microphone variables */
   m_pMicrophone = NULL;

   /* Diaglog variables */
   m_pDiaglogClientHandler_Microphone = NULL;

   /* Diaglib variables */
   //m_pDiaglibHandler_SetMicVoltage = NULL;

   /* Generate microphone component */
   m_pMicrophone = OSAL_NEW fc_audiomanager_tclMicrophone();
   if (m_pMicrophone == NULL)
   {
      ETG_TRACE_FATAL(( "fc_audiomanager_tclMicrophone_Diagnosis: creation of microphone component failed" ));
      FATAL_M_ASSERT_ALWAYS();
   }

    /* Generate microphone Diaglog class  */
    m_pDiaglogClientHandler_Microphone = OSAL_NEW fc_audiomanager_tclDiaglogClientHandler_Microphone(this);
    if (m_pDiaglogClientHandler_Microphone == NULL)
    {
       ETG_TRACE_FATAL(( "fc_audiomanager_tclMicrophone_Diagnosis: creation of diaglog ClientHandler for microphone failed" ));
       FATAL_M_ASSERT_ALWAYS();
    }

    /* If the Diaglib Interface is later required.
       Generate Diaglib IF (diagnosis)
    */
/*
    diaglib::tclServiceDiaglib* m_pDiaglibService = NULL;
    m_pDiaglibService = m_poMainApp->m_poAudio_Diagnosis->m_pDiaglibService;
    m_pDiaglibHandler_SetMicVoltage = OSAL_NEW fc_audiomanager_tclDiaglibHandler_SetMicVoltage(m_pDiaglibService);
    if (m_pDiaglibHandler_SetMicVoltage == NULL)
    {
       ETG_TRACE_FATAL(( "fc_audiomanager_tclMicrophone_Diagnosis: creation of diaglib Set Mic Voltage handler handler for diagnosis failed" ));
       FATAL_M_ASSERT_ALWAYS();
    }
*/
    vAddObserver();

}




/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::~fc_audiomanager_tclMicrophone_Diagnosis
 *
 * DESCRIPTION:   Destructor
 *
 * PARAMETERS:    None
 *
 * RETURN TYPE:   None
 *
 *******************************************************************************/
fc_audiomanager_tclMicrophone_Diagnosis::~fc_audiomanager_tclMicrophone_Diagnosis()
{
   InternalCommunicationAdapter::getInstance()->POMessages->DeRegisterObserver(this);
   m_poMainApp = OSAL_NULL;
   m_poAudioService = OSAL_NULL;

   /* Delete Diaglog handler for diagnosis */
   if (m_pDiaglogClientHandler_Microphone != NULL)
   {
        OSAL_DELETE m_pDiaglogClientHandler_Microphone;
        m_pDiaglogClientHandler_Microphone = NULL;
   }

   /* If the Diaglib Interface is later required.
      Delete Diaglib handler for diagnosis
   */
/*
   if (m_pDiaglibHandler_SetMicVoltage != NULL)
   {
        OSAL_DELETE m_pDiaglibHandler_SetMicVoltage;
        m_pDiaglibHandler_SetMicVoltage = NULL;
   }
*/
   /* Delete microphone component */
   if (m_pMicrophone != NULL)
   {
        OSAL_DELETE m_pMicrophone;
        m_pMicrophone = NULL;
   }


}

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::vProcessTimer
 *
 * DESCRIPTION:   Timer function.
 *
 * PARAMETERS:    None
 *
 * RETURN TYPE:   None
 *
 *******************************************************************************/
tVoid fc_audiomanager_tclMicrophone_Diagnosis::vProcessTimer(tVoid)
{
   vMakeMicDiag();
}




/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::vMakeMicDiag
 *
 * DESCRIPTION:   Main function to process the Mic diagmosis.
 *
 * PARAMETERS:    None
 *
 * RETURN TYPE:   None
 *
 *******************************************************************************/
tVoid fc_audiomanager_tclMicrophone_Diagnosis::vMakeMicDiag(tVoid)
{
//here we make the feature for g3g projects

#if defined (VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SUZUKI_LINUX)
   ETG_TRACE_USR4(("vMakeMicDiag(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SUZUKI_LINUX"));
   vMakeMicDiagSUZ();

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX
   ETG_TRACE_USR4(("vMakeMicDiag(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX"));
   if(m_bNormalAppState && m_bNormalVoltageState)
    vMakeMicDiagPSA();

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI  || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SMART) //SMART HW is similar to AIVI 
   ETG_TRACE_USR4(("vMakeMicDiag(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI/SMART"));
   vMakeMicDiagAIVI();

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CAF
   ETG_TRACE_USR4(("vMakeMicDiag(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CAF"));
   vMakeMicDiagCAF();

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CHERYM31T
   ETG_TRACE_USR4(("vMakeMicDiag(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CHERYM31T"));
   vMakeMicDiagCAF();

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_JAC_CII
   ETG_TRACE_USR4(("vMakeMicDiag(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_JAC_CII"));
   vMakeMicDiagJAC();
   
#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_INF4CV
//implementation not available
//inf4cv use a different MIC, MIC1_DIAG and MIC2_DIAG pins are not available in hardware.hence no diagnosis required in inf4c for mic

#else
   ETG_TRACE_USR4(("vMakeMicDiag(): VARIANT is not known"));
#endif

}


/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::vMakeMicDiagSUZ
 *
 * DESCRIPTION:   makes mic diagnosis
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************/
tVoid fc_audiomanager_tclMicrophone_Diagnosis::vMakeMicDiagSUZ()
{
    tsMicData *tsMicDataElem;
    tU8 MicNr = 1;


    if (m_pMicrophone == NULL)
    {
        ETG_TRACE_FATAL(( "vMakeMicDiagSUZ: Pointer is NULL" ));
        return;
    }

    if (MicDataList[1].MicValid) //Is the microphone 1 configured?
    {

        tsMicDataElem = m_pMicrophone->tenGetMicData(MicNr);
        //Get microphone State and Current and Detection from microphone component (class)


        if((tsMicDataElem->MicState != MicDataList[MicNr].MicState) ||
           (tsMicDataElem->MicCurrent != MicDataList[MicNr].MicCurrent) ||
           (tsMicDataElem->MicDetection != MicDataList[MicNr].MicDetection))

        {//this part will be entered at the first time because MicState differs at the first time from MicDataList.MicState (initially undefined)
            MicDataList[MicNr].MicState = tsMicDataElem->MicState;
            MicDataList[MicNr].MicCurrent = tsMicDataElem->MicCurrent;
            MicDataList[MicNr].MicDetection = tsMicDataElem->MicDetection;

            ETG_TRACE_USR2(("vMakeMicDiagSUZ(): changed values: MicrophoneCurrent = %d MicrophoneConnectionStatus = %d MicrophoneDetection = %d"
                            , MicDataList[MicNr].MicCurrent, MicDataList[MicNr].MicState, MicDataList[MicNr].MicDetection));

            /************************************************************
              DIAGLIB: Connect to the masc_fi interface (update property)
             ***********************************************************/
            if(m_poAudioService)
            {
              ETG_TRACE_USR2(("vMakeMicDiagSUZ(): updateClients call"));
              (tVoid)m_poAudioService->updateClients(MIDW_MASCFFI_C_U16_MICROPHONECONNECTIONSTATUS);
            }
        }
    }
}


/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::vMakeMicDiagPSA
 *
 * DESCRIPTION:   makes mic diagnosis
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************/
tVoid fc_audiomanager_tclMicrophone_Diagnosis::vMakeMicDiagPSA()
{
    tsMicData *tsMicDataElem;
    tBool bValuesChanged = FALSE;
    tU8 MicNr;


    if (m_pMicrophone == NULL)
    {
        ETG_TRACE_FATAL(( "vMakeMicDiagPSA: Pointer is NULL" ));
        return;
    }


    if (MicDataList[1].MicValid) //Is the microphone 1 configured?
    {
        MicNr = 1;
        tsMicDataElem = m_pMicrophone->tenGetMicData(MicNr);
        //Get microphone State and Current and Detection from microphone component (class)

    if(tsMicDataElem->MicState != m_tMicState[0])
    {
      m_tMicState[0] = tsMicDataElem->MicState;
      ETG_TRACE_USR2(("vMakeMicDiagPSA(): Mic 1 values changed"));
    }
    else
    {
        if((tsMicDataElem->MicState != MicDataList[MicNr].MicState) ||
           (tsMicDataElem->MicCurrent != MicDataList[MicNr].MicCurrent) ||
           (tsMicDataElem->MicDetection != MicDataList[MicNr].MicDetection))

            {
            MicDataList[MicNr].MicState = tsMicDataElem->MicState;
            MicDataList[MicNr].MicCurrent = tsMicDataElem->MicCurrent;
            MicDataList[MicNr].MicDetection = tsMicDataElem->MicDetection;

            ETG_TRACE_USR2(("vMakeMicDiagPSA(): changed values: MicrophoneCurrent = %d MicrophoneConnectionStatus = %d MicrophoneDetection = %d"
                            , MicDataList[MicNr].MicCurrent, MicDataList[MicNr].MicState, MicDataList[MicNr].MicDetection));

            ETG_TRACE_USR2(("vMakeMicDiagPSA(): Mic 1 values changed and updating to diaglog"));
            bValuesChanged = TRUE;
            }
    }
    }

    if (MicDataList[2].MicValid) //Is the microphone 2 configured?
    {
        MicNr = 2;
        tsMicDataElem = m_pMicrophone->tenGetMicData(MicNr);
        //Get microphone State and Current and Detection from microphone component (class)

    if(tsMicDataElem->MicState != m_tMicState[1])
    {
      m_tMicState[1] = tsMicDataElem->MicState;
      ETG_TRACE_USR2(("vMakeMicDiagPSA(): Mic 2 values changed"));
    }
    else
    {

        if((tsMicDataElem->MicState != MicDataList[MicNr].MicState) ||
           (tsMicDataElem->MicCurrent != MicDataList[MicNr].MicCurrent) ||
           (tsMicDataElem->MicDetection != MicDataList[MicNr].MicDetection))

            {
            MicDataList[MicNr].MicState = tsMicDataElem->MicState;
            MicDataList[MicNr].MicCurrent = tsMicDataElem->MicCurrent;
            MicDataList[MicNr].MicDetection = tsMicDataElem->MicDetection;

            ETG_TRACE_USR2(("vMakeMicDiagPSA(): changed values: MicrophoneCurrent = %d MicrophoneConnectionStatus = %d MicrophoneDetection = %d"
                            , MicDataList[MicNr].MicCurrent, MicDataList[MicNr].MicState, MicDataList[MicNr].MicDetection));

            ETG_TRACE_USR2(("vMakeMicDiagPSA(): Mic 2 values changed and updating to diaglog"));
            bValuesChanged = TRUE;
      }
        }
    }

    if (bValuesChanged)
    {
        /************************************************************
          DIAGLIB: Connect to the masc_fi interface (update property)
         ***********************************************************/
        if(m_poAudioService)
        {
            //Sent the new mic states (masc_fi object) to the clients
            ETG_TRACE_USR2(("vMakeMicDiagPSA(): updateClients call"));
            (tVoid)m_poAudioService->updateClients(MIDW_MASCFFI_C_U16_MICROPHONECONNECTIONSTATUS);
        }

        /***********************************************************
          DIAGLOG DTCs: Connect to the diaglog interface
         **********************************************************/
        if(m_pDiaglogClientHandler_Microphone)
        {
            if (MicDataList[1].MicValid) //Is the microphone 1 configured?
                m_pDiaglogClientHandler_Microphone->vAdaptDiaglogMicStatus(1); //Adapt and save Mic1 Data to diaglog object for sent

            if (MicDataList[2].MicValid) //Is the microphone 2 configured?
                m_pDiaglogClientHandler_Microphone->vAdaptDiaglogMicStatus(2); //Adapt and save Mic2 Data to diaglog object for sent

            // Sent the new mic TestResult (DTCs) to diaglog.
            ETG_TRACE_USR2(("vMakeMicDiagPSA(): send the new mic TestResults (DTCs)to the diaglog"));
            m_pDiaglogClientHandler_Microphone->vSendTestResultToDiaglog();
        }


    }else //if (bValuesChanged)
    {
        ETG_TRACE_USR2(("vMakeMicDiagPSA(): Mic values are no changed"));
    }

}

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::vMakeMicDiagAIVI
 *
 * DESCRIPTION:   makes mic diagnosis
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************/
 tVoid fc_audiomanager_tclMicrophone_Diagnosis::vMakeMicDiagAIVI()
 {
     tBool bValuesChanged = FALSE;
     tU8 MicNr;
     tsMicData *tsMicDataElem;

     if (m_pMicrophone == NULL)
     {
         ETG_TRACE_FATAL(( "vMakeMicDiagAIVI: Pointer is NULL" ));
         return;
     }


     if (MicDataList[1].MicValid) //Is the microphone 1 configured?
     {
         MicNr = 1;
         tsMicDataElem = m_pMicrophone->tenGetMicData(MicNr);

         //Get microphone State and Current and Detection from microphone component (class)

         if (tsMicDataElem->MicState != MicDataList[MicNr].MicState)
         {//this part will be entered at the first time because MicState differs at the first time from MicDataList.MicState (initially undefined)
             MicDataList[MicNr].MicState = tsMicDataElem->MicState;

             ETG_TRACE_USR2(("vMakeMicDiagAIVI(): Mic 1 value changed"));
             ETG_TRACE_USR2(("vMakeMicDiagAIVI(): changed value: MicrophoneConnectionStatus = %d", MicDataList[MicNr].MicState));

             bValuesChanged = TRUE;
         }

     }

     if (MicDataList[2].MicValid) //Is the microphone 2 configured?
     {
         MicNr = 2;
         tsMicDataElem = m_pMicrophone->tenGetMicDataAIVI(MicNr);
         //Get microphone State and Current and Detection from microphone component (class)

         if(tsMicDataElem->MicState != MicDataList[MicNr].MicState)
         {//this part will be entered at the first time because MicState differs at the first time from MicDataList.MicState (initially undefined)
             MicDataList[MicNr].MicState = tsMicDataElem->MicState;

             ETG_TRACE_USR2(("vMakeMicDiagAIVI(): Mic 2 value changed"));
             ETG_TRACE_USR2(("vMakeMicDiagAIVI(): changed value: MicrophoneConnectionStatus = %d", MicDataList[MicNr].MicState));

             bValuesChanged = TRUE;
         }
     }
/*
     if (MicDataList[3].MicValid) //Is the ANC microphone 1 configured? (on the position 3 of the list is the ANC microphone)
     {
         MicNr = 3;
         MicState = m_pMicrophone->tenGetMicDataAIVI(MicNr);
         //Get microphone State and Current and Detection from microphone component (class)

         if(tsMicDataElem->MicState != MicDataList[MicNr].MicState)
         {//this part will be entered at the first time because MicState differs at the first time from MicDataList.MicState (initially undefined)
             MicDataList[MicNr].MicState = tsMicDataElem->MicState;

             ETG_TRACE_USR2(("vMakeMicDiagAIVI(): ANC Mic value changed"));
             ETG_TRACE_USR2(("vMakeMicDiagAIVI(): changed value: MicrophoneConnectionStatus = %d", MicDataList[MicNr].MicState));

             bValuesChanged = TRUE;
         }
     }
*/


     if (bValuesChanged)
     {
         /************************************************************
           DIAGLIB: Connect to the masc_fi interface (update property)
          ***********************************************************/
         if(m_poAudioService)
         {
             //Sent the new mic states (masc_fi object) to the clients
             ETG_TRACE_USR2(("vMakeMicDiagAIVI(): updateClients call"));
             (tVoid)m_poAudioService->updateClients(MIDW_MASCFFI_C_U16_MICROPHONECONNECTIONSTATUS);
         }

         /***********************************************************
           DIAGLOG DTCs: Connect to the diaglog interface
          **********************************************************/

         if(m_pDiaglogClientHandler_Microphone)
         {
             if (MicDataList[1].MicValid) //Is the microphone 1 configured?
                 m_pDiaglogClientHandler_Microphone->vAdaptDiaglogMicStatus(1); //Adapt and save Mic1 Data to diaglog object for sent

             if (MicDataList[2].MicValid) //Is the microphone 2 configured?
                 m_pDiaglogClientHandler_Microphone->vAdaptDiaglogMicStatus(2); //Adapt and save Mic2 Data to diaglog object for sent
/*
             if (MicDataList[3].MicValid) //Is the microphone 3 configured?
                 m_pDiaglogClientHandler_Microphone->vAdaptDiaglogMicStatus(3); //Adapt and save Mic3 (ANC Microphone) Data to diaglog object for sent
*/
             // Sent the new mic TestResult (DTCs) to diaglog.
             ETG_TRACE_USR2(("vMakeMicDiagAIVI(): send the new mic TestResults(DTCs) to the diaglog"));
             m_pDiaglogClientHandler_Microphone->vSendTestResultToDiaglog();
         }

     }else //if (bValuesChanged)
     {
         ETG_TRACE_USR2(("vMakeMicDiagAIVI(): Mic values are no changed"));
     }


 }


/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::vMakeMicDiagCAF
 *
 * DESCRIPTION:   makes mic diagnosis
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************/
tVoid fc_audiomanager_tclMicrophone_Diagnosis::vMakeMicDiagCAF()
{
    tsMicData *tsMicDataElem;
    tU8 MicNr = 1;


    if (m_pMicrophone == NULL)
    {
        ETG_TRACE_FATAL(( "vMakeMicDiagCAF: Pointer is NULL" ));
        return;
    }

    if (MicDataList[1].MicValid) //Is the microphone 1 configured?
    {
        tsMicDataElem = m_pMicrophone->tenGetMicData(MicNr);
        //Get microphone State and Current and Detection from microphone component (class)


        if((tsMicDataElem->MicState != MicDataList[MicNr].MicState) ||
          (tsMicDataElem->MicCurrent != MicDataList[MicNr].MicCurrent) ||
          (tsMicDataElem->MicDetection != MicDataList[MicNr].MicDetection))

       {//this part will be entered at the first time because MicState differs at the first time from MicDataList.MicState (initially undefined)
           MicDataList[MicNr].MicState = tsMicDataElem->MicState;
           MicDataList[MicNr].MicCurrent = tsMicDataElem->MicCurrent;
           MicDataList[MicNr].MicDetection = tsMicDataElem->MicDetection;

            ETG_TRACE_USR2(("vMakeMicDiagCAF(): changed values: MicrophoneCurrent = %d MicrophoneConnectionStatus = %d MicrophoneDetection = %d"
                            , MicDataList[MicNr].MicCurrent, MicDataList[MicNr].MicState, MicDataList[MicNr].MicDetection));

            /************************************************************
              DIAGLIB: Connect to the masc_fi interface (update property)
             ***********************************************************/
            if(m_poAudioService)
            {
              ETG_TRACE_USR2(("vMakeMicDiagCAF(): updateClients call"));
              (tVoid)m_poAudioService->updateClients(MIDW_MASCFFI_C_U16_MICROPHONECONNECTIONSTATUS);
            }

            /***********************************************************
              DIAGLOG DTCs: Connect to the diaglog interface
             **********************************************************/
            if(m_pDiaglogClientHandler_Microphone)
            {
                //Adapt and save Mic Data to diaglog object for sent
                m_pDiaglogClientHandler_Microphone->vAdaptDiaglogMicStatus(1);

                // Sent the new Mic TestResult (DTCs) to diaglog.
                ETG_TRACE_USR2(("vMakeMicDiagCAF(): send the new mic TestResults (DTCs)to the diaglog"));
                m_pDiaglogClientHandler_Microphone->vSendTestResultToDiaglog();
            }

        }else
        {
            ETG_TRACE_USR2(("vMakeMicDiagCAF(): Mic values are no changed"));
        }
    }
}



/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::vMakeMicDiagJAC
 *
 * DESCRIPTION:   makes mic diagnosis
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************/
tVoid fc_audiomanager_tclMicrophone_Diagnosis::vMakeMicDiagJAC()
{

    //tsMicData *tsMicDataElem;
    tU8 MicNr = 1;
    ETG_TRACE_USR4((" MicNr: %d", MicNr));

    if (m_pMicrophone == NULL)
    {
        ETG_TRACE_FATAL(( "vMakeMicDiagJAC: Pointer is NULL" ));
        return;
    }

    if (MicDataList[1].MicValid) //Is the microphone 1 configured?
    {
        //tsMicDataElem = m_pMicrophone->tenGetMicData(MicNr);
        //Get microphone State and Current and Detection from microphone component (class)

        //What shall happens with the data?
        //Diaglib or Diaglog interface?
    }
}


/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::vGetMicConfig()
 *
 * DESCRIPTION:   get microphone configuration
 *
 * PARAMETERS:    tBool *mic1_config, tBool *mic2_config
 *
 * RETURN TYPE:
 *
 *******************************************************************************/
tVoid fc_audiomanager_tclMicrophone_Diagnosis::vGetMicConfig()
{

#if defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SUZUKI_LINUX
    MicDataList[1].MicValid = TRUE;

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX
    //Read KDS configuration for microphone (datapool)

 #if defined  DP_U16_KDSADR_TEL_FCT_AUDIO
    dp_tclKdsTEL_Fct_AUDIO tclDpKdsAudio;
    tU8 u8ReturnStatus;
    tU8 u8MicNr;

    u8ReturnStatus = tclDpKdsAudio.u8GetAUDIONumberOfMicrophones(u8MicNr);
    if (u8ReturnStatus != DP_U8_ELEM_STATUS_VALID && u8ReturnStatus != DP_U8_ELEM_STATUS_INITVAL)
    {
       ETG_TRACE_ERR(("vGetMicConfig(): Invalid Return status at reading Audio Number of microphones from KDS, Return status error=%i", u8ReturnStatus));
       MicDataList[1].MicValid = TRUE; //only one microphone
    }
    else
    {
        ETG_TRACE_USR2(("vGetMicConfig(): Number of microphones=%d", u8MicNr));
        if ( u8MicNr == 0) { //1 Microphone

            ETG_TRACE_USR2(("vGetMicConfig(): KDS Microphone configuration value is 0 => 1 microphone"));
            MicDataList[1].MicValid = TRUE;

        }else if (u8MicNr == 1) { // 2 Microphones

            ETG_TRACE_USR2(("vGetMicConfig(): KDS Microphone configuration value is 1 => 2 microphones"));
            MicDataList[1].MicValid = TRUE;
            MicDataList[2].MicValid = TRUE;
        }else
            ETG_TRACE_USR2(("vGetMicConfig(): KDS Microphone configuration value is wrong = %d", u8MicNr));
    }
 //#else
    //MicDataList[1].MicValid = TRUE; //only one microphone
 #endif

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SMART)
    //No KDS Configuration is available
    MicDataList[1].MicValid = TRUE;  //Mics 1 and 2 have the same results => 1 calculation

  if (DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "Microphone", &m_u8MicVal, 1))
    {
    ETG_TRACE_USR2(("vGetMicConfig(): KDS Microphone configuration value is = %d", m_u8MicVal));
    if(m_u8MicVal == MIC_2_RENAULT)
      MicDataList[2].MicValid = TRUE;
    else if(m_u8MicVal==MIC_NONE)
         MicDataList[1].MicValid = FALSE;
       else{}
  }
    //MicDataList[3].MicValid = TRUE;  //ANC Mics 1, 2 and 3 have the same results => 1 Calculation

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_JAC_CII
    //MicDataList[1].MicValid = TRUE;
    //MicDataList[2].MicValid = TRUE;
#else
    MicDataList[1].MicValid = TRUE;
#endif

}


/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::ptsGetMicDataListElem()
 *
 * DESCRIPTION:
 *
 * PARAMETERS:
 *
 * RETURN TYPE:  pointer of ptsMicData for the giving MicNr
 *
 *******************************************************************************/
tsMicData* fc_audiomanager_tclMicrophone_Diagnosis::ptsGetMicDataListElem(tU8 MicNr)
{
    return &(MicDataList[MicNr]);
}



/* ******************* Masc_fi Functions ***************************************/

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::bSendMicrophoneConnectionStatus
 *
 * DESCRIPTION:   Helper function to set property parameter for microphone diagnosis.
 *
 * PARAMETERS:    roOutMsg:      (O)   out message
 *                poInMsg:       (->I) in message
 *
 * RETURN TYPE:   TRUE: success
 *                FALSE: failed
 *
 *******************************************************************************/
tBool fc_audiomanager_tclMicrophone_Diagnosis::bSendMicrophoneConnectionStatus(amt_tclServiceData& roOutMsg, amt_tclServiceData* poInMsg)
{
    tBool m_bRetVal = FALSE;
    (tVoid)poInMsg;
    //(tVoid)roOutMsg;

    ETG_TRACE_USR4(("bSendMicrophoneConnectionStatus (NEW)(): Entered"));

    if (MicDataList[1].MicValid) //Is the microphone 1 configured?
        vAdaptMascfiMicStatus(1); //Adapt and save Mic1 Data to masc_fi object for sent


    if (MicDataList[2].MicValid) //Is the microphone 2 configured?
        vAdaptMascfiMicStatus(2); //Adapt and save Mic2 Data to masc_fi object for sent

    if (bExistTestDataToSend)
    {
       /* Sent the masc_fi microphone connection status parameter to the diaglib */
       fi_tclVisitorMessage oCCaMsg(oFiMicrophoneConnectionStatusObject);
       m_bRetVal = oCCaMsg.bHandOver(&roOutMsg);

       // Always destroy the FI data object before leaving its creation scope
       oFiMicrophoneConnectionStatusObject.vDestroy();

       ETG_TRACE_USR4(("bSendMicrophoneConnectionStatus (NEW)(): mascfi message is sent"));
       bExistTestDataToSend = FALSE;
    }

    return m_bRetVal;
}



/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::vAdaptMascfiMicStatus
 *
 * DESCRIPTION:   Adapt the MicData (MicState, Micdatection, MicCurrect) to the
 *                masc_fi Mic data midw_fi_tcl_T_MicrophoneConnectionStatus and save it.
 *
 *
 *******************************************************************************/
tVoid fc_audiomanager_tclMicrophone_Diagnosis::vAdaptMascfiMicStatus(tU8 MicNr)
{
    midw_fi_tcl_MicrophoneConnectionStatus oMicrophoneConnectionStatus;

   // Set the parameters of the FI data element

    oMicrophoneConnectionStatus.e8MicrophoneNumber.enType = (midw_fi_tcl_e8_MicrophoneNumber::tenType)MicNr; //midw_fi_tcl_e8_MicrophoneNumber::FI_EN_MIC_1;
    oMicrophoneConnectionStatus.u16MicrophoneCurrent = (tU16)(MicDataList[MicNr].MicCurrent); //gen4 [-wcoversion] compiler warning fix

    switch (MicDataList[MicNr].MicState)
    {
        case MIC_STATE_NORMAL:
            oMicrophoneConnectionStatus.e8MicrophoneConnectionStatus.enType = midw_fi_tcl_e8_MicrophoneConnectionStatus::FI_EN_FC_AUDIO_DIAG_MIC_CONNECTION_OK;
            ETG_TRACE_USR2(("vAdaptMascfiMicStatus(): ConnectionStatus for MIC %d is NORMAL", MicNr));
            break;
        case MIC_STATE_OPEN:
            oMicrophoneConnectionStatus.e8MicrophoneConnectionStatus.enType = midw_fi_tcl_e8_MicrophoneConnectionStatus::FI_EN_FC_AUDIO_DIAG_MIC_CONNECTION_OPEN;
            ETG_TRACE_USR2(("vAdaptMascfiMicStatus(): ConnectionStatus for MIC %d is OPEN", MicNr));
            break;
        case MIC_STATE_SHORT_TO_BATTERY:
            oMicrophoneConnectionStatus.e8MicrophoneConnectionStatus.enType = midw_fi_tcl_e8_MicrophoneConnectionStatus::FI_EN_FC_AUDIO_DIAG_MIC_CONNECTION_SHORT_TO_BATTERY;
            ETG_TRACE_USR2(("vAdaptMascfiMicStatus(): ConnectionStatus for MIC %d is SHORT TO BATTERY", MicNr));
            break;
        case MIC_STATE_SHORT_TO_GROUND:
            oMicrophoneConnectionStatus.e8MicrophoneConnectionStatus.enType = midw_fi_tcl_e8_MicrophoneConnectionStatus::FI_EN_FC_AUDIO_DIAG_MIC_CONNECTION_SHORT;
            ETG_TRACE_USR2(("vAdaptMascfiMicStatus(): ConnectionStatus for MIC %d is SHORT TO GROUND", MicNr));
            break;
        default:
            oMicrophoneConnectionStatus.e8MicrophoneConnectionStatus.enType = midw_fi_tcl_e8_MicrophoneConnectionStatus::FI_EN_FC_AUDIO_DIAG_MIC_CONNECTION_UNDEFINED;
            ETG_TRACE_USR2(("vAdaptMascfiMicStatus(): ConnectionStatus for MIC %d is UNDEFINED", MicNr));
            break;
    }

    switch (MicDataList[MicNr].MicDetection)
    {
        case MIC_DET_ABSENT:
            oMicrophoneConnectionStatus.e8MicrophoneDetection.enType = midw_fi_tcl_e8_MicrophoneDetection::FI_EN_MIC_DET_ABSENT;
            ETG_TRACE_USR2(("vAdaptMascfiMicStatus(): Detection for MIC %d is ABSENT", MicNr));
            break;
        case MIC_DET_PRESENT:
            oMicrophoneConnectionStatus.e8MicrophoneDetection.enType = midw_fi_tcl_e8_MicrophoneDetection::FI_EN_MIC_DET_PRESENT;
            ETG_TRACE_USR2(("vAdaptMascfiMicStatus(): Detection for MIC %d is PRESENT", MicNr));
            break;
        default:
            oMicrophoneConnectionStatus.e8MicrophoneDetection.enType = midw_fi_tcl_e8_MicrophoneDetection::FI_EN_MIC_DET_UNKNOWN;
            ETG_TRACE_USR2(("vAdaptMascfiMicStatus(): Detection for MIC %d is UNKNOWN", MicNr));
            break;
    }


    //Add the testData to the ConnectionStatusList
    oFiMicrophoneConnectionStatusObject.MicrophoneConnectionStatusList.MicrophoneConnectionStatusList.push_back(oMicrophoneConnectionStatus);

    // Always destroy the FI data object before leaving its creation scope
    oMicrophoneConnectionStatus.vDestroy();

    ETG_TRACE_USR2(("vAdaptMascfiMicStatus(): ConnectionStatus Values for MIC %d are stored", MicNr));

    bExistTestDataToSend = TRUE;

    return;
}

/************************************************ End of Masc_fi Functions *********************************************/



/************************************************************************************************************************/
/************************ Functions to communicate with them fc_audiomanager, to get the spm states  ********************/
/************************************************************************************************************************/


/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::vAddObserver
 *
 * DESCRIPTION:   Add observer for application state.
 *
 * PARAMETERS:    None
 *
 * RETURN TYPE:   None
 *
 *******************************************************************************/
tVoid fc_audiomanager_tclMicrophone_Diagnosis::vAddObserver(tVoid)
{
   InternalCommunicationAdapter * pPO = InternalCommunicationAdapter::getInstance();
   if(pPO)
   {
      pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_ApplicationStatus);
      pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_MuteState_HandleVoltage);
   }
}

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::MessageNotification
 *
 * DESCRIPTION:   Notification handler for application state.
 *
 * PARAMETERS:    None
 *
 * RETURN TYPE:   None
 *
 *******************************************************************************/
tVoid fc_audiomanager_tclMicrophone_Diagnosis::MessageNotification(PO_MessageConfig::enID MsgId)
{
   if(PO_MessageConfig::ID_ApplicationStatus == MsgId)
   {
      tenApplicationStates enApplicationState = AUD_APP_PWR_EVENT_UNDEFINED;

      InternalCommunicationAdapter * pPO = InternalCommunicationAdapter::getInstance();
      if(pPO)
      {
         const ID_ApplicationStatus * pMsg = pPO->POMessages->QueryMessage<ID_ApplicationStatus>(MsgId);
         if(pMsg)
         {
            enApplicationState = pMsg->value.m_State;
         }
      }

      if((AUD_APP_PWR_EVENT_APP_NORMAL == enApplicationState) || (AUD_APP_PWR_EVENT_APP_DIAGNOSIS == enApplicationState))
      {
         vEnterAppStateNormal();
      }
      else if((AUD_APP_PWR_EVENT_APP_PAUSE == enApplicationState) || (AUD_APP_PWR_EVENT_APP_OFF == enApplicationState))
      {
         vLeaveAppStateNormal();
      }
   }
   else if(PO_MessageConfig::ID_MuteState_HandleVoltage == MsgId)
   {
      tenVoltageState enVoltageState = EN_AUDIO_VOLTAGESTATE_CRITICAL_LOW_VOLTAGE_START;

      InternalCommunicationAdapter * pPO = InternalCommunicationAdapter::getInstance();
      if(pPO)
      {
         const ID_MuteState_HandleVoltage * pMsg = pPO->POMessages->QueryMessage<ID_MuteState_HandleVoltage>(MsgId);
         if(pMsg)
         {
            enVoltageState = pMsg->en_VoltageState;
         }
      }

      if((EN_AUDIO_VOLTAGESTATE_CRITICAL_LOW_VOLTAGE_END == enVoltageState) || (EN_AUDIO_VOLTAGESTATE_HIGH_VOLTAGE_END == enVoltageState) || (EN_AUDIO_VOLTAGESTATE_CRITICAL_HIGH_VOLTAGE_END == enVoltageState))
      {
         vEnterVoltageStateNormal();
      }
      else if((EN_AUDIO_VOLTAGESTATE_CRITICAL_LOW_VOLTAGE_START == enVoltageState) || (EN_AUDIO_VOLTAGESTATE_HIGH_VOLTAGE_START == enVoltageState) ||(EN_AUDIO_VOLTAGESTATE_CRITICAL_HIGH_VOLTAGE_START == enVoltageState))
      {
         vLeaveVoltageStateNormal();
      }
   }
}




/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::vEnterAppStateNormal
 *
 * DESCRIPTION:   Handle power state change to NORMAL.
 *
 * PARAMETERS:    None
 *
 * RETURN TYPE:   None
 *
 *******************************************************************************/
tVoid fc_audiomanager_tclMicrophone_Diagnosis::vEnterAppStateNormal(tVoid)
{
   ETG_TRACE_USR2(("vEnterAppStateNormal()"));

   // start timer if not started
   if(FALSE == m_bTimerStarted)
   {
      if(m_poMainApp)
      {
      if (m_u8MicVal == MIC_NONE)
      {
        ETG_TRACE_USR4(("vEnterStateNormal(): MIC Diagnosis is not performed"));
        return;
      }
         m_bTimerStarted = m_poMainApp->bStartTimer(FC_AUDIOMANAGER_TIMER_ID_MIC_DIAG, FC_AUDIOMANAGER_TIMER_MIC_DIAG_FIRST_TICK, FC_AUDIOMANAGER_TIMER_MIC_DIAG_TICK_INTERVAL);

         if(m_bTimerStarted)
         {
            ETG_TRACE_USR4(("vEnterStateNormal(): bStartTimer success"));
         }
         else
         {
            ETG_TRACE_ERR(("vEnterStateNormal(): bStartTimer failed"));
         }
      }
   }

   m_bNormalAppState = TRUE;
}

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::vLeaveAppStateNormal
 *
 * DESCRIPTION:   Handle power state change from NORMAL.
 *
 * PARAMETERS:    None
 *
 * RETURN TYPE:   None
 *
 *******************************************************************************/
tVoid fc_audiomanager_tclMicrophone_Diagnosis::vLeaveAppStateNormal(tVoid)
{
   ETG_TRACE_USR2(("vLeaveAppStateNormal()"));
   if(m_poMainApp)
   {
       if(m_poMainApp->bStopTimer(FC_AUDIOMANAGER_TIMER_ID_MIC_DIAG))
    {
          ETG_TRACE_USR4(("vLeaveAppStateNormal(): bStopTimer success"));
          m_bTimerStarted = FALSE;
    }
    else
    {
        ETG_TRACE_ERR(("vLeaveAppStateNormal(): bStopTimer failed"));
    }
   }
   m_bNormalAppState = FALSE;
}

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::vEnterVoltageStateNormal
 *
 * DESCRIPTION:   Handle power state change to NORMAL.
 *
 * PARAMETERS:    None
 *
 * RETURN TYPE:   None
 *
 *******************************************************************************/
tVoid fc_audiomanager_tclMicrophone_Diagnosis::vEnterVoltageStateNormal(tVoid)
{
   ETG_TRACE_USR2(("vEnterVoltageStateNormal()"));

   m_bNormalVoltageState = TRUE;
}

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone_Diagnosis::vLeaveVoltageStateNormal
 *
 * DESCRIPTION:   Handle power state change from NORMAL.
 *
 * PARAMETERS:    None
 *
 * RETURN TYPE:   None
 *
 *******************************************************************************/
tVoid fc_audiomanager_tclMicrophone_Diagnosis::vLeaveVoltageStateNormal(tVoid)
{
   ETG_TRACE_USR2(("vLeaveVoltageStateNormal()"));

   m_bNormalVoltageState = FALSE;
}



