/*******************************************************************************
* FILE:          fc_audiomanager_Microphone.cpp
* PROJECT:
* SW-COMPONENT:  fc_audiomanager microphone diagnosis
*-------------------------------------------------------------------------------
*
* DESCRIPTION:   Microphone diagnosis.
*
*-------------------------------------------------------------------------------
*
* COPYRIGHT:    (c) 2015 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author                      | Modification
* 02.02.15  | SoftTec Gaidatzi-Lken      | Inital version
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/

#include "fc_audiomanager_Microphone.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#include "fc_audiomanager_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_MIC_DIAGNOSIS
#include "trcGenProj/Header/fc_audiomanager_Microphone.cpp.trc.h"
#endif


#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#include "dp_audio_if.h"
#include "dp_diagdebug_if.h"

#define DEFAULT_ADC_REFERENCE_VALUE       3300   //Default ADC reference voltage in mV. The individuell ADC reference value from each target
                                                 //is measured at the production diagnosis and is written at the datapool.
                                                 //But the individuell ADC reference value is just calculated at the ADC driver and in the
                                                 //reading ADC value just included => This Default ADC value has to be included at my calculation.
                                                 //Thomas Mehring and Holger Schwark have confirmed this on the 12.09.2016
/*

#define VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SUZUKI_LINUX
#define VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX
#define VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI
#define VARIANT_S_FTR_ENABLE_FEAT_AUDIO_JAC_CII
#define VARIANT_S_FTR_ENABLE_FEAT_AUDIO_G3G_LINUX //This define is old and not used here
#define VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX //GM is not used here, mic diagnosis is on streamrouter


*/

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone::fc_audiomanager_tclMicrophone
 *
 * DESCRIPTION:   Default Constructor
 *
 * PARAMETERS:    None
 *
 * RETURN TYPE:   None
 *
 *******************************************************************************/
fc_audiomanager_tclMicrophone::fc_audiomanager_tclMicrophone()
{
    tsMicDataElem.MicState = MIC_STATE_UNKNOWN;
    tsMicDataElem.MicCurrent = 0;
    tsMicDataElem.MicDetection = MIC_DET_UNKNOWN;
    tsMicDataElem.MicValid =false;//coverity fix for initialisation
}

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone::~fc_audiomanager_tclMicrophone
 *
 * DESCRIPTION:   Destructor
 *
 * PARAMETERS:    None
 *
 * RETURN TYPE:   None
 *
 *******************************************************************************/
fc_audiomanager_tclMicrophone::~fc_audiomanager_tclMicrophone()
{
}


/*=================================================================================
                         Public Functions
  =================================================================================*/

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone::tenGetMicStateCurrentDet
 *
 * DESCRIPTION:   Selects the correct function based upon the hardware
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************/
//lint -e715 u8MicNr is not used here but is a parameter for the reference functions
tsMicData *fc_audiomanager_tclMicrophone::tenGetMicData(tU8 u8MicNr)
{


#if defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SUZUKI_LINUX
   ETG_TRACE_USR4(("tenGetMicData(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SUZUKI_LINUX"));
   return tenGetMicDataSUZ(u8MicNr);

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX
   ETG_TRACE_USR4(("tenGetMicData(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX"));
   return tenGetMicDataPSA(u8MicNr);

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI  || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SMART) // HW is similar to AIVI
   ETG_TRACE_USR4(("tenGetMicData(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI/SMART"));
   return tenGetMicDataAIVI(u8MicNr);

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CAF
   ETG_TRACE_USR4(("tenGetMicData(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CAF"));
   return tenGetMicDataCAF(u8MicNr);

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CHERYM31T
   ETG_TRACE_USR4(("tenGetMicData(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CHERYM31T"));
   return tenGetMicDataCAF(u8MicNr);

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_JAC_CII
   ETG_TRACE_USR4(("tenGetMicData(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_JAC_CII"));
   return tenGetMicDataJAC(u8MicNr);
   
#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_INF4CV
//implementation not available
//inf4cv use a different MIC, MIC1_DIAG and MIC2_DIAG pins are not available in hardware.hence no diagnosis required in inf4c for mic

#else //Standard is suzuki?
   ETG_TRACE_USR4(("tenGetMicData(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SUZUKI_LINUX"));
   return tenGetMicDataSUZ(u8MicNr);
#endif

}

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone::tenGetMicDetection
 *
 * DESCRIPTION:   Selects the correct function based upon the hardware
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************/
tenMicDetection fc_audiomanager_tclMicrophone::tenGetMicDetection(tU8 u8MicNr)const
{


#if defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SUZUKI_LINUX
   ETG_TRACE_USR4(("tenGetMicDetection(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SUZUKI_LINUX"));
   return tenGetMicDetectionSUZ(u8MicNr);

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX
   ETG_TRACE_USR4(("tenGetMicDetection(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX"));
   return tenGetMicDetectionPSA(u8MicNr);

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI
   ETG_TRACE_USR4(("tenGetMicDetection(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI"));
   return MIC_DET_NOT_IMPLEMENTED;

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_INF4CV || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SMART)
   ETG_TRACE_USR4(("tenGetMicDetection(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_INF4CV"));
   return MIC_DET_NOT_IMPLEMENTED;

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CAF
   ETG_TRACE_USR4(("tenGetMicDetection(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CAF"));
   return MIC_DET_NOT_IMPLEMENTED;

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CHERYM31T
   ETG_TRACE_USR4(("tenGetMicDetection(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CHERYM31T"));
   return MIC_DET_NOT_IMPLEMENTED;

#elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_JAC_CII
   ETG_TRACE_USR4(("tenGetMicDetection(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_JAC_CII"));
   return tenGetMicDetectionJAC(u8MicNr);

#else //Standard is suzuki?
   ETG_TRACE_USR4(("tenGetMicDetection(): VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SUZUKI_LINUX"));
   return tenGetMicDetectionSUZ(u8MicNr);
#endif
   ETG_TRACE_USR4((" u8MicNr: %d", u8MicNr));


}
//lint +e715 u8MicNr

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone::s32GetMicVoltage
 *
 * DESCRIPTION:   Provides Voltage value of microphone
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************/
tS32 fc_audiomanager_tclMicrophone::s32GetMicVoltage(tU8 u8MicNr)const
{
    return getADCmVoltage(u8MicNr);
}

/*=================================================================================
                                      S U Z U K I
  =================================================================================*/

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone::s32GetMicCurrent
 *
 * DESCRIPTION:   Provides Current value of microphone only for Suzuki
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************
tS32 fc_audiomanager_tclMicrophone::s32GetMicCurrentSUZ(tU8 u8MicNr)const
{
    tS32 s32MicADCVoltage = s32GetMicVoltage(u8MicNr);
    tS32 MicCurrentuA = 0;
    tS32 MicCurrentmA10 = 0;

    if(s32MicADCVoltage > -1)
    {
     //check for errors

     if (s32MicADCVoltage > 2650 && s32MicADCVoltage < 3000)
     {
       ETG_TRACE_USR2(
           ("s32GetMicCurrent(): Short to Ground,  Microphone ADC Voltage = %dmV", s32MicADCVoltage));
       MicCurrentuA = 6000;
     }
     else if (s32MicADCVoltage > 2650 && s32MicADCVoltage < 3000)
     {
       ETG_TRACE_USR2(
           ("s32GetMicCurrent(): Mic Ok, Microphone ADC Voltage = %dmV", s32MicADCVoltage));
       //MicCurrentuA = (s32MicVoltageADCVoltage * 7818)/1000 - 15418;
       MicCurrentuA = (s32MicADCVoltage * 21360)/1000 - 43540;

       if(MicCurrentuA < 0)
       {
         MicCurrentuA = 0;
       }
     }

     tS32 MicCurrentmA = MicCurrentuA/1000;
     MicCurrentmA10 = MicCurrentuA/100;

     ETG_TRACE_USR2(("s32GetMicCurrent(): MicCurrentmA=%dmA; MicCurrentmA10=(%d/10)mA", MicCurrentmA, MicCurrentmA10));

  }
  return MicCurrentmA10;

}
*/


/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone::tenGetMicDataSUZ
 *
 * DESCRIPTION:   Provides State, Current and Detection values of microphone
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************/
tsMicData *fc_audiomanager_tclMicrophone::tenGetMicDataSUZ(tU8 u8MicNr)
{
  tS32 s32MicADCVoltage = s32GetMicVoltage(u8MicNr);
  tS32 MicCurrentuA = 0;

  tsMicDataElem.MicState = MIC_STATE_UNKNOWN;
  tsMicDataElem.MicCurrent = 0;
  tsMicDataElem.MicDetection = MIC_DET_UNKNOWN;


  //1279
  if(s32MicADCVoltage > -1)
  {
     //check for errors

     if (s32MicADCVoltage < 1900)
     {
       ETG_TRACE_USR2(
           ("tenGetMicStateCurrentDetSUZ(): MIC State = OPEN LOAD, Microphone ADC Voltage = %dmV", s32MicADCVoltage));
       MicCurrentuA = 0;
     tsMicDataElem.MicState = MIC_STATE_OPEN;

     }
     else if (s32MicADCVoltage > 3000)
     {
       ETG_TRACE_USR2(
           ("tenGetMicStateCurrentDetSUZ(): MIC State = SHORT TO BATTEY, Microphone ADC Voltage = %dmV", s32MicADCVoltage));
       MicCurrentuA = 0;
     tsMicDataElem.MicState = MIC_STATE_SHORT_TO_BATTERY;
     }
     else if (s32MicADCVoltage > 2650)
     {
       ETG_TRACE_USR2(
           ("tenGetMicStateCurrentDetSUZ(): MIC State = SHORT TO GROUND, Microphone ADC Voltage = %dmV", s32MicADCVoltage));
       MicCurrentuA = 6000;
     tsMicDataElem.MicState = MIC_STATE_SHORT_TO_GROUND;
     }
     else
     {
       ETG_TRACE_USR2(
           ("tenGetMicStateCurrentDetSUZ(): MIC State = MIC OK, Microphone ADC Voltage = %dmV", s32MicADCVoltage));
       //MicCurrentuA = (s32MicVoltageADCVoltage * 7818)/1000 - 15418; // ???

       MicCurrentuA = (s32MicADCVoltage * 21360)/1000 - 43540;  //For Suzuki
       if(MicCurrentuA < 0)
       {
         MicCurrentuA = 0;
       }
     tsMicDataElem.MicState = MIC_STATE_NORMAL;
     }
     tS32 MicCurrentmA = MicCurrentuA/1000;
     tS32 MicCurrentmA10 = MicCurrentuA/100;
     ETG_TRACE_USR2(("tenGetMicStateCurrentDetSUZ(): MicCurrentmA=%dmA; MicCurrentmA10=(%d/10)mA", MicCurrentmA, MicCurrentmA10));

   tsMicDataElem.MicCurrent = MicCurrentmA10;
     tsMicDataElem.MicDetection = tenGetMicDetectionSUZ(u8MicNr);

  }
  return &tsMicDataElem;
}


/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone::tenGetMicDetectionSUZ
 *
 * DESCRIPTION:   Provides the detection state of microphone
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************/
tenMicDetection fc_audiomanager_tclMicrophone::tenGetMicDetectionSUZ(tU8 u8MicNr)const
{
    (tVoid)u8MicNr; //lint warning
    tS8  s8MicDetectState = s8GetGPIOState(OSAL_EN_MIC_DETECT);
    tenMicDetection  MicDetection;


    if(s8MicDetectState > -1)
    {
       if(s8MicDetectState == FALSE)
       {
           MicDetection = MIC_DET_ABSENT;
       }
       else if(s8MicDetectState == TRUE)
       {
           MicDetection = MIC_DET_PRESENT;
       }
       else
       {
           ETG_TRACE_ERR(("tenGetMicDetectionSUZ(): strange value for a logic level: %d", s8MicDetectState));
           MicDetection = MIC_DET_UNKNOWN;
       }
    }
    else
    {
       ETG_TRACE_ERR(("vMakeMicDiagSUZ(): MicDetect fails"));
       MicDetection = MIC_DET_UNKNOWN;
    }

    return MicDetection;
}

/*=================================================================================
                                      P S A
  =================================================================================*/

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone::tenGetMicDataPSA
 *
 * DESCRIPTION:   Provides State, Current and Detection values of microphone
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************/
tsMicData *fc_audiomanager_tclMicrophone::tenGetMicDataPSA(tU8 u8MicNr)
{
  tS32 s32MicADCVoltage = s32GetMicVoltage(u8MicNr); //get ADCVoltage for microphone u8MicNr in mV
  tS32 MicCurrentmA = 0;

  tsMicDataElem.MicState = MIC_STATE_UNKNOWN;
  tsMicDataElem.MicCurrent = 0;
  tsMicDataElem.MicDetection = MIC_DET_UNKNOWN;


  if(s32MicADCVoltage > -1)
  {
     //check for errors

     if (s32MicADCVoltage <= 100) //SHORT_TO_GROUND
     {
       ETG_TRACE_USR2(
           ("tenGetMicStateCurrentDetPSA(): MIC State = SHORT TO GROUND, Microphone ADC Voltage = %dmV for Mic %d", s32MicADCVoltage, u8MicNr));
       MicCurrentmA = 6;
     tsMicDataElem.MicState = MIC_STATE_SHORT_TO_GROUND;
       tsMicDataElem.MicDetection = MIC_DET_ABSENT;

     }
     else if (s32MicADCVoltage > 3000)//SHORT_TO_BATTERY
     {
       ETG_TRACE_USR2(
           ("tenGetMicStateCurrentDetPSA(): MIC State = SHORT TO BATTERY, Microphone ADC Voltage = %dmV for Mic %d", s32MicADCVoltage, u8MicNr));
       MicCurrentmA = 0;
     tsMicDataElem.MicState = MIC_STATE_SHORT_TO_BATTERY;
       tsMicDataElem.MicDetection = MIC_DET_ABSENT;
     }
     else if (s32MicADCVoltage >= 2800)//OPEN
     {
       ETG_TRACE_USR2(
           ("tenGetMicStateCurrentDetPSA(): MIC State = OPEN LOAD, Microphone ADC Voltage = %dmV for Mic %d", s32MicADCVoltage, u8MicNr));
       MicCurrentmA = 0;
     tsMicDataElem.MicState = MIC_STATE_OPEN;
       tsMicDataElem.MicDetection = MIC_DET_ABSENT;
     }
     else if (s32MicADCVoltage > 1500)//NORMAL
     {
       ETG_TRACE_USR2(("tenGetMicStateCurrentDetPSA(): MIC State = MIC OK, Microphone ADC Voltage = %dmV for Mic %d", s32MicADCVoltage, u8MicNr));

       MicCurrentmA = (2857 - s32MicADCVoltage)*1000 / 121429; //For PSA

//coverity fix: MicCurrentmA varies from 0.4 to 11.175,never be less than zero.
//      if(MicCurrentmA < 0)
//       {
//        MicCurrentmA = 0;
//       }
     tsMicDataElem.MicState = MIC_STATE_NORMAL;
       tsMicDataElem.MicDetection = MIC_DET_PRESENT;
     }
     else //OVERCURRENT: It can not happen
     {
       MicCurrentmA = 0;
       tsMicDataElem.MicDetection = MIC_DET_UNKNOWN;
     }

     tS32 MicCurrentmA10 = MicCurrentmA*10;

     ETG_TRACE_USR2(("tenGetMicStateCurrentDetPSA(): MicCurrentmA = %dmA; MicCurrentmA10 = %dmA10 for Mic %d", MicCurrentmA, MicCurrentmA10, u8MicNr));

   tsMicDataElem.MicCurrent = MicCurrentmA10;
  }
  else
  {
      tsMicDataElem.MicCurrent = 0;
      tsMicDataElem.MicDetection = MIC_DET_UNKNOWN;
  }

  ETG_TRACE_USR2(("tenGetMicStateCurrentDetPSA(): MicDetect = %d for Mic %d", tsMicDataElem.MicDetection, u8MicNr));

  return &tsMicDataElem;
}


/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone::tenGetMicDetectionPSA
 *
 * DESCRIPTION:   Provides the detection state of microphone
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************/
tenMicDetection fc_audiomanager_tclMicrophone::tenGetMicDetectionPSA(tU8 u8MicNr)const
{
    (tVoid)u8MicNr; //Lint warning

    //GPIO MIC_DETECT not exist at PSA
    tenMicDetection MicDetection = MIC_DET_NOT_IMPLEMENTED;

    return MicDetection;
}



/*=================================================================================
                                      A - I V I
  =================================================================================*/

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone::s32GetMicCurrent
 *
 * DESCRIPTION:   Provides Current value of microphone only for Suzuki
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************/
tsMicData *fc_audiomanager_tclMicrophone::tenGetMicDataAIVI(tU8 u8MicNr)
{
    tS32 s32MicADCVoltage = s32GetMicVoltage(u8MicNr); //get ADCVoltage for microphone u8MicNr in mV

    tsMicDataElem.MicState = MIC_STATE_UNKNOWN;
    tsMicDataElem.MicCurrent = 0;
    tsMicDataElem.MicDetection = MIC_DET_UNKNOWN;


    if(s32MicADCVoltage > -1)
    {
       //check for errors

       if (s32MicADCVoltage <= 680) //SHORT_TO_GROUND AND OPEN
       {
         ETG_TRACE_USR2(
             ("s32GetMicStateAIVI(): MIC State = OPEN or SHORT TO GROUND, Microphone ADC Voltage = %dmV for Mic %d", s32MicADCVoltage, u8MicNr));
         //MicState = MIC_STATE_SHORT_TO_GROUND;
         tsMicDataElem.MicState = MIC_STATE_OPEN;

       }
       else if (s32MicADCVoltage > 2721)//SHORT_TO_BATTERY
       {
         ETG_TRACE_USR2(
             ("s32GetMicStateAIVI(): MIC State = SHORT TO BATTERY, Microphone ADC Voltage = %dmV for Mic %d", s32MicADCVoltage, u8MicNr));
         tsMicDataElem.MicState = MIC_STATE_SHORT_TO_BATTERY;
       }
/*
       else if (s32MicADCVoltage >= 2800)//OPEN
       {
         ETG_TRACE_USR2(
             ("s32GetMicStateAIVI(): MIC State = OPEN LOAD, Microphone ADC Voltage = %dmV for Mic %d", s32MicADCVoltage, u8MicNr));
         tsMicDataElem.MicState = MIC_STATE_OPEN;
       }
*/
       else //NORMAL
       {
         ETG_TRACE_USR2(("s32GetMicStateAIVI(): MIC State = MIC OK, Microphone ADC Voltage = %dmV for Mic %d", s32MicADCVoltage, u8MicNr));
         tsMicDataElem.MicState = MIC_STATE_NORMAL;
       }

    }

    return &tsMicDataElem;

}
/*=================================================================================
                                  C A F   F O R D
  =================================================================================*/

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone::tenGetMicDataCAF
 *
 * DESCRIPTION:   Provides State, Current and Detection values of microphone
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************/
tsMicData *fc_audiomanager_tclMicrophone::tenGetMicDataCAF(tU8 u8MicNr)
{
  tS32 s32MicADCVoltage = s32GetMicVoltage(u8MicNr); //get ADCVoltage for microphone u8MicNr in mV
  tS32 MicCurrentmA = 0;

  tsMicDataElem.MicState = MIC_STATE_UNKNOWN;
  tsMicDataElem.MicCurrent = 0;
  tsMicDataElem.MicDetection = MIC_DET_UNKNOWN;


  if(s32MicADCVoltage > -1)
  {
     //check for errors

     if (s32MicADCVoltage == 0) //SHORT_TO_GROUND
     {
       ETG_TRACE_USR2(
           ("tenGetMicStateCurrentDetCAF(): MIC State = SHORT TO GROUND, Microphone ADC Voltage = %dmV for Mic %d", s32MicADCVoltage, u8MicNr));
       MicCurrentmA = 6;
     tsMicDataElem.MicState = MIC_STATE_SHORT_TO_GROUND;
       tsMicDataElem.MicDetection = MIC_DET_ABSENT;

     }
     else if (s32MicADCVoltage >= 3214 && s32MicADCVoltage <= 4300) //For B sample (Old sample 3082-5096) //SHORT_TO_BATTERY
     {
       ETG_TRACE_USR2(
           ("tenGetMicStateCurrentDetCAF(): MIC State = SHORT TO BATTERY, Microphone ADC Voltage = %dmV for Mic %d", s32MicADCVoltage, u8MicNr));
       MicCurrentmA = 0;
     tsMicDataElem.MicState = MIC_STATE_SHORT_TO_BATTERY;
       tsMicDataElem.MicDetection = MIC_DET_ABSENT;
     }
     else if (s32MicADCVoltage >= 2882 && s32MicADCVoltage <= 3123) //For B sample (Old sample 2752-2948) //OPEN
     {
       ETG_TRACE_USR2(
           ("tenGetMicStateCurrentDetCAF(): MIC State = OPEN LOAD, Microphone ADC Voltage = %dmV for Mic %d", s32MicADCVoltage, u8MicNr));
       MicCurrentmA = 0;
     tsMicDataElem.MicState = MIC_STATE_OPEN;
       tsMicDataElem.MicDetection = MIC_DET_ABSENT;
     }
     else if (s32MicADCVoltage >= 2082 && s32MicADCVoltage <= 2480) //For B sample (Old sample 2404-2605) //NORMAL
     {
       ETG_TRACE_USR2(("tenGetMicStateCurrentDetCAF(): MIC State = MIC OK, Microphone ADC Voltage = %dmV for Mic %d", s32MicADCVoltage, u8MicNr));

/*   For A-Sample
       //MicCurrentmA = (8934-(s32MicADCVoltage/0,3125))/207; //For CAF:: and after calculation the next line is going out
       MicCurrentmA = (2791 - s32MicADCVoltage) / 64; //For CAF A Sample
*/
       //For B-Sample new resistor values (22.3.2016)
       //MicCurrentmA = (8221-(s32MicADCVoltage/0,35714))/378; //For CAF:: and after calculation the next line is going out
       MicCurrentmA = (2936 - s32MicADCVoltage) / 135; //For CAF B Sample

//coverity fix: MicCurrentmA varies from 6.32 to 3.37, never becomes <0
//       if(MicCurrentmA < 0)
//       {
//         MicCurrentmA = 0;
//       }
     tsMicDataElem.MicState = MIC_STATE_NORMAL;
       tsMicDataElem.MicDetection = MIC_DET_PRESENT;
     }else
     {
         MicCurrentmA = 0;
         tsMicDataElem.MicDetection = MIC_DET_UNKNOWN;
     }

     tS32 MicCurrentmA10 = MicCurrentmA*10;

     ETG_TRACE_USR2(("tenGetMicStateCurrentDetCAF(): MicCurrentmA = %dmA; MicCurrentmA10 = %dmA10 for Mic %d", MicCurrentmA, MicCurrentmA10, u8MicNr));

   tsMicDataElem.MicCurrent = MicCurrentmA10;
  }
  else
  {
      tsMicDataElem.MicCurrent = 0;
      tsMicDataElem.MicDetection = MIC_DET_UNKNOWN;
  }

  ETG_TRACE_USR2(("tenGetMicStateCurrentDetCAF(): MicDetect = %d for Mic %d", tsMicDataElem.MicDetection, u8MicNr));

  return &tsMicDataElem;
}


/*=================================================================================
                                      J A C
  =================================================================================*/

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone::tenGetMicDataJAC
 *
 * DESCRIPTION:   Provides State, Current and Detection values of microphone
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************/

tsMicData *fc_audiomanager_tclMicrophone::tenGetMicDataJAC(tU8 u8MicNr)
{
    (tVoid)u8MicNr; //Lint warning

    tsMicDataElem.MicState = MIC_STATE_NOT_IMPLEMENTED;
    tsMicDataElem.MicCurrent = 0;
    tsMicDataElem.MicDetection = MIC_DET_NOT_IMPLEMENTED;

/*
    tS32 s32MicADCVoltage = s32GetMicVoltage(u8MicNr);


    if(s32MicADCVoltage > -1)
    {
      //current calculation not possible in this project

       //check for errors
       if(s32MicADCVoltage < 100)
       {
         ETG_TRACE_USR2(("tenGetMicStateCurrentDetJAC(): MIC State = SHORT TO GROUND, Microphone ADC Voltage = %dmV", s32MicADCVoltage));
       }
       else if(s32MicADCVoltage > 3170)
       {
         ETG_TRACE_USR2(("tenGetMicStateCurrentDetJAC(): MIC State = SHORT TO BATTERY, Microphone ADC Voltage = %dmV", s32MicADCVoltage));
       }
       else
       {
         ETG_TRACE_USR2(("tenGetMicStateCurrentDetJAC(): MIC State = MIC OK or OPEN LOAD, Microphone ADC Voltage = %dmV", s32MicADCVoltage));
       }
    }
*/

    return &tsMicDataElem;

}

/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone::tenGetMicDetectionJAC
 *
 * DESCRIPTION:   Provides the detection state of microphone
 *
 * PARAMETERS:
 *
 * RETURN TYPE:
 *
 *******************************************************************************/

tenMicDetection fc_audiomanager_tclMicrophone::tenGetMicDetectionJAC(tU8 u8MicNr)const
{
    (tVoid)u8MicNr; //Lint warning

    tenMicDetection  MicDetection = MIC_DET_NOT_IMPLEMENTED;
    ETG_TRACE_USR2(("tenGetMicDetectionJAC(): not implemented"));

    return MicDetection;

}




/************************************************************************************************************************/
/* Funktionen um die Mic-Werte aus der entsprechenden Hardware zu lesen */
/************************************************************************************************************************/


/*******************************************************************************
 *
 * FUNCTION:      fc_audiomanager_tclMicrophone::getADCuVoltage
 *
 * DESCRIPTION:   reads ADC and compute the voltage
 *
 * PARAMETERS:    ADC channel
 *
 * RETURN TYPE:   voltage at ADC in ?V
 *
 *******************************************************************************/
tS32 fc_audiomanager_tclMicrophone::getADCmVoltage(tU8 u8MicNr)const
{
     OSAL_tIODescriptor hFd = 0;
     OSAL_tenAccess enAccess = OSAL_EN_READWRITE;
     tS32 s32Ret = 0;
     tU16 *tU16ADCdata = NULL ;
     tPS8 ps8ADCdata = NULL;
     tU32 NoOfBlocks  = 1;
     tU32 u32NoOfBlocks = 1;
     tU32 U32_minblksize =2;
     //ETG_TRACE_USR2(("getADCmVoltage(): open"));

     /* Channel Open */
     switch (u8MicNr) {
     case 1:
         hFd = OSAL_IOOpen((tCString)OSAL_C_STRING_DEVICE_ADC_MIC_DIAG, enAccess);
         break;
     case 2:
         hFd = OSAL_IOOpen((tCString)OSAL_C_STRING_DEVICE_ADC_MIC_DIAG_2, enAccess);
         break;
     case 3: //For AIVI project Mic 3 is the ANC MIC 1 and for this Mic is the device OSAL_C_STRING_DEVICE_ADC_MIC_DIAG_2 rensonsible
         hFd = OSAL_IOOpen((tCString)OSAL_C_STRING_DEVICE_ADC_MIC_DIAG_2, enAccess);
         break;
     default:
         ETG_TRACE_ERR(("getADCmVoltage(): unknown MicNr"));
         s32Ret = -1;
         return s32Ret;
     }


     if(OSAL_ERROR != hFd)
     {
        //ETG_TRACE_USR2(("getADCmVoltage(): allocate"));
        tU16ADCdata = (tU16 *)OSAL_pvMemoryAllocate((tU32)U32_minblksize* NoOfBlocks);
        if (tU16ADCdata != NULL)
        {
          //ETG_TRACE_USR2(("getADCmVoltage(): read"));
           ps8ADCdata = (tPS8)(tVoid *)tU16ADCdata;
           u32NoOfBlocks = (tU32)OSAL_s32IORead(hFd,(tPS8)ps8ADCdata, (tU32)U32_minblksize* NoOfBlocks);

           if((OSAL_ERROR == (tS32)u32NoOfBlocks) ||( ((tU32)U32_minblksize* NoOfBlocks)  !=  u32NoOfBlocks))
           {
             ETG_TRACE_ERR(("getADCmVoltage(): read fails"));
             s32Ret = -1;
           }
           /* Get ADC data */
           else
           {

             OSAL_trAdcConfiguration ADC_Resolution;
             ADC_Resolution.u8AdcResolution = 0;
             if(OSAL_ERROR == OSAL_s32IOControl(hFd,OSAL_C_S32_IOCTRL_ADC_GET_CONFIG,(intptr_t)(&ADC_Resolution)))
             {
               ETG_TRACE_ERR(("getADCmVoltage(): control fails"));
               s32Ret = -1;
             }
             else
             {
              tU16ADCdata = (tU16*)(tVoid*)ps8ADCdata;
              //ETG_TRACE_USR2(("getADCmVoltage(): *tU16ADCdata=%d; resolution=%d", *tU16ADCdata, ADC_Resolution.u8AdcResolution));
              tS32 ADC_Max = (1 << ADC_Resolution.u8AdcResolution)-1;//maximum value which can deliver ADC

              //Coverity warning: checking ADC_Max is zero
              if(ADC_Max!=0)
              {


              tS32 ADC_InputVoltageForMicCurrent = DEFAULT_ADC_REFERENCE_VALUE * (*tU16ADCdata)/ADC_Max;//Voltage at the input terminal of the ADC in mV
              s32Ret = ADC_InputVoltageForMicCurrent;
              ETG_TRACE_USR2(("getADCmVoltage(): ADC Max value = %04x; ADC Input Voltage for Microphone = %dmV", ADC_Max, ADC_InputVoltageForMicCurrent));
        }
             }
           }
           //ETG_TRACE_USR2(("getADCmVoltage(): close"));

           /* Channel Close */
           if(OSAL_ERROR  == OSAL_s32IOClose ( hFd ) )
           {
             //s32Ret = -1;
             ETG_TRACE_ERR(("getADCmVoltage(): OSAL_s32IOClose fails"));
           }

           OSAL_vMemoryFree(tU16ADCdata);
           tU16ADCdata = NULL;
           ps8ADCdata = NULL;
        }
        else
        {
          s32Ret = -1;
          ETG_TRACE_ERR(("getADCmVoltage(): OSAL_pvMemoryAllocate fails"));
      OSAL_s32IOClose ( hFd );//coverity Warn fix
        }
     }
     else
     {
       s32Ret = -1;
       ETG_TRACE_ERR(("getADCmVoltage(): OSAL_IOOpen fails"));
     }

     return s32Ret;

}

/*****************************************************************************
*
* FUNCTION:     s8GetGPIOState()
*
* PARAMETER:    OSAL_enGpioPins GPIO_ID
*
* RETURNVALUE:  tS8, "0" or "1" on success  or "-1" in case of error
*
* DESCRIPTION:  Read the state of a GPIO Pin
*
*****************************************************************************/
tS8 fc_audiomanager_tclMicrophone::s8GetGPIOState(OSAL_enGpioPins GPIO_ID) const
{
   OSAL_tIODescriptor hFd;
   tS8 s8Ret = -1;
   OSAL_tenAccess enAccess = OSAL_EN_READWRITE;
   OSAL_tGPIODevID DevID = (OSAL_tGPIODevID) GPIO_ID;
   OSAL_trGPIOData Data = {DevID, TRUE};

   /* Open the device */
   hFd = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO,enAccess);
   if ( OSAL_ERROR == hFd )
   {
      ETG_TRACE_ERR(("s8GetGPIOState(): OSAL_IOOpen fails"));
   }
   else
   {
    /* Get state of input pin */
      if (OSAL_ERROR == OSAL_s32IOControl(hFd,OSAL_C_32_IOCTRL_GPIO_IS_STATE_ACTIVE, (intptr_t) &Data))
      {
        ETG_TRACE_ERR(("s8GetGPIOState(): OSAL_C_32_IOCTRL_GPIO_IS_STATE_ACTIVE fails"));
      }
      else
      {
         /* success */
         ETG_TRACE_USR2(("s8GetGPIOState(): State: %d", Data.unData.bState));
     s8Ret = (tS8)Data.unData.bState;
      }

      /* Close the device */
      if(OSAL_ERROR  == OSAL_s32IOClose ( hFd ) )
      {
        ETG_TRACE_ERR(("s8GetGPIOState(): OSAL_s32IOClose fails"));
      }
   }

   return s8Ret;
}


