/*
 * beep_file_reader.cpp
 *
 *  Created on: Aug 26, 2016
 *      Author: rjk2kor
 */

#include "beep_player_alsahw.h"
#include "beep_file_reader.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#include "../../fc_audiomanager_trace.h"
#include "../../fc_audiomanager_trace_macros.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_BEEP
#include "trcGenProj/Header/beep_file_reader.cpp.trc.h"
#endif

/**
 * Constructor
 */
beep_file_reader::beep_file_reader(const char* file_path,bool bLoop, int recurrenceperiod)
:m_fptr(NULL)
 ,m_info()
,m_bLoop(bLoop)
,m_playable_samples(0)
,m_output_channels(BEEP_ALSA_DEV_OUTPUT_CHANNELS)//Default initialization
{
  if(file_path)
  {
    m_fptr = sf_open(file_path,SFM_READ,&m_info);
    if(m_fptr != NULL)
    {
      ETG_TRACE_USR3(("Beep_file_Reader::Opened file: %s",file_path));
      ETG_TRACE_USR3(("Channels: %d, format:%d, frames:%d, samplerate:%d",m_info.channels,m_info.format,(int)m_info.frames,m_info.samplerate));
      m_playable_samples = m_info.frames*recurrenceperiod/RECURRENCE_PERIOD_MAX;
      ETG_TRACE_USR3(("Playable samples Calculated as : %d",m_playable_samples));
      //Playable samples to be adjusted based on recurrence period later
      if(m_playable_samples > m_info.frames)
      {
        ETG_TRACE_FATAL(("There is some error in calculation... BUG HERE"));
        m_playable_samples = m_info.frames;
      }
    }
    else
    {
      ETG_TRACE_FATAL(("Error Opening File : %s",sf_error_number(sf_error(m_fptr)) ));
    }
  }
}
/**
 * Destructor
 */
beep_file_reader::~beep_file_reader()
{
  ETG_TRACE_USR4(("Destructor of Beep File Reader"));
  if(m_fptr)
  {
    ETG_TRACE_USR4(("Calling sf_close"));
    sf_close(m_fptr);
    m_fptr = NULL;
  }
}
/**
 * Helper function to check if the buffer is valid
 */
bool beep_file_reader::bIsValid()
{
  return (m_fptr != NULL);
}
/**
 * Helper function to set recurrence period
 */
bool beep_file_reader::bSetRecurrencePeriod(int recurrenceperiod)
{
  ETG_TRACE_USR3(("beep_file_reader::bSetRecurrencePeriod, new Recurrence Period : %d ",recurrenceperiod));
  if(recurrenceperiod)
  {
    m_playable_samples = m_info.frames*recurrenceperiod/RECURRENCE_PERIOD_MAX;
    //Playable samples to be adjusted based on recurrence period later
    if(m_playable_samples > m_info.frames)
    {
      ETG_TRACE_FATAL(("There is some error in calculation... BUG HERE"));
      m_playable_samples = m_info.frames;
    }
    ETG_TRACE_USR3(("Playable samples Calculated as : %d",m_playable_samples));
    return true;
  }
  return false;
}

/**
 * Helper to fill the buffer
 */
sf_count_t beep_file_reader::uFillBuffer(short int* bufferptr,sf_count_t frames)
{
  short int* buffer = NULL;

  if(m_output_channels != m_info.channels)
  {
    //Create a buffer to hold so many number of samples
    buffer = new short int[frames*m_output_channels];
    memset(buffer,0,frames*m_output_channels);
  }
  else
  {
    buffer = bufferptr;
  }

  sf_count_t written=0;
  if(buffer && frames && m_fptr)
  {
    sf_count_t readcount = 0;
    sf_count_t required_frames = frames;

    //Get current position
    sf_count_t current_position = sf_seek(m_fptr,0,SEEK_CUR);

    //Adjust position if required according the soft limit of playable samples and looping is enabled
    if((current_position >= m_playable_samples))
    {
      if(m_bLoop)
      {
        //This is a incorrect position, moving to the beginning of the file
        current_position = sf_seek(m_fptr,0,SEEK_SET);
      }
      else
      {
        current_position = m_playable_samples;
      }
    }
    ETG_TRACE_USR4(("Current Position : %d, loop:%d, Playable Samples:%d ",current_position,m_bLoop,(int)m_playable_samples));

    //Creating a variable to improve readability. Below variable will give remaining samples to the configured buffer end.
    sf_count_t remain_samples_to_end_limit = m_playable_samples - current_position;

    //See if we have the required frames from the current position to the soft limit of buffer end
    readcount = sf_readf_short( m_fptr,
                  &buffer[written*m_info.channels],
                  (required_frames <= remain_samples_to_end_limit) ? required_frames:remain_samples_to_end_limit
                  );
    if(readcount)
    {
      written += readcount;
      //A read operation has been performed, update the required frames variable
      required_frames =  required_frames - written;//In case of a failue to read, value of written is 0
    }
    else
    {
      ETG_TRACE_USR3(("Nothing to read"));
      required_frames = 0;
    }

    //If we didnt get all the frames needed, check if loop flag is set to true, then we can loop and fill the data
    if(m_bLoop && (required_frames))
    {
      //Now seek to the beginning of the file
      current_position = sf_seek(m_fptr,0,SEEK_SET);

      //Now we can read and write as if this is a ring buffer...
      while(required_frames)
      {
        if(required_frames > m_playable_samples)
        {
          readcount = sf_readf_short(m_fptr, &buffer[written*m_info.channels], m_playable_samples);
          if(readcount)
          {
            //We have reached the end of buffer... set the file position to beginning
            sf_seek(m_fptr,0,SEEK_SET);
          }
          else
          {
            ETG_TRACE_FATAL(("Read Failure 1"));
            required_frames = 0;
            break;
          }

        }
        else
        {
          readcount = sf_readf_short(m_fptr, &buffer[written*m_info.channels], required_frames);
          if(readcount == 0)
          {
            ETG_TRACE_FATAL(("Read Failure 2"));
            required_frames = 0;
            break;
          }
        }
        required_frames = required_frames - readcount;//Update remaing frames
        written = written + readcount;//Update written frames counter
      }
    }
  }

  if(frames != written)
  {
    ETG_TRACE_USR3(("Total frames Requested : %d, written to buffer :%d",frames,written));
  }
  else
  {
    //Do conversions required..

    //A stereo to mono conversion is required here
    if(m_output_channels < m_info.channels)
    {
      ETG_TRACE_USR4(("Conversion to Mono"));
      vInterleavedStereo_To_Mono(buffer,bufferptr,written);
    }
    //A Mono to stereo conversion is required
    else if (m_output_channels > m_info.channels)
    {
      ETG_TRACE_USR4(("Conversion to Stereo"));
      vMakeInterleavedStereo(buffer,buffer,bufferptr,written);
    }
  }
  if(m_output_channels != m_info.channels)
  {
    delete[] buffer;//Remove memory allocated
  }

  return written;
}
/**
 * Helper function to interleave
 */
void beep_file_reader::vMakeInterleavedStereo(const short int * in_L,     // mono input buffer (left channel)
                const short int * in_R,     // mono input buffer (right channel)
        short int * out,            // stereo output buffer
                const size_t num_samples)  // number of samples
{
  if(!(in_L && in_R && out))
  {
    return;
  }
    for (size_t i = 0; i < num_samples; ++i)
    {
        out[i * 2] = in_L[i];
        out[i * 2 + 1] = in_R[i];
    }
}
/**
 * Here the output buffer has to be half the size of non interleaved buffer
 */
void beep_file_reader::vInterleavedStereo_To_Mono(const short int * noninterleavedbuf,     // mono input buffer (left channel)
        short int * out,            // stereo output buffer
                const size_t num_samples)  // number of samples
{
  if(!(noninterleavedbuf && out))
  {
    return;
  }
    for (size_t i = 0; i < num_samples; ++i)
    {
      out[i] = (short int)((noninterleavedbuf[i*2]+ noninterleavedbuf[i*2+1])/2); //gen4 compiler warning fix
    }
}
/**
 * Helper function
 */
void beep_file_reader::vSetLoop(bool bEnable)
{
  m_bLoop = bEnable;
}
/**
 * Helper function to reset loop to false and set the softlimit of playable samples to end of file
 */
void beep_file_reader::vReset()
{
  m_bLoop = false;
  m_playable_samples = m_info.frames;
}
/**
 * Helper function to set ouput channels
 */
void beep_file_reader::vSetOutputChannels(int channels)
{
  m_output_channels = channels;
}
/**
 * Helper function to copy current settings to another reader
 */
void beep_file_reader::vCopySettingsTo(beep_file_reader* x)
{
  x->m_bLoop = m_bLoop;
  x->m_playable_samples = x->m_info.frames * m_playable_samples/m_info.frames;
  x->m_output_channels = m_output_channels;
  ETG_TRACE_USR3(("Copied settings : New Samples to Play : %d",x->m_playable_samples));
}
