/*******************************************************************************
*
* FILE:          Beep_file_Player.cpp
*
* SW-COMPONENT:  Beep file player application
*
* DESCRIPTION:   Beep file player class implementation
*
* AUTHOR:        pmh7kor
*
* COPYRIGHT:    (c) 2015 RBEI, Bangalore
*
*******************************************************************************/


/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#include "../../fc_audiomanager_trace.h"
#include "../../fc_audiomanager_trace_macros.h"
#include "fc_audiomanager_main.h"

#include "sndfile_backend_player.h"
#include "gst_backend_player.h"

#include "Beep_file_Player.h"

#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_BEEP
#include "trcGenProj/Header/Beep_file_Player.cpp.trc.h"
#endif

using namespace std;

/******************************************************************************/
/*                                                                            */
/* GLOBAL VARIABLES                                                           */
/*                                                                            */
/******************************************************************************/

/* Beep_file_Player()
*
* Brief  : This is the constructor for class Beep_file_Player.Here
* class variables are initialized
*
* Params : Beep_playback_status_IF* m_ptrController
* Return : None
*/
Beep_file_Player::Beep_file_Player(Beep_playback_status_IF* m_ptrController)
:m_poController(m_ptrController)
,m_supported_backend_list()
,m_active_backend(NULL)
{
  //Contructor of beep file player
}

/* ~Beep_file_Player()
*
* Brief            : This is the destructor for class Beep_file_Player.
* Here objects used in the class are appropriately freed.
*
* Input Parameters : None
* Return Value     : None
*/

Beep_file_Player::~Beep_file_Player()
{
  ETG_TRACE_USR3(("~Beep_file_Player()"));
  m_poController = NULL; m_active_backend = NULL;
}
/*
* tBool Beep_file_Player::bInitializePlayer()
*
* Brief            : Initializes the Player, by registering for TTFis loopback
*
* Input Parameters : None
* Return Value     : TRUE
*/
bool Beep_file_Player::bInitializePlayer()
{
  ETG_TRACE_USR3(("Beep_file_Player::bInitializePlayer"));
  //Initialize different backends here
  m_supported_backend_list.push_back(new sndfile_backend_player(m_poController));
  m_supported_backend_list.push_back(new gst_backend_player(m_poController));

  for(vector<beep_player_backend_IF*>::iterator it = m_supported_backend_list.begin();it != m_supported_backend_list.end();it++)
  {
    if(*it != NULL)
    {
      (*it)->bInitializePlayer();
    }
  }
  return TRUE;
}
/*
* tBool Beep_file_Player::bDeinitPlayer()
*
* Brief            : DeInitializes the Player
*
* Input Parameters : None
* Return Value     : returns TRUE if Deinit is done
*/
bool Beep_file_Player::bDeinitPlayer()
{
  ETG_TRACE_USR3(("Beep_file_Player::bDeinitPlayer"));
  for(vector<beep_player_backend_IF*>::iterator it = m_supported_backend_list.begin();it != m_supported_backend_list.end();it++)
  {
    if(*it != NULL)
    {
      (*it)->bStop(STOP_AFTER_PLAYING_BUFFER);
      (*it)->vTearDown();
      (*it)->bDeinitPlayer();
    }
  }
  return TRUE;
}

/*
* tVoid Beep_file_Player::vTearDown()
*
* Brief            : unlinks the gstreamer elements and free them.
*
* Input Parameters : None
* Return Value     : None
*/

void Beep_file_Player::vTearDown()
{
  ETG_TRACE_USR3(("Beep_file_Player::vTearDown"));
  if(m_active_backend)
  {
    m_active_backend->vTearDown();
    //Once teardown is complete, say that there is active backend
    m_active_backend = NULL;
  }
  else
  {
    ETG_TRACE_USR3(("Beep_file_Player::vTearDown() No Active backend exists"));
  }
}
/*
* tBool Beep_file_Player::bPrepare(const char *p_filePath, const char *p_deviceName,tBool repeatMode)
*
* Brief            :  Creates the buffer and a reader
*
* Input Parameters :
* p_filePath     the file to play
* p_deviceName      Alsa Device on which file to be played
* Return Value     : TRUE if success, else FALSE
*/
bool Beep_file_Player::bPrepare(const char *p_filePath, const char *p_deviceName, bool bRepeat, unsigned int recurrence)
{
  ETG_TRACE_USR3(("Beep_file_Player::bPrepare() Enter"));
  if(p_filePath && p_deviceName)
  {
    for(vector<beep_player_backend_IF*>::iterator it = m_supported_backend_list.begin();it != m_supported_backend_list.end();it++)
    {
      if(*it != NULL)
      {
        if((*it)->bPrepare(p_filePath,p_deviceName,bRepeat,recurrence))
        {
          m_active_backend = *it;
          ETG_TRACE_USR3(("Beep_file_Player::bPrepare() Delegated to supported backend"));
          return true;
        }
        else
        {
          (*it)->vTearDown();
        }
      }
    }
  }
  ETG_TRACE_ERR(("Beep_file_Player::bPrepare() No supported backend exists to play this file !!!"));
  return false;
}

/*
* tBool Beep_file_Player::bPlay()
*
* Brief            : Function to play the Beep File
*
* Input Parameters : None
* Return Value     : True if playing success, else FALSE if State cant be set.
*/

bool Beep_file_Player::bPlay()
{
  if(m_active_backend)
  {
    return m_active_backend->bPlay();
  }
  else
  {
    ETG_TRACE_USR3(("Beep_file_Player::vTearDown() No Active backend exists"));
  }
  return false;
}

/*
* tBool Beep_file_Player::bStop(beep_player_stop_mode mode)
*
* Brief            : Function to stop playing the Beep File if playing
*
* Input Parameters :
*      mode   : Mode to use for stopping.
* Return Value     : True if STOP success, else FALSE if State cant be set.
*/

bool Beep_file_Player::bStop(beep_player_stop_mode mode)
{
  ETG_TRACE_USR3(("Beep_file_Player::bStop() Enter, Mode:%d",mode));

  if(m_active_backend)
  {
    return m_active_backend->bStop(mode);
  }
  else
  {
    ETG_TRACE_USR3(("Beep_file_Player::bStop() No Active backend exists"));
  }
  return true;
}
/*
* tBool Beep_file_Player::bUpdateRecurrence(int recurrence)
*
* Brief            : Function to update the recurrence of the file currently playing
*
* Input Parameters : recurrence : Recurrence period
* Return Value     : True if success, else FALSE if recurrence period cannot be set.
*/
bool Beep_file_Player::bUpdateRecurrence(int recurrence)
{
  if(m_active_backend)
  {
    return m_active_backend->bUpdateRecurrence(recurrence);
  }
  else
  {
    ETG_TRACE_USR3(("Beep_file_Player::bUpdateRecurrence() No Active backend exists"));
  }
  return false;
}

/*
* tBool Beep_file_Player::bUpdateActiveFile(const char* new_active_file)
*
* Brief            : Function
*
* Input Parameters : None
* Return Value     : True if playing success, else FALSE if State cant be set.
*/
bool Beep_file_Player::bUpdateActiveFile(const char* new_active_file)
{
  if(m_active_backend && new_active_file)
  {
    return m_active_backend->bUpdateActiveFile(new_active_file);
  }
  else
  {
    ETG_TRACE_USR3(("Beep_file_Player::bUpdateActiveFile() No Active backend exists"));
  }
  return false;
}
