/*******************************************************************************
*
* FILE:          Beep_PrioHandler.cpp
*
* SW-COMPONENT:  Beep
*
* DESCRIPTION:   Beep Priority handling
*
* AUTHOR:        Vaijayanti Vadiraj
*
* COPYRIGHT:    (c) 2016 RBEI
*
*******************************************************************************/
/******************************************************************************/
/* INCLUDES                                                                   */
/******************************************************************************/


#include "fc_audiomanager_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_BEEP
#include "trcGenProj/Header/Beep_PrioHandler.cpp.trc.h"
#endif

#include "Beep_PrioHandler.h"

#include "Beep_defines.h"
//#include "Beep_Controller.h"
#include "BeepController_IF.h"
#include "BeepNotification_IF.h"
#include "beepprioconf.h"

#include <algorithm>

using namespace std;

Beep_PrioHandler* Beep_PrioHandler::Beep_PrioHandlerInstance = NULL;

/*******************************************************************************
*
* FUNCTION: constructor
*
* DESCRIPTION: builds the static configuration for priority table
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
Beep_PrioHandler::Beep_PrioHandler(BeepNotification_IF* pBeepNotification_IF,vector <BeepPrioConf*>& rfPrioConfTable)
  :
aud_BeepPrio_ubNextBeepID(0)
  ,aud_BeepPrio_ubState(AUD_BEEPPRIOSTATE_OFF)
  ,BeepPrioConfTable(rfPrioConfTable)
  ,lstBeepControlData()
  ,poBeepNotification(pBeepNotification_IF)
{
  Beep_PrioHandler::Beep_PrioHandlerInstance = this;
}

/*******************************************************************************
*
* FUNCTION: ClearRequest
*
* DESCRIPTION: this is signalled from the lower layer when current playing beep
*         is over. The queued beep requests will be started if any.
*
* PARAMETER: beepId.
*
* RETURNVALUE: None.
*
*******************************************************************************/
void Beep_PrioHandler::ClearRequest(tU8 beepId)
{
  BeepData objBeepData;
  tU8 notifyValue;

  vector<BeepData>::iterator itBeepData = lstBeepControlData.begin();
  ETG_TRACE_USR4(("Beep_PrioHandler::ClearRequest() - beepName :%d, aud_BeepPrio_ubState :%d", beepId, aud_BeepPrio_ubState));
  if(lstBeepControlData.empty())
  {
    return;
  }

  objBeepData = lstBeepControlData.front();

  notifyValue = BEEP_STOPPED;
  if (aud_BeepPrio_ubState == AUD_BEEPPRIOSTATE_TMPOFF)
  {
    notifyValue = BEEP_QUEUED;
  }
  objBeepData.poBeepNotification_IF->NotifyBeepState(objBeepData.ubBeepName, objBeepData.ubBeepID, notifyValue, objBeepData.u32BeepToken);
  if(itBeepData->ubBeepName == beepId)
  {
    ETG_TRACE_USR4(("Beep_PrioHandler::clearing Beep :%d",beepId));
    lstBeepControlData.erase(itBeepData);
    aud_BeepPrio_ubState = AUD_BEEPPRIOSTATE_OFF;
  }

  ETG_TRACE_USR4(("Beep_PrioHandler::ClearRequest - size of queue :%d", lstBeepControlData.size()));
  vPlayBeepIfQueueNotEmpty();

}

/*******************************************************************************
* FUNCTION: vPlayBeepIfQueueNotEmpty
*
* DESCRIPTION: To play beeps in queue it not empty
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
void Beep_PrioHandler::vPlayBeepIfQueueNotEmpty()
{
  if(lstBeepControlData.empty() == false)
  {
    BeepData objBeepData = lstBeepControlData.front();
    if(objBeepData.poBeepNotification_IF != NULL)
    {
      ETG_TRACE_USR4(("Beep_PrioHandler::vPlayBeepIfQueueNotEmpty() Next beep playing , aud_BeepPrio_ubState :%d, BeepName :%d", aud_BeepPrio_ubState, objBeepData.ubBeepName));
      aud_BeepPrio_ubState = AUD_BEEPPRIOSTATE_ON;
      objBeepData.tenState = AUD_BEEPPRIOSTATE_ON;
      objBeepData.poBeepNotification_IF->NotifyBeepState(objBeepData.ubBeepName, objBeepData.ubBeepID, BEEP_START_IP,objBeepData.u32BeepToken);

      ETG_TRACE_USR4(("Beep_PrioHandler: Sending command to Beep controller to start Beep :%02x",
        ETG_ENUM(T_e8_Beep_InternalID,objBeepData.ubBeepName)));

      NORMAL_M_ASSERT(objBeepData.bGetExtendedData());//Get extended data using param interface
      poBeepNotification->m_poBeepController->StartBeep(objBeepData);
    }
  }
}

/*******************************************************************************
* FUNCTION: vIdleBeep
*
* DESCRIPTION: this is signalled from the beep/actuator to cancel a beep
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
void Beep_PrioHandler::vIdleBeep(tU8 ubStopType, tU8 ubBeepID)
{
  if(lstBeepControlData.size() == 0)
  {
    return;
  }
  ETG_TRACE_USR4(("Beep_PrioHandler::vIdleBeep() called : Queue size : %d,First Beep in Queue : %d, Beep to be Stopped :%d",lstBeepControlData.size(), lstBeepControlData.front().ubBeepID,ubBeepID));
  if(lstBeepControlData.front().ubBeepID == ubBeepID)
  {
    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(poBeepNotification);
    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(poBeepNotification->m_poBeepController);
    poBeepNotification->m_poBeepController->IdleBeep(ubStopType);
  }
  else
  {
    removeFromQueue(ubBeepID);
  }
}

/*******************************************************************************
* FUNCTION: u8SetRequest
*
* DESCRIPTION: this is signalled from the beep/actuator to start a beep
*         based on the state of the priority handler engine, a beep is
*         started/queued/removed/rejected/temporary stopped
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
tU8 Beep_PrioHandler::u8SetRequest(tU8 beepName,  BeepNotification_IF* beep, BeepParameter_IF* beep_param, tU8* beepId)
{
  BeepData objBeepData;

  tU8 ubRet = 0;
  ETG_TRACE_USR4(("Beep_PrioHandler::u8SetRequest() - BeepName :%d", beepName));
  ETG_TRACE_USR4(("Beep_PrioHandler::aud_BeepPrio_ubState :%d", aud_BeepPrio_ubState));

  PrintBeepsInTheQueue();
  if(!poBeepNotification->bIsBeepIdValid(beepName))
  {
    ETG_TRACE_USR4(("Beep_PrioHandler::u8SetRequest() - BeepName :%d is invalid", beepName));
    return BEEP_REJECTED;
  }
  if(lstBeepControlData.size() >= MAX_BEEP_QUEUESIZE)
  {
    return BEEP_REJECTED;
  }
  if(aud_BeepPrio_ubState == AUD_BEEPPRIOSTATE_OFF) // hadnling new beep request when no beep is active
  {
    if(bCheckDoNotPutInQueue(beepName) == true)
    {
      return BEEP_REJECTED;
    }
    aud_BeepPrio_ubState = AUD_BEEPPRIOSTATE_ON;
    objBeepData.ubBeepName = beepName;
    objBeepData.ubBeepID = this->aud_BeepPrio_ubNextBeepID;
    objBeepData.poBeepNotification_IF = beep;
    objBeepData.poBeepParameter_IF =  beep_param;
    NORMAL_M_ASSERT(objBeepData.bGetExtendedData());//Get extended data using param interface


    *beepId = this->aud_BeepPrio_ubNextBeepID;
    aud_BeepPrio_ubNextBeepID++;
    lstBeepControlData.insert(lstBeepControlData.begin(), objBeepData);
    ubRet = BEEP_START_IP;

    ETG_TRACE_USR4(("Beep_PrioHandler: Sending command to Beep controller to start Beep :%02x",
      ETG_ENUM(T_e8_Beep_InternalID,objBeepData.ubBeepName)));

    AUD_POINTER_CHECK_CALL(objBeepData.poBeepNotification_IF);
    //Notify that we are about to start
    objBeepData.poBeepNotification_IF->NotifyBeepState(objBeepData.ubBeepName, objBeepData.ubBeepID, BEEP_START_IP,objBeepData.u32BeepToken);

    //ETG_TRACE_USR1(("Beep_PrioHandler: Sending command to Beep controller to start Beep :%d", ETG_ENUM(midw_fi_tcl_e8_AudioBeepType::enType, objBeepData.ubBeepName)));
    poBeepNotification->m_poBeepController->StartBeep(objBeepData);
  }
  else // handling new beep when a beep is already running
  {
    *beepId = this->aud_BeepPrio_ubNextBeepID;
    aud_BeepPrio_ubNextBeepID++;
    bCheckForRemove(beepName);
    if(bCheckForUniqueReject(beepName) == true)
    {
      ETG_TRACE_USR4(("Beep_PrioHandler: Beep request rejected(unique) for Beep :%d", beepName));
      ubRet = BEEP_UNIQUE_REJECTED;
      return ubRet;
    }
    if(bCheckForReject(beepName) == true) // tempstop/reject, ImmediateStop/Reject
    {
      ETG_TRACE_USR4(("Beep_PrioHandler: Beep request rejected for Beep :%d", beepName));
      ubRet = BEEP_REJECTED;
    }
    else //tempstop/insert, ImmediateStop/insert
    {
      objBeepData.ubBeepName = beepName;
      objBeepData.ubBeepID = *beepId;
      objBeepData.poBeepNotification_IF = beep;
      objBeepData.poBeepParameter_IF =  beep_param;
      //*beepId = objBeepData.ubBeepID;  //stops testtone
      NORMAL_M_ASSERT(objBeepData.bGetExtendedData());//Get extended beep data to objBeepData

      //aud_BeepPrio_ubNextBeepID++;
      if(bCheckForInsertOrReplaceBeep(beepName,&objBeepData) == true)
        ubRet = BEEP_QUEUED;
    }
    if(bCheckForStopImmediately(beepName) == true)
    {
      ETG_TRACE_USR4(("Beep_PrioHandler: Beep request stopped ongoing beep and removed other related beeps"));
      ubRet = BEEP_STOPPED;
    }
    if(bcheckForTempStop(beepName) == true) //stopped beep should be in queue first before inserting a new beep, for Tempstop/Insert case
      ubRet = BEEP_QUEUED;

  }
  return ubRet;
}
/*******************************************************************************
* FUNCTION: bCheckForInsertOrReplaceBeep(tU8 beepName,BeepData* bd)
*
* DESCRIPTION: helper function to make sure queue size is not exceeding
*         upper limit
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/

bool Beep_PrioHandler::bCheckForInsertOrReplaceBeep(tU8 beepName,BeepData* bd)
{
  ETG_TRACE_USR4(("Beep_PrioHandler:bCheckForInsertOrReplaceBeep() : %d",beepName));
  vector<BeepData>::iterator itBeepData;
  if((BeepPrioConfTable[beepName]->test(BeepPrioConfTable[beepName]->bitsReplace, lstBeepControlData.begin()->ubBeepName)))
  {
    itBeepData = lstBeepControlData.begin()+1;
    //itBeepData = find(lstBeepControlData.begin()+1,lstBeepControlData.end(),objBeepData);
    //below insert is to for Replace functionality , if replace is set to true
    for(; itBeepData != lstBeepControlData.end(); itBeepData++)
    {
      if(itBeepData->ubBeepName == beepName){
        ETG_TRACE_USR4(("Beep_PrioHandler: Beep already present in Queue"));
        break;
      }
    }
    if(itBeepData == lstBeepControlData.end()){
      ETG_TRACE_USR4(("Beep_PrioHandler: inserting new Beep in Queue"));
      lstBeepControlData.insert(lstBeepControlData.begin()+1, *bd);
    }
    poBeepNotification->m_poBeepController->IdleBeep(BEEP_STOP_IMMEDIATE);
    return true;
  }
  else
  {
    vInsertWithPrio(beepName,bd);
    ETG_TRACE_USR4(("Beep_PrioHandler: Beep queued :%d", bd->ubBeepName));
    return true;
  }
  return false;
}
/*******************************************************************************
* FUNCTION: getSizeOfQueue
*
* DESCRIPTION: helper function to make sure queue size is not exceeding
*         upper limit
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
tU8 Beep_PrioHandler::getSizeOfQueue(void) const
{
  return (tU8)(this->lstBeepControlData.size());
}

bool Beep_PrioHandler::bCheckDoNotPutInQueue(tU8 newBeepName)
{
  if(BeepPrioConfTable.empty())
  {
    NORMAL_M_ASSERT_ALWAYS();
  }
  return BeepPrioConfTable[newBeepName]->doNotPutInQueue;
}
/*******************************************************************************
* FUNCTION: bCheckForReject
*
* DESCRIPTION: returns true if new beep has to be rejected because of another
*        beep that is either playing or in queue which does not allow
*        this beep to be put in queue
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
bool Beep_PrioHandler::bCheckForReject(tU8 newBeepName)
{
  vector<BeepData>::iterator itBeepData;
  for(itBeepData = lstBeepControlData.begin(); itBeepData != lstBeepControlData.end(); itBeepData++)
  {
    if((BeepPrioConfTable[newBeepName]->test(BeepPrioConfTable[newBeepName]->bitsRejectReason, itBeepData->ubBeepName)) == true)
    {
      ETG_TRACE_USR4(("Beep_PrioHandler:bCheckForReject()"));
      return true;
    }
  }
  if(newBeepName == midw_fi_tcl_e8_AudioBeepType::FI_EN_AUDIO_BEEP_WARN_NOT_DEF)
  {
    return true;
  }
  else
  {
    return false;
  }
}

/*******************************************************************************
* FUNCTION: bCheckForUniqueReject
*
* DESCRIPTION: returns true if new beep has to be rejected because of same
*        beep is either playing or in queue (exception is w.r.t click)
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
bool Beep_PrioHandler::bCheckForUniqueReject(tU8 newBeepName)
{
  vector<BeepData>::iterator itBeepData;
  for(itBeepData = lstBeepControlData.begin(); itBeepData != lstBeepControlData.end(); itBeepData++)
  {
    if((newBeepName == itBeepData->ubBeepName) && (BeepPrioConfTable[newBeepName]->ubUnique == 1))
    {
      ETG_TRACE_USR4(("Beep_PrioHandler:bCheckForUniqueReject()"));
      return true;
    }
  }
  return false;
}

/*******************************************************************************
* FUNCTION: bCheckForRemove
*
* DESCRIPTION: if new beep has more significance than compared to a
*        certain beep in the queue, then the beep in the queue is
*        removed from the queue. Currently playing beep is never
*        disturbed in this case.
*        This is determined by priority rules
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
void Beep_PrioHandler::bCheckForRemove(tU8 newBeepName)
{
  ETG_TRACE_USR4(("Beep_PrioHandler:bCheckForRemove()"));
  vector<BeepData>::iterator itBeepData = lstBeepControlData.begin();

  if(lstBeepControlData.size() == 1)
  {
    return;
  }

  /**
  * If any IPA beep is active, do nothing
  * Currently the beep IPA_NOSOUND has all the list of IPA beeps, so using it. No other special reason.
  */
  if(BeepPrioConfTable[EN_AUDIO_BEEP_IPA_NOSOUND]->test(BeepPrioConfTable[EN_AUDIO_BEEP_IPA_NOSOUND]->bitsRemove, itBeepData->ubBeepName))
    //if(((tU64)1<<itBeepData->ubBeepName)&(ANY_IPA_BEEP_TO_BE_FLUSHED))
  {
    //do nothing
  }
  else if((BeepPrioConfTable[newBeepName]->test(BeepPrioConfTable[newBeepName]->bitsRemove, itBeepData->ubBeepName)) || \
    (BeepPrioConfTable[newBeepName]->test(BeepPrioConfTable[newBeepName]->bitsReplace, itBeepData->ubBeepName)))
  {
    return;
  }


  itBeepData = lstBeepControlData.begin()+1;
  while(itBeepData != lstBeepControlData.end())
  {
    if((BeepPrioConfTable[newBeepName]->test(BeepPrioConfTable[newBeepName]->bitsRemove, itBeepData->ubBeepName)) || \
      (BeepPrioConfTable[newBeepName]->test(BeepPrioConfTable[newBeepName]->bitsReplace, itBeepData->ubBeepName)))
    {

      tU8 playingBeep = lstBeepControlData.begin()->ubBeepName;
      tU8 beepToBeRemoved = itBeepData->ubBeepName;
      tU8 sequence = BeepPrioConfTable[playingBeep]->enSequence;

      if(((sequence != EN_BEEP_SEQUENCE_MAX)&&(BeepPrioConfTable[playingBeep]->enSequence != BeepPrioConfTable[beepToBeRemoved]->enSequence)) ||\
        (sequence == EN_BEEP_SEQUENCE_MAX))
      {
        if(itBeepData == lstBeepControlData.end()-1)
        {
          lstBeepControlData.erase(itBeepData);
          break;
        }
        else
        {
          itBeepData = lstBeepControlData.erase(itBeepData);
          itBeepData--;
        }
      }
    }
    itBeepData++;
  };
}

/*******************************************************************************
* FUNCTION: bcheckForTempStop(tU8 beepName)
*
* DESCRIPTION: returns true if current running beep has to be temporarily stopped
*        else false
*
* PARAMETER:
*
* RETURNVALUE: bool.
*******************************************************************************/
bool Beep_PrioHandler::bcheckForTempStop(tU8 beepName)
{
  BeepData objBeepDataTemp = lstBeepControlData.front();
  vector<BeepData>::iterator itBeepData = lstBeepControlData.begin();
  ETG_TRACE_USR4(("Beep_PrioHandler:bcheckForTempStop() : %d",beepName));
  if(BeepPrioConfTable[beepName]->test(BeepPrioConfTable[beepName]->bitsTmpStop, lstBeepControlData.begin()->ubBeepName))
  {
    aud_BeepPrio_ubState = AUD_BEEPPRIOSTATE_TMPOFF;
    lstBeepControlData.begin()->tenState = AUD_BEEPPRIOSTATE_TMPOFF;
    for(; itBeepData != lstBeepControlData.end(); ++itBeepData)
    {
      if((itBeepData->ubBeepName == objBeepDataTemp.ubBeepName)&&(itBeepData != lstBeepControlData.begin())){
        lstBeepControlData.erase(itBeepData);
        ETG_TRACE_USR4(("Beep_PrioHandler: Temp Stop Beep already present in Queue, So, removing to insert at the end of Queue"));
        break;
      }
    }
    ETG_TRACE_USR4(("Beep_PrioHandler: inserting Temp stop Beep in Queue"));
    lstBeepControlData.push_back(objBeepDataTemp);

    ETG_TRACE_USR4(("Beep_PrioHandler: Beep temporarily stopped"));
    poBeepNotification->m_poBeepController->IdleBeep(BEEP_STOP_IMMEDIATE);
    return true;
  }
  return false;
}
/*******************************************************************************
* FUNCTION: bCheckForStopImmediately(tU8 newBeepName)
*
* DESCRIPTION: returns true if current running beep has to be stopped immediately
*        else false
*
* PARAMETER:
*
* RETURNVALUE: bool.
*******************************************************************************/
bool Beep_PrioHandler::bCheckForStopImmediately(tU8 newBeepName)
{
  ETG_TRACE_USR4(("Beep_PrioHandler:bCheckForStopImmediately()"));
  bool bRet = false;
  vector<BeepData>::iterator itBeepData = lstBeepControlData.begin();

  bCheckForRemove(newBeepName);
  if(BeepPrioConfTable[newBeepName]->test(BeepPrioConfTable[newBeepName]->bitsStopImmediately, itBeepData->ubBeepName))
  {
    poBeepNotification->m_poBeepController->IdleBeep(BEEP_STOP_IMMEDIATE);
    bRet = true;
  }
  return bRet;
}
/*******************************************************************************
* FUNCTION: vInsertWithPrio
*
* DESCRIPTION: New beep gets inserted in the queue and gets placed according
*        to its priority
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
void Beep_PrioHandler::vInsertWithPrio(tU8 newBeepName, BeepData* pBeepData)
{
  tBool blNewBeepInserted = false;
  vector<BeepData>::iterator itBeepData;
  if(BeepPrioConfTable[newBeepName]->doNotPutInQueue == true)
  {
    return;
  }

  if(lstBeepControlData.size() == 1)
  {
    lstBeepControlData.push_back(*pBeepData);
    PrintBeepsInTheQueue();
    return;
  }

  for(itBeepData = lstBeepControlData.begin()+1; itBeepData != lstBeepControlData.end(); itBeepData++)
  {
    if((BeepPrioConfTable[newBeepName]->ubPrio < BeepPrioConfTable[itBeepData->ubBeepName]->ubPrio) && \
      (BeepPrioConfTable[(itBeepData-1)->ubBeepName]->enNextBeep == EN_AUDIO_BEEP_INVALID))
    {
      if(BeepPrioConfTable[itBeepData->ubBeepName]->enSequence == EN_BEEP_SEQUENCE_MAX)
      {
        lstBeepControlData.insert(itBeepData,*pBeepData);
        blNewBeepInserted = true;
        break;
      }
      else if(BeepPrioConfTable[itBeepData->ubBeepName]->enSequence != BeepPrioConfTable[newBeepName]->enSequence)
      {
        lstBeepControlData.insert(itBeepData,*pBeepData);
        blNewBeepInserted = true;
        break;
      }
    }
  }
  if(blNewBeepInserted == false)
  {
    lstBeepControlData.push_back(*pBeepData);
    blNewBeepInserted = true;
  }
  PrintBeepsInTheQueue();

}

/*******************************************************************************
* FUNCTION: removeFromQueue
*
* DESCRIPTION: returns beep in particular position in queue
*        Mostly used during unit testing
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
void Beep_PrioHandler::removeFromQueue(tU8 beepId)
{
  vector<BeepData>::iterator itBeepData;
  vector<BeepData>::iterator itResultBeepData;
  for(itBeepData = lstBeepControlData.begin();itBeepData != lstBeepControlData.end(); itBeepData++)
  {
    if(itBeepData->ubBeepID == beepId)
    {
      ETG_TRACE_USR4(("Beep_PrioHandler::removeFromQueue():%d", beepId));
      itBeepData->poBeepNotification_IF->NotifyBeepState(itBeepData->ubBeepName, itBeepData->ubBeepID, (tU8)BEEP_STOPPED,itBeepData->u32BeepToken);
      itResultBeepData = lstBeepControlData.erase(itBeepData);
        break;

    }
  }
}

/*******************************************************************************
* FUNCTION: test
*
* DESCRIPTION: returns true if a particular bit is set in a bit mask
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
tBool BeepPrioConf::test(std::vector<tU8>& list, tU8 beepid)
{
  return(list.end() != find(list.begin(),list.end(),beepid));
}

/*******************************************************************************
* FUNCTION: BeepData Constructor
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
BeepData::BeepData()
{
  tenState = AUD_BEEPPRIOSTATE_OFF;
  ubBeepName = 0;
  poBeepNotification_IF = NULL;
  poBeepParameter_IF = NULL;
  ubBeepID = 0;


  memset(&m_beepparam,0,sizeof(BeepControlData::BeepPara));
  memset(&m_chimeparam,0,sizeof(BeepControlData::ChimePara));
  memset(&m_channeloffsets,0,sizeof(BeepControlData::ChVolOff));

  m_u8Source = 0x11;
  m_s16VolumeStep = 0;
  m_s16Gain = -480;
  m_bIsValid = false;
  u32BeepToken =0;
}
/*******************************************************************************
* FUNCTION: BeepData
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
bool BeepData::bGetExtendedData()
{
  if(poBeepParameter_IF != NULL)
  {
    poBeepParameter_IF->u8UpdateBeepMainData(ubBeepName);
    //Get beep source
    if(poBeepNotification_IF != NULL)
      poBeepNotification_IF->SetAdasNotificationPointer(ubBeepName);
    m_u8Source = poBeepParameter_IF->GetBeepSource(ubBeepName);

    //Get volume offsets
    poBeepParameter_IF->GetBeepVolumeOffsets(ubBeepName, &m_channeloffsets);
    //Get volume gain
    m_s16Gain = poBeepParameter_IF->GetBeepVolume(ubBeepName);
    //Get volume step
    m_s16VolumeStep = poBeepParameter_IF->GetIncrementalVolumeStep();
    //Get Signal tone parameters
    poBeepParameter_IF->GetSignalTonePara(ubBeepName, &m_beepparam);
    //Get Chime parameters
    poBeepParameter_IF->GetSignalChimePara(ubBeepName, &m_chimeparam);
    //Get Beep Token
    u32BeepToken = poBeepParameter_IF->GetRequestToken();

    if(poBeepNotification_IF != NULL)
    {
      //Get Path
      m_filepath = poBeepNotification_IF->GetBeepFilePath(ubBeepName);

      m_bIsValid = true;
    }

    return true;
  }
  return false;
}

/*******************************************************************************
* FUNCTION: getInstance
*
* DESCRIPTION: returns instance of prio handler
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
Beep_PrioHandler* Beep_PrioHandler::getInstance()
{
  return Beep_PrioHandler::Beep_PrioHandlerInstance;
}

/*******************************************************************************
* FUNCTION: vClearQueue
*
* DESCRIPTION: Helper function
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
void Beep_PrioHandler::vClearQueue(void)
{
  lstBeepControlData.clear();
  aud_BeepPrio_ubState= AUD_BEEPPRIOSTATE_OFF;
}

/*******************************************************************************
* FUNCTION: vInit
*
* DESCRIPTION: Helper function. (Does not get called on start up)..
*        Mostly used during unit testing
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
void Beep_PrioHandler::vInit(void)
{
  aud_BeepPrio_ubNextBeepID = 0;
  aud_BeepPrio_ubState = AUD_BEEPPRIOSTATE_OFF;
}


Beep_PrioHandler::~Beep_PrioHandler()
{
  poBeepNotification = NULL;
}

/*******************************************************************************
* FUNCTION: blCheckForBeepInQueue
*
* DESCRIPTION: returns true is a particular beep is found in queue
*        Mostly used during unit testing
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
bool Beep_PrioHandler::blCheckForBeepInQueue(tU8 beepName)
{
  vector<BeepData>::iterator itBeepData;
  if(lstBeepControlData.size() == 0)
  {
    return false;
  }
  for(itBeepData = lstBeepControlData.begin();itBeepData != lstBeepControlData.end(); itBeepData++)
  {
    if(itBeepData->ubBeepName == beepName)
    {
      return true;
    }
  }
  return false;
}

/*******************************************************************************
* FUNCTION: u8GetPrioHandlerState
*
* DESCRIPTION: returns state of prio handler
*        Mostly used during unit testing
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
tU8 Beep_PrioHandler::u8GetPrioHandlerState(void) const
{
  return aud_BeepPrio_ubState;
}

/*******************************************************************************
* FUNCTION: u8GetBeepName
*
* DESCRIPTION: returns beep in particular position in queue
*        Mostly used during unit testing
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
tU8 Beep_PrioHandler::u8GetBeepName(tU8 listIndex)
{
  if(lstBeepControlData.size() == 0)
  {
    return 0xff;
  }
  return(lstBeepControlData[listIndex].ubBeepName);
}


/*******************************************************************************
* FUNCTION: PrintBeepsInTheQueue
*
* DESCRIPTION: Helper function to print beeps in the queue
*
* PARAMETER:
*
* RETURNVALUE: None.
*******************************************************************************/
tVoid Beep_PrioHandler::PrintBeepsInTheQueue(tVoid)
{
  vector<BeepData>::iterator itBeepData;
  ETG_TRACE_USR4(("Beep_PrioHandler: Printing Beeps In The Queue"));

  for(itBeepData = lstBeepControlData.begin();itBeepData != lstBeepControlData.end(); itBeepData++)
  {
    ETG_TRACE_USR4(("Beep_PrioHandler: Beep :%02x  with Priority: %d",
      ETG_ENUM(T_e8_Beep_InternalID,itBeepData->ubBeepName), BeepPrioConfTable[itBeepData->ubBeepName]->ubPrio));
  }
}

void Beep_PrioHandler::vSetBeepNotification_IF(BeepNotification_IF* pBeepNotification_IF)
{
  aud_BeepPrio_ubNextBeepID = 0;
  aud_BeepPrio_ubState = AUD_BEEPPRIOSTATE_OFF;

#if 0

#define AUD_BEEP_DATA(intBeepName, a,b,c,d,e,f,g,h,i,j,k,l,m,n) BeepPrioConfTable.push_back(OSAL_NEW BeepPrioConf(intBeepName, a,h,i,(tU64)j,(tU64)k,(tU64)l,(tU64)m,n));
#include "beeppriority.conf"
#undef AUD_BEEP_DATA

#define AUD_BEEP_PRIO_RULE_STOP_IMMEDIATELY(stopBeep, beepToBeStopped) BeepPrioConfTable.at(stopBeep)->bitsStopImmediately = beepToBeStopped;
#include "beeppriority.conf"
#undef AUD_BEEP_PRIO_RULE_STOP_IMMEDIATELY

#define AUD_BEEP_DO_NOT_PUT_IN_QUEUE(beepNotToBePutInQueue) BeepPrioConfTable.at(beepNotToBePutInQueue)->doNotPutInQueue = true;
#include "beeppriority.conf"
#undef AUD_BEEP_DO_NOT_PUT_IN_QUEUE

#define AUD_BEEP_SEQUENCES(beepName, sequence) BeepPrioConfTable.at(beepName)->enSequence = sequence;
#include "beeppriority.conf"
#undef AUD_BEEP_SEQUENCES
#endif

  poBeepNotification = pBeepNotification_IF;
  Beep_PrioHandler::Beep_PrioHandlerInstance = this;
}
