/*******************************************************************************
 *
 * FILE:          Beep_Main.cpp
 *
 * SW-COMPONENT:  Beep application
 *
 * DESCRIPTION:   Beep Main Class implementation
 *
 * AUTHOR:        Ramesh Babu K
 *
 * COPYRIGHT:    (c) 2012 RBEI, Bangalore
 *
 *******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_BEEP
#include "trcGenProj/Header/Beep_Main.cpp.trc.h"
#endif


/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/

#include "Beep_Main.h"
#include "../fc_audiomanager_trace.h"
#include "../mute/stream/StreamMute.h"

#include "../InternalComponentCommunication/Messages/Beep/IDNotifyBeepState.h"
#include "../InternalComponentCommunication/Messages/SoundConfig/IDSetSoundSystemConfig.h"
#include "../InternalComponentCommunication/Messages/Beep/IDNotifyBeepFileStatus.h"
#include "../InternalComponentCommunication/Messages/Beep/IDCCAStartBeepFile.h"
#include "../InternalComponentCommunication/Messages/Beep/ID_NotifyBeepName.h"
#include "../InternalComponentCommunication/Messages/BOSE_Amp/ID_BOSE_Amplifier_ConnectionStatus.h"
#include "../InternalComponentCommunication/Messages/BOSE_Amp/ID_SonarInfo.h"

#include "Beep_Actuator.h"
#include "Beep_Controller.h"
#include "Priority/Beep_PrioHandler.h"
#include "../fc_audiomanager_service_Audio_Function.h"

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CSM_RNAIVI
//#include "Beep_CSMSignalInterface.h"
#include "Beep_Adas.h"
#include "fc_audiomanager_clienthandler_Buzzer.h"
#endif
#include "Beep_defines.h"

#include "../aud_sinkmanager/vd_adr3/vd_adr3Msg_If.h"

/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/

///*******************************************************************************
// *
// * FUNCTION: Beep_Main::Beep_Main()
// *
// * DESCRIPTION: Constructor.
// *
// * PARAMETER: None.
// *
// * RETURNVALUE: None.
// *
// *******************************************************************************/

Beep_Main::Beep_Main(void):IF_MessageObserver<PO_MessageConfig::enID>("Beep_Main Constructor")
{
  ETG_TRACE_USR4(("Beep_Main() entered."));
  m_poMainAppInstance = NULL;
  m_poBeepPrioHandler = NULL;
  m_pStreamMute = NULL;
  m_poBeepActuator = NULL;
  u8AmpConnectionState = 0x00;

  m_poAudio_Function =NULL;
    u8BeepName = 0xFF;
    u8BeepState = 0xFF;
  m_u8SoundSystemconfig = SOUND_CONFIG_ITEM_VALUE_SOUNDSYSTEM1;
  m_rsp_beep_token = 0;
  vResetInternalData();
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CSM_RNAIVI
  m_poBeepAdas = NULL;
  //m_poBeep_CSMSignalInterface = NULL;
  m_BuzzerclientHandler = NULL;
#endif
}

Beep_Main::Beep_Main(fc_audiomanager_tclApp* poMainAppl, fc_audiomanager_tclService_Audio_Function* m_poAudio_Func)
   : IF_MessageObserver<PO_MessageConfig::enID>("Beep_Main")
{
  m_poMainAppInstance = poMainAppl;
  m_poAudio_Function = m_poAudio_Func;
    m_pStreamMute = StreamMute::getInstance();
  m_poBeepPrioHandler = new Beep_PrioHandler(this,BeepData.m_BeepPrioConfTable);
  m_poBeepController = new Beep_Controller(m_poBeepPrioHandler, m_pStreamMute, m_poMainAppInstance,this);
  m_poBeepActuator = new Beep_Actuator(m_poBeepPrioHandler, m_poBeepController,m_poAudio_Function->GetVolumeManagerPtr());
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CSM_RNAIVI
  m_poBeepAdas = new Beep_Adas(this,m_poBeepPrioHandler,m_poAudio_Function->GetVolumeManagerPtr());
  if(((u8GetOEMType() == OEM_TYPE_INFINITI) || (u8GetOEMType() == OEM_TYPE_NISSAN))) //Bose MCAN Amplifier configured, creating clienthandler of buzzer

    m_BuzzerclientHandler = new fc_audiomanager_tclclienthandler_Buzzer(m_poMainAppInstance,this);
  else
    m_BuzzerclientHandler = NULL;
#endif
    u8BeepName = 0xFF;
    u8BeepState = 0xFF;

  m_u8SoundSystemconfig = SOUND_CONFIG_ITEM_VALUE_SOUNDSYSTEM1;
  u8AmpConnectionState = 0x00;
  m_rsp_beep_token = 0;

   vResetInternalData();
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this, PO_MessageConfig::ID_CCAStart_Beep);
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this, PO_MessageConfig::IDSetSoundSystemConfig);
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this, PO_MessageConfig::ID_CCAStart_BeepFile);
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this,PO_MessageConfig::ID_BOSE_Amplifier_ConnectionStatus);

   ETG_TRACE_USR2(("Beep_Main: Observer added for ID_CCAStart_Beep & IDSetSoundSystemConfig & IDCCAStartBeepFile"));

}

/*******************************************************************************
 *
 * FUNCTION: Beep_Main::~Beep_Main()
 *
 * DESCRIPTION: Destructor.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
Beep_Main::~Beep_Main()
{
  ETG_TRACE_USR4(("~Beep_Main() entered."));
  InternalCommunicationAdapter::getInstance()->POMessages->DeRegisterObserver(this);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CSM_RNAIVI
  if(m_poBeepAdas != NULL)
  {
    delete (m_poBeepAdas);
    m_poBeepAdas = NULL;
  }
  if(m_BuzzerclientHandler != NULL)
  {
    delete (m_BuzzerclientHandler);
    m_BuzzerclientHandler = NULL;
  }
#endif

  delete (m_poBeepPrioHandler);
    delete (m_poBeepController);
    //delete(m_pStreamMute);
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pStreamMute);
  m_pStreamMute->vDeleteMuteResponseTimer();
  m_pStreamMute = NULL;

  if (m_poBeepActuator != NULL)
  {
    delete(m_poBeepActuator);
    m_poBeepActuator = NULL;
  }
    m_poAudio_Function = NULL;
    m_poMainAppInstance = NULL;
    u8BeepName = 0xFF;
    m_u8SoundSystemconfig = 0x0;
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Main::MessageNotification(PO_MessageConfig::enID MsgId)
 *
 * DESCRIPTION: Function called on new Beep PO message.
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/

// This function is called by post office on new incoming messages.
void Beep_Main::MessageNotification(PO_MessageConfig::enID MsgId)
{
  switch(MsgId)
  {
  case (PO_MessageConfig::ID_CCAStart_Beep):
    {
      const ID_CCAStart_Beep* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage < ID_CCAStart_Beep > (MsgId);
      AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pMsg);
      ETG_TRACE_USR2(("Beep_Main: MessageNotification(ID_CCAStart_Beep): %d, %d", pMsg->BeepType, pMsg->Resource));

    setBeep(pMsg->BeepType);
    break;
  }
  case (PO_MessageConfig::IDSetSoundSystemConfig):
    {
      const IDSetSoundSystemConfig* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage < IDSetSoundSystemConfig > (MsgId);
      AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pMsg);
      m_u8SoundSystemconfig = pMsg->u8SoundConfig;
      ETG_TRACE_USR2(("Beep_Main: MessageNotification(IDSetSoundSystemConfig): %d", m_u8SoundSystemconfig));
      break;
    }
  case (PO_MessageConfig::ID_CCAStart_BeepFile):
    {
      const ID_CCAStart_BeepFile* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage < ID_CCAStart_BeepFile > (MsgId);
      AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pMsg);
      ETG_TRACE_USR2(("Beep_Main: MessageNotification(ID_CCAStart_BeepFile)"));
      setBeepFile(pMsg->File_Path,pMsg->oConfig,pMsg->m_u32Token);
     break;
    }

  case (PO_MessageConfig::ID_BOSE_Amplifier_ConnectionStatus):
      {
         const ID_BOSE_Amplifier_ConnectionStatus* pBoMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage<ID_BOSE_Amplifier_ConnectionStatus>(MsgId);
       AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pBoMsg);
         ETG_TRACE_USR4(("Beep_Controllers::MessageNotification:ID_BOSE_Amplifier_ConnectionStatus : %d",pBoMsg->connectionstate));
       u8AmpConnectionState = pBoMsg->connectionstate;
     break;
      }
  default:
    break;
  }

}

/*******************************************************************************
 *
 * FUNCTION: Beep_Main::setBeep(tU8 tBeepType)
 *
 * DESCRIPTION: Function called on new Beep PO message.
 *
 * PARAMETER: tU8 tBeepType
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Main::setBeep(tU8 tBeepType)
{
  tU8 tU8State = BEEP_STOPPED;
  tU8 tU8BeepId = 0xFF;

    if(tBeepType != EN_AUDIO_BEEP_INVALID)
    {
    //Check if the requested beep has a mapping to internal beep id
	AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poAudio_Function);
	if((m_poAudio_Function->IsSBRVariant()) && (tBeepType == EN_AUDIO_BEEP_CLICK))
	{
		ETG_TRACE_USR4(("Its a SBR varaint and Its a click beep"));
		#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CSM_RNAIVI
		u8BeepName = EN_AUDIO_BEEP_CLICK_WAV;    //mapping CLICK beep to CLICK_WAV
		#endif
	}
	else
	{
    tU8 u8InternalBeepID = BeepData.u8GetIntID(tBeepType);

    u8BeepName = u8InternalBeepID;
	ETG_TRACE_USR4(("Beep_Main_setBeep NOT a SBR variant and NOT a click beep"));
	}
    ETG_TRACE_USR4(("Beep_Main_setBeep BeepName:%d", u8BeepName));

      if(NULL != m_poBeepPrioHandler)
      {
        tU8State = m_poBeepPrioHandler->u8SetRequest(u8BeepName, this,this, &tU8BeepId);
      }

      switch(tU8State)
      {
        case BEEP_REJECTED:
          NotifyBeepState( u8BeepName, tU8BeepId, 0x00,0);
          break;
        case BEEP_UNIQUE_REJECTED:
          break;
        default:
        {
        while (BeepData.m_beep_store[u8BeepName]->m_u8Next != 0xFF)
          {
            tBeepType = BeepData.m_beep_store[u8BeepName]->m_u8Next;
            if(NULL != m_poBeepPrioHandler)
            {
               u8UpdateBeepMainData(tBeepType);
               m_poBeepPrioHandler->u8SetRequest(tBeepType, this, this,&tU8BeepId);
            }
          }
        }
        break;
        }
    }
  else
  {
    ETG_TRACE_ERR(("!!! E R R O R !!! Beep_Main_setBeep Invalid BeepName:%d", tBeepType));
  }
}
/*******************************************************************************
 *
 * FUNCTION: Beep_Main::GetBeepSource(tU8 tUBeepName)
 *
 * DESCRIPTION: To get Beep source
 *
 * PARAMETER:
 *
 * RETURNVALUE: None
 *
 *******************************************************************************/
tU8 Beep_Main::GetBeepSource(tU8 tUBeepName)
{
  ETG_TRACE_USR4(("Beep_Main_GetBeepSource %d",  tUBeepName));
  return BeepData.m_beep_store[tUBeepName]->m_u8Source;
}
/*******************************************************************************
 * FUNCTION: Beep_Main::u8UpdateBeepMainData(tU8 tBeepName)
 *
 * DESCRIPTION: Function to update Beep info for the beep in queue.
 *
 * PARAMETER: tU8 tBeepName
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Main::u8UpdateBeepMainData(tU8 tBeepName)
{
  ETG_TRACE_USR4(("Beep_Main_u8UpdateBeepMainData %d",  tBeepName));
  u8BeepName = tBeepName;
  bUpdateInternalData();
}
/*******************************************************************************
 * FUNCTION: Beep_Main::setBeepFile(std::string File_Path,BeepFile_Config& rfConfig, tU32 u32token)
 *
 * DESCRIPTION: Function called on new Beep PO message.
 *
 * PARAMETER: std::string   File_Path   -> Path of the beep file
 *            BeepFile_Config   rfConfig  -> Configuration for the beep
 *            tU32        u32token    -> Token of the beep
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Main::setBeepFile(const std::string File_Path,const BeepFile_Config& rfConfig,const tU32& u32token)
{
  tU8 tU8State = BEEP_STOPPED;
  tU8 tU8BeepId = 0xFF;
  //Map the received beep to internal beep ID EN_AUDIO_BEEP_PRIO_BEEP_GST_INTERNAL_01
  ETG_TRACE_USR4(("Beep_Main_setBeepFile > VolumeType:%d, VoldB=%d, token=%d, Path=%s ",rfConfig.VolumeType,rfConfig.Volume,u32token,File_Path.c_str()));

  //Assign the beep to an internal ID
  u8BeepName = EN_AUDIO_BEEP_GST_INTERNAL;

  //Check if the beep is supported in the project. If not supported, simply reject the beep
  if(BeepData.bIsBeepValid(u8BeepName))
  {
    //Send this beep to priohandler
    if(NULL != m_poBeepPrioHandler)
    {
      //Copy the beep path internally.
      m_req_beep_path = File_Path;
      m_req_beep_token = u32token;
      //Update other required data
      bUpdateInternalData((const BeepFile_Config*)&rfConfig,u32token);
      //Trigger a request
      tU8State = m_poBeepPrioHandler->u8SetRequest(u8BeepName, this,this, &tU8BeepId);
    }

    switch(tU8State)
    {
      case BEEP_REJECTED:
        NotifyBeepState( u8BeepName, tU8BeepId, 0x00, u32token);
        break;
      case BEEP_UNIQUE_REJECTED:
        break;
      default:
        break;
      }
  }
  else
  {
    ETG_TRACE_ERR(("!!! E R R O R !!! Beep_Main_setBeepFile Not Configured in Project ! ! ! !"));
  }
}



/*******************************************************************************
 *
 * FUNCTION: Beep_Main::GetSignalTonePara()
 *
 * DESCRIPTION: Interface function to get Signal Tone parameters.
 *
 * PARAMETER: tU8 tUBeepName, BeepControlData::BeepPara *BeepParameter
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Main::GetSignalTonePara(tU8 tUBeepName, BeepControlData::BeepPara *BeepParameter)
{
  ETG_TRACE_USR4(("Beep_Main_GetSignalTonePara %d",  tUBeepName));
  memcpy(BeepParameter,&m_req_beep_BeepPara,sizeof(BeepControlData::BeepPara));
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Main::GetSignalChimePara()
 *
 * DESCRIPTION: Interface function to get Signal Chime parameters.
 *
 * PARAMETER: tU8 tUBeepName, BeepControlData::ChimePara *ChimeParameter
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Main::GetSignalChimePara(tU8 tUBeepName, BeepControlData::ChimePara *ChimeParameter)
{
  ETG_TRACE_USR4(("Beep_Main_GetSignalChimePara %d",  tUBeepName));
  memcpy(ChimeParameter,&m_req_beep_ChimePara,sizeof(BeepControlData::ChimePara));
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Main::GetBeepVolumeOffsets(tU8 tUBeepName, BeepControlData::ChVolOff *VolOffsets)
 *
 * DESCRIPTION: Interface function to get volume offsets for Beep
 *
 * PARAMETER: tU8 tUBeepName
 *
 * RETURNVALUE: tS16
 *
 *******************************************************************************/
tVoid Beep_Main::GetBeepVolumeOffsets(tU8 tUBeepName, BeepControlData::ChVolOff *VolOffsets)
{
  ETG_TRACE_USR4(("GetBeepVolumeOffsets  BeepName:%d", tUBeepName));

#if defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX
  GetBeepVolumeOffsetsPSA(tUBeepName,VolOffsets);
#else
  GetBeepVolumeOffsetsNormal(tUBeepName,VolOffsets);
#endif


 if(u8AmpConnectionState){
  ETG_TRACE_USR4(("GetBeepVolumeOffsets AMP connected"));
  VolOffsets->LF       =  CHANNEL_MUTE;
  VolOffsets->RF       =  CHANNEL_MUTE;
  VolOffsets->CE       =  CHANNEL_MUTE;
  VolOffsets->SUB      =  CHANNEL_MUTE;
  VolOffsets->LR       =  0;
  VolOffsets->RR       =  CHANNEL_MUTE;
  VolOffsets->RampLin  =  0x50;//Fast Mute
  VolOffsets->RampDB   =  0x64;//Fast Mute
  }
  else
  {
   VolOffsets->LR       =  CHANNEL_MUTE;
  VolOffsets->RR       =  CHANNEL_MUTE;
  VolOffsets->RampLin  =  0x50;//Fast Mute
  VolOffsets->RampDB   =  0x64;//Fast Mute
 }
}
/*******************************************************************************
 *
 * FUNCTION: Beep_Main::GetBeepVolumeOffsetsPSA(tU8 tUBeepName, BeepControlData::ChVolOff *VolOffsets)
 *
 * DESCRIPTION: function to get volume offsets for Beep
 *
 * PARAMETER: tU8 tUBeepName,BeepControlData::ChVolOff *VolOffsets
 *
 * RETURNVALUE: None
 *
 *******************************************************************************/
tVoid Beep_Main::GetBeepVolumeOffsetsPSA(tU8 tUBeepName, BeepControlData::ChVolOff *VolOffsets)
{
  ETG_TRACE_USR4(("GetBeepVolumeOffsetsPSA %d",m_u8SoundSystemconfig));
  if (m_u8SoundSystemconfig == SOUND_CONFIG_ITEM_VALUE_SOUNDSYSTEM4) //for PSA, Beep to be heard on SUBWOOFER & CENTRE Speakers for Gen3 Amplifier
  {
    VolOffsets->CE       =  BeepData.m_beep_store[tUBeepName]->m_s16Volume;
    VolOffsets->SUB      =  BeepData.m_beep_store[tUBeepName]->m_s16Volume;
    VolOffsets->LF     =  CHANNEL_MUTE;
    VolOffsets->RF     =  CHANNEL_MUTE;
  }
  else // For Gen1,Gen2,HiFi Surround(Analog) Amplifiers, volume should be set to Front,CE
  {
    VolOffsets->LF       =  BeepData.m_beep_store[tUBeepName]->m_s16Volume;
    VolOffsets->RF       =  BeepData.m_beep_store[tUBeepName]->m_s16Volume;
    VolOffsets->CE       =  BeepData.m_beep_store[tUBeepName]->m_s16Volume;
    VolOffsets->SUB      =  CHANNEL_MUTE;
  }
}
/*******************************************************************************
 *
 * FUNCTION: Beep_Main::GetBeepVolumeOffsetsNormal(tU8 tUBeepName, BeepControlData::ChVolOff *VolOffsets)
 *
 * DESCRIPTION: function to get volume offsets for Beep
 *
 * PARAMETER: tU8 tUBeepName,BeepControlData::ChVolOff *VolOffsets
 *
 * RETURNVALUE: None
 *
 *******************************************************************************/
tVoid Beep_Main::GetBeepVolumeOffsetsNormal(tU8 tUBeepName, BeepControlData::ChVolOff *VolOffsets)
{
  ETG_TRACE_USR4(("GetBeepVolumeOffsetsNormal"));
  VolOffsets->LF       =  BeepData.m_beep_store[tUBeepName]->m_s16Volume;
  VolOffsets->RF       =  BeepData.m_beep_store[tUBeepName]->m_s16Volume;
  VolOffsets->CE       =  CHANNEL_MUTE;
  VolOffsets->SUB      =  CHANNEL_MUTE;
}
/*******************************************************************************
 *
 * FUNCTION: Beep_Actuator::GetSoundSystemConfig()
 *
 * DESCRIPTION: Interface function to sound system configuration
 *
 * PARAMETER: None
 *
 * RETURNVALUE: tU8
 *
 *******************************************************************************/
tU8 Beep_Main::GetSoundSystemConfig()
{
  return m_u8SoundSystemconfig;
}
/*******************************************************************************
 *
 * FUNCTION: Beep_Main::NotifyBeepState(tU8 tBeepID, tU8 tBeepState)
 *
 * DESCRIPTION: Beep state Notification
 *
 * PARAMETER: tU8 tBeepID, tU8 tBeepState
 *
 * RETURNVALUE: None
 *
 *******************************************************************************/
tVoid Beep_Main::NotifyBeepState(tU8 tUBeepName,tU8 tBeepID, tU8 tBeepState, tU32 u32ResponseToken)
{
  ETG_TRACE_USR4(("Entered NotifyBeepState BeepName : %d BeepID:%d, State:%d, token:%d",tUBeepName,tBeepID, tBeepState,u32ResponseToken));

  //Check if the beep has an external ID mapping, only then send out a beep status update

  if(tBeepState == BEEP_START_IP)
  {
    tBeepState = midw_fi_tcl_e8_AudioBeepStatus::FI_EN_AUDIO_BEEP_STATUS_ON;
  if(tUBeepName ==EN_AUDIO_BEEP_GST_INTERNAL && u8AmpConnectionState)
  {
  ETG_TRACE_USR4(("Beep_Main::NotifyBeepState file requested informing Bose to play in front centre config"));
    ID_SonarInfo oSonarInfo(EN_OBSTACLEZONE_FrontCenter);
    InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oSonarInfo);

  }

  }
  else
  {

    tBeepState = midw_fi_tcl_e8_AudioBeepStatus::FI_EN_AUDIO_BEEP_STATUS_OFF;

  if(tUBeepName ==EN_AUDIO_BEEP_GST_INTERNAL && u8AmpConnectionState)
  {
  ETG_TRACE_USR4(("Beep_Main::NotifyBeepState file requested informing Bose to play in front centre config"));
    ID_SonarInfo oSonarInfo(EN_OBSTACLEZONE_NoSound);
    InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oSonarInfo);

  }
  }

  u8BeepState = tBeepState;
  m_rsp_beep_token = u32ResponseToken;

    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poAudio_Function);

    //Check if the beep has an external ID mapping, only then send out a beep status update to HMI
    if (m_poAudio_Function->updateClients(MIDW_MASCFFI_C_U16_BEEPSTATUS) != AIL_EN_N_NO_ERROR)
     {
        ETG_TRACE_ERR(("vSendBeepSatus : eUpdateClients():-> Updating the clients failed"));
     }
     else
     {
        ETG_TRACE_USR4(("vSendBeepSatus: eUpdateClients():-> Updating the clients passed"));
     }
}

/*******************************************************************************
 *
 * FUNCTION:  Beep_Main::bSendCCAStatusMessage(amt_tclServiceData& roOutMsg, amt_tclServiceData* poInMsg)

 *******************************************************************************/
tBool Beep_Main::bSendCCAStatusMessage(amt_tclServiceData& roOutMsg, amt_tclServiceData* poInMsg ) const
 {
   (tVoid)poInMsg;
   (tVoid)roOutMsg;

   midw_mascffi_tclMsgBeepStatusStatus oBeepStatus;
   tBool bRetVal = false;

  oBeepStatus.BeepStatus.enType = (midw_fi_tcl_e8_AudioBeepStatus::tenType)u8BeepState;
  oBeepStatus.Resource.enType = (midw_fi_tcl_e8_ResourceNo::tenType)0x01;
  oBeepStatus.u32Token = m_rsp_beep_token;
  fi_tclVisitorMessage oCCaMsg(oBeepStatus);

   oBeepStatus.vDestroy();
   bRetVal = oCCaMsg.bHandOver(&roOutMsg);
   return bRetVal;
 }

/*******************************************************************************
 *
 * FUNCTION:  Beep_Main::bIsBeepIdValid(tU8 tUBeepName)

 *******************************************************************************/
tBool Beep_Main::bIsBeepIdValid(tU8 tUBeepName)
{
  return BeepData.bIsBeepValid(tUBeepName);
}
/*******************************************************************************
 *
 * FUNCTION:  Beep_Main::GetImxBeepData(tU8 tUBeepName)

 *******************************************************************************/
std::string Beep_Main::GetBeepFilePath(tU8 tUBeepName)
{
  if(tUBeepName != EN_AUDIO_BEEP_GST_INTERNAL)
  {
    beep_datatype* ptrBeepData = BeepData.m_beep_store[tUBeepName];
    if(ptrBeepData == NULL)
    {
      NORMAL_M_ASSERT_ALWAYS();
      return NULL;
    }
    beep_datatype_imx* tmp = dynamic_cast<beep_datatype_imx*>(ptrBeepData);
    if(tmp != NULL)
      m_req_beep_path = tmp->m_filepath;
  }
  ETG_TRACE_USR4(("Beep_Main::GetBeepFilePath, beep path is :%s ",m_req_beep_path.c_str()));
  return m_req_beep_path;
}
/*******************************************************************************
 *
 * FUNCTION:  Beep_Main::GetBeepVolume(tU8 tUBeepName)

 *******************************************************************************/

tS16 Beep_Main::GetBeepVolume(tU8 /*tUBeepName*/)
{
  ETG_TRACE_USR4(("Beep_Main::GetBeepVolume, beep volume is :%d ",m_req_beep_volume));
  return m_req_beep_volume;
};

/*******************************************************************************
 *
 * FUNCTION:  Beep_Main::GetBeepVolume(tU8 tUBeepName)

 *******************************************************************************/
tU32 Beep_Main::GetRequestToken()
{
  ETG_TRACE_USR4(("Beep_Main::GetRequestToken, beep token is :%d ",m_req_beep_token));
  return m_req_beep_token;
}

/*******************************************************************************
 *
 * FUNCTION:  Beep_Main::vResetInternalData()
 * Description : Resets the internal data variables : m_req_beep_ChimePara, m_req_beep_BeepPara, m_req_beep_volume to initial state
 *******************************************************************************/
tVoid Beep_Main::vResetInternalData()
{
  memset(&m_req_beep_ChimePara,0,sizeof(BeepControlData::ChimePara));
  memset(&m_req_beep_BeepPara,0,sizeof(BeepControlData::BeepPara));
  m_req_beep_volume = (-120 * ADR_DB_RESOLUTION);//Mute
  m_req_beep_token = 0;
}

/*******************************************************************************
 *
 * FUNCTION:  Beep_Main::bUpdateInternalData()
 * Description : Updates the internal data variables : m_req_beep_ChimePara, m_req_beep_BeepPara, m_req_beep_path, m_req_beep_volume to initial state
 *******************************************************************************/
tBool Beep_Main::bUpdateInternalData(const BeepFile_Config* ptrConfig, tU32 u32Token)
{
  ETG_TRACE_USR4(("Beep_Main_bUpdateInternalData %d",  u8BeepName));
  if(!BeepData.bIsBeepValid(u8BeepName))
  {
    ETG_TRACE_USR4(("Invalid Beep Id, Cannot process this request "));
    return false;
  }
  tBool bretval = false;
  if(u8BeepName != EN_AUDIO_BEEP_GST_INTERNAL)
    vResetInternalData();//Clear the data first

  beep_datatype* ptrBeepData = BeepData.m_beep_store[u8BeepName];
  if(ptrBeepData == NULL)
  {
    NORMAL_M_ASSERT_ALWAYS();
    return false;
  }

  /*************************
   * UPDATE VOLUME FOR BEEP
   *************************/
  //Common parameter : Get the volume of the beep
  if((ptrConfig == NULL)&&(u8BeepName != EN_AUDIO_BEEP_GST_INTERNAL))
  {
    m_req_beep_volume  = ptrBeepData->m_s16Gain;
    m_req_beep_token   = u32Token;
  }

  //Check and update, if user specific volume has to be applied for this beep.
  if(ptrConfig != NULL)
  {
    //Beep file path is already updated in Beep_Main::setBeepFile. So just update the volume
    if(BEEP_VOL_TYPE_USER_SPECIFIC == ptrConfig->VolumeType)//User specific gain has to be applied
    {
      m_req_beep_volume = (tS16)(ptrConfig->Volume * ADR_DB_RESOLUTION);
      ETG_TRACE_USR4(("Applying User Specific Volume: %d *1/4 dB",m_req_beep_volume));
    }
    else
      m_req_beep_volume  = ptrBeepData->m_s16Gain;
  }

  /*************************************
   * UPDATE DATA REQUIRED TO PLAY BEEP
   *************************************/
  //Check if the requested beep is a ADR beep. For ADR beep we donot support any user configuration.
  //All the required data has to be fetched from Data store in case of ADR beep.
  if(BeepData.bIsAdrBeep(u8BeepName))
  {
    ETG_TRACE_USR4(("bUpdateInternalData,bIsAdrBeep"));
    beep_datatype_adr* tmp = dynamic_cast<beep_datatype_adr*>(ptrBeepData);
    if(tmp != NULL)
    {
      //1. Fetch Chime parameters
      m_req_beep_ChimePara.Pulses       = tmp->m_u16Repeat;
      m_req_beep_ChimePara.AttackTime     = (tU16)(tmp->m_u16RiseTime + tmp->m_u16OnTime) ;
      m_req_beep_ChimePara.AttackTimeLog     = 0x7FFF;
      m_req_beep_ChimePara.AttackTimeLin     = tmp->m_u16RiseTime;
      m_req_beep_ChimePara.ReleaseTime     = (tU16)(tmp->m_u16PeriodTime - (tmp->m_u16RiseTime + tmp->m_u16OnTime));
      m_req_beep_ChimePara.ReleaseTimeLog   = 0x7FFF;
      m_req_beep_ChimePara.ReleaseTimeLin   = tmp->m_u16FallTime;
      m_req_beep_ChimePara.SignalToneAction   = tmp->m_u8SignalToneAction;
      //2. Fetch Beep Parameters
      m_req_beep_BeepPara.BeepId         = tmp->m_u8Generator;
      m_req_beep_BeepPara.Freq         = tmp->m_u16Freq;
      m_req_beep_BeepPara.Gain         = -6;  //-6; //currently configured to -6db

      bretval = true;
    }
    else
    {
      NORMAL_M_ASSERT_ALWAYS();
    }
  }
  //This beep is not an ADR beep. So this should be and IMX beep.
  //For IMX beep, we support user configuration of path.
  //For user configured imx beeps, m_req_beep_path variable is updated in Beep_Main::setBeepFile() function.
  //So if the user has not provided any configuration, then fetch the file path information from data store.
   else if((ptrConfig == NULL)&&(BeepData.bIsImxBeep(u8BeepName) && (u8BeepName != EN_AUDIO_BEEP_GST_INTERNAL)))
  {
    ETG_TRACE_USR4(("Beep_Main_bUpdateInternalData, Beep data is preconfigured. Imx_Beep. %d",  u8BeepName));
    beep_datatype_imx* tmp = dynamic_cast<beep_datatype_imx*>(ptrBeepData);
    if(tmp != NULL)
    {
      //1. Fetch Path
      m_req_beep_path = tmp->m_filepath;
      bretval = true;
    }
  }
  return bretval;
}
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CSM_RNAIVI
/*******************************************************************************
 *
 * FUNCTION: Beep_Main::vSendMethodStartToBuzzer(tU8 tBeepName)
 *
 * DESCRIPTION: Function to send the buzzer request to switch control
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Main::vSendBeepRequestToBuzzer(tU8 tBeepName)
{

  ETG_TRACE_USR4(("Beep_Main :: vSendBeepRequestToBuzzer() called for Beep Id : %d", tBeepName));
    beep_datatype* ptrBeepData = BeepData.m_beep_store[tBeepName];
    beep_datatype_adr* adrBeep = dynamic_cast<beep_datatype_adr*>(ptrBeepData);
    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(adrBeep);
        AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_BuzzerclientHandler);
  if(u8GetOEMType() == OEM_TYPE_INFINITI)
  {
  ETG_TRACE_USR4(("Beep_Main :: vSendBeepRequestToBuzzer()   OEM_TYPE_INFINITI called for Beep Id : %d", tBeepName));
    m_BuzzerclientHandler->vSendBuzzerRequest(tBeepName,adrBeep);
  }
  else if(u8GetOEMType() ==OEM_TYPE_NISSAN && m_poAudio_Function->bIsBoseAmpConnected())
  {
  ETG_TRACE_USR4(("Beep_Main :: vSendBeepRequestToBuzzer() OEM_TYPE_NISSAN called for Beep Id : %d", tBeepName));
  m_BuzzerclientHandler->VPlayBuzzerBeepGPIO(tBeepName,adrBeep);


  }

}
#endif

/*******************************************************************************
 *
 * FUNCTION: Beep_Main::u8GetOEMType()
 *
 * DESCRIPTION: to read the OEM(Variant Type) from PD Configuration
 *
 * PARAMETER:
 *
 * RETURNVALUE: tU8.
 *
 *******************************************************************************/
tU8 Beep_Main::u8GetOEMType()
{
  tU8 u8VariantInfo;
  if(DP_S32_NO_ERR == DP_s32GetConfigItem("CMVariantCoding","OEMType", &u8VariantInfo, 1))
    {
    ETG_TRACE_USR4(("OEM Type read from PD-KDS : %d", u8VariantInfo));
  }
  else
    ETG_TRACE_ERR(("reading OEMType failed"));
  return u8VariantInfo;
}
