/*******************************************************************************
 *
 * FILE:          Beep_DataStore.h
 *
 * SW-COMPONENT:  Beep Data Store
 *
 * PROJECT:
 *
 * DESCRIPTION:   Central store where all the beep related data is stored
 *
 * AUTHOR:       Ranjit K
 *
 * COPYRIGHT:    (c) 2015 RBEI, Bangalore
 *
 *******************************************************************************/
#ifndef _BEEP_DATASTORE_H_IF_INCLUDE_
#define _BEEP_DATASTORE_H_IF_INCLUDE_

//#define SYSTEM_S_IMPORT_INTERFACE_VECTOR
//#define SYSTEM_S_IMPORT_INTERFACE_MAP
//#define SYSTEM_S_IMPORT_INTERFACE_STRING
//#include <stl_pif.h>



#include <string>
#include <map>
#include <vector>
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"



using namespace std;

class BeepPrioConf;

/***************************************************************************************************
* Default configuration for IMX beeps
****************************************************************************************************/
//Defaults
#define BEEP_SRC_GST_PLAYER            (0x11) //Cue1
#define BEEP_ALSA_DEV_GST_PLAYER_DEFAULT    "AdevAudioSampler1Out"
#define BEEP_FILE_BASE_DIR            "/var/opt/bosch/static/audio/sounds/"

/***************************************************************************************************
* Class declaration : Beep data type
****************************************************************************************************/
class beep_datatype
{
private:
  //Explicit privates
  beep_datatype();
  beep_datatype(const beep_datatype&);
public:
  tU8    m_u8BeepId;//ID of the beep
  tU8    m_u8BeepExtId;//External ID of the beep
  tU8    m_u8Source;//Source to be used to generate the beep
  tS16  m_s16Volume;
  tS16  m_s16Gain;
  tU8    m_u8Next;
  tU16  m_u16Repeat;
  tU8    m_u8Generator;
  beep_datatype(tU8 BeepId, tU8 u8BeepExtId, tU8 uSource, tS16 volume,tS16  s16Gain, tU8 Next, tU16 Repeat,tU8 u8Generator);
  virtual ~beep_datatype(){}
};

/***************************************************************************************************
* Class declaration : Beep data type for beeps generated using ADR3
****************************************************************************************************/
class beep_datatype_adr : public beep_datatype
{
private:
  //Explicit privates
  beep_datatype_adr();
  beep_datatype_adr(const beep_datatype_adr&);
public:
  tU16  m_u16Freq;
  tU16  m_u16RiseTime;
  tU16  m_u16OnTime;
  tU16  m_u16FallTime;
  tU16  m_u16PeriodTime;
  tU8    m_u8SignalToneAction;

  beep_datatype_adr(tU8 u8BeepId,
    tU8 u8BeepExtId,
    tU8 u8Source,
    tU16 u16Freq,
    tS16 s16Volume,
    tS16 s16Gain,
    tU8 u8Next,
    tU16 u16Repeat,
    tU16 u16RiseTime,
    tU16 u16OnTime,
    tU16 u16FallTime,
    tU16 u16PeriodTime,
    tU8 u8SignalToneAction,
    tU8 u8Generator );

  virtual ~beep_datatype_adr(){}
};

/***************************************************************************************************
* Class declaration : Beep data type for beeps generated from iMX
****************************************************************************************************/
class beep_datatype_imx : public beep_datatype
{
private:
  beep_datatype_imx();
  beep_datatype_imx(const beep_datatype_imx&);
public:
  string m_filepath;
  beep_datatype_imx(tU8 BeepId, tU8 u8BeepExtId, tU8 Source,string File_Path,tS16 s16Volume,tS16 Gain,tU8 Next,tU16 Repeat,tU8 Generator);
  virtual ~beep_datatype_imx(){}
};

/***************************************************************************************************
* Class Declaration : Central data store for all beep related data
****************************************************************************************************/

class beep_datastore
{
private:
  bool m_bAdr3BeepConfigured;                //Flag to check ADR3 beeps are configured
  bool m_bImxBeepConfigured;                // Flag to check IMX beeps are configured
  string m_sAlsaDev;

public:
  beep_datastore();                    // Constructor
  ~beep_datastore();                    // Destructor

  beep_datatype* pGetBeepData(tU8 u8BeepID);        //Utility function to query the beep data using beep ID, please donot delete the return value

private:
  void vInit();                      //Helper function to initialize beep data store
  void vDeInit();                      //Helper function to deinitialize beep data store

  void vLoadAdrBeepData();                //Internal helper to load ADR3 beep data from the configuration file
  void vLoadImxBeepData();                //Internal helper to load IMX beep data from the configuration file
  void vLoadBeepPrio();                  //Internal helper function to load beep priority rules from configuration file
  void vLoadImxBeepConfig();                              //Internal helper function to load the configuration for imx beeps
public:
  map<tU8,beep_datatype*> m_beep_store;          //Map to handle all the beep related data
  vector <BeepPrioConf*> m_BeepPrioConfTable;        //Vector to handle the beep priority objects

  bool bIsAdr3BeepConfigured()        { return m_bAdr3BeepConfigured; }      //Helper function which tells if adr3 beeps are configured
  bool bIsImxBeepConfigured()          { return m_bImxBeepConfigured ; }      //Helper function which tells if imx beeps are configured
  bool bIsBeepValid(tU8 u8BeepId);
  string sGetAlsaDevice(tU8 u8Resource);//Helper function to get the alsa device to be used for a particular resource
  bool bIsAdrBeep(tU8 u8BeepId);
  bool bIsImxBeep(tU8 u8BeepId);

  tU8  u8GetExtID(tU8 u8InternalBeepName); //Helper function to convert an internal Id to an external ID
  tU8  u8GetIntID(tU8 u8ExternalBeepName);//Helper function to convert an external ID to an internal ID
};

#endif //#ifdef _BEEP_DATASTORE_H_IF_INCLUDE_


