/*******************************************************************************
 *
 * FILE:          Beep_Actuator.cpp
 *
 * SW-COMPONENT:  Beep Actuator application
 *
 * DESCRIPTION:   Beep Actuator Class implementation
 *
 * AUTHOR:        Ramesh Babu K
 *
 * COPYRIGHT:    (c) 2012 RBEI, Bangalore
 *
 *******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_BEEP
#include "trcGenProj/Header/Beep_Actuator.cpp.trc.h"
#endif


#include "../../../PostOffice/PostOffice.hpp"
#include "CCA/aud_CCAMsg.h"
#include "BeepController_IF.h"
#include "Beep_Actuator.h"
#include "Beep_Controller.h"
#include "fc_audiomanager_trace.h"
#include "../InternalComponentCommunication/Messages/Beep/IDNotifyAudioTestToneStatus.h"


#include "../InternalComponentCommunication/Messages/Beep/IDCCAStartAudioTestTone.h"
#include "Beep_defines.h"
#include "Priority/BeepPrioHandler_IF.h"

#include "../InternalComponentCommunication/DataTypes/TypeDefines/ADR3StreamType.h"
//#include "../ExternItf/CCA/FI_Message/FI_Includes.h"

/******************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/******************************************************************************/
#define ACTUATOR_OFFSET  0
#if defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SMART) || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_INF4CV)
#define ACTUATOR_DBV_OFFSET  -75
#else
#define ACTUATOR_DBV_OFFSET  -125   //-31.25 dB to get 1V voltage reference
#endif
/******************************************************************************/
/*                                                                            */
/* GLOBAL VARIABLES                                                           */
/*                                                                            */
/******************************************************************************/


/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/
/*******************************************************************************
 *
 * FUNCTION: Beep_Actuator::Beep_Actuator()
 *
 * DESCRIPTION: Constructor.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/

 Beep_Actuator::Beep_Actuator(void):IF_MessageObserver<PO_MessageConfig::enID>("Beep_Actuator Constructor")
{
  ETG_TRACE_USR4(("Beep_Actuator() entered."));
  m_poBeepPrioHandler = NULL;
  m_poBeepController = NULL;
  m_poVolumeMgr = NULL;
    u8ChannelSelect = 0;
    u32Frequency = 0;
    u8VolType = 0;
    s16Volume = 0;
    u8TestToneType = 0x00;
    u8Requester = 0x00;
    u8State =  midw_fi_tcl_e8_Aud_TestToneControl::FI_EN_FC_AUDIO_TEST_TONE_OFF;
    u8BeepId = 0x00;
}

 Beep_Actuator::Beep_Actuator(BeepPrioHandler_IF* poBeepPrioHandler, BeepController_IF* poBeepController,VolumeManager::VolumeManager* poVolumeMgr)
   : IF_MessageObserver<PO_MessageConfig::enID>("Beep_Actuator")
{
  ETG_TRACE_USR4(("Beep_Actuator() entered."));
  m_poBeepPrioHandler = poBeepPrioHandler;
  m_poBeepController = poBeepController;
  m_poVolumeMgr = poVolumeMgr;
    u8ChannelSelect = 0;
    u32Frequency = 0;
    u8VolType = 0;
    s16Volume = 0;
    u8TestToneType = 0x00;
    u8Requester = 0x00;
    u8State =  midw_fi_tcl_e8_Aud_TestToneControl::FI_EN_FC_AUDIO_TEST_TONE_OFF;
    u8BeepId = 0x00;

   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this, PO_MessageConfig::ID_CCAStart_AudioTestTone);
   ETG_TRACE_USR2(("Beep_Actuator: Observer added for ID_CCAStart_AudioTestTone"));


}

/*******************************************************************************
 *
 * FUNCTION: Beep_Actuator::~Beep_Actuator()
 *
 * DESCRIPTION: Destructor.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
Beep_Actuator::~Beep_Actuator()
{
  ETG_TRACE_USR4(("~Beep_Actuator() entered."));
  InternalCommunicationAdapter::getInstance()->POMessages->DeRegisterObserver(this);
    m_poBeepPrioHandler = NULL;
    m_poBeepController = NULL;
  m_poVolumeMgr = NULL;
}

/*******************************************************************************/

// This function is called by post office on new incoming messages.
void Beep_Actuator::MessageNotification(PO_MessageConfig::enID MsgId)
{




   const ID_CCAStart_AudioTestTone* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage < ID_CCAStart_AudioTestTone > (MsgId);
       AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pMsg);
      ETG_TRACE_USR4(("Beep_Actuator: MessageNotification, generator: adr3 : %d, %d, %d, %d, %d"
    , pMsg->SpeakerType, pMsg->Frequency, pMsg->VolumeType, pMsg->Volume, pMsg->Requester));

   setActuatorBeep(pMsg->SpeakerType, pMsg->Frequency, pMsg->VolumeType, pMsg->Volume, pMsg->Requester);














}

/*******************************************************************************
 *
 * FUNCTION: Beep_Actuator::setBeep(tU8 tBeepType)
 *
 * DESCRIPTION: Function called on new Beep PO message.
 *
 * PARAMETER: tU8 tBeepType
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Actuator::setActuatorBeep(tU8 tChannelSelect, tU32 tFrequency, tU8 tVolType, tS16 tVolume, tU8 tRequester)
{
  u8ChannelSelect = tChannelSelect;
  u32Frequency = tFrequency;
  u8VolType = tVolType;
  s16Volume = tVolume;
  tU8 tU8State;
  tU8 tU8BeepId;

  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poBeepPrioHandler);
    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poBeepController);
  ETG_TRACE_USR4(("Beep_Actuator_setBeep ChannelSelect:%d  Frequency:%d  VolType:%d  Volume:%d Requester:%d", u8ChannelSelect, u32Frequency, u8VolType, s16Volume, tRequester));

  if((u8Requester == tRequester)&&(u8State == midw_fi_tcl_e8_Aud_TestToneControl::FI_EN_FC_AUDIO_TEST_TONE_ON))
  {
    if(tChannelSelect == 0x00)
    {
      m_poBeepPrioHandler->vIdleBeep(BEEP_STOP_IMMEDIATE, u8BeepId);
    }
    else
    {
      m_poBeepController->SetBeepLocation();
    }
  }
  else if((u8State == midw_fi_tcl_e8_Aud_TestToneControl::FI_EN_FC_AUDIO_TEST_TONE_OFF)&&(tChannelSelect != 0x00))
  {
    u8Requester = tRequester;
    if(tRequester == TESTTONE_HMI)
    {
      u8TestToneType = EN_AUDIO_BEEP_ACTUATOR_HMI;
    }
    else if(tRequester == TESTTONE_DIAGNOSIS)
    {
      u8TestToneType = EN_AUDIO_BEEP_ACTUATOR_DIAGNOSIS;
    }
    else
    {
          ETG_TRACE_FATAL(("Beep_Actuator::Requester is Invalid"));
          NORMAL_M_ASSERT_ALWAYS();
    }
    tU8State = m_poBeepPrioHandler->u8SetRequest(u8TestToneType, this,this, &tU8BeepId);
    switch(tU8State)
      {
      case BEEP_REJECTED:
        NotifyBeepState( u8TestToneType, tU8BeepId, BEEP_STOPPED,0);
        
        break;
      case BEEP_UNIQUE_REJECTED:
        break;
      default:

        u8State = tU8State;
        break;
      }
  }
  else
  {
    //Request from the new requester not handled when already test tone is running (or) no channel is selected.
  }
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Actuator::setActuatorBeepState(tU8 tBeepState,tU8 tBeepId)
 *
 * DESCRIPTION: Function to update the Beep Actuator state based on its current state.
 *
 * PARAMETER: tU8 tBeepState,
 *        tU8 tBeepId
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Actuator::setActuatorBeepState(tU8 tBeepState,tU8 tBeepId)
{
  switch(tBeepState)
  {
  case BEEP_REJECTED:
    NotifyBeepState( u8TestToneType, tBeepId, BEEP_STOPPED,0);
    u8State = BEEP_STOPPED;
    // Beep Id update is not required as this Beep is rejected.
    break;
  case BEEP_UNIQUE_REJECTED:
      // no update of Beep Id or state is needed, as the Beep is already playing or queued.
    break;
  default:
    //u8BeepId = tBeepId;
    u8State = tBeepState;
    break;
  }
}

tVoid Beep_Actuator::SetBeepIdnStatus(tU8 /*tBeepName*/, tU8 tBeepId)
{
  ETG_TRACE_USR4(("Beep_Actuator::SetAdasSoundStatus() called :%d",tBeepId));
  u8BeepId = tBeepId;
  // to be used only for ADAS
}
/*******************************************************************************
 *
 * FUNCTION: Beep_Actuator::GetBeepSource(tU8 tUBeepName)
 *
 * DESCRIPTION: Function returns the source to be selected
 *
 * PARAMETER: tU8 tUBeepName
 *
 * RETURNVALUE: tU8.
 *
 *******************************************************************************/
tU8 Beep_Actuator::GetBeepSource(tU8 tUBeepName)
{
  tU8 u8Ret = BEEP_SRC_SOUNDGENERATOR1;
  ETG_TRACE_USR4(("Beep_Actuator_GetBeepSource for Beep=%d, returns source: %d",  tUBeepName,u8Ret));
  return u8Ret;
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Actuator::GetSignalTonePara()
 *
 * DESCRIPTION: Interface function to get Signal Tone parameters.
 *
 * PARAMETER: tU8 tUBeepName, BeepControlData::BeepPara *BeepParameter
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Actuator::GetSignalTonePara(tU8 tUBeepName, BeepControlData::BeepPara *BeepParameter)
{
  (tVoid)tUBeepName;  //Lint
  ETG_TRACE_USR4(("Beep_Actuator_GetSignalTonePara"));
  BeepParameter -> BeepId = SINUSOIDAL;
  BeepParameter -> Freq = (tU16) u32Frequency;
  BeepParameter -> Gain = -6; //configured same as Beeps
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Actuator::GetBeepVolume()
 *
 * DESCRIPTION: Interface function to get volume for Actuator Beep
 *
 * PARAMETER: tU8 tUBeepName
 *
 * RETURNVALUE: tS16
 *
 *******************************************************************************/
tS16 Beep_Actuator::GetBeepVolume(tU8 tUBeepName)
{
  (tVoid)tUBeepName;  //Lint
  tS16 tVolume;
  ETG_TRACE_USR4(("Beep_Actuator_GetVolume"));

  switch(u8VolType)
  {
    case midw_fi_tcl_e8_VolumeType::FI_EN_DB_LEVEL:
      tVolume = (tS16)(s16Volume*ADR_DB_RESOLUTION);
      break;
    case midw_fi_tcl_e8_VolumeType::FI_EN_DBV_LEVEL:
      tVolume = (tS16)(ACTUATOR_DBV_OFFSET +  s16Volume*ADR_DB_RESOLUTION);
      break;
    case midw_fi_tcl_e8_VolumeType::FI_EN_INCREMENT:
    {
      AUD_POINTER_CHECK_CALL(m_poVolumeMgr);
      //Getting the db value from VolumeManager for volume step for incrmental volume type
      tVolume = (tS16)(m_poVolumeMgr->s16getDBValue((tU8)s16Volume,EN_AUDIO_SOURCE_STREAM_DEFAULT) * ADR_DB_RESOLUTION); //EN_AUDIO_SOURCE_STREAM_DEFAULT
    }
    break;
    default:
      tVolume = -160;
      break;
  }

  return tVolume;
}
/*******************************************************************************
 *
 * FUNCTION: tS16 Beep_Actuator::GetIncrementalVolumeStep()
 *
 * DESCRIPTION: Interface function to get volume step data if the volume type is incremental type
 *
 * PARAMETER: NONE
 *
 * RETURNVALUE: tS16
 *
 *******************************************************************************/
tS16 Beep_Actuator::GetIncrementalVolumeStep()
{
  if(midw_fi_tcl_e8_VolumeType::FI_EN_INCREMENT == u8VolType)
    return s16Volume;    //returning step value
  return 0; //if vol type is not incremental, returning volume step as 0;
}
/*******************************************************************************
 *
 * FUNCTION: Beep_Actuator::GetBeepVolumeOffsets(tU8 tUBeepName, BeepControlData::ChVolOff *VolOffsets)
 *
 * DESCRIPTION: Interface function to get volume offsets for Beep
 *
 * PARAMETER: tU8 tUBeepName
 *
 * RETURNVALUE: tS16
 *
 *******************************************************************************/
tVoid Beep_Actuator::GetBeepVolumeOffsets(tU8 tUBeepName, BeepControlData::ChVolOff *VolOffsets)
{
  (tVoid)tUBeepName;  //Lint
  ETG_TRACE_USR4(("Beep_Actuator_GetBeepVolumeOffsets"));
  VolOffsets->LF       =  (tS16)((u8ChannelSelect & midw_fi_tcl_b8_ChannelSelect::FI_C_U8_BIT_CHANNEL_LF )  ? ACTUATOR_OFFSET : CHANNEL_MUTE);
  VolOffsets->RF       =  (tS16)((u8ChannelSelect & midw_fi_tcl_b8_ChannelSelect::FI_C_U8_BIT_CHANNEL_RF)  ? ACTUATOR_OFFSET : CHANNEL_MUTE);
  VolOffsets->LR       =  (tS16)((u8ChannelSelect & midw_fi_tcl_b8_ChannelSelect::FI_C_U8_BIT_CHANNEL_LR)  ? ACTUATOR_OFFSET : CHANNEL_MUTE);
  VolOffsets->RR       =  (tS16)((u8ChannelSelect & midw_fi_tcl_b8_ChannelSelect::FI_C_U8_BIT_CHANNEL_RR)  ? ACTUATOR_OFFSET : CHANNEL_MUTE);
  VolOffsets->CE       =  (tS16)((u8ChannelSelect & midw_fi_tcl_b8_ChannelSelect::FI_C_U8_BIT_CHANNEL_CENT)  ? ACTUATOR_OFFSET : CHANNEL_MUTE);
  VolOffsets->SUB     =  (tS16)((u8ChannelSelect & midw_fi_tcl_b8_ChannelSelect::FI_C_U8_BIT_CHANNEL_SUB)  ? ACTUATOR_OFFSET : CHANNEL_MUTE);
  VolOffsets->RampLin  =  0x08;//very Fast Mute
  VolOffsets->RampDB   =  0x320;//very Fast Mute
}
/*******************************************************************************
 *
 * FUNCTION: Beep_Actuator::GetSoundSystemConfig()
 *
 * DESCRIPTION: Interface function to sound system configuration
 *
 * PARAMETER: None
 *
 * RETURNVALUE: tU8
 *
 *******************************************************************************/
tU8 Beep_Actuator::GetSoundSystemConfig()
{
   //nothing implemented for now and not used in Beep Actuator Currently
  return 0;
}
/*******************************************************************************
 *
 * FUNCTION: Beep_Actuator::GetRequestToken()
 *
 * DESCRIPTION: Get Token of the Beep request
 *
 * PARAMETER: None
 *
 * RETURNVALUE: tU8
 *
 *******************************************************************************/
tU32  Beep_Actuator::GetRequestToken()
{
   //nothing implemented for now and not used in Beep Actuator Currently
  return 0;
}


/*******************************************************************************
 *
 * FUNCTION: Beep_Actuator::NotifyBeepState(tU8 tBeepID, tU8 tBeepState)
 *
 * DESCRIPTION: Beep state Notification
 *
 * PARAMETER: tU8 tBeepID, tU8 tBeepState
 *
 * RETURNVALUE: None
 *
 *******************************************************************************/
tVoid Beep_Actuator::NotifyBeepState(tU8 tUBeepName,tU8 tBeepID, tU8 tBeepState, tU32 /*u32ResponseToken*/)
{
  (tVoid)tUBeepName; //Lint
  (tVoid)tBeepID; //Lint
  ETG_TRACE_USR4(("Entered NotifyActuatorBeepState State:%d",tBeepState));

  switch(tBeepState)
  {
  case BEEP_STOPPED:
      {
      ETG_TRACE_USR4(("Beep_Acutator Demuting Other Streams"));
                AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poBeepController);
      m_poBeepController->DemuteOtherStreams();


      //Update internal state and notify clients
      u8State = midw_fi_tcl_e8_Aud_TestToneControl::FI_EN_FC_AUDIO_TEST_TONE_OFF;
      ID_NotifyAudioTestToneStatus oNotifyAudioTestToneStatus(u8ChannelSelect, u8State);
      InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oNotifyAudioTestToneStatus);
      }
      break;
  case BEEP_START_IP:
      {
      ETG_TRACE_USR4(("Beep_Acutator Muting Other Streams"));
                AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poBeepController);
      m_poBeepController->MuteOtherStreams();

      //Update internal state and notify clients
      u8State = midw_fi_tcl_e8_Aud_TestToneControl::FI_EN_FC_AUDIO_TEST_TONE_ON;
      ID_NotifyAudioTestToneStatus oNotifyAudioTestToneStatus(u8ChannelSelect, u8State);
      InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oNotifyAudioTestToneStatus);
      }
      break;
  default:
      {
      ETG_TRACE_USR4(("Beep_Acutator Other Stream Mute Status Unchanged"));
      }
      break;

  }
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Actuator::GetBeepFilePath(tU8 tBeepID)
 *
 * DESCRIPTION: Utility function to return the beep file name
 *
 * PARAMETER: tU8 tBeepID
 *
 * RETURNVALUE: None
 *
 *******************************************************************************/
std::string Beep_Actuator::GetBeepFilePath(tU8 /*tUBeepName*/)
{
  return std::string();
}
