/**
 * @file ArkamysFeatureHandlerRNAIVI2.cpp
 * @author pau4kor
 * @copyright (c) 2019 RBEI
 * @addtogroup fc_audiomanager
 * @{
 */


#include <vector>
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include <etrace_if.h>
#include "fc_audiomanager_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_ARKAMYS
#include "trcGenProj/Header/ArkamysFeatureHandlerRNAIVI2.cpp.trc.h"
#endif

/* Include Diagnostics FI interface */
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
//included in service_audio_function
//#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGNOSTICSFI_TYPES
//#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGNOSTICSFI_FUNCTIONIDS
//#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGNOSTICSFI_SERVICEINFO
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_ERRORCODES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_SERVICEINFO
#define MIDW_FI_S_IMPORT_INTERFACE_FI_TYPES

#include <midw_fi_if.h>
#include "ArkamysFeatureHandlerRNAIVI2.h"
#include "ArkamysStateCalculatorRNAIVI2.h"
#include "InternalComponentCommunication/Messages/Source/IDSetSource.h"
#include "InternalComponentCommunication/Messages/Source/IDIntSource.h"
#include "InternalComponentCommunication/Messages/mute/IDNotifyMute.h"
#include "InternalComponentCommunication/Messages/settings/IDSettings.h"
#include "InternalComponentCommunication/Messages/Sound/ID_ArkamysModeSet.h"
#include "InternalComponentCommunication/Messages/Diag/IDDiagLinearMode.h"
#include "InternalComponentCommunication/Messages/Diag/IDCCAStartAudDiagMode.h"
#include "InternalComponentCommunication/DataTypes/TypeDefines/ADR3Settings.h"
#include "InternalComponentCommunication/DataTypes/TypeDefines/ADR3StreamType.h"
#include "InternalComponentCommunication/DataTypes/TypeDefines/Fading_Reason.h"
#include "InternalComponentCommunication/Messages/Startup/IDNotifyStartup.h"
#include "mute/stream/StreamMute_IF.h"
#include "mute/stream/StreamMute.h"

#include "fc_audiomanager_service_Audio_Function.h"

//For Diaglog/ITC
#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include "vd_diaglog_if.h"


#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#include "dp_audio_if.h"
#include "dp_diagdebug_if.h" // For Arkamys EOL Datapool

#include "vd_adr3Msg_If.h"

#include "config/dp/AudioUserDpIfSelect.h"

// Number of elements
#define NELT(_array) (sizeof(_array) / sizeof(_array[0]))
#define DATA_LEN 2
// Only for g3g view build
#ifndef ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_07
#define ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_07 0x8607
#endif

ArkamysFeatureHandlerRNAIVI2* ArkamysFeatureHandlerRNAIVI2::getInstance()
{
   static ArkamysFeatureHandlerRNAIVI2 theInstance;
   return &theInstance;
}

ArkamysFeatureHandlerRNAIVI2::ArkamysFeatureHandlerRNAIVI2():
      IF_MessageObserver<PO_MessageConfig::enID>("Arkamys Mute Result Observer")
{

   /*
    * Initialize System settings
    */
   m_u32ReportedArkamysLib    = 0;
   m_enArkamysMode            = EN_ARKAMYS_BYPASS;

   /*
    * Initialize Feature State relevant settings
    */
   m_bEngModeActive              = false;
   m_enCurSourceSetting          = ArkamysConfigRNAIVI2::ARKAMYS_SOURCE_INVALID;
   m_enArkamysSourcrGrp          = ArkamysConfigRNAIVI2::ARKAMYS_GRP_UNDEFINED;

   /*
    * Initialize Current Feature Settings
    */
   m_u8VirtualSubEnabled   = ArkamysConfigRNAIVI2::STATE_UNKNOWN;
   m_bVirtualSubAvailable  = false;

   /*
    * Initialize pointers
    */
   m_pPO                         = InternalCommunicationAdapter::getInstance();
   m_poMainAppl                  = NULL;
   m_poResultHandler             = NULL;

   m_pEolInterface               = ArkamysEolInterfaceRNAIVI2::getInstance();
   m_pADRInterface               = new ArkamysADRInterface();
   m_bDeleteArkamysADRInterface  = true;
   m_pArkamysCommInterface       = new ArkamysCommInterface();
   m_bDeleteArkamysCommInterface = true;

   /*
    * Initialize misc members
    */
   m_enState            = ARKAMYS_FEATURE_STATE_IDLE;
   m_u8MuteRegId        = 0;
   m_hTimer             = OSAL_C_INVALID_HANDLE;

   /*
    * Initialize auditorium members
    */
   m_bEntStreamMuted = false;

}

// Lint does not recognize the delete flags (i.e. m_bDeleteArkamysCommInterface), so it thinks that the pointer is not freed correctly.
//lint -save -e1740 pointer member 'ArkamysFeatureHandlerRNAIVI2::m_pArkamysCommInterface' not directly freed or zeroed by destructor
ArkamysFeatureHandlerRNAIVI2::~ArkamysFeatureHandlerRNAIVI2()
{
   InternalCommunicationAdapter::getInstance()->POMessages->DeRegisterObserver(this);
   m_pPO             = NULL;
   m_poMainAppl      = NULL;
   m_pEolInterface   = NULL;
   m_poResultHandler = NULL;

   if (m_bDeleteArkamysCommInterface)
   {
      delete m_pArkamysCommInterface;
      m_pArkamysCommInterface = NULL;
   }

   if (m_bDeleteArkamysADRInterface)
   {
      delete m_pADRInterface;
      m_pADRInterface = NULL;
   }

   // delete timer
   if (m_hTimer != OSAL_C_INVALID_HANDLE)
   {
      (void)OSAL_s32TimerDelete(m_hTimer);
      m_hTimer = OSAL_C_INVALID_HANDLE;
   }


}
//lint -restore

void ArkamysFeatureHandlerRNAIVI2::vTimerCallback(tPVoid pvArg)
{
   (tVoid)pvArg;    /* unused parameter */
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::vTimerCallback"));
   fc_audiomanager_tclApp::vEventFromOtherThreads(FC_AUDIOMANAGER_EVENT_ID_ARKAMYS);
}

tenArkamysMode ArkamysFeatureHandlerRNAIVI2::enGetAudioMode()
{
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::enGetAudioMode"));
  return m_enArkamysMode;
}

ArkamysConfigRNAIVI2::tenArkamysAmplifier ArkamysFeatureHandlerRNAIVI2::enGetKDSAmplifierInformation()
{
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::enGetKDSAmplifierInformation"));
     //read item from the KDS key's bytes- stream

  ArkamysConfigRNAIVI2::tenArkamysAmplifier enArkamysAmplifier = ArkamysConfigRNAIVI2::ARKAMYS_AMP_INTERNAL_AMPLIFIER;

  tU8 u8Amplifier = 0; //initializing with internal amplifier

  if((DP_S32_NO_ERR == DP_s32GetConfigItem("SystemConfiguration1","OutputInformation", &u8Amplifier, 1)))
  {
    ETG_TRACE_USR4(("Amplifier value  : %d", u8Amplifier));

    switch(u8Amplifier)
    {
    case INTERNALAMP_2SPEAKER :
    case EXTERNAL_ANC :
    case INTERNALAMP_4SPEAKER:
    case EXTERNAL_ANC_FRONT_ONLY:
      enArkamysAmplifier = ArkamysConfigRNAIVI2::ARKAMYS_AMP_INTERNAL_AMPLIFIER;
      break;
    case PREMIUM_AMP1:
    case EXTERNALAMP_BOSEWOMCAN:
    case EXTERNAL_DAIMLER_1:
    case EXTERNAL_DAIMLER_2:
    case BOSE_PERSONAL_SPACE:
      enArkamysAmplifier =  ArkamysConfigRNAIVI2::ARKAMYS_AMP_PREMIUM_1;
      break;
    case PREMIUM_AMP2:
    case EXTERNALAMP_BOSE2CH_MCAN:
    case EXTERNALAMP_BOSE_5_1CH_MCAN:
      enArkamysAmplifier = ArkamysConfigRNAIVI2::ARKAMYS_AMP_PREMIUM_2;
      break;
    default :
      break;
    }
  }
  else
  {
    ETG_TRACE_ERR(("Error in diagnosis read "));
    //return 0;
  }

  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2 amplifier %i", ETG_CENUM(ArkamysConfigRNAIVI2::tenArkamysAmplifier, enArkamysAmplifier)));

  return enArkamysAmplifier;

}

ArkamysConfigRNAIVI2::tenArkamysDeviceVariant ArkamysFeatureHandlerRNAIVI2::enGetKDSLibInformation()
{
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::enGetKDSLibInformation"));
     //read item from the KDS key's bytes- stream

  ArkamysConfigRNAIVI2::tenArkamysDeviceVariant enArkamysDevVariant = ArkamysConfigRNAIVI2::ARKAMYS_DEV_INVALID;

  tU8 u8ArkamysClassicLib = 0; //disabled
  if((DP_S32_NO_ERR == DP_s32GetConfigItem("CMVariantCoding","ArkamysClassicSupport", &u8ArkamysClassicLib, 1)))
  {
    ETG_TRACE_USR4(("Arkamys classic support  : %d", u8ArkamysClassicLib));
    if(u8ArkamysClassicLib)
         enArkamysDevVariant =  ArkamysConfigRNAIVI2::ARKAMYS_DEV_CLASSIC_ONLY;
  }
  else
  {
    ETG_TRACE_ERR(("Error in diagnosis read "));
    //return 0;
  }

  tU8 u8ArkamysAudiLib = 0; //disabled
  if((DP_S32_NO_ERR == DP_s32GetConfigItem("CMVariantCoding","ArkamysAuditoriumSupport", &u8ArkamysAudiLib, 1)))
  {
    ETG_TRACE_USR4(("Arkamys auditorium support  : %d", u8ArkamysAudiLib));
    if(u8ArkamysAudiLib)
       enArkamysDevVariant =  ArkamysConfigRNAIVI2::ARKAMYS_DEV_ALL_OTHER;
  }
  else
  {
    ETG_TRACE_ERR(("Error in diagnosis read "));
    //return 0;
  }

  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2 device variant %i", ETG_CENUM(ArkamysConfigRNAIVI2::tenArkamysDeviceVariant, enArkamysDevVariant)));

  return enArkamysDevVariant;

}

ArkamysConfigRNAIVI2::tenArkamysEOLLib ArkamysFeatureHandlerRNAIVI2::enGetArkamysEOLLibType()
{
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::enGetArkamysEOLLibType"));

  ArkamysConfigRNAIVI2::tenArkamysEOLLib enArkamysEOLLib = ArkamysConfigRNAIVI2::ARKAMYS_EOL_INVALID;

  tU32 ArkamysEOLLibType = ntohl(ArkamysEolInterfaceRNAIVI2::getInstance() -> getEOLLibType());

  if(ArkamysEOLLibType == ARKAMYS_LIBTYPE_CLASSIC)
    enArkamysEOLLib = ArkamysConfigRNAIVI2::ARKAMYS_EOL_CLASSIC;


  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2 EOL Lib type %i", ETG_CENUM(ArkamysConfigRNAIVI2::tenArkamysEOLLib, enArkamysEOLLib)));

  return enArkamysEOLLib;

}

void ArkamysFeatureHandlerRNAIVI2::vSetArkamysMode()
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::vSetArkamysMode"));

   if(m_u32ReportedArkamysLib == 0)//wait until arkamys library version is available
     return;

   tBool bLibVersionMatch = true;

   if((m_u32ReportedArkamysLib < ARKAMYS_LIB_MIN_VERSION) || (m_u32ReportedArkamysLib > ARKAMYS_LIB_MAX_VERSION))
   {
     bLibVersionMatch = false;
   }

   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::vSetArkamysMode EOL version and library version match (%s)", bLibVersionMatch? "true":"false"));

   //check for valid EOL
   tBool bValidEOLLoaded = ArkamysEolInterfaceRNAIVI2::getInstance()->isDataFromDatapool();

   m_enArkamysMode = ArkamysStateCalculatorRNAIVI2::enCalculateArkamysMode(enGetKDSAmplifierInformation(),
                                       enGetArkamysEOLLibType(),
                                       bLibVersionMatch,
                                       enGetKDSLibInformation(),
                                       bValidEOLLoaded
                                       );


   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2 Arkamys Audio mode set to %i", ETG_CENUM(tenArkamysMode ,m_enArkamysMode)));

   tU8 u8ArkamysMode = m_enArkamysMode;

   //update ASPP
   vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_ARKAMYS_MODE, VD_ADR3_OPTYPE_SET
        , 1, &u8ArkamysMode, ARKAMYS_MODE);//send data to ADR3

   //update EOL interface
   ArkamysEolInterfaceRNAIVI2::getInstance()->setArkamysAudioMode(m_enArkamysMode);

   //update HMI
   AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poMainAppl);
   fc_audiomanager_tclService_Audio_Function *poAudioFunction = m_poMainAppl->pGetAudioFunctionptr();

   AUD_POINTER_CHECK_CALL_NO_RET_VALUE(poAudioFunction);
   poAudioFunction->updateClients(MIDW_MASCFFI_C_U16_AUDIOMODE);

   ID_ArkamysModeSet oArkamysModeSet(m_enArkamysMode);
   InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oArkamysModeSet);
}

/**
 * bStartup()
 * Perform startup checks and corresponding actions. To be called only once at the beginning of the lifecycle
 */
tBool ArkamysFeatureHandlerRNAIVI2::bStartup(fc_audiomanager_tclApp* poMainAppl)
{
   AudioUserDpIf* pAudUsrDpIf = NULL;

   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::bStartup() entered."));

   ETG_TRACE_USR4(("Requesting Arkamys Lib from ADR..."));

   m_ActionQueue.Add(AkActionRNAIVI2(AkActionRNAIVI2::ARKAMYS_ACTION_GET_LIBVERSION, 0));
   m_ActionQueue.Event(AkActionRNAIVI2::ARKAMYS_EVENT_CONTINUE);

   m_poMainAppl = poMainAppl;

   // create timer
   tS32 tmrRes = OSAL_s32TimerCreate((OSAL_tpfCallback)vTimerCallback, 0, &m_hTimer);
   if (tmrRes != OSAL_OK)
   {
      m_hTimer = OSAL_C_INVALID_HANDLE;
      ETG_TRACE_ERR(("ArkamysFeatureHandlerRNAIVI2::bStartup(): OSAL_s32TimerCreate failed."));
   }


   // Load the Eol Data from the Datapool
   tBool bLoadEolSuccess = false;
   bLoadEolSuccess = ArkamysEolInterfaceRNAIVI2::getInstance() -> loadEolData();

   if (!bLoadEolSuccess)
   {
      ETG_TRACE_SYS_MIN(("ArkamysFeatureHandlerRNAIVI2:bStartup() Loading of EOL data failed. Continuing without EOL data."));
      //return false;
   }


   // Register for Stream Mute
   m_u8MuteRegId = StreamMute::getInstance() -> registerForStreamMute(StreamMute_IF::INTERNAL_BEH, this);

   // Observe relevant PO messages
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_SetSource);
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_Settings);
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_DiagLinearMode);
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_CCAStart_AudDiagMode);
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_NotifyStartup);
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_CCA_GetDiagResult);
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_NotifyMute);
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::IDIntSource);
   pAudUsrDpIf = AudioUserDpIfSelect::pGetAudUsrDpIf();

   if(pAudUsrDpIf)
   {
     tS16 s16VirtualSub= 0x00;//coverity fix

     pAudUsrDpIf->s32GetSubwoofer(s16VirtualSub);
     m_bVirtualSubAvailable = static_cast<tBool>(s16VirtualSub & 0x01);

   }

   // Since some settings may have changed, update and apply states
   vUpdateStatesAndApply();

   ETG_TRACE_USR1(("ArkamysFeatureHandlerRNAIVI2:bStartup() done."));

   return true;
}

void ArkamysFeatureHandlerRNAIVI2::vUpdateStatesAndApply()
{
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pEolInterface);
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::vUpdateStatesAndApply() entered."));

  // check if Arkamys Feature should be enabled
  if (m_enArkamysMode == EN_ARKAMYS_BYPASS)
  {
    ETG_TRACE_USR2(("Arkamys Feature is disabled -> Not checking for state changes."));
    ETG_TRACE_USR2(("    EOL Arkamys Lib: %x",ntohl( ArkamysEolInterfaceRNAIVI2::getInstance() -> getEOLArkamysLibVersion())));
    ETG_TRACE_USR2(("    ADR Arkamys Lib: %x", m_u32ReportedArkamysLib));
    ETG_TRACE_USR2(("    EOL Arkamys Lib type: %x", ntohl(ArkamysEolInterfaceRNAIVI2::getInstance() -> getEOLLibType())));
    ETG_TRACE_USR2(("    Arkamys source: %i", ETG_CENUM(ArkamysConfigRNAIVI2::tenSrcSetting, m_enCurSourceSetting)));
    return;
  }

  /*******************classic features****************/
  tBool bVSubwoofer = bUpdateVirtualSubwooferState();

  // Bail out when nothing to do
  if (!bVSubwoofer)
  {
    ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::vUpdateStatesAndApply(): nothing to do."));
    return;
  }


  m_ActionQueue.Add(AkActionRNAIVI2(AkActionRNAIVI2::ARKAMYS_ACTION_MUTE, 0));
  if(!m_bEntStreamMuted)//if stream is already muted then do not wait for mute status
    m_ActionQueue.Add(AkActionRNAIVI2(AkActionRNAIVI2::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI2::ARKAMYS_EVENT_MUTED));  // Wait until muted

  tBool bActionsAdded = false;

  //if diagnosis session is active or source is set to phone,speech -> set only arkamys bypass
  if((m_bEngModeActive) || (m_enCurSourceSetting == ArkamysConfigRNAIVI2::ARKAMYS_SOURCE_OTHER) )
  {
    //activate bypass
    m_ActionQueue.Add(AkActionRNAIVI2(AkActionRNAIVI2::ARKAMYS_ACTION_SET_MODE, EN_ARKAMYS_BYPASS));
    m_ActionQueue.Add(AkActionRNAIVI2(AkActionRNAIVI2::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI2::ARKAMYS_EVENT_ASPP_OK));  // Wait until adr ok
    bActionsAdded = true;

  }

  else
  {

    /*********************classic only***************************/

    m_ActionQueue.Add(AkActionRNAIVI2(AkActionRNAIVI2::ARKAMYS_ACTION_SEND_FLUSH_CMD, 0));
    m_ActionQueue.Add(AkActionRNAIVI2(AkActionRNAIVI2::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI2::ARKAMYS_EVENT_ASPP_OK));
    //bass enhancer
    if (bVSubwoofer)
    {
      addVirtualSubwooferToActionQueue(m_u8VirtualSubEnabled);
      bActionsAdded = true;
    }

    //start classic
    m_ActionQueue.Add(AkActionRNAIVI2(AkActionRNAIVI2::ARKAMYS_ACTION_SET_MODE, m_enArkamysMode));
    m_ActionQueue.Add(AkActionRNAIVI2(AkActionRNAIVI2::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI2::ARKAMYS_EVENT_ASPP_OK));  // Wait until adr ok

  }

  // Demute
  m_ActionQueue.Add(AkActionRNAIVI2(AkActionRNAIVI2::ARKAMYS_ACTION_DEMUTE, 0));

  if (!bActionsAdded)
  {
    // Cancel queue if nothing was added
    ETG_TRACE_USR4(("Cancel ActionQueue (no relevant changes)."));
    m_ActionQueue.Cancel();
  }
  else
  {
    // Start Queue
    m_ActionQueue.Event(AkActionRNAIVI2::ARKAMYS_EVENT_CONTINUE);
  }
}

/**
 * setLinearMode()
 * @param bLinearModeEnable set true to enable linear mode, false to disable
 * @description While linear mode is active, the Arkamys features are switched off or set to a flat configuration
 */
void ArkamysFeatureHandlerRNAIVI2::setLinearMode(tBool bLinearModeEnable)
{
   m_bEngModeActive = bLinearModeEnable;
   vUpdateStatesAndApply();
}

/**
 *setSource()
 *@param enSource sets the source info based on primary, secondary or other sources
 *@description source information is set inside arkamys module
 */
void ArkamysFeatureHandlerRNAIVI2::setSource(ArkamysConfigRNAIVI2::tenSrcSetting enSource)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::setSource entered with source %i.", ETG_CENUM(ArkamysConfigRNAIVI2::tenSrcSetting, enSource)));

   if(enSource == ArkamysConfigRNAIVI2::ARKAMYS_SOURCE_INVALID) // do nothing for an invalid source
    return;

   m_enCurSourceSetting = enSource;
   vUpdateStatesAndApply();
}

/**
 *setVirtualSubwoofer()
 *@description setting of virtual subwoofer/bass enhancer in arkamys classic mode
 */
void ArkamysFeatureHandlerRNAIVI2::setVirtualSubwoofer(tBool bOn)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::setVirtualSubwoofer(%s)", bOn ? "On" : "Off"));

   m_bVirtualSubAvailable = bOn;
   vUpdateStatesAndApply();
}

/**
 *Mute()
 *@description setting of mute to ADR3
 */
void ArkamysFeatureHandlerRNAIVI2::Mute()
{
   AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pArkamysCommInterface);
   m_pArkamysCommInterface -> Mute(m_u8MuteRegId);
}

/**
 *Demute()
 *@description setting of demute to ADR3
 */
void ArkamysFeatureHandlerRNAIVI2::Demute()
{
   AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pArkamysCommInterface);
   m_pArkamysCommInterface -> Demute(m_u8MuteRegId);
}

/**
 *u32GetEOLVersionFormat()
 *@description convert the ASPP lib version to EOL format
 */
tVoid ArkamysFeatureHandlerRNAIVI2::vGetEOLVersionFormat(tU32& u32Version)
{
    tU32 version[3] = {0};
    //extract major, minor and patch version from ASPP version.
    //M*10000+m*100+p
    //- M: Major (0..99), m: Minor (0..99), p: Patch (0..99)
    if(u32Version > 9999)
    {
        version[0] = u32Version/10000;
        u32Version = u32Version % 10000;
    }

    if(u32Version > 99)
    {
        version[1] = u32Version/100;
        u32Version = u32Version % 100;
    }
    version[2] = u32Version;

    //coversion to EOL format
    //example - 1.2.3 --> 10203
    u32Version  = version[0];
    u32Version <<= 8;
    u32Version |= version[1];
    u32Version <<= 8;
    u32Version |= version[2];

}
/**
 *AdrMessageRx()
 *@description receives messages sent from ADR for function ID FE0(ArkamysParameter)
 */
//void ArkamysFeatureHandlerRNAIVI2::AdrMessageRx(tU8 cmd, tU32 adr, tU32 len, tS32* data)
void ArkamysFeatureHandlerRNAIVI2::AsppMessageRx(ID_ArkamysParameter arkamysParameter)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::AsppMessageRx(cmd:%i adr:%08X len:%08X data[0]:%08X)",
         ETG_CENUM(ArkamysMessage::tenAdrOpType, arkamysParameter.arkamysCommand), arkamysParameter.arkamysAddress, arkamysParameter.arkamysLength, arkamysParameter.arkamysValues[0]));
   if (arkamysParameter.arkamysCommand == ArkamysStaticMessage::ArkamysOpTypeStatus)
   {
      tU32 u32ArkAddress = ntohl(arkamysParameter.arkamysAddress);

      // Check if lib ack was received
      if(u32ArkAddress == ARKAMYS_LIB_ACK_ADDR )
      {
          vAsppOkCallback();//to continue the queue
      }
      else
      {
          if (m_pADRInterface != NULL)
              m_pADRInterface -> vAdrOkCallback();//for processing the queue
          m_ActionQueue.Event(AkActionRNAIVI2::ARKAMYS_EVENT_DATA_RECEIVED);

          tU32 u32ArkMsgLen  = ntohl(arkamysParameter.arkamysLength);

          tU16 u16BlockId, u16ParamId;
          u16BlockId = (tU16)(u32ArkAddress >> 2);
          u16BlockId = (tU16)(u16BlockId & 0xFF00);
          u16ParamId = (tU16)(u32ArkAddress & 0x00FF);
          ETG_TRACE_USR4(("Block Id set to (%x), Param Id set to (%x).", u16BlockId, u16ParamId));

          ArkamysConfigRNAIVI2::tArkParamList *pArkLibParams = ArkamysConfigRNAIVI2::getArkamysParamConfig(ArkamysConfigRNAIVI2::ARK_LIB_VERSION);
          if(pArkLibParams == NULL)
              return;
          if(
             (pArkLibParams->arkblockid == u16BlockId) &&
             (pArkLibParams->arkparamid == u16ParamId) &&
             (pArkLibParams->size == u32ArkMsgLen)
            )// Check if lib version was received
          {
              m_u32ReportedArkamysLib = arkamysParameter.arkamysValues[0];//data sent by arkamys lib is always in little-endian
              ETG_TRACE_USR4(("SBR Arkamys Library version - ASPP format %i (0x%X).", m_u32ReportedArkamysLib, m_u32ReportedArkamysLib));
              vGetEOLVersionFormat(m_u32ReportedArkamysLib);
              ETG_TRACE_USR4(("SBR Arkamys Library version - EOL format %i (0x%X).", m_u32ReportedArkamysLib, m_u32ReportedArkamysLib));
              vSetArkamysMode();//set arkamys mode in sbr, hmi
              vUpdateStatesAndApply();
          }
          else
          {
              ETG_TRACE_USR1(("ArkamysFeatureHandlerRNAIVI2::AsppMessageRx Version Parameter is not relevant"));
          }
      }

   }
}

/**
 *vStartTimer()
 *@description starting of osal timer
 */
void ArkamysFeatureHandlerRNAIVI2::vStartTimer(OSAL_tMSecond milliseconds)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::vStartTimer(%i)", milliseconds));
   if (m_hTimer != OSAL_C_INVALID_HANDLE)
   {
      tS32 tmrRes = OSAL_s32TimerSetTime(m_hTimer, milliseconds, 0);
      if (tmrRes != OSAL_OK)
      {
         ETG_TRACE_ERR(("ArkamysFeatureHandlerRNAIVI2::vStartTimer(): OSAL_s32TimerSetTime failed."));
      }
   }
}

/**
 *vStopTimer()
 *@description stopping of osal timer
 */
void ArkamysFeatureHandlerRNAIVI2::vStopTimer()
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::vStopTimer()"));
   if (m_hTimer != OSAL_C_INVALID_HANDLE)
   {
      tS32 tmrOk = OSAL_s32TimerSetTime(m_hTimer, 0, 0);
      if (tmrOk != OSAL_OK)
      {
         ETG_TRACE_ERR(("ArkamysFeatureHandlerRNAIVI2::vStopTimer(): OSAL_s32TimerSetTime failed."));
      }
   }
}

/**
 *timerExpired()
 *@description callback function for timer expiry
 */
void ArkamysFeatureHandlerRNAIVI2::timerExpired(tPVoid pvArg)
{
   pvArg = pvArg; // not used
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::timerExpired(%p)", pvArg));
   m_ActionQueue.Event(AkActionRNAIVI2::ARKAMYS_EVENT_TIMEOUT);
}

/**
 *vAdrOkCallback()
 *@description ADR3 callback for data connection availability
 */
void ArkamysFeatureHandlerRNAIVI2::vAsppOkCallback()
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::vAsppOkCallback(). TID = %i", OSAL_ThreadWhoAmI()));
   if (m_pADRInterface != NULL)
   {
      m_pADRInterface -> vAdrOkCallback();
   }
   else
   {
      ETG_TRACE_ERR(("m_pArkamysADRInterface == NULL"));
   }
   m_ActionQueue.Event(AkActionRNAIVI2::ARKAMYS_EVENT_ASPP_OK);
}

/**
 *vResetArkamysSettingsFlags()
 *@description to trigger resetting of arkamys features during ADR reset
 */
void ArkamysFeatureHandlerRNAIVI2::vResetArkamysSettingsFlags()
{
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::vResetArkamysSettingsFlags().called "));
  //disable classic settings  for recalculation of states
  m_u8VirtualSubEnabled = ArkamysConfigRNAIVI2::STATE_UNKNOWN;
  vUpdateStatesAndApply();
}

/**
 *MessageNotification()
 *@description receives P.O messages
 */
void ArkamysFeatureHandlerRNAIVI2::MessageNotification(PO_MessageConfig::enID MsgId)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::MessageNotification(%d)", ETG_CENUM(PO_MessageConfig::enID, MsgId)));

   if (MsgId == PO_MessageConfig::ID_SetSource)
   {
      const MSG_SetSource* pMsg = m_pPO->POMessages->QueryMessage<MSG_SetSource>(MsgId);
      ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2: Source setting received: Stream:%i Type:%i"
            , ETG_CENUM(tenStream, pMsg -> enStream), ETG_CENUM(tenSourceType, pMsg -> enSourceType)));

      setSource(ArkamysConfigRNAIVI2::getArkamysSource(pMsg -> enStream, pMsg -> enSourceType));

   }

   else if (MsgId == PO_MessageConfig::IDIntSource)
   {
      const IDIntSource* pMsg = m_pPO->POMessages->QueryMessage<IDIntSource>(MsgId);
      ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2: Message IDIntSource(%i, %i) received."
            , pMsg->value.m_Channel
            ,  ETG_CENUM(tenInternalSource, pMsg->value.m_Source)));

      if(pMsg->value.m_Channel == midw_fi_tcl_e8_AudioChannel::FI_EN_AUDIO_CHANNEL_EXC)
      {
          vSendArkamysSrcGrp(pMsg->value.m_Source);
      }


   }

   else if (MsgId == PO_MessageConfig::ID_Settings)
   {
      const MSG_Settings* pMsg = m_pPO->POMessages->QueryMessage<MSG_Settings>(MsgId);
      ETG_TRACE_USR2(("ArkamysFeatureHandlerRNAIVI2: ID_Settings received with type: %i and parameter: %i.", ETG_CENUM(tenSettingType, pMsg -> enSettingType), pMsg -> s16Steps));
      switch (pMsg -> enSettingType)
      {
      default:
         ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2: Setting %i ignored (not relevant for Arkamys handling).", ETG_CENUM(tenSettingType, pMsg -> enSettingType)));
         break;

      case EN_AUDIO_SETTINGS_VIRTUAL_SUB_WOOFER :
          ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2: Settings Virtual Sub woofer = %i", pMsg -> s16Steps));
          setVirtualSubwoofer(static_cast<tBool>(pMsg -> s16Steps));
          break;
      }
   }
   else if (MsgId == PO_MessageConfig::ID_NotifyStartup)
   {
       ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2: NotifyStartup received."));

       const ID_NotifyStartup* pMsg = InternalCommunicationAdapter::getInstance() -> POMessages -> QueryMessage<ID_NotifyStartup>(MsgId);
       if(pMsg -> enOn == EN_AUDIO_ADR3_AVAILABLE)
       {
           //resend arkamys classic information due to ADR reset
           vResetArkamysSettingsFlags();
       }
   }
   else if (MsgId == PO_MessageConfig::ID_DiagLinearMode || MsgId == PO_MessageConfig::ID_CCAStart_AudDiagMode)
   {
      enum
      {
         AK_LINEAR_MODE_NO_CHANGE,
         AK_LINEAR_MODE_ON,
         AK_LINEAR_MODE_OFF
      } enLinMode = AK_LINEAR_MODE_NO_CHANGE;

      if (MsgId == PO_MessageConfig::ID_DiagLinearMode)
      {
         const ID_DiagLinearMode*         pMsg_LinModePhase = m_pPO->POMessages->QueryMessage<ID_DiagLinearMode>(MsgId);
         enLinMode = (pMsg_LinModePhase->enDiagLinearModePhase == EN_LINEARMODE_SET) ? AK_LINEAR_MODE_ON :
                     (pMsg_LinModePhase->enDiagLinearModePhase == EN_LINEARMODE_UNFREEZE) ? AK_LINEAR_MODE_OFF :
                     AK_LINEAR_MODE_NO_CHANGE;
      }
      else if (MsgId == PO_MessageConfig::ID_CCAStart_AudDiagMode)
      {
         const ID_CCAStart_AudDiagMode*   pMsg_DiagMode     = m_pPO->POMessages->QueryMessage<ID_CCAStart_AudDiagMode>(MsgId);
         enLinMode = (pMsg_DiagMode -> DiagAud == midw_fi_tcl_e8_DiagAudMode::FI_EN_AUDIO_DIAG_AUD_MODE_LINEAR_ON) ?  AK_LINEAR_MODE_ON :
                     (pMsg_DiagMode -> DiagAud == midw_fi_tcl_e8_DiagAudMode::FI_EN_AUDIO_DIAG_AUD_MODE_LINEAR_OFF) ?  AK_LINEAR_MODE_OFF :
                     AK_LINEAR_MODE_NO_CHANGE;
      }


      if (enLinMode == AK_LINEAR_MODE_ON)
      {
         setLinearMode(true);
      }
      else if (enLinMode == AK_LINEAR_MODE_OFF) //if (pMsg_LinModePhase->enDiagLinearModePhase == EN_LINEARMODE_UNFREEZE)
      {
         setLinearMode(false);
      }
      else
      {
         ETG_TRACE_USR4(("Not changing linear mode."));
      }
   }

  else if(MsgId == PO_MessageConfig::ID_CCA_GetDiagResult)
   {
    ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2: PO message for for DiagResult Recieved"));
     const ID_CCA_GetDiagResult*   pMsg_DiagResult    = m_pPO->POMessages->QueryMessage<ID_CCA_GetDiagResult>(MsgId);


     if(pMsg_DiagResult != NULL)
     {
       m_poResultHandler = &(pMsg_DiagResult->roResultHandler);

       for (tU16 i=0; i < pMsg_DiagResult -> roRequestedITCs.size(); i++)
       {
         if(pMsg_DiagResult -> roRequestedITCs[i] == ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_07)
            vHandleGetDiagResult();
       }
     }
   }
  else if(MsgId == PO_MessageConfig::ID_NotifyMute)
  {
   const ID_NotifyMute* pMsg = m_pPO->POMessages->QueryMessage<ID_NotifyMute>(MsgId);
   AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pMsg);
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2 :: ADR mute notification received  with action: %x and Stream: %x ."
     , pMsg->enMuteResult, pMsg->enStream));
   if(pMsg->enStream == EN_AUDIO_SOURCE_STREAM_MAIN)
   {
     if(pMsg->enMuteResult == EN_AUDIO_MUTE_MUTERESULT_MUTE)
       m_bEntStreamMuted = true;
     else
       m_bEntStreamMuted = false;
   }
  }

}

/**
 *vSendArkamysSrcGrp()
 *@description to map adr sourcetype to arkamys source grp
 */
void ArkamysFeatureHandlerRNAIVI2::vSendArkamysSrcGrp(tenInternalSource enIntSource)
{
    m_enArkamysSourcrGrp =  ArkamysConfigRNAIVI2::getArkamysSourceGrp(enIntSource);
    if(m_enArkamysSourcrGrp == ArkamysConfigRNAIVI2::ARKAMYS_GRP_UNDEFINED)//do not forward to Arkamys
    {
        ETG_TRACE_FATAL(("ArkamysFeatureHandlerRNAIVI2::Invalid Arkamys group set %i", ETG_CENUM(ArkamysConfigRNAIVI2::tenSrcGrpSetting, m_enArkamysSourcrGrp)));
        return;
    }

    if(m_enArkamysSourcrGrp == ArkamysConfigRNAIVI2::ARKAMYS_GRP_ANNOUNCEMENT)//do not forward to Arkamys if announcement sources
    {
        ETG_TRACE_USR1(("ArkamysFeatureHandlerRNAIVI2::Announcement Arkamys group set %i", ETG_CENUM(ArkamysConfigRNAIVI2::tenSrcGrpSetting, m_enArkamysSourcrGrp)));
        return;
    }

    m_ActionQueue.Add(AkActionRNAIVI2(AkActionRNAIVI2::ARKAMYS_ACTION_SET_SOURCE_GRP, m_enArkamysSourcrGrp));
    m_ActionQueue.Event(AkActionRNAIVI2::ARKAMYS_EVENT_CONTINUE);
}
/**
 *vHandleGetDiagResult()
 *@description Handler for Diaglog property SENDNEXTTESTRESULT
 */
void ArkamysFeatureHandlerRNAIVI2::vHandleGetDiagResult()
{

  ETG_TRACE_USR2(("ArkamysFeatureHandlerRNAIVI2::vHandleGetDiagResult entered"));
  tTestResultList oFiTestResultObject;

  dp_tclArkamysDPArkamys_Eol oDpEol;
  if(oDpEol.bReloadDpElement())//reload dp data for ITC handling
    ArkamysEolInterfaceRNAIVI2::getInstance()->loadEolData();

  tBool bresult;
  bresult = ArkamysEolInterfaceRNAIVI2::getInstance()->isDataFromDatapool();//get datapool read status

  if (m_poResultHandler != NULL)
  {
    tTestResult oTestData;
    // Set the parameters of the FI data object
    oTestData.u16ID = ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_07;

    if (bresult)
    {
      oTestData.enResult = EN_PASSED;
      ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::TroubleCode:0x%x, Diagnosis Test Result = PASSED",ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_07));
    }
    else
    {
      oTestData.enResult = EN_FAILED;
      ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::TroubleCode:0x%x, Diagnosis Test Result = FAILED",ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_07));
    }

    oFiTestResultObject.push_back(oTestData);
    m_poResultHandler->vSaveDiagResult(oFiTestResultObject);
    oFiTestResultObject.clear();
  }
  else
  {
    ETG_TRACE_ERR(("ArkamysFeatureHandlerRNAIVI2::DiagLog result handler not set"));
  }
}

/**
 *addVirtualSubwooferToActionQueue()
 *@description adding virtual subwoofer to message queue
 */
void ArkamysFeatureHandlerRNAIVI2::addVirtualSubwooferToActionQueue(tU8 u8On)
{
  if (m_enState != ARKAMYS_FEATURE_STATE_IDLE)
  {
    ETG_TRACE_USR3(("ArkamysFeatureHandlerRNAIVI2::addVirtualSubwooferToActionQueue(): not executed. State not idle, but %i", ETG_CENUM(tenArkamysHandlerState, m_enState)));
    return;
  }

  if(u8On == ArkamysConfigRNAIVI2::STATE_UNKNOWN)
  {
    ETG_TRACE_USR3(("ArkamysFeatureHandlerRNAIVI2::addVirtualSubwooferToActionQueue(): not executed. Invalid request for Bass Enhancer: %i", ETG_CENUM(ArkamysConfigRNAIVI2::tenBassEnhancerState, u8On)));
    return;
  }

   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::addVirtualSubwooferToActionQueue(%i)", u8On));

   m_ActionQueue.Add(AkActionRNAIVI2(AkActionRNAIVI2::ARKAMYS_ACTION_SET_VIRTUAL_SUB, u8On));        // Set chunk
   m_ActionQueue.Add(AkActionRNAIVI2(AkActionRNAIVI2::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI2::ARKAMYS_EVENT_ASPP_OK));

}

/**
 *StreamMuteNotify()
 *@description receives the mute/demute notification
 */
void ArkamysFeatureHandlerRNAIVI2::StreamMuteNotify(StreamMuteNotify_IF::tenMuteNotification muteAck, tU8 stream, tU16 sink)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::StreamMuteNotify()"));
   stream = stream; // Hello lint
   if(muteAck == StreamMuteNotify_IF::MUTE_ACK_MUTE)
   {
      m_ActionQueue.Event(AkActionRNAIVI2::ARKAMYS_EVENT_MUTED);
   }
   else if(muteAck == StreamMuteNotify_IF::MUTE_ACK_DEMUTE)
   {
      m_ActionQueue.Event(AkActionRNAIVI2::ARKAMYS_EVENT_DEMUTED);
   }
}

void ArkamysFeatureHandlerRNAIVI2::SetArkamysCommInterface(ArkamysCommInterface *pArkamysCommInterface)
{
   if (m_pArkamysCommInterface != NULL && m_bDeleteArkamysCommInterface)
   {
      delete m_pArkamysCommInterface;
      m_pArkamysCommInterface = NULL;
   }

   // The following is secured by m_bDeleteArkamysCommInterface, but Lint doesn't recognize this.
   //lint -save -e423 Creation of memory leak in assignment to 'ArkamysFeatureHandlerRNAIVI2::m_pArkamysCommInterface'
   m_pArkamysCommInterface = pArkamysCommInterface;
   //lint -restore
   m_bDeleteArkamysCommInterface = FALSE;
}

void ArkamysFeatureHandlerRNAIVI2::SetArkamysADRInterface(ArkamysADRInterface *pArkamysADRInterface)
{
   if (m_pADRInterface != NULL && m_bDeleteArkamysADRInterface)
   {
      delete m_pADRInterface;
      m_pADRInterface = NULL;
   }

   // The following is secured by m_bDeleteArkamysADRInterface, but Lint doesn't recognize this.
   //lint -save -e423 Creation of memory leak in assignment to 'ArkamysFeatureHandlerRNAIVI2::m_pArkamysADRInterface'
   m_pADRInterface = pArkamysADRInterface;
   //lint -restore
   m_bDeleteArkamysADRInterface = FALSE;
}


/**
 * returns true when VirtualSubwoofer state has changed
 */
tBool ArkamysFeatureHandlerRNAIVI2::bUpdateVirtualSubwooferState()
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI2::bUpdateVirtualSubwooferState() entered."));
   ETG_TRACE_USR4(("    Parameters: Engmode:%i, Source:%i", m_bEngModeActive, ETG_CENUM(ArkamysConfigRNAIVI2::tenSrcSetting, m_enCurSourceSetting)));
   tU8 u8NewState = ArkamysStateCalculatorRNAIVI2::u8CalculateVirtualSubwooferState(m_bEngModeActive,  m_enCurSourceSetting, m_bVirtualSubAvailable);

   if (u8NewState != m_u8VirtualSubEnabled)
   {
      ETG_TRACE_USR4(("Virtual Subwoofer state changed: %i -> %i.", m_u8VirtualSubEnabled, u8NewState));
      m_u8VirtualSubEnabled = u8NewState;
      return true;
   }

   ETG_TRACE_USR4(("Virtual Subwoofer state not changed (is: %i).", m_u8VirtualSubEnabled));

   return false;
}

/**
 * TraceStatus()
 * Trace-Output current state. Called via TTFis command "AUDIOMANAGER_ArkamysDebug PRINT_STATUS"
 */
void ArkamysFeatureHandlerRNAIVI2::TraceStatus()
{
   // Query ADR Version from Registry (as seen in FC_tclEngModeDiagnosisClientHandler.cpp)
   reg_tclRegKey t_reg;

   char chararray[60] = { '\0' };
   if (t_reg.bOpen("/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/VERSIONS") == true)
   {
    if (t_reg.bQueryString("ADR_SW_VERSION", chararray, sizeof(chararray)) == false)
      {
       chararray[0] = 0;
      }
      t_reg.vClose();
   }



   ETG_TRACE_FATAL(("\t +-----------------------------------------------------------------------------------------\t"));
   ETG_TRACE_FATAL(("\t + Arkamys Audio Mode:   \t%i                                    \t\t\t\t\t", ETG_CENUM(tenArkamysMode, m_enArkamysMode)));
   ETG_TRACE_FATAL(("\t |                                                               \t\t\t\t\t"));

   ETG_TRACE_FATAL(("\t +ASPP Version                                                    \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t |   %s                                                          \t\t\t\t\t", chararray ));
   ETG_TRACE_FATAL(("\t |   Arkamys Lib:        \t%x                                    \t\t\t\t\t", m_u32ReportedArkamysLib));

   ETG_TRACE_FATAL(("\t |                                                               \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t +EOL (loaded from %s)                                           \t\t\t\t\t", ArkamysEolInterfaceRNAIVI2::getInstance() -> isDataFromDatapool() ? "Datapool" : "Internal" ));
   ETG_TRACE_FATAL(("\t |   Supported Arkamys Lib:  \t%x                                    \t\t\t\t\t", ntohl(ArkamysEolInterfaceRNAIVI2::getInstance() -> getEOLArkamysLibVersion())));
   ETG_TRACE_FATAL(("\t |   Supported Lib Type: \t0x%x                                  \t\t\t\t\t", ntohl(ArkamysEolInterfaceRNAIVI2::getInstance() -> getEOLLibType())));

  // ETG_TRACE_FATAL(("\t |                                                                   \t\t\t\t\t"));
  // ETG_TRACE_FATAL(("\t +KDS Values                                                         \t\t\t\t\t"));
  // ETG_TRACE_FATAL(("\t |                                                                   \t\t\t\t\t"));
  // ETG_TRACE_FATAL(("\t |                                                                   \t\t\t\t\t"));

   ETG_TRACE_FATAL(("\t +External state                                                 \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t |   Linear Mode:         \t%s                                   \t\t\t\t\t", m_bEngModeActive ? "on" : "off"));
   ETG_TRACE_FATAL(("\t |   Source:              \t%i                                   \t\t\t\t\t", ETG_CENUM(ArkamysConfigRNAIVI2::tenSrcSetting, m_enCurSourceSetting)));
   ETG_TRACE_FATAL(("\t |   Source group:        \t%i                                   \t\t\t\t\t", ETG_CENUM(ArkamysConfigRNAIVI2::tenSrcGrpSetting, m_enArkamysSourcrGrp)));
   ETG_TRACE_FATAL(("\t |                                                               \t\t\t\t\t"));

   ETG_TRACE_FATAL(("\t +Classic Feature state                                          \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t |                                                               \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t |   Virtual Subwoofer:   \t%i                                   \t\t\t\t\t", ETG_CENUM(ArkamysConfigRNAIVI2::tenArkamysFeatureState, m_u8VirtualSubEnabled)));
   ETG_TRACE_FATAL(("\t |                                                               \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t +-----------------------------------------------------------------------------------------\t"));
}

