/**
 * @file ArkamysFeatureHandlerRNAIVI.cpp
 * @author pau4kor
 * @copyright (c) 2015 RBEI
 * @addtogroup fc_audiomanager
 * @{
 */


#include <vector>
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"


#include "AudioProxy/aud_gio_dbus_handler.h"
#include "AudioProxy/aud_gio_dbus_audproc_proxy.h"

#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include <etrace_if.h>
#include "fc_audiomanager_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_ARKAMYS
#include "trcGenProj/Header/ArkamysFeatureHandlerRNAIVI.cpp.trc.h"
#endif

/* Include Diagnostics FI interface */
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
//included in service_audio_function
//#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGNOSTICSFI_TYPES
//#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGNOSTICSFI_FUNCTIONIDS
//#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGNOSTICSFI_SERVICEINFO
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_ERRORCODES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_SERVICEINFO
#define MIDW_FI_S_IMPORT_INTERFACE_FI_TYPES

#include <midw_fi_if.h>
#include "ArkamysFeatureHandlerRNAIVI.h"
#include "ArkamysStateCalculatorRNAIVI.h"
#include "InternalComponentCommunication/Messages/Source/IDSetSource.h"
#include "InternalComponentCommunication/Messages/mute/IDNotifyMute.h"
#include "InternalComponentCommunication/Messages/settings/IDSettings.h"
#include "InternalComponentCommunication/Messages/Sound/ID_ArkamysFeature.h"
#include "InternalComponentCommunication/Messages/Sound/ID_ArkamysModeSet.h"
#include "InternalComponentCommunication/Messages/Diag/IDDiagLinearMode.h"
#include "InternalComponentCommunication/Messages/Diag/IDCCAStartAudDiagMode.h"
#include "InternalComponentCommunication/DataTypes/TypeDefines/ADR3Settings.h"
#include "InternalComponentCommunication/DataTypes/TypeDefines/ADR3StreamType.h"
#include "InternalComponentCommunication/DataTypes/TypeDefines/Fading_Reason.h"
#include "InternalComponentCommunication/Messages/Startup/IDNotifyStartup.h"
#include "InternalComponentCommunication/Messages/Startup/ID_NotifyAudioProcess.h"
#include "mute/stream/StreamMute_IF.h"
#include "mute/stream/StreamMute.h"

/**
 * MSP_SOUND_ROOM_EFFECT values
 * below include need to be added
 * #include "../../../../../../ai_audio/components/AudioManager/include/audiomanagertypes.h"
 * using namespace am;
 */
#define AM_MSP_SND_ROOM_EFFECT_LEVEL0  ((int16_t) 0)
#define AM_MSP_SND_ROOM_EFFECT_LEVEL1  ((int16_t) 1)
#define AM_MSP_SND_ROOM_EFFECT_LEVEL2  ((int16_t) 2)
#define AM_MSP_SND_ROOM_EFFECT_LEVEL3  ((int16_t) 3)
#define AM_MSP_SND_ROOM_EFFECT_LEVEL4  ((int16_t) 4)

#include "fc_audiomanager_service_Audio_Function.h"

//For Diaglog/ITC
#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include "vd_diaglog_if.h"


#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#include "dp_audio_if.h"
#include "dp_diagdebug_if.h" // For Arkamys EOL Datapool

#include "vd_adr3Msg_If.h"

#include "config/dp/AudioUserDpIfSelect.h"

// Number of elements
#define NELT(_array) (sizeof(_array) / sizeof(_array[0]))
#define DATA_LEN 2
// Only for g3g view build
#ifndef ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_07
#define ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_07 0x8607
#endif

ArkamysFeatureHandlerRNAIVI* ArkamysFeatureHandlerRNAIVI::getInstance()
{
   static ArkamysFeatureHandlerRNAIVI theInstance;
   return &theInstance;
}

ArkamysFeatureHandlerRNAIVI::ArkamysFeatureHandlerRNAIVI():
      IF_MessageObserver<PO_MessageConfig::enID>("Arkamys Mute Result Observer")
{

   /*
    * Initialize System settings
    */
   m_u32ReportedArkamysLib    = 0;
   m_enArkamysMode            = EN_ARKAMYS_BYPASS;
   m_bAudioProcAvailable      = false;

   /*
    * Initialize Feature State relevant settings
    */
   m_bEngModeActive              = false;
   m_enCurSourceSetting          = ArkamysConfigRNAIVI::ARKAMYS_SOURCE_INVALID;


   /*
    * Initialize Current Feature Settings
    */
   m_u8FMEnhancerEnabled   = ArkamysConfigRNAIVI::STATE_UNKNOWN;
   m_bFMEnhancerAvailable  = true;
   m_u8VirtualSubEnabled   = ArkamysConfigRNAIVI::STATE_UNKNOWN;
   m_bVirtualSubAvailable  = false;
   m_u8SoundStageMode      = ArkamysConfigRNAIVI::STATE_UNKNOWN;
   m_bFMEnhancerLinearMode = false;


   /*
    * Initialize pointers
    */
   m_pPO                         = InternalCommunicationAdapter::getInstance();
   m_poMainAppl                  = OSAL_NULL;
   m_poResultHandler             = OSAL_NULL;

   m_pEolInterface               = ArkamysEolInterfaceRNAIVI::getInstance();
   m_pADRInterface               = OSAL_NEW ArkamysADRInterface();
   m_bDeleteArkamysADRInterface  = true;
   m_pArkamysCommInterface       = OSAL_NEW ArkamysCommInterface();
   m_bDeleteArkamysCommInterface = true;

   /*
    * Initialize misc members
    */
   m_enState            = ARKAMYS_FEATURE_STATE_IDLE;
   m_u8MuteRegId        = 0;
   m_hTimer             = OSAL_C_INVALID_HANDLE;

   /*
    * Initialize auditorium members
    */
   m_enRequestedAuditoriumAmbianceMode   = ArkamysConfigRNAIVI::ARKAMYS_AMBIANCE_FLAT;
   m_enCurAuditoriumAmbianceMode         = m_enRequestedAuditoriumAmbianceMode;
   m_u8FMEnhancer2Enabled    = ArkamysConfigRNAIVI::STATE_UNKNOWN;
   m_bFMEnhancer2Available   = false;
   m_u8AGCEnabled            = ArkamysConfigRNAIVI::STATE_UNKNOWN;
   m_bAGCAvailable           = false;
   m_s16InputLeveDeviation   = 0;
   m_bEntStreamMuted = false;

}

// Lint does not recognize the delete flags (i.e. m_bDeleteArkamysCommInterface), so it thinks that the pointer is not freed correctly.
//lint -save -e1740 pointer member 'ArkamysFeatureHandlerRNAIVI::m_pArkamysCommInterface' not directly freed or zeroed by destructor
ArkamysFeatureHandlerRNAIVI::~ArkamysFeatureHandlerRNAIVI()
{
   InternalCommunicationAdapter::getInstance()->POMessages->DeRegisterObserver(this);
   m_pPO             = OSAL_NULL;
   m_poMainAppl      = OSAL_NULL;
   m_pEolInterface   = OSAL_NULL;
   m_poResultHandler = OSAL_NULL;

   if (m_bDeleteArkamysCommInterface)
   {
      OSAL_DELETE m_pArkamysCommInterface;
      m_pArkamysCommInterface = OSAL_NULL;
   }

   if (m_bDeleteArkamysADRInterface)
   {
      OSAL_DELETE m_pADRInterface;
      m_pADRInterface = OSAL_NULL;
   }

   // delete timer
   if (m_hTimer != OSAL_C_INVALID_HANDLE)
   {
      (void)OSAL_s32TimerDelete(m_hTimer);
      m_hTimer = OSAL_C_INVALID_HANDLE;
   }


}
//lint -restore

void ArkamysFeatureHandlerRNAIVI::vTimerCallback(tPVoid pvArg)
{
   (tVoid)pvArg;    /* unused parameter */
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::vTimerCallback"));
   fc_audiomanager_tclApp::vEventFromOtherThreads(FC_AUDIOMANAGER_EVENT_ID_ARKAMYS);
}

tenArkamysMode ArkamysFeatureHandlerRNAIVI::enGetAudioMode()
{
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::enGetAudioMode"));
  return m_enArkamysMode;
}

void ArkamysFeatureHandlerRNAIVI::vSetAudioProcAudioMode(tenArkamysMode enArkamysMode)
{
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::vSetAudioProcAudioMode entered with mode %i", ETG_CENUM(tenArkamysMode, enArkamysMode)));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poMainAppl);
  tU32 u32AudioMode = 0;//classic by default
  switch(enArkamysMode)
  {
  case EN_ARKAMYS_CLASSIC :
  case EN_ARKAMYS_AUDITORIUM:
  case EN_ARKAMYS_PREMIUM1:
    u32AudioMode = enArkamysMode - 2;
    break;
  case EN_ARKAMYS_PREMIUM2:
    u32AudioMode = enArkamysMode - 3;
    break;
  default :
    //u32AudioMode = 0;//classic
    break;
  }
  tBool bretval = m_poMainAppl->poDBus()->poGetAudioProcProxy()->set_aplAudioMode(u32AudioMode);
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::Audio process Audio mode property set returned with %d", bretval));
}


ArkamysConfigRNAIVI::tenArkamysAmplifier ArkamysFeatureHandlerRNAIVI::enGetKDSAmplifierInformation()
{
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::enGetKDSAmplifierInformation"));
     //read item from the KDS key's bytes- stream

  ArkamysConfigRNAIVI::tenArkamysAmplifier enArkamysAmplifier = ArkamysConfigRNAIVI::ARKAMYS_AMP_INTERNAL_AMPLIFIER;

  tU8 u8Amplifier = 0; //initializing with internal amplifier

  if((DP_S32_NO_ERR == DP_s32GetConfigItem("SystemConfiguration1","OutputInformation", &u8Amplifier, 1)))
  {
    ETG_TRACE_USR4(("Amplifier value  : %d", u8Amplifier));

    switch(u8Amplifier)
    {
    case INTERNALAMP_2SPEAKER :
    case EXTERNAL_ANC :
    case INTERNALAMP_4SPEAKER:
    case EXTERNAL_ANC_FRONT_ONLY:
      enArkamysAmplifier = ArkamysConfigRNAIVI::ARKAMYS_AMP_INTERNAL_AMPLIFIER;
      break;
    case PREMIUM_AMP1:
    case EXTERNALAMP_BOSEWOMCAN:
    case EXTERNAL_DAIMLER_1:
    case EXTERNAL_DAIMLER_2:
    case BOSE_PERSONAL_SPACE:
      enArkamysAmplifier =  ArkamysConfigRNAIVI::ARKAMYS_AMP_PREMIUM_1;
      break;
    case PREMIUM_AMP2:
    case EXTERNALAMP_BOSE2CH_MCAN:
    case EXTERNALAMP_BOSE_5_1CH_MCAN:
      enArkamysAmplifier = ArkamysConfigRNAIVI::ARKAMYS_AMP_PREMIUM_2;
      break;
    default :
      break;
    }
  }
  else
  {
    ETG_TRACE_ERR(("Error in diagnosis read "));
    //return 0;
  }

  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI amplifier %i", ETG_CENUM(ArkamysConfigRNAIVI::tenArkamysAmplifier, enArkamysAmplifier)));

  return enArkamysAmplifier;

}

ArkamysConfigRNAIVI::tenArkamysDeviceVariant ArkamysFeatureHandlerRNAIVI::enGetKDSLibInformation()
{
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::enGetKDSLibInformation"));
     //read item from the KDS key's bytes- stream

  ArkamysConfigRNAIVI::tenArkamysDeviceVariant enArkamysDevVariant = ArkamysConfigRNAIVI::ARKAMYS_DEV_ALL_OTHER;

  tU8 u8ArkamysClassicLib = 0; //disabled
  tBool bArkamysClassicEnabled = false;//disabled
  if((DP_S32_NO_ERR == DP_s32GetConfigItem("CMVariantCoding","ArkamysClassicSupport", &u8ArkamysClassicLib, 1)))
  {
    ETG_TRACE_USR4(("Arkamys classic support  : %d", u8ArkamysClassicLib));
    if(u8ArkamysClassicLib)
      bArkamysClassicEnabled = true;
  }
  else
  {
    ETG_TRACE_ERR(("Error in diagnosis read "));
    //return 0;
  }

  tU8 u8ArkamysAudiLib = 0; //disabled
    tBool bArkamysAuditoriumEnabled = false;//disabled
  if((DP_S32_NO_ERR == DP_s32GetConfigItem("CMVariantCoding","ArkamysAuditoriumSupport", &u8ArkamysAudiLib, 1)))
  {
    ETG_TRACE_USR4(("Arkamys auditorium support  : %d", u8ArkamysAudiLib));
    if(u8ArkamysAudiLib)
      bArkamysAuditoriumEnabled = true;
  }
  else
  {
    ETG_TRACE_ERR(("Error in diagnosis read "));
    //return 0;
  }


  //logic to detemine the device variant
  if(bArkamysClassicEnabled && (!bArkamysAuditoriumEnabled))
    enArkamysDevVariant =  ArkamysConfigRNAIVI::ARKAMYS_DEV_CLASSIC_ONLY;


  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI device variant %i", ETG_CENUM(ArkamysConfigRNAIVI::tenArkamysDeviceVariant, enArkamysDevVariant)));

  return enArkamysDevVariant;

}

ArkamysConfigRNAIVI::tenArkamysEOLLib ArkamysFeatureHandlerRNAIVI::enGetArkamysEOLLibType()
{
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::enGetArkamysEOLLibType"));
     //read item from the KDS key's bytes- stream

  ArkamysConfigRNAIVI::tenArkamysEOLLib enArkamysEOLLib = ArkamysConfigRNAIVI::ARKAMYS_EOL_INVALID;

  tU32 ArkamysEOLLibType = ntohl(ArkamysEolInterfaceRNAIVI::getInstance() -> getEOLLibType());

  if(ArkamysEOLLibType == ARKAMYS_LIBTYPE_CLASSIC)
    enArkamysEOLLib = ArkamysConfigRNAIVI::ARKAMYS_EOL_CLASSIC;
  else if(ArkamysEOLLibType == ARKAMYS_LIBTYPE_AUDITORIUM)
    enArkamysEOLLib = ArkamysConfigRNAIVI::ARKAMYS_EOL_AUDITORIUM;

  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI EOL Lib type %i", ETG_CENUM(ArkamysConfigRNAIVI::tenArkamysEOLLib, enArkamysEOLLib)));

  return enArkamysEOLLib;

}

tU32 ArkamysFeatureHandlerRNAIVI::u32GetIMXLibVersion()
{
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::u32GetIMXLibVersion"));
  AUD_POINTER_CHECK_CALL(m_poMainAppl);
    tU32 u32Version = 0;

    std::vector<tS32>  version(3, 0);
    version = m_poMainAppl->poDBus()->poGetAudioProcProxy()->get_aplArkamysLibraryVersion();

    if(version.size() == 3)
    {
      u32Version  = version[0];
      u32Version <<= 8;
      u32Version |= version[1];
      u32Version <<= 8;
      u32Version |= version[2];
    }
    else
      ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::wrong size of lib version %d", version.size()));

    ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::IMXLibVersion is 0x%4x", u32Version));
  return u32Version;
}

void ArkamysFeatureHandlerRNAIVI::vSetArkamysMode()
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::vSetArkamysMode"));

   if((!m_bAudioProcAvailable) || (m_u32ReportedArkamysLib == 0))//wait until audio process and arkamys library version is available
     return;

   tBool bLibVersionMatch = true;
   tU32 ArkamysEOLADR3LibVersion = ntohl(ArkamysEolInterfaceRNAIVI::getInstance() -> getEOLADR3LibVersion());

   //since two different versions are compatible check EOL and ADR reported lib version against hardcoded values
   if((ArkamysEOLADR3LibVersion < ARKAMYS_ADR3_LIB_MIN_VERSION) || (ArkamysEOLADR3LibVersion > ARKAMYS_ADR3_LIB_MAX_VERSION))
   {
     bLibVersionMatch = false;
   }

   if((m_u32ReportedArkamysLib < ARKAMYS_ADR3_LIB_MIN_VERSION) || (m_u32ReportedArkamysLib > ARKAMYS_ADR3_LIB_MAX_VERSION))
   {
     bLibVersionMatch = false;
   }

   //in case of auditorium library check IMX version also
   if( (ntohl(ArkamysEolInterfaceRNAIVI::getInstance()->getEOLLibType()) == ARKAMYS_LIBTYPE_AUDITORIUM) )// auditorium lib
   {
     ETG_TRACE_USR2(("ArkamysFeatureHandlerRNAIVI::vSetArkamysMode auditorium lib"));
     tU32 ArkamysEOLIMXLibVersion  = ntohl(ArkamysEolInterfaceRNAIVI::getInstance() -> getEOLIMXLibVersion());
     if((ArkamysEOLIMXLibVersion < ARKAMYS_IMX_LIB_MIN_VERSION) || (ArkamysEOLIMXLibVersion > ARKAMYS_IMX_LIB_MAX_VERSION))
     {
       bLibVersionMatch = false;
     }
     tU32 ArkamysIMXLibVersion = u32GetIMXLibVersion();
     if((ArkamysIMXLibVersion < ARKAMYS_IMX_LIB_MIN_VERSION) || (ArkamysIMXLibVersion > ARKAMYS_IMX_LIB_MAX_VERSION))
     {
       bLibVersionMatch = false;
     }
   }
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::vSetArkamysMode EOL version and library version match (%s)", bLibVersionMatch? "true":"false"));

   m_enArkamysMode = ArkamysStateCalculatorRNAIVI::enCalculateArkamysMode(enGetKDSAmplifierInformation(),
                                                                       enGetArkamysEOLLibType(),
                                       bLibVersionMatch,
                                       enGetKDSLibInformation()
                                       );


   //taken from audio process - override arkamys-mode if file is present
   FILE            *pFile = NULL;                  /* pointer to config data file */
   const char      *IsAuditorium = "/var/opt/bosch/dynamic/audiomanager/audproc_force_auditorium.txt";

   /***************************************************
    * force system configuration from file if present
    */
   pFile = fopen(IsAuditorium, "r");

   if(NULL != pFile)
   {
     m_enArkamysMode = EN_ARKAMYS_AUDITORIUM;
     fclose(pFile);
   }

   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI Arkamys Audio mode set to %i", ETG_CENUM(tenArkamysMode ,m_enArkamysMode)));

   tU8 u8ArkamysMode = m_enArkamysMode;
   if(u8ArkamysMode == EN_ARKAMYS_PREMIUM1 || u8ArkamysMode == EN_ARKAMYS_PREMIUM2)//for premium mode system should run in default classic config
     u8ArkamysMode = EN_ARKAMYS_CLASSIC;

   //update ADR3
   vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_ARKAMYS_MODE, VD_ADR3_OPTYPE_SET
        , 1, &u8ArkamysMode, ARKAMYS_MODE);//send data to ADR3

   //update EOL interface
   ArkamysEolInterfaceRNAIVI::getInstance()->setArkamysAudioMode(m_enArkamysMode);

   //update AudioProcess
   vSetAudioProcAudioMode(m_enArkamysMode);

   //update HMI
   AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poMainAppl);
   fc_audiomanager_tclService_Audio_Function *poAudioFunction = m_poMainAppl->pGetAudioFunctionptr();

   AUD_POINTER_CHECK_CALL_NO_RET_VALUE(poAudioFunction);
   poAudioFunction->updateClients(MIDW_MASCFFI_C_U16_AUDIOMODE);

   ID_ArkamysModeSet oArkamysModeSet(m_enArkamysMode);
   InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oArkamysModeSet);
}

/**
 * bStartup()
 * Perform startup checks and corresponding actions. To be called only once at the beginning of the lifecycle
 */
tBool ArkamysFeatureHandlerRNAIVI::bStartup(fc_audiomanager_tclApp* poMainAppl)
{
   AudioUserDpIf* pAudUsrDpIf = NULL;

   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::bStartup() entered."));

   ETG_TRACE_USR4(("Requesting Arkamys Lib from ADR..."));
   m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_GET_LIBVERSION, 0));
   m_ActionQueue.Event(AkActionRNAIVI::ARKAMYS_EVENT_CONTINUE);

   m_poMainAppl = poMainAppl;

   // create timer
   tS32 tmrRes = OSAL_s32TimerCreate((OSAL_tpfCallback)vTimerCallback, 0, &m_hTimer);
   if (tmrRes != OSAL_OK)
   {
      m_hTimer = OSAL_C_INVALID_HANDLE;
      ETG_TRACE_ERR(("ArkamysFeatureHandlerRNAIVI::bStartup(): OSAL_s32TimerCreate failed."));
   }


   // Load the Eol Data from the Datapool
   tBool bLoadEolSuccess = ArkamysEolInterfaceRNAIVI::getInstance() -> loadEolData();

   if (!bLoadEolSuccess)
   {
      ETG_TRACE_SYS_MIN(("ArkamysFeatureHandlerRNAIVI:bStartup() Loading of EOL data failed. Continuing without EOL data."));
      //return false;
   }


   // Register for Stream Mute
   m_u8MuteRegId = StreamMute::getInstance() -> registerForStreamMute(StreamMute_IF::INTERNAL_BEH, this);

   // Observe relevant PO messages
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_SetSource);
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_Settings);
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_ArkamysFeature);
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_DiagLinearMode);
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_CCAStart_AudDiagMode);
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_NotifyStartup);
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_NotifyAudioProcess);
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_CarSetting);
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_CCA_GetDiagResult);
   m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_NotifyMute);
   pAudUsrDpIf = AudioUserDpIfSelect::pGetAudUsrDpIf();

   if(pAudUsrDpIf)
   {
     tS16 s16VirtualSub = 0x00;//coverity fix
     tU8 u8Ambiance;

     pAudUsrDpIf->s32GetSubwoofer(s16VirtualSub);
     pAudUsrDpIf->s32GetRoomEffect(u8Ambiance);
     pAudUsrDpIf->s32GetAutomaticGainControl(m_bAGCAvailable);

     m_bVirtualSubAvailable = static_cast<tBool>(s16VirtualSub & 0x01);
     m_enRequestedAuditoriumAmbianceMode = static_cast< ArkamysConfigRNAIVI::tenAuditoriumAmbiance>(u8Ambiance);
   }

   // Since some settings may have changed, update and apply states
   vUpdateStatesAndApply();

   ETG_TRACE_USR1(("ArkamysFeatureHandlerRNAIVI:bStartup() done."));

   return true;
}

void ArkamysFeatureHandlerRNAIVI::vUpdateStatesAndApply()
{
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pEolInterface);
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::vUpdateStatesAndApply() entered."));

  // check if Arkamys Feature should be enabled
  // if invalid source check is removed then the same should be handled in the state calculator
  if (
    (!m_bAudioProcAvailable) ||
    ((m_u32ReportedArkamysLib < ARKAMYS_ADR3_LIB_MIN_VERSION) || (m_u32ReportedArkamysLib > ARKAMYS_ADR3_LIB_MAX_VERSION)) ||
    (m_enCurSourceSetting == ArkamysConfigRNAIVI::ARKAMYS_SOURCE_INVALID)
     )
  {
    ETG_TRACE_USR2(("Arkamys Feature is disabled -> Not checking for state changes."));
    ETG_TRACE_USR2(("    EOL Arkamys Lib: %x",ntohl( ArkamysEolInterfaceRNAIVI::getInstance() -> getEOLADR3LibVersion())));
    ETG_TRACE_USR2(("    ADR Arkamys Lib: %i", m_u32ReportedArkamysLib));
    ETG_TRACE_USR2(("    EOL Arkamys Lib type: %x", ntohl(ArkamysEolInterfaceRNAIVI::getInstance() -> getEOLLibType())));
    ETG_TRACE_USR2(("    Audio Process proxy Available: %s", m_bAudioProcAvailable?"yes":"no"));
    ETG_TRACE_USR2(("    Arkamys source: %i", ETG_CENUM(ArkamysConfigRNAIVI::tenSrcSetting, m_enCurSourceSetting)));
    return;
  }

  /*******************classic features****************/
  tBool bSSS4       = bUpdateSSS4State();
  tBool bVSubwoofer = bUpdateVirtualSubwooferState();

  /*******************auditorium features*************/
  tBool bAmbiance      = bUpdateAmbianceState();
  tBool bFMEnhancer2   = bUpdateFMEnhancer2State();
  tBool bAGC           = bUpdateAGCState();

    //common feature
  tBool bSourceEq = bUpdateFMEnhancerState();

  // Bail out when nothing to do
  if (!bSSS4  && !bVSubwoofer && !bSourceEq && !bAmbiance && !bFMEnhancer2 && !bAGC)
  {
    ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::vUpdateStatesAndApply(): nothing to do."));
    return;
  }


  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_MUTE, 0));
  if(!m_bEntStreamMuted)//if stream is already muted then do not wait for mute status
    m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_MUTED));  // Wait until muted


  //bypass in ADR3
  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_MODE, EN_ARKAMYS_BYPASS));   // Start bypass
  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));  // Wait until adr ok


  tBool bActionsAdded = false;

  //fm enhancer1
  if (bSourceEq)
  {
    addFMEnhancerToActionQueue(m_u8FMEnhancerEnabled);
    bActionsAdded = true;
  }

  //if diagnosis session is active or source is set to phone,speech or arkamys is disabled--> set only arkamys bypass
  if((m_enArkamysMode == EN_ARKAMYS_BYPASS)|| ((m_bEngModeActive) && (m_bFMEnhancerLinearMode)) || (m_enCurSourceSetting == ArkamysConfigRNAIVI::ARKAMYS_SOURCE_OTHER) )
  {
    //bypass in IMX
    m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_BYPASS_AUD_PROC, true));   // activate bypass
    bActionsAdded = true;

  }

  else
  {
    //bypass in IMX
    if(m_bEngModeActive)
    {
      m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_BYPASS_AUD_PROC, true));   // activate bypass in case of linear mode
    }
    else
    {
      m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_BYPASS_AUD_PROC, false));   // deactivate bypass
      //bActionsAdded = true;
    }

    /*********************classic only***************************/
    //sound staging
    if (bSSS4)
    {
      // update SSS4
      addSSS4ToActionQueue(m_u8SoundStageMode);
      bActionsAdded = true;
    }

    //bass enhancer
    if (bVSubwoofer)
    {
      addVirtualSubwooferToActionQueue(m_u8VirtualSubEnabled);
      bActionsAdded = true;
    }


    /*********************auditorium only*************************/
    //imx library update - auto gain control
    if (bAGC)
    {
      addAGCAudProcToActionQueue(m_u8AGCEnabled);
      bActionsAdded = true;
    }

    //fm enhancer2
    if (bFMEnhancer2)
    {
      addFMEnhancer2ToActionQueue(m_u8FMEnhancer2Enabled);
      bActionsAdded = true;
    }

    //ambiance
    if (bAmbiance)
    {
      // update SSS4
      addAmbianceToActionQueue(m_enCurAuditoriumAmbianceMode);//adr3 lib
      addAmbianceAudProcToActionQueue(m_enCurAuditoriumAmbianceMode);//imx lib
      bActionsAdded = true;
    }

    //special case for FMEnhancer2 wherein the values for FMEnhancer2 should remain unchanged during linear audio mode
    //triggered from developer test menu & should not be enabled during diagnosis session
    if((m_enArkamysMode == EN_ARKAMYS_AUDITORIUM) && (m_bEngModeActive))
    {
      addFMEnhancer2ToActionQueue(m_u8FMEnhancer2Enabled);
      bActionsAdded = true;
    }

    //start classic or auditorium or premium
    m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_MODE, m_enArkamysMode));
    m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));  // Wait until adr ok

  }

  // Demute
  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_DEMUTE, 0));

  if (!bActionsAdded)
  {
    // Cancel queue if nothing was added
    ETG_TRACE_USR4(("Cancel ActionQueue (no relevant changes)."));
    m_ActionQueue.Cancel();
  }
  else
  {
    // Start Queue
    m_ActionQueue.Event(AkActionRNAIVI::ARKAMYS_EVENT_CONTINUE);
  }
}

/**
 * setLinearMode()
 * @param bLinearModeEnable set true to enable linear mode, false to disable
 * @description While linear mode is active, the Arkamys features are switched off or set to a flat configuration
 */
void ArkamysFeatureHandlerRNAIVI::setLinearMode(tBool bLinearModeEnable)
{
   m_bEngModeActive = bLinearModeEnable;
   vUpdateStatesAndApply();
}

/**
 *setSource()
 *@param enSource sets the source info based on primary, secondary or other sources
 *@description source information is set inside arkamys module
 */
void ArkamysFeatureHandlerRNAIVI::setSource(ArkamysConfigRNAIVI::tenSrcSetting enSource)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::setSource entered with source %i.", ETG_CENUM(ArkamysConfigRNAIVI::tenSrcSetting, enSource)));

   if(enSource == ArkamysConfigRNAIVI::ARKAMYS_SOURCE_INVALID) // do nothing for an invalid source
    return;

   m_enCurSourceSetting = enSource;
   vUpdateStatesAndApply();
}

/**
 *setFMEnhancer()
 *@description setting of FM enhancer filters in arkamys mode
 */
void ArkamysFeatureHandlerRNAIVI::setFMEnhancer(tBool bOn)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::setFMEnhancer(%s)", bOn ? "On" : "Off"));
   //both FM enhancer 1 and FM enhancer2 has to be changed
   m_bFMEnhancerAvailable = bOn;
   vUpdateStatesAndApply();
}

/**
 *setVirtualSubwoofer()
 *@description setting of virtual subwoofer/bass enhancer in arkamys classic mode
 */
void ArkamysFeatureHandlerRNAIVI::setVirtualSubwoofer(tBool bOn)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::setVirtualSubwoofer(%s)", bOn ? "On" : "Off"));

   m_bVirtualSubAvailable = bOn;
   vUpdateStatesAndApply();
}

/**
 *Mute()
 *@description setting of mute to ADR3
 */
void ArkamysFeatureHandlerRNAIVI::Mute()
{
   AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pArkamysCommInterface);
   m_pArkamysCommInterface -> Mute(m_u8MuteRegId);
}

/**
 *Demute()
 *@description setting of demute to ADR3
 */
void ArkamysFeatureHandlerRNAIVI::Demute()
{
   AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pArkamysCommInterface);
   m_pArkamysCommInterface -> Demute(m_u8MuteRegId);
}


/**
 *AdrMessageRx()
 *@description receives messages sent from ADR for function ID FE0(ArkamysParameter)
 */
//void ArkamysFeatureHandlerRNAIVI::AdrMessageRx(tU8 cmd, tU32 adr, tU32 len, tS32* data)
void ArkamysFeatureHandlerRNAIVI::AdrMessageRx(ID_ArkamysParameter arkamysParameter)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::AdrMessageRx(cmd:%i adr:%08X len:%08X data[0]:%08X)",
         ETG_CENUM(ArkamysMessage::tenAdrOpType, arkamysParameter.arkamysCommand), arkamysParameter.arkamysAddress, arkamysParameter.arkamysLength, arkamysParameter.arkamysValues[0]));
   if (arkamysParameter.arkamysCommand == ArkamysStaticMessage::ArkamysOpTypeStatus)
   {
      m_ActionQueue.Event(AkActionRNAIVI::ARKAMYS_EVENT_DATA_RECEIVED);

      // Check if lib version was received
      if (ntohl(arkamysParameter.arkamysAddress) == ARKAMYS_LIB_VERSION_ARM_ADR && ntohl(arkamysParameter.arkamysLength) == 1)
      {
         m_u32ReportedArkamysLib = ntohl(arkamysParameter.arkamysValues[0]);
         ETG_TRACE_USR4(("ADR Arkamys Library version - EOL format %i (0x%X).", m_u32ReportedArkamysLib, m_u32ReportedArkamysLib));

         vSetArkamysMode();//set arkamys mode in adr3, hmi and audioprocess
         vUpdateStatesAndApply();
      }
      else
      {
    ETG_TRACE_USR1(("ArkamysFeatureHandlerRNAIVI::AdrMessageRx Parameter is not relevant"));
      }
   }
}

/**
 *vStartTimer()
 *@description starting of osal timer
 */
void ArkamysFeatureHandlerRNAIVI::vStartTimer(OSAL_tMSecond milliseconds)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::vStartTimer(%i)", milliseconds));
   if (m_hTimer != OSAL_C_INVALID_HANDLE)
   {
      tS32 tmrRes = OSAL_s32TimerSetTime(m_hTimer, milliseconds, 0);
      if (tmrRes != OSAL_OK)
      {
         ETG_TRACE_ERR(("ArkamysFeatureHandlerRNAIVI::vStartTimer(): OSAL_s32TimerSetTime failed."));
      }
   }
}

/**
 *vStopTimer()
 *@description stopping of osal timer
 */
void ArkamysFeatureHandlerRNAIVI::vStopTimer()
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::vStopTimer()"));
   if (m_hTimer != OSAL_C_INVALID_HANDLE)
   {
      tS32 tmrOk = OSAL_s32TimerSetTime(m_hTimer, 0, 0);
      if (tmrOk != OSAL_OK)
      {
         ETG_TRACE_ERR(("ArkamysFeatureHandlerRNAIVI::vStopTimer(): OSAL_s32TimerSetTime failed."));
      }
   }
}

/**
 *timerExpired()
 *@description callback function for timer expiry
 */
void ArkamysFeatureHandlerRNAIVI::timerExpired(tPVoid pvArg)
{
   pvArg = pvArg; // not used
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::timerExpired(%p)", pvArg));
   m_ActionQueue.Event(AkActionRNAIVI::ARKAMYS_EVENT_TIMEOUT);
}

/**
 *vAdrOkCallback()
 *@description ADR3 callback for data connection availability
 */
void ArkamysFeatureHandlerRNAIVI::vAdrOkCallback()
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::adrOkCallback(). TID = %i", OSAL_ThreadWhoAmI()));
   if (m_pADRInterface != OSAL_NULL)
   {
      m_pADRInterface -> vAdrOkCallback();
   }
   else
   {
      ETG_TRACE_ERR(("m_pArkamysADRInterface == OSAL_NULL"));
   }
   m_ActionQueue.Event(AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK);
}

/**
 *vResetArkamysSettingsFlags()
 *@description to trigger resetting of arkamys features during ADR reset
 */
void ArkamysFeatureHandlerRNAIVI::vResetArkamysSettingsFlags()
{
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::vResetArkamysSettingsFlags().called "));
  //disable classic settings  for recalculation of states
  m_u8FMEnhancerEnabled = ArkamysConfigRNAIVI::STATE_UNKNOWN;
  m_u8VirtualSubEnabled = ArkamysConfigRNAIVI::STATE_UNKNOWN;
  m_u8SoundStageMode = ArkamysConfigRNAIVI::STATE_UNKNOWN;
  //disable auditorium settings  for recalculation of states
  m_u8AGCEnabled = ArkamysConfigRNAIVI::STATE_UNKNOWN;
  m_u8FMEnhancer2Enabled = ArkamysConfigRNAIVI::STATE_UNKNOWN;
  m_enCurAuditoriumAmbianceMode = ArkamysConfigRNAIVI::ARKAMYS_AMBIANCE_FLAT;
  vUpdateStatesAndApply();
}

/**
 *MessageNotification()
 *@description receives P.O messages
 */
void ArkamysFeatureHandlerRNAIVI::MessageNotification(PO_MessageConfig::enID MsgId)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::MessageNotification(%d)", ETG_CENUM(PO_MessageConfig::enID, MsgId)));

   if (MsgId == PO_MessageConfig::ID_SetSource)
   {
      const MSG_SetSource* pMsg = m_pPO->POMessages->QueryMessage<MSG_SetSource>(MsgId);
      ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI: Source setting received: Stream:%i Type:%i"
            , ETG_CENUM(tenStream, pMsg -> enStream), ETG_CENUM(tenSourceType, pMsg -> enSourceType)));

      // only for FM Enhancer 2 in Auditorium mode
       vSetFMEnhancer2(pMsg -> enSourceType);

      setSource(ArkamysConfigRNAIVI::getArkamysSource(pMsg -> enStream, pMsg -> enSourceType));
   }

   else if (MsgId == PO_MessageConfig::ID_ArkamysFeature)
   {
      const ID_ArkamysFeature* pMsg = m_pPO->POMessages->QueryMessage<ID_ArkamysFeature>(MsgId);
      ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI: Message ID_ArkamysFeature(%i, %i) received."
            , ETG_CENUM(ArkamysConfigRNAIVI::tenArkamysFeature, pMsg -> enArkamysFeature)
            , pMsg -> u16ArkamysParameter));

      switch (pMsg -> enArkamysFeature)
      {
      case ArkamysConfigRNAIVI::ARKAMYS_FEATURE_SSS4:
        m_u8SoundStageMode = static_cast<tU8>(pMsg -> u16ArkamysParameter);
         vUpdateStatesAndApply();
         break;
      case ArkamysConfigRNAIVI::ARKAMYS_FEATURE_VIRTUAL_SUBWOOFER:
         m_u8VirtualSubEnabled = pMsg -> u16ArkamysParameter != 0;
         setVirtualSubwoofer(m_u8VirtualSubEnabled);
        // m_ActionQueue.Event(AkActionRNAIVI::ARKAMYS_EVENT_CONTINUE);
        // vUpdateStatesAndApply();
         break;

      default:
         break;
      }
   }

   else if (MsgId == PO_MessageConfig::ID_Settings)
   {
      const MSG_Settings* pMsg = m_pPO->POMessages->QueryMessage<MSG_Settings>(MsgId);
      ETG_TRACE_USR2(("ArkamysFeatureHandlerRNAIVI: ID_Settings received with type: %i and parameter: %i.", ETG_CENUM(tenSettingType, pMsg -> enSettingType), pMsg -> s16Steps));
      switch (pMsg -> enSettingType)
      {
      default:
         ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI: Setting %i ignored (not relevant for Arkamys handling).", ETG_CENUM(tenSettingType, pMsg -> enSettingType)));
         break;

      case EN_AUDIO_SETTINGS_VIRTUAL_SUB_WOOFER :
          ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI: Settings Virtual Sub woofer = %i", pMsg -> s16Steps));
          setVirtualSubwoofer(static_cast<tBool>(pMsg -> s16Steps));
          break;
      case EN_AUDIO_SETTINGS_FM_ENHANCER:
          ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI: Settings FM Enhancer = %i", pMsg -> s16Steps));
          setFMEnhancer(static_cast<tBool>(pMsg -> s16Steps));
          break;
      case EN_AUDIO_SETTINGS_ROOM_EFFECT:
            {
               ArkamysConfigRNAIVI::tenAuditoriumAmbiance ambMode =
                 (pMsg -> s16Steps == AM_MSP_SND_ROOM_EFFECT_LEVEL0) ? ArkamysConfigRNAIVI::ARKAMYS_AMBIANCE_NATURAL :
                     (pMsg -> s16Steps == AM_MSP_SND_ROOM_EFFECT_LEVEL1) ? ArkamysConfigRNAIVI::ARKAMYS_AMBIANCE_LIVE :
                     (pMsg -> s16Steps == AM_MSP_SND_ROOM_EFFECT_LEVEL2) ? ArkamysConfigRNAIVI::ARKAMYS_AMBIANCE_CLUB :
                     (pMsg -> s16Steps == AM_MSP_SND_ROOM_EFFECT_LEVEL3) ? ArkamysConfigRNAIVI::ARKAMYS_AMBIANCE_LOUNGE :
                     ArkamysConfigRNAIVI::ARKAMYS_AMBIANCE_FLAT;
               ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI: Ambiance Setting %i received and translated to %i.", pMsg -> s16Steps, ETG_CENUM( ArkamysConfigRNAIVI::tenAuditoriumAmbiance, ambMode)));
               vSetAmbiance(ambMode);
            }
               break;

      case EN_AUDIO_SETTINGS_AUTO_GAIN_CONTROL:
          ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI: Settings AGC = %i", pMsg -> s16Steps));
          vSetAGCMode(static_cast<tBool>(pMsg -> s16Steps));
          break;

      }
   }
   else if (MsgId == PO_MessageConfig::ID_NotifyStartup)
   {
      ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI: NotifyStartup received."));

      const ID_NotifyStartup* pMsg = InternalCommunicationAdapter::getInstance() -> POMessages -> QueryMessage<ID_NotifyStartup>(MsgId);
    if(pMsg -> enOn == EN_AUDIO_ADR3_AVAILABLE)
    {
      //resend arkamys classic information due to ADR reset
      vResetArkamysSettingsFlags();
    }
   }
   else if (MsgId == PO_MessageConfig::ID_DiagLinearMode || MsgId == PO_MessageConfig::ID_CCAStart_AudDiagMode)
   {
      enum
      {
         AK_LINEAR_MODE_NO_CHANGE,
         AK_LINEAR_MODE_ON,
         AK_LINEAR_MODE_OFF
      } enLinMode = AK_LINEAR_MODE_NO_CHANGE;

      if (MsgId == PO_MessageConfig::ID_DiagLinearMode)
      {
         const ID_DiagLinearMode*         pMsg_LinModePhase = m_pPO->POMessages->QueryMessage<ID_DiagLinearMode>(MsgId);
         enLinMode = (pMsg_LinModePhase->enDiagLinearModePhase == EN_LINEARMODE_SET) ? AK_LINEAR_MODE_ON :
                     (pMsg_LinModePhase->enDiagLinearModePhase == EN_LINEARMODE_UNFREEZE) ? AK_LINEAR_MODE_OFF :
                     AK_LINEAR_MODE_NO_CHANGE;
         //handling of FM enhancer during linear mode. ToDO : need to create a new PO message to handle
         m_bFMEnhancerLinearMode = (pMsg_LinModePhase->enDiagLinearModePhase == EN_LINEARMODE_SET) ? true :
                                   (pMsg_LinModePhase->enDiagLinearModePhase == EN_LINEARMODE_UNFREEZE) ? false :
                                   m_bFMEnhancerLinearMode;

      }


      else if (MsgId == PO_MessageConfig::ID_CCAStart_AudDiagMode)
      {
         const ID_CCAStart_AudDiagMode*   pMsg_DiagMode     = m_pPO->POMessages->QueryMessage<ID_CCAStart_AudDiagMode>(MsgId);
         enLinMode = (pMsg_DiagMode -> DiagAud == midw_fi_tcl_e8_DiagAudMode::FI_EN_AUDIO_DIAG_AUD_MODE_LINEAR_ON) ?  AK_LINEAR_MODE_ON :
                     (pMsg_DiagMode -> DiagAud == midw_fi_tcl_e8_DiagAudMode::FI_EN_AUDIO_DIAG_AUD_MODE_LINEAR_OFF) ?  AK_LINEAR_MODE_OFF :
                     AK_LINEAR_MODE_NO_CHANGE;
      }


      if (enLinMode == AK_LINEAR_MODE_ON)
      {
         setLinearMode(true);
      }
      else if (enLinMode == AK_LINEAR_MODE_OFF) //if (pMsg_LinModePhase->enDiagLinearModePhase == EN_LINEARMODE_UNFREEZE)
      {
         setLinearMode(false);
      }
      else
      {
         ETG_TRACE_USR4(("Not changing linear mode."));
      }
   }

  else if (MsgId == PO_MessageConfig::ID_NotifyAudioProcess)
   {
       ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI: PO message for AudioProcess available received"));
       m_bAudioProcAvailable = true;
       vSetArkamysMode();
   }

  else if(MsgId == PO_MessageConfig::ID_CarSetting)
  {
    ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI: PO message for Car speed received"));
    const ID_CarSetting*   pMsg_CarSpeedValue     = m_pPO->POMessages->QueryMessage<ID_CarSetting>(MsgId);
    vSendArkamysCurrentSpeedToAudProc(pMsg_CarSpeedValue->s16Value);
  }

  else if(MsgId == PO_MessageConfig::ID_CCA_GetDiagResult)
   {
    ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI: PO message for for DiagResult Recieved"));
     const ID_CCA_GetDiagResult*   pMsg_DiagResult    = m_pPO->POMessages->QueryMessage<ID_CCA_GetDiagResult>(MsgId);


     if(pMsg_DiagResult != OSAL_NULL)
     {
       m_poResultHandler = &(pMsg_DiagResult->roResultHandler);

       for (tU16 i=0; i < pMsg_DiagResult -> roRequestedITCs.size(); i++)
       {
         if(pMsg_DiagResult -> roRequestedITCs[i] == ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_07)
            vHandleGetDiagResult();
       }
     }
   }
  else if(MsgId == PO_MessageConfig::ID_NotifyMute)
  {
   const ID_NotifyMute* pMsg = m_pPO->POMessages->QueryMessage<ID_NotifyMute>(MsgId);
   AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pMsg);
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI :: ADR mute notification received  with action: %x and Stream: %x ."
     , pMsg->enMuteResult, pMsg->enStream));
   if(pMsg->enStream == EN_AUDIO_SOURCE_STREAM_MAIN)
   {
     if(pMsg->enMuteResult == EN_AUDIO_MUTE_MUTERESULT_MUTE)
       m_bEntStreamMuted = true;
     else
       m_bEntStreamMuted = false;
   }
  }

}


/**
 *vHandleGetDiagResult()
 *@description Handler for Diaglog property SENDNEXTTESTRESULT
 */
void ArkamysFeatureHandlerRNAIVI::vHandleGetDiagResult()
{

  ETG_TRACE_USR2(("ArkamysFeatureHandlerRNAIVI::vHandleGetDiagResult entered"));
  tTestResultList oFiTestResultObject;

  dp_tclArkamysDPArkamys_Eol oDpEol;
  if(oDpEol.bReloadDpElement())//reload dp data for ITC handling
    ArkamysEolInterfaceRNAIVI::getInstance()->loadEolData();

  tBool bresult;
  bresult = ArkamysEolInterfaceRNAIVI::getInstance()->isDataFromDatapool();//get datapool read status

  if (m_poResultHandler != OSAL_NULL)
  {
    tTestResult oTestData;
    // Set the parameters of the FI data object
    oTestData.u16ID = ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_07;

    if (bresult)
    {
      oTestData.enResult = EN_PASSED;
      ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::TroubleCode:0x%x, Diagnosis Test Result = PASSED",ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_07));
    }
    else
    {
      oTestData.enResult = EN_FAILED;
      ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::TroubleCode:0x%x, Diagnosis Test Result = FAILED",ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_07));
    }

    oFiTestResultObject.push_back(oTestData);
    m_poResultHandler->vSaveDiagResult(oFiTestResultObject);
    oFiTestResultObject.clear();
  }
  else
  {
    ETG_TRACE_ERR(("ArkamysFeatureHandlerRNAIVI::DiagLog result handler not set"));
  }
}


/**
 *addSSS4ToActionQueue()
 *@description adding sound staging to message queue
 */
void ArkamysFeatureHandlerRNAIVI::addSSS4ToActionQueue(tU8 u8On)
{
  if (m_enState != ARKAMYS_FEATURE_STATE_IDLE)
  {
    ETG_TRACE_USR3(("ArkamysFeatureHandlerRNAIVI::addSSS4ToActionQueue(): not executed. State not idle, but %i", ETG_CENUM(tenArkamysHandlerState, m_enState)));
    return;
  }

  if(u8On == ArkamysConfigRNAIVI::STATE_UNKNOWN)
  {
    ETG_TRACE_USR3(("ArkamysFeatureHandlerRNAIVI::addSSS4ToActionQueue(): not executed. Invalid request for Sound Stage: %i", ETG_CENUM(ArkamysConfigRNAIVI::tenSoundStageState, u8On)));
    return;
  }

   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::AddSSS4ToActionQueue(%i)", u8On));

       //pau4kor:workaround for ADR messages bigger than 1004 bytes of data, to be removed if the size is changed
      //In arkamys classic EOL, at offset 1814 the message is of size 2112 bytes.
      //These 2112 bytes will be sent as 3 messages to ADR3 as the limit for each message is 1004 bytes
      //However, the event queue will continue execution on receiving the ADR ok callback for 1st of the three messages. To prevent this, additional 3 Wait events are added.
   m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_SSS4, (u8On << 16) | 0));        // Set SS4 chunk
   m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));
   m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));
   m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));

   //4 offsets are sent in 1 msg starting from 2nd offset
   m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_SSS4, (u8On << 16) | 1));            // Set SSS4 chunk
   m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));

//   Reset command is not included in EOL data
 //  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SEND_RESET_CMD, 0));
 //  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));

}

/**
 *addVirtualSubwooferToActionQueue()
 *@description adding virtual subwoofer to message queue
 */
void ArkamysFeatureHandlerRNAIVI::addVirtualSubwooferToActionQueue(tU8 u8On)
{
  if (m_enState != ARKAMYS_FEATURE_STATE_IDLE)
  {
    ETG_TRACE_USR3(("ArkamysFeatureHandlerRNAIVI::addVirtualSubwooferToActionQueue(): not executed. State not idle, but %i", ETG_CENUM(tenArkamysHandlerState, m_enState)));
    return;
  }

  if(u8On == ArkamysConfigRNAIVI::STATE_UNKNOWN)
  {
    ETG_TRACE_USR3(("ArkamysFeatureHandlerRNAIVI::addVirtualSubwooferToActionQueue(): not executed. Invalid request for Bass Enhancer: %i", ETG_CENUM(ArkamysConfigRNAIVI::tenBassEnhancerState, u8On)));
    return;
  }

   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::addVirtualSubwooferToActionQueue(%i)", u8On));

   //improvement solution for NCG3D-120447. sending 2 bass enhancer msgs isntead of 4
   m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_VIRTUAL_SUB, (u8On << 16) | 0));        // Set chunk
   m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));

   m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_VIRTUAL_SUB, (u8On << 16) | 2));        // Set chunk
   m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));

 //  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SEND_RESET_CMD, 0));
//   m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));
}

/**
 *addFMEnhancerToActionQueue()
 *@description adding FM enhancer to message queue
 */
void ArkamysFeatureHandlerRNAIVI::addFMEnhancerToActionQueue(tU8 u8On)
{
  if (m_enState != ARKAMYS_FEATURE_STATE_IDLE)
  {
    ETG_TRACE_USR3(("ArkamysFeatureHandlerRNAIVI::addFMEnhancerToActionQueue(): not executed. State not idle, but %i", ETG_CENUM(tenArkamysHandlerState, m_enState)));
    return;
  }

  if(u8On == ArkamysConfigRNAIVI::STATE_UNKNOWN)
  {
    ETG_TRACE_USR3(("ArkamysFeatureHandlerRNAIVI::addFMEnhancerToActionQueue(): not executed. Invalid request for FM Enhancer: %i", ETG_CENUM(ArkamysConfigRNAIVI::tenFMEnhancerState, u8On)));
    return;
  }
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::addFMEnhancerToActionQueue(%i)", u8On));

   m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_FM_ENHANCER, u8On ));        // Set chunk
   m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));

}


/*
 * Auditorium related features
 */
/**
 *addAmbianceToActionQueue()
 *@description adding ambiance to message queue
 */
void ArkamysFeatureHandlerRNAIVI::addAmbianceToActionQueue(ArkamysConfigRNAIVI::tenAuditoriumAmbiance enAmbianceMode)
{
  if (m_enState != ARKAMYS_FEATURE_STATE_IDLE)
  {
    ETG_TRACE_USR3(("ArkamysFeatureHandlerRNAIVI::addAmbianceToActionQueue(): not executed. State not idle, but %i", ETG_CENUM(tenArkamysHandlerState, m_enState)));
    return;
  }

  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::addAmbianceToActionQueue(%i)", ETG_CENUM(ArkamysConfigRNAIVI::tenAuditoriumAmbiance, enAmbianceMode)));
       //pau4kor:workaround for ADR messages bigger than 1004 bytes of data, to be removed if the size is changed
      //In arkamys classic EOL, at offset 1814 the message is of size 2112 bytes.
      //These 2112 bytes will be sent as 3 messages to ADR3 as the limit for each message is 1004 bytes
      //However, the event queue will continue execution on receiving the ADR ok callback for 1st of the three messages. To prevent this, additional 3 Wait events are added.
  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_AMBIANCE, (enAmbianceMode << 16) | 0));        // Set chunk
  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));
  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));
  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));

  //4 offsets are sent in 1 msg starting from 2nd offset
  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_AMBIANCE, (enAmbianceMode << 16) | 1));        // Set chunk
  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));

  //2 offsets are sent in 1 msg starting from 5th offset
  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_AMBIANCE, (enAmbianceMode << 16) | 5));        // Set chunk
  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));

  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_AMBIANCE, (enAmbianceMode << 16) | 7));        // Set chunk
  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));

   //  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SEND_RESET_CMD, 0));
  //   m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));
}

/**
 *addAmbianceAudProcToActionQueue()
 *@description adding ambiance for audio process to message queue
 */
void ArkamysFeatureHandlerRNAIVI::addAmbianceAudProcToActionQueue(ArkamysConfigRNAIVI::tenAuditoriumAmbiance enAmbianceMode)
{
  if (m_enState != ARKAMYS_FEATURE_STATE_IDLE)
  {
    ETG_TRACE_USR3(("ArkamysFeatureHandlerRNAIVI::addAmbianceAudProcToActionQueue(): not executed. State not idle, but %i", ETG_CENUM(tenArkamysHandlerState, m_enState)));
    return;
  }
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::addAmbianceAudProcToActionQueue(%i)", ETG_CENUM(ArkamysConfigRNAIVI::tenAuditoriumAmbiance, enAmbianceMode)));
  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_AMBIANCE_AUD_PROC, enAmbianceMode));        // Set to audio process
}

/**
 *addAGCAudProcToActionQueue()
 *@description adding AGC for audio process to message queue
 */
void ArkamysFeatureHandlerRNAIVI::addAGCAudProcToActionQueue(tU8 u8On)
{
  if (m_enState != ARKAMYS_FEATURE_STATE_IDLE)
  {
    ETG_TRACE_USR3(("ArkamysFeatureHandlerRNAIVI::addAGCAudProcToActionQueue(): not executed. State not idle, but %i", ETG_CENUM(tenArkamysHandlerState, m_enState)));
    return;
  }

  if(u8On == ArkamysConfigRNAIVI::STATE_UNKNOWN)
  {
    ETG_TRACE_USR3(("ArkamysFeatureHandlerRNAIVI::addAGCAudProcToActionQueue(): not executed. Invalid request for AGC: %i", ETG_CENUM(ArkamysConfigRNAIVI::tenAGCState, u8On)));
    return;
  }

  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::addAGCAudProcToActionQueue(%i)", u8On));
  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_AGC_AUD_PROC, u8On));        // Set to audio process
}

/**
 *addFMEnhancer2ToActionQueue()
 *@description adding Fm enhancer 2 to message queue
 */
void ArkamysFeatureHandlerRNAIVI::addFMEnhancer2ToActionQueue(tU8 u8On)
{
  if (m_enState != ARKAMYS_FEATURE_STATE_IDLE)
  {
    ETG_TRACE_USR3(("ArkamysFeatureHandlerRNAIVI::addFMEnhancer2ToActionQueue(): not executed. State not idle, but %i", ETG_CENUM(tenArkamysHandlerState, m_enState)));
    return;
  }

  if(u8On == ArkamysConfigRNAIVI::STATE_UNKNOWN)
  {
    ETG_TRACE_USR3(("ArkamysFeatureHandlerRNAIVI::addFMEnhancer2ToActionQueue(): not executed. Invalid request for FM Enhancer2: %i", ETG_CENUM(ArkamysConfigRNAIVI::tenFMEnhancer2State, u8On)));
    return;
  }

  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SET_FM_ENHANCER2, (u8On << 16) | 0));        // Set chunk
  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));

   //  m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_SEND_RESET_CMD, 0));
  //   m_ActionQueue.Add(AkActionRNAIVI(AkActionRNAIVI::ARKAMYS_ACTION_WAIT_EVENT, AkActionRNAIVI::ARKAMYS_EVENT_ADR_OK));
}

/**
 *vSetFMEnhancer2()
 *@description setting of FM enhancer2 based on source
 */
tVoid ArkamysFeatureHandlerRNAIVI::vSetFMEnhancer2(tenSourceType enSourceType)
{
  ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI: vSetFMEnhancer2:SourceType:%i", ETG_CENUM(tenSourceType, enSourceType)));

    //fm enhancer 2 is used only for FM/DAB sources
  if((enSourceType == EN_AUDIO_SOURCE_SOURCETYPE_FM) || (enSourceType == EN_AUDIO_SOURCE_SOURCETYPE_DAB))
     m_bFMEnhancer2Available = true;

   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI: FM Enhancer2 set to : %d", m_bFMEnhancer2Available));

}

/**
 *vSetAGCMode()
 *@description setting of AGC based on HMI request
 */
tVoid ArkamysFeatureHandlerRNAIVI::vSetAGCMode(tBool bOn)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::setAGCMode(%s)", bOn ? "On" : "Off"));

   m_bAGCAvailable = bOn;
   vUpdateStatesAndApply();
}

/**
 *vSetAmbiance()
 *@description setting of ambiance based on HMI request
 */
tVoid ArkamysFeatureHandlerRNAIVI::vSetAmbiance(ArkamysConfigRNAIVI::tenAuditoriumAmbiance enAmbMode)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::vSetAmbiance(%i)",ETG_CENUM(ArkamysConfigRNAIVI::tenAuditoriumAmbiance, enAmbMode)));

   m_enRequestedAuditoriumAmbianceMode = enAmbMode;
   vUpdateStatesAndApply();
}

/**
 *vSendArkamysCurrentSpeedToAudProc()
 *@description setting of speed info in audio process
 */
tVoid ArkamysFeatureHandlerRNAIVI::vSendArkamysCurrentSpeedToAudProc(tS16 speedValue)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::vSendArkamysCurrentSpeedToAudProc(%i)", speedValue));
   AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poMainAppl);

   if(m_enArkamysMode != EN_ARKAMYS_AUDITORIUM)//do not set speed in imx library if auditorium mode is not active
    return;

   tBool bretval = m_poMainAppl->poDBus()->poGetAudioProcProxy()->set_aplArkamysCurrentSpeed(speedValue);
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::Audio process Current speed property set returned with %d", bretval));

}

/**
 *vSendAGCModeToAudProc()
 *@description setting of AGC in audio process
 */
tVoid ArkamysFeatureHandlerRNAIVI::vSendAGCModeToAudProc(tBool bOn)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::bSendAGCModeToAudProc(%i)", bOn));
   AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poMainAppl);

   if(m_bEngModeActive)//do not set AGC in imx library if linear mode is active
     return;

   tBool bretval = m_poMainAppl->poDBus()->poGetAudioProcProxy()->set_aplArkamysAvcActivate(bOn);
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::Audio process AGC mode property set returned with %d", bretval));

}

/**
 *vSendAmbianceToAudProc()
 *@description setting of ambiance in audio process
 */
tVoid ArkamysFeatureHandlerRNAIVI::vSendAmbianceToAudProc(ArkamysConfigRNAIVI::tenAuditoriumAmbiance enAmbianceMode)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::vSendAmbianceToAudProc(%i)", ETG_CENUM(ArkamysConfigRNAIVI::tenAuditoriumAmbiance, enAmbianceMode)));
   AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poMainAppl);

   if(m_bEngModeActive)//do not set Ambiance in imx library if linear mode is active
     return;

   tBool bretval = m_poMainAppl->poDBus()->poGetAudioProcProxy()->set_aplArkamysAmbienceSelect((tS32)enAmbianceMode);
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::Audio process Ambiance property set returned with %d", bretval));

}

/**
 *ArkamysInputLevelDeviation()
 *@description receives the Input level deviation value from ADR
 */
void ArkamysFeatureHandlerRNAIVI::ArkamysInputLevelDeviation(tU8 *u8Data, tU16 length)
{
     ETG_TRACE_USR4(("Got Arkamys Input Level deviation Data: Len:%x ", length));
     //AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poMainAppl);

     //copy input level deviation value to member variable
     OSAL_pvMemoryCopy(&m_s16InputLeveDeviation, u8Data, sizeof(tS16));

     m_s16InputLeveDeviation = ntohs(m_s16InputLeveDeviation);
     //m_s16InputLeveDeviation /= 128;//unit is 1_128dB in ADR3
     ETG_TRACE_USR4(("Input Level deviation value: 0x%x ", m_s16InputLeveDeviation));

     AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poMainAppl);

     tBool bretval = m_poMainAppl->poDBus()->poGetAudioProcProxy()->set_aplArkamysInputLevelDeviation(m_s16InputLeveDeviation);
     ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::Audio process Input Level Deviation property set returned with %d", bretval));

}

/**
 *vSendArkamysModeToAudProc()
 *@description setting of arkamys bypass mode value in audio process
 */
tVoid ArkamysFeatureHandlerRNAIVI::vSendArkamysBypassModeToAudProc(tBool bBypassMode)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::vSendArkamysBypassModeToAudProc(%d)", bBypassMode));
   AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poMainAppl);

   tBool bretval = m_poMainAppl->poDBus()->poGetAudioProcProxy()->set_aplArkamysByPassActivate(bBypassMode);
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::Audio process Arkamys Bypass mode property set returned with %d", bretval));

}

/**
 *ArkamysLevelLoss()
 *@description receives the level loss value from audio process
 */

void ArkamysFeatureHandlerRNAIVI::ArkamysLevelLoss(tS16 levelLoss)
{
     ETG_TRACE_USR4(("Got Arkamys Level Loss Data: Len:%i ", levelLoss));

     tS16 s16LevelLoss = (tS16)(levelLoss * 128);//unit is 1/128db

     tU8 pU8Data[DATA_LEN] = {0};
     pU8Data[0]     = (tU8) (((s16LevelLoss) & 0xFF00)>>8);
     pU8Data[1]     = (tU8) ((s16LevelLoss) & 0x00FF);

     //update ADR3
     vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_ARKAMYS_LEVEL_LOSS, VD_ADR3_OPTYPE_SET
          , DATA_LEN, pU8Data, ARKAMYS_LEVEL_LOSS);//send data to ADR3
}

/**
 *StreamMuteNotify()
 *@description receives the mute/demute notification
 */
void ArkamysFeatureHandlerRNAIVI::StreamMuteNotify(StreamMuteNotify_IF::tenMuteNotification muteAck, tU8 stream, tU16 sink)
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::StreamMuteNotify() sink %d",sink));
   stream = stream; // Hello lint
   if(muteAck == StreamMuteNotify_IF::MUTE_ACK_MUTE)
   {
      m_ActionQueue.Event(AkActionRNAIVI::ARKAMYS_EVENT_MUTED);
   }
   else if(muteAck == StreamMuteNotify_IF::MUTE_ACK_DEMUTE)
   {
      m_ActionQueue.Event(AkActionRNAIVI::ARKAMYS_EVENT_DEMUTED);
   }
}

void ArkamysFeatureHandlerRNAIVI::SetArkamysCommInterface(ArkamysCommInterface *pArkamysCommInterface)
{
   if (m_pArkamysCommInterface != OSAL_NULL && m_bDeleteArkamysCommInterface)
   {
      OSAL_DELETE m_pArkamysCommInterface;
      m_pArkamysCommInterface = OSAL_NULL;
   }

   // The following is secured by m_bDeleteArkamysCommInterface, but Lint doesn't recognize this.
   //lint -save -e423 Creation of memory leak in assignment to 'ArkamysFeatureHandlerRNAIVI::m_pArkamysCommInterface'
   m_pArkamysCommInterface = pArkamysCommInterface;
   //lint -restore
   m_bDeleteArkamysCommInterface = FALSE;
}

void ArkamysFeatureHandlerRNAIVI::SetArkamysADRInterface(ArkamysADRInterface *pArkamysADRInterface)
{
   if (m_pADRInterface != OSAL_NULL && m_bDeleteArkamysADRInterface)
   {
      OSAL_DELETE m_pADRInterface;
      m_pADRInterface = OSAL_NULL;
   }

   // The following is secured by m_bDeleteArkamysADRInterface, but Lint doesn't recognize this.
   //lint -save -e423 Creation of memory leak in assignment to 'ArkamysFeatureHandlerRNAIVI::m_pArkamysADRInterface'
   m_pADRInterface = pArkamysADRInterface;
   //lint -restore
   m_bDeleteArkamysADRInterface = FALSE;
}


/**
 * returns true when VirtualSubwoofer state has changed
 */
tBool ArkamysFeatureHandlerRNAIVI::bUpdateVirtualSubwooferState()
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::bUpdateVirtualSubwooferState() entered."));
   ETG_TRACE_USR4(("    Parameters: Engmode:%i, Source:%i", m_bEngModeActive, ETG_CENUM(ArkamysConfigRNAIVI::tenSrcSetting, m_enCurSourceSetting)));
   tU8 u8NewState = ArkamysStateCalculatorRNAIVI::u8CalculateVirtualSubwooferState(m_bEngModeActive,  m_enCurSourceSetting, m_bVirtualSubAvailable,m_enArkamysMode);

   if (u8NewState != m_u8VirtualSubEnabled)
   {
      ETG_TRACE_USR4(("Virtual Subwoofer state changed: %i -> %i.", m_u8VirtualSubEnabled, u8NewState));
      m_u8VirtualSubEnabled = u8NewState;
      return true;
   }

   ETG_TRACE_USR4(("Virtual Subwoofer state not changed (is: %i).", m_u8VirtualSubEnabled));

   return false;
}

/**
 * returns true when SSS4 state has changed
 */
tBool ArkamysFeatureHandlerRNAIVI::bUpdateSSS4State()
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::bUpdateSSS4State() entered."));
   ETG_TRACE_USR4(("    Parameters: Engmode:%i, Source:%i", m_bEngModeActive, ETG_CENUM(ArkamysConfigRNAIVI::tenSrcSetting, m_enCurSourceSetting)));
   tU8 u8NewState = ArkamysStateCalculatorRNAIVI::u8CalculateSSS4State(m_bEngModeActive, m_enCurSourceSetting,m_enArkamysMode);

   if (u8NewState != m_u8SoundStageMode)
   {
      ETG_TRACE_USR4(("Sound Staging state changed: %i -> %i.", m_u8SoundStageMode, u8NewState));
      m_u8SoundStageMode = u8NewState;
      return true;
   }

   ETG_TRACE_USR4(("Sound staging state not changed (is: %i).", m_u8SoundStageMode));

   return false;
}

/**
 * returns true when FMEnhancer state has changed
 */
tBool ArkamysFeatureHandlerRNAIVI::bUpdateFMEnhancerState()
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::bUpdateFMEnhancerState() entered."));
   ETG_TRACE_USR4(("    Parameters: EngMode:%i, Source:%i, RequestMode(DTM):%i", m_bFMEnhancerLinearMode, ETG_CENUM(ArkamysConfigRNAIVI::tenSrcSetting, m_enCurSourceSetting), m_bFMEnhancerAvailable));
   tU8 u8NewState = ArkamysStateCalculatorRNAIVI::u8CalculateFMEnhancerState(m_bFMEnhancerLinearMode, m_enCurSourceSetting, m_bFMEnhancerAvailable, m_enArkamysMode);

   if (u8NewState != m_u8FMEnhancerEnabled)
   {
      ETG_TRACE_USR4(("FM Enhancer state changed: %i -> %i.", m_u8FMEnhancerEnabled, u8NewState));
      m_u8FMEnhancerEnabled = u8NewState;
      return true;
   }

   ETG_TRACE_USR4(("FM Enhancer state not changed (is: %i).", m_u8FMEnhancerEnabled));

   return false;
}


/*
 * auditorium related settings
 */

/**
 * returns true when Ambiance state has changed
 */
tBool ArkamysFeatureHandlerRNAIVI::bUpdateAmbianceState()
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::bUpdateAmbianceState() entered."));
   ETG_TRACE_USR4(("    Parameters: Engmode:%i, UserAmbiance:%i Source:%i", m_bEngModeActive, ETG_CENUM(ArkamysConfigRNAIVI::tenAuditoriumAmbiance, m_enRequestedAuditoriumAmbianceMode), ETG_CENUM(ArkamysConfigRNAIVI::tenSrcSetting, m_enCurSourceSetting)));
   ArkamysConfigRNAIVI::tenAuditoriumAmbiance enNewState = ArkamysStateCalculatorRNAIVI::enCalculateAmbianceState(m_bEngModeActive, m_enRequestedAuditoriumAmbianceMode, m_enCurSourceSetting,m_enArkamysMode);

   if (enNewState != m_enCurAuditoriumAmbianceMode)
   {
      ETG_TRACE_USR4(("Ambiance state changed: %i -> %i.", ETG_CENUM(ArkamysConfigRNAIVI::tenAuditoriumAmbiance, m_enCurAuditoriumAmbianceMode), ETG_CENUM(ArkamysConfigRNAIVI::tenAuditoriumAmbiance, enNewState)));
      m_enCurAuditoriumAmbianceMode = enNewState;
      return true;
   }

   ETG_TRACE_USR4(("Ambiance state not changed (is: %i request was: %i)."
         , ETG_CENUM(ArkamysConfigRNAIVI::tenAuditoriumAmbiance, m_enCurAuditoriumAmbianceMode), ETG_CENUM(ArkamysConfigRNAIVI::tenAuditoriumAmbiance, m_enRequestedAuditoriumAmbianceMode)));

   return false;
}

/**
 * returns true when FMEnhancer2 state has changed
 */
tBool ArkamysFeatureHandlerRNAIVI::bUpdateFMEnhancer2State()
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::bUpdateFMEnhancer2State() entered."));
   ETG_TRACE_USR4(("    Parameters: EngMode:%i, Source:%i, RequestMode(Source):%i, RequestMode(DTM):%i", m_bFMEnhancerLinearMode, ETG_CENUM(ArkamysConfigRNAIVI::tenSrcSetting, m_enCurSourceSetting), m_bFMEnhancer2Available, m_bFMEnhancerAvailable));
   tU8 u8NewState = ArkamysStateCalculatorRNAIVI::u8CalculateFMEnhancer2State(m_bFMEnhancerLinearMode,m_bFMEnhancer2Available,m_bFMEnhancerAvailable, m_enArkamysMode);

   if (u8NewState != m_u8FMEnhancer2Enabled)
   {
      ETG_TRACE_USR4(("FM Enhancer2 state changed: %i -> %i.", m_u8FMEnhancer2Enabled, u8NewState));
      m_u8FMEnhancer2Enabled = u8NewState;
      return true;
   }

   ETG_TRACE_USR4(("FM Enhancer2 state not changed (is: %i).", m_u8FMEnhancer2Enabled));

   return false;
}

/**
 * returns true when AGC state has changed
 */
tBool ArkamysFeatureHandlerRNAIVI::bUpdateAGCState()
{
   ETG_TRACE_USR4(("ArkamysFeatureHandlerRNAIVI::bUpdateAGCState() entered."));
   ETG_TRACE_USR4(("    Parameters: Engmode:%i, UserAGC:%i Source:%i", m_bEngModeActive, m_bAGCAvailable, ETG_CENUM(ArkamysConfigRNAIVI::tenSrcSetting, m_enCurSourceSetting)));

   tU8 u8NewState = ArkamysStateCalculatorRNAIVI::u8CalculateAGCState(m_bEngModeActive, m_bAGCAvailable,m_enArkamysMode);

   if (u8NewState != m_u8AGCEnabled)
   {
      ETG_TRACE_USR4(("AGC state changed: %i -> %i.", m_u8AGCEnabled, u8NewState));
      m_u8AGCEnabled = u8NewState;
      return true;
   }

   ETG_TRACE_USR4(("AGC state not changed (is: %i).", m_u8AGCEnabled));
   return false;
}


/**
 * TraceStatus()
 * Trace-Output current state. Called via TTFis command "AUDIOMANAGER_ArkamysDebug PRINT_STATUS"
 */
void ArkamysFeatureHandlerRNAIVI::TraceStatus()
{
   // Query ADR Version from Registry (as seen in FC_tclEngModeDiagnosisClientHandler.cpp)
   reg_tclRegKey t_reg;

   char chararray[60] = { '\0' };
   if (t_reg.bOpen("/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/VERSIONS") == true)
   {
    if (t_reg.bQueryString("ADR_SW_VERSION", chararray, sizeof(chararray)) == false)
      {
       chararray[0] = 0;
      }
      t_reg.vClose();
   }



   ETG_TRACE_FATAL(("\t +-----------------------------------------------------------------------------------------\t"));
   ETG_TRACE_FATAL(("\t + Arkamys Audio Mode:         \t%i                                  \t\t\t\t\t", ETG_CENUM(tenArkamysMode, m_enArkamysMode)));
   ETG_TRACE_FATAL(("\t |                                                                   \t\t\t\t\t"));

   ETG_TRACE_FATAL(("\t +ADR Version                                                        \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t |   %s                                                              \t\t\t\t\t", chararray ));
   ETG_TRACE_FATAL(("\t |   Arkamys Lib:  \t%x                                              \t\t\t\t\t", m_u32ReportedArkamysLib));

   ETG_TRACE_FATAL(("\t |                                                                   \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t +EOL (loaded from %s)                                               \t\t\t\t\t", ArkamysEolInterfaceRNAIVI::getInstance() -> isDataFromDatapool() ? "Datapool" : "Internal" ));
   ETG_TRACE_FATAL(("\t |   Supported ADR Lib:  \t%x                                        \t\t\t\t\t", ntohl(ArkamysEolInterfaceRNAIVI::getInstance() -> getEOLADR3LibVersion())));
   ETG_TRACE_FATAL(("\t |   Supported IMX Lib:  \t%x                                        \t\t\t\t\t", ntohl(ArkamysEolInterfaceRNAIVI::getInstance() -> getEOLIMXLibVersion())));
   ETG_TRACE_FATAL(("\t |   Supported Lib Type:  \t0x%x                                     \t\t\t\t\t", ntohl(ArkamysEolInterfaceRNAIVI::getInstance() -> getEOLLibType())));

   ETG_TRACE_FATAL(("\t |                                                                   \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t +KDS Values                                                         \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t |                                                                   \t\t\t\t\t"));
   //ETG_TRACE_FATAL(("\t |   Steering:            \t%s                                       \t\t\t\t\t", m_pEolInterface -> getRightSteering() ? "Right" : "Left"));
   ETG_TRACE_FATAL(("\t |                                                                   \t\t\t\t\t"));

   ETG_TRACE_FATAL(("\t +External state                                                     \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t |   Linear Mode:         \t%s                                       \t\t\t\t\t", m_bEngModeActive ? "on" : "off"));
   ETG_TRACE_FATAL(("\t |   Source:              \t%i                                       \t\t\t\t\t", ETG_CENUM(ArkamysConfigRNAIVI::tenSrcSetting, m_enCurSourceSetting)));
   ETG_TRACE_FATAL(("\t |                                                                   \t\t\t\t\t"));

   ETG_TRACE_FATAL(("\t +Common Feature state                                              \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t |                                                                   \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t |   Source EQ:            \t%i                                      \t\t\t\t\t",ETG_CENUM(ArkamysConfigRNAIVI::tenArkamysFeatureState, m_u8FMEnhancerEnabled)));
   ETG_TRACE_FATAL(("\t |                                                                   \t\t\t\t\t"));

   ETG_TRACE_FATAL(("\t +Classic Feature state                                              \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t |                                                                   \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t |   Virtual Subwoofer:    \t%i                                 \t\t\t\t\t", ETG_CENUM(ArkamysConfigRNAIVI::tenArkamysFeatureState, m_u8VirtualSubEnabled)));
   ETG_TRACE_FATAL(("\t |   Sound Staging:        \t%i                                 \t\t\t\t\t", ETG_CENUM(ArkamysConfigRNAIVI::tenArkamysFeatureState, m_u8SoundStageMode)));
   ETG_TRACE_FATAL(("\t |                                                                   \t\t\t\t\t"));

   ETG_TRACE_FATAL(("\t +Auditorium Feature state                                           \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t |                                                                   \t\t\t\t\t"));
   ETG_TRACE_FATAL(("\t |   Sound Staging Ambiance:        \t%i                             \t\t\t\t\t", ETG_CENUM(ArkamysConfigRNAIVI::tenAuditoriumAmbiance, m_enCurAuditoriumAmbianceMode)));
   ETG_TRACE_FATAL(("\t |                    (requested):  \t%i                             \t\t\t\t\t", ETG_CENUM(ArkamysConfigRNAIVI::tenAuditoriumAmbiance, m_enRequestedAuditoriumAmbianceMode)));
   ETG_TRACE_FATAL(("\t |   FM Enhancer2:                  \t%i                             \t\t\t\t\t", ETG_CENUM(ArkamysConfigRNAIVI::tenArkamysFeatureState, m_u8FMEnhancer2Enabled)));
   ETG_TRACE_FATAL(("\t |   AGC:                           \t%i                             \t\t\t\t\t", ETG_CENUM(ArkamysConfigRNAIVI::tenArkamysFeatureState, m_u8AGCEnabled)));
   ETG_TRACE_FATAL(("\t +-----------------------------------------------------------------------------------------\t"));
}

