/**
 * @file ArkamysADRInterface.cpp
 * @author CM-AI wie1hi
 * @copyright (c) 2014 Robert Bosch Car Multimedia GmbH
 * @addtogroup fc_audiomanager
 * @description This class provides a method to send data to the ADR3. It is the lowest layer in the Arkamys framework
 * @{
 */
#include "../fc_audiomanager_main.h"
#include <netinet/in.h>
#include <pthread.h>

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include <etrace_if.h>
#include "../fc_audiomanager_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_ARKAMYS
#include "trcGenProj/Header/ArkamysADRInterface.cpp.trc.h"
#endif

#include "../aud_sinkmanager/vd_adr3/vd_adr3Msg_If.h"
#include "../InternalComponentCommunication/Messages/settings/IDSettings.h"
#include "../InternalComponentCommunication/InternalCommunicationAdapter.h"
#include "../InternalComponentCommunication/Messages/Sound/ID_Arkamys.h"
#include "ArkamysADRInterface.h"


tU8 ArkamysStaticMessage::getADRCommand()
{
   switch (command)
   {
      case ArkamysOpTypeSet:
         return VD_ADR3_OPTYPE_SET;
      case ArkamysOpTypeGet:
         return VD_ADR3_OPTYPE_GET;
      case ArkamysOpTypeStatus:
         return VD_ADR3_OPTYPE_STATUS;
      case ArkamysOpTypeError:
         return VD_ADR3_OPTYPE_ERROR;
      default:
         return 0xFF;
   }
}

ArkamysStaticMessage::tenAdrOpType ArkamysStaticMessage::ADRToArkamysCommand(tU8 adrCommand)
{
   switch (adrCommand)
   {
   case VD_ADR3_OPTYPE_SET:
      return ArkamysOpTypeSet;
   case VD_ADR3_OPTYPE_GET:
      return ArkamysOpTypeGet;
   case VD_ADR3_OPTYPE_STATUS:
      return ArkamysOpTypeStatus;
   case VD_ADR3_OPTYPE_ERROR:
      return ArkamysOpTypeError;
   default:
      return ArkamysOpTypeInvalid;
   }
}

ArkamysADRInterface::ArkamysADRInterface()
{
   m_bMutexInitialized = false;
   m_bAdrBusy = false;
   pthread_mutex_init(&m_Mutex, NULL);
}

ArkamysADRInterface::~ArkamysADRInterface()
{
   pthread_mutex_destroy(&m_Mutex);
}

/**
 * SendToADR()
 * @description Send Arkamys Command to ADR
 * @param msg Pointer to the message to be sent
 * @param bSwapData Set to true when the data (msg->data) is in Host Byte Order and needs swapping
 */
void ArkamysADRInterface::SendToADR(ArkamysStaticMessage msg)
{
   pthread_mutex_lock(&m_Mutex);
   ETG_TRACE_USR4(("mutex_lock TID = %i", OSAL_ThreadWhoAmI()));
   ETG_TRACE_USR4(("ArkamysADRInterface::SendToADR() entered. Queueing message. TID = %i", OSAL_ThreadWhoAmI()));
   m_ArkamysMessageQueue.push(msg);
   if (!m_bAdrBusy)
   {
      ETG_TRACE_USR4(("mutex_unlock TID = %i", OSAL_ThreadWhoAmI()));
      vProcessQueue(); // sets m_bAdrBusy = true
      pthread_mutex_unlock(&m_Mutex);
   }
   else
   {
      ETG_TRACE_USR4(("ADR busy. Message queued."));
      ETG_TRACE_USR4(("mutex_unlock TID = %i", OSAL_ThreadWhoAmI()));
      pthread_mutex_unlock(&m_Mutex);
   }
}

void ArkamysADRInterface::vProcessQueue()
{

   if (!m_ArkamysMessageQueue.empty())
   {
      ArkamysStaticMessage msg = m_ArkamysMessageQueue.front();
      m_ArkamysMessageQueue.pop();

      ETG_TRACE_USR4(("ArkamysADRInterface::vProcessQueue() entered.\n"
            "Parameter:\n"
            "msg.command: %i\n"
            "msg.size:    %i\n"
            "msg.data:    %02x\n"
            , ETG_CENUM(ArkamysStaticMessage::tenAdrOpType, msg.command)
            , msg.size
            , ETG_LIST_LEN(msg.size), ETG_LIST_PTR_T8(msg.data)
            ));

      m_bAdrBusy = true;
      msg.command = msg.getADRCommand();

      //send the arkamys message to ADR3
      ID_Arkamys oMsg(msg);
      InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oMsg);

      ETG_TRACE_USR4(("Message processed. Messages left: %i", m_ArkamysMessageQueue.size()));
   }
}

void ArkamysADRInterface::vAdrOkCallback()
{
   pthread_mutex_lock(&m_Mutex);
   ETG_TRACE_USR4(("mutex_lock TID = %i", OSAL_ThreadWhoAmI()));
   ETG_TRACE_USR4(("ArkamysADRInterface::vAdrOkCallback(). TID = %i", OSAL_ThreadWhoAmI()));
   m_bAdrBusy = false;
   ETG_TRACE_USR4(("mutex_unlock TID = %i", OSAL_ThreadWhoAmI()));
   vProcessQueue();  // sets m_bAdrBusy = true when queue is not empty
   pthread_mutex_unlock(&m_Mutex);
}

/**
 * Mute()
 * @description Send mute command to ADR
 */
//void ArkamysADRInterface::Mute()
//{
//   ETG_TRACE_USR4(("ArkamysADRInterface::Mute()"));
//
//   tU8 msg[4];
//   msg [0] = TRC::enAdr3MsgIf;
//   msg [1] = TRC::enADRArkamysVolume;
//   msg [2] = ArkamysFeatureHandler::ARKAMYS_VOLUME_MSG_MUTE;
//   msg [3] = EN_AUDIO_MUTE_MUTEACTION_On;
//   fc_audiomanager_tclApp::vRxFromOtherThreads(sizeof(msg), msg);
//}

/**
 * Demute()
 * @description Send demute command to ADR
 */
//void ArkamysADRInterface::Demute()
//{
//   ETG_TRACE_USR4(("ArkamysADRInterface::Demute()"));
//
//   tU8 msg[4];
//   msg [0] = TRC::enAdr3MsgIf;
//   msg [1] = TRC::enADRArkamysVolume;
//   msg [2] = ArkamysFeatureHandler::ARKAMYS_VOLUME_MSG_MUTE;
//   msg [3] = EN_AUDIO_MUTE_MUTEACTION_Off;
//   fc_audiomanager_tclApp::vRxFromOtherThreads(sizeof(msg), msg);
//}

/**
 * MuteRear()
 * @description Send mute command for rear speakers to ADR
 */
//void ArkamysADRInterface::MuteRear()
//{
//   ETG_TRACE_USR4(("ArkamysADRInterface::MuteRear()"));
//
//   tU8 msg[4];
//   msg [0] = TRC::enAdr3MsgIf;
//   msg [1] = TRC::enADRArkamysVolume;
//   msg [2] = ArkamysFeatureHandler::ARKAMYS_VOLUME_MSG_MUTE_REAR;
//   msg [3] = EN_AUDIO_MUTE_MUTEACTION_On;
//   fc_audiomanager_tclApp::vRxFromOtherThreads(sizeof(msg), msg);
//}

/**
 * DeMuteFront()
 * @description Send demute command for front speakers to ADR
 */
//void ArkamysADRInterface::DemuteFront()
//{
//   ETG_TRACE_USR4(("ArkamysADRInterface::DemuteFront()"));
//
//   tU8 msg[4];
//   msg [0] = TRC::enAdr3MsgIf;
//   msg [1] = TRC::enADRArkamysVolume;
//   msg [2] = ArkamysFeatureHandler::ARKAMYS_VOLUME_MSG_MUTE_FRONT;
//   msg [3] = EN_AUDIO_MUTE_MUTEACTION_Off;
//   fc_audiomanager_tclApp::vRxFromOtherThreads(sizeof(msg), msg);
//}


void ArkamysADRInterface::SetBalance(tS16 steps)
{
   ETG_TRACE_USR4(("ArkamysADRInterface::SetBalance(%d)", steps));
   MSG_Settings oMsg(EN_AUDIO_SOURCE_STREAM_MAIN, EN_AUDIO_SETTINGS_BALANCE, steps);
   InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oMsg);
}

void ArkamysADRInterface::SetFader(tS16 steps)
{
   ETG_TRACE_USR4(("ArkamysADRInterface::SetFader(%d)", steps));
   MSG_Settings oMsg(EN_AUDIO_SOURCE_STREAM_MAIN, EN_AUDIO_SETTINGS_FADER, steps);
   InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oMsg);
}

tBool ArkamysADRInterface::bExistMsgToSend()
{
   if(m_ArkamysMessageQueue.size() > 0)
     return true;
   else
     return false;
}


