/*!
 * vdamp-plugin-sender-func.h
 *
 *  Created on: Nov 16, 2013
 *      Author: rjk2kor
 * *****************************************************************************************
 * Version History
 * Date      Modified by    Change Description
 * *****************************************************************************************
 * 11/26/2013  Ranjit K    Added doxygen support.
 * *****************************************************************************************
 */

#ifndef VDAMP_PLUGIN_SENDER_FUNC_H_
#define VDAMP_PLUGIN_SENDER_FUNC_H_

#include "vdamp-plugin-types.h"
#include <map>
#include <sys/types.h>
#include <stdarg.h>
#include "system_types.h"
using namespace std;

/*****************************************************************************************
 * PLUGIN SENDER FUNCTIONAL INTERFACES
 *****************************************************************************************/
/**
 * \class vd_amp_powerState_senderIF
 * \brief This class describes the Sender interface for Power state.<br>
 * <B>Direction : VD Amp to plugin </B><br>
 * When VD Amp wants to set the power state of the connected amplifier, it will use an instance of this class to do so.
 * If the connected amplifier supports setting the power state, then the plugin developer should implement a class that
 * would derive from this class and with the pure virtual function overridden. Pointer to the instance of the new class
 * can be provided to VD Amp using the sender interface. Refer vdamp_plugin_sender::oGetInterfaceData() for more info.
 * \note Calls to the member functions of this class will be made from the context of VD Amp CCA Thread.
 */
class vd_amp_powerState_senderIF
{
public:
  vd_amp_powerState_senderIF(){}
  virtual ~vd_amp_powerState_senderIF(){}
  /*!
   * \fn void vAmp_PowerState(amp_tenPowerState enState)
   * \brief  This function would be triggered by the plugin manager to set  the power state of the connected amplifier
   * \note  This function will be to called from the context of CCA Thread of VD Amp.
   * \param enState Desired power state of the amplifier
   * \return None.
   */
  virtual void vAmp_PowerState(amp_tenPowerState enState) = 0;

};
/**
 * \class vd_amp_ApplicationState_senderIF
 * \brief This class describes the Sender interface for ApplicationState.<br>
 * <B>Direction : VD Amp to plugin </B><br>
 * When VD Amp wants to inform the App state to the connected amplifier, it will use an instance of this class to do so.
 *  Pointer to the instance of the new class
 * can be provided to VD Amp using the sender interface. Refer vdamp_plugin_sender::oGetInterfaceData() for more info.
 * \note Calls to the member functions of this class will be made from the context of VD Amp CCA Thread.
 */
class vd_amp_ApplicationState_senderIF
{
public:
  vd_amp_ApplicationState_senderIF(){}
  virtual ~vd_amp_ApplicationState_senderIF(){}

   /*!
   * \fn void vAmp_ApplicationState(uint16_t appState)
   * \brief  This function would be triggered by the plugin manager to send  the Application state
   * \note  This function will be to called from the context of CCA Thread of VD Amp.
   * \param appstate Desired ApplicationState
   * \return None.
   */
  virtual void vAmp_ApplicationState(uint16_t appState) = 0;
};
/**
 * \class vd_amp_source_senderIF
 * \brief This class describes the Sender interface for Source.<br>
 * <B>Direction : VD Amp to plugin </B><br>
 * When VD Amp wants to inform the current source to the connected amplifier, it will use an instance of this class to do so.
 * If the connected amplifier supports setting the current source, then the plugin developer should implement a class that
 * would derive from this class and with the pure virtual function overridden. Pointer to the instance of the new class
 * can be provided to VD Amp using the sender interface. Refer vdamp_plugin_sender::oGetInterfaceData() for more info.
 * \note Calls to the member functions of this class will be made from the context of VD Amp CCA Thread.
 */
class vd_amp_source_senderIF
{
public:
  vd_amp_source_senderIF(){}
  virtual ~vd_amp_source_senderIF(){}
  /*!
   * \fn void vAmp_Source(amp_tenStream enStream,amp_tenSource enSource)
   * \brief This function would be triggered by the plugin manager to set the source information
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param enStream Stream for which the volume has to be set
   * \param enSource Source.
   * \return None.
   */
  virtual void vAmp_Source(amp_tenStream enStream,amp_tenSource enSource) = 0;
  /*!
   * \fn void vAmp_Source_unMapped(amp_tenStream enStream,amp_tenSource_unmapped enSource)
   * \brief This function would be triggered by the plugin manager to set the source information
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param enStream Stream for which the volume has to be set
   * \param enSource Unmapped Source ID, same as CCA ID.
   * \return None.
   */
  virtual void vAmp_Source_Unmapped(amp_tenStream enStream,amp_tenSource_unmapped enSource)
  {
    (tVoid)enStream;
    (tVoid)enSource;
  };//Dummy implementation ...has to be overridden by the plugins which require the unmapped set of source list

};
/**
 * \class vd_amp_volume_senderIF
 * \brief This class describes the Sender interface for Volume.<br>
 * <B>Direction : VD Amp to plugin </B><br>
 * When VD Amp wants to set the volume of connected amplifier, it will use an instance of this class to do so.
 * If the connected amplifier supports setting volume, then the plugin developer should implement a class that
 * would derive from this class and with the pure virtual function overridden. Pointer to the instance of the new class
 * can be provided to VD Amp using the sender interface. Refer vdamp_plugin_sender::oGetInterfaceData() for more info.
 * \note Calls to the member functions of this class will be made from the context of VD Amp CCA Thread.
 */
class vd_amp_volume_senderIF
{
public:
  vd_amp_volume_senderIF(){}
  virtual ~vd_amp_volume_senderIF(){}
  /*!
   * \fn void vAmp_Volume(amp_tenStream enStream,uint8_t ubStep,int16_t swdB,uint16_t uwRampLin,uint16_t uwRampdB)
   * \brief This function would be triggered by the plugin manager to set the volume of the connected amplifier
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param enStream Stream for which the volume has to be set
   * \param ubStep Range 0x00 - 0xFF. Step size 1.
   * \param swdB   0 --> max Level e.g. -200 --> -50dB. Unit : quarter_db. Step Size 1.
   * \param uwRampLin  TBD.
   * \param uwRampdB   TBD.
   * \return None.
   */
  virtual void vAmp_Volume(amp_tenStream enStream,uint8_t ubStep,int16_t swdB,uint16_t uwRampLin,uint16_t uwRampdB) = 0;
  /*!
   * \fn void vAmp_VolumeRelative(amp_tenVolumeType VolumeType, uint8_t VolumeStep, amp_tenExtAmpVolumeType VolGroup)
   * \brief This function would be triggered by the plugin manager to set the Relative volume of the connected amplifier
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param amp_tenVolumeType VolumeType -> Type of volume (Relative or Absolute)
   * \param uint8_t VolumeStep -> Range 0 to 4
   * \param amp_tenExtAmpVolumeType VolGroup -> Group to which Volume should be applied.
   * \return None.
   */
  virtual void vAmp_VolumeRelative(amp_tenVolumeType VolumeType, uint8_t VolumeStep, amp_tenExtAmpVolumeType VolGroup) = 0 ;
  /*!
   * \fn void vAmp_VolumeList(const map< amp_tenExtAmpVolumeType, tS8 >& VolumeList)
   * \brief This function would be triggered by the plugin manager to send the list of all volumes to the connected amplifier
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param const map< amp_tenExtAmpVolumeType, tS8 >& VolumeList -> Map with list of volumegroup and values.
   * \return None.
   */
  virtual void vAmp_VolumeList(const map< amp_tenExtAmpVolumeType, tS8 >& VolumeList) = 0;
   /*!
   * \fn void vAmp_DuckingValue(int16_t dB, amp_tenStream enStream)
   * \brief This function would be triggered by the plugin manager to send the ducking attenuation and ducking limitation dB value to the connected amplifier
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param int16_t dB, amp_tenStream enStream -> dBvalues and the stream.
   * \return None.
   */
  virtual void vAmp_DuckingValue(uint8_t dB, amp_tenStream enStream)
   {
    (tVoid)dB;
    (tVoid)enStream;
  };//Dummy implementation ...has to be overridden by the plugins which require the Duckingvalue
};

/**
 * \class vd_amp_VolumeOffset_senderIF
 * \brief This class describes the Sender interface for Volume Offset.<br>
 * <B>Direction : VD Amp to plugin </B><br>
 * When VD Amp wants to set the volume offset of connected amplifier, it will use an instance of this class to do so.
 * If the connected amplifier supports setting volume offset, then the plugin developer should implement a class that
 * would derive from this class and with the pure virtual function overridden. Pointer to the instance of the new class
 * can be provided to VD Amp using the sender interface. Refer vdamp_plugin_sender::oGetInterfaceData() for more info.
 * \note Calls to the member functions of this class will be made from the context of VD Amp CCA Thread.
 */
class vd_amp_VolumeOffset_senderIF
{
public:
  vd_amp_VolumeOffset_senderIF(){}
  virtual ~vd_amp_VolumeOffset_senderIF(){}

  /*!
   * \fn void vAmp_VolumeOffset(amp_tenStream enStream,int16_t swOffset_LF,int16_t swOffset_RF,int16_t swOffset_LR,int16_t swOffset_RR,int16_t swOffsest_CE,int16_t swOffset_SUB)
   * \brief This function would be triggered by VD Amp to set the volume offset of the connected amplifier
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param enStream Stream for which the volume has to be set
   * \param swOffset_LF  0 --> max Level e.g. -200 --> -50dB. Unit : quarter_db. Step Size 1.
   * \param swOffset_RF  0 --> max Level e.g. -200 --> -50dB. Unit : quarter_db. Step Size 1.
   * \param swOffset_LR  0 --> max Level e.g. -200 --> -50dB. Unit : quarter_db. Step Size 1.
   * \param swOffset_RR  0 --> max Level e.g. -200 --> -50dB. Unit : quarter_db. Step Size 1.
   * \param swOffsest_CE  0 --> max Level e.g. -200 --> -50dB. Unit : quarter_db. Step Size 1.
   * \param swOffset_SUB  0 --> max Level e.g. -200 --> -50dB. Unit : quarter_db. Step Size 1.
   * \return None.
   */
  virtual void vAmp_VolumeOffset(amp_tenStream enStream,int16_t swOffset_LF,int16_t swOffset_RF,int16_t swOffset_LR,int16_t swOffset_RR,int16_t swOffsest_CE,int16_t swOffset_SUB)= 0;
};

/**
 * \class vd_amp_InputVolumeOffset_senderIF
 * \brief This class describes the Sender interface for Input Volume Offset.<br>
 * <B>Direction : VD Amp to plugin </B><br>
 * When VD Amp wants to set the input volume offset of connected amplifier, it will use an instance of this class to do so.
 * If the connected amplifier supports setting input volume offset, then the plugin developer should implement a class that
 * would derive from this class and with the pure virtual function overridden. Pointer to the instance of the new class
 * can be provided to VD Amp using the sender interface. Refer vdamp_plugin_sender::oGetInterfaceData() for more info.
 * \note Calls to the member functions of this class will be made from the context of VD Amp CCA Thread.
 */
class vd_amp_InputVolumeOffset_senderIF
{
public:
  vd_amp_InputVolumeOffset_senderIF(){}
  virtual ~vd_amp_InputVolumeOffset_senderIF(){}
  /*!
   * \fn void vAmp_InputVolumeOffset(amp_tenStream enStream, uint16_t uwOffset)
   * \brief This function would be triggered by the plugin manager to set the Input volume offset of the connected amplifier
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param enStream Stream for which the input volume offset has to be set.
   * \param uwOffset  0 --> max Level e.g. -200 --> -50dB. Unit : quarter_db. Step Size 1.
   * \return None.
   */
  virtual void vAmp_InputVolumeOffset(amp_tenStream enStream, uint16_t uwOffset) = 0;
};
/**
 * \class vd_amp_settings_senderIF
 * \brief This class describes the Sender interface for Settings.<br>
 * <B>Direction : VD Amp to plugin </B><br>
 * When VD Amp wants to set the settings of connected amplifier (eg: bass, treble etc), it will use an instance of this class to do so.
 * If the connected amplifier supports sound processing settings , then the plugin developer should implement a class that
 * would derive from this class and with the pure virtual function overridden. Pointer to the instance of the new class
 * can be provided to VD Amp using the sender interface. Refer vdamp_plugin_sender::oGetInterfaceData() for more info.
 * \note Calls to the member functions of this class will be made from the context of VD Amp CCA Thread.
 */
class vd_amp_settings_senderIF
{
public:
  vd_amp_settings_senderIF(){}
  virtual ~vd_amp_settings_senderIF(){}
  /*!
   * \fn void vAmp_Settings(amp_tenStream enStream, amp_tenSettingType enSettingType, int16_t swSteps)
   * \brief This function would be triggered by the plugin manager to set the settings of the connected amplifier
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param enStream Stream for which the settings have to be set.
   * \param enSettingType Type of the setting
   * \param swSteps  Step Size 1. Negative for decrement, positive for increment
   * \return None.
   */
  virtual void vAmp_Settings(amp_tenStream enStream, amp_tenSettingType enSettingType, int16_t swSteps) = 0;
    /*!
   * \fn void vAmp_Settings_List(uint8_t* values, uint16_t Length)
   * \brief This function would be triggered to inform the plugin regarding the sound settings values
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param values list of sound settings.
   * \param Length length of the sound settings list
   * \return None.
   */
  virtual void vAmp_Settings_List(uint8_t* values, uint16_t Length)
  {
    (tVoid)values;
    (tVoid)Length;
  };//Dummy implementation ...has to be overridden by the plugins which require the sound setting list
};
/**
 * \class vd_amp_SetParamFilter_senderIF
 * \brief This class describes the Sender interface for SetParamFilter.<br>
 * <B>Direction : VD Amp to plugin </B><br>
 * When VD Amp wants to set the Param Filter of connected amplifier , it will use an instance of this class to do so.
 * If the connected amplifier supports this, then the plugin developer should implement a class that
 * would derive from this class and with the pure virtual function overridden. Pointer to the instance of the new class
 * can be provided to VD Amp using the sender interface. Refer vdamp_plugin_sender::oGetInterfaceData() for more info.
 * \note Calls to the member functions of this class will be made from the context of VD Amp CCA Thread.
 */
class vd_amp_SetParamFilter_senderIF
{
public:
  vd_amp_SetParamFilter_senderIF(){}
  virtual ~vd_amp_SetParamFilter_senderIF(){}
  /*!
   * \fn void vAmp_SetParamFilter(uint8_t ubFilterID, amp_tenParamFilterChar enChar,uint8_t ubOrder, uint16_t uwFreq, int16_t swGain,uint8_t ubQ)
   * \brief This function would be invoked by the plugin manager to set the param filter of the connected amplifier
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param ubFilterID Filter ID. TBD.
   * \param enChar Filter Char
   * \param ubOrder  Range 0x01 - 0x04. Step size 1.
   * \param uwFreq  Range 0x0014 - 0x4E20. Step size 1. Unit: Hz.
   * \param swGain  Range 0xFF88 - 0x0078. Step size 1. Unit: quarter_DB
   * \param ubQ  Range 0x00 - 0x0A. Step size 1.
   * \return None.
   */
  virtual void vAmp_SetParamFilter(uint8_t ubFilterID, amp_tenParamFilterChar enChar,uint8_t ubOrder, uint16_t uwFreq, int16_t swGain,uint8_t ubQ) = 0;
};
/**
 * \class vd_amp_SetGraphFilter_senderIF
 * \brief This class describes the Sender interface for Set Graph Filter.<br>
 * <B>Direction : VD Amp to plugin </B><br>
 * When VD Amp wants to set the Graph Filter of connected amplifier , it will use an instance of this class to do so.
 * If the connected amplifier supports this, then the plugin developer should implement a class that
 * would derive from this class and with the pure virtual function overridden. Pointer to the instance of the new class
 * can be provided to VD Amp using the sender interface. Refer vdamp_plugin_sender::oGetInterfaceData() for more info.
 * \note Calls to the member functions of this class will be made from the context of VD Amp CCA Thread.
 */
class vd_amp_SetGraphFilter_senderIF
{
public:
  vd_amp_SetGraphFilter_senderIF(){}
  virtual ~vd_amp_SetGraphFilter_senderIF(){}
  /*!
   * \fn void vAmp_SetGraphFilter(uint8_t ubEqID, uint8_t ubBand, int16_t swGain, amp_tenFrequencyIndex enFreqIndex)
   * \brief This function would be invoked by the plugin manager to set graph filter of the connected amplifier
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param ubEqID If more than one equaliser is present in an instance, this parameter is used to select one of these.
   * \param ubBand Each Band of the Equaliser has one filter that can be controlled. In case of Graphic Equaliser the frequency is constant.
   *               In case of an Parametric-EQ all parameters, including the frequency, can be changed. Step size 1.
   * \param swGain  Range 0xFF88 - 0x0078. Step size 1. Unit: quarter_DB
   * \param enFreqIndex  Frequency index
   * \return None.
   */
  virtual void vAmp_SetGraphFilter(uint8_t ubEqID, uint8_t ubBand, int16_t swGain, amp_tenFrequencyIndex enFreqIndex)= 0;
};
/**
 * \class vd_amp_mute_senderIF
 * \brief This class describes the Sender interface for mute.<br>
 * <B>Direction : VD Amp to plugin </B><br>
 * When VD Amp wants to set the mute state of connected amplifier , it will use an instance of this class to do so.
 * If the connected amplifier supports this, then the plugin developer should implement a class that
 * would derive from this class and with the pure virtual function overridden. Pointer to the instance of the new class
 * can be provided to VD Amp using the sender interface. Refer vdamp_plugin_sender::oGetInterfaceData() for more info.
 * \note Calls to the member functions of this class will be made from the context of VD Amp CCA Thread.
 */
class vd_amp_mute_senderIF
{
public:
  vd_amp_mute_senderIF(){}
  virtual ~vd_amp_mute_senderIF(){}
  /*!
   * \fn void vAmp_Mute(amp_tenStream enStream,amp_tenMuteAction ubMuteAction)
   * \brief This function would be invoked by the plugin manager to set the mute state of the connected amplifier
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param enStream Stream
   * \param ubMuteAction Mute action to be performed.
   * \return None.
   */
  virtual void vAmp_Mute(amp_tenStream enStream,amp_tenMuteAction ubMuteAction)=0;
};

/**
 * \class vd_amp_BeepParam_senderIF
 * \brief This class describes the Sender interface for beep parameters.<br>
 * <B>Direction : VD Amp to plugin </B><br>
 * When VD Amp wants to set the beep parameters of connected amplifier , it will use an instance of this class to do so.
 * If the connected amplifier supports this, then the plugin developer should implement a class that
 * would derive from this class and with the pure virtual function overridden. Pointer to the instance of the new class
 * can be provided to VD Amp using the sender interface. Refer vdamp_plugin_sender::oGetInterfaceData() for more info.
 * \note Calls to the member functions of this class will be made from the context of VD Amp CCA Thread.
 */
class vd_amp_BeepParam_senderIF
{
public:
  vd_amp_BeepParam_senderIF(){}
  virtual ~vd_amp_BeepParam_senderIF(){}
  /*!
   * \fn void vAmp_BeepParam(amp_tenBeepID enSignalBeepID, uint8_t ubSignalToneID,uint16_t uwFreq,int16_t swGain)
   * \brief This function would be invoked by the plugin manager to set the beep parameters of the connected amplifier
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param enSignalBeepID Beep ID
   * \param ubSignalToneID Signal tone ID. Range 0x01 - 0x04. Step size 1.
   * \param uwFreq  Range 0x0014 - 0x4E20. Step size 1. Unit: Hz.
   * \param swGain  Range 0xFF88 - 0x0078. Step size 1. Unit: quarter_DB
   * \return None.
   */
  virtual void vAmp_BeepParam(amp_tenBeepID enSignalBeepID, uint8_t ubSignalToneID,uint16_t uwFreq,int16_t swGain)=0;
};
/**
 * \class vd_amp_Beep_senderIF
 * \brief This class describes the Sender interface for beep.<br>
 * <B>Direction : VD Amp to plugin </B><br>
 * When VD Amp wants to play beeps from connected amplifier , it will use an instance of this class to do so.
 * If the connected amplifier supports this, then the plugin developer should implement a class that
 * would derive from this class and with the pure virtual function overridden. Pointer to the instance of the new class
 * can be provided to VD Amp using the sender interface. Refer vdamp_plugin_sender::oGetInterfaceData() for more info.
 * \note Calls to the member functions of this class will be made from the context of VD Amp CCA Thread.
 */
class vd_amp_Beep_senderIF
{
public:
  vd_amp_Beep_senderIF(){}
  virtual ~vd_amp_Beep_senderIF(){}
  /*!
   * \fn void vAmp_Beep(ubSignalBeepID, uwPulses, uwAttackTime, uwAttackTimeLog, uwAttackTimeLin,uwReleaseTime, uwReleaseTimeLog, uwReleaseTimeLin, enSignalToneAction)
   * \brief This function would be invoked by the plugin manager to start beep of the connected amplifier
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param ubSignalBeepID Beep ID
   * \param uwPulses  Number of pulses. Step size 1.
   * \param uwAttackTime Is the time from starting of the fade-in to starting of fade-out. Step size 1.
   * \param uwAttackTimeLog Set the time constant for fade in U=Umax*(1-e^(-t/tau)). Step size 1. Unit ms.
   * \param uwAttackTimeLin Step size 1. Unit ms.
   * \param uwReleaseTime Is the time from starting of the fade-out to starting of fade-in of the next pulse (if there is any).Step size 1. Unit ms.
   * \param uwReleaseTimeLog Set the time constant for fade out U=Umax*e^(-t/tau). Step size 1. Unit ms.
   * \param uwReleaseTimeLin Step size 1. Unit ms.
   * \param enSignalToneAction Signal tone action
   * \return None.
   */
  virtual void vAmp_Beep(amp_tenBeepID enSignalBeepID,
      uint16_t uwPulses,
      uint16_t uwAttackTime,
      uint16_t uwAttackTimeLog,
      uint16_t uwAttackTimeLin,
      uint16_t uwReleaseTime,
      uint16_t uwReleaseTimeLog,
      uint16_t uwReleaseTimeLin,
      amp_tenSignalToneAction  enSignalToneAction) = 0;
};
/**
 * \class vd_amp_CarSetting_senderIF
 * \brief This class describes the Sender interface car setting.<br>
 * <B>Direction : VD Amp to plugin </B><br>
 * When VD Amp wants to update the car settings of connected amplifier , it will use an instance of this class to do so.
 * If the connected amplifier supports this, then the plugin developer should implement a class that
 * would derive from this class and with the pure virtual function overridden. Pointer to the instance of the new class
 * can be provided to VD Amp using the sender interface. Refer vdamp_plugin_sender::oGetInterfaceData() for more info.
 * \note Calls to the member functions of this class will be made from the context of VD Amp CCA Thread.
 */
class vd_amp_CarSetting_senderIF
{
public:
  vd_amp_CarSetting_senderIF(){}
  virtual ~vd_amp_CarSetting_senderIF(){}
  /*!
   * \fn void vAmp_CarSetting(amp_tenCarSettingType enSettingType, int16_t swCarSettingValue)
   * \brief This function would be invoked by the plugin manager to set the car settings of the connected amplifier
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param enSettingType Setting type
   * \param swCarSettingValue Setting value. Step size 1.
   * \return None.
   */
  virtual void vAmp_CarSetting(amp_tenCarSettingType enSettingType, int16_t swCarSettingValue)=0;
};

/**
 * \class vd_amp_Surround_senderIF
 * \brief This class describes the Sender interface surround.<br>
 * <B>Direction : VD Amp to plugin </B><br>
 * When VD Amp wants to update the surround settings of connected amplifier , it will use an instance of this class to do so.
 * If the connected amplifier supports this, then the plugin developer should implement a class that
 * would derive from this class and with the pure virtual function overridden. Pointer to the instance of the new class
 * can be provided to VD Amp using the sender interface. Refer vdamp_plugin_sender::oGetInterfaceData() for more info.
 * \note Calls to the member functions of this class will be made from the context of VD Amp CCA Thread.
 */
class vd_amp_Surround_senderIF
{
public:
  vd_amp_Surround_senderIF(){}
  virtual ~vd_amp_Surround_senderIF(){}
  /*!
   * \fn void vAmp_Surround(uint8_t ubMode, int8_t sbLevel, int8_t sbCenterLevel)
   * \brief This function would be invoked by the plugin manager to set the surround of the connected amplifier
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param ubMode Mode.
   * \param sbLevel Level.
   * \param sbCenterLevel Center Level.
   * \return None.
   */
  virtual void vAmp_Surround(uint8_t ubMode, int8_t sbLevel, int8_t sbCenterLevel)=0;
};

class vd_amp_PDCVolumeOffsetIF
  {
public:
  vd_amp_PDCVolumeOffsetIF(){}
  virtual ~vd_amp_PDCVolumeOffsetIF(){}
  /*!
   * \fn void vAmp_PDCVolumeLevel(uint8_t ubMode, int8_t sbLevel, int8_t sbCenterLevel)
   * \brief This function would be invoked by the plugin manager to set the PDC Volume level of the connected amplifier
   * \note  This function will be called from the context of CCA Thread of VD Amp.
   * \param ubMode Mode.
   * \param sbLevel Level.
   * \param sbCenterLevel Center Level.
   * \return None.
   */
  virtual void vAmp_PDCVolumeLevel(tS16 s16DBValue)=0;
};
/******************************************************************************************
 * SENDER INTERFACES COLLECTION
 ******************************************************************************************/
/**
 * \class vdamp_plugin_sender_interfaces
 * \brief This class is a collection of all possible sender interfaces for an external amplifier.
 * When VD Amp wants to get the list of supported sender interfaces of connected amplifier , it will invoke vdamp_plugin_sender::oGetInterfaceData()
 * function of the plugin management interface. The plugin is expected to update the pointers to the supported sender interfaces in an instance of
 * this class and provide it to VD Amp.
 */
class vdamp_plugin_sender_interfaces
{
public:
  vdamp_plugin_sender_interfaces()
  :m_ptrpowerSender(NULL),m_ptrsourceSender(NULL),
  m_ptrvolumeSender(NULL),m_ptrVolumeOffsetSender(NULL),m_ptrInputVolumeOffsetSender(NULL),
  m_ptrsettingsSender(NULL),m_ptrSetParamFilterSender(NULL),m_ptrSetGraphFilterSender(NULL),
  m_ptrmuteSenderIF(NULL),m_ptrBeepParamSender(NULL),m_ptrBeepSender(NULL),m_ptrCarSettingSender(NULL),m_ptrSurroundSender(NULL),m_ptrPDCVolumeOffsetSender(NULL),m_ptrappstateSender(NULL)
  { }

  ~vdamp_plugin_sender_interfaces()
  {
    m_ptrpowerSender= NULL;
    m_ptrsourceSender= NULL;
    m_ptrvolumeSender= NULL;
    m_ptrVolumeOffsetSender= NULL;
    m_ptrInputVolumeOffsetSender= NULL;
    m_ptrsettingsSender= NULL;
    m_ptrSetParamFilterSender= NULL;
    m_ptrSetGraphFilterSender= NULL;
    m_ptrmuteSenderIF= NULL;
    m_ptrBeepParamSender= NULL;
    m_ptrBeepSender= NULL;
    m_ptrCarSettingSender= NULL;
    m_ptrSurroundSender= NULL;
    m_ptrPDCVolumeOffsetSender = NULL;
    m_ptrappstateSender = NULL;
  }
  vd_amp_powerState_senderIF* m_ptrpowerSender;/*!< Pointer to instance of power sender.*/
  vd_amp_source_senderIF* m_ptrsourceSender ;/*!< Pointer to instance of source sender.*/
  vd_amp_volume_senderIF* m_ptrvolumeSender;/*!< Pointer to instance of volume sender.*/
  vd_amp_VolumeOffset_senderIF* m_ptrVolumeOffsetSender;/*!< Pointer to instance of volume offset sender.*/
  vd_amp_InputVolumeOffset_senderIF* m_ptrInputVolumeOffsetSender;/*!< Pointer to instance of input volume offset sender.*/
  vd_amp_settings_senderIF* m_ptrsettingsSender;/*!< Pointer to instance of settings sender.*/
  vd_amp_SetParamFilter_senderIF* m_ptrSetParamFilterSender;/*!< Pointer to instance of set param filter sender.*/
  vd_amp_SetGraphFilter_senderIF* m_ptrSetGraphFilterSender;/*!< Pointer to instance of set graph filter sender.*/
  vd_amp_mute_senderIF* m_ptrmuteSenderIF;/*!< Pointer to instance of mute sender.*/
  vd_amp_BeepParam_senderIF* m_ptrBeepParamSender;/*!< Pointer to instance of beep param sender.*/
  vd_amp_Beep_senderIF* m_ptrBeepSender;/*!< Pointer to instance of beep sender.*/
  vd_amp_CarSetting_senderIF* m_ptrCarSettingSender;/*!< Pointer to instance of car setting sender.*/
  vd_amp_Surround_senderIF* m_ptrSurroundSender;/*!< Pointer to instance of surround sender.*/
  vd_amp_PDCVolumeOffsetIF* m_ptrPDCVolumeOffsetSender;/*!< Pointer to instance of PDC Volume Offset sender.*/
  vd_amp_ApplicationState_senderIF* m_ptrappstateSender;/*!< Pointer to instance of Appstate sender.*/
};

#endif /* VDAMP_PLUGIN_SENDER_FUNC_H_ */
