/*!
*******************************************************************************
* \file              arl_tclISrcVFsm.cpp
*******************************************************************************
*  - PROJECT:        LCN2KAI
*  - SW-COMPONENT:   Audio Routing Library (ARL)
*  - DESCRIPTION:    Implements ISrc Virtual FSM
*  - COPYRIGHT:      &copy; 2012 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 26.06.2012 \version 2.0 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
* - Remove AHL inheritance to enable usage by object reference instead of inheritance
* - Remove MOST dependencies
* - Adapt to Project needs
*
* \date 15.09.2011 \version 1.3 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - GMNGA-16161 JIRA issue fix - MOST error handling for invalid length/missing
* parameter use cases.
*
* \date 05.09.2011 \version 1.2 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - GMNGA-15492 JIRA issue fix - Allocate error value modified to Wrong/invalid
* parameter
* - GMNGA-15455 JIRA issue fix - Source activity error handling for wrong
* paramters
*
* \date 25.08.2011 \version 1.1 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Added routing state handling within the VFSM.
* - GMNGA-14770 JIRA issue fix - Correction/improvement in service availability
* handling.
*
* \date 10.02.2011 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_STREAMROUTERFI_FUNCTIONIDS
#include <midw_fi_if.h>


#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include <generic_msgs_if.h>

#include "arl_ISourceTypes.h"
#include "arl_tclISrcVFsm.h"
#include "arl_tclISourceFsm.h"
#include "arl_tclISource.h"
#include "arl_tclBaseClienthandler.h"
#include "arl_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ARL_ISOURCE_VFSM
#include "trcGenProj/Header/arl_tclISrcVFsm.cpp.trc.h"
#endif

/******************************************************************************
| defines and macros (scope: module-local)
|----------------------------------------------------------------------------*/

/*!
 * \define  ARL_U32_SRV_AVAIL_MAGIC
 * \description Service availability magic.
 * This macro returns a magic value based on Service Id & Availability of the
 * service
 *
 * \param   [srvid]:    (I)   Service Id (tU16)
 * \param   [avail]:    (I)   Availability (tBool)
 *
 */

#define ARL_U32_SRV_AVAIL_MAGIC(srvid, avail) (((tU32)srvid)|((tU32)avail << 16))

/*!
 * \define  ARL_U32_SERVICE_AVAILABILE
 * \description Service Available macro
 * This macro returns a unique number based on Service ID.
 *
 * \param   [srvid]:    (I)   Service Id (tU16)
 *
 */

#define ARL_U32_SERVICE_AVAILABILE(srvid)     (((tU32)srvid)|((tU32)TRUE << 16))

/*!
 * \define  ARL_U32_SERVICE_UNAVAILABILE
 * \description Service Available macro
 * This macro returns a unique number based on Service ID.
 *
 * \param   [srvid]:    (I)   Service Id (tU16)
 *
 */

#define ARL_U32_SERVICE_UNAVAILABILE(srvid)   (((tU32)srvid)|((tU32)FALSE << 16))

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/


/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function prototype (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| Utilizing the Framework for message map abstraction.
|----------------------------------------------------------------------------*/

BEGIN_MSG_MAP(arl_tclISrcVFsm, ahl_tclBaseWork)

   // Method start handlers.
   ON_MESSAGE_SVCDATA(ARL_EN_ALLOCATE, AMT_C_U8_CCAMSG_OPCODE_METHODSTART, vOnAllocStart)
   ON_MESSAGE_SVCDATA(ARL_EN_DEALLOCATE, AMT_C_U8_CCAMSG_OPCODE_METHODSTART, vOnDeAllocStart)
   ON_MESSAGE_SVCDATA(ARL_EN_SOURCEACTIVITY, AMT_C_U8_CCAMSG_OPCODE_METHODSTART, vOnSrcActStart)

   ON_MESSAGE_SVCDATA(ARL_EN_LOOPBACK, AMT_C_U8_CCAMSG_OPCODE_METHODSTART, vOnLoopbackStart)
   ON_MESSAGE_SVCDATA(ARL_EN_SRCACTTIMEREXPIRY, AMT_C_U8_CCAMSG_OPCODE_METHODSTART, vOnSrcActTimerExpiryStart)

   // Stream router method result handlers.
   ON_MESSAGE_SVCDATA(MIDW_STREAMROUTERFI_C_U16_ALLOCATEROUTE, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnAllocRouteResult)
   ON_MESSAGE_SVCDATA(MIDW_STREAMROUTERFI_C_U16_DEALLOCATEROUTE, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnDeAllocRouteResult)
   ON_MESSAGE_SVCDATA(MIDW_STREAMROUTERFI_C_U16_SETPLAY, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnSetPlayResult)
   ON_MESSAGE_SVCDATA(MIDW_STREAMROUTERFI_C_U16_SETSTOP, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnSetStopResult)
   ON_MESSAGE_SVCDATA(MIDW_STREAMROUTERFI_C_U16_SENDFGCOMMAND, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnFGGraphCmdResult)

   // Stream router status handlers.
   ON_MESSAGE_SVCDATA(MIDW_STREAMROUTERFI_C_U16_ROUTINGSTATE, AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnRouteState)

   // AV Manager method result handlers.
//   ON_MESSAGE_SVCDATA(MOST_AVMANFI_C_U16_REQUESTAVACTIVATION, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnReqAVActivResult)
//   ON_MESSAGE_SVCDATA(MOST_AVMANFI_C_U16_REQUESTAVDEACTIVATION, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnReqAVDeActivResult)

   // Stream router error handlers.
   ON_MESSAGE_SVCDATA(MIDW_STREAMROUTERFI_C_U16_ALLOCATEROUTE, AMT_C_U8_CCAMSG_OPCODE_ERROR, vOnStrRouError)
   ON_MESSAGE_SVCDATA(MIDW_STREAMROUTERFI_C_U16_DEALLOCATEROUTE, AMT_C_U8_CCAMSG_OPCODE_ERROR, vOnStrRouError)
   ON_MESSAGE_SVCDATA(MIDW_STREAMROUTERFI_C_U16_SETPLAY, AMT_C_U8_CCAMSG_OPCODE_ERROR, vOnStrRouError)
   ON_MESSAGE_SVCDATA(MIDW_STREAMROUTERFI_C_U16_SETSTOP, AMT_C_U8_CCAMSG_OPCODE_ERROR, vOnStrRouError)
   ON_MESSAGE_SVCDATA(MIDW_STREAMROUTERFI_C_U16_SENDFGCOMMAND, AMT_C_U8_CCAMSG_OPCODE_ERROR, vOnStrRouError)

   // AV Manager router error handlers.
//   ON_MESSAGE_SVCDATA(MOST_AVMANFI_C_U16_REQUESTAVACTIVATION, AMT_C_U8_CCAMSG_OPCODE_ERROR, vOnAVManError)
//   ON_MESSAGE_SVCDATA(MOST_AVMANFI_C_U16_REQUESTAVDEACTIVATION, AMT_C_U8_CCAMSG_OPCODE_ERROR, vOnAVManError)

END_MSG_MAP()

/******************************************************************************
| function implementation (scope: external-interfaces)
|----------------------------------------------------------------------------*/

/******************************************************************************
** FUNCTION:  arl_tclISrcVFsm::arl_tclISrcVFsm(ahl_tclBaseOneThreadA..
******************************************************************************/

arl_tclISrcVFsm::arl_tclISrcVFsm //AHL but now only for ASF created inside
(
   ahl_tclBaseOneThreadApp* const cpoApp
   , arl_tclISource* const cpoISource
   , arl_tclStrClientHandler_ASF_IF* poAsfStrClh
):arl_tclISourceVFsm(cpoApp, cpoISource, poAsfStrClh)
{

   ETG_TRACE_COMP(("[%x]Creating arl_tclISourceVFsm for ASF"
         ,ETG_ENUM(ail_u16AppId, m_u16AppId)));

}

arl_tclISrcVFsm::arl_tclISrcVFsm //AHL
(
   ahl_tclBaseOneThreadApp* const cpoApp
   , arl_tclISource* const cpoISource
):arl_tclISourceVFsm(cpoApp, cpoISource)
{

   ETG_TRACE_COMP(("[%x]Creating arl_tclISourceVFsm"
         ,ETG_ENUM(ail_u16AppId, m_u16AppId)));

}


arl_tclISrcVFsm::arl_tclISrcVFsm //AIL
(
      ail_tclAppInterfaceRestricted* const cpoApp
   , arl_tclISourceIF* const cpoISource
):arl_tclISourceVFsm(cpoApp, cpoISource)
{
   ETG_TRACE_COMP(("[%x]Creating AIL based arl_tclISourceVFsm"
         ,ETG_ENUM(ail_u16AppId, m_u16AppId)));

}


/******************************************************************************
** FUNCTION:  virtual arl_tclISrcVFsm::~arl_tclISrcVFsm()
******************************************************************************/

/*virtual*/
arl_tclISrcVFsm::~arl_tclISrcVFsm()
{
   ETG_TRACE_COMP(("[%x]Destroying arl_tclISrcVFsm"
         ,ETG_ENUM(ail_u16AppId, m_u16AppId)));
}

/******************************************************************************
** FUNCTION:  tVoid arl_tclISrcVFsm::vOnAllocStart(amt_tclServiceData* poMessa.
******************************************************************************/

tVoid arl_tclISrcVFsm::vOnAllocStart(amt_tclServiceData* poMessage)
{
   arl_tISrcAllocStart oAllocStart(*poMessage);

   if (TRUE == oAllocStart.bIsValid())
   {
      ETG_TRACE_USR4(("[%x]Allocate MS triggered. SrcNum:%d, TotalNumSrcs:%d"
            ,ETG_ENUM(ail_u16AppId, m_u16AppId)
            , ETG_ENUM(midw_fi_tcl_e8_AudSource::enType, oAllocStart.u8Source.enType), m_oISrcFsmList.size()));

      ETG_TRACE_USR4(("for Sink %d",oAllocStart.u16Sink));

      // check if this source number is inside the list
      tBool bFound=false;
      for(tU8 i=0; i<m_oISrcFsmList.size(); i++)
      {
         if((static_cast<tU8>(oAllocStart.u8Source.enType) == m_oISrcFsmList[i]->u8GetSourceNumber()))
         {
            bFound = true;
            arl_tclISourceFsm& rfoISrcFsm =  (*(m_oISrcFsmList[i]));

            // Trigger Allocate
            rfoISrcFsm.vOnAllocStart(oAllocStart);
            break;
         }
      }
      if(bFound == false)
      {
         // A source number is requested which is not supported by this source.
         // Send error message.
         if (OSAL_NULL != m_cpoISource)
         {
            ETG_TRACE_ERR(("Allocate MS Error. ErrCode: [not supported]"));
         }
      }

   }  // if (TRUE == oAllocStart.bIsValid())
   else
   {
      // Message decoding failed return error code 0x05
      // Send error message.

      if (OSAL_NULL != m_cpoISource)
      {
         ETG_TRACE_ERR(("Allocate MS Error. ErrCode: [not supported]"));
      }
   }  // End of if-else; if (TRUE == oAllocStart.bIsValid())

}  // tVoid arl_tclISrcVFsm::vOnAllocStart(amt_tclServiceData* poMessage)

/******************************************************************************
** FUNCTION:  tVoid arl_tclISrcVFsm::vOnDeAllocStart(amt_tclServiceD..
******************************************************************************/

tVoid arl_tclISrcVFsm::vOnDeAllocStart(amt_tclServiceData* poMessage)
{
   arl_tISrcDeallocStart oDeAllocStart(*poMessage);

   if (TRUE ==  oDeAllocStart.bIsValid())
   {
      ETG_TRACE_USR4(("[%x]DeAllocate MS triggered. SrcNum:%d, TotalNumSrcs:%d"
            ,ETG_ENUM(ail_u16AppId, m_u16AppId)
            , ETG_ENUM(midw_fi_tcl_e8_AudSource::tenType,oDeAllocStart.u8Source.enType), m_oISrcFsmList.size()));

      // check if this source number is inside the list
      tBool bFound=false;
      for(tU8 i=0; i<m_oISrcFsmList.size(); i++)
      {
         if((static_cast<tU8>(oDeAllocStart.u8Source.enType) == m_oISrcFsmList[i]->u8GetSourceNumber()))
         {
            bFound = true;
            arl_tclISourceFsm& rfoISrcFsm =  (*(m_oISrcFsmList[i]));

            // Trigger DeAllocate
            rfoISrcFsm.vOnDeAllocStart(oDeAllocStart);
            break;
         }
      }
      if(bFound == false)
      {
         // A source number is requested which is not supported by this source.
         // Send error message.
         if (OSAL_NULL != m_cpoISource)
         {
            ETG_TRACE_ERR(("Allocate MS Error. ErrCode: [not supported]"));
         }
      }

   }  // if (TRUE ==  oDeAllocStart.bIsValid())
   else
   {
      // Message decoding failed return error code 0x05
      // Send error message.
      if (OSAL_NULL != m_cpoISource)
      {
         ETG_TRACE_ERR(("Allocate MS Error. ErrCode: [not supported]"));
      }
   }  // End of if-else; if (TRUE ==  oDeAllocStart.bIsValid())
}  // tVoid arl_tclISrcVFsm::vOnDeAllocStart(amt_tclServiceData* poMessage)

/******************************************************************************
** FUNCTION:  tVoid arl_tclISrcVFsm::vOnSrcActStart(amt_tclServiceData* poMess..
******************************************************************************/

tVoid arl_tclISrcVFsm::vOnSrcActStart(amt_tclServiceData* poMessage)
{
   arl_tISrcActivityStart oSrcActivity(*poMessage);

   if (TRUE == oSrcActivity.bIsValid())
   {

      ETG_TRACE_USR4(("[%x]Source Activity triggered. SrcNum:%d, TotalNumSrcs:%d"
            ,ETG_ENUM(ail_u16AppId, m_u16AppId)
            , ETG_ENUM(midw_fi_tcl_e8_AudSource::tenType,oSrcActivity.u8Source.enType), m_oISrcFsmList.size()));

      // check if this source number is inside the list
      tBool bFound=false;
      for(tU8 i=0; i<m_oISrcFsmList.size(); i++)
      {
        if((static_cast<tU8>(oSrcActivity.u8Source.enType) == m_oISrcFsmList[i]->u8GetSourceNumber()))
         {
            bFound = true;
            arl_tclISourceFsm& rfoISrcFsm =  (*(m_oISrcFsmList[i]));

            // Trigger Source activity
            rfoISrcFsm.vOnSrcActStart(oSrcActivity);
            break;
         }
      }
      if(bFound == false)
      {
         // A source number is requested which is not supported by this source.
         // Send error message.
         if (OSAL_NULL != m_cpoISource)
         {
            ETG_TRACE_ERR(("Allocate MS Error. ErrCode: [not supported]"));
         }
      }



   }  // if (TRUE == oSrcActivity.bIsValid())
   else
   {
      // Message decoding failed return error code 0x05
      // Send error message.
      if (OSAL_NULL != m_cpoISource)
      {
         ETG_TRACE_ERR(("Allocate MS Error. ErrCode:[not supported]"));
      }
   }  // End of if-else; if (TRUE == oSrcActivity.bIsValid())
}  // tVoid arl_tclISrcVFsm::vOnSrcActStart(amt_tclServiceData* poMessage)

tVoid arl_tclISrcVFsm::vOnSrcActTimerExpiryStart(amt_tclServiceData* poMessage)
{
  arl_tISrcActTimerExpiryStart oSrcActTimerExpiry(*poMessage);

   if (TRUE == oSrcActTimerExpiry.bIsValid())
   {

      ETG_TRACE_USR4(("[%x]SrcActTimerExpiry triggered. SrcNum:%d"
            ,ETG_ENUM(ail_u16AppId, m_u16AppId)
            , ETG_ENUM(midw_fi_tcl_e8_AudSource::tenType,oSrcActTimerExpiry.u8Source.enType)));

      // check if this source number is inside the list
      tBool bFound=false;
      for(tU8 i=0; i<m_oISrcFsmList.size(); i++)
      {
        if((static_cast<tU8>(oSrcActTimerExpiry.u8Source.enType) == m_oISrcFsmList[i]->u8GetSourceNumber()))
         {
            bFound = true;
            arl_tclISourceFsm& rfoISrcFsm =  (*(m_oISrcFsmList[i]));

            // Trigger Source activity
            rfoISrcFsm.vOnSrcActTimerExpiry();
            break;
         }
      }
      if(bFound == false)
      {
         // A source number is requested which is not supported by this source.
         // Send error message.
         if (OSAL_NULL != m_cpoISource)
         {
            ETG_TRACE_ERR(("SrcActTimerExpiry Error. ErrCode: [not supported]"));
         }
      }
   }  // if (TRUE == oSrcActivity.bIsValid())
   else
   {
      // Message decoding failed return error code 0x05
      // Send error message.
      if (OSAL_NULL != m_cpoISource)
      {
         ETG_TRACE_ERR(("SrcActTimerExpiry Error. ErrCode:[not supported]"));
      }
   }  // End of if-else; if (TRUE == oSrcActivity.bIsValid())

   oSrcActTimerExpiry.vDestroy();
}  // tVoid arl_tclISrcVFsm::vOnSrcActTimerExpiryStart(amt_tclServiceData* poMessage)


tVoid arl_tclISrcVFsm::vOnLoopbackStart(amt_tclServiceData* poMessage)
{
  if(ARL_EN_LOOPBACK == poMessage->u16GetFunctionID())
  {
    gm_tclStreamMessage oVisitorMsg(poMessage);
    tPCUChar pcu8Data = (tPCUChar)oVisitorMsg.pcocGetData();
       tU8 u8Loopbackfunction =  *(pcu8Data + 0);

    ETG_TRACE_USR2(("vOnLoopbackStart() u8LoopbackFunctionId = 0x%x, u32Length =0x%x,\n pu8Data=0x%02x"
        , u8Loopbackfunction, 6, ETG_LIST_LEN(6), ETG_LIST_PTR_T8(poMessage)));

       switch(u8Loopbackfunction)
       {
            case ARL_EN_LOOPBACK_SRCACTRESULT:
              vOnSrcActLoopbackStart(pcu8Data+1);
              break;

            case ARL_EN_LOOPBACK_SETMUTESTATE:
              vOnSetMuteStateLoopbackStart(pcu8Data+1);
                 break;
            case ARL_EN_LOOPBACK_SENDFGGRAPHCMD:
              vOnSendFGGraphCommandLoopbackStart(pcu8Data+1);
                 break;
            case ARL_EN_LOOPBACK_SETAUDIOROUTEREQUEST:
              vOnSetAudioRouteRequestLoopbackStart(pcu8Data+1);
                 break;
            case ARL_EN_LOOPBACK_SETAUDIOPROPERTY:
              vOnSetAudioPropertyLoopbackStart(pcu8Data+1);
                 break;
            case ARL_EN_LOOPBACK_SETSOURCEAVAILABILITY:
              vOnSetSourceAvailabilityLoopbackStart(pcu8Data+1);
                 break;
            default:
              ETG_TRACE_ERR(("arl_tclISrcVFsm::vOnLoopbackStart() error - loopbackfunctionid not valid"));
              break;
       }
  }
}  // tVoid arl_tclISrcVFsm::vOnLoopbackStart(amt_tclServiceData* poMessage)

tVoid arl_tclISrcVFsm::vOnSrcActLoopbackStart(tPCUChar pcu8Data)
{
  if(NULL == pcu8Data)
  {
    ETG_TRACE_ERR(("vOnSrcActLoopbackStart Error. data pointer NULL"));
    return;
  }

  arl_tenSource src = (arl_tenSource) *(pcu8Data + 0);
  arl_tenActivity act = (arl_tenActivity) *(pcu8Data + 1);

     // check if this source number is inside the list
     tBool bFound=false;

     for(tU8 i=0; i<m_oISrcFsmList.size(); i++)
     {
       if((static_cast<tU8>(src) == m_oISrcFsmList[i]->u8GetSourceNumber()))
       {
         bFound = true;
         arl_tclISourceFsm& rfoISrcFsm =  (*(m_oISrcFsmList[i]));

         // Trigger Source activity result
         rfoISrcFsm.vActResult(act);
         break;
       }
     }
     if(bFound == false)
     {
       // A source number is requested which is not supported by this source.
       // Send error message.
       if (OSAL_NULL != m_cpoISource)
       {
         ETG_TRACE_ERR(("vOnSrcActLoopbackStart Error. ErrCode: [not supported]"));
       }
     }
}

tVoid arl_tclISrcVFsm::vOnSetMuteStateLoopbackStart(tPCUChar pcu8Data)
{
  if(NULL == pcu8Data)
  {
    ETG_TRACE_ERR(("vOnSetMuteStateLoopbackStart Error. data pointer NULL"));
    return;
  }

  arl_tenSource enSrcNum = (arl_tenSource) *(pcu8Data + 0);
  arl_tenMuteAction enMuteAction = (arl_tenMuteAction) *(pcu8Data + 1);

  tU32 u32MuteReason = ((tU32)*(pcu8Data + 2)) <<24;
  u32MuteReason = u32MuteReason | (((tU32)*(pcu8Data + 3)) <<16);
  u32MuteReason = u32MuteReason | (((tU32)*(pcu8Data + 4)) <<8);
  u32MuteReason = u32MuteReason | ((tU32)*(pcu8Data + 5));
  arl_enMuteReason enmuteReason = (arl_enMuteReason)u32MuteReason;

    vSetMuteState(enSrcNum, enMuteAction, enmuteReason);

    if(m_cpoISource != NULL)
    {
    if(m_cpoISource->eUpdateClients(ARL_EN_MUTE_REQ) == AIL_EN_N_NO_ERROR)
    {
      ETG_TRACE_USR2(("vOnSetMuteStateLoopbackStart eUpdateClients called without error "));
    }
    }
}

tVoid arl_tclISrcVFsm::vOnSendFGGraphCommandLoopbackStart(tPCUChar pcu8Data)
{
  if(NULL == pcu8Data)
  {
    ETG_TRACE_ERR(("vOnSendFGGraphCommandLoopbackStart Error. data pointer NULL"));
    return;
  }
  arl_tenSource enSrcNum = (arl_tenSource) *(pcu8Data + 0);
  arl_tenFGCommand enFGCmd = (arl_tenFGCommand) *(pcu8Data + 1);

  bSendFGGraphCmd(enSrcNum, enFGCmd);
}

tVoid arl_tclISrcVFsm::vOnSetAudioRouteRequestLoopbackStart(tPCUChar pcu8Data)
{
  if(NULL == pcu8Data)
  {
    ETG_TRACE_ERR(("vOnSetAudioRouteRequestLoopbackStart Error. data pointer NULL"));
    return;
  }
  arl_tenSource enSourceClass = (arl_tenSource) *(pcu8Data + 0);
  tU16 u16SubSource = (tU16)(((tU16)*(pcu8Data + 1)) <<8);
  u16SubSource = (tU16) (u16SubSource | ((tU16)*(pcu8Data + 2)));
  SourceID SrcId(enSourceClass, u16SubSource);
  arl_tenActivity enRequest = (arl_tenActivity) *(pcu8Data + 3);
  tU32 u32Reason = ((tU32)*(pcu8Data + 4)) <<24;
  u32Reason = u32Reason | (((tU32)*(pcu8Data + 5)) <<16);
  u32Reason = u32Reason | (((tU32)*(pcu8Data + 6)) <<8);
  u32Reason = u32Reason | ((tU32)*(pcu8Data + 7));
  arl_enRouteReqReason enRouteReqReason = (arl_enRouteReqReason)u32Reason;

  vSetRouteRequest(SrcId, enRequest, enRouteReqReason);

    if(m_cpoISource != NULL)
    {
    if(m_cpoISource->eUpdateClients(ARL_EN_ROUTEREQUEST) == AIL_EN_N_NO_ERROR)
    {
      ETG_TRACE_USR2(("vOnSetAudioRouteRequestLoopbackStart eUpdateClients called without error "));
    }
    }
}

tVoid arl_tclISrcVFsm::vOnSetAudioPropertyLoopbackStart(tPCUChar pcu8Data)
{
  if(NULL == pcu8Data)
  {
    ETG_TRACE_ERR(("vOnSetAudioPropertyLoopbackStart Error. data pointer NULL"));
    return;
  }
  arl_tenSource enSrcNum = (arl_tenSource) *(pcu8Data + 0);
  arl_tenSourcePropType enPropType = (arl_tenSourcePropType) *(pcu8Data + 1);
  tS32 s32PropValue = ((tS32)*(pcu8Data + 2)) <<24;
  s32PropValue = s32PropValue | (((tS32)*(pcu8Data + 3)) <<16);
  s32PropValue = s32PropValue | (((tS32)*(pcu8Data + 4)) <<8);
  s32PropValue = s32PropValue | ((tS32)*(pcu8Data + 5));
  tU16 u16SubSource =(tU16) (((tU16)*(pcu8Data + 6)) <<8);
  u16SubSource = (tU16) (u16SubSource | ((tU16)*(pcu8Data + 7)));

  vSetAudioProperty(enSrcNum, enPropType, s32PropValue, u16SubSource);

    if(m_cpoISource != NULL)
    {
    if(m_cpoISource->eUpdateClients(ARL_EN_AUDIOPROPERTY) == AIL_EN_N_NO_ERROR)
    {
      ETG_TRACE_USR2(("vOnSetAudioPropertyLoopbackStart eUpdateClients called without error "));
    }
    }
}

tVoid arl_tclISrcVFsm::vOnSetSourceAvailabilityLoopbackStart(tPCUChar pcu8Data)
{
  if(NULL == pcu8Data)
  {
    ETG_TRACE_ERR(("vOnSetSourceAvailabilityLoopbackStart Error. data pointer NULL"));
    return;
  }
  arl_tenSrcAvailability srcAvailability = (arl_tenSrcAvailability) *(pcu8Data + 0);
  arl_tenAvailabilityReason availabilityReason = (arl_tenAvailabilityReason) *(pcu8Data + 1);
  arl_tenSource enSource = (arl_tenSource) *(pcu8Data + 2);
  tU16 u16SubSource = (tU16)(((tU16)*(pcu8Data + 3)) <<8);
  u16SubSource = (tU16)(u16SubSource | ((tU16)*(pcu8Data + 4)));

  vSetSourceAvailability(srcAvailability, availabilityReason, enSource, u16SubSource);

    if(m_cpoISource != NULL)
    {
    if(m_cpoISource->eUpdateClients(ARL_EN_SOURCEAVAILABLE) == AIL_EN_N_NO_ERROR)
    {
      ETG_TRACE_USR2(("vOnSetSourceAvailabilityLoopbackStart eUpdateClients called without error "));
    }
    }
}

/******************************************************************************
** FUNCTION:  tVoid arl_tclISrcVFsm::vOnSourceInfo(amt_tclServiceData* ..
******************************************************************************/

tVoid arl_tclISrcVFsm::vOnSourceInfo(amt_tclServiceData* )
{
   ETG_TRACE_ERR(("[%x]vOnSourceInfo is not supported"
         ,ETG_ENUM(ail_u16AppId, m_u16AppId)));
}  // tVoid arl_tclISrcVFsm::vOnSourceInfo(amt_tclServiceData *poMessage)




/******************************************************************************
** FUNCTION:  tVoid arl_tclISrcVFsm::vOnAllocRouteResult(amt_tclServiceData* ..
******************************************************************************/

tVoid arl_tclISrcVFsm::vOnAllocRouteResult(amt_tclServiceData* poMessage)
{
   arl_tAllocRouteResult oMsgAllocMr(*poMessage);

   // Get the source number which is embedded into the command counter ;-)
   tU16 u16SrcNum   =  poMessage->u16GetCmdCounter();

   ETG_TRACE_USR4(("[%x]Allocate Route result received. SrcNum:%d, TotalNumSrcs:%d"
         ,ETG_ENUM(ail_u16AppId, m_u16AppId)
         , u16SrcNum, m_oISrcFsmList.size()));

   // Route Allocated On StartUp.
   if(u16SrcNum & ARL_MASK_ROUALLOC_STARTUP)
   {
       u16SrcNum &= ARL_MAX_SOURCE_NUMBER;

       // we have to search again if we find the source number inside the vector elements
       // check if this source number is inside the list
       tBool bFound=false;
       for(tU8 i=0; i<m_oISrcFsmList.size(); i++)
       {
         if(u16SrcNum == m_oISrcFsmList[i]->u8GetSourceNumber())
         {
            bFound = true;
            arl_tclISourceFsm& rfoISrcFsm =  (*(m_oISrcFsmList[i]));

            // Invoke the FSM
            rfoISrcFsm.vOnStartUpAllocRouteResult(u16SrcNum);
            break;
         }
       }
       if(bFound == false)
       {
          NORMAL_M_ASSERT(ARL_MAX_SOURCE_NUMBER >= u16SrcNum);
          NORMAL_M_ASSERT(0 != u16SrcNum);
          NORMAL_M_ASSERT_ALWAYS();
       }
       // end of search

   }
   else
   {
       // we have to search again if we find the source number inside the vector elements
       // check if this source number is inside the list
       tBool bFound=false;
       for(tU8 i=0; i<m_oISrcFsmList.size(); i++)
       {
         if(u16SrcNum == m_oISrcFsmList[i]->u8GetSourceNumber())
         {
            bFound = true;
            arl_tclISourceFsm& rfoISrcFsm =  (*(m_oISrcFsmList[i]));

            // Invoke the FSM
            rfoISrcFsm.vOnAllocRouteResult(oMsgAllocMr);
            break;
         }
       }
       // end of search
       if(bFound == false)
       {
          NORMAL_M_ASSERT(ARL_MAX_SOURCE_NUMBER >= u16SrcNum);
          NORMAL_M_ASSERT(0 != u16SrcNum);
          NORMAL_M_ASSERT_ALWAYS();

       }
   }



}  // tVoid arl_tclISrcVFsm::vOnAllocRouteResult(amt_tclServiceData* poMessage)

/******************************************************************************
** FUNCTION:  tVoid arl_tclISrcVFsm::vOnDeAllocRouteResult(amt_tclServiceData..
******************************************************************************/

tVoid arl_tclISrcVFsm::vOnDeAllocRouteResult(amt_tclServiceData* poMessage)
{
   arl_tDeAllocRouteResult oMsgDeAllocMr(*poMessage);

   // Get the source number which is embedded into the command counter ;-)
   tU16 u16SrcNum   =  poMessage->u16GetCmdCounter();

   ETG_TRACE_USR4(("[%x]DeAllocate Route result received. SrcNum:%d, TotalNumSrcs:%d"
         ,ETG_ENUM(ail_u16AppId, m_u16AppId)
         , u16SrcNum, m_oISrcFsmList.size()));


   // Route Allocated On StartUp.
   if(u16SrcNum & ARL_MASK_ROUALLOC_STARTUP)
   {
       // No Action
   }
   else
   {
       // we have to search again if we find the source number inside the vector elements
       // check if this source number is inside the list
       tBool bFound=false;
       for(tU8 i=0; i<m_oISrcFsmList.size(); i++)
       {
         if(u16SrcNum == m_oISrcFsmList[i]->u8GetSourceNumber())
         {
            bFound = true;
            arl_tclISourceFsm& rfoISrcFsm =  (*(m_oISrcFsmList[i]));

            // Invoke the FSM
            rfoISrcFsm.vOnDeAllocRouteResult(oMsgDeAllocMr);
            break;
         }
       }
       // end of search
       if(bFound == false)
       {
          NORMAL_M_ASSERT(ARL_MAX_SOURCE_NUMBER >= u16SrcNum);
          NORMAL_M_ASSERT(0 != u16SrcNum);
          NORMAL_M_ASSERT_ALWAYS();
       }
   }


}  // tVoid arl_tclISrcVFsm::vOnDeAllocRouteResult(amt_tclServiceData* poMessage)

/******************************************************************************
** FUNCTION:  tVoid arl_tclISrcVFsm::vOnSetPlayResult(amt_tclServiceData* poM..
******************************************************************************/

tVoid arl_tclISrcVFsm::vOnSetPlayResult(amt_tclServiceData* poMessage)
{
   arl_tSetPlayResult oMsgSetPlay(*poMessage);

   // Get the source number which is embedded into the command counter ;-)
   tU16 u16SrcNum   =  poMessage->u16GetCmdCounter();

   ETG_TRACE_USR4(("[%x]Set Play result received. SrcNum:%d, TotalNumSrcs:%d"
         ,ETG_ENUM(ail_u16AppId, m_u16AppId)
         , u16SrcNum, m_oISrcFsmList.size()));


    // we have to search again if we find the source number inside the vector elements
    // check if this source number is inside the list
    tBool bFound=false;
    for(tU8 i=0; i<m_oISrcFsmList.size(); i++)
    {
      if(u16SrcNum == m_oISrcFsmList[i]->u8GetSourceNumber())
      {
         bFound = true;
         arl_tclISourceFsm& rfoISrcFsm =  (*(m_oISrcFsmList[i]));

         // Invoke the FSM
         rfoISrcFsm.vOnSetPlayResult(oMsgSetPlay);
         break;
      }
    }
    // end of search
    if(bFound == false)
    {
       NORMAL_M_ASSERT(ARL_MAX_SOURCE_NUMBER >= u16SrcNum);
       NORMAL_M_ASSERT(0 != u16SrcNum);
       NORMAL_M_ASSERT_ALWAYS();
    }

}  // tVoid arl_tclISrcVFsm::vOnSetPlayResult(amt_tclServiceData* poMessage)

/******************************************************************************
** FUNCTION:  tVoid arl_tclISrcVFsm::vOnSetStopResult(amt_tclServiceData* poM..
******************************************************************************/

tVoid arl_tclISrcVFsm::vOnSetStopResult(amt_tclServiceData* poMessage)
{
   arl_tSetStopResult oMsgSetStop(*poMessage);

   // Get the source number which is embedded into the command counter ;-)
   tU16 u16SrcNum   =  poMessage->u16GetCmdCounter();

   ETG_TRACE_USR4(("[%x]Set Stop result received. SrcNum:%d, TotalNumSrcs:%d"
         ,ETG_ENUM(ail_u16AppId, m_u16AppId)
         , u16SrcNum, m_oISrcFsmList.size()));


    // we have to search again if we find the source number inside the vector elements
    // check if this source number is inside the list
    tBool bFound=false;
    for(tU8 i=0; i<m_oISrcFsmList.size(); i++)
    {
      if(u16SrcNum == m_oISrcFsmList[i]->u8GetSourceNumber())
      {
         bFound = true;
         arl_tclISourceFsm& rfoISrcFsm =  (*(m_oISrcFsmList[i]));

         // Invoke the FSM
         rfoISrcFsm.vOnSetStopResult(oMsgSetStop);
         break;
      }
    }
    // end of search
    if(bFound == false)
    {
      NORMAL_M_ASSERT(ARL_MAX_SOURCE_NUMBER >= u16SrcNum);
      NORMAL_M_ASSERT(0 != u16SrcNum);
      NORMAL_M_ASSERT_ALWAYS();
    }


}  // tVoid arl_tclISrcVFsm::vOnSetStopResult(amt_tclServiceData* poMessage)


/******************************************************************************
** FUNCTION:  tVoid arl_tclISrcVFsm::vOnFGGraphCmdResult(amt_tclS..
******************************************************************************/

tVoid arl_tclISrcVFsm::vOnFGGraphCmdResult(amt_tclServiceData* )
{
   // Directly forward to Player IF
   ETG_TRACE_USR4(("[%x] vOnFGGraphCmdResult"
         ,ETG_ENUM(ail_u16AppId, m_u16AppId)));

   if(m_cpoISource != NULL)
   {
      m_cpoISource->vOnFGGraphCmdResult(ARL_EN_NO_ERROR);
   }
}

/******************************************************************************
** FUNCTION:  tVoid arl_tclISrcVFsm::vOnStrRouError(amt_tclServiceData* poMes..
******************************************************************************/

tVoid arl_tclISrcVFsm::vOnStrRouError(amt_tclServiceData* poMessage)
{
   amt_tclServiceDataError oErrorMsg(poMessage);

   // Get the source number which is embedded into the command counter ;-)
   tU16 u16SrcNum   =  oErrorMsg.u16GetCmdCounter();

   ETG_TRACE_ERR(("[%x]Stream router Error msg received. SrcNum:%d, TotalNumSrcs:%d"
         ,ETG_ENUM(ail_u16AppId, m_u16AppId)
         , u16SrcNum, m_oISrcFsmList.size()));

    // we have to search again if we find the source number inside the vector elements
    // check if this source number is inside the list
    tBool bFound=false;
    for(tU8 i=0; i<m_oISrcFsmList.size(); i++)
    {
      if(u16SrcNum == m_oISrcFsmList[i]->u8GetSourceNumber())
      {
         bFound = true;
         arl_tclISourceFsm& rfoISrcFsm =  (*(m_oISrcFsmList[i]));
         arl_tenISourceError enError   =  ARL_EN_ISRC_INVALID_ERR;

         switch (oErrorMsg.u16GetFunctionID())
         {
            case MIDW_STREAMROUTERFI_C_U16_ALLOCATEROUTE:
            {
               // Invoke the state machine.
               rfoISrcFsm(rfoISrcFsm.rfcoGetTriggerId().m_s16StrRouAllocateErr);
               enError   =  ARL_EN_ISRC_ALLOC_ERR;
            }  // case MIDW_STREAMROUTERFI_C_U16_ALLOCATEROUTE:
            break;

            case MIDW_STREAMROUTERFI_C_U16_DEALLOCATEROUTE:
            {
               // Invoke the state machine.
               rfoISrcFsm(rfoISrcFsm.rfcoGetTriggerId().m_s16StrRouDeAllocateErr);
               enError   =  ARL_EN_ISRC_DEALLOC_ERR;
            }  // case MIDW_STREAMROUTERFI_C_U16_DEALLOCATEROUTE:
            break;

            case MIDW_STREAMROUTERFI_C_U16_SETPLAY:
            {
               // Invoke the state machine.
               rfoISrcFsm(rfoISrcFsm.rfcoGetTriggerId().m_s16StrRouSetPlayErr);
               enError   =  ARL_EN_ISRC_SRCACT_ON_ERR;
            }  // case MIDW_STREAMROUTERFI_C_U16_SETPLAY:
            break;

            case MIDW_STREAMROUTERFI_C_U16_SETSTOP:
            {
               // Invoke the state machine.
               rfoISrcFsm(rfoISrcFsm.rfcoGetTriggerId().m_s16StrRouSetStopErr);
               enError   =  ARL_EN_ISRC_SRCACT_OFF_ERR;
            }  // case MIDW_STREAMROUTERFI_C_U16_SETSTOP:
            break;

            case MIDW_STREAMROUTERFI_C_U16_SENDFGCOMMAND:
            {
               // Directly forward to Player IF
               if(m_cpoISource != NULL)
               {
                  m_cpoISource->vOnFGGraphCmdResult(ARL_EN_UNKNOWN_ERROR);
               }
            }  // case MIDW_STREAMROUTERFI_C_U16_SETSTOP:
            break;

            default:
            {
               // Control should never reach here. Assert always.
               NORMAL_M_ASSERT_ALWAYS();

            }  // default:
            break;

         }  // switch (oErrorMsg.u16GetFunctionID())


         if (OSAL_NULL != m_cpoISource)
         {
            ETG_TRACE_ERR(("Notifying Error:%d to Application received for FID:%d."
               , ETG_ENUM(ARL_ISRC_ERR, enError), oErrorMsg.u16GetFunctionID()));

            m_cpoISource->vOnError((tU8)u16SrcNum, enError);
         }

         break; // leave for loop
      }
    }
    if(bFound == false)
    {
       if(0 != (u16SrcNum & ARL_MAX_SOURCE_NUMBER))
       {
         //Early Allocation condition. Source Number is masked with Start up Mask (0x80)
       }
       else
       {
         NORMAL_M_ASSERT(ARL_MAX_SOURCE_NUMBER >= u16SrcNum);
         NORMAL_M_ASSERT(0 != u16SrcNum);
         NORMAL_M_ASSERT_ALWAYS();
       }
    }


}  // tVoid arl_tclISrcVFsm::vOnStrRouError(amt_tclServiceData* poMessage)

/******************************************************************************
** FUNCTION:  tBool arl_tclISrcVFsm::bServiceAvailability(tU16 u16ServiceId, ..
******************************************************************************/

/*virtual*/
tBool arl_tclISrcVFsm::bServiceAvailability(tU16 u16ServiceId, tBool bAvail)
{

   if(bAvail)
   {
      ETG_TRACE_USR4(("[%x]Service ID: %d, available"
            ,ETG_ENUM(ail_u16AppId, m_u16AppId)
            , ETG_ENUM(ail_u16ServiceId, u16ServiceId)));
   }else{
      ETG_TRACE_USR4(("[%x]Service ID: %d, NOT available"
               ,ETG_ENUM(ail_u16AppId, m_u16AppId)
               , ETG_ENUM(ail_u16ServiceId, u16ServiceId)));
   }


   // Check for service availability & make the source available/unavailable.
   tBool bSrvAvail   = FALSE;

   if(m_poStrRouClientASF != OSAL_NULL)
   {
    if(m_poStrRouClientASF->bIfServiceAvailable())
    {
     bSrvAvail = TRUE;
    }
   }

   if(m_poStrRouClient != OSAL_NULL)
   {
      if(m_poStrRouClient->bIfServiceAvailable())
      {
         bSrvAvail = TRUE;
      }
   }

   if(m_poStrRouClientAIL != OSAL_NULL)
   {
      if(m_poStrRouClientAIL->bIfServiceAvailable())
      {
         bSrvAvail = TRUE;
      }
   }

   if (OSAL_NULL != m_cpoISource)
   {
      /// NOTE: Don't play with service, as the FBlock would be unavailable in
      /// case communications partners are unavailable, which might be an issue
      /// for the MOST compliance test.
      /// So make the source unavailable.
      m_cpoISource->vSetSrcAvailable(bSrvAvail);
   }

   // Dispatch the service availability.
   for (arl_tISrcFsmListIter iterFsm = m_oISrcFsmList.begin();
         iterFsm < m_oISrcFsmList.end(); ++iterFsm)
   {
      arl_tclISourceFsm& rfoISrcFsm =  (*(*iterFsm));

      // Dispatch the trigger based on service availability magic macro ;-)
      switch (ARL_U32_SRV_AVAIL_MAGIC(u16ServiceId, bAvail))
      {
         case ARL_U32_SERVICE_AVAILABILE(CCA_C_U16_SRV_STREAMROUTER):
         {
            // Trigger Stream router Available.
            rfoISrcFsm(rfoISrcFsm.rfcoGetTriggerId().m_s16StrRouAvail);
         }
         break;

         case ARL_U32_SERVICE_UNAVAILABILE(CCA_C_U16_SRV_STREAMROUTER):
         {
            // Trigger Stream router unavailable.
            rfoISrcFsm(rfoISrcFsm.rfcoGetTriggerId().m_s16StrRouUnavail);
         }
         break;

         case ARL_U32_SERVICE_AVAILABILE(CCA_C_U16_SRV_FB_AVMANAGER):
         {
            // Only for self triggered source trigger the avail/unavailability
            if (TRUE == rfoISrcFsm.bIsSelfTrig())
            {
               // Trigger AVManager router Available.
               rfoISrcFsm(rfoISrcFsm.rfcoGetTriggerId().m_s16AVManAvail);
            }
         }
         break;

         case ARL_U32_SERVICE_UNAVAILABILE(CCA_C_U16_SRV_FB_AVMANAGER):
         {
            // Only for self triggered source trigger the avail/unavailability
            if (TRUE == rfoISrcFsm.bIsSelfTrig())
            {
               // Trigger AVManager router unavailable.
               rfoISrcFsm(rfoISrcFsm.rfcoGetTriggerId().m_s16AVManUnavail);
            }
         }
         break;

         default:
         {
            // Developer error. Should never be here. ;-)
            NORMAL_M_ASSERT_ALWAYS();
         }  // default:
         break;

      }  // switch (ARL_U32_SRV_AVAIL_MAGIC(u16ServiceId, bAvail))

   }  // for (arl_tISrcFsmListIter iterFsm = m_oISrcFsmList.begin(); iterFsm <..

   return TRUE;

}  // tBool arl_tclISrcVFsm::bServiceAvailability(tU16 u16ServiceId, tBool bAv..

/******************************************************************************
** FUNCTION:  tVoid arl_tclISrcVFsm::vOnRouteState(const amt_tc..
******************************************************************************/

tVoid arl_tclISrcVFsm::vOnRouteState(amt_tclServiceData* poMessage)
{
   arl_tRouteState oMsgRouteState(*poMessage);

   // Dispatch the routing state to all the state machines.
   for (arl_tISrcFsmListIter iterFsm = m_oISrcFsmList.begin();
         iterFsm < m_oISrcFsmList.end(); ++iterFsm)
   {
      arl_tclISourceFsm& rfoISrcFsm =  (*(*iterFsm));

      rfoISrcFsm.vOnRouteState(oMsgRouteState);
   }

}  // tVoid arl_tclISrcVFsm::vOnRouteState(amt_tclServiceData* poMessage)

////////////////////////////////////////////////////////////////////////////////
// <EOF>
