/*!
*******************************************************************************
* \file              arl_tclISourceVFsm.cpp
*******************************************************************************
*  - PROJECT:        LCN2KAI
*  - SW-COMPONENT:   Audio Routing Library (ARL)
*  - DESCRIPTION:    Implements ISource Virtual FSM
*  - COPYRIGHT:      &copy; 2011 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 26.06.2012 \version 2.0 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
* - Remove AHL inheritance to enable usage by object reference instead of inheritance
* - Remove MOST dependencies
* - Adapt to Project needs
*
* \date 25.08.2011 \version 1.1 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Special handling for stream router class due to routing state property
*
* \date 03.02.2011 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#include "arl_ISourceTypes.h"
#include "arl_tclISourceVFsm.h"
#include "arl_tclBaseClienthandler.h"
#include "arl_tclStrClienthandler.h"

#include "arl_tclStrClientHandler_ASF_IF.h"
#include "arl_tclStrClientHandler_AIL.h"

#include "arl_LookupTable.h"
#include "arl_tclISourceFsm.h"
#include "arl_tclISource.h"
#include "arl_Trace.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#define AHL_S_IMPORT_INTERFACE_CCA_EXTENSION
#include <ahl_if.h>

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#define FI_S_IMPORT_INTERFACE_BASE_TYPES
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "common_fi_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ARL_ISOURCE_VFSM
#include "trcGenProj/Header/arl_tclISourceVFsm.cpp.trc.h"
#endif

/******************************************************************************
| defines and macros (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function prototype (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function implementation (scope: external-interfaces)
|----------------------------------------------------------------------------*/

/******************************************************************************
** FUNCTION:  arl_tclISourceVFsm::arl_tclISourceVFsm(ahl_tclBaseOneThreadA..
******************************************************************************/

arl_tclISourceVFsm::arl_tclISourceVFsm // AHL with ASF STR
(
   ahl_tclBaseOneThreadApp* const cpoApp
   , arl_tclISource* const cpoISource
   , arl_tclStrClientHandler_ASF_IF* poAsfStrClh
):m_u16AppId(cpoApp->u16GetAppId()), m_cpoISource(cpoISource), m_poStrRouClient(OSAL_NULL)
, m_poStrRouClientAIL(OSAL_NULL), m_poStrRouClientASF(poAsfStrClh), m_oISrcFsmList()
{
   ETG_TRACE_COMP(("[TASK:%d]Creating ISource VFSM", OSAL_ThreadWhoAmI()));

   NORMAL_M_ASSERT(OSAL_NULL != cpoApp);
   NORMAL_M_ASSERT(OSAL_NULL != m_cpoISource);


   // but we need to pass pointer on us to asf clh
   if(m_poStrRouClientASF != OSAL_NULL)
   {
     m_poStrRouClientASF->setVFsm(this);
   }

   NORMAL_M_ASSERT(OSAL_NULL != m_poStrRouClientASF);

   ETG_TRACE_COMP(("Created ASF Stream router client[0x%08x]"
      , ARL_ETG_OBJ_ADDRESS(m_poStrRouClientASF)));
   /// Initialize the FSM
   vInitFsm();

}  // arl_tclISourceVFsm::arl_tclISourceVFsm(ahl_tclBaseOneThreadApp* const ..

arl_tclISourceVFsm::arl_tclISourceVFsm // AHL
(
   ahl_tclBaseOneThreadApp* const cpoApp
   , arl_tclISource* const cpoISource
):m_u16AppId(cpoApp->u16GetAppId()), m_cpoISource(cpoISource), m_poStrRouClient(OSAL_NULL)
, m_poStrRouClientAIL(OSAL_NULL), m_poStrRouClientASF(OSAL_NULL), m_oISrcFsmList()
{
   ETG_TRACE_COMP(("[TASK:%d]Creating ISource VFSM", OSAL_ThreadWhoAmI()));

   NORMAL_M_ASSERT(OSAL_NULL != cpoApp);
   NORMAL_M_ASSERT(OSAL_NULL != m_cpoISource);

   m_poStrRouClient  =  OSAL_NEW arl_tclStrClienthandler(cpoApp, this
                           , CCA_C_U16_SRV_STREAMROUTER, ARL_STRROU_FI_MAJOR_VERSION
                           , ARL_STRROU_FI_MINOR_VERSION);

   NORMAL_M_ASSERT(OSAL_NULL != m_poStrRouClient);

   ETG_TRACE_COMP(("Created AHL Stream router client[0x%08x]"
      , ARL_ETG_OBJ_ADDRESS(m_poStrRouClient)));
   /// Initialize the FSM
   vInitFsm();

}  // arl_tclISourceVFsm::arl_tclISourceVFsm(ahl_tclBaseOneThreadApp* const ..


arl_tclISourceVFsm::arl_tclISourceVFsm //AIL
(
      ail_tclAppInterfaceRestricted* const cpoApp
   , arl_tclISourceIF* const cpoISource
):m_u16AppId(cpoApp->u16GetAppId()), m_cpoISource(cpoISource), m_poStrRouClient(OSAL_NULL)
, m_poStrRouClientAIL(OSAL_NULL),  m_poStrRouClientASF(OSAL_NULL), m_oISrcFsmList()
{
   ETG_TRACE_COMP(("[TASK:%d]Creating ISource VFSM", OSAL_ThreadWhoAmI()));

   NORMAL_M_ASSERT(OSAL_NULL != cpoApp);
   NORMAL_M_ASSERT(OSAL_NULL != m_cpoISource);

   m_poStrRouClientAIL  =  OSAL_NEW arl_tclStrClientHandler_AIL(cpoApp, this
                           , CCA_C_U16_SRV_STREAMROUTER, ARL_STRROU_FI_MAJOR_VERSION
                           , ARL_STRROU_FI_MINOR_VERSION);

   NORMAL_M_ASSERT(OSAL_NULL != m_poStrRouClientAIL);

   ETG_TRACE_COMP(("Created Stream router client[0x%08x]"
      , ARL_ETG_OBJ_ADDRESS(m_poStrRouClient)));
   /// Initialize the FSM
   vInitFsm();

}  // arl_tclISourceVFsm::arl_tclISourceVFsm(ahl_tclBaseOneThreadApp* const ..


tVoid arl_tclISourceVFsm::vOnServiceState( tU16 u16ServiceId, tU16 u16ServerId, tU16 u16RegisterId, tU8  u8ServiceState, tU16 u16SubId)
{
   if(m_poStrRouClientAIL != NULL)
   {
      m_poStrRouClientAIL->vOnServiceState( u16ServiceId, u16ServerId, u16RegisterId, u8ServiceState, u16SubId);
   }
}


tVoid arl_tclISourceVFsm::vOnAsyncRegisterConf(tU16 u16RegisterId, tU16 u16ServerAppId, tU16 u16ServiceId, tU16 u16TargetSubId)
{
   if(m_poStrRouClientAIL!=NULL)
   {
      m_poStrRouClientAIL->vOnAsyncRegisterConf(u16RegisterId,u16ServerAppId, u16ServiceId, u16TargetSubId);
   }
}

tVoid arl_tclISourceVFsm::vOnNewAppStateNotify(tU32 u32OldAppState, tU32 u32AppState)
{
   ETG_TRACE_USR4(("[%d] arl_tclISourceVFsm::vOnNewAppStateNotify ",
               ETG_ENUM(ail_u16AppId, m_u16AppId)));
   if(m_poStrRouClientAIL!=NULL)
   {
      m_poStrRouClientAIL->vOnNewAppState(u32OldAppState,u32AppState);
   }

   if(AMT_C_U32_STATE_OFF == u32AppState)
   {
      //New AppState need to trigger all sources
      tU8 u8Loop = 1;

      for (arl_tISrcFsmListIter iterFsm = m_oISrcFsmList.begin();
            iterFsm < m_oISrcFsmList.end(); ++iterFsm, ++u8Loop)
      {
         ETG_TRACE_USR1(("Trigger ISource FSM[0x%08x] for Source Number[%d]: AppStateChange from %u to %ul"
               , ARL_ETG_OBJ_ADDRESS((*iterFsm)), (*iterFsm)->u8GetSourceNumber(), u32OldAppState, u32AppState ));
         // Invoke the state machine.
         arl_tclISourceFsm& rfoISrcFsm =  *(*iterFsm);
         rfoISrcFsm(rfoISrcFsm.rfcoGetTriggerId().m_s16AppState_Off);
      }
   }
}

/******************************************************************************
** FUNCTION:  virtual arl_tclISourceVFsm::~arl_tclISourceVFsm()
******************************************************************************/

/*virtual*/
arl_tclISourceVFsm::~arl_tclISourceVFsm()
{
   try
   {
      ETG_TRACE_COMP(("[%d]Destroying ISource VFSM",  ETG_ENUM(ail_u16AppId, m_u16AppId)));

      tU8 u8Loop = 1;

      for (arl_tISrcFsmListIter iterFsm = m_oISrcFsmList.begin();
         iterFsm < m_oISrcFsmList.end(); ++iterFsm, ++u8Loop)
      {
         ETG_TRACE_COMP(("Iterate MuteState map"));
         if(m_MuteState[static_cast<arl_tenSource>((*iterFsm)->u8GetSourceNumber())] != NULL)
         {
            ETG_TRACE_COMP(("Deleting m_MuteState for Source Number[%d]"
                  ,(*iterFsm)->u8GetSourceNumber()));
            OSAL_DELETE m_MuteState[static_cast<arl_tenSource>((*iterFsm)->u8GetSourceNumber())];
            m_MuteState[static_cast<arl_tenSource>((*iterFsm)->u8GetSourceNumber())] = NULL;
         }

         ETG_TRACE_COMP(("Deleting ISource FSM[0x%08x] for Source Number[%d]"
            , ARL_ETG_OBJ_ADDRESS((*iterFsm)), (*iterFsm)->u8GetSourceNumber() ));
#if 0
         ETG_TRACE_COMP(("Deleting ISource FSM[0x%08x] for Source Number[%d]"
            , ARL_ETG_OBJ_ADDRESS((*iterFsm)), u8Loop));
#endif
         OSAL_DELETE *iterFsm;
         *iterFsm =  OSAL_NULL;
      }

      ETG_TRACE_COMP(("Deleting Stream router client[0x%08x]"
            , ARL_ETG_OBJ_ADDRESS(m_poStrRouClient)));

      OSAL_DELETE m_poStrRouClient;
      OSAL_DELETE m_poStrRouClientAIL;

      m_oISrcFsmList.clear();

   }
   catch (...)
   {
   }

   m_poStrRouClient  =  OSAL_NULL;
   m_poStrRouClientAIL =  OSAL_NULL;
   m_poStrRouClientASF =  OSAL_NULL;
}  // arl_tclISourceVFsm::~arl_tclISourceVFsm()

/******************************************************************************
** FUNCTION:  tVoid arl_tclISourceVFsm::vInitFsm()
******************************************************************************/

tVoid arl_tclISourceVFsm::vInitFsm()
{
   arl_tSourceList mySourceList;

   // Get total number of sources existing for this service.
   arl_s16GetNumSrc(m_u16AppId, mySourceList);

   // If Number of sources is  > 0
   if (0 != mySourceList.size())
   {
      arl_tSourceList::iterator sourceIterator;

      // Create state machines for each source within the lookup table
      for (sourceIterator = mySourceList.begin(); sourceIterator != mySourceList.end();
            ++sourceIterator)
      {
         m_oISrcFsmList.push_back(poCreateISrcFsm(*sourceIterator));
         m_MuteState[static_cast<arl_tenSource>((*sourceIterator))] = NULL;
      }

   }  // if (ARL_ERROR != s16NumSrcs)
   else
   {
      ETG_TRACE_ERR(("ARL Lookup Table returned 0 Sources for AppId: %d!!"
         , ETG_ENUM(ail_u16AppId, m_u16AppId)));

      NORMAL_M_ASSERT_ALWAYS();

   }  // End of if-else; if (ARL_ERROR != s16NumSrcs)
}  // tVoid arl_tclISourceVFsm::vInitFsm()

/******************************************************************************
** FUNCTION:  tVoid arl_tclISourceVFsm::vSetMuteState(...
******************************************************************************/
tVoid arl_tclISourceVFsm::vSetMuteState(arl_tenSource enSrcNum, arl_tenMuteAction enMuteAction, audioroutinglib::arl_tclBitmask<arl_enMuteReason> muteReason)
{

   //First check if the given source is maintained by application
   if(arl_bGetISrcAppId(enSrcNum) == m_u16AppId)
   {
      ETG_TRACE_USR2(("0x%x vSetMuteState: Request Source %d, MuteAction: %d, muteReason: %d",
                 m_u16AppId,
                 ETG_ENUM(arl_tenSource,    enSrcNum),
                 ETG_ENUM(arl_tenMuteState, enMuteAction),
                 muteReason.getValue()));
      if(m_MuteState[enSrcNum] == NULL)
         {
            m_MuteState[enSrcNum] = new arl_stSetMuteTuple;
            m_MuteState[enSrcNum]->muteReason = muteReason;
            m_MuteState[enSrcNum]->muteAction = enMuteAction;
         }else{
            m_MuteState[enSrcNum]->muteAction = enMuteAction;
            m_MuteState[enSrcNum]->muteReason = muteReason;
         }

   }else{
      ETG_TRACE_ERR(("[%d] vSetMuteState: APPLICATION NOT ALLOWED TO PLACE REQUESTS FOR SOURCE: %d, MUTEACTION: %d, MUTEREASON: 0x%x",
                  ETG_ENUM(ail_u16AppId,     m_u16AppId  ),
                  ETG_ENUM(arl_tenSource,    enSrcNum    ),
                  ETG_ENUM(arl_tenMuteState, enMuteAction),
                  muteReason.getValue()));
   }
}

/******************************************************************************
** FUNCTION:  tVoid arl_tclISourceVFsm::vSetRouteRequest(...
******************************************************************************/
// daw2hi: updated to use SourceID where subsource is included
tVoid arl_tclISourceVFsm::vSetRouteRequest(SourceID SrcId, arl_tenActivity enRequest, audioroutinglib::arl_tclBitmask<arl_enRouteReqReason> reason)
{
   //First check if the given source is maintained by application
   if(arl_bGetISrcAppId(SrcId.enSourceClass) == m_u16AppId)
   {
      ETG_TRACE_USR2(("[%d]vSetRouteRequest: Request Source %d, SubSrc %d, Activity: %d, reason: %d",
            ETG_ENUM(ail_u16AppId, m_u16AppId),
            ETG_ENUM(arl_tenSource, SrcId.enSourceClass),
            SrcId.u16SubSource,
            ETG_ENUM(arl_tenActivity, enRequest),
            reason.getValue()));

// In Gen4 we have adapted SDS_TTS to clear expiered requests. It is in the responsibility of the component to handle it correct
// SDS_TTS will have several different audio sources
#ifdef VARIANT_S_FTR_ENABLE_CONFIG_AUDIO_A2S1
    if(m_u16AppId != CCA_C_U16_APP_SDS_TTS)  vClearRouteReq();
#else
    vClearRouteReq();
#endif
      m_RouteReq[SrcId] = enRequest;
   }else{
      ETG_TRACE_ERR(("[%d] vSetRouteRequest: APPLICATION NOT ALLOWED TO PLACE REQUESTS FOR SOURCE: %d, SUBSOURCE: %d, ACTIVITY: %d, REASON: 0x%x",
            ETG_ENUM(ail_u16AppId,     m_u16AppId  ),
            ETG_ENUM(arl_tenSource, SrcId.enSourceClass),
            SrcId.u16SubSource,
            ETG_ENUM(arl_tenActivity, enRequest),
            reason.getValue()));
   }
}
/******************************************************************************
** FUNCTION:  tVoid arl_tclISourceVFsm::vClearRouteReq(...
******************************************************************************/
// paj5kor: Clear previous route requests of application
tVoid arl_tclISourceVFsm::vClearRouteReq()
{
  ETG_TRACE_USR4(("vClearRouteReq is entered"));
  std::map<SourceID,  arl_tenActivity>::iterator routeReqMapIter = m_RouteReq.begin();
  while( routeReqMapIter != m_RouteReq.end())
  {
    ETG_TRACE_USR4(("arl_tclISourceVFsm::vClearRouteReq  Route Req map Size = %d", m_RouteReq.size()));
      ETG_TRACE_USR4(("arl_tclISourceVFsm::vClearRouteReq::Clearing source from RouteReq map is Source=%d SubSrc=%d",
        (*routeReqMapIter).first.enSourceClass,
        (*routeReqMapIter).first.u16SubSource));
      m_RouteReq.erase(routeReqMapIter);
      routeReqMapIter = m_RouteReq.begin();//Reinitialize iterator to beginning
  }
  ETG_TRACE_USR4(("arl_tclISourceVFsm::vClearRouteReq is ended, Route Req map Size = %d", m_RouteReq.size()));
}
/******************************************************************************
** FUNCTION:  tVoid arl_tclISourceVFsm::vSetRouteRequest(...
******************************************************************************/
tVoid arl_tclISourceVFsm::vSetSourceAvailability(arl_tenSrcAvailability srcAvailability,
                                                 arl_tenAvailabilityReason availabilityReason,
                                                 arl_tenSource enSource,
                                                 tU16 u16SubSource)
{
   //First check if the given source is maintained by application
   if(arl_bGetISrcAppId(enSource) == m_u16AppId)
   {
      ETG_TRACE_USR2(("[%d]vSetSourceAvailability: Source %d, SubSource %d, Availability: %d, Reason: %d",
            ETG_ENUM(ail_u16AppId,              m_u16AppId),
            ETG_ENUM(arl_tenSource,             enSource),
            u16SubSource,
            ETG_ENUM(arl_tenSrcAvailability,    srcAvailability),
            ETG_ENUM(arl_tenAvailabilityReason, availabilityReason)));
      arl_stSrcAvailabilityTuple sat = {srcAvailability, availabilityReason};
      m_SrcAvailability[SourceID(enSource, u16SubSource)] = sat;//arl_stSrcAvailabilityTuple(srcAvailability, availabilityReason);
   }else{
      ETG_TRACE_ERR(("[%d]vSetSourceAvailability: APPLICATION NOT ALLOWED TO PLACE REQUESTS FOR SOURCE: %d, SUBSOURCE %d, ACTIVITY: %d, REASON: 0x%x",
            ETG_ENUM(ail_u16AppId,              m_u16AppId),
            ETG_ENUM(arl_tenSource,             enSource),
            u16SubSource,
            ETG_ENUM(arl_tenSrcAvailability,    srcAvailability),
            ETG_ENUM(arl_tenAvailabilityReason, availabilityReason)));
   }
}
tVoid arl_tclISourceVFsm::vSetAudioProperty(arl_tenSource enSource,
                         arl_tenSourcePropType srcPropType,
                         tS32 SourcePropValue,tU16 u16SubSource)
{
   //First check if the given source is maintained by application
   ETG_TRACE_USR4(("vSetAudioProperty is entered"));
   if(arl_bGetISrcAppId(enSource) == m_u16AppId)
   {
      arl_stSrcAudioPropertyTuple sat = {srcPropType, SourcePropValue};
      m_SrcAudioProperty[SourceID(enSource, u16SubSource)] = sat;//arl_stSrcAvailabilityTuple(srcAvailability, availabilityReason);
    ETG_TRACE_USR4(("m_SrcAudioProperty is updated with information received"));
   }else{
   }
}

// here we do not really need to care for ASF. FG only used in Gen2.
tBool arl_tclISourceVFsm::bSendFGGraphCmd(arl_tenSource enSrcNum, arl_tenFGCommand enFGCmd)
{
   tBool bServiceAvail = FALSE;

   if(m_poStrRouClient != NULL)
   {
      bServiceAvail = m_poStrRouClient->bIfServiceAvailable();
   }

   if(m_poStrRouClientAIL != NULL)
   {
      bServiceAvail = m_poStrRouClientAIL->bIfServiceAvailable();
   }

   if (true == bServiceAvail)
   {
       //Get UCID for source
       tU16 u16UCID = arl_s16SrcUCID(enSrcNum);
       // Prepare the Allocate route message.
       midw_streamrouterfi_tclMsgSendFGCommandMethodStart oMsgFGCmd;
       oMsgFGCmd.u16UCID  =  u16UCID;

       //actually there is no SubRoute so
       //we can set this to zero
       oMsgFGCmd.u16FGRouteID = 0;
       oMsgFGCmd.eCmd.enType = static_cast<midw_fi_tcl_e8_streamrouter_fgcommand::tenType>(enFGCmd);

       ETG_TRACE_USR1(("[%d]Sending FG command message to stream router\
       UCID:0x%x, SubRoute:%d, Cmd:%d", ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
       , oMsgFGCmd.u16UCID, oMsgFGCmd.u16FGRouteID,
       ETG_ENUM(arl_tenFGCommand, static_cast<arl_tenFGCommand>(oMsgFGCmd.eCmd.enType))));

       // Trigger the sending of the stream router message.
       if( m_oISrcFsmList.front()!=NULL)
       {
          return m_oISrcFsmList.front()->bSendStrRouMsg(oMsgFGCmd);
       }
       return FALSE;
   }
   ETG_TRACE_ERR(("[%d]bServiceAvail == FALSE",
              ETG_ENUM(ail_u16AppId, m_u16AppId)));
   return FALSE;
}

/******************************************************************************
** FUNCTION:  tVoid arl_tclISrcVFsm::vOnSrcActResult(arl_tenSource src..
******************************************************************************/

tVoid arl_tclISourceVFsm::vOnSrcActResult(arl_tenSource src, arl_tenActivity act)
{
   // we have to search again if we find the source number inside the vector elements
       // check if this source number is inside the list
       tBool bFound=false;
       for(tU8 i=0; i<m_oISrcFsmList.size(); i++)
       {
         if(src == static_cast<arl_tenSource>(m_oISrcFsmList[i]->u8GetSourceNumber()))
         {
            bFound = true;
            arl_tclISourceFsm& rfoISrcFsm =  (*(m_oISrcFsmList[i]));
            rfoISrcFsm.vActResult(act);
            //rfoISrcFsm(rfoISrcFsm.rfcoGetTriggerId().m_s16AppAckActivity);
         }
       }
       if(bFound == false)
       {
            NORMAL_M_ASSERT_ALWAYS();
       }
}


/******************************************************************************
** FUNCTION:  arl_tclISourceFsm* arl_tclISourceVFsm::poCreateISrcFsm(tU8 u8Sr..
******************************************************************************/

arl_tclISourceFsm* arl_tclISourceVFsm::poCreateISrcFsm(tU8 u8SrcNum) const
{
   arl_tclISourceFsm* poISourceFsm = OSAL_NULL;
   //Check if we are AIL or AHL
   if(m_poStrRouClientAIL != NULL)
   {
      poISourceFsm   =  OSAL_NEW arl_tclISourceFsm(u8SrcNum, m_cpoISource, m_poStrRouClientAIL);
   }
   if(m_poStrRouClient != NULL)
   {
         //Because of Lint we introduce nonsense
         if(poISourceFsm!=NULL)
         {
            OSAL_DELETE(poISourceFsm);
         }
         //end of nonsense
         poISourceFsm   =  OSAL_NEW arl_tclISourceFsm(u8SrcNum, m_cpoISource, m_poStrRouClient);
   }

   if(m_poStrRouClientASF != NULL)
   {
         //Because of Lint we introduce nonsense
         if(poISourceFsm!=NULL)
         {
            OSAL_DELETE(poISourceFsm);
         }
         //end of nonsense
         poISourceFsm   =  OSAL_NEW arl_tclISourceFsm(u8SrcNum, m_cpoISource, m_poStrRouClientASF);
   }
   ETG_TRACE_COMP(("Created ISource FSM[0x%08x] for Source number[%d]"
         , ARL_ETG_OBJ_ADDRESS(poISourceFsm), u8SrcNum));

   NORMAL_M_ASSERT(OSAL_NULL != poISourceFsm);
   return poISourceFsm;
}  // arl_tclISourceFsm* arl_tclISourceVFsm::poCreateISrcFsm(tU8 u8SrcNum) const


tBool arl_tclISourceVFsm::bOnAllocRouteResult(tU16 u16UCID, int srvID, int fktID, tU8 u8Src,
    std::vector< std::string > AlsaList_in, std::vector< std::string > AlsaList_out)
{
  ETG_TRACE_FATAL(("bOnAllocRouteResult processed"));
    tBool bRet=false;
  // would be nice to create amt_tclServiceData and call the bOnMessage
  midw_streamrouterfi_tclMsgAllocateRouteMethodResult outmsg;
  outmsg.u16UCID = u16UCID;
//  outmsg.listInputDev.strALSADev = AlsaList_in;
//  outmsg.listOutputDev.strALSADev = AlsaList_out;

  std::vector<std::string>::iterator it;
  for(it=AlsaList_in.begin(); it != AlsaList_in.end();++it)
  {
    midw_fi_tclString oString((*it).c_str());
    outmsg.listInputDev.strALSADev.push_back(oString);
  }
  for(it=AlsaList_out.begin(); it != AlsaList_out.end();++it)
  {
    midw_fi_tclString oFiString((*it).c_str());
    outmsg.listOutputDev.strALSADev.push_back(oFiString);
  }

  fi_tclVisitorMessage oResultMsg(outmsg);

  oResultMsg.vInitServiceData(
               CCA_C_U16_APP_FC_AUDIO_ROUTE_MGR,  // Source AppID
               this->m_u16AppId, // Target AppID
                     0,                // stream type ??
                     0,           // stream counter
                     0,                // RegisterID ??
                     u8Src,            // CmdCounter,
                     (tU16)srvID,
                     (tU16)fktID,           // Function ID MIDW_STREAMROUTERFI_C_U16_ALLOCATEROUTE
                     CCA_C_U8_OPCODE_METHODRESULT,       // Opcode CCA_C_U8_OPCODE_METHODRESULT (0x07)
                     0,0,0,0);

    if(true == bOnMessage(&oResultMsg))
    {
      bRet = true;
    }

    // ToDo: destroy message
    oResultMsg.bDelete();
    return bRet;
}

tBool arl_tclISourceVFsm::bOnDeAllocRouteResult(tU16 u16UCID, int srvID, int fktID, tU8 u8Src) //need also ucid
{
  ETG_TRACE_FATAL(("bOnDeAllocRouteResult processed"));
  // would be nice to create amt_tclServiceData and call the bOnMessage
  midw_streamrouterfi_tclMsgDeAllocateRouteMethodResult outmsg;
  outmsg.u16UCID = u16UCID;
  fi_tclVisitorMessage oResultMsg(outmsg);
  tBool bRet= false;

  oResultMsg.vInitServiceData(
               CCA_C_U16_APP_FC_AUDIO_ROUTE_MGR,  // Source AppID
               this->m_u16AppId, // Target AppID
                     0,                // stream type ??
                     0,           // stream counter ??
                     0,                // RegisterID
                     u8Src,            // CmdCounter,
                     (tU16)srvID,
                     (tU16)fktID,           // Function ID MIDW_STREAMROUTERFI_C_U16_ALLOCATEROUTE
                     CCA_C_U8_OPCODE_METHODRESULT,       // Opcode CCA_C_U8_OPCODE_METHODRESULT (0x07)
                     0,0,0,0);

    if(true == bOnMessage(&oResultMsg))
  {
    bRet = true;
  }

    // ToDo: destroy message
    oResultMsg.bDelete();
  return bRet;
}

// TODo: check parameters
tBool arl_tclISourceVFsm::bOnPlayResult(tU16 u16UCID, int srvID, int fktID, tU8 u8Src) //need also ucid
{
  ETG_TRACE_FATAL(("bOnPlayResult processed"));
  // would be nice to create amt_tclServiceData and call the bOnMessage
  midw_streamrouterfi_tclMsgSetPlayMethodResult outmsg;
  outmsg.u16UCID = u16UCID;
  fi_tclVisitorMessage oResultMsg(outmsg);
  tBool bRet= false;

  oResultMsg.vInitServiceData(
               CCA_C_U16_APP_FC_AUDIO_ROUTE_MGR,  // Source AppID
               this->m_u16AppId, // Target AppID
                     0,                // stream type ??
                     0,           // stream counter ??
                     0,                // RegisterID ??
                     u8Src,             // CmdCounter (should be source number u8Src ?)
                     (tU16)srvID,
                     (tU16)fktID,           // Function ID MIDW_STREAMROUTERFI_C_U16_ALLOCATEROUTE
                     CCA_C_U8_OPCODE_METHODRESULT,       // Opcode CCA_C_U8_OPCODE_METHODRESULT (0x07)
                     0,0,0,0);

    if(true == bOnMessage(&oResultMsg))
  {
    bRet = true;
  }

    // ToDo: destroy message
    oResultMsg.bDelete();
  return bRet;
}

// TODo: check parameters
tBool arl_tclISourceVFsm::bOnStopResult(tU16 u16UCID, int srvID, int fktID, tU8 u8Src) //need also ucid
{
  ETG_TRACE_FATAL(("bOnStopResult processed"));
  // would be nice to create amt_tclServiceData and call the bOnMessage
  midw_streamrouterfi_tclMsgSetStopMethodResult outmsg;
  outmsg.u16UCID = u16UCID;
  fi_tclVisitorMessage oResultMsg(outmsg);
  tBool bRet= false;

  oResultMsg.vInitServiceData(
               CCA_C_U16_APP_FC_AUDIO_ROUTE_MGR,  // Source AppID
               this->m_u16AppId, // Target AppID
                     0,                // stream type ??
                     0,           // stream counter ??
                     0,                // RegisterID ??
                     u8Src,             // CmdCounter (should be source number u8Src ?)
                     (tU16)srvID,
                     (tU16)fktID,           // Function ID MIDW_STREAMROUTERFI_C_U16_ALLOCATEROUTE
                     CCA_C_U8_OPCODE_METHODRESULT,       // Opcode CCA_C_U8_OPCODE_METHODRESULT (0x07)
                     0,0,0,0);

    if(true == bOnMessage(&oResultMsg))
  {
    bRet = true;
  }

    oResultMsg.bDelete();
  return bRet;
}

//tBool arl_tclISourceVFsm::bOnRoutingStateStatus(std::vector<midw_fi_tcl_streamrouter_route_entry,
//    std::allocator<midw_fi_tcl_streamrouter_route_entry> > fiRouteEntryList, int srvID, int fktID)

tBool arl_tclISourceVFsm::bOnRoutingStateStatus(std::vector<midw_fi_tcl_streamrouter_route_entry> fiRouteEntryList,
    int srvID, int fktID)
{
  midw_streamrouterfi_tclMsgRoutingStateStatus outmsg;

  // fill parameter
//  std::vector<midw_fi_tcl_streamrouter_route_entry>::iterator it;
//  for(it=fiRouteEntryList.begin(); it!=fiRouteEntryList.end();++it)
//  {
//    outmsg.routing_list.sREntry.push_back(*it);
//  }
  outmsg.routing_list.sREntry = fiRouteEntryList;

  fi_tclVisitorMessage oStatustMsg(outmsg);
  tBool bRet= false;

  oStatustMsg.vInitServiceData(
               CCA_C_U16_APP_FC_AUDIO_ROUTE_MGR,  // Source AppID
               this->m_u16AppId, // Target AppID
                     0,                // stream type ??
                     0,           // stream counter ??
                     0,                // RegisterID ??
                     0,             // CmdCounter (should be source number u8Src ?)
                     (tU16)srvID,
                     (tU16)fktID,           // Function ID MIDW_STREAMROUTERFI_C_U16_ALLOCATEROUTE
                     CCA_C_U8_OPCODE_STATUS,       // Opcode CCA_C_U8_OPCODE_STATUS
                     0,0,0,0);

    if(true == bOnMessage(&oStatustMsg))
  {
    bRet = true;
  }

    oStatustMsg.bDelete();
  return bRet;
}


/******************************************************************************
** FUNCTION:  tBool arl_tclISourceVFsm::bOnMessage(amt_tclServiceData*..
******************************************************************************/

tBool arl_tclISourceVFsm::bOnMessage(amt_tclServiceData* poMessage)
{
   tBool bRetVal  =  true;

   if (FALSE == ahl_tclBaseWork::bDefaultSvcDataHandler(this, poMessage))
   {
      // Programmer error for message map in the ISource FSM.
      ETG_TRACE_ERR(("[TASK:%d]Message map is incomplete, FID[0x%x]!!"
         , OSAL_ThreadWhoAmI(), poMessage->u16GetFunctionID()));

      NORMAL_M_ASSERT_ALWAYS();
      bRetVal  =  FALSE;
   }

   return bRetVal;

}  // tBool arl_tclISourceVFsm::bOnMessage(amt_tclServiceData* poMessage)

/******************************************************************************
** FUNCTION:  tBool arl_tclISourceVFsm::bServiceAvailability(tU16 u16ServiceI..
******************************************************************************/

/*virtual*/
tBool arl_tclISourceVFsm::bServiceAvailability(tU16 u16ServiceId, tBool bAvail)
{
   ETG_TRACE_ERR(("[TASK:%d]Service availability not handled!!", OSAL_ThreadWhoAmI()));

   // Avoid Lint - Parameter needs to be neglected without LINT objection.
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u16ServiceId);
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(bAvail);

   return FALSE;

}  // tBool arl_tclISourceVFsm::bServiceAvailability(tU16 u16ServiceId, tBool..

////////////////////////////////////////////////////////////////////////////////
// <EOF>
