/*!
*******************************************************************************
* \file              arl_tclISourceVFsm.h
*******************************************************************************
*  - PROJECT:        LCN2KAI
*  - SW-COMPONENT:   Audio Routing Library (ARL)
*  - DESCRIPTION:    Implements ISource Virtual FSM base class
*  - COPYRIGHT:      &copy; 2011 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 26.06.2012 \version 2.0 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
* - Remove AHL inheritance to enable usage by object reference instead of inheritance
* - Remove MOST dependencies
* - Adapt to Project needs
*
* \date 02.02.2011 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

#ifndef ARL_TCLISOURCEVFSM_H_
#define ARL_TCLISOURCEVFSM_H_

/******************************************************************************
| includes of component-internal interfaces, if necessary
| (scope: component-local)
|----------------------------------------------------------------------------*/

/// OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include <osal_if.h>

//#define SYSTEM_S_IMPORT_INTERFACE_ITERATOR   ///< Import STL IF - Iterator
//#define SYSTEM_S_IMPORT_INTERFACE_VECTOR     ///< Import STL IF - Vector
//#define SYSTEM_S_IMPORT_INTERFACE_STRING
//#include <stl_pif.h>

#include  <string>
#include  <iterator>
#include  <vector>





//using namespace std;


/* use AIL template */
#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"

#define AHL_S_IMPORT_INTERFACE_BASEWORK
#include <ahl_if.h>

#include "arl_tclISourceIF.h"
#include "arl_tclStrClientHandler_AIL.h"

#include "arl_tclStrClientHandler_ASF_IF.h"
/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| forward declarations
|----------------------------------------------------------------------------*/

class ahl_tclBaseOneThreadApp;
class arl_tclISource;
class arl_tclISource_AILBase;
class arl_tclBaseClienthandler;
//class arl_tclBaseClienthandlerAIL;
class arl_tclISourceFsm;

/*!
 * \class arl_tclISourceVFsm
 * \brief
 * \b Motivation:
 * A virtual finite state machine implementation. The VFSM is a hybrid
 * finite state model which is used to acheive a superior design clarity.
 * This separates the control part of the design from the data processing part.
 * The idea of VFSM model facilitates an executable specification & introduces
 * an execution model.
 *
 * \par
 * This base class is the "Super" state in which all state machines for each
 * source resides.
 *
 * Features implemented:
 *
 *    - Creation & management of FSM's based on source numbers.
 *    - Configuring the different souce FSM's.
 *    - Internally provides additional features for message mapping.
 *    - Virtual implementation to map service availability.
 *
 */


class arl_tclISourceVFsm : public ahl_tclBaseWork
{

   friend class arl_tclISourceIF;
   friend class arl_tclISource_AIL;

public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  arl_tclISourceVFsm::arl_tclISourceVFsm(ahl_tclBaseOneThreadA..
   ***************************************************************************/
   /*!
    * \brief   Parameterized Constructor, based on Dependency Injection
    *          Principle (DIP)
    * \param   [cpoApp]:      (->I) Main Application pointer.
    * \param   [cpoISource]:  (->I) ISource service pointer.
    * \retval  NONE
    * \date 03.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclISourceVFsm //AHL
   (
      ahl_tclBaseOneThreadApp* const cpoApp
      , arl_tclISource* const cpoISource
   );

   arl_tclISourceVFsm  //AIL
   (
         ail_tclAppInterfaceRestricted* const cpoApp
      , arl_tclISourceIF* const cpoISource
   );

   arl_tclISourceVFsm // AHL with ASF STR
   (
      ahl_tclBaseOneThreadApp* const cpoApp
      , arl_tclISource* const cpoISource
      , arl_tclStrClientHandler_ASF_IF* poAsfStrClh
   );

   tVoid vOnAsyncRegisterConf(tU16 u16RegisterId, tU16 u16ServerAppId, tU16 u16ServiceId, tU16 u16TargetSubId);
   virtual tVoid vOnNewAppStateNotify(tU32 /*u32OldAppState*/, tU32 /*u32AppState*/);
   /***************************************************************************
   ** FUNCTION:  virtual arl_tclISourceVFsm::~arl_tclISourceVFsm()
   ***************************************************************************/
   /*!
    * \brief   Destructor
    * \param   NONE
    * \retval  NONE
    * \date 03.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual ~arl_tclISourceVFsm();

   /***************************************************************************
   ** FUNCTION:  tBool arl_tclISourceVFsm::bOnMessage(amt_tclServiceData* poM..
   ***************************************************************************/
   /*!
    * \brief   Function to handle Incoming message.
    * \param   [poMessage]:    (->I)   Pointer to incoming message.
    * \retval  [tBool]: TRUE if incoming message was handled, FALSE otherwise.
    * \date 07.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bOnMessage(amt_tclServiceData* poMessage);

   // not clear what params I can use here. I would need to create a amt_tclServiceData out of it
   tBool bOnAllocRouteResult(tU16 u16UCID, int srvID, int fktID, tU8 u8Src,
       std::vector<std::string> AlsaList_in, std::vector<std::string> AlsaList_out);
   //tBool vOnAllocRouteResult(tU16 u16UCID, int srvID, int fktID, tU8 u8Src);

   tBool bOnDeAllocRouteResult(tU16 u16UCID, int srvID, int fktID, tU8 u8Src);


   tBool bOnPlayResult(tU16 u16UCID, int srvID, int fktID, tU8 u8Src);
   tBool bOnStopResult(tU16 u16UCID, int srvID, int fktID, tU8 u8Src);

   //tBool bOnRoutingStateStatus();
//   tBool bOnRoutingStateStatus(std::vector<midw_fi_tcl_streamrouter_route_entry,
//       allocator<midw_fi_tcl_streamrouter_route_entry> > fiRouteEntryList,int srvID, int fktID);

   tBool bOnRoutingStateStatus(std::vector<midw_fi_tcl_streamrouter_route_entry> fiRouteEntryList,int srvID, int fktID);



   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceVFsm::bServiceAvailability(tU16..
   ***************************************************************************/
   /*!
    * \brief   Signals the service availabiltiy/unavailability of service.
    * \param   [u16ServiceId]:      (I)   Service Id
    * \param   [bAvail]:            (I)   Availability
    * \retval  [tBool]: TRUE service availability was consumed, FALSE otherwise.
    * \date 10.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bServiceAvailability(tU16 u16ServiceId, tBool bAvail = TRUE);

   virtual tVoid vOnServiceState( tU16 u16ServiceId, tU16 u16ServerId, tU16 u16RegisterId, tU8  u8ServiceState, tU16 u16SubId);

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceVFsm::vSetMuteState(t...
   ***************************************************************************/
   /*!
    * \brief   Request from Player for de-/mute
    * \param   [enSrcNum]:      Source
    * \param   [enMuteAction]:  Request De-/Mute
    * \param   [muteReason]:    Reason for the request
    * \retval  NONE
    * \date 12.11.2012 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
    * \date 04.03.2014 \adapted by daw2hi to use SourceID where subsource is included
    **************************************************************************/
   virtual tVoid vSetRouteRequest(SourceID SrcId, arl_tenActivity enRequest, audioroutinglib::arl_tclBitmask<arl_enRouteReqReason> reason);
   virtual std::map<SourceID, arl_tenActivity> const  tGetRouteRequestMap() { return m_RouteReq; }


   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceVFsm::vSetMuteState(t...
   ***************************************************************************/
   /*!
    * \brief   Announcement of Player about availability change of a source
    * \param   [enSrcNum]:      Source
    * \param   [srcAvailability]:  the availability of given source
    * \param   [availabilityReason]:    Reason for the change
    * \retval  NONE
    * \date 26.11.2013 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
    **************************************************************************/
   virtual tVoid vSetSourceAvailability(arl_tenSrcAvailability srcAvailability, arl_tenAvailabilityReason availabilityReason, arl_tenSource enSource, tU16 u16SubSource = 0);
   virtual std::map<SourceID, arl_stSrcAvailabilityTuple> const  tGetSourceAvailabilityMap() { return m_SrcAvailability; }


   virtual tVoid vSetAudioProperty(arl_tenSource enSource,arl_tenSourcePropType srcPropType, tS32 SourcePropValue, tU16 u16SubSource = 0);
   virtual std::map<SourceID, arl_stSrcAudioPropertyTuple> const  tGetSrcAudioProperty() { return m_SrcAudioProperty; }
   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceVFsm::vSetMuteState(t...
   ***************************************************************************/
   /*!
    * \brief   Request from Player for de-/mute
    * \param   [enSrcNum]:      Source
    * \param   [enMuteAction]:  Request De-/Mute
    * \param   [muteReason]:    Reason for the request
    * \retval  NONE
    * \date 12.11.2012 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
    **************************************************************************/
   virtual tVoid vSetMuteState(arl_tenSource enSrcNum, arl_tenMuteAction enMuteAction, audioroutinglib::arl_tclBitmask<arl_enMuteReason> muteReason);

   virtual std::map<arl_tenSource, arl_stSetMuteTuple * > const  tGetMuteStateMap() { return m_MuteState; }

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceVFsm::bSendFGGraphCmd(t...
   ***************************************************************************/
   /*!
    * \brief   Send FG command to StreamRouter
    * \param   [u16UCID]:      ucid
    * \param   [enFGCmd]:  command
    * \retval  TRUE if command was send else FALSE
    * \date 06.12.2012 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
    **************************************************************************/
   tBool bSendFGGraphCmd(arl_tenSource enSrcNum, arl_tenFGCommand enFGCmd);

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISrcVFsm::vOnSrcActResult(arl_tenSource..
   ***************************************************************************/
   /*!
    * \brief   Function to handle source activity result.
    * \param   [src]:    Source that activity is getting acknowledged
    * \param   [act]:    Activity that is acknowledged for given source
    * \retval  NONE
    * \date 12.12.2012 \author Ruben Volkmer (mailto:ITB-Solutions.de)
    **************************************************************************/
   tVoid vOnSrcActResult(arl_tenSource src, arl_tenActivity act);


   tU16 u16GetAppId() { return m_u16AppId; }

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *******************************PROTECTED************************************
   ***************************************************************************/

   /*!
    * \typedef vector<arl_tclISourceFsm*> arl_tISrcFsmList
    * Data structure for holding ISource FSM's.
    */

   typedef std::vector<arl_tclISourceFsm*> arl_tISrcFsmList;

   /*!
    * \typedef vector<arl_tclISourceFsm*>::iterator arl_tISrcFsmListIter
    * Data structure for iterating through the ISource FSM List.
    */

   typedef std::vector<arl_tclISourceFsm*>::iterator arl_tISrcFsmListIter;

   /***************************************************************************
   ** FUNCTION:  arl_tclISourceVFsm::arl_tclISourceVFsm()
   ***************************************************************************/
   /*!
    * \brief   Default Constructor
    * \param   NONE
    * \retval  NONE
    * \note    Default constructor is declared protected to disable it. So
    *          that any attempt to create without parameter will be caught by
    *          the compiler.
    * \date 03.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclISourceVFsm();      // No definition exists.

   /***************************************************************************
   ** FUNCTION:  arl_tclISourceVFsm::arl_tclISourceVFsm(const arl_tclISourceV..
   ***************************************************************************/
   /*!
    * \brief   Copy Constructor
    * \param   [rfcoISourceVFsm]: (I) Const reference to object to be copied
    * \retval  NONE
    * \note    Default copy constructor is declared protected to disable it. So
    *          that any attempt to copy will be caught by the compiler.
    * \date 03.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclISourceVFsm(const arl_tclISourceVFsm& rfcoISourceVFsm); // No definition exists.

   /***************************************************************************
   ** FUNCTION:  arl_tclISourceVFsm& arl_tclISourceVFsm::operator=(co ..
   ***************************************************************************/
   /*!
    * \brief   Assignment Operator
    * \param   [rfcoISourceVFsm]: (I) Const reference to object to be copied
    * \retval  [arl_tclISourceVFsm&]: Reference to this pointer.
    * \note    Assignment operator is declared protected to disable it. So
    *          that any attempt for assignment will be caught by the compiler.
    * \date 03.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclISourceVFsm& operator=(const arl_tclISourceVFsm& rfcoISourceVFsm);


   /*!
    * \addtogroup tclMem
    */
   /*! @{*/
   /// Collection of Source specific actual and requested MuteState
   std::map<arl_tenSource, arl_stSetMuteTuple * > m_MuteState;

   /// Collection of Source specific actual and requested MuteState
   //  daw2hi: updated 1st param to be a SourceID where subsource is included
   std::map<SourceID, arl_tenActivity> m_RouteReq;

   /// Collection of Source specific actual and requested MuteState
   std::map<SourceID, arl_stSrcAvailabilityTuple> m_SrcAvailability;

   /// Collection of Source specific audio property
   std::map<SourceID, arl_stSrcAudioPropertyTuple> m_SrcAudioProperty;
   /// Application id
   tU16 m_u16AppId;
   /// ISource service handler pointer.
   arl_tclISourceIF* const m_cpoISource;

   /// Stream router client handler.
   arl_tclBaseClienthandler*    m_poStrRouClient;
   arl_tclBaseClienthandlerAIL* m_poStrRouClientAIL;

   // We should now have the ASF Proxy variant
   arl_tclStrClientHandler_ASF_IF* m_poStrRouClientASF;

   /// ISource Fsm List
   arl_tISrcFsmList m_oISrcFsmList;



   /*! @}*/

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  arl_tclISourceFsm* arl_tclISourceVFsm::poCreateISrcFsm(tU8 u..
   ***************************************************************************/
   /*!
    * \brief   Create a source FSM
    * \param   [u8SrcNum]:    (I) Source number
    * \retval  [arl_tclISourceFsm*]: Pointer to ISource FSM.
    * \date 03.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclISourceFsm* poCreateISrcFsm(tU8 u8SrcNum) const;

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISourceVFsm::vInitFsm()
   ***************************************************************************/
   /*!
    * \brief   Initialize all the FSM's
    * \param   NONE
    * \retval  NONE
    * \date 03.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vInitFsm();

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISourceVFsm::vClearRouteReq()
   ***************************************************************************/
   /*!
    * \brief   Clear previous route requests of media player
    * \param   NONE
    * \retval  NONE
    * \date 25.07.2014 \author Parusharam J (RBEI/ECV3)
    **************************************************************************/
   tVoid vClearRouteReq();
   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/

}; // class arl_tclISourceVFsm

#endif   // #ifndef ARL_TCLISOURCEVFSM_H_

////////////////////////////////////////////////////////////////////////////////
// <EOF>
