/*!
********************************************************************************
* \file              arl_ISourceTypes.h
********************************************************************************
*  - PROJECT:        LCN2KAI
*  - SW-COMPONENT:   Audio Routing Library (ARL)
*  - DESCRIPTION:    iSource interface types
*  - COPYRIGHT:      &copy; 2010 - 2012 Robert Bosch Car Multimedia Gmbh
********************************************************************************
* \date 26.06.2012 \version 2.0 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
* - Remove AHL inheritance to enable usage by object reference instead of inheritance
* - Remove MOST dependencies
* - Adapt to Project needs
*
* \date 25.08.2011 \version 1.1 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Added new types based on:
*     -# Stream router routing state
*     -# MOST Error codes
*
* \date 15.12.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
*******************************************************************************/

#ifndef ARL_ISOURCETYPES_H_
#define ARL_ISOURCETYPES_H_

/******************************************************************************
| includes of component-internal interfaces, if necessary
| (scope: component-local)
|----------------------------------------------------------------------------*/

#define FI_S_IMPORT_INTERFACE_FI_MESSAGE

#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_AUD_SRC_ROUTEFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_AUD_SRC_ROUTEFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_AUD_SRC_ROUTEFI_ERRORCODES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_AUD_SRC_ROUTEFI_SERVICEINFO
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_STREAMROUTERFI_TYPES
#include <midw_fi_if.h>

#define AMT_S_IMPORT_INTERFACE_GENERIC
#include <amt_if.h>

// Includes for (Removed_bpstl)
//#define SYSTEM_S_IMPORT_INTERFACE_ITERATOR   ///< Import STL IF - Iterator
//#define SYSTEM_S_IMPORT_INTERFACE_VECTOR     ///< Import STL IF - Vector
//#include <stl_pif.h>





#include <iterator>
#include <vector>
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"



//using namespace std;

#include "arl_tclBitmask.h"

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

#define ARL_ERROR                   ((tS16)-1)
#define ARL_MASK_ROUALLOC_STARTUP   ((tU8)0x80)
#define ARL_MAX_SOURCE_NUMBER       ((tU8)0x7F)
#define ARL_U32_DEFAULT_MSG_SIZE    ((tU32)0)

#define ARL_ETG_OBJ_ADDRESS(ptr)    ((OSAL_NULL != ptr)?(ptr):(0x00000000))

#define ARL_ERR_EN_PROCESSINGERR    0x42

#define ARL_ERROR_SRCACT            0x03
#define ARL_ERROR_ALLOC             0x01
#define ARL_ERROR_DEALLOC           0x02

#define ARL_ROUTESTATE_EN_INIT      midw_fi_tcl_e8_streamrouter_routestate::FI_EN_INIT
#define ARL_ROUTESTATE_EN_STOP      midw_fi_tcl_e8_streamrouter_routestate::FI_EN_STOP
#define ARL_ROUTESTATE_EN_PLAY      midw_fi_tcl_e8_streamrouter_routestate::FI_EN_PLAY
#define ARL_ROUTESTATE_EN_KILL      midw_fi_tcl_e8_streamrouter_routestate::FI_EN_KILL

// Timer Defines
#define ARL_SOURCEACTIVITY_TIMER_DURATION    (tCU32)16000 //10 second
#define ARL_STARTUP_DEALLOC_TIMER_DURATION   (tCU32)10000 //10 second

// @TODO - Change the major/minor versions to build xml
#define ARL_STRROU_FI_MAJOR_VERSION    1
#define ARL_STRROU_FI_MINOR_VERSION    0

#ifndef AUDIO_ROUTING_LIB_SERVICE__FI_MAJOR_VERSION
 #define AUDIO_ROUTING_LIB_SERVICE__FI_MAJOR_VERSION 1
#endif

/******************************************************************************
| enums (scope: global)
|----------------------------------------------------------------------------*/

/*!
 * \enum
 * iSource function id definition/mapping as specified in GM F-Catalog.
 */

enum
{
   ARL_EN_MUTE_REQ               =  MIDW_AUD_SRC_ROUTEFI_C_U16_MUTEREQUEST       ///< Function id value for Property of MuteRequest
   , ARL_EN_MUTE_STATE           =  MIDW_AUD_SRC_ROUTEFI_C_U16_MUTESTATE       ///< Function id value for Property of MuteState
   , ARL_EN_ALLOCATE             =  MIDW_AUD_SRC_ROUTEFI_C_U16_ALLOCATE       ///< Function id value for Allocate in ISource interface
   , ARL_EN_DEALLOCATE           =  MIDW_AUD_SRC_ROUTEFI_C_U16_DEALLOCATE       ///< Function id value for Deallocate in ISource interface
   , ARL_EN_SOURCEACTIVITY       =  MIDW_AUD_SRC_ROUTEFI_C_U16_SOURCEACTIVITY       ///< Function id value for SourceActivity in ISource interface
   , ARL_EN_ROUTEREQUEST         =  MIDW_AUD_SRC_ROUTEFI_C_U16_REQUESTAUDIOROUTE       ///< Function id value for RouteRequest in ISource interface
   , ARL_EN_SRCACTTIMEREXPIRY    =  MIDW_AUD_SRC_ROUTEFI_C_U16_SRCACTTIMEREXPIRY       ///< Function id value for SrcActTimerExpiry in ISource interface
   , ARL_EN_SOURCEAVAILABLE      =  MIDW_AUD_SRC_ROUTEFI_C_U16_SOURCEAVAILABILITY       ///< Function id value for SourceAvailable in ISource interface
   , ARL_EN_AUDIOPROPERTY        =  MIDW_AUD_SRC_ROUTEFI_C_U16_SETAUDIOPROPERTY       ///< Function id value for Set Audio Property in ISource interface
   , ARL_EN_AUDIOROUTEREQUESTREJECT = MIDW_AUD_SRC_ROUTEFI_C_U16_REQUESTAUDIOROUTEREJECT
   , ARL_EN_LOOPBACK                   =  0xFFFE   ///< Function id value for  loopback in ISource interface
   , ARL_EN_INVALID              =  MIDW_AUD_SRC_ROUTEFI_C_U16_INVALID_FUNCTIONID      ///< Invalid Function id value
};

enum
{
   ARL_EN_LOOPBACK_SRCACTRESULT               =  0x01       ///< Function id for Source Activity Result loopback
   , ARL_EN_LOOPBACK_SETMUTESTATE             =  0x02      ///< Function id for Set Mute state loopback
   , ARL_EN_LOOPBACK_SENDFGGRAPHCMD       =  0x03      ///< Function id for Send FilterGraph Command loopback
   , ARL_EN_LOOPBACK_SETAUDIOROUTEREQUEST       =  0x04      ///< Function id for Set Audio Route Request loopback
   , ARL_EN_LOOPBACK_SETAUDIOPROPERTY      =  0x05      ///< Function id for Set Audio Property loopback
   , ARL_EN_LOOPBACK_SETSOURCEAVAILABILITY     =  0x06      ///< Function id for Set Source Availability loopback
};

/*!
 * \enum arl_tenSrcState
 * iSource Source States.
 */

typedef enum
{
   ARL_EN_NO_ERROR        =  0          ///< Source available state
   , ARL_EN_UNKNOWN_ERROR =  1          ///< Source unavailable state

} arl_tenFGGraphCmd_Result;

/*!
 * \enum arl_tenSrcState
 * iSource Source States.
 */

typedef enum
{
   ARL_EN_SRC_REQ_AV_FAIL           =  -3          ///< Request for AV activation has failed (Requested State)
   , ARL_EN_SRC_TRIG_AV_ACT_ERR     =  -2          ///< Request for AV activation returned an error (Trigger for state change)
   , ARL_EN_SRC_TRIG_ERR            =  -1          ///< General Source related error (Trigger for state change)
   , ARL_EN_SRC_IDLE                =  0           ///< Source is idle (Default state)
   , ARL_EN_SRC_PLAYING             =  1           ///< Source is playing (Playing state)
   , ARL_EN_SRC_REQ_AV_ACT          =  2           ///< Requested AV activation (Requested State)
   , ARL_EN_SRC_REQ_AV_DEACT        =  3           ///< Requested AV Deactivation (Requested State)
   , ARL_EN_SRC_TRIG_REQ_AV_SUCC    =  4           ///< Request for AV activation succeeded (Trigger for state change)
   , ARL_EN_SRC_REQ_ALLOC_ROUTE     =  5           ///< Source requested for allocation route (Requested State)
   , ARL_EN_SRC_REQ_DEALLOC_ROUTE   =  6           ///< Source requested for deallocation of route (Requested State)
   , ARL_EN_SRC_REQ_SET_PLAY        =  7           ///< Source requested for Set Play (Requested State)
   , ARL_EN_SRC_REQ_SET_STOP        =  8           ///< Source requested for Set Stop (Requested State)
   , ARL_EN_SRC_TRIG_UCID_ACTIVE    =  9           ///< Source recieved UCID is already active (Trigger for state change)

   // General state irrespective of Source numbers
   , ARL_EN_SRC_AVAIL               =  10          ///< Source available state
   , ARL_EN_SRC_UNAVAIL             =  11          ///< Source unavailable state

} arl_tenSrcState;

typedef enum
{
     ARL_EN_ROUTE_STATE_NO_ALLOC         =  0x00        ///< Default State, Not Required to Allocate Route on Startup
   , ARL_EN_ROUTE_STATE_REQ_ALLOCATE     =  0x01       ///< ID value for Route Allocate on Startup
   , ARL_EN_ROUTE_STATE_ALLOCATED        =  0x02       ///< ID value for Allocated Route State
   , ARL_EN_ROUTE_STATE_REQ_DEALLOC      =  0x03       ///< ID value for Deallocate Route Request
   , ARL_EN_ROUTE_STATE_INVALID          =  0xFF       ///< Invalid Function id value
}arl_tenStartUpRouState;

/*!
 * \typedef enum arl_tenActivity
 * \brief Source activity definition
 */

typedef enum
{
   ARL_EN_ISRC_ACT_OFF     = midw_fi_tcl_e8_SrcActivity::FI_EN_OFF   ///< Source may be not activated for longer
   , ARL_EN_ISRC_ACT_PAUSE = midw_fi_tcl_e8_SrcActivity::FI_EN_PAUSE ///< Source temporarily paused
   , ARL_EN_ISRC_ACT_ON    = midw_fi_tcl_e8_SrcActivity::FI_EN_ON    ///< Source active.

} arl_tenActivity;

/*!
 * \typedef enum arl_tenFGCommand
 * \brief Command Enumeration for the Filtergraph of Platform
 */

typedef enum
{
   ARL_EN_FGCMD_MICMUTE_ON    = midw_fi_tcl_e8_streamrouter_fgcommand::FI_EN_ECNR_MICMUTE_ON
   , ARL_EN_FGCMD_MICMUTE_OFF = midw_fi_tcl_e8_streamrouter_fgcommand::FI_EN_ECNR_MICMUTE_OFF
   , ARL_EN_FGCMD_EC_ON = midw_fi_tcl_e8_streamrouter_fgcommand::FI_EN_ECNR_ECHOCANCELLATION_ON ///< Echo cancellation on
   , ARL_EN_FGCMD_EC_OFF = midw_fi_tcl_e8_streamrouter_fgcommand::FI_EN_ECNR_ECHOCANCELLATION_OFF ///< Echo cancellation off
   , ARL_EN_FGCMD_GAIN_LOW = midw_fi_tcl_e8_streamrouter_fgcommand::FI_EN_ECNR_GAIN_LOW ///< Low gain
   , ARL_EN_FGCMD_GAIN_MED = midw_fi_tcl_e8_streamrouter_fgcommand::FI_EN_ECNR_GAIN_MED ///< Medium gain
   , ARL_EN_FGCMD_GAIN_HIG = midw_fi_tcl_e8_streamrouter_fgcommand::FI_EN_ECNR_GAIN_HIG ///< High gain
   , ARL_EN_FGCMD_MIC_NR_ON = midw_fi_tcl_e8_streamrouter_fgcommand::FI_EN_ECNR_MICNOISEREDUCTION_ON ///< Microphone Noise reduction on
   , ARL_EN_FGCMD_MIC_NR_OFF = midw_fi_tcl_e8_streamrouter_fgcommand::FI_EN_ECNR_MICNOISEREDUCTION_OFF ///< Microphone Noise reduction off
} arl_tenFGCommand;

/*!
 * \typedef enum arl_tenMuteState
 * \brief Mute state definition, reflecting actual system state
 */

typedef enum
{
   ARL_EN_MUTESTATE_DEMUTE = midw_fi_tcl_e8_MuteState::FI_EN_DEMUTE  ///< Audio is audible.
   , ARL_EN_MUTESTATE_MUTE = midw_fi_tcl_e8_MuteState::FI_EN_MUTE    ///< Audio is NOT audible.
   , ARL_EN_MUTESTATE_DEMUTE_RUNNING = midw_fi_tcl_e8_MuteState::FI_EN_DEMUTERUNNING ///<  Demute Ramp is running, audio is getting audible
} arl_tenMuteState;

// *********************************
// !!!!    OBSOLETE DEFINES     !!!!
//   valid for period of transition
// *********************************
#define ARL_EN_ISRC_MUTE_ON      ARL_EN_MUTESTATE_MUTE
#define ARL_EN_ISRC_MUTE_OFF     ARL_EN_MUTESTATE_DEMUTE
#define ARL_EN_ISRC_MUTE_RUNNING ARL_EN_MUTESTATE_DEMUTE
#define ARL_EN_ISRC_MUTE_INVALID ARL_EN_MUTESTATE_DEMUTE
// *********************************
// OBSOLTE Defines,
// End
// *********************************



/*!
 * \typedef enum arl_tenMuteAction
 * \brief Mute action definition, client wanted state
 */
typedef enum
{
   ARL_EN_MUTEACTION_DEMUTE = midw_fi_tcl_e8MuteAction::FI_EN_DEMUTE  ///< Audio is wanted to be audible.
   , ARL_EN_MUTEACTION_MUTE = midw_fi_tcl_e8MuteAction::FI_EN_MUTE    ///< Audio is wanted to be NOT audible.
} arl_tenMuteAction;

/*!
 * \typedef enum arl_enMuteReason
 * \brief Bit definitions for MuteReasons
 */
typedef enum
{
   ARL_EN_ROUTEREQ_REASON_OTHER
} arl_enRouteReqReason;

/*!
 * \typedef enum arl_enMuteReason
 * \brief Bit definitions for MuteReasons
 */
typedef enum
{
   ARL_EN_MUTE_REASON_DIAG = midw_fi_tcl_b32MuteReason::FI_C_U32_BIT_MUTE_REAS_DIAG  ///< Diagnosis .
   , ARL_EN_MUTE_REASON_OTHER = midw_fi_tcl_b32MuteReason::FI_C_U32_BIT_MUTE_REAS_OTHER  ///< other Mute Reason
   , ARL_EN_MUTE_REASON_SRC_SEL = midw_fi_tcl_b32MuteReason::FI_C_U32_BIT_MUTE_REAS_SOURCE_SEL  ///< Source selection
   , ARL_EN_MUTE_REASON_TRACK_START = midw_fi_tcl_b32MuteReason::FI_C_U32_BIT_MUTE_REAS_START_OF_TRACK  ///< Track start
   , ARL_EN_MUTE_REASON_USR_ACT = midw_fi_tcl_b32MuteReason::FI_C_U32_BIT_MUTE_REAS_USER_ACTION  ///< User action
} arl_enMuteReason;

//todo: CMG3G-12179 the below defines to be removed after application changes the source name to web_ent
#define ARL_SRC_ECORE_MAIN_1 ARL_SRC_WEB_ENT
#define ARL_SRC_ECORE_MAIN_2 ARL_SRC_WEB_INFO_USR
typedef enum
{
   ARL_SRC_NONE                  = midw_fi_tcl_e8_AudSource::FI_EN_NONE
   , ARL_SRC_FM                  = midw_fi_tcl_e8_AudSource::FI_EN_FM
   , ARL_SRC_FMTA                = midw_fi_tcl_e8_AudSource::FI_EN_FM_TA
   , ARL_SRC_AM                  = midw_fi_tcl_e8_AudSource::FI_EN_AM
   , ARL_SRC_DAB                 = midw_fi_tcl_e8_AudSource::FI_EN_DAB
   , ARL_SRC_DABTA               = midw_fi_tcl_e8_AudSource::FI_EN_DAB_TA
   , ARL_SRC_XM                  = midw_fi_tcl_e8_AudSource::FI_EN_XMTUN
   , ARL_SRC_CPRA                = midw_fi_tcl_e8_AudSource::FI_EN_CPRA
   , ARL_SRC_CDDA                = midw_fi_tcl_e8_AudSource::FI_EN_CDDA
   , ARL_SRC_AUX                 = midw_fi_tcl_e8_AudSource::FI_EN_AUX
   , ARL_SRC_SPI_BTAUDIO         = midw_fi_tcl_e8_AudSource::FI_EN_SPI_BT_AUDIO
   , ARL_SRC_SPI_USB             = midw_fi_tcl_e8_AudSource::FI_EN_SPI_USB
   , ARL_SRC_SPI_VR_USB          = midw_fi_tcl_e8_AudSource::FI_EN_SPI_VR_USB
   , ARL_SRC_SPI_VR_BTAUDIO      = midw_fi_tcl_e8_AudSource::FI_EN_SPI_VR_BT_AUDIO
   , ARL_SRC_SPI_TTS             = midw_fi_tcl_e8_AudSource::FI_EN_SPI_TTS
   , ARL_SRC_BTAUDIO             = midw_fi_tcl_e8_AudSource::FI_EN_BT_AUDIO
   , ARL_SRC_PHONE               = midw_fi_tcl_e8_AudSource::FI_EN_PHONE
   , ARL_SRC_AUX2                = midw_fi_tcl_e8_AudSource::FI_EN_AUX2
   , ARL_SRC_VRU                 = midw_fi_tcl_e8_AudSource::FI_EN_VRU
   , ARL_SRC_SMS                 = midw_fi_tcl_e8_AudSource::FI_EN_SMS
   , ARL_SRC_NAVI                = midw_fi_tcl_e8_AudSource::FI_EN_NAVI
   , ARL_SRC_CUE                 = midw_fi_tcl_e8_AudSource::FI_EN_CUE
   , ARL_SRC_MIC                 = midw_fi_tcl_e8_AudSource::FI_EN_MIC
   , ARL_SRC_PHONE2              = midw_fi_tcl_e8_AudSource::FI_EN_PHONE2
   , ARL_SRC_CPRA2               = midw_fi_tcl_e8_AudSource::FI_EN_CPRA2
   , ARL_SRC_IAPPS               = midw_fi_tcl_e8_AudSource::FI_EN_IAPPS
   , ARL_SRC_SPI_MAIN            = midw_fi_tcl_e8_AudSource::FI_EN_SPI_MAIN
   , ARL_SRC_SPI_PHONE           = midw_fi_tcl_e8_AudSource::FI_EN_SPI_PHONE
   , ARL_SRC_SPI_VR              = midw_fi_tcl_e8_AudSource::FI_EN_SPI_VR
   , ARL_SRC_SPI_MIX             = midw_fi_tcl_e8_AudSource::FI_EN_SPI_MIX
   , ARL_SRC_PHONE_ANALOG_EXT    = midw_fi_tcl_e8_AudSource::FI_EN_PHONE_ANALOG_EXT
   , ARL_SRC_PHONE_ECALL         = midw_fi_tcl_e8_AudSource::FI_EN_PHONE_ECALL
   , ARL_SRC_PHONE_OUTBAND_RING  = midw_fi_tcl_e8_AudSource::FI_EN_PHONE_OUTBAND_RING
   , ARL_SRC_PHONE_VR       = midw_fi_tcl_e8_AudSource::FI_EN_PHONE_VR
   , ARL_SRC_SPI_ALERT           = midw_fi_tcl_e8_AudSource::FI_EN_SPI_ALERT
   , ARL_SRC_DRM               = midw_fi_tcl_e8_AudSource::FI_EN_DRM
   , ARL_SRC_PHONE_WAIT          = midw_fi_tcl_e8_AudSource::FI_EN_PHONE_WAIT
   , ARL_SRC_TCU_ECALL           = midw_fi_tcl_e8_AudSource::FI_EN_TCU_ECALL
   , ARL_SRC_TCU_SERVICECALL     = midw_fi_tcl_e8_AudSource::FI_EN_TCU_SERVICECALL
   , ARL_SRC_TCU_TTS             = midw_fi_tcl_e8_AudSource::FI_EN_TCU_TTS
   , ARL_SRC_SPI_TRANSIENT       = midw_fi_tcl_e8_AudSource::FI_EN_SPI_TRANSIENT
   , ARL_SRC_WEB_ENT             = midw_fi_tcl_e8_AudSource::FI_EN_WEB_ENT
   , ARL_SRC_WEB_INFO_USR        = midw_fi_tcl_e8_AudSource::FI_EN_WEB_INFO_USR
   , ARL_SRC_SXM_TTS             = midw_fi_tcl_e8_AudSource::FI_EN_SXM_TTS
   , ARL_SRC_SPI_ML_INFO         = midw_fi_tcl_e8_AudSource::FI_EN_SPI_ML_INFO
   , ARL_SRC_SPI_VR_STEREO       = midw_fi_tcl_e8_AudSource::FI_EN_SPI_VR_STEREO
   , ARL_SRC_BLU_RAY             = midw_fi_tcl_e8_AudSource::FI_EN_BLU_RAY
   , ARL_SRC_EXTERNAL_CD         = midw_fi_tcl_e8_AudSource::FI_EN_EXTERNAL_CD
   , ARL_SRC_SXM_NEWS            = midw_fi_tcl_e8_AudSource::FI_EN_SXM_NEWS
   , ARL_SRC_IPA_TTS             = midw_fi_tcl_e8_AudSource::FI_EN_IPA_TTS
   , ARL_SRC_DTV         = midw_fi_tcl_e8_AudSource::FI_EN_DTV
   , ARL_SRC_AMTI         = midw_fi_tcl_e8_AudSource::FI_EN_AM_TI
   , ARL_SRC_WEB_INFO_SYSTEM   = midw_fi_tcl_e8_AudSource::FI_EN_WEB_INFO_SYSTEM
   , ARL_SRC_WEB_MIX       = midw_fi_tcl_e8_AudSource::FI_EN_WEB_MIX
   , ARL_SRC_WEB_TTS       = midw_fi_tcl_e8_AudSource::FI_EN_WEB_TTS
   , ARL_SRC_VPA_ENT       = midw_fi_tcl_e8_AudSource::FI_EN_VPA_MAIN
   , ARL_SRC_VPA_MIX       = midw_fi_tcl_e8_AudSource::FI_EN_VPA_TTS
   , ARL_SRC_VPA_SDS       = midw_fi_tcl_e8_AudSource::FI_EN_VPA_SDS
   , ARL_SRC_USB_VIDEO       = midw_fi_tcl_e8_AudSource::FI_EN_USB_VIDEO
   , ARL_SRC_RSE_IN         = midw_fi_tcl_e8_AudSource::FI_EN_RSE_IN
   , ARL_SRC_NAVI_BY_SDS         = midw_fi_tcl_e8_AudSource::FI_EN_NAVI_BY_SDS
   ,ARL_SRC_AUTO_DRIVE        = midw_fi_tcl_e8_AudSource::FI_EN_AUTO_DRIVE
   ,ARL_SRC_DRVASS_VIDEO      = midw_fi_tcl_e8_AudSource::FI_EN_DRVASS_VIDEO
   ,ARL_SRC_EXT_MEDIA1        = midw_fi_tcl_e8_AudSource::FI_EN_EXT_MEDIA1
   ,ARL_SRC_DRIVE_ECO_TTS        = midw_fi_tcl_e8_AudSource::FI_EN_ECO_TTS
   ,ARL_SRC_EXT_NAV        = midw_fi_tcl_e8_AudSource::FI_EN_EXT_NAVI
   ,ARL_SRC_WX             = midw_fi_tcl_e8_AudSource::FI_EN_WX
   ,ARL_SRC_WEB_SDS             = midw_fi_tcl_e8_AudSource::FI_EN_WEB_SDS
   ,ARL_SRC_INTERCOM             = midw_fi_tcl_e8_AudSource::FI_EN_INTERCOM
   ,ARL_SRC_MCS_AUDIO            = midw_fi_tcl_e8_AudSource::FI_EN_MCS_AUDIO
   ,ARL_SRC_MCS_VIDEO               = midw_fi_tcl_e8_AudSource::FI_EN_MCS_VIDEO
   ,ARL_SRC_NAVI_HORIZON                    = midw_fi_tcl_e8_AudSource::FI_EN_NAVI_HORIZON
   ,ARL_SRC_NAVI_HORIZON_RETRIGGER           = midw_fi_tcl_e8_AudSource::FI_EN_NAVI_HORIZON_RETRIGGER
   ,ARL_SRC_SDS_PROMPT                 = midw_fi_tcl_e8_AudSource::FI_EN_SDS_PROMPT
   ,ARL_SRC_NAVI_XURGENT                     = midw_fi_tcl_e8_AudSource::FI_EN_NAVI_XURGENT
   ,ARL_SRC_NAVI_XURGENT_RETRIGGER           = midw_fi_tcl_e8_AudSource::FI_EN_NAVI_XURGENT_RETRIGGER
   ,ARL_SRC_NAVI_ROUTE_AND_DEST_INFO         = midw_fi_tcl_e8_AudSource::FI_EN_NAVI_ROUTE_AND_DEST_INFO
   ,ARL_SRC_NAVI_ROUTE_AND_DEST_INFO_RETRIGGER     = midw_fi_tcl_e8_AudSource::FI_EN_NAVI_ROUTE_AND_DEST_INFO_RETRIGGER
   ,ARL_SRC_NAVI_EXTERNAL_TRIGGERED         = midw_fi_tcl_e8_AudSource::FI_EN_NAVI_EXTERNAL_TRIGGERED
   ,ARL_SRC_NAVI_EXTERNAL_TRIGGERED_RETRIGGER     = midw_fi_tcl_e8_AudSource::FI_EN_NAVI_EXTERNAL_TRIGGERED_RETRIGGER
   ,ARL_SRC_NAVI_DEBUG                 = midw_fi_tcl_e8_AudSource::FI_EN_NAVI_DEBUG
   ,ARL_SRC_NAVI_DEBUG_RETRIGGER           = midw_fi_tcl_e8_AudSource::FI_EN_NAVI_DEBUG_RETRIGGER
   ,ARL_SRC_NAVI_SOUND                 = midw_fi_tcl_e8_AudSource::FI_EN_NAVI_SOUND
   ,ARL_SRC_NAVI_SOUND_RETRIGGER                           = midw_fi_tcl_e8_AudSource::FI_EN_NAVI_SOUND_RETRIGGER
   ,ARL_SRC_NAVI_DRIVING_RECOMMENDATION                    = midw_fi_tcl_e8_AudSource::FI_EN_NAVI_DRIVING_RECOMMENDATION
   ,ARL_SRC_NAVI_DRIVING_RECOMMENDATION_RETRIGGER          = midw_fi_tcl_e8_AudSource::FI_EN_NAVI_DRIVING_RECOMMENDATION_RETRIGGER
   ,ARL_SRC_HMI_SMS                                        = midw_fi_tcl_e8_AudSource::FI_EN_HMI_SMS
   ,ARL_SRC_SXM_TTS_SPEECH                                 = midw_fi_tcl_e8_AudSource::FI_EN_SXM_TTS_SPEECH
   ,ARL_SRC_TTS_DEFAULT                                    = midw_fi_tcl_e8_AudSource::FI_EN_TTS_DEFAULT
   ,ARL_SRC_MIC_PERMANENT                                  = midw_fi_tcl_e8_AudSource::FI_EN_MIC_PERMANENT
   ,ARL_SRC_SXM_IP                                         = midw_fi_tcl_e8_AudSource::FI_EN_SXM_IP
   ,ARL_SRC_WX_ALERT                                       = midw_fi_tcl_e8_AudSource::FI_EN_WX_ALERT
} arl_tenSource;

typedef enum
{
   ARL_SRC_VOLUME_LOCK                = midw_fi_tcl_e8_AudioProperty::FI_EN_AUDIO_PROP_VOLUME_LOCK,
     ARL_SRC_INPUT_GAIN          = midw_fi_tcl_e8_AudioProperty::FI_EN_AUDIO_PROP_INPUT_GAIN,
   ARL_SRC_IPOD_DUCKING        = midw_fi_tcl_e8_AudioProperty::FI_EN_AUDIO_PROP_IPOD_DUCKING
} arl_tenSourcePropType;
/*!
 * \typedef enum arl_tenSrcAvailability
 * \brief states of an audiosource
 */
typedef enum
{
   ARL_EN_SRC_NOT_AVAILABLE = 0            ///<  No device is connected.
   , ARL_EN_SRC_PLAYABLE_UNKNOWN             ///<  Device is connected, playable content unknown
   , ARL_EN_SRC_PLAYABLE                     ///<  Device connected playable content found
   , ARL_EN_SRC_NOT_PLAYABLE                 ///<  Device connected, no playable content found or defect
} arl_tenSrcAvailability;


/*!
 * \typedef enum arl_tenAvailabilityReason
 * \brief reasons for a change in the availability
 */
typedef enum
{
   ARL_EN_REASON_NEWMEDIA = 0   ///< Device is newly inserted
   , ARL_EN_REASON_SAMEMEDIA      ///< Device was already there before
   , ARL_EN_REASON_NOMEDIA        ///< Media removed
   , ARL_EN_REASON_TEMPERATURE    ///< Over-/Undertemperature
   , ARL_EN_REASON_VOLTAGE        ///< Over-/Undervoltage
   , ARL_EN_REASON_ERROR          ///< Device Access/ Read Error
   , ARL_EN_REASON_NO_CONTENT     ///< Device has no conttent
   , ARL_EN_REASON_OVERCURRENT    ///< Device has over current
   , ARL_EN_REASON_INTERNALDISCONNECT  ///< Device has internal disconnect
} arl_tenAvailabilityReason;


   typedef std::vector<tU8> arl_tSourceList;
   typedef std::vector<tU8>::iterator arl_tSourceListIter;

   typedef struct
   {
      audioroutinglib::arl_tclBitmask<arl_enMuteReason>  muteReason;
      arl_tenMuteAction muteAction;
   } arl_stSetMuteTuple;
   typedef struct
   {
      arl_tenSourcePropType  PropType;
      tS32 PropValue;
   } arl_stSrcAudioPropertyTuple;

typedef struct arl_stSrcAvailabilityTuple
{
   //arl_stSrcAvailabilityTuple(arl_tenSrcAvailability srcAvailability, arl_tenAvailabilityReason chgReason) : availability(srcAvailability), reason(chgReason) {}
   arl_tenSrcAvailability    availability;
   arl_tenAvailabilityReason reason;
} arl_stSrcAvailabilityTuple;

struct SourceID
{
   arl_tenSource enSourceClass;
   tU16          u16SubSource;

   SourceID(arl_tenSource srcClass, tU16 subSource) :enSourceClass(srcClass),u16SubSource(subSource){}

   bool operator<( const SourceID & rhs ) const
   {
      return (
            ((((tU32)enSourceClass  )<< 16) | (tU32)  u16SubSource) <
            ((((tU32)rhs.enSourceClass)<< 16) | (tU32)rhs.u16SubSource) );
   }

   bool operator==(const SourceID& rhs) const
            {
      return (
            (rhs.enSourceClass == enSourceClass) &&
            (rhs.u16SubSource == u16SubSource) );
            }

   bool operator!=(const SourceID& rhs) const
               {
      return (
            (rhs.enSourceClass != enSourceClass) ||
            (rhs.u16SubSource != u16SubSource) );
               }

   SourceID& operator =(const SourceID& o)
   {
      enSourceClass = (o.enSourceClass);
      u16SubSource = (o.u16SubSource);
      return *this;
   }
};

/******************************************************************************
| typedefs (scope: global)
|----------------------------------------------------------------------------*/

/*!
 * \class arl_tclMsgIdentifiers
 * \brief Message identifier datatype to capture the identifiers from incoming
 * message.
 *
 */

class arl_tclMsgIdentifiers
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  arl_tclMsgIdentifiers::arl_tclMsgIdentifiers(const amt_tc..
   ***************************************************************************/
   /*!
    * \brief   Parameterized Constructor, based on Dependency Injection
    *          Principle (DIP)
    * \param   [rfcoServData]:      (I)   Service data object/incoming message.
    * \retval  NONE
    * \date 03.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/

   arl_tclMsgIdentifiers(const amt_tclServiceData& rfcoServData)
      :m_u16FunctionId(rfcoServData.u16GetFunctionID())
         , m_u16SourceAppId(rfcoServData.u16GetSourceAppID())
         , m_u16RegisterId(rfcoServData.u16GetRegisterID())
         , m_u16CmdCounter(rfcoServData.u16GetCmdCounter())
         , m_u32Timestamp(rfcoServData.u32GetTimestamp()) {};

   /***************************************************************************
   ** FUNCTION:  arl_tclMsgIdentifiers::arl_tclMsgIdentifiers(const arl_tclMsg..
   ***************************************************************************/
   /*!
    * \brief   Copy Constructor
    * \param   [rfcoMsgId]: (I) Const reference to object to be copied
    * \retval  NONE
    * \date 11.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/

   arl_tclMsgIdentifiers(const arl_tclMsgIdentifiers& rfcoMsgId)
   {
      m_u16FunctionId   =  rfcoMsgId.m_u16FunctionId;
      m_u16SourceAppId  =  rfcoMsgId.m_u16SourceAppId;
      m_u16RegisterId   =  rfcoMsgId.m_u16RegisterId;
      m_u16CmdCounter   =  rfcoMsgId.m_u16CmdCounter;
      m_u32Timestamp    =  rfcoMsgId.m_u32Timestamp;

   }; // arl_tclMsgIdentifiers(const arl_tclMsgIdentifiers& rfcoMsgId)

   /***************************************************************************
   ** FUNCTION:  arl_tclMsgIdentifiers& arl_tclMsgIdentifiers::operator=(co ..
   ***************************************************************************/
   /*!
    * \brief   Assignment Operator
    * \param   [rfcoMsgId]: (I) Const reference to object to be copied
    * \retval  [arl_tclMsgIdentifiers&]: Reference to this pointer.
    * \date 11.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/

   arl_tclMsgIdentifiers& operator= (const arl_tclMsgIdentifiers& rfcoMsgId)
   {
      // Self protection.
      if (this != &rfcoMsgId)
      {
         m_u16FunctionId   =  rfcoMsgId.m_u16FunctionId;
         m_u16SourceAppId  =  rfcoMsgId.m_u16SourceAppId;
         m_u16RegisterId   =  rfcoMsgId.m_u16RegisterId;
         m_u16CmdCounter   =  rfcoMsgId.m_u16CmdCounter;
         m_u32Timestamp    =  rfcoMsgId.m_u32Timestamp;
      }  // if (this != &rfcoMsgId)

      return *this;

   }; // arl_tclMsgIdentifiers& operator= (const arl_tclMsgIdentifiers& rfcoMs..

   /***************************************************************************
   ** FUNCTION:  virtual arl_tclMsgIdentifiers::~arl_tclMsgIdentifiers()
   ***************************************************************************/
   /*!
    * \brief   Destructor
    * \param   NONE
    * \retval  NONE
    * \date 03.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual ~arl_tclMsgIdentifiers() {};

   /***************************************************************************
   ** FUNCTION:  inline tU16 arl_tclMsgIdentifiers::u16GetFunctionId() const
   ***************************************************************************/
   /*!
    * \brief   Get the Function id
    * \param   NONE
    * \retval  [tU16]: Source App Id.
    * \date 03.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   inline tU16 u16GetFunctionId() const { return m_u16FunctionId; };

   /***************************************************************************
   ** FUNCTION:  inline tU16 arl_tclMsgIdentifiers::u16GetSourceAppId() const
   ***************************************************************************/
   /*!
    * \brief   Get the Source App id
    * \param   NONE
    * \retval  [tU16]: Source App Id.
    * \date 03.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   inline tU16 u16GetSourceAppId() const { return m_u16SourceAppId; };

   /***************************************************************************
   ** FUNCTION:  inline tU16 arl_tclMsgIdentifiers::u16GetRegisterId() const
   ***************************************************************************/
   /*!
    * \brief   Get the Register id
    * \param   NONE
    * \retval  [tU16]: Register Id.
    * \date 03.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   inline tU16 u16GetRegisterId() const { return m_u16RegisterId; };

   /***************************************************************************
   ** FUNCTION:  inline tU16 arl_tclMsgIdentifiers::u16GetCmdCounter() const
   ***************************************************************************/
   /*!
    * \brief   Get the command counter
    * \param   NONE
    * \retval  [tU16]: Command counter value.
    * \date 03.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   inline tU16 u16GetCmdCounter() const { return m_u16CmdCounter; };

   /***************************************************************************
   ** FUNCTION:  inline tU32 arl_tclMsgIdentifiers::u32GetTimestamp() const
   ***************************************************************************/
   /*!
    * \brief   Get the command counter
    * \param   NONE
    * \retval  [tU32]: Time stamp
    * \date 03.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   inline tU32 u32GetTimestamp() const { return m_u32Timestamp; };

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *******************************PROTECTED************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  arl_tclMsgIdentifiers::arl_tclMsgIdentifiers()
   ***************************************************************************/
   /*!
    * \brief   Default Constructor
    * \param   NONE
    * \retval  NONE
    * \note    Default constructor is declared protected to disable it. So
    *          that any attempt to create without parameter will be caught by
    *          the compiler.
    * \date 11.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclMsgIdentifiers();

   /*!
    * \addtogroup tclMem
    */
   /*! @{*/

   /// Function Id
   tU16 m_u16FunctionId;
   /// Source App Id
   tU16 m_u16SourceAppId;
   /// Register Id
   tU16 m_u16RegisterId;
   /// Command counter
   tU16 m_u16CmdCounter;
   /// Time stamp
   tU32 m_u32Timestamp;

   /*! @}*/

   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/

}; // class arl_tclMsgIdentifiers

/*!
 * \note
 * Internally using the Audio cue Most Fi type to extract the payload.
 * The variable parameter which depends on the Source is service id, which will
 * be taken care by the service handler of ISource service base class.
 * These types will not be exposed to other application, but will be used
 * internally by the ARL framework.
 */

/*!
 * \class arl_tclISrcAllocStart
 * \brief Extended Fi data type for Source Allocation method start.
 *
 */

class arl_tclISrcAllocStart : public midw_aud_src_routefi_tclMsgAllocateMethodStart
   , public arl_tclMsgIdentifiers
{
public:
   arl_tclISrcAllocStart(amt_tclServiceData &rfoServData);
   virtual ~arl_tclISrcAllocStart();
   tBool bIsValid();
private:
   tBool m_bValid;
};

/*!
 * \class arl_tclISrcDeallocStart
 * \brief Extended Fi data type for Source DeAllocation method start.
 *
 */

class arl_tclISrcDeallocStart : public midw_aud_src_routefi_tclMsgDeAllocateMethodStart
   , public arl_tclMsgIdentifiers
{
public:
   arl_tclISrcDeallocStart(amt_tclServiceData &rfoServData);
   virtual ~arl_tclISrcDeallocStart();
   tBool bIsValid();
private:
   tBool m_bValid;
};


/*!
 * \class arl_tclISrcActivityStart
 * \brief Extended Fi data type for Source activity method start.
 *
 */

class arl_tclISrcActivityStart : public midw_aud_src_routefi_tclMsgSourceActivityMethodStart
   , public arl_tclMsgIdentifiers
{
public:
  arl_tclISrcActivityStart(amt_tclServiceData &rfoServData);
  virtual ~arl_tclISrcActivityStart();
   tBool bIsValid();
private:
   tBool m_bValid;
};

class arl_tclISrcActTimerExpiryStart : public midw_aud_src_routefi_tclMsgSrcActTimerExpiryMethodStart
   , public arl_tclMsgIdentifiers
{
public:
  arl_tclISrcActTimerExpiryStart(amt_tclServiceData &rfoServData);
  virtual ~arl_tclISrcActTimerExpiryStart();
   tBool bIsValid();
private:
   tBool m_bValid;
};


/*!
 * \typedef arl_tclISrcAllocStart arl_tISrcAllocStart
 * \brief Allocate method causes the source to occupy bandwidth for
 * streaming data.
 */

typedef arl_tclISrcAllocStart arl_tISrcAllocStart;

/*!
 * \typedef arl_tclISrcDeallocStart arl_tISrcDeallocStart
 * \brief Deallocate method causes the source to free occupied Streaming
 * bandwidth.
 */

typedef arl_tclISrcDeallocStart arl_tISrcDeallocStart;

/*!
 * \typedef arl_tclISrcActivityStart arl_tISrcActivityStart
 * \brief Method to control the activity of a source
 */
typedef arl_tclISrcActivityStart arl_tISrcActivityStart;

typedef arl_tclISrcActTimerExpiryStart arl_tISrcActTimerExpiryStart;



/*!
 * \typedef midw_fi_tcl_e8_SrcActivity arl_tSrcActivity
 * \brief Source activity.
 */
typedef midw_fi_tcl_e8_SrcActivity arl_tSrcActivity;



class arl_tAllocRouteResult : public midw_streamrouterfi_tclMsgAllocateRouteMethodResult
{
public:
  arl_tAllocRouteResult(amt_tclServiceData &rfoServData);
  virtual ~arl_tAllocRouteResult();
   tBool bIsValid();
private:
   tBool m_bValid;
};
class arl_tDeAllocRouteStart : public midw_streamrouterfi_tclMsgDeAllocateRouteMethodStart
{
public:
  arl_tDeAllocRouteStart(amt_tclServiceData &rfoServData);
  arl_tDeAllocRouteStart();
  virtual ~arl_tDeAllocRouteStart();
   tBool bIsValid();
private:
   tBool m_bValid;
};
class arl_tDeAllocRouteResult : public midw_streamrouterfi_tclMsgDeAllocateRouteMethodResult
{
public:
  arl_tDeAllocRouteResult(amt_tclServiceData &rfoServData);
  virtual ~arl_tDeAllocRouteResult();
   tBool bIsValid();
private:
   tBool m_bValid;
};

class arl_tSetPlayStart : public midw_streamrouterfi_tclMsgSetPlayMethodStart
{
public:
  arl_tSetPlayStart(amt_tclServiceData &rfoServData);
  arl_tSetPlayStart();
  virtual ~arl_tSetPlayStart();
   tBool bIsValid();
private:
   tBool m_bValid;
};

class arl_tSetPlayResult : public midw_streamrouterfi_tclMsgSetPlayMethodResult
{
public:
  arl_tSetPlayResult(amt_tclServiceData &rfoServData);
  virtual ~arl_tSetPlayResult();
   tBool bIsValid();
private:
   tBool m_bValid;
};
class arl_tSrcActTimerExpiryStart : public midw_aud_src_routefi_tclMsgSrcActTimerExpiryMethodStart
{
public:
  arl_tSrcActTimerExpiryStart(amt_tclServiceData &rfoServData);
  arl_tSrcActTimerExpiryStart();
  virtual ~arl_tSrcActTimerExpiryStart();
   tBool bIsValid();
private:
   tBool m_bValid;
};

class arl_tSetStopStart : public midw_streamrouterfi_tclMsgSetStopMethodStart
{
public:
  arl_tSetStopStart(amt_tclServiceData &rfoServData);
  arl_tSetStopStart();
  virtual ~arl_tSetStopStart();
   tBool bIsValid();
private:
   tBool m_bValid;
};

class arl_tSetStopResult : public midw_streamrouterfi_tclMsgSetStopMethodResult
{
public:
  arl_tSetStopResult(amt_tclServiceData &rfoServData);
  arl_tSetStopResult();
  virtual ~arl_tSetStopResult();
   tBool bIsValid();
private:
   tBool m_bValid;
};
class arl_tRouteState : public midw_streamrouterfi_tclMsgRoutingStateStatus
{
public:
  arl_tRouteState(amt_tclServiceData &rfoServData);
  arl_tRouteState();
  virtual ~arl_tRouteState();
   tBool bIsValid();
private:
   tBool m_bValid;
};

/*!
 * \typedef midw_fi_tcl_streamrouter_routing_list arl_tRouteList
 * \brief Route List
 */

typedef midw_fi_tcl_streamrouter_routing_list arl_tRouteList;

/*!
 * \typedef midw_fi_tcl_streamrouter_route_entry arl_tRouteEntry
 * \brief Routing entry
 */
typedef midw_fi_tcl_streamrouter_route_entry arl_tRouteEntry;

/*!
 * \typedef midw_fi_tcl_e8_streamrouter_routestate arl_tenRouteState
 * \brief Routing state enumeration.
 */
typedef midw_fi_tcl_e8_streamrouter_routestate arl_tenRouteState;

/*!
 * \typedef vector<arl_tRouteEntry>::iterator arl_tRouteEntryIter
 * \brief Routing list iterator
 */
typedef std::vector<arl_tRouteEntry>::const_iterator arl_tRouteEntryIter;

/*!
 * \typedef struct arl_tISrcProp
 * \brief Datatype definition for a ISource Entry configuration.
 */

typedef struct arl_rISrcProp
{
   /// UCID
   tU16 m_u16Ucid;
   /// AV Logical Channel Id
   //@todo remove this AVManager shit
   //arl_tLogChannel m_oLogChannel;
   /// Self Triggered?
   tBool m_bIsSelfTrig;
   /// Source FBlock Id
   tU8 m_u8SourceFBlock;
   /// Source Instance Id
   tU8 m_u8SourceInstID;
   /// Number of audio channels
   tU8 m_u8AudioChannels;
   /// Early Route Allocation
   tBool m_bAllocOnStartUp;

   tU16 m_u16Sink;
   /****************************************************************************
   ** FUNCTION:  arl_rISrcProp& arl_rISrcProp:: operator= (const arl_rISrcE..
   ****************************************************************************/
   /*!
    * \brief   Assignment operator
    * \param   [rfcoISrcProp]:     (I)   Reference of the object to be assigned
    * \retval  [arl_rISrcProp&]: Reference of this pointer.
    * \date 24.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    ***************************************************************************/
   arl_rISrcProp& operator= (const arl_rISrcProp& rfcoISrcProp);

} arl_tISrcProp;


#endif   // #ifndef ARL_ISOURCETYPES_H_

////////////////////////////////////////////////////////////////////////////////
// <EOF>
