/*!
*******************************************************************************
* \file              arl_tclFsmTransition.cpp
*******************************************************************************
*  - PROJECT:        LCN2KAI
*  - SW-COMPONENT:   Audio Routing Library (ARL)
*  - DESCRIPTION:    Implements FSM Transitions
*  - COPYRIGHT:      &copy; 2011 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 26.06.2012 \version 2.0 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
* - Remove AHL inheritance to enable usage by object reference instead of inheritance
* - Remove MOST dependencies
* - Adapt to Project needs
*
* \date 10.01.2011 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#include "arl_tclFsmTransition.h"
#include "arl_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ARL_GENERIC_FSM
#include "trcGenProj/Header/arl_tclFsmTransition.cpp.trc.h"
#endif

/******************************************************************************
| defines and macros (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function prototype (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function implementation (scope: external-interfaces)
|----------------------------------------------------------------------------*/

/******************************************************************************
** FUNCTION:  arl_tclFsmTransition::arl_tclFsmTransition(tS16 s16NextState..
******************************************************************************/

arl_tclFsmTransition::arl_tclFsmTransition
(
   tS16 s16NextStateID
   , tS16 s16TriggerID
):m_s16NextStateID(s16NextStateID), m_s16TriggerID(s16TriggerID)
, m_oTransFuncs(), m_oConditions()
{
   ETG_TRACE_COMP(("[TASK:%d]Created a new State Transition with NextStateId: %d, Triggerid: %d"
      , OSAL_ThreadWhoAmI(), s16NextStateID, s16TriggerID));

   m_oTransFuncs.clear();
   m_oConditions.clear();
} // arl_tclFsmTransition::arl_tclFsmTransition(tS16 s16NextStateID, tS16..

/******************************************************************************
** FUNCTION:  virtual arl_tclFsmTransition::~arl_tclFsmTransition()
******************************************************************************/

/*virtual*/
arl_tclFsmTransition::~arl_tclFsmTransition()
{
   try
   {
      ETG_TRACE_COMP(("[TASK:%d]Deleting Transition for NextStateId: %d, Triggerid: %d"
         , OSAL_ThreadWhoAmI(), m_s16NextStateID, m_s16TriggerID));

      m_oTransFuncs.clear();
      m_oConditions.clear();
   }
   catch(...)
   {
   }
}  // arl_tclFsmTransition::~arl_tclFsmTransition()

/******************************************************************************
** FUNCTION:  tS16 arl_tclFsmTransition::s16GetNextStateID() const
******************************************************************************/

tS16 arl_tclFsmTransition::s16GetNextStateID() const
{
   return m_s16NextStateID;
}  // tS16 arl_tclFsmTransition::s16GetNextStateID()

/******************************************************************************
** FUNCTION:  tS16 arl_tclFsmTransition::s16GetTriggerID() const
******************************************************************************/

tS16 arl_tclFsmTransition::s16GetTriggerID() const
{
   return m_s16TriggerID;
}  // tS16 arl_tclFsmTransition::s16GetTriggerID()

/******************************************************************************
** FUNCTION:  tBool arl_tclFsmTransition::bActivated()
******************************************************************************/

tBool arl_tclFsmTransition::bActivated()
{
   tBool bTrigger    =  TRUE;

   for (arl_tCondListIter iterCond = m_oConditions.begin();
       ((TRUE == bTrigger) && (iterCond < m_oConditions.end())); ++iterCond)
   {
      arl_tFsmCondition* poFsmCond =  *iterCond;

      if (OSAL_NULL != poFsmCond)
      {
         bTrigger = (bTrigger && (*poFsmCond)());  // Execute the condition
      }  // if (OSAL_NULL != poFsmCond)

   }  // for (arl_tCondListIter iterCond = m_oConditions.begin(); iterCond <..

   ETG_TRACE_COMP(("[TASK:%d]Transition active: %d, NextStateId: %d, Triggerid: %d"
         , OSAL_ThreadWhoAmI(), bTrigger, m_s16NextStateID, m_s16TriggerID));

   return bTrigger;

}  // tBool arl_tclFsmTransition::bActivated()

/******************************************************************************
** FUNCTION:  tVoid arl_tclFsmTransition::operator()()
******************************************************************************/

tVoid arl_tclFsmTransition::operator ()()
{
   if (TRUE == bActivated())
   {
      tU8 u8Loop = 0;
      for(arl_tFunctorListIter iterFunc = m_oTransFuncs.begin();
         iterFunc < m_oTransFuncs.end(); ++iterFunc, ++u8Loop)
      {
         arl_tFsmFunctor* poFsmFunctor   =  *iterFunc;

         if (OSAL_NULL != poFsmFunctor)
         {
            tBool bRetVal = (*poFsmFunctor)();   // Execute the function

            ETG_TRACE_USR1(("[TASK:%d]Transition Triggerid: %d function(%d) ExecStat:%d"
               , OSAL_ThreadWhoAmI(), m_s16TriggerID, u8Loop
               , ETG_ENUM(ARL_STATUS, bRetVal)));

         }  // if (OSAL_NULL != poFsmFunctor)

      }  // for(arl_tFunctorListIter iterFunc = m_oTransFuncs.begin(); iter..

   }  // if (TRUE == bTrigger)
   else
   {
      // Condition failure.
   }  // End of if-else; if (TRUE == bActivated())

}  // tVoid arl_tclFsmTransition::operator ()() const

/******************************************************************************
** FUNCTION:  tVoid arl_tclFsmTransition::vAddCond(arl_tFsmCondition* const cp..
******************************************************************************/

tVoid arl_tclFsmTransition::vAddCond(arl_tFsmCondition *const cpoFsmCondition)
{
   m_oConditions.push_back(cpoFsmCondition);
}  // tVoid arl_tclFsmTransition::vAddCond(arl_tFsmCondition *const cpoFsmC..

/******************************************************************************
 ** FUNCTION:  tVoid arl_tclFsmTransition::vAdd(arl_tFsmFunctor* const cpoF..
******************************************************************************/

tVoid arl_tclFsmTransition::vAdd(arl_tFsmFunctor *const cpoFsmFunctor)
{
   m_oTransFuncs.push_back(cpoFsmFunctor);
}  // tVoid arl_tclFsmTransition::vAdd(arl_tFsmFunctor *const cpoFsmFunc..

////////////////////////////////////////////////////////////////////////////////
// <EOF>
