/*!
*******************************************************************************
* \file              arl_tclFsmState.cpp
*******************************************************************************
*  - PROJECT:        LCN2KAI
*  - SW-COMPONENT:   Audio Routing Library (ARL)
*  - DESCRIPTION:    Implements FSM State
*  - COPYRIGHT:      &copy; 2011 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 26.06.2012 \version 2.0 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
* - Remove AHL inheritance to enable usage by object reference instead of inheritance
* - Remove MOST dependencies
* - Adapt to Project needs
*
* \date 10.01.2011 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#include "arl_tclFsmState.h"
#include "arl_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ARL_GENERIC_FSM
#include "trcGenProj/Header/arl_tclFsmState.cpp.trc.h"
#endif

/******************************************************************************
| defines and macros (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function prototype (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function implementation (scope: external-interfaces)
|----------------------------------------------------------------------------*/

/******************************************************************************
** FUNCTION:  arl_tclFsmState::arl_tclFsmState(tS16 s16ParentStateId, tCSt..
******************************************************************************/

arl_tclFsmState::arl_tclFsmState(tS16 s16ParentStateId, tCString cszName)
:m_s16ParentStateId(s16ParentStateId), m_cszStateName(cszName), m_oEntryFunc()
, m_oExitFunc(), m_oTransitions()
{
   ETG_TRACE_COMP(("[TASK:%d]Created a new State with ParentStateId: %d, Name: %s"
      , OSAL_ThreadWhoAmI(), m_s16ParentStateId, m_cszStateName));
} // arl_tclFsmState::arl_tclFsmState(tS16 s16ParentStateId, tCString cs..

/******************************************************************************
** FUNCTION:  virtual arl_tclFsmState::~arl_tclFsmState()
******************************************************************************/

/*virtual*/
arl_tclFsmState::~arl_tclFsmState()
{
   try
   {
      ETG_TRACE_COMP(("[TASK:%d]Destroying State: %s", OSAL_ThreadWhoAmI()
         , m_cszStateName));

      m_oEntryFunc.clear();
      m_oExitFunc.clear();
      m_oTransitions.clear();
   }
   catch(...)
   {
   }
   m_cszStateName    =  OSAL_NULL;
}  // arl_tclFsmState::~arl_tclFsmState()

/******************************************************************************
** FUNCTION:  tVoid arl_tclFsmState::operator()(const arl_tenOnState enOnState)
******************************************************************************/

tVoid arl_tclFsmState::operator ()(const arl_tenOnState enOnState)
{
   ETG_TRACE_USR1(("[TASK:%d]Initiated %d action for State: %s", OSAL_ThreadWhoAmI()
      , ETG_ENUM(ARL_ONSTATE_ACTION, enOnState) , m_cszStateName));

   switch (enOnState)
   {
      case ARL_EN_STATE_ENTRY:
      {
         tU8 u8Loop  =  0;
         for (arl_tFunctorListIter iterFunc = m_oEntryFunc.begin();
             iterFunc < m_oEntryFunc.end(); ++iterFunc, ++u8Loop)
         {
            arl_tFsmFunctor* poFsmFunctor   =  *iterFunc;

            if (OSAL_NULL != poFsmFunctor)
            {
               tBool bRetVal = (*poFsmFunctor)();   // Execute the function

               ETG_TRACE_USR2(("State ENTRY function(%d) exited with ExecStat:%d"
                  , u8Loop, ETG_ENUM(ARL_STATUS, bRetVal)));

            }  // if (OSAL_NULL != poFsmFunctor)
         }
      }  // case ARL_EN_STATE_ENTRY:
      break;

      case ARL_EN_STATE_EXIT:
      {
         tU8 u8Loop  =  0;
         for (arl_tFunctorListIter iterFunc = m_oExitFunc.begin();
             iterFunc < m_oExitFunc.end(); ++iterFunc, ++u8Loop)
         {
            arl_tFsmFunctor* poFsmFunctor   =  *iterFunc;

            if (OSAL_NULL != poFsmFunctor)
            {
               tBool bRetVal =   (*poFsmFunctor)();   // Execute the function

               ETG_TRACE_USR2(("State EXIT function(%d) exited with ExecStat:%d"
                  , u8Loop, ETG_ENUM(ARL_STATUS, bRetVal)));

            }  // if (OSAL_NULL != poFsmFunctor)
         }
      }  // case ARL_EN_STATE_EXIT:
      break;

      default:
      {
         ETG_TRACE_ERR(("Unknown State action functionality!!!!"));
      }  // default:
      break;
   }  // switch(enOnState)

}  // tVoid arl_tclFsmState::operator ()(const arl_tenOnState enOnState)

/******************************************************************************
** FUNCTION:  arl_tclFsmTransition* arl_tclFsmState::poGetTransition(..
******************************************************************************/

arl_tclFsmTransition* arl_tclFsmState::poGetTransition(tS16 s16TriggerId)
{
   ETG_TRACE_USR1(("[TASK:%d]Getting a transition with TriggerId: %d State %s"
      , OSAL_ThreadWhoAmI(), s16TriggerId, m_cszStateName));

   arl_tclFsmTransition* poFsmTrans  =  OSAL_NULL;

   for (arl_tTransitionListIter iterTrans = m_oTransitions.begin();
         iterTrans < m_oTransitions.end(); ++iterTrans)
   {
      if ((*iterTrans)->s16GetTriggerID() == s16TriggerId)
      {
         poFsmTrans  =  *iterTrans;

         ETG_TRACE_USR1(("TriggerId: %d was found, Next StateId: %d"
            , s16TriggerId, (*iterTrans)->s16GetNextStateID()));

         break;
      }  // if (iterTrans->s16GetTriggerID() == s16TriggerId)
      else
      {
         // Trigger id did not match
      }  // End of if-else; if (iterTrans->s16GetTriggerID() == s16TriggerId)
   }

   return poFsmTrans;
}  // arl_tclFsmTransition* arl_tclFsmState::poGetTransition(tS16 s16Tr..

/******************************************************************************
** FUNCTION:  tVoid arl_tclFsmState::vAdd(arl_tFsmFunctor* const cpoFsmF...
******************************************************************************/

tVoid arl_tclFsmState::vAdd
(
   arl_tFsmFunctor* const cpoFsmFunctor
   , arl_tenOnState enOnState
)
{
   ETG_TRACE_USR1(("[TASK:%d]Adding %d action for State: %s", OSAL_ThreadWhoAmI()
      , ETG_ENUM(ARL_ONSTATE_ACTION, enOnState) , m_cszStateName));

   switch (enOnState)
   {
      case ARL_EN_STATE_ENTRY:
      {
         m_oEntryFunc.push_back(cpoFsmFunctor);
      }  // case ARL_EN_STATE_ENTRY:
      break;

      case ARL_EN_STATE_EXIT:
      {
         m_oExitFunc.push_back(cpoFsmFunctor);
      }  // case ARL_EN_STATE_EXIT:
      break;

      default:
      {
         ETG_TRACE_ERR(("Unknown State action functionality!!!!"));
      }  // default:
      break;
   }
}  // tVoid arl_tclFsmState::vAdd(arl_tFsmFunctor *const cpoFsmFunctor, ..

/******************************************************************************
** FUNCTION:  tVoid arl_tclFsmState::vAdd(arl_tclFsmTransition *const cpoFs..
******************************************************************************/

tVoid arl_tclFsmState::vAdd(arl_tclFsmTransition *const cpoFsmTransition)
{
   ETG_TRACE_USR1(("[TASK:%d]Adding transition with TriggerId:%d for State: %s"
      , OSAL_ThreadWhoAmI(), cpoFsmTransition->s16GetTriggerID(), m_cszStateName));

   m_oTransitions.push_back(cpoFsmTransition);
}  // tVoid arl_tclFsmState::vAdd(arl_tclFsmTransition *const cpoFsmTransition)

/******************************************************************************
** FUNCTION:  tCString arl_tclFsmState::cszGetStateName() const
******************************************************************************/

tCString arl_tclFsmState::cszGetStateName() const
{
   return m_cszStateName;
}  // tCString arl_tclFsmState::cszGetStateName() const

/******************************************************************************
** FUNCTION:  tS16 arl_tclFsmState::s16GetParentStateId() const
******************************************************************************/

tS16 arl_tclFsmState::s16GetParentStateId() const
{
   return m_s16ParentStateId;
}  // tS16 arl_tclFsmState::s16GetParentStateId() const

////////////////////////////////////////////////////////////////////////////////
// <EOF>
