/*!
********************************************************************************
* \file              arl_tclFsm.cpp
********************************************************************************
*  - PROJECT:        LCN2KAI
*  - SW-COMPONENT:   Audio Routing Library (ARL)
*  - DESCRIPTION:    Finite State Machine (FSM)
*  - COPYRIGHT:      &copy; 2011 Robert Bosch Car Multimedia Gmbh
********************************************************************************
* \date 26.06.2012 \version 2.0 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
* - Remove AHL inheritance to enable usage by object reference instead of inheritance
* - Remove MOST dependencies
* - Adapt to Project needs
*
* \date 10.01.2011 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
*******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#include "arl_tclFsm.h"
#include "arl_tclFsmTrigger.h"
#include "arl_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ARL_GENERIC_FSM
#include "trcGenProj/Header/arl_tclFsm.cpp.trc.h"
#endif

/******************************************************************************
| defines and macros (scope: module-local)
|----------------------------------------------------------------------------*/
#define ARL_FSM_DEFAULT_NAME(name)  ((OSAL_NULL != name)?(name):("FSM_DEFAULT_NAME"))

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function prototype (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function implementation (scope: external-interfaces)
|----------------------------------------------------------------------------*/

/******************************************************************************
** FUNCTION:  arl_tclFsm::arl_tclFsm(tCString cszName = OSAL_NULL)
******************************************************************************/

/*explicit*/
arl_tclFsm::arl_tclFsm(tCString cszName):m_bFsmFinalized(FALSE)
, m_s16CurrentStateId(ARL_FSM_STATE_ID_INVALID), m_cszFsmName(cszName)
, m_oStates(), m_oTriggers()
{
   ETG_TRACE_COMP(("[TASK:%d]Creating FSM: %s", OSAL_ThreadWhoAmI()
      , ARL_FSM_DEFAULT_NAME(m_cszFsmName)));
} // arl_tclFsm::arl_tclFsm(tCString cszName)

/******************************************************************************
** FUNCTION:  virtual arl_tclFsm::~arl_tclFsm()
******************************************************************************/

/*virtual*/
arl_tclFsm::~arl_tclFsm()
{
   ETG_TRACE_COMP(("[TASK:%d]Destroying FSM: %s", OSAL_ThreadWhoAmI()
      , ARL_FSM_DEFAULT_NAME(m_cszFsmName)));

   m_cszFsmName   =  OSAL_NULL;
} // arl_tclFsm::~arl_tclFsm()

/******************************************************************************
** FUNCTION:  tS16 arl_tclFsm::s16AddState(tCString cszStateName, tS16 s1..
******************************************************************************/

tS16 arl_tclFsm::s16AddState(tCString cszStateName, tS16 s16ParentStateId)
{
   tS16 s16StateId   =  ARL_FSM_STATE_ID_INVALID;

   if (FALSE == m_bFsmFinalized)
   {
      s16StateId  =  ((tS16)m_oStates.size());
      arl_tclFsmState oState(s16ParentStateId, cszStateName);

      m_oStates.push_back(oState);

      ETG_TRACE_USR1(("[TASK:%d]Adding state:%d to Parent state:%d State:%s in FSM"
         , OSAL_ThreadWhoAmI(), s16StateId, s16ParentStateId
         , ARL_FSM_DEFAULT_NAME(cszStateName)));

   }  // if (FALSE == m_bFsmFinalized)
   else
   {
      // FSM finalized hence cannot add a new state
      ETG_TRACE_ERR(("FSM finalized, a new state cannot be added!!!!"));
   }  // End of if-else; if (FALSE == m_bFsmFinalized)

   return s16StateId;

}  // tS16 arl_tclFsm::s16AddState(tS16 s16ParentStateId, tCString cszState..

/******************************************************************************
** FUNCTION:  tS16 arl_tclFsm::s16AddTrigger(tCString cszTrigName)
******************************************************************************/

tS16 arl_tclFsm::s16AddTrigger(tCString cszTrigName)
{
   tS16 s16TriggerId   =  ARL_FSM_TRIG_ID_INVALID;

   if (FALSE == m_bFsmFinalized)
   {
      s16TriggerId  =  ((tS16)m_oTriggers.size());
      arl_tclFsmTrigger oTrigger(cszTrigName);

      m_oTriggers.push_back(oTrigger);

      ETG_TRACE_USR1(("[TASK:%d]Adding trigger:%d name: %s to FSM", OSAL_ThreadWhoAmI()
         , s16TriggerId, ARL_FSM_DEFAULT_NAME(cszTrigName)));

   }  // if (FALSE == m_bFsmFinalized)
   else
   {
      // FSM finalized hence cannot add a new trigger
      ETG_TRACE_ERR(("FSM finalized, a new trigger cannot be added!!!!"));
   }  // End of if-else; if (FALSE == m_bFsmFinalized)

   return s16TriggerId;

}  // tS16 arl_tclFsm::s16AddTrigger(tCString cszTrigName)

/******************************************************************************
** FUNCTION:  tBool arl_tclFsm::bAdd(tS16 s16StateId, arl_tFsmFunctor* co..
******************************************************************************/

tBool arl_tclFsm::bAdd
(
   tS16 s16StateId
   , arl_tFsmFunctor* const cpoFsmFunctor
   , arl_tclFsmState::arl_tenOnState enOnState
)
{
   tBool bRetVal  =  TRUE;

   if (FALSE == m_bFsmFinalized)
   {
      if (TRUE == bIsValidState(s16StateId))
      {
         m_oStates[(tU16)s16StateId].vAdd(cpoFsmFunctor, enOnState);
      }  // if (TRUE == bIsValidState(s16StateId))
      else
      {
         // Error
         bRetVal  =  FALSE;
         ETG_TRACE_ERR(("Invalid state id!!!!"));
      }  // End of if-else; if (TRUE == bIsValidState(s16StateId))

   }  // if (FALSE == m_bFsmFinalized)
   else
   {
      // FSM finalized
      bRetVal  =  FALSE;
      ETG_TRACE_ERR(("FSM finalized, a new on state action cannot be added!!!!"));
   }  // End of if-else; if (FALSE == m_bFsmFinalized)

   return bRetVal;

}  // tBool arl_tclFsm::bAdd(tS16 s16StateId, arl_tFsmFunctor *const cpo..

/******************************************************************************
** FUNCTION:  tBool arl_tclFsm::bSetInitState(tS16 s16StateId)
******************************************************************************/

tBool arl_tclFsm::bSetInitState(tS16 s16StateId)
{
   tBool bRetVal  =  TRUE;

   if (FALSE == m_bFsmFinalized)
   {
      bRetVal  =  bSetState(s16StateId);
   }  // if (FALSE == m_bFsmFinalized)
   else
   {
      // FSM finalized
      bRetVal  =  FALSE;
      ETG_TRACE_ERR(("FSM finalized, a new INIT state cannot be set!!!!"));
   }  // End of if-else; if (FALSE == m_bFsmFinalized)

   return bRetVal;

}  // tBool arl_tclFsm::bSetInitState(tS16 s16StateId)

/******************************************************************************
** FUNCTION:  tBool arl_tclFsm::bSetState(tS16 s16StateId)
******************************************************************************/

tBool arl_tclFsm::bSetState(tS16 s16StateId)
{
   tBool bRetVal  =  TRUE;

   if (TRUE == bIsValidState(s16StateId))
   {
      m_s16CurrentStateId  =  s16StateId;

      ETG_TRACE_USR1(("[TASK:%d]Setting current state:%d to FSM: %s"
         , OSAL_ThreadWhoAmI(), s16StateId, ARL_FSM_DEFAULT_NAME(m_cszFsmName)));

   }  // if (TRUE == bIsValidState(s16StateId))
   else
   {
      bRetVal  =  FALSE;
      ETG_TRACE_ERR(("Invalid state id!!!!"));
   }  // End of if-else; if (TRUE == bIsValidState(s16StateId))

   return bRetVal;
}  // tBool arl_tclFsm::bSetState(tS16 s16StateId)

/******************************************************************************
** FUNCTION:  tBool arl_tclFsm::bIsValidState(tS16 s16StateId) const
******************************************************************************/

tBool arl_tclFsm::bIsValidState(tS16 s16StateId) const
{
   return ((ARL_FSM_STATE_ID_INVALID < s16StateId)
         && (s16StateId < ((tS16)m_oStates.size())));

}  // tBool arl_tclFsm::bIsValidState(tS16 s16StateId) const

/******************************************************************************
** FUNCTION:  tBool arl_tclFsm::bIsValidTrigger(tS16 s16Trigger) const
******************************************************************************/

tBool arl_tclFsm::bIsValidTrigger(tS16 s16Trigger) const
{
   return ((ARL_FSM_TRIG_ID_INVALID < s16Trigger)
         && (s16Trigger < ((tS16)m_oTriggers.size())));

}  // tBool arl_tclFsm::bIsValidTrigger(tS16 s16Trigger) const

/******************************************************************************
** FUNCTION:  tVoid arl_tclFsm::operator()(tS16 s16TriggerId)
******************************************************************************/

tVoid arl_tclFsm::operator ()(tS16 s16TriggerId)
{
   if (TRUE == bIsValidTrigger(s16TriggerId))
   {
      // Get Trigger name for tracing
      tCString cszTrigName =  m_oTriggers[(tU16)s16TriggerId].cszGetName();
      tBool bTrigInvoked   =  FALSE;

      if (TRUE == bIsValidState(m_s16CurrentStateId))
      {
         ETG_TRACE_USR1(("[TASK:%d]Trigger id: %d initiated in State:%d for FSM: %s."
               , OSAL_ThreadWhoAmI(), s16TriggerId, m_s16CurrentStateId
               , ARL_FSM_DEFAULT_NAME(m_cszFsmName)));

         ETG_TRACE_USR1(("Trigger Name: %s initiated", cszTrigName));

         arl_tclFsmStateListIter iterCurrState =
            (m_oStates.begin() + m_s16CurrentStateId);

         tBool bCheckParentState =  TRUE;

         while (TRUE == bCheckParentState)
         {
            arl_tclFsmTransition* const poFsmTrans =
               iterCurrState->poGetTransition(s16TriggerId);

            if (OSAL_NULL != poFsmTrans)
            {
               tS16 s16NextStateId  =  poFsmTrans->s16GetNextStateID();

               if (TRUE == bIsValidState(s16NextStateId))
               {
                  // Get the next state.
                  arl_tclFsmState& rfoNextState  =   m_oStates[(tU16)s16NextStateId];

                  ETG_TRACE_USR1(("State change:[Next State(%d)]<--[State(%s)]"
                     , s16NextStateId, iterCurrState->cszGetStateName()));

                  bTrigInvoked   =  poFsmTrans->bActivated();

                  if (TRUE == bTrigInvoked)
                  {
                     // Do the state change.
                     // 1. Invoke the current state exit actions.
                     (*iterCurrState)(ARL_ON_STATE_EXIT);

                     // 2. Invoke transition functions
                     (*poFsmTrans)();

                     // 3. Change the state.
                     (tVoid)bSetState(s16NextStateId);

                     // 4. Invoke the next state entry actions.
                     rfoNextState(ARL_ON_STATE_ENTRY);

                     ETG_TRACE_USR1(("State changed to [State(%d)(%s)]"
                        , s16NextStateId, rfoNextState.cszGetStateName()));
                  }  // if (TRUE == bTrigInvoked)
                  else
                  {
                     ETG_TRACE_USR1(("Transition is not active."));
                  }  // End of if-else; if (TRUE == bTrigInvoked)

               }  // if (TRUE == bIsValidState(s16NextStateId))
               else
               {
                  // Next state is not valid one.
               }  // End of if-else; if (TRUE == bIsValidState(poFsmTrans->s16Get..

            }  // if (OSAL_NULL != cpoTrans)
            else
            {
               // Trigger not invoked as transition could not be found.
               ETG_TRACE_USR1(("Transition for trigger:%d, State: %s", s16TriggerId
                  , iterCurrState->cszGetStateName()));
            }  // End of if-else; if (OSAL_NULL != cpoTrans)

            tS16 s16ParentStateId   =  iterCurrState->s16GetParentStateId();

            if ((FALSE == bTrigInvoked)
               && (ARL_FSM_STATE_ID_INVALID != s16ParentStateId))
            {
               iterCurrState  =  (m_oStates.begin() + s16ParentStateId);

               ETG_TRACE_USR1(("Getting parent state for trigger:%d, State: %s"
                  , s16TriggerId, iterCurrState->cszGetStateName()));

            }  // if((FALSE == bTrigInvoked)&& (ARL_FSM_STATE_ID_INVALID..
            else
            {
               // No more parent states to check ;-), break
               bCheckParentState =  FALSE;

               ETG_TRACE_USR1(("No more parent states to check, exiting the search"));
            }  // End of if-else; if((FALSE == bTrigInvoked)&& (ARL_FSM_S16..

         }  //while (TRUE == bCheckParentState)

         if (FALSE == bTrigInvoked)
         {
            ETG_TRACE_ERR(("[TASK:%d]Trigger id: %d was not invoked for FSM: %s.!!!!"
               , OSAL_ThreadWhoAmI(), s16TriggerId
               , ARL_FSM_DEFAULT_NAME(m_cszFsmName)));

         }  // if (FALSE == bTrigInvoked)

      }  // if (TRUE == bIsValidState(m_s16CurrentStateId))
      else
      {
      }  // End of if-else; if (TRUE == bIsValidState(m_s16CurrentStateId))

   }  // if (TRUE == bIsValidTrigger(s16TriggerId))
   else
   {
      // Error trace.
      ETG_TRACE_ERR(("[TASK:%d]Invalid Trigger id: %d!!!!", OSAL_ThreadWhoAmI()
         , s16TriggerId));
   }  // End of if-else; if (TRUE == bIsValidTrigger(s16TriggerId))

}  // tVoid arl_tclFsm::operator ()(tS16 s16TriggerId)

/******************************************************************************
** FUNCTION:  tS16 arl_tclFsm::s16GetCurrentState() const
******************************************************************************/

tS16 arl_tclFsm::s16GetCurrentState() const
{
   return m_s16CurrentStateId;
}  // tS16 arl_tclFsm::s16GetCurrentState() const

/******************************************************************************
** FUNCTION:  tBool arl_tclFsm::bFsmFinalize()
******************************************************************************/

tBool arl_tclFsm::bFsmFinalize()
{
   if (FALSE == m_bFsmFinalized)
   {
      ETG_TRACE_USR1(("[TASK:%d]FSM: %s finalized.", OSAL_ThreadWhoAmI()
         , ARL_FSM_DEFAULT_NAME(m_cszFsmName)));

      if (TRUE == bIsValidState(m_s16CurrentStateId))
      {
         // Finalize the FSM
         m_bFsmFinalized   =  TRUE;

         // Trigger the entry into initial state.
         (m_oStates[(tU16)m_s16CurrentStateId])(ARL_ON_STATE_ENTRY);
      }
      else
      {
         ETG_TRACE_ERR(("Current state while finalization is invalid!!!!"));
      }
   }
   else
   {
      // Nothing to do. FSM already finalized. - Developer error.
      NORMAL_M_ASSERT_ALWAYS();
   }

   return m_bFsmFinalized;
}  // tBool arl_tclFsm::bFsmFinalize()

/******************************************************************************
* FUNCTION:  arl_tclFsmTransition* arl_tclFsm::poAddTransition(tS16 s16Source..
******************************************************************************/

arl_tclFsmTransition* arl_tclFsm::poAddTransition
(
   tS16 s16SourceStateId
   , tS16 s16DestStateId
   , tS16 s16TriggerId
)
{
   arl_tclFsmTransition* poTransition  =  OSAL_NULL;

   if (FALSE == m_bFsmFinalized)
   {
      if ((TRUE == bIsValidState(s16SourceStateId))   // Source State is valid
         && (TRUE == bIsValidState(s16DestStateId))   // Dest State is valid
         && (TRUE == bIsValidTrigger(s16TriggerId)))  // Trigger is valid
      {
         // Create a new transition.
         poTransition   =  OSAL_NEW arl_tclFsmTransition(s16DestStateId, s16TriggerId);

         NORMAL_M_ASSERT(OSAL_NULL != poTransition);

         // Add to the state transitions.
         m_oStates[(tU16)s16SourceStateId].vAdd(poTransition);

         ETG_TRACE_USR1(("[TASK:%d]Added transition for \
            [SourceState(%d)]--[Trigger(%d)]-->[DestState(%d)] for FSM: %s."
            , OSAL_ThreadWhoAmI(), s16SourceStateId, s16DestStateId, s16TriggerId
            , ARL_FSM_DEFAULT_NAME(m_cszFsmName)));

      }
      else
      {
         // Developer error something wrong in SM design.
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
   else
   {
      // Finalized, cannot add new transitions - developer error.
      NORMAL_M_ASSERT_ALWAYS();
   }

   return poTransition;
}  // arl_tclFsmTransition* arl_tclFsm::poAddTransition(tS16 s16Source..

/******************************************************************************
** FUNCTION:  tBool arl_tclFsm::bAddTransitionFunc(arl_tclFsmTransition* c..
******************************************************************************/

tBool arl_tclFsm::bAddTransitionFunc
(
   arl_tclFsmTransition* const cpoTransition
   , arl_tFsmFunctor* const cpoFunction
) const
{
   tBool bRetVal  =  FALSE;

   if (FALSE == m_bFsmFinalized)
   {
      if ((OSAL_NULL != cpoTransition) && (OSAL_NULL != cpoFunction))
      {
         cpoTransition->vAdd(cpoFunction);
         bRetVal  =  TRUE;

         ETG_TRACE_USR1(("[TASK:%d]Added transition function for FSM: %s."
            , OSAL_ThreadWhoAmI(), ARL_FSM_DEFAULT_NAME(m_cszFsmName)));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Adding transition function after finalization is invalid!!"));
   }

   return bRetVal;
}  // tBool arl_tclFsm::bAddTransitionFunc(arl_tclFsmTransition* const cpo..

/******************************************************************************
** FUNCTION:  tBool arl_tclFsm::bAddCondition(arl_tclFsmTransition* cons..
******************************************************************************/

tBool arl_tclFsm::bAddCondition
(
   arl_tclFsmTransition* const cpoTransition
   , arl_tFsmCondition* const cpoCond
) const
{
   tBool bRetVal  =  FALSE;

   if (FALSE == m_bFsmFinalized)
   {
      if ((OSAL_NULL != cpoTransition) && (OSAL_NULL != cpoCond))
      {
         cpoTransition->vAddCond(cpoCond);
         bRetVal  =  TRUE;

         ETG_TRACE_USR1(("[TASK:%d]Added condition function for FSM: %s."
            , OSAL_ThreadWhoAmI(), ARL_FSM_DEFAULT_NAME(m_cszFsmName)));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Adding condition function after finalization is invalid!!"));
   }

   return bRetVal;
}  // tBool arl_tclFsm::bAddCondition(arl_tclFsmTransition* const cpoTransiti..

/******************************************************************************
** FUNCTION:  tBool arl_tclFsm::UT_SetState(tS16 s16StateId)
** Only for ARL Unit Test this function is called
******************************************************************************/
tVoid arl_tclFsm::vSetState(tS16 s16StateId)
{
  m_s16CurrentStateId  =  s16StateId;
}
////////////////////////////////////////////////////////////////////////////////
// <EOF>
