

/*******************************************************************************
*
* FILE:          FC_StreamRouter_main.cpp
*
* SW-COMPONENT:  FC_StreamRouter application
*
* PROJECT:
*
* DESCRIPTION:   CCA Application.
*
* AUTHOR:       TMS Neumann
*
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/


#include "FC_StreamRouter_main.h"

#include "FC_StreamRouter_service_StreamRouter.h"

#ifndef VARIANT_S_FTR_ENABLE_GM_STR
  #include "FC_StreamRouter_clienthandler_AudioMgmt.h"
#endif

#include "FC_StreamRouter_trace_def.h"

#define STR_STR_STR
#include "FC_StreamRouter_ucid_defs.h"

#include "FC_StreamRouter_fwdtable.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_STREAMROUTER_APPLICATION
#include "trcGenProj/Header/FC_StreamRouter_main.cpp.trc.h"
#endif

// needed for cca list data type
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_STREAMROUTERFI_TYPES
#include "midw_fi_if.h"

#ifdef VARIANT_S_FTR_ENABLE_GM_STR
  #include "FC_StreamRouter_configMic.h"
#endif

#ifndef VARIANT_S_FTR_ENABLE_GM_STR
  #define NO_DIAGNOSIS
#endif

#ifndef NO_DIAGNOSIS
  #include "Diagnostic/FC_StreamRouter_Diagnosis.h"
#endif

#include "FC_StreamRouter_routingtable.h"
#include "FC_StreamRouter_mostwrp.h"

#include <alsa/asoundlib.h>


#define FC_STREAMROUTER_TIMER_ID 0

#define FC_STREAMROUTER_TIMER_TIME_UNTIL_FIRST_TICK 1000
#define FC_STREAMROUTER_TIMER_TICK_INTERVALL 20000

#define STR_CFGMASK_GST         0x0004
#define STR_CFGMASK_GST_SUSUKI  0x0008


// Static self reference to be used in function callbacks.
fc_streamrouter_tclApp* fc_streamrouter_tclApp::m_poMainAppInstance = NULL;

static OSAL_tThreadID _tStartGstreamerThreadID = OSAL_NULL;

/*******************************************************************************
*
* FUNCTION: fc_streamrouter_tclApp::fc_streamrouter_tclApp():ahl_tclBaseOneThreadApp()
*
* DESCRIPTION: Constructor.
*
*              Initialize static self reference to be used in function
*              callbacks.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_streamrouter_tclApp::fc_streamrouter_tclApp():ahl_tclBaseOneThreadApp()
{
  ETG_TRACE_USR4(("fc_streamrouter_tclApp() entered."));

  sndcard_handle = NULL;
  m_poStreamRouter = NULL;

#ifdef VARIANT_S_FTR_ENABLE_GM_STR
  m_poConfigMic = NULL;
#endif

  pFwdTable = NULL;
  rtable = NULL;

#ifndef VARIANT_S_FTR_ENABLE_GM_STR
  m_poClientHandlerAudioMgmt = NULL;
#endif

#ifndef NO_DIAGNOSIS
  m_poDiagnosis = NULL;
#endif

  m_mostwrp = NULL;
  m_most_fastev = NULL;

  fc_streamrouter_tclApp::m_poMainAppInstance = this;
}

/*******************************************************************************
*
* FUNCTION: fc_streamrouter_tclApp::~fc_streamrouter_tclApp()
*
* DESCRIPTION: Destructor.
*
*              Invalidate static self reference.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_streamrouter_tclApp::~fc_streamrouter_tclApp()
{
  ETG_TRACE_USR4(("~fc_streamrouter_tclApp() entered."));

  sndcard_handle = NULL;
  m_poStreamRouter = NULL;

#ifdef VARIANT_S_FTR_ENABLE_GM_STR
  m_poConfigMic = NULL;
#endif

  pFwdTable = NULL;
  rtable = NULL;

#ifndef VARIANT_S_FTR_ENABLE_GM_STR
  m_poClientHandlerAudioMgmt = NULL;
#endif

#ifndef NO_DIAGNOSIS
  m_poDiagnosis = NULL;
#endif

  m_mostwrp = NULL;
  m_most_fastev = NULL;

  fc_streamrouter_tclApp::m_poMainAppInstance = NULL;
} //lint !e1540

fc_streamrouter_tclApp* fc_streamrouter_tclApp::theServer()
{

   if(m_poMainAppInstance == NULL)
   {
      m_poMainAppInstance = new fc_streamrouter_tclApp();
   }
   return m_poMainAppInstance;
}

fc_streamrouter_tclService_StreamRouter* fc_streamrouter_tclApp::getStreamrouterService()
{
  return m_poStreamRouter;
}

/*******************************************************************************
*
* FUNCTION: tBool fc_streamrouter_tclApp::bOnInit()
*
* DESCRIPTION: This function is called by the CCA framework before starting any
*              CCA message related communication action. As default this
*              function always returns TRUE. The user has to implement all his
*              local application initialization (create client and service
*              handler, create worker threads, allocate application memory,...).
*              If returns TRUE, initialization could be successfully performed
*              and framework will register application, else, a reset of the
*              system is forced immediately.
*
*              This function is the counterpart of vOnApplicationClose().
*
* PARAMETER: None.
*
* RETURNVALUE: TRUE  = Application successfully initialized.
*              FALSE = Application NOT successfully initialized.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::bOnInit().
*******************************************************************************/
tBool fc_streamrouter_tclApp::bOnInit()
{
  tBool bSuccess = TRUE;

  ETG_TRACE_USR4(("bOnInit() entered."));

  m_poStreamRouter = OSAL_NEW fc_streamrouter_tclService_StreamRouter(this);
  if(NULL == m_poStreamRouter)
  {
    bSuccess = FALSE;
    ETG_TRACE_FATAL(( "bOnInit: creation of Service_StreamRouter class failed" ));
    FATAL_M_ASSERT_ALWAYS();
  }

#ifdef VARIANT_S_FTR_ENABLE_GM_STR
  m_poConfigMic = OSAL_NEW fc_streamrouter_tclConfigMic();

  if (NULL == m_poConfigMic)
  {
    bSuccess = FALSE;
    ETG_TRACE_FATAL(( "bOnInit: creation of ConfigMic class failed" ));
    FATAL_M_ASSERT_ALWAYS();
  }
#endif

#ifndef VARIANT_S_FTR_ENABLE_GM_STR
  m_poClientHandlerAudioMgmt = OSAL_NEW fc_streamrouter_tclClientHandler_audiomgmt(this);
  if(m_poClientHandlerAudioMgmt == NULL)
  {
    bSuccess = FALSE;
    ETG_TRACE_FATAL(( "bOnInit: creation of  AudioMgmt ClientHandler failed" ));
    FATAL_M_ASSERT_ALWAYS();
  }
#endif

  if(FALSE == bStartTimer(FC_STREAMROUTER_TIMER_ID,
                          FC_STREAMROUTER_TIMER_TIME_UNTIL_FIRST_TICK,
                          FC_STREAMROUTER_TIMER_TICK_INTERVALL))
  {
    bSuccess = FALSE;
    FATAL_M_ASSERT_ALWAYS();
  }

  rtable = new routingtable();
  if(rtable == NULL)
  {
    ETG_TRACE_FATAL((">>> ! UNABLE INSTANCIATE ROUTINGTABLE ! <<<"));
  }

  pFwdTable = new FwdTable();
  if(pFwdTable == NULL)
  {
    ETG_TRACE_FATAL((">>> ! UNABLE INSTANCIATE FORWARDINGTABLE ! <<<"));
  }

#ifndef NO_DIAGNOSIS
  /* generate the diagnosis class */
  m_poDiagnosis = OSAL_NEW fc_streamrouter_tclDiagnosis(this);
  if (m_poDiagnosis == NULL)
  {
     bSuccess = FALSE;
     ETG_TRACE_FATAL(( "bOnInit: creation of diagnosis class for Diagnostic failed" ));
     FATAL_M_ASSERT_ALWAYS();
  }
#endif

  m_mostwrp = OSAL_NEW fc_streamrouter_mostwrp();
  if(m_mostwrp == NULL)
  {
     bSuccess = FALSE;
     ETG_TRACE_FATAL(("bOnInit: creation of mostwrp failed!"));
     FATAL_M_ASSERT_ALWAYS();
  }

  m_most_fastev = OSAL_NEW fc_streamrouter_most_fastev(this);
  if(m_most_fastev == NULL)
  {
    bSuccess = FALSE;
    ETG_TRACE_FATAL(("bOnInit: creation of most_fastev failed!"));
    FATAL_M_ASSERT_ALWAYS();
  }

#ifdef STREAMROUTER_TRACE_IN
  vPlugTraceIn(TR_TTFIS_BOSCH_AUDIO, (OSAL_tpfCallback)vTraceInput_CB);
#endif

  return bSuccess;
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_streamrouter_tclApp::vOnApplicationClose()
*
* DESCRIPTION: This function is called by the CCA framework to indicate the
*              imminent shutdown of the application. The user has to implement
*              the de-initialization of all his local application data (destroy
*              client and service handler, destroy worker threads, free
*              application memory, ... ). After this function call the
*              destructor of this application will be called.
*
*              This function is the counterpart of bOnInit().
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnApplicationClose().
*******************************************************************************/
tVoid fc_streamrouter_tclApp::vOnApplicationClose()
{
  ETG_TRACE_USR4(("vOnApplicationClose() entered."));

  bStopTimer(FC_STREAMROUTER_TIMER_ID);

  try
  {
   if(m_poStreamRouter != NULL)
    {
      OSAL_DELETE m_poStreamRouter;
      m_poStreamRouter = NULL;
    }

#ifndef VARIANT_S_FTR_ENABLE_GM_STR
   if(m_poClientHandlerAudioMgmt != NULL)
    {
      OSAL_DELETE m_poClientHandlerAudioMgmt;
      m_poClientHandlerAudioMgmt = NULL;
    }
#endif
  }

  catch(...)
  {}

#if 0
  if(m_mostwrp != NULL)
  {
    OSAL_DELETE m_mostwrp;
    m_mostwrp = NULL;
  }

  if(m_most_fastev != NULL)
  {
    OSAL_DELETE m_most_fastev;
    m_most_fastev = NULL;
  }
#endif

  if(pFwdTable != NULL)
  {
    OSAL_DELETE pFwdTable;
    pFwdTable = NULL;
  }

  if(rtable != NULL)
  {
    OSAL_DELETE rtable;
    rtable = NULL;
  }

#ifdef VARIANT_S_FTR_ENABLE_GM_STR
  if(m_poConfigMic != NULL)
  {
    OSAL_DELETE m_poConfigMic;
    m_poConfigMic = NULL;
  }
#endif

#ifndef NO_DIAGNOSIS
  if (m_poDiagnosis != NULL)
  {
    OSAL_DELETE m_poDiagnosis;
    m_poDiagnosis = NULL;
  }
#endif

#ifdef STREAMROUTER_TRACE_IN
  vUnplugTrace(TR_TTFIS_BOSCH_AUDIO);
#endif

  ahl_tclBaseOneThreadApp::vOnApplicationClose();
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_streamrouter_tclApp::vOnLoadSettings()
*
* DESCRIPTION: This function is called by the CCA framework to trigger the
*              loading of last mode settings. The user has to load the
*              previously stored last mode settings from persistent memory.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnLoadSettings().
*******************************************************************************/
tVoid fc_streamrouter_tclApp::vOnLoadSettings()
{
  //ETG_TRACE_USR4(("vOnLoadSettings() entered."));
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_streamrouter_tclApp::vOnSaveSettings()
*
* DESCRIPTION: This function is called by the CCA framework to trigger the
*              storage of last mode settings. The user has to store the last
*              mode settings to persistent memory.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnSaveSettings().
*******************************************************************************/
tVoid fc_streamrouter_tclApp::vOnSaveSettings()
{
  //ETG_TRACE_USR4(("vOnSaveSettings() entered."));
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_streamrouter_tclApp::vOnTimer(tU16 u16TimerId)
*
* DESCRIPTION: This function is called by the CCA framework on the expiration
*              of a previously via function bStartTimer() started timer. The
*              expired timer is forwarded to the respective service or
*              client-handler via a call of vProcessTimer().
*
*              The function is called from this applications context and
*              therefore no interthread programming rules must be considered
*              and the application methods and/or member variables can be
*              accessed without using the static self reference
*              'm_poMainAppInstance'.
*
* PARAMETER: [IN] u16TimerId = Identifier of the expired timer.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnTimer().
*******************************************************************************/
tVoid fc_streamrouter_tclApp::vOnTimer(tU16 u16TimerId)
{
  ETG_TRACE_USR4(("vOnTimer() entered."));

  if(m_poStreamRouter != NULL)
  {
    m_poStreamRouter->vProcessTimer(u16TimerId);
  }
  else
  {
    NORMAL_M_ASSERT_ALWAYS();
  }

#if 0
  switch(u16TimerId)
  {
    case FC_STREAMROUTER_TIMER_ID:
      {

        if(NULL != m_poStreamRouter)
        {
          m_poStreamRouter->vProcessTimer(u16TimerId);
          //m_poStreamRouter->vSendTestMessage();
        }
        else
        {
          NORMAL_M_ASSERT_ALWAYS();
        }

        /*if(NULL != m_poClientHandlerMostMic)
        {
          m_poClientHandlerMostMic->vProcessTimer(u16TimerId);
        }
        else
        {
          NORMAL_M_ASSERT_ALWAYS();
        }*/

      }
      break;
    default:
      {
      }
  }
#endif
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_streamrouter_tclApp::vOnLoopback
*                 (tU16 u16ServiceID, amt_tclServiceData* poMessage)
*
* DESCRIPTION: This function is called by the CCA framework if a message from
*              one if this applications services is received (so called self
*              or loopback messages). The message is forwarded to the respective
*              service via a call of vOnLoopbackService().
*
* PARAMETER: [IN] u16ServiceID = Identifier of the service from where the
*                                message was sent.
*
*            [IN] poMessage = Message object.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnLoopback().
*******************************************************************************/
tVoid fc_streamrouter_tclApp::vOnLoopback(tU16 u16ServiceID, amt_tclServiceData* poMessage)
{
  ETG_TRACE_USR4(("vOnLoopback() entered. (Service ID: %d)", u16ServiceID));

  switch(u16ServiceID)
  {
    case CCA_C_U16_SRV_STREAMROUTER:
    {
      if(NULL != m_poStreamRouter)
      {
        m_poStreamRouter->vOnLoopbackService(poMessage);
      }
      else
      {
        NORMAL_M_ASSERT_ALWAYS();
      }
    }
    break;

    default:
    {
      ETG_TRACE_ERR(("vOnLoopback(). Unknown ServiceID %u.", u16ServiceID));
    }
  }
}

/*******************************************************************************
*
* FUNCTION: AllocateRoute(tU16 appid, tU16 ucid, tS32 stm_msgbuf_id, void* ccainfo)
*
* DESCRIPTION: This function allocate a Route
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
int fc_streamrouter_tclApp::AllocateRoute(tU16 appid, tU16 ucid, tS32 stm_msgbuf_id, void* ccainfo)
{
  ETG_TRACE_USR4(("Allocating Routes without Sink, using default Sink 1"));
  return  AllocateRoute(appid, ucid, (tU16)1, stm_msgbuf_id, ccainfo);
}
int fc_streamrouter_tclApp::AllocateRoute(tU16 appid, tU16 ucid, tU16 sink, tS32 stm_msgbuf_id, void* ccainfo)
{
  (void)stm_msgbuf_id;
  int ret = STR_ERR__OK;

  ETG_TRACE_USR4(("Allocating Routes for ucid 0x%x...", (tU16)ucid));

  if((rtable == NULL) || (pFwdTable == NULL))
  {
    ETG_TRACE_ERR(("FATAL: Data tables are NULL!"));
    return STR_ERR__INTERNAL;
  }

  rtable->lock();

  if(rtable->is_ucid((int)ucid) == 0)
  {
    int idx = 0;
    tU16 strid = getSTRID(ucid); // take care of EOL configuration
    ETG_TRACE_USR4(("strid: %d", strid));

    if(strid != 0)
    {
      int cfgid = pFwdTable->get_configid(strid, idx);
      ETG_TRACE_USR4(("cfgid: %d", cfgid));

      while(cfgid > 0)
      {
        ROUTINGENTRY* rentry = rtable->new_entry((int)appid, (int)ucid, (int)cfgid);
        if(rentry != NULL)
        {
          CONFIGENTRY* cfg = pFwdTable->get_cfg_entry(cfgid);
          if(cfg != NULL)
          {
            rentry->kill = false;
            rentry->strid = strid;

            ETG_TRACE_USR4(("Setting up cfgid: %d.", (tU8)cfgid));

            if(cfg->mask & STR_CFGMASK_BTCLK)
            {
              ETG_TRACE_USR4(("Setting up BT master clk"));
              mask_btclk_init();
            }
            ret = ret | AddRoute(rentry);

            if((ret & STR_ERR__MASK) != 0) // handle fatal errors
            {
              ETG_TRACE_FATAL(("ERR: FATAL ERRORS DURING ALLOCATION! (ucid: 0x%x, cfgid: %d): 0x%x (%s)", (tU16)ucid, (tU16)cfgid, (tU32)ret, getErrorString(ret)));
              RemoveRoutesPerUCID(ucid);
              break;
            }
            else // handle warnings
            {
              if(ret != 0)
              {
                ETG_TRACE_FATAL(("WRN: THERE WERE WARNINGS DURING ALLOCATION! (ucid: 0x%x, cfgid: %d): 0x%x (%s)", (tU16)ucid, (tU16)cfgid, (tU32)ret, getErrorString(ret)));
              }
              ETG_TRACE_USR4(("Allocated!"));
              ETG_TRACE_USR4(("ALSA device: \"%s\"", cfg->alsadev));
            }
          }
          else
          {
            ETG_TRACE_FATAL(("ERR: Error in forwarding table (configuration not found)!"));
            RemoveRoutesPerUCID(ucid);
            ret = ret | STR_ERR__INTERNAL;
            break;
          }
        }
        else
        {
          ETG_TRACE_ERR(("ERR: Did not get a free entry in routing table!"));
          RemoveRoutesPerUCID(ucid);
          ret = ret | STR_ERR__INTERNAL;
          break;
        }

        ++idx;
        cfgid = pFwdTable->get_configid(strid, idx);
      }
    }
    else
    {
      ETG_TRACE_ERR(("ERR: Invalid ucid!"));
      ret =  ret | STR_ERR__INTERNAL;
    }
  }

  if((ret & STR_ERR__MASK) == 0)
  {
    fill_cca_info(ucid, ccainfo);
  }

#ifndef VARIANT_S_FTR_ENABLE_GM_STR
  if((ret & STR_ERR__MASK) == 0)
  {
    int strid = getSTRID(ucid);
    STRIDENTRY* pIDEntry = pFwdTable->get_strid_entry(strid);

    int adrSrcNr           = (pIDEntry->adr).adrsrcnr;
    const char* adrStream  = (pIDEntry->adr).adrstream;
    tU32 adrStreamNr       = (pIDEntry->adr).adrstreamnr;
    const char* audProcAdev       = (pIDEntry->adr).audprocAdev;

    ETG_TRACE_USR4(("STRIDENTRY.strid       = %d", pIDEntry->strid));
    ETG_TRACE_USR4(("STRIDENTRY.name        = \"%s\"", pIDEntry->name));
    ETG_TRACE_USR4(("STRIDENTRY.configid[0] = %d", *(pIDEntry->configid)));

    ETG_TRACE_USR4(("adrSrcNr: \"%d\"", adrSrcNr));
    ETG_TRACE_USR4(("adrStream: \"%s\"", adrStream));
    ETG_TRACE_USR4(("adrStreamNr: \"%d\"", adrStreamNr));
    ETG_TRACE_USR4(("audProcAdev: \"%s\"", audProcAdev));

    ETG_TRACE_USR4(("sink: \"%d\"", sink));
    if(m_poClientHandlerAudioMgmt != NULL)
    {
      m_poClientHandlerAudioMgmt->bSendAmpDevAudioRoute((tU8)adrSrcNr, sink, adrStreamNr, audProcAdev, ccainfo, ACTION_ALLOCTAE);
    }
  }
  else
  {
    ETG_TRACE_ERR(("ERR: UCID either already allocated or invalid, no action (bSendAmpDevAudioRoute not called)"));
  }
#endif

  rtable->unlock();
  return ret;
}

/*******************************************************************************
*
* FUNCTION: AddRoute(ROUTINGENTRY *rentry))
*
* DESCRIPTION: This function add a Route in the RENTRY Table
*              (change sequence for BT latency optimization)
*
* PARAMETER:
*
* RETURNVALUE: None.
*
*******************************************************************************/
int fc_streamrouter_tclApp::AddRoute(ROUTINGENTRY *rentry)
{
  if(rentry == (ROUTINGENTRY*)NULL) return STR_ERR__INTERNAL;
  if(pFwdTable == NULL) return STR_ERR__INTERNAL;
  int iret = STR_ERR__OK;

  CONFIGENTRY* cfg = pFwdTable->get_cfg_entry(rentry->cfgid);
  if(cfg != NULL)
  {
    if((cfg->most.active) && (m_mostwrp != NULL))
    {
      int aret = m_mostwrp->allocate((unsigned int)(cfg->most.bwidth), (bool)(cfg->most.direction), 0, (unsigned int)(cfg->most.mcfgid));
      if((aret == 0) && (m_mostwrp->get_connectionlabel() != 0))
      {
        rentry->bwidth = cfg->most.bwidth; // not provided by acs, therefore copied from config
        rentry->conlabel = m_mostwrp->get_connectionlabel();
      }
      else
      {
        iret = STR_ERR__MSCM_CHANNEL_ALLOC;
      }
    }
  }
  return iret;
}

/*******************************************************************************
*
* FUNCTION: DeAllocateRoute(tU16 appid, tU16 ucid)
*
* DESCRIPTION: This function deallocate a Route
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
int fc_streamrouter_tclApp::DeAllocateRoute(tU16 appid, tU16 ucid)
{
  (void)appid;

  ETG_TRACE_USR4(("Deallocating routes for ucid 0x%x...", ucid));

  if((rtable == NULL) || (pFwdTable == NULL))
  {
    ETG_TRACE_ERR(("FATAL: Data tables are NULL!"));
    NORMAL_M_ASSERT_ALWAYS();
    return STR_ERR__INTERNAL;
  }


#ifndef VARIANT_S_FTR_ENABLE_GM_STR
    int strid = getSTRID(ucid);
    STRIDENTRY* pIDEntry = pFwdTable->get_strid_entry(strid);
    if(pIDEntry == NULL)
    {
       return STR_ERR__INTERNAL;
    }

    int adrSrcNr           = (pIDEntry->adr).adrsrcnr;
    const char* adrStream  = (pIDEntry->adr).adrstream;
    tU32 adrStreamNr       = (pIDEntry->adr).adrstreamnr;
    const char* audProcAdev       = (pIDEntry->adr).audprocAdev;

    ETG_TRACE_USR4(("STRIDENTRY.strid       = %d", pIDEntry->strid));
    ETG_TRACE_USR4(("STRIDENTRY.name        = \"%s\"", pIDEntry->name));
    ETG_TRACE_USR4(("STRIDENTRY.configid[0] = %d", *(pIDEntry->configid)));

    ETG_TRACE_USR4(("adrSrcNr: \"%d\"", adrSrcNr));
    ETG_TRACE_USR4(("adrStream: \"%s\"", adrStream));
    ETG_TRACE_USR4(("adrStreamNr: \"%d\"", adrStreamNr));
    ETG_TRACE_USR4(("audProcAdev: \"%s\"", audProcAdev));

    if(m_poClientHandlerAudioMgmt != NULL)
    {
      m_poClientHandlerAudioMgmt->bSendAmpDevAudioRoute((tU8)adrSrcNr, adrStreamNr, audProcAdev, NULL, ACTION_DEALLOCTAE);
    }

#endif
  rtable->lock();
  RemoveRoutesPerUCID(ucid);
  rtable->unlock();

  return STR_ERR__OK;
}

/*******************************************************************************
*
* FUNCTION: RemoveRoutesPerUCID(int ucid)
*
* DESCRIPTION: This function remove a route from the RENTRY Table per UCID
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
int fc_streamrouter_tclApp::RemoveRoutesPerUCID(int ucid)
{
  ROUTINGENTRY* rentry = 0;
  int idx = 0;
  int ret = 0;

  if(rtable == NULL)
  {
    ETG_TRACE_FATAL(("fc_streamrouter_tclApp::RemoveRoutesPerUCID(): NULLPTR ERROR!"));
    NORMAL_M_ASSERT_ALWAYS();
    return STR_ERR__INTERNAL;
  }

  rentry = rtable->get_next_entry(ucid, &idx);
  while(rentry != NULL)
  {
    ETG_TRACE_USR4(("Removing subroute cfgid %d.", (tU8)rentry->cfgid));
    ret = RemoveRoute(rentry);
    CONFIGENTRY* cfg = pFwdTable->get_cfg_entry(rentry->cfgid);
    if(cfg != NULL)
    {
      if(cfg->mask & STR_CFGMASK_BTCLK)
      {
        ETG_TRACE_USR4(("mask_btclk_deinit()..."));
        mask_btclk_deinit();
      }
    }
    if((ret & STR_ERR__MASK) != 0) // fatals
    {
      ETG_TRACE_FATAL(("ERR: FATAL ERRORS DURING DEALLOCATION! (ucid: 0x%x, cfgid: %d): 0x%x (%s)", (tU16)rentry->ucid, (tU16)rentry->cfgid, (tU32)ret, getErrorString(ret)));
    }
    else
    {
      if(ret != 0)
      {
        ETG_TRACE_FATAL(("WRN: THERE WERE WARNINGS DURING DEALLOCATION! (ucid: 0x%x, cfgid: %d): 0x%x (%s)", (tU16)rentry->ucid, (tU16)rentry->cfgid, (tU32)ret, getErrorString(ret)));
      }
    }
    ETG_TRACE_USR4(("Routing entry removed. (appid: 0x%x, ucid: 0x%x, cfgid: %d)", (tU16)rentry->appid, (tU16)rentry->ucid, (tU16)rentry->cfgid));
    ++idx;
    rentry = rtable->get_next_entry(ucid, &idx);
  }
  int nDelRoutes = rtable->free_entry((int)ucid);
  ETG_TRACE_USR4(("%d subroute entries removed", nDelRoutes));

  return 0; // this error is not handled, because a deallocation just throws error messages (no reallocation of the whole route)
}

/*******************************************************************************
*
* FUNCTION: RemoveRoute(ROUTINGENTRY *rentry)
*
* DESCRIPTION: This function remove a route from the RENTRY Table
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
int fc_streamrouter_tclApp::RemoveRoute(ROUTINGENTRY *rentry)
{
  if(rentry == NULL) return STR_ERR__INTERNAL;
  if(pFwdTable == NULL) return STR_ERR__INTERNAL;
  int iret = STR_ERR__OK;

  CONFIGENTRY* cfg = pFwdTable->get_cfg_entry(rentry->cfgid);
  if(rentry->conlabel != 0)
  {
    if(m_mostwrp != NULL)
    {
      if(cfg == NULL) return STR_ERR__INTERNAL;
      m_mostwrp->deallocate(rentry->conlabel, (unsigned int)(cfg->most.mcfgid));
    }
    rentry->conlabel = 0;
  }
  return iret;
}

/*******************************************************************************
*
* FUNCTION: Play(ROUTINGENTRY* rentry)
*
* DESCRIPTION: This function sets the route on Play
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
int fc_streamrouter_tclApp::Play(ROUTINGENTRY* rentry)
{
  if(rentry == NULL) return STR_ERR__INTERNAL;
  if(pFwdTable == NULL) return STR_ERR__INTERNAL;

  if(rentry->playing == false)
  {
    rentry->playing = true;

    CONFIGENTRY* cfg = pFwdTable->get_cfg_entry(rentry->cfgid);
    if(cfg != NULL)
    {
      ETG_TRACE_USR4(("Setting up cfgid: %d.", (tU16)cfg->configid));

      if(cfg->mask & STR_CFGMASK_GST) //if ((STRID_TESTMICINTOMOST == rentry->strid) || (STRCFG_STRMICROIN == rentry->strid))
      {
        ETG_TRACE_USR4(("Starting gstreamer"));
        mask_start_gstreamer(STR_CFGMASK_GST);
      }

      if(cfg->mask & STR_CFGMASK_GST_SUSUKI)
      {
        ETG_TRACE_USR4(("Starting gstreamer for SUSUKI"));
        mask_start_gstreamer(STR_CFGMASK_GST_SUSUKI);
      }

      if(cfg->most.active == true)
      {
        if(rentry->conlabel != 0)
        {
          if(m_mostwrp != NULL) m_mostwrp->demute(rentry->conlabel, cfg->most.mcfgid);
        }
      }
    }
  }
  else
  {
    return STR_WRN__ROUTE_STATE;
  }

  return STR_ERR__OK;
}

/*******************************************************************************
*
* FUNCTION: SetPlay(tU16 appid, tU16 ucid, tU8 ramptype)
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
int fc_streamrouter_tclApp::SetPlay(tU16 appid, tU16 ucid, tU8 ramptype)
{
  (void)appid;
  int ret = STR_ERR__STM_PLAY;

  ETG_TRACE_USR4(("Route start... (ucid 0x%x)", ucid));

  if((rtable == NULL) || (pFwdTable == NULL))
  {
    ETG_TRACE_ERR(("FATAL: Data tables are NULL!"));
    NORMAL_M_ASSERT_ALWAYS();
    return STR_ERR__INTERNAL;
  }
  rtable->lock();
  if(rtable->is_ucid((int)ucid) != 0)
  {
    ret = SetPlayPerUCID(ucid, ramptype);
  }
  rtable->unlock();

  return ret;
}

/*******************************************************************************
*
* FUNCTION: SetPlayPerUCID(tU16 ucid, tU8 ramptype, int* depth)
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
int fc_streamrouter_tclApp::SetPlayPerUCID(tU16 ucid, tU8 ramptype)
{
  (void)ramptype;
  int idx = 0;
  int ret = STR_ERR__OK;

  if(rtable == NULL)
  {
    ETG_TRACE_FATAL(("fc_streamrouter_tclApp::SetPlayPerUCID(): NULLPTR ERROR!"));
    NORMAL_M_ASSERT_ALWAYS();
    return STR_ERR__INTERNAL;
  }

  ROUTINGENTRY* rentry = rtable->get_next_entry(ucid, &idx);
  while(rentry != NULL)
  {
    ret = Play(rentry);
    if((ret & STR_ERR__MASK) != 0) // fatals
    {
      ETG_TRACE_FATAL(("ERR: ERRORS DURING ROUTE PLAY! (ucid: 0x%x, cfgid: %d): 0x%x (%s)", (tU16)ucid, (tU16)rentry->cfgid, (tU32)ret, getErrorString(ret)));
      SetStopPerUCID(ucid, ramptype);
      return ret;
    }
    else
    {
      if(ret != 0)
      {
        ETG_TRACE_FATAL(("WRN: WARNINGS DURING ROUTE PLAY! (ucid: 0x%x, cfgid: %d): 0x%x (%s)", (tU16)ucid, (tU16)rentry->cfgid, (tU32)ret, getErrorString(ret)));
      }
    }
    ++idx;
    rentry = rtable->get_next_entry(ucid, &idx);
  }
  return ret;
}

/*******************************************************************************
*
* FUNCTION: Stop(ROUTINGENTRY* rentry)
*
* DESCRIPTION: This function sets the route on Stop
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
int fc_streamrouter_tclApp::Stop(ROUTINGENTRY* rentry)
{
  int iret = STR_ERR__OK;

  if(rentry == NULL) return STR_ERR__INTERNAL;
  if(pFwdTable == NULL) return STR_ERR__INTERNAL;

  if(rentry->playing == true)
  {
    rentry->playing = false;

    CONFIGENTRY* cfg = pFwdTable->get_cfg_entry(rentry->cfgid);
    if (cfg != NULL)
    {
      ETG_TRACE_USR4(("Setting up cfgid: %d.", (tU16)cfg->configid));

      if((cfg->mask & STR_CFGMASK_GST) || (cfg->mask & STR_CFGMASK_GST_SUSUKI)) //if ((STRID_TESTMICINTOMOST == rentry->strid) || (STRCFG_STRMICROIN == rentry->strid))
      {
        ETG_TRACE_USR4(("Stop gsttreamer"));
        mask_stop_gstreamer();
      }

      if(cfg->most.active == true)
      {
        if(rentry->conlabel != 0)
        {
          if(m_mostwrp != NULL) m_mostwrp->mute(rentry->conlabel, cfg->most.mcfgid);
        }
      }
    }
  }
  else
  {
    return STR_WRN__ROUTE_STATE;
  }

  return iret;
}

/*******************************************************************************
*
* FUNCTION: SetStop(tU16 appid, tU16 ucid, tU8 ramptype)
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
int fc_streamrouter_tclApp::SetStop(tU16 appid, tU16 ucid, tU8 ramptype)
{
  (void)appid;

  int ret = STR_ERR__OK;
#ifdef VARIANT_S_FTR_ENABLE_GM_STR
  ret = STR_ERR__STM_STOP;
#endif

  ETG_TRACE_USR4(("Route stop... (ucid 0x%x)", ucid));

  if((rtable == NULL) || (pFwdTable == NULL))
  {
    ETG_TRACE_ERR(("FATAL: Data tables are NULL!"));
    return STR_ERR__INTERNAL;
  }

  rtable->lock();
  if(rtable->is_ucid((int)ucid) != 0)
  {
    ret = SetStopPerUCID(ucid, ramptype);
  }
  rtable->unlock();

  return ret;
}

/*******************************************************************************
*
* FUNCTION: SetStopPerUCID(tU16 ucid, tU8 ramptype, int* depth)
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
int fc_streamrouter_tclApp::SetStopPerUCID(tU16 ucid, tU8 ramptype)
{
  (void)ramptype;
  int idx = 0;
  int ret = STR_ERR__OK;

  if(rtable == NULL)
  {
    ETG_TRACE_FATAL(("fc_streamrouter_tclApp::SetStopPerUCID(): NULLPTR ERROR!"));
    NORMAL_M_ASSERT_ALWAYS();
    return STR_ERR__INTERNAL;
  }

  ROUTINGENTRY* rentry = rtable->get_next_entry(ucid, &idx);
  while(rentry != NULL)
  {
    ret = Stop(rentry);
    if((ret & STR_ERR__MASK) != 0) // fatals
    {
      ETG_TRACE_FATAL(("ERR: ERRORS DURING ROUTE STOP! (ucid: 0x%x, cfgid: %d): 0x%x (%s)", (tU16)ucid, (tU16)rentry->cfgid, (tU32)ret, getErrorString(ret)));
      return ret;
    }
    else
    {
      if(ret != 0)
      {
        ETG_TRACE_FATAL(("WRN: WARNINGS DURING ROUTE STOP! (ucid: 0x%x, cfgid: %d): 0x%x (%s)", (tU16)ucid, (tU16)rentry->cfgid, (tU32)ret, getErrorString(ret)));
      }
    }
    ++idx;
    rentry = rtable->get_next_entry(ucid, &idx);
  }
  return STR_ERR__OK;
}

/*******************************************************************************
*
* FUNCTION: fill_cca_info(int ucid, void* ccainfo)
*
* DESCRIPTION: This function generate the cca message and fill this with rentry table infos
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
int fc_streamrouter_tclApp::fill_cca_info(int ucid, void* ccainfo)
{
  if(ccainfo == NULL) return -1;

  if((rtable == NULL) || (pFwdTable == NULL))
  {
    ETG_TRACE_FATAL(("fc_streamrouter_tclApp::fill_cca_info(): NULLPTR ERROR!"));
    NORMAL_M_ASSERT_ALWAYS();
    return -1;
  }

  midw_streamrouterfi_tclMsgAllocateRouteMethodResult* rmsg = (midw_streamrouterfi_tclMsgAllocateRouteMethodResult*)ccainfo;

  rmsg->listOutputDev.strALSADev.clear();
  rmsg->listInputDev.strALSADev.clear();

  rmsg->u16UCID                = (tU16)ucid;
  rmsg->u32STM_MsgBufID        = 0;

  rmsg->u16MLBChannelAddress   = 0;
  rmsg->u16MOSTConnectionLabel = 0;
  rmsg->u8BlkWidth             = 0;
  rmsg->u32SmHandle            = 0;

  int idx = 0;
  ROUTINGENTRY* rentry = rtable->get_next_entry(ucid, &idx);
  while(rentry != 0)
  {
    if(rentry->conlabel != 0)
    {
      rmsg->u16MOSTConnectionLabel = (tU16)rentry->conlabel;
      rmsg->u8BlkWidth = (tU8)rentry->bwidth;
    }

    CONFIGENTRY* cfg = pFwdTable->get_cfg_entry(rentry->cfgid);
    if(cfg != NULL)
    {
      if(cfg->alsadev[0] != 0)
      {
        if(cfg->direction == 0)
        {
          rmsg->listOutputDev.strALSADev.push_back(midw_fi_tclString(cfg->alsadev));
        }
        else
        {
          rmsg->listInputDev.strALSADev.push_back(midw_fi_tclString(cfg->alsadev));
        }
      }
    }
    ++idx;
    rentry = rtable->get_next_entry(ucid, &idx);
  }

  return 0;
}

/*******************************************************************************
*
* FUNCTION: GetStreamProp(tU16 appid, tU16 ucid)
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
int fc_streamrouter_tclApp::GetStreamProp(tU16 appid, tU16 ucid)
{
  (void)appid;
  ETG_TRACE_USR4(("Getting the stream properties for ucid 0x%x ...", (tU16)ucid));
#if 0
  int idx = 0;
  if((rtable == NULL) || (pFwdTable == NULL))
  {
    ETG_TRACE_ERR(("FATAL: Data tables are NULL!"));
    return STR_ERR__INTERNAL;
  }

  ROUTINGENTRY* rentry = rtable->get_next_entry(ucid, &idx);
  while(rentry != 0)
  {
    FG_StreamProp oStreamProp = {0};

    STM_ret ret = STM_sendCommand(rentry->stm_route, FG_CMD_GETSTREAMPROP, &oStreamProp);
    if(STM_ER_OK == ret)
    {
      ETG_TRACE_USR4((">>> Stream properties (ucidsub: 0x%x) >>>", (tU16)rentry->ucidsub));
      ETG_TRACE_USR4(("sample rate: %d, bit rate: %d, format: %d, interleave: %d, channels: %d"
            , oStreamProp.audio.sampleRate, oStreamProp.audio.bitRate, oStreamProp.audio.format, oStreamProp.audio.interleave
            , oStreamProp.audio.channels.count));

      for(tU8 u8Loop = 0; u8Loop < oStreamProp.audio.channels.count; ++u8Loop)
      {
        ETG_TRACE_USR4(("Channel: %d", oStreamProp.audio.channels.ch[u8Loop]));
      }
    }
    else
    {
      ETG_TRACE_USR4(("ERR: STM_sendCommand GETSTREAMPROP: %d", (int)ret));
    }

    ++idx;
    rentry = rtable->get_next_entry(ucid, &idx);
  }
#endif
  return STR_ERR__OK;
}

/*******************************************************************************
*
* FUNCTION: PrintSubroutes(int ucid)
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
int fc_streamrouter_tclApp::PrintSubroutes(int ucid)
{
  if((rtable == NULL) || (pFwdTable == NULL))
  {
    ETG_TRACE_ERR(("FATAL: Data tables are NULL!"));
    return STR_ERR__INTERNAL;
  }

  int idx = 0;
  ucid = rtable->get_next_ucid(ucid);
  while(ucid > 0)
  {
    ETG_TRACE_USR4((">>> Configured subroutes (ucid 0x%x) <<<", ucid));
    ROUTINGENTRY* rentry = rtable->get_next_entry(ucid, &idx);
    while(rentry != NULL)
    {
      CONFIGENTRY* centry = pFwdTable->get_cfg_entry(rentry->cfgid);
      if(centry != NULL)
      {

  #if 0
        ETG_TRACE_USR4(("[ cfgid %02d | appid 0x%04x | ucidsub 0x%04x | msgbufid 0x%08x | connlabel 0x%04x | chnlid 0x%04x | bwidth %02d",
                        (tU8)rentry->cfgid, (tU16)rentry->appid, (tU16)rentry->ucidsub, (tU32)rentry->stm_msgbuf_id, (tU16)rentry->mcsm_desc.ConnectionLabelID,
                        (tU16)rentry->mcsm_desc.MLBChannelID, (tU8)rentry->mcsm_desc.BlockWidth));
  #endif
        ETG_TRACE_USR4(("[ alsa device : \"%s\"", centry->alsadev));
        ETG_TRACE_USR4(("  -"));

      }
      ++idx;
      rentry = rtable->get_next_entry(ucid, &idx);
    }
    idx = 0;
    ucid = rtable->get_next_ucid(ucid);
  }

  return STR_ERR__OK;
}

/*******************************************************************************
*
* FUNCTION: getSTRID(int ucid)
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
tU16 fc_streamrouter_tclApp::getSTRID(int ucid)
{
  //if((pFwdTable == NULL) || (m_poConfigMic == NULL))
  if(pFwdTable == NULL)
  {
    ETG_TRACE_FATAL(("fc_streamrouter_tclApp::getSTRID(): NULLPTR ERROR!"));
    NORMAL_M_ASSERT_ALWAYS();
    return 0;
  }

  int strid = pFwdTable->get_strid(ucid);
  tU16 retval = (tU16)strid;

#if 0
  if(strid == STRID_EOL)
  {
    retval = STRID_MP1TOINT;
  }
#endif

  return retval;
}

/*******************************************************************************
*
* FUNCTION: AllocateMostMic(void)
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
tBool fc_streamrouter_tclApp::AllocateMostMic(void)
{
  tBool ret = true;
//  ETG_TRACE_USR4(("AllocateMostMic... src: %d", (tU8)mostmic_srcnum));
#if 0
  if(m_poClientHandlerMostMic == NULL)
  {
    ETG_TRACE_FATAL(("fc_streamrouter_tclApp::AllocateMostMic(): NULLPTR ERROR!"));
    NORMAL_M_ASSERT_ALWAYS();
    return false;
  }

  ret = (tBool)m_poClientHandlerMostMic->sendAllocate(mostmic_instance, mostmic_srcnum);

  if(ret == true)
  {
    ETG_TRACE_ERR(("Successfully allocated!"));
  }
  else
  {
    ETG_TRACE_ERR(("Failed to allocate!"));
    ret = false;
  }
#endif
  return ret;
}

/*******************************************************************************
*
* FUNCTION: DeallocateMostMic(void)
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
tBool fc_streamrouter_tclApp::DeallocateMostMic(void)
{
  tBool ret = true;
  //ETG_TRACE_USR4(("DeallocateMostMic... src: %d", (tU8)mostmic_srcnum));
#if 0
  if(m_poClientHandlerMostMic == NULL)
  {
    ETG_TRACE_FATAL(("fc_streamrouter_tclApp::DeallocateMostMic(): NULLPTR ERROR!"));
    NORMAL_M_ASSERT_ALWAYS();
    return false;
  }

  ret = (tBool)m_poClientHandlerMostMic->sendDeallocate(mostmic_instance, mostmic_srcnum);
  if(ret == true)
  {
    ETG_TRACE_ERR(("Sent successfully!"));
  }
  else
  {
    ETG_TRACE_ERR(("Failed to send deallocate message!"));
    ret = false;
  }
#endif
  return ret;
}

/*******************************************************************************
*
* FUNCTION: GetMostMicParams(tU8* src_num, tU16* blk_width, tU16 *con_label)
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
tBool fc_streamrouter_tclApp::GetMostMicParams(tU8 src_num, tU16* blk_width, tU16 *con_label, tU8 u8Instance)
{
  (void)src_num;
  (void)blk_width;
  (void)con_label;
  (void)u8Instance;
  tBool bRetVal  =  TRUE;
#if 0
  str_tMostMicConn oMostMicConn;

  if(m_poClientHandlerMostMic == NULL)
  {
    ETG_TRACE_FATAL(("m_poClientHandlerMostMic == NULL"));
    return FALSE;
  }

  if(TRUE == m_poClientHandlerMostMic->bGetMostMicProp(u8Instance, oMostMicConn, src_num))
  {
    if(OSAL_NULL != blk_width)
      *blk_width  =  oMostMicConn.blk_width;

    if(OSAL_NULL != con_label)
      *con_label  =  oMostMicConn.con_label;
  }
  else
  {
    bRetVal  =  FALSE;
  }
#endif
  return bRetVal;
}

/*virtual*/
tVoid fc_streamrouter_tclApp::vOnEvent(OSAL_tEventMask nEvent)
{
  ETG_TRACE_USR4(("fc_streamrouter_tclApp::vOnEvent() - event received %d", nEvent));
  if(m_poStreamRouter != NULL)
  {
    //m_poStreamRouter->vOnEvent(nEvent);
    if((nEvent & STR_EV_NETOFF) || (nEvent & STR_EV_CRITICALUNLOCK) || (nEvent & STR_EV_INICPROTECTED))
    {
      handle_error();
    }
  }
  else
  {
    ETG_TRACE_FATAL(("fc_streamrouter_tclApp::vOnEvent() - event %d received, but service is NULL! ", nEvent));
  }
}

/*******************************************************************************
*
* FUNCTION: vTraceInput_CB(const tUChar* puchData)
*
* DESCRIPTION: Function for handling of the Input Trace
*
* PARAMETER:
*
* RETURNVALUE: None.
*
*******************************************************************************/
#ifdef STREAMROUTER_TRACE_IN
void fc_streamrouter_tclApp::vTraceInput_CB(const tUChar* puchData)
{
  tU8 u8MsgLen  = puchData[0]; //byte 0 -> number of transmitted dat
  tU8 u8MsgCode = 0;
  //tU8 u8MsgData = 0;

  tU32 arg = 0;

  if(u8MsgLen > 0)
  {
    u8MsgCode = puchData[1]; //byte 1 -> start of transmitted data
  }

  if(u8MsgLen > 4)
  {
    arg |= puchData[2] << 24;
    arg |= puchData[3] << 16;
    arg |= puchData[4] << 8;
    arg |= puchData[5];
  }

  //ETG_TRACE_USR4(("vTraceInput_CB() entered. u8MsgCode = 0x%4x; u8MsgData = 0x%4x", u8MsgCode, u8MsgData));

  switch(u8MsgCode)
  {
    case 0:     // GetVersion
      {
        ETG_TRACE_USR4(("Hello2."));
        if(OSAL_NULL != m_poMainAppInstance)
        {
          m_poMainAppInstance->m_poStreamRouter->TTFis_LoopbackCommand(0, arg, STREAMROUTER_C_U16_IFID_TEST);
        }
      }
      break;
    case 2:     // Allocate
      {
        if((OSAL_NULL != m_poMainAppInstance) && (OSAL_NULL != m_poMainAppInstance->m_poStreamRouter))
        {
          m_poMainAppInstance->m_poStreamRouter->TTFis_LoopbackCommand(0, arg, STREAMROUTER_C_U16_IFID_ALLOCATE);
        }
      }
      break;
    case 3:     // Play
      {
        if((OSAL_NULL != m_poMainAppInstance) && (OSAL_NULL != m_poMainAppInstance->m_poStreamRouter))
        {
          m_poMainAppInstance->m_poStreamRouter->TTFis_LoopbackCommand(0, arg, STREAMROUTER_C_U16_IFID_PLAY);

        }
      }
      break;
    case 4:     // Stop
      {
        if((OSAL_NULL != m_poMainAppInstance) && (OSAL_NULL != m_poMainAppInstance->m_poStreamRouter))
        {
          m_poMainAppInstance->m_poStreamRouter->TTFis_LoopbackCommand(0, arg, STREAMROUTER_C_U16_IFID_STOP);
        }
      }
      break;
    case 5:     // Deallocate
      {
        if((OSAL_NULL != m_poMainAppInstance) && (OSAL_NULL != m_poMainAppInstance->m_poStreamRouter))
        {
          m_poMainAppInstance->m_poStreamRouter->TTFis_LoopbackCommand(0, arg, STREAMROUTER_C_U16_IFID_DEALLOCATE);
        }
      }
      break;
    case 7:     // GetStreamProp
      {
        ETG_TRACE_USR4(("Hello."));
        #if 0
        if((OSAL_NULL != m_poMainAppInstance) && (OSAL_NULL != m_poMainAppInstance->m_poStreamRouter))
        {
          m_poMainAppInstance->m_poStreamRouter->TTFis_LoopbackCommand(0, arg, STREAMROUTER_C_U16_IFID_GETSREAMPROP);
        }
        #endif
      }
      break;
    case 8:     // PrintSubroutes
      {
        if((OSAL_NULL != m_poMainAppInstance) && (OSAL_NULL != m_poMainAppInstance->m_poStreamRouter))
        {
          m_poMainAppInstance->m_poStreamRouter->TTFis_LoopbackCommand(0, arg, STREAMROUTER_C_U16_IFID_PRINTCONFSRT);
        }
      }
      break;
    case 0x0b:  // SetMostMicInstance
      {
        ETG_TRACE_USR4(("Hello."));
        //m_poMainAppInstance->mostmic_instance = arg;
      }
      break;
    case 0x0c:  // SetADCPWM
      {
        ETG_TRACE_USR4(("Hello."));
        #if 0
        if((OSAL_NULL != m_poMainAppInstance) && (OSAL_NULL != m_poMainAppInstance->m_poConfigMic))
        {
          ETG_TRACE_USR4(("setting ADC PWM to %d", (tS32)arg));
          m_poMainAppInstance->m_poConfigMic->vSetTimerMicPwm((tS32)arg);
        }
        #endif
      }
      break;
    case 0x0D:  // SetMic1Config
      {
        ETG_TRACE_USR4(("SetMic1Config..."));
#ifdef VARIANT_S_FTR_ENABLE_GM_STR
        if((OSAL_NULL != m_poMainAppInstance) && (OSAL_NULL != m_poMainAppInstance->m_poConfigMic))
        {
          tU8 u8MicConf=(tU8)arg;
          m_poMainAppInstance->m_poConfigMic->mic1cfg = u8MicConf;
          if(m_poMainAppInstance->m_poConfigMic->bConfigureMic1(u8MicConf, m_poMainAppInstance->m_poConfigMic->u8GetMic2cfg()) != true)
          {
            ETG_TRACE_ERR(("ERR: Unable to set EOL gpio configuration for Mic1!"));
          }
        }
        ETG_TRACE_USR4(("Done."));
#endif
      }
      break;
    case 0x0E:  // SetMic2Config
      {
        ETG_TRACE_USR4(("Hello."));
#if 0
        if((OSAL_NULL != m_poMainAppInstance) && (OSAL_NULL != m_poMainAppInstance->m_poConfigMic))
        {
          tU8 u8MicConf=(tU8)arg;
          m_poMainAppInstance->m_poConfigMic->mic2cfg = u8MicConf;
          if(m_poMainAppInstance->m_poConfigMic->bConfigureMic2(u8MicConf, m_poMainAppInstance->m_poConfigMic->u8GetMic1cfg()) != true)
          {
            ETG_TRACE_ERR(("ERR: Unable to set EOL gpio configuration for Mic2!"));
          }
        }
#endif
      }
      break;
    case 0x0f:  // SetMostMicSrcNum
      {
        ETG_TRACE_USR4(("Hello."));
        //m_poMainAppInstance->mostmic_srcnum = arg;
      }
      break;
    case 0x10:  // PrintConfigTable
      {
        ETG_TRACE_USR4(("Hello."));
        #if 0
        if((OSAL_NULL != m_poMainAppInstance) && (OSAL_NULL != m_poMainAppInstance->pFwdTable))
        {
          m_poMainAppInstance->pFwdTable->vPrint();
        }
        #endif
      }
      break;
    case 0x11:  // GetRoutingState
      {
        if((OSAL_NULL != m_poMainAppInstance) && (OSAL_NULL != m_poMainAppInstance->m_poStreamRouter))
        {
          m_poMainAppInstance->m_poStreamRouter->TTFis_LoopbackCommand(0, 0, STREAMROUTER_C_U16_IFID_ROUTINGSTATE);
        }
      }
      break;
    case 0x12:  // GetMicConfig
      {
        ETG_TRACE_USR4(("GetMicConfig..."));
#ifdef VARIANT_S_FTR_ENABLE_GM_STR
        tU8 mic1;
        if((OSAL_NULL != m_poMainAppInstance) && (OSAL_NULL != m_poMainAppInstance->m_poConfigMic))
        {
          mic1 = m_poMainAppInstance->m_poConfigMic->u8GetMic1cfg();
          //mic2 = m_poMainAppInstance->m_poConfigMic->u8GetMic2cfg();
          ETG_TRACE_USR4(("Config Mic:%d ", mic1));
        }
        else
        {
          ETG_TRACE_ERR(("ERR: Get Mic Config, pointer is NULL"));
        }
        ETG_TRACE_USR4(("Done."));
#endif
      }
      break;
    case 0x13:  // GetMicDelay
      {
        ETG_TRACE_USR4(("Hello."));
#if 0
        if((OSAL_NULL != m_poMainAppInstance) && (OSAL_NULL != m_poMainAppInstance->m_poConfigMic))
        {
          tU8 mic1Delay = 0, mic2Delay = 0;
          m_poMainAppInstance->m_poConfigMic->vGetMicDelay(&mic1Delay, &mic2Delay);
          ETG_TRACE_USR4(("Mic1 Delay: %d", mic1Delay));
          ETG_TRACE_USR4(("Mic2 Delay: %d", mic2Delay));
        }
        else
        {
          ETG_TRACE_ERR(("ERR: Get Mic Delay, pointer is NULL"));
        }
#endif
      }
      break;
    case 0x14:  // SetMicDelay
      {
        ETG_TRACE_USR4(("Hello."));
#if 0
        if((OSAL_NULL != m_poMainAppInstance) && (OSAL_NULL != m_poMainAppInstance->m_poConfigMic))
        {
          ETG_TRACE_USR4(("setting EOL Mic delay to %d", (tS32)arg));
          m_poMainAppInstance->m_poConfigMic->vSetMicDelay((tS32)arg);
        }
        else
        {
          ETG_TRACE_ERR(("ERR: Set Mic Delay, pointer is NULL"));
        }
#endif
      }
      break;
    case 0x19:  // DisableErrRec
      {
#if 0
        if(m_poMainAppInstance != NULL)
        {
          m_poMainAppInstance->enabled_errorh_rec = TRUE;
        }
        OSAL_tIODescriptor file = OSAL_IOCreate((tCString)STR_PATH_ERRH_ENABLE_REC, OSAL_EN_READWRITE);
        if(file == OSAL_ERROR)
        {
          ETG_TRACE_USR4(("ERR: Failed to create file!"));
        }
        else
        {
          OSAL_s32IOClose(file);
        }
#endif
      }
      break;
    case 0x20:  // EnableErrRec
      {
#if 0
        if(m_poMainAppInstance != NULL)
        {
          m_poMainAppInstance->enabled_errorh_rec = FALSE;
        }
        OSAL_s32IORemove((tCString)STR_PATH_ERRH_ENABLE_REC);
#endif
      }
      break;
    default:
      break;
  }
}
#endif

void fc_streamrouter_tclApp::handle_error(void)
{
  if((rtable == NULL) || (m_poStreamRouter == NULL)) return;

  tBool routes_found = FALSE;

  int ucid = rtable->get_next_ucid(0);
  while(ucid > 0)
  {
    if(rtable->get_ucid_mostcfg(ucid, NULL, NULL, NULL) == TRUE)
    {
      rtable->set_ucid_kill(ucid, TRUE);
      routes_found = TRUE;
    }
    ucid = rtable->get_next_ucid(ucid);
  }

  if(routes_found == TRUE)
  {
    m_poStreamRouter->sendRoutingStateProperty();
    remove_kill_marked_routes();
    m_poStreamRouter->sendRoutingStateProperty();
    ETG_TRACE_FATAL(("fc_streamrouter_tclApp::handle_error() - MOST ROUTES DEALLOCATED!"));
  }
}

void fc_streamrouter_tclApp::remove_kill_marked_routes(void)
{
  tBool kill;

  if(rtable == NULL)
  {
    ETG_TRACE_FATAL(("fc_streamrouter_tclApp::remove_kill_marked_routes(): NULLPTR ERROR!"));
    NORMAL_M_ASSERT_ALWAYS();
    return;
  }

  int ucid = rtable->get_next_ucid(0);
  while(ucid > 0)
  {
    kill = FALSE;
    if(rtable->get_ucid_routecfg(ucid, NULL, NULL, NULL, &kill) == TRUE)
    {
      if(kill == TRUE) RemoveRoutesPerUCID(ucid);
    }
    ucid = rtable->get_next_ucid(ucid);
  }
}

tCString fc_streamrouter_tclApp::getErrorString(int error_code) const
{
   tCString cszErrorString = OSAL_NULL;

   switch(error_code)
   {
   case STR_ERR__OK:
      {
         cszErrorString = "OK";
      }
      break;
   case STR_ERR__STM_ROUTE_REQ:
      {
         cszErrorString = "STM_ROUTE_REQ";
      }
      break;
   case STR_ERR__STM_ROUTE_FREE:
      {
         cszErrorString = "STM_ROUTE_FREE";
      }
      break;
   case STR_ERR__STM_SET_MLB:
      {
         cszErrorString = "STM_SET_MLB";
      }
      break;
   case STR_ERR__STM_PLAY:
      {
         cszErrorString = "STM_PLAY";
      }
      break;
   case STR_ERR__STM_STOP:
      {
         cszErrorString = "STM_STOP";
      }
      break;
   case STR_ERR__MSCM_CHANNEL_ALLOC:
      {
         cszErrorString = "MSCM_CHANNEL_ALLOC";
      }
      break;
   case STR_ERR__MSCM_CHANNEL_DEALLOC:
      {
         cszErrorString = "MSCM_CHANNEL_DEALLOC";
      }
      break;
   case STR_ERR__INTERNAL:
      {
         cszErrorString = "INTERNAL";
      }
      break;
   case STR_ERR__RESOURCE_TABLE:
      {
         cszErrorString = "RESOURCE_TABLE";
      }
      break;
   case STR_ERR__MOST_MIC_ALLOC:
      {
         cszErrorString = "MOST_MIC_ALLOC";
      }
      break;
   case STR_ERR__MOST_MIC_DEALLOC:
      {
         cszErrorString = "MOST_MIC_DEALLOC";
      }
      break;
   case STR_WRN__STM_FLUSH:
      {
         cszErrorString = "STM_FLUSH";
      }
      break;
   case STR_WRN__STM_STOP:
      {
         cszErrorString = "STM_STOP";
      }
      break;
   case STR_WRN__ROUTE_STATE:
      {
         cszErrorString = "ROUTE_STATE";
      }
      break;
   default:
      {
         cszErrorString = "?";
      }
      break;
   }

   return cszErrorString;
}

int fc_streamrouter_tclApp::mask_btclk_init(void)
{
#ifdef VARIANT_S_FTR_PRJ_VAG
  int fret = 0;
  snd_pcm_t *handle = (snd_pcm_t*)sndcard_handle;
  char *snd_card_str = "AdevBTMasterClock";

  snd_pcm_hw_params_t *hw_params;

  if(handle == NULL)
  {
    int err = 0;
    if((err = snd_pcm_open(&handle, snd_card_str, SND_PCM_STREAM_PLAYBACK, 0)) < 0)
    {
      ETG_TRACE_FATAL(("can not open audio device: %s", snd_strerror(err)));
      fret = 1;
    }
    else
    {
      sndcard_handle = (void*)handle;

      ETG_TRACE_FATAL(("Set hw_params"));
      if((err = snd_pcm_hw_params_malloc(&hw_params)) < 0)
      {
        ETG_TRACE_FATAL(("can not allocate hardware parameter structure (%s)", snd_strerror(err)));
      }

      if((err = snd_pcm_hw_params_any(handle, hw_params)) < 0)
      {
        ETG_TRACE_FATAL(("can not initialize hardware parameter structure (%s)", snd_strerror(err)));
      }

      if((err = snd_pcm_hw_params_set_access(handle, hw_params, SND_PCM_ACCESS_RW_INTERLEAVED)) < 0)
      {
        ETG_TRACE_FATAL(("can not set access type (%s)", snd_strerror (err)));
      }

      if((err = snd_pcm_hw_params_set_format(handle, hw_params, SND_PCM_FORMAT_S16_LE)) < 0)
      {
        ETG_TRACE_FATAL(("can not set sample format (%s)", snd_strerror (err)));
      }

      if((err = snd_pcm_hw_params_set_rate(handle, hw_params, 48000, 0)) < 0)
      {
        ETG_TRACE_FATAL(("can not set sample rate (%s)", snd_strerror (err)));
      }

      if((err = snd_pcm_hw_params_set_channels(handle, hw_params, 1)) < 0)
      {
        ETG_TRACE_FATAL(("can not set channel count (%s)", snd_strerror (err)));
      }

      unsigned int uval = 2;
      int dir = 0;
      //if ((err = snd_pcm_hw_params_set_periods (capture_handle, hw_params, uval, dir)) < 0) {
      if((err = snd_pcm_hw_params_set_periods_min(handle, hw_params, &uval, &dir)) < 0)
      {
        ETG_TRACE_FATAL(("can not set periods (%s)", snd_strerror(err)));
      }

      snd_pcm_uframes_t period_size = 0;
      if((err = snd_pcm_hw_params_get_period_size_max(hw_params, &period_size, &dir)) < 0)
      {
        ETG_TRACE_FATAL(("can not get period_size max (%s)", snd_strerror (err)));
      }

      ETG_TRACE_FATAL(("Set period_size to %d", period_size));
      if((err = snd_pcm_hw_params_set_period_size(handle, hw_params, period_size, 0)) < 0)
      {
        ETG_TRACE_FATAL(("can not set period_size (%s)", snd_strerror(err)));
      }

      if((err = snd_pcm_hw_params(handle, hw_params)) < 0)
      {
        ETG_TRACE_FATAL(("can not set parameters (%s)", snd_strerror(err)));
      }

      snd_pcm_uframes_t buffer_frames;
      if((err = snd_pcm_hw_params_get_buffer_size(hw_params, &buffer_frames)) < 0)
      {
        ETG_TRACE_FATAL(("can not get buffer size (%s)", snd_strerror (err)));
      }

      snd_pcm_hw_params_free(hw_params);

      snd_pcm_sw_params_t *sw_params;
      if((err = snd_pcm_sw_params_malloc(&sw_params)) < 0)
      {
        ETG_TRACE_FATAL(("can not allocate software parameters structure (%s)", snd_strerror(err)));
      }

      if((err = snd_pcm_sw_params_current(handle, sw_params)) < 0)
      {
        ETG_TRACE_FATAL(("can not initialize software parameters structure (%s)", snd_strerror(err)));
      }

      if((err = snd_pcm_sw_params_set_avail_min(handle, sw_params, 0)) < 0)
      {
        ETG_TRACE_FATAL(("can not set start mode (%s)", snd_strerror(err)));
      }

      snd_pcm_uframes_t boundary;
      if((err = snd_pcm_sw_params_get_boundary(sw_params, &boundary)) < 0)
      {
        ETG_TRACE_FATAL(("can not get boundary (%s)", snd_strerror(err)));
      }

      snd_pcm_uframes_t stop_threshold = boundary;
      if((err = snd_pcm_sw_params_set_stop_threshold(handle, sw_params, stop_threshold)) < 0)
      {
        ETG_TRACE_FATAL(("can not set start mode (%s)", snd_strerror(err)));
      }

      if((err = snd_pcm_sw_params_set_start_threshold(handle, sw_params, 1)) < 0)
      {
        ETG_TRACE_FATAL(("can not set start mode (%s)", snd_strerror(err)));
      }

      if((err = snd_pcm_sw_params(handle, sw_params)) < 0)
      {
        ETG_TRACE_FATAL(("can not set software parameters (%s)", snd_strerror(err)));
      }

      if((err = snd_pcm_prepare(handle)) < 0)
      {
        ETG_TRACE_FATAL(("can not prepare audio interface for use (%s)", snd_strerror(err)));
      }

      //signed short *buf;
      //unsigned int BUFSAMPLES = 0;

      signed short buf[256] = {0};
      unsigned int bufsize = 256;
      //Zum starten :
      if((err = snd_pcm_writei(handle, &buf, bufsize)) < 0)
      {
        ETG_TRACE_FATAL(("writei failed (%s)", snd_strerror(err)));
      }
    }
  }
  else
  {
    ETG_TRACE_FATAL(("Masterclock already initialized (dev already opened)!"));
    fret = 1;
  }
  return fret;
#else
    return 0;
#endif
}

int fc_streamrouter_tclApp::mask_btclk_deinit(void)
{
#ifdef VARIANT_S_FTR_PRJ_VAG
  snd_pcm_t *handle = (snd_pcm_t*)sndcard_handle;
  int err = 0;

  if(sndcard_handle != NULL)
  {
    if((err = snd_pcm_close(handle)) < 0)
    {
      ETG_TRACE_FATAL(("can't close audio device %s", snd_strerror(err)));
    }
    else
    {
      ETG_TRACE_FATAL(("success."));
      handle = NULL;
    }
  }
  else
  {
    ETG_TRACE_FATAL(("device was closed already."));
  }
  sndcard_handle = (void*)handle;
#else
  return 0;
#endif
}

tVoid fc_streamrouter_tclApp::mask_start_gstreamer(tS32 t32Variant)
{

  intptr_t var = t32Variant;
  // setup thread to start the gstreamer
  OSAL_trThreadAttribute rThAttr;
  char sThreadName[]   = "GstThread";
  rThAttr.szName       = sThreadName;
  rThAttr.u32Priority  = OSAL_C_U32_THREAD_PRIORITY_NORMAL;
  rThAttr.s32StackSize = 16384;
  rThAttr.pfEntry      = (OSAL_tpfThreadEntry)vStartGstreamer;
  rThAttr.pvArg        = (void*)var;

  _tStartGstreamerThreadID = OSAL_ThreadCreate(&rThAttr);

  if(OSAL_ERROR == _tStartGstreamerThreadID)
  {
    ETG_TRACE_FATAL(("Start Gstreamer Thread creation failed!"))
    return;
  }

  if(OSAL_s32ThreadActivate(_tStartGstreamerThreadID) == OSAL_ERROR)
  {
    ETG_TRACE_FATAL(("Start Gstreamer Thread activation failed!"))
    return;
  }
  else
  {
    ETG_TRACE_USR4(( "Start Gstreamer Thread created succesfully" ));
  }

}

tVoid fc_streamrouter_tclApp::vStartGstreamer(void* arg)
{
    ETG_TRACE_USR4(( "Start arecord" ));
    intptr_t t32Variant = (intptr_t)arg;

    if (t32Variant == STR_CFGMASK_GST) //if (STRID_TESTMICINTOMOST == rentry->strid)
    {
        ETG_TRACE_USR4(( "Start arecord for GM" ));
        //ETG_TRACE_USR4(( "Variant = %d", t32Variant));
        system("arecord -D AdevMicroInStereo -c 2 -f s16_le -r 16000 | aplay -D AdevMicroOutStereo -c 2 -f s16_le -r 48000");  // start arecord audiotest signal

        //ETG_TRACE_USR4(( "Start gst-launch" ));
        //system("gst-launch -v alsasrc device= AdevMicroInStereo ! audioconvert ! audioresample ! audio/x-raw-int, rate=16000 ! alsasink device= AdevMicroOutStereo");  // start gst-launch audiotest signal
        //system("gst-launch -v alsasrc device=AdevMicroInStereo ! audioconvert ! wavenc ! decodebin ! alsasink device=AdevMicroOutStereo");  // start gst-launch audiotest signal
    }
    else if (t32Variant == STR_CFGMASK_GST_SUSUKI) //if (STRCFG_STRMICROIN == rentry->strid)
    {
        ETG_TRACE_USR4(( "Start arecord for SUSUKI" ));
        //ETG_TRACE_USR4(( "Variant = %d", t32Variant));
        system("arecord -D AdevMicro1In -c 1 -f s16_le -r 16000 | aplay -D AdevEnt1Out -c 2 -f s16_le -r 48000");  // start arecord audiotest signal for SUSUKI
    }

}

tVoid fc_streamrouter_tclApp::mask_stop_gstreamer()
{
  ETG_TRACE_USR4(("Stop Gstreamer Thread"));
  if(OSAL_NULL != _tStartGstreamerThreadID)
  {
    tS32 s32ReturnValue = OSAL_s32ThreadDelete( _tStartGstreamerThreadID );
    _tStartGstreamerThreadID = OSAL_NULL;
    if(s32ReturnValue == OSAL_OK)
    {
      ETG_TRACE_USR4(("StartGstreamerThreadID: Thread Deletion Successful"));
    }
    else
    {
      ETG_TRACE_USR4(("StartGstreamerThreadID: Thread Deletion failed"));
    }
  }
  else
  {
    ETG_TRACE_USR4(("Invalid Thread ID: StartGstreamerThreadID"));
  }

  //Kill arecord process
  system("killall arecord");

  //Kill gst-launch process
  //system("killall gst-launch-0.10");
}


