/******************************************************************************
 * FILE        : ecnr-sse-handler.c
 * PROJECT     : Gen3 and Gen4
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2013 RBCM GMBH
 * HISTORY     :
 * Date        | Author                 | Modification
 * 19.08.2013  | Patrick Rey            | initial version
 *             |                        | SSE 3.X engine feature handling
 * 29.11.2013  | Patrick Rey            | configuration set update
 * 08.03.2019  | Ranjit Susal(RBEI/ECO) | SSE 4.X engine feature handling
 *****************************************************************************/

#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX

#ifdef D_ECNR_LIB_SSE_3_17
#include "sse.h"
#include "sse_err.h"
#endif

#ifdef D_ECNR_LIB_SSE_4_03
#include "sse.h"
#include "sse_param.h"
#endif


#include "ecnr-types.h"
#include "ecnr-error.h"
#include "ecnr-common-defs.h"
#include "ecnr-object.h"
#include "ecnr-service.h"
#include "ecnr-alsa.h"
#include "ecnr-engine-handler.h"
#include "ecnr-sse-handler.h"

#define ECNR_S_IMPORT_INTERFACE_DP
#include "ecnr-datapool-default.h"
#include "ecnr-configuration-data.h"


/*******************************************************************************
              VERSION COMPILER
*******************************************************************************/
//#define SSE_TEST_STUB

/*******************************************************************************
              DEFINES
*******************************************************************************/
//#define SSE_FRAMEZISE_SR_8000             64 //prio2 lint fix Info 715
//#define SSE_FRAMEZISE_SR_16000            128//prio2 lint fix Info 715

#ifdef D_ECNR_LIB_SSE_3_17
#define ECNR_MAX_STRING_SIZE  SSE_MAX_STRING_SIZE
#elif  D_ECNR_LIB_SSE_4_03
#define ECNR_MAX_STRING_SIZE  200
#endif

#define SSE_MAX_LOG_FILE_SIZE             10000
#define LOG_BSD_HEADER_INFO_FILE "/var/opt/bosch/dynamic/audiomanager/ecnr/ecnr_sse_bsd_info.txt"

/* Needed for Trace */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "ecnr-trace.h"
#include "etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ECNR_ENGINE_BE
#include "trcGenProj/Header/ecnr-sse-handler.c.trc.h"
#endif

/*******************************************************************************
              Constant definition
*******************************************************************************/

const short int DEFAULT_MIC_CHANNEL_CNT = 1;       /* Microphone input/output channel count */
const short int DEFAULT_RECV_CHANNEL_CNT = 1;      /* Receive input/output channel count */
const short int DEFAULT_RECV_CHANNEL_CAR_PLAY_CNT = 0;      /* Receive input/output channel count */
const short int DEFAULT_REF_CHANNEL_CNT = 0;       /* Reference input channel count */

const unsigned int DEFAULT_EXT_MEM_SIZE = 70000;   /* Size of external SSE memory */

/*--- SCD (SSE Configuration Data): 1xMic, 1xRef, 1xRcv (the SCD formate is based on
      32 bit words. We ensure 32 bit words by using int32_t as data type.         */
/*const int32_t ScdExample[] = {
   0x0001efff, 0x00010009, 0x2bd47b40, 0x00010004, 0x00000000, 0x00000008, 0xc8b2442f,
   0x00000000, 0x00000000, 0x00000000, 0x00000006, 0x8af35bf8, 0x00000000, 0x6e544a82,
   0x00000001, 0xbd7f967b, 0x00000001 };
*/

/*******************************************************************************
              STRUCTURE DEFINITION
*******************************************************************************/
typedef struct
{
    int             SSEInitOk;
    tsseMain*       pMain;              /* pointer to SSE main data structure             */
    tecnrConfigData ConfigData;         /* configuration data structure                   */
    tsseExtMem      SSEMemTab;          /* SSE memory table for using external memory     */
    unsigned int    iArrayLen;          /* length of SSE version array                    */
    const int*      pVersionArray;      /* SSE version array                              */
    const char*     pVersionString;     /* SSE version string                             */
    const char*     pVersionComment;    /* SSE version comment                            */
    char*           BSDInfo;            //[SSE_MAX_BSD_INFO_STRING_SIZE];// BSD Info
    tI32            ConfigCRC;          /* CRC value of loaded .scd configuration file : available  from SSE 4.x lib onwards */
    tecnrTuneIO     sSseDebugBuffer;    /* Debug data from/to Tuning Tool                 */
} tSseState;


typedef struct
{
    const gchar* ecnr_sse_param_name;
          guint  ecnr_sse_param_id;
           gint  ecnr_sse_param_channel;
          guint  ecnr_sse_param_size;
           void* ecnr_sse_param_ref;
            int  ecnr_sse_param_val;
}ecnr_sse_parameter;


#define SSE_PARM(pname, name_id, ch, size, ref, val) \
  {                                    \
    .ecnr_sse_param_name    = pname,   \
    .ecnr_sse_param_id      = name_id, \
    .ecnr_sse_param_channel = ch,      \
    .ecnr_sse_param_size    = size,    \
    .ecnr_sse_param_ref     = ref,     \
    .ecnr_sse_param_val     = val,     \
  }

#define SSE_PARM_I8_PER_VAL(name, val) \
  SSE_PARM(#name, name, 0, sizeof(gint8), NULL, (int)val)
#define SSE_PARM_UI16_PER_VAL(name, val) \
  SSE_PARM(#name, name, 0, sizeof(guint16), NULL, (int)val)
#define SSE_PARM_I32_PER_VAL(name, val) \
  SSE_PARM(#name, name, 0, sizeof(gint32), NULL, (int)val)

#define SSE_PARM_UI32_PER_VAL(name, val) \
  SSE_PARM(#name, name, 0, sizeof(guint32), NULL, (int)val)

//Below macro's are unused in the source file due to that observed lint errors.
//To solve lint errors below Macro's are commented. In future if below macro's are usable, please un-comment it and use it.
//lint fix "Info 750"
/*
//#define SSE_PARM_I8_PER_REF(name, ref, size) \
  SSE_PARM(#name, name, 0, (size * sizeof(gint8)), (void*)ref, 0)
//#define SSE_PARM_UI16_PER_REF(name, ref, size) \
  SSE_PARM(#name, name, 0, (size * sizeof(guint16)), (void*)ref, 0)
//#define SSE_PARM_I32_PER_REF(name, ref, size) \
  SSE_PARM(#name, name, 0, (size * sizeof(gint32)), (void*)ref, 0)
*/



/**********************************************
 * SSE TEST FUNCTIONS - TOP
 ***********************************************/
//#define D_SSE_TEST_PERFORMANCE
#ifdef D_SSE_TEST_PERFORMANCE

#define D_NB_TEST_STEP_MAX      100

typedef struct _sse_time_test
{
    struct timespec fct_call_entry_time;
    struct timespec fct_call_exit_time;
 /*   guint32 fct_call_exec_time; */ /* remove comment when it is used*/
}sse_time_test;


static sse_time_test* ptstub = (sse_time_test*)NULL;
static gint ecnr_sse_test_next_step = 0;
static gint sse_test_nb_step = 0;
static GString *test_fct_name;


static void ecnr_sse_handler_performance_test_init(gchar* fct_name_to_test, gint nb_test_step)
{

    if(fct_name_to_test)
    {
        pr_message("TEST ENTER for function %s with %d step\n", fct_name_to_test, nb_test_step);
    }

    if(!ptstub)
    {
        if(!nb_test_step)
            sse_test_nb_step = D_NB_TEST_STEP_MAX;
        else
            sse_test_nb_step = nb_test_step;

        ptstub = g_malloc0( sse_test_nb_step * sizeof(sse_time_test));
    }

    test_fct_name = g_string_new (NULL);
    if(fct_name_to_test)
        g_string_append(test_fct_name, fct_name_to_test);
    ecnr_sse_test_next_step = 0;

    return;
}

static void ecnr_sse_handler_performance_test_terminate(void)
{
    sse_time_test* pd = (sse_time_test*)NULL;
    gint i = 0;
    guint32 exec_time = 0;

    if(test_fct_name->str)
        pr_message("TEST EXIT for function %s with %d step\n", test_fct_name->str, sse_test_nb_step);

    /* evaluate all result and write in a file */
    if(!ptstub   || !sse_test_nb_step)
    {
        pr_message("No Result available - 1 \n");
        return;
    }

    for(i = 0; i < sse_test_nb_step; i++, pd++)
    {

        if(i == ecnr_sse_test_next_step)
        {
            pr_message("terminate test with %d step \n", ecnr_sse_test_next_step);
            break;
        }
        pd = &ptstub[i];

        if(pd)
        {
            pr_message("%d %s call ENTRY time : [%05u,%lu]\n", (i+1), (test_fct_name->str)? test_fct_name->str :"NULL", (unsigned int)pd->fct_call_entry_time.tv_sec, (unsigned long int)pd->fct_call_entry_time.tv_nsec);
            pr_message("%d %s call EXIT time : [%05u,%lu]\n", (i+1), (test_fct_name->str)? test_fct_name->str :"NULL", (unsigned int)pd->fct_call_exit_time.tv_sec, (unsigned long int)pd->fct_call_exit_time.tv_nsec);


            if(pd->fct_call_entry_time.tv_sec == pd->fct_call_exit_time.tv_sec)
            {
                exec_time = pd->fct_call_exit_time.tv_nsec - pd->fct_call_entry_time.tv_nsec;
            }
            else /* next second*/
            {
                exec_time = (1000000000 - pd->fct_call_entry_time.tv_nsec) + pd->fct_call_exit_time.tv_nsec;
            }
            /* calculate execution time */
            pr_message("%d %s call EXEC time : [%lu]\n", (i+1), (test_fct_name->str)? test_fct_name->str :"NULL", exec_time);


        }
        else
            pr_message("No Result available - 2\n");
    }

    g_string_free(test_fct_name, TRUE);
    if(ptstub)
    {
        g_free(ptstub);
        ptstub = NULL;
    }

    return;
}

inline static void ecnr_sse_handler_performance_start_next_test_step(void)
{

    if(!ptstub  || (ecnr_sse_test_next_step >= sse_test_nb_step))
        return;

    sse_time_test* p = &ptstub[ecnr_sse_test_next_step];

    clock_gettime(CLOCK_MONOTONIC, &p->fct_call_entry_time);
    return;
}

inline static void ecnr_sse_handler_performance_stop_current_test_step(void)
{
    if(!ptstub  || (ecnr_sse_test_next_step >= sse_test_nb_step))
        return;

    sse_time_test* p = &ptstub[ecnr_sse_test_next_step++];

    clock_gettime(CLOCK_MONOTONIC, &p->fct_call_exit_time);

    return;
}

#endif

 /**********************************************
 * SSE TEST FUNCTIONS - END
 ***********************************************/

/*******************************************************************************
*
* FUNCTION: ecnr_sse_configuration_set_aa_vr_wb_pre
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static int ecnr_sse_load_configuration_set(ecnr_sse_parameter* pparam, gint param_nb)
{
    ecnr_sse_parameter* pdata = (ecnr_sse_parameter*)NULL;
    int i = 0;
    int err = ECNR_ERR_OK;

    ETG_TRACE_USR2(("[ecnr_sse_load_configuration_set]: ENTERED"));

    for (pdata = pparam;  i < (int)param_nb; i++, pdata++)
    {
        if (pdata->ecnr_sse_param_id)
        {
            if(pdata->ecnr_sse_param_ref)
            {
                err = ecnr_sse_handler_set_data(pdata->ecnr_sse_param_id,
                                  (unsigned int)pdata->ecnr_sse_param_channel,
                                                pdata->ecnr_sse_param_size,
                                                pdata->ecnr_sse_param_ref);

                int* value = (int*)(pdata->ecnr_sse_param_ref);

                if(err == ECNR_ERR_OK)
                {
                    pr_message("succefull set of param(%s) with value(%d)\n",  pdata->ecnr_sse_param_name, *value);
                    ETG_TRACE_USR2(("[ecnr_sse_load_configuration_set]: succefull set of value(%d) to the parameter->%s",  *value, pdata->ecnr_sse_param_name));
                }
                else
                {
                    pr_warning("fails to set param(%s) with value(%d)\n",  pdata->ecnr_sse_param_name, *value);
                }
            }
            else
            {
                err = ecnr_sse_handler_set_data
                      (
                                pdata->ecnr_sse_param_id,
                  (unsigned int)pdata->ecnr_sse_param_channel,
                                pdata->ecnr_sse_param_size,
                        (void*)&pdata->ecnr_sse_param_val
                       );

                if(err == ECNR_ERR_OK)
                {
                    pr_message("successful set of param(%s) with value(%d)\n",  pdata->ecnr_sse_param_name, pdata->ecnr_sse_param_val);
                    ETG_TRACE_USR2(("[ecnr_sse_load_configuration_set]: successful set of value(%d) to the parameter->%s",  pdata->ecnr_sse_param_val, pdata->ecnr_sse_param_name));
                }
                else
                    pr_warning("fails to set param(%s) with value(%d)\n",  pdata->ecnr_sse_param_name, pdata->ecnr_sse_param_val);
            }
        }
    }

    ETG_TRACE_USR2(("[ecnr_sse_load_configuration_set]: EXIT"));
    return  err;
}

/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/

static                tSseState  m_sseState;
static       unsigned     short  m_CurrConfigId = 0;
static const unsigned       int  SSE_C_EXT_MEM_SIZE = 800000; // 400000; //200000;
static                 gboolean  m_ecnr_sse_handler_performance_enable = FALSE;

#ifdef D_KDS_SUPPORT
static const unsigned char        InternalMajorVersion = 2;   // Used to determine compatibility with BSD in KDS, see InternalMajorVersion in KDSDataHeader
static const unsigned short       InternalMinorVersion = 1;   // Used to determine compatibility with BSD in KDS, see InternalMinorVersion in KDSDataHeader
#endif

#ifdef D_ECNR_LIB_SSE_3_17
static short            SSE_NRFloor = 0;        // NRFloor value
static const char       NoBSDInfoStr[] = "BSD info not available";
#endif

#ifdef D_ECNR_LIB_SSE_4_03
static const char       NoSCDComment[] = "SCD Comment not available";
#endif


/*******************************************************************************
              STATIC FUNCTION  DECLARATIONS
*******************************************************************************/
#ifdef SSE_TEST_STUB
#include "ecnr-sse-handler-test-stub.c"

#define sseProcess ecnr_sse_handler_test_stub_sseProcess

#endif



/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_log_bsd_header_info
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void ecnr_sse_handler_log_bsd_header_info
(
    unsigned short              data_set_id,
    const struct EcnrConfig     *pdata
) //void
{
    FILE        *fd_info = NULL;   /* pointer to config data file */
    const char  *info_loger = LOG_BSD_HEADER_INFO_FILE;
    long        size;
    int     ret;

    pr_debug("ENTER info loger \"%s\"\n", info_loger);

    /* File name is empty string or NULL pointer means we do not have a
     * config file (not really an error)
    */


    /* test file length*/
    fd_info = fopen (info_loger,"rb");

    if (NULL != fd_info)
    {
        fseek (fd_info, 0, SEEK_END);   // non-portable
        size=ftell (fd_info);
        fclose (fd_info);
    }
    else
    {
        pr_debug("no file %s was found\n", info_loger);
        size = 0;
    }

    if(size > SSE_MAX_LOG_FILE_SIZE)
    {
        pr_debug("remove the file: %s with size %ld\n", info_loger, size);
        ret = remove(info_loger);
        if (ret != ECNR_ERR_OK)
        {
            pr_debug("ecnr_sse_bsd_info.txt removal failed = %d", ret);
        }
    }
    else
        pr_debug("the current size of the file: %s is %ld", info_loger, size);

    /* open file in write mode and position the stream pointer at the end of the file*/
    fd_info = fopen(info_loger, "a+");
    if(NULL == fd_info)
    {
        pr_warning("info loger %s open failed", info_loger);
        return;
    }

    /*set up new header
        ****************  load new ecnr configuration at    ******************
        * - ecnr_configuration_id       :
        * - Datapool_id                 :
        * - loaded bsd file with CRC    :
        * - bsd file name               :
        * - opeth file name             :
        ***********************************************************************
    */

    if(pdata)
    {
        struct timespec ts;

        clock_gettime(CLOCK_MONOTONIC, &ts);

        fprintf (fd_info,"****************  load new data_set_id_%d at time(s) [%05u,%.9lu]\n",data_set_id ,(unsigned int)ts.tv_sec,(unsigned long int)ts.tv_nsec);
        fprintf (fd_info,"* - Info details              :%s\n","the configuration was loaded from the data pool" );
        fprintf (fd_info,"* - ecnr_configuration_id     :%d\n",pdata->configId);
        fprintf (fd_info,"* - datapool_id               :%d\n",pdata->EcnrdPBaseKey);
#ifdef D_ECNR_LIB_SSE_3_17
        fprintf (fd_info,"* - bsd file name             :%s\n",pdata->bsd_str?pdata->bsd_str:"NULL");
        fprintf (fd_info,"* - opeth file name           :%s\n",pdata->opeth_str?pdata->opeth_str:"NULL");
        fprintf (fd_info,"* - loaded bsd file with CRC  :%s\n",m_sseState.BSDInfo?m_sseState.BSDInfo:"NULL");
#elif  D_ECNR_LIB_SSE_4_03
        fprintf (fd_info,"* - SCD file name             :%s\n",pdata->bsd_str?pdata->bsd_str:"NULL");
        fprintf (fd_info,"* - opeth file name           :%s\n",pdata->opeth_str?pdata->opeth_str:"NULL");
        fprintf (fd_info,"* - SCD file CRC              :%x\n",m_sseState.ConfigCRC);
        fprintf (fd_info,"* - SCD comments              :%s\n",m_sseState.BSDInfo?m_sseState.BSDInfo:"NULL");
#endif
        fprintf (fd_info,"*/\n");
        fclose (fd_info);
    }
    else
    {
        pr_debug("no loger info available\n");
        fclose (fd_info);
    }

    pr_debug("EXIT");
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_log_bsd_default_data_set_info
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void ecnr_sse_handler_log_bsd_default_data_set_info
(
    unsigned short              data_set_id,
    const struct EcnrConfig     *pdata
) //void
{
    FILE        *fd_info = NULL;   /* pointer to config data file */
    const char  *info_loger = LOG_BSD_HEADER_INFO_FILE;
    long        size;
    int     ret;
    pr_debug("ENTER info loger \"%s\"\n", info_loger);

    /* File name is empty string or NULL pointer means we do not have a
     * config file (not really an error)
    */


    /* test file length*/
    fd_info = fopen (info_loger,"rb");

    if (NULL != fd_info)
    {
        fseek (fd_info, 0, SEEK_END);   // non-portable
        size=ftell (fd_info);
        fclose (fd_info);
    }
    else
    {
        pr_debug("no file %s was found\n", info_loger);
        size = 0;
    }

    if(size > SSE_MAX_LOG_FILE_SIZE)
    {
        pr_debug("remove the file: %s with size %ld\n", info_loger, size);
        ret = remove(info_loger);
        if (ret != ECNR_ERR_OK)
        {
            pr_debug("ecnr_sse_bsd_info unsuccessful = %d", ret);
        }
    }
    else
        pr_debug("the current size of the file: %s is %ld", info_loger, size);

    /* open file in write mode and position the stream pointer at the end of the file*/
    fd_info = fopen(info_loger, "a+");
    if(NULL == fd_info)
    {
        pr_warning("info loger %s open failed", info_loger);
        return;
    }

    /*set up new header
        ****************  load new ecnr configuration at    ******************
        * - ecnr_configuration_id       :
        * - Datapool_id                 : "default dataset"
        * - loaded bsd file with CRC    :
        * - bsd file name               :
        * - opeth file name             : "undefined"
        ***********************************************************************
    */

    if(pdata)
    {
        struct timespec ts;

        clock_gettime(CLOCK_MONOTONIC, &ts);

        fprintf (fd_info,"****************  load default data_set_id_%d at time(s) [%05u,%.9lu]\n",data_set_id ,(unsigned int)ts.tv_sec,(unsigned long int)ts.tv_nsec);
        fprintf (fd_info,"* - Info details              :%s\n","the default data set was loaded" );
        fprintf (fd_info,"* - ecnr_configuration_id     :%d\n",pdata->configId);
        fprintf (fd_info,"* - datapool_id               :%d\n",pdata->EcnrdPBaseKey);
#ifdef D_ECNR_LIB_SSE_3_17
        fprintf (fd_info,"* - bsd file name             :%s\n",pdata->bsd_str?pdata->bsd_str:"NULL");
        fprintf (fd_info,"* - opeth file name           :%s\n",pdata->opeth_str?pdata->opeth_str:"NULL");
        fprintf (fd_info,"* - loaded bsd file with CRC  :%s\n",m_sseState.BSDInfo?m_sseState.BSDInfo:"NULL");
#elif  D_ECNR_LIB_SSE_4_03
        fprintf (fd_info,"* - scd file name             :%s\n",pdata->bsd_str?pdata->bsd_str:"NULL");
        fprintf (fd_info,"* - opeth file name           :%s\n",pdata->opeth_str?pdata->opeth_str:"NULL");
        fprintf (fd_info,"* - SCD file CRC              :%x\n",m_sseState.ConfigCRC);
        fprintf (fd_info,"* - SCD comments              :%s\n",m_sseState.BSDInfo?m_sseState.BSDInfo:"NULL");
#endif
        fprintf (fd_info,"*/\n");
        fclose (fd_info);
    }
    else
    {
        pr_debug("no loger info available\n");
        fclose (fd_info);
    }

    pr_debug("EXIT");
}


/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_log_new_data_pool_info
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
/*static*/ void ecnr_sse_handler_log_new_data_pool_info
(
    unsigned short              data_set_id,
    const struct EcnrConfig     *pdata
) //void
{
    FILE        *fd_info = NULL;   /* pointer to config data file */
    const char  *info_loger = LOG_BSD_HEADER_INFO_FILE;
    long        size;
    int      ret;
    pr_debug("ENTER info loger \"%s\"\n", info_loger);

    /* File name is empty string or NULL pointer means we do not have a
     * config file (not really an error)
    */


    /* test file length*/
    fd_info = fopen (info_loger,"rb");

    if (NULL != fd_info)
    {
        fseek (fd_info, 0, SEEK_END);   // non-portable
        size=ftell (fd_info);
        fclose (fd_info);
    }
    else
    {
        pr_debug("no file %s was found\n", info_loger);
        size = 0;
    }

    if(size > SSE_MAX_LOG_FILE_SIZE)
    {
        pr_debug("remove the file: %s with size %ld\n", info_loger, size);
        ret = remove(info_loger);
        if (ret != ECNR_ERR_OK)
        {
            pr_debug("ecnr_sse_bsd_info.txt removal failed = %d", ret);
        }
    }
    else
        pr_debug("the current size of the file: %s is %ld", info_loger, size);

    /* open file in write mode and position the stream pointer at the end of the file*/
    fd_info = fopen(info_loger, "a+");
    if(NULL == fd_info)
    {
        pr_warning("info loger %s open failed", info_loger);
        return;
    }

    /*set up new header
        ****************  load new ecnr configuration at    ******************
        * - ecnr_configuration_id       :
        * - Datapool_id                 :
        * - loaded bsd file with CRC    :
        * - bsd file name               :
        * - opeth file name             : "undefined"
        ***********************************************************************
    */

    if(pdata)
    {
        struct timespec ts;

        clock_gettime(CLOCK_MONOTONIC, &ts);

        fprintf (fd_info,"****************  load new data_pool data_set_id_%d at time(s) [%05u,%.9lu]\n",data_set_id ,(unsigned int)ts.tv_sec,(unsigned long int)ts.tv_nsec);
        fprintf (fd_info,"* - Info details              :%s\n","a new data pool content was loaded via ITC request" );
        fprintf (fd_info,"* - ecnr_configuration_id     :%d\n",pdata->configId);
        fprintf (fd_info,"* - datapool_id               :%d\n",pdata->EcnrdPBaseKey);
        fprintf (fd_info,"* - bsd file name             :%s\n",pdata->bsd_str?pdata->bsd_str:"NULL");
        fprintf (fd_info,"* - opeth file name           :%s\n",pdata->opeth_str?pdata->opeth_str:"NULL");
        fprintf (fd_info,"* - loaded bsd file with CRC  :%s\n","available only after ecnrInitialize call ");
        fprintf (fd_info,"*/\n");
        fclose (fd_info);
    }
    else
    {
        pr_debug("no loger info available\n");
        fclose (fd_info);
    }

    pr_debug("EXIT");
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_log_config_data
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static int ecnr_sse_handler_log_config_data
(
    unsigned short              data_set_id,
    const struct EcnrConfig     *pdata,
    ecnr_log_type               log_type
) //void)
{
    int            err = SSE_ERR_OK;
    unsigned int   paramsize = 0;
    short          nrfloor;
    char*          pt =(char*)NULL;

    pr_debug("Entered \n");


    m_sseState.ConfigCRC=0;

    err = ecnr_sse_handler_get_config_data_crc(&m_sseState.ConfigCRC);

    if(SSE_ERR_OK!=err)
    {
        pr_warning("!! get SSE config data CRC returned with error -> %x !! \n",err);
        err = SSE_ERR_OK;
    }

#ifdef D_ECNR_LIB_SSE_3_17
    /* Get BSD Info - might be needed for traces later
     * It does not use err, because failure of this call won't influence the library's functionality in any way
     */

    err = ecnr_sse_handler_get_data(sse_BSDInfoStr, 0, &paramsize, (void *)NULL);
    pr_debug("retrieved paramsize = %d, param = %d\n", paramsize, sse_BSDInfoStr );

    m_sseState.BSDInfo = g_malloc0(paramsize);
    pt = (char*)g_malloc0(paramsize);
    if(pt)//m_sseState.BSDInfo)
    {
        m_sseState.BSDInfo = pt;

        err = ecnr_sse_handler_get_data(sse_BSDInfoStr, 0, &paramsize, (void *)m_sseState.BSDInfo);
        if(err == SSE_ERR_OK)
        {
            pr_debug("retrieved BSD info with size = %d\n", paramsize);
        }
        else
        {
            strncpy(m_sseState.BSDInfo, NoBSDInfoStr, sizeof(NoBSDInfoStr));
            pr_debug("retrieve SSE info failed paramsize = %d, param = %d", paramsize, sse_BSDInfoStr );
        }
    }
#elif  D_ECNR_LIB_SSE_4_03
    /* Get SCD Comment - might be needed for traces later
     * It does not use err, because failure of this call won't influence the library's functionality in any way
     */
    paramsize=ECNR_MAX_CONFIG_COMMENT_LENGTH;

    m_sseState.BSDInfo = g_malloc0(paramsize);
    pt = (char*)g_malloc0(paramsize);
    if(pt)
    {
        m_sseState.BSDInfo = pt;

        err = ecnr_sse_handler_get_data(sse_ScdComment, 0, &paramsize, (void *)m_sseState.BSDInfo);
        if(err == SSE_ERR_OK)
        {
            pr_debug("retrieved SCD info with size = %d\n", paramsize);
        }
        else
        {
            strncpy(m_sseState.BSDInfo, NoSCDComment, sizeof(NoSCDComment));
            pr_debug("retrieve SCD comment failed parameter size = %d, param = %d", paramsize, sse_ScdComment );
        }
    }
#endif

#ifdef D_ECNR_LIB_SSE_3_17
    /* Read NRFloor value - it is used to turn NR on and off
     * If this fails, it could be because NR is disabled in the BSD which is not really an issue.
     * Therefore the error code is ignored
     */
    paramsize = sizeof(nrfloor);

    err = ecnr_sse_handler_get_data(sse_NRFloor, 0, &paramsize, (void *)&nrfloor);
    if(err == SSE_ERR_OK)
        SSE_NRFloor = nrfloor;
    else
        SSE_NRFloor = 0;
#elif  D_ECNR_LIB_SSE_4_03
      /* NR floor could not be able to retrieve in D_ECNR_LIB_SSE_4_03 */
    (void)nrfloor;
#endif

    /* log bsd file header info */
    switch(log_type)
    {
        case ecnr_log_type_init_from_dp:
            ecnr_sse_handler_log_bsd_header_info(data_set_id, pdata);
            break;
        case ecnr_log_type_init_from_default:
            ecnr_sse_handler_log_bsd_default_data_set_info(data_set_id, pdata);
            break;
        case ecnr_log_type_load_new_dp:
            ecnr_sse_handler_log_new_data_pool_info(data_set_id, pdata);
            break;
        default:
            break;
    }

    pr_debug("EXIT with err = %d \n", err);
    return err;
}


/*******************************************************************************
              FUNCTION  DECLARATION
*******************************************************************************/

/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_process_audio
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/

int ecnr_sse_handler_process_audio(stecnr_alsa_state* pstate)
{
    int err = 0;
#if   D_ECNR_LIB_SSE_3_17
    short int *ptemp = (short int *)NULL;
    guint32 ElemCnt = 0;
#endif


    //pr_debug("entered");
    ETG_TRACE_USR4(("[ecnr_sse_handler_process_audio]: ENTERED"));
    if(m_sseState.pMain)
    {
#ifdef D_SSE_TEST_PERFORMANCE
        ecnr_sse_handler_performance_start_next_test_step();
#endif

#if   D_ECNR_LIB_SSE_3_17
        if(pstate->ecnr_enable_asr_preprocessing)
        {
            ptemp = *pstate->MicOut[0];
            *pstate->MicOut[0] = pstate->AsrPrepare;
        }
        else
        {
            /*do nothing */
        }
#endif

        if(m_ecnr_sse_handler_performance_enable)
        {
            struct timespec ts_call, ts_ret;
            glong exec_time = 0;

            clock_gettime(CLOCK_MONOTONIC, &ts_call);
            /*------ signal processing function of the SSE module -*/
#if   D_ECNR_LIB_SSE_3_17
            err = sseProcess
                  (
                       m_sseState.pMain
                      ,*pstate->MicIn
                      ,*pstate->RecvIn
                      ,*pstate->RefIn
                      ,*pstate->MicOut
                      ,*pstate->RecvOut
                      ,NULL
                   );
#elif D_ECNR_LIB_SSE_4_03
            tsseAudioIO  audioIO;
            memset( &audioIO, 0, sizeof(tsseAudioIO) );

            audioIO.AudioBufferSampleCnt = pstate->ecnr_frame_shift_out;
            audioIO.MicInBufferCnt   = pstate->ecnr_mic_in_cnt;
            audioIO.MicProcBufferCnt = pstate->ecnr_mic_proc_cnt;
            audioIO.RcvInBufferCnt   = pstate->ecnr_recv_in_cnt;
            audioIO.RcvProcBufferCnt = pstate->ecnr_recv_in_cnt;
            audioIO.RefInBufferCnt   = pstate->ecnr_ref_in_cnt;
            audioIO.MicInBuffers     = *pstate->MicIn;
            audioIO.MicProcBuffers   = *pstate->MicOut;
            audioIO.RcvInBuffers     = *pstate->RecvIn;
            audioIO.RcvProcBuffers   = *pstate->RecvOut;
            audioIO.RefInBuffers     = *pstate->RefIn;
            err = sseProcess
                  (
                      m_sseState.pMain
                     ,audioIO
                     ,NULL
                  );
#endif


            clock_gettime(CLOCK_MONOTONIC, &ts_ret);

            if(ts_call.tv_sec == ts_ret.tv_sec)
            {
                exec_time = ts_ret.tv_nsec - ts_call.tv_nsec;
            }
            else /* next second*/
            {
                exec_time = (1000000000 - ts_call.tv_nsec) + ts_ret.tv_nsec;
            }

            ETG_TRACE_USR4(("[ecnr_sse_handler_process_audio]: sseProcess EXEC time : [%ld]\n", exec_time));
        }
        else
        {
            /*------ signal processing function of the SSE module -*/
#if   D_ECNR_LIB_SSE_3_17
            err = sseProcess
                  (
                     m_sseState.pMain
                    ,*pstate->MicIn
                    ,*pstate->RecvIn
                    ,*pstate->RefIn
                    ,*pstate->MicOut
                    ,*pstate->RecvOut
                    ,NULL
                  );
#elif D_ECNR_LIB_SSE_4_03
            tsseAudioIO  audioIO;
            memset( &audioIO, 0, sizeof(tsseAudioIO) );

            audioIO.AudioBufferSampleCnt = pstate->ecnr_frame_shift_out;
            audioIO.MicInBufferCnt   = pstate->ecnr_mic_in_cnt;
            audioIO.MicProcBufferCnt = pstate->ecnr_mic_proc_cnt;
            audioIO.RcvInBufferCnt   = pstate->ecnr_recv_in_cnt;
            audioIO.RcvProcBufferCnt = pstate->ecnr_recv_in_cnt;
            audioIO.RefInBufferCnt   = pstate->ecnr_ref_in_cnt;
            audioIO.MicInBuffers     = *pstate->MicIn;
            audioIO.MicProcBuffers   = *pstate->MicOut;
            audioIO.RcvInBuffers     = *pstate->RecvIn;
            audioIO.RcvProcBuffers   = *pstate->RecvOut;
            audioIO.RefInBuffers     = *pstate->RefIn;
            err = sseProcess
                  (
                      m_sseState.pMain
                     ,audioIO
                     ,NULL
                  );
#endif
        }

#if   D_ECNR_LIB_SSE_3_17
        if(pstate->ecnr_enable_asr_preprocessing)
        {
            *pstate->MicOut[0] = ptemp;
        }

        if(!err && pstate->ecnr_enable_asr_preprocessing)
        {
            //*pstate->MicOut[0] = ptemp;
            ElemCnt = 0;

            err = ssePrepareDataForASR
                  (
                      m_sseState.pMain
                     ,pstate->AsrPrepare
                     ,pstate->ecnr_frame_shift_out
                     ,*pstate->MicOut[0]
                     ,pstate->ecnr_frame_shift_out
                     ,&ElemCnt
                   );


            if(err)
            {
                pr_critical("call of ssePrepareDataForASR returns error:%d and elemcnt:%d with frame_size:%d\n", err, ElemCnt, pstate->ecnr_frame_shift_out );
                ETG_TRACE_USR4(("[ecnr_sse_handler_process_audio]: call of ssePrepareDataForASR returns error:%d and elemcnt:%d with frame_size:%d", err, ElemCnt, pstate->ecnr_frame_shift_out));
            }

            if(ElemCnt != pstate->ecnr_frame_shift_out)
                pr_debug("call of ssePrepareDataForASR returns elemcnt:%d with frame_size:%d\n", ElemCnt, pstate->ecnr_frame_shift_out);
        }
#endif

#ifdef D_SSE_TEST_PERFORMANCE
        ecnr_sse_handler_performance_stop_current_test_step();
#endif
    }
    else
    {
        pr_warning("no sse valid instance \n");
        err= SSE_ERR_NOT_CREATED;
    }

    ETG_TRACE_USR4(("[ecnr_sse_handler_process_audio]: EXIT"));
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_process_audio_with_debug_data
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/

int ecnr_sse_handler_process_audio_with_debug_data(stecnr_alsa_state* pstate)
{
    int err = ECNR_ERR_OK;
#if   D_ECNR_LIB_SSE_3_17
    short int *ptemp = (short int *)NULL;
    guint32 ElemCnt = 0;
#endif
    //pr_debug("entered");
    ETG_TRACE_USR4(("[ecnr_sse_handler_process_audio_with_debug_data]: ENTERED"));
    if(m_sseState.pMain)
    {
#ifdef D_SSE_TEST_PERFORMANCE
        ecnr_sse_handler_performance_start_next_test_step();
#endif

#if   D_ECNR_LIB_SSE_3_17
        if(pstate->ecnr_enable_asr_preprocessing)
        {
            ptemp = *pstate->MicOut[0];
            *pstate->MicOut[0] = pstate->AsrPrepare;
        }
        else
        {
            /*do nothing */
        }
#endif

        if(pstate->pTuneIO->InBufferUsedSize)
        {
            pr_debug("sseProcess %d Debug data \n", pstate->pTuneIO->InBufferUsedSize);
        }

        /*------ signal processing function of the SSE module -*/
#if   D_ECNR_LIB_SSE_3_17

        err = sseProcess( m_sseState.pMain,
               *pstate->MicIn ,
               *pstate->RecvIn,
               *pstate->RefIn,// NULL,
               *pstate->MicOut,
               *pstate->RecvOut,
     (tsseDebug*)pstate->pTuneIO);

#elif D_ECNR_LIB_SSE_4_03
        tsseAudioIO  audioIO;
        memset( &audioIO, 0, sizeof(tsseAudioIO) );

        audioIO.AudioBufferSampleCnt = pstate->ecnr_frame_shift_out;
        audioIO.MicInBufferCnt   = pstate->ecnr_mic_in_cnt;
        audioIO.MicProcBufferCnt = pstate->ecnr_mic_proc_cnt;
        audioIO.RcvInBufferCnt   = pstate->ecnr_recv_in_cnt;
        audioIO.RcvProcBufferCnt = pstate->ecnr_recv_in_cnt;
        audioIO.RefInBufferCnt   = pstate->ecnr_ref_in_cnt;
        audioIO.MicInBuffers     = *pstate->MicIn;
        audioIO.MicProcBuffers   = *pstate->MicOut;
        audioIO.RcvInBuffers     = *pstate->RecvIn;
        audioIO.RcvProcBuffers   = *pstate->RecvOut;
        audioIO.RefInBuffers     = *pstate->RefIn;

        err = sseProcess
              (
                  m_sseState.pMain
                 ,audioIO
                 ,(tsseTuneIO*)pstate->pTuneIO
              );
#endif



        if(pstate->pTuneIO->OutBufferSize != BUFFER_LENGTH)
        {
            pr_debug("return sseProcess %d Debug data \n", pstate->pTuneIO->OutBufferSize);
        }

#if   D_ECNR_LIB_SSE_3_17
        if(pstate->ecnr_enable_asr_preprocessing)
        {
            *pstate->MicOut[0] = ptemp;
        }

        if(!err && pstate->ecnr_enable_asr_preprocessing)
        {
            //*pstate->MicOut[0] = ptemp;
            ElemCnt = 0;

            err = ssePrepareDataForASR ( m_sseState.pMain,
                        pstate->AsrPrepare,
                        pstate->ecnr_frame_shift_out,
                        *pstate->MicOut[0],
                        pstate->ecnr_frame_shift_out,
                        &ElemCnt);


            if(err)
                pr_critical(" call of ssePrepareDataForASR returns error:%d and elemcnt:%d with frame_size:%d\n", err, ElemCnt, pstate->ecnr_frame_shift_out );

            if(ElemCnt != pstate->ecnr_frame_shift_out)
                pr_debug("call of ssePrepareDataForASR returns element:%d with frame_size:%d\n", ElemCnt, pstate->ecnr_frame_shift_out);
        }
#endif

#ifdef D_SSE_TEST_PERFORMANCE
        ecnr_sse_handler_performance_stop_current_test_step();
#endif
    }
    else
    {
        pr_warning("ecnr_sse_handler_Process_audio:: no sse valid instance \n");
        err = SSE_ERR_NOT_CREATED;
    }
    //pr_debug("EXIT");

    ETG_TRACE_USR4(("[ecnr_sse_handler_process_audio_with_debug_data]: EXIT"));
    return err;
}
/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_Process_debug_data_only
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/

int ecnr_sse_handler_Process_debug_data_only(stecnr_alsa_state* pstate)
{
    int err = ECNR_ERR_OK;

    ETG_TRACE_USR4(("[ecnr_sse_handler_Process_debug_data_only]: ENTERED"));

    //pr_debug("entered");
    if(m_sseState.pMain)
    {
        /*------ signal processing function of the SSE module -*/
#if   D_ECNR_LIB_SSE_3_17

        err = sseProcess( m_sseState.pMain,
                             NULL ,
                             NULL,
                             NULL,
                             NULL,
                             NULL,
                 (tsseDebug*)pstate->pTuneIO);

#elif D_ECNR_LIB_SSE_4_03
        tsseAudioIO  audioIO;
        memset( &audioIO, 0, sizeof(tsseAudioIO) );
        err = sseProcess
              (
                  m_sseState.pMain
                 ,audioIO
                 ,(tsseTuneIO*)pstate->pTuneIO
              );
#endif

    }
    else
    {
        pr_warning("ecnr_sse_handler_Process_audio:: no sse valid instance \n");
        err= SSE_ERR_NOT_CREATED;
    }
    //pr_debug("EXIT");
    ETG_TRACE_USR4(("[ecnr_sse_handler_Process_debug_data_only]: EXIT"));
    return err;
}



/*******************************************************************************
*
* FUNCTION: ecnr-sse-handler-create
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int ecnr_sse_handler_create(int memOptLevel)
{
    int err = ECNR_ERR_OK;
    pr_debug("ENTERED \n");
    ETG_TRACE_USR3(("[ecnr_sse_handler_create]: ENTERED"));

#ifdef D_ECNR_LIB_SSE_3_17
    err = sseCreate(&m_sseState.pMain, NULL, (tsseMemOptLevel)memOptLevel);
#elif  D_ECNR_LIB_SSE_4_03
    err = sseCreate(&m_sseState.pMain, NULL);
#endif

    if (!m_sseState.pMain)
        pr_critical("sse instance creation failed \n");

    ETG_TRACE_USR3(("[ecnr_sse_handler_create]: EXIT"));
    (void)memOptLevel;
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_get_data
*
* DESCRIPTION: To retrieve the data set
*
* PARAMETER: [IN] Id of the data to be retrieved
*
* RETURNVALUE: int.
*
*******************************************************************************/


int ecnr_sse_handler_get_data
(
    const unsigned  int  DataID,
    const unsigned  int  iChannel,
          unsigned  int* pSize,
                   void* pData
)
{
    int err = ECNR_ERR_OK;

    pr_debug("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_sse_handler_get_data]: ENTERED"));

    if(m_sseState.pMain)
    {
#if   D_ECNR_LIB_SSE_3_17
        err = sseGetData(m_sseState.pMain, (tsseDataID)DataID, iChannel, pSize, pData );
#elif D_ECNR_LIB_SSE_4_03
        err = sseGetData(m_sseState.pMain, DataID, (int)iChannel, pSize, pData );
#endif
        pr_debug("parameter %x\n", DataID);

        ETG_TRACE_USR3(("[ecnr_sse_handler_get_data]: call sseGetData with DataID->%x, iChannel->%d, pSize->%d", DataID, iChannel, *pSize));
    }
    else
    {
        pr_debug("no sse valid instance");
        err = SSE_ERR_NOT_CREATED ;
    }

    ETG_TRACE_USR3(("[ecnr_sse_handler_get_data]: EXIT"));
    pr_debug("EXIT\n");

    return err;
}



typedef enum _reqType
{
  ECNR_SSE_REQ_SET,
  ECNR_SSE_REQ_GET
}ecnr_sse_req_type;

typedef struct _ReqData
{
    ecnr_sse_req_type   req_type;
    unsigned int        data_id;
    unsigned int        channel;
    unsigned int        size;
    void*               data;
}ecnr_sse_reg_data;

#define ECNR_SSE_MAX_REQ_NB 10
static ecnr_sse_reg_data ecnr_sse_req_liste[ECNR_SSE_MAX_REQ_NB];
static guint ecnr_sse_req_cnt = 0;

static int ecnr_sse_add_new_req(ecnr_sse_reg_data* data)
{
    ecnr_sse_reg_data* new_req_data = (ecnr_sse_reg_data*)NULL;

    if(!data)
    {
        pr_warning(" data structure pointer invalid\n");
        return ECNR_ERR_OK;
    }


    if(ecnr_sse_req_cnt == ECNR_SSE_MAX_REQ_NB)
    {
        pr_warning("max number of request is reached, this new request is ignored \n");
        return ECNR_ERR_OK;
    }

    ETG_TRACE_USR3(("[ecnr_sse_add_new_req]: add new request for parameter %d", data->data_id));
    pr_message("add new request for parameter %d\n", data->data_id);

    /* store new request */
    new_req_data = &ecnr_sse_req_liste[ecnr_sse_req_cnt];

    new_req_data->req_type  = data->req_type;
    new_req_data->data_id   = data->data_id;
    new_req_data->channel   = data->channel;
    new_req_data->size      = data->size;

    new_req_data->data = g_malloc0(data->size);
    if(new_req_data->data)
        memcpy(new_req_data->data, data->data, data->size);

    ecnr_sse_req_cnt++;

    ETG_TRACE_USR3(("[ecnr_sse_add_new_req]: EXIT"));
    return ECNR_ERR_OK;
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_process_pending_req
*
* DESCRIPTION: To set data in teh sse Engine
*
* PARAMETER: [IN] Id of the data to be retrieved
*
* RETURNVALUE: int.
*
*******************************************************************************/

int ecnr_sse_process_pending_req(void)
{
    int                 err = ECNR_ERR_OK;
    ecnr_sse_reg_data*  pending_data = (ecnr_sse_reg_data*)NULL;
    guint list_count = 0;
    guint i = 0;

    if(!ecnr_sse_req_cnt)
        return ECNR_ERR_OK;

    if(!m_sseState.pMain)
    {
        pr_warning("no ase library instance available\n");
        return ECNR_ERR_OK;
    }

    list_count = ecnr_sse_req_cnt;

    ETG_TRACE_USR3(("[ecnr_sse_process_pending_req]: %d pending request", list_count));
    pr_message("%d pending request\n", list_count);

    /* walk through all prending requets */
    for( i = 0; i < list_count; i++)
    {
        pending_data = &ecnr_sse_req_liste[i];

        if((NULL != pending_data) && (NULL != pending_data->data))
        {
          if(pending_data->req_type == ECNR_SSE_REQ_SET)
          {

            if(pending_data->size < 5)
            {
                int value = 0;
                memcpy(&value,pending_data->data,pending_data->size);
                pr_message("proceed to sseSetData pending request with parameter->%d, size->%d, value->%d\n", pending_data->data_id, pending_data->size, value);
                ETG_TRACE_USR3(("[ecnr_sse_process_pending_req]: proceed to sseSetData pending request with parameter->%d, size->%d, value->%d", pending_data->data_id, pending_data->size, value));
            }
            else
            {
                ETG_TRACE_USR3(("[ecnr_sse_process_pending_req]: proceed to sseSetData pending request with parameter %d", pending_data->data_id));
                pr_message("proceed to sseSetData pending request with parameter %d\n", pending_data->data_id);
            }


#if   D_ECNR_LIB_SSE_3_17
        err = sseSetData( m_sseState.pMain,
              (tsseDataID)pending_data->data_id,
                     (int)pending_data->channel,
                          pending_data->size,
                          pending_data->data );
#elif D_ECNR_LIB_SSE_4_03
        err = sseSetData( m_sseState.pMain,
                          pending_data->data_id,
                     (int)pending_data->channel,
                          &pending_data->size,
                          pending_data->data );
#endif

            if(pending_data->data)
            {
                g_free(pending_data->data );
                pending_data->data = (void*)NULL;
            }

            if(err)
            {
               pr_warning(" sseSetData fails with dataid:%d, size:%d \n",pending_data->data_id, pending_data->size );
            }
        }
        else if(pending_data->req_type == ECNR_SSE_REQ_GET)
        {
            pr_warning("currently not supported\n");
        }
        else
        {
            /* do nothing */
        }

        ecnr_sse_req_cnt--;
        }
        else
        {
            err = ECNR_ERR_NULL_POINTER;
            ETG_TRACE_ERR(("[ecnr_sse_process_pending_req]: pending_data is Null %d at %d\n", err,i));
            pr_critical("pending_data is Null %d at %d\n", err,i);
        }

    }

    ETG_TRACE_USR3(("[ecnr_sse_process_pending_req]: EXIT"));
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_set_data
*
* DESCRIPTION: To set data in teh sse Engine
*
* PARAMETER: [IN] Id of the data to be retrieved
*
* RETURNVALUE: int.
*
*******************************************************************************/

int ecnr_sse_handler_set_data
(
    const unsigned  int  DataID,
    const unsigned  int  iChannel,
          unsigned  int  iSize,
                   void* pData
)
{
    int err = ECNR_ERR_OK;
    struct ecnr_instance_object* my_obj = NULL;

    my_obj =  ecnr_service_instance_active_get();

    if(my_obj != NULL)
    {
        if (my_obj->ecnr_service_terminate_req)
        {
            pr_debug("my_obj->ecnr_service_terminate_req hit");
            return ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE;
        }
     }
 
    if(pData == NULL)
    {
       pr_critical("pdata is NULL sse handler set data failed");
       ETG_TRACE_ERR(("[ecnr_sse_handler_set_data]: pdata is NULL\n"));
       return ECNR_ERR_NULL_POINTER;
    }

    if(iSize <= 4)
    {
        int value = 0;
        (void)memcpy((void*)&value, pData, iSize);
        ETG_TRACE_USR3(("[ecnr_sse_handler_set_data]: DataId(%x), Channel(%d), Size(%d), value(%d)", DataID,iChannel, iSize, value));
        pr_debug("DataId(%x), Channel(%d), Size(%d), value(%d)\n", DataID,iChannel, iSize, value);
    }
    else
        pr_debug("DataId(%x), Channel(%d), Size(%d)\n", DataID, iChannel, iSize);

    if(m_sseState.pMain)
    {
        ecnr_audio_state audio_state = ecnr_service_audio_get_state();

        if(   (ECNR_AS_STREAM_SILENCE       == audio_state)
            ||(ECNR_AS_STREAM_INIT          == audio_state)
            ||(ECNR_AS_STREAM_TERMINATED    == audio_state))
        {
            if(iSize <= 4)
            {
                int value = 0;
                (void)memcpy((void*)&value, pData, iSize);
                pr_debug("DataId(%x), Channel(%d), Size(%d), value(%d)\n", DataID,iChannel, iSize, value);
            }
            else
            {
                pr_debug("DataId(%x), Channel(%d), Size(%d)\n", DataID, iChannel, iSize);
            }

#if   D_ECNR_LIB_SSE_3_17
            err = sseSetData
                  (
                      m_sseState.pMain
                     ,(tsseDataID)DataID
                     ,iChannel
                     ,iSize
                     ,pData
                   );
#elif D_ECNR_LIB_SSE_4_03
            err = sseSetData
                  (
                      m_sseState.pMain
                     ,DataID
                     ,(int)iChannel
                     ,&iSize
                     ,pData
                   );
#endif
        }
        else
        {
            /* store a request which will be processed within the streaming thread
               prior to the call of sseProcess()*/

            ecnr_sse_reg_data new_req_data;

            new_req_data.req_type  = ECNR_SSE_REQ_SET;
            new_req_data.data_id   = DataID;
            new_req_data.channel   = iChannel;
            new_req_data.size      = iSize;
            new_req_data.data      = pData;


            err = ecnr_sse_add_new_req(&new_req_data);
        }

    }
    else
    {
        pr_debug("no sse valid instance");
        err = SSE_ERR_NOT_CREATED ;
    }

    if(err)
    {
        pr_debug("sseSetData for parameter:%d  is failed with error: %d \n", DataID, err);
    }
    return err;
}

/*******************************************************************************
*
* FUNCTION:
*
* DESCRIPTION: To set data in the sse Engine
*
* PARAMETER: [IN] Id of the data to be retrieved
*
* RETURNVALUE: int.
*
*******************************************************************************/
int ecnr_sse_handler_initialize
(
    const tecnrConfigData*  pConfigData,
    unsigned short       ConfigId
)
{
    int err = ECNR_ERR_OK;
    guint frame_shift = 0;
    guint sample_rate = 0;
    guint period_size_calc = 0;
    guint period_size_conf = 0;
    unsigned short ecnr_config_id = 0;
    gboolean bdocheck = TRUE;

    pr_debug("ENTERED");
    ETG_TRACE_USR3(("[ecnr_sse_handler_initialize]: ENTERED"));

    if(m_sseState.pMain)
    {
        if(!pConfigData)
        {
#ifdef D_ECNR_LIB_SSE_3_17
            pr_debug("SSE initialization without configuration data");
            ETG_TRACE_USR3(("[ecnr_sse_handler_initialize]: SSE initialization without configuration data"));
            err = sseInitialize(m_sseState.pMain,NULL);
            bdocheck = FALSE;
#elif  D_ECNR_LIB_SSE_4_03
            pr_debug("SSE initialization without configuration data");
            ETG_TRACE_USR3(("[ecnr_sse_handler_initialize]: SSE initialization without configuration data"));
            err = sseInitialize(m_sseState.pMain);
            bdocheck = FALSE;
#endif

        }
        else
        {
#ifdef D_ECNR_LIB_SSE_3_17
            pr_debug("SSE initialization with configuration data");
            ETG_TRACE_USR3(("[ecnr_sse_handler_initialize]: SSE initialization with configuration data"));

            err = sseInitialize
                  (
                          m_sseState.pMain
                         ,(tsseConfigData*)pConfigData
                   );
#elif  D_ECNR_LIB_SSE_4_03

            pr_debug("SSE initialization with configuration data");
            ETG_TRACE_USR3(("[ecnr_sse_handler_initialize]: SSE initialization with configuration data"));




            for(int i = 0; i< pConfigData->ConfigBufferCnt ;i++)
            {
                err = sseSetConfigData
                      (
                           m_sseState.pMain
                          ,(void*)pConfigData->i32ConfigBuffer[i]
                          ,pConfigData->ConfigBufferSize[i]
                      );
                if(SSE_ERR_OK != err)
                	pr_debug("sseSetConfigData failed with error(%d) for configuration set(%d)\n",err,i);
            }

            err = sseInitialize(m_sseState.pMain);

#endif
        }
    }
    else
    {
        pr_warning("no sse valid instance");
        err = SSE_ERR_NOT_CREATED ;
    }

#ifdef D_SSE_TEST_PERFORMANCE
    ecnr_sse_handler_performance_test_init("sseProcess", 100);
#endif

#ifdef SSE_TEST_STUB
    err = ecnr_sse_handler_test_stub_prepare();
#endif

    if(!err && bdocheck)
    {
        /* add check of period size */
        frame_shift = ecnr_alsa_get_frame_shift_in();
        sample_rate = ecnr_alsa_get_sample_rate_in();

        if(sample_rate)
        {
            period_size_calc = (1000 * frame_shift ) / sample_rate;

            ecnr_config_id = ecnr_configuration_get_ecnr_config_Id(ConfigId);
            #ifdef D_SUPPORT_PERIOD_PARAMETER
            period_size_conf = ecnr_configuration_get_period(ecnr_config_id);
            #else
            period_size_conf = ecnr_configuration_get_prefill(ecnr_config_id);
            #endif

            if(period_size_calc != period_size_conf)
            {
                ETG_TRACE_ERR(("[ecnr_sse_handler_initialize]: mismatch between the period size in the bsd file -> %dms  and the configured size -> %dms ",
                period_size_calc,
                period_size_conf));

                pr_warning("mismatch between the period size in the bsd file -> %dms  and the configured size -> %dms\n", period_size_calc,  period_size_conf);
                err = ECNR_ERR_PERIODE_SIZE_INVALID;

                /* here write error memory */
                    ETG_TRACE_ERRMEM(("[ECNR][ecnr_sse_handler_initialize]: mismatch between the period size in the bsd file -> %dms  and the configured size -> %dms, load the default configuration",
                    period_size_calc,
                    period_size_conf));
            }
        }
    }
    else
    {
        pr_debug("sseInitialize failed with err -> %d, period size check will not be done \n", err);
    }

    ETG_TRACE_USR3(("[ecnr_sse_handler_initialize]: EXIT"));
    pr_debug("EXIT with error -> %d", err);
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_sse_create_instance
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

/*static*/  int ecnr_sse_handler_sse_create_instance (void)
{
    int            err = SSE_ERR_OK;

    pr_debug("ENTERED");
    ETG_TRACE_USR3(("[ecnr_sse_handler_sse_create_instance]: ENTERED"));

    /* Get and display SSE version number
    * (Note that no SSE instance needs to be created before)
    */
    ecnr_sse_get_version(&m_sseState.iArrayLen, &m_sseState.pVersionArray, &m_sseState.pVersionString, &m_sseState.pVersionComment);

    /* Make sure there are no leftovers or memory leaks from a previous session
    */
    ecnr_configuration_free_Configuration_Data(&m_sseState.ConfigData);

    /* Set m_sseState.SSEInitOk to false.
    * If all goes well, it will be set to true
    */
    m_sseState.SSEInitOk = FALSE;

   /* m_sseState is guaranteed to be initialized with 0 on startup
    * So if m_sseState.SSEMemTab[0].pMem is still NULL we need to allocate the buffers
    * Buffers are persistant between config changes, so this needs to be done only once
    */
    if(m_sseState.SSEMemTab[0].pMem == NULL)
    {
        m_sseState.SSEMemTab[0].pMem = g_malloc0(SSE_C_EXT_MEM_SIZE);

        if(m_sseState.SSEMemTab[0].pMem == NULL)
        {
            pr_debug("memory allocation for sse failed\n");
            ETG_TRACE_USR4(("[ecnr_sse_handler_sse_create_instance]: memory allocation for sse failed"));
            return SSE_ERR_ALLOC;
        }
        else
        {
            m_sseState.SSEMemTab[0].Size = SSE_C_EXT_MEM_SIZE;
#ifdef D_ECNR_LIB_SSE_3_17
            m_sseState.SSEMemTab[0].Type = ssePersistMem;
#elif  D_ECNR_LIB_SSE_4_03
            /* 4.03 library always uses persistent memory */
#endif

        }
    }

    /* Reinitialize every time
     * We need to do this, because the application will configure SSE after each init.
     * This is required because we can't receive messages if the audio route is not configured.
     * Therefore we have to provide a well defined status after each call to init.
     *
     * Delete a previous instance of SSE
     * If the previous instance was already deleted or there was no previous
     * instance, m_sseState.pMain will be NULL which is a valid input for sseDestroy()
     */

    if(m_sseState.pMain)
    {
#if defined(D_ECNR_LIB_SSE_3_17) || defined(D_ECNR_LIB_SSE_4_03)
        err = sseDestroy(&m_sseState.pMain);
#endif
        if(err != SSE_ERR_OK)
        {
            pr_debug("failed to sse Destroy");
            return SSE_ERR_ALLOC;
        }
        ETG_TRACE_USR4(("[ecnr_sse_handler_sse_create_instance]: sseDestroy was successfully called"));
        pr_debug("sseDestroy was successfully called");
    }

    /* Create an instance of SSE
     *
     * First stage of the initialization process. This must be done at the very
     * beginning, before anything except for fetching the version number can be
     * done with SSE. If SSEMemTab is initialized and contains at least one
     * valid pointer to a memory buffer, only memory of this buffer is used
     * within SSE (i.e. no internal memory allocation)
     */

    if(!m_sseState.pMain)
    {

#ifdef D_ECNR_LIB_SSE_3_17
        err = sseCreate(&m_sseState.pMain, m_sseState.SSEMemTab, sseMemNoOpt);
#elif  D_ECNR_LIB_SSE_4_03
        err = sseCreate(&m_sseState.pMain, m_sseState.SSEMemTab);
#endif

        if(err != SSE_ERR_OK)
        {
            ecnr_critical(err, "ecnr_sse_handler_sse_create_instance");
            return err;
        }
        ETG_TRACE_USR4(("[ecnr_sse_handler_sse_create_instance]: a new SSE instance was created"));
        pr_debug("a new sse instance was created");
    }

    /* If initialization was successful, store it in m_sseState.SSEInitOk.
     * Otherwise leave it as FALSE so we know that the setup failed inside the callback function
     */
    m_sseState.SSEInitOk = TRUE;
    //pr_debug("m_sseState.SSEInitOk = %d \n ", m_sseState.SSEInitOk);


    ETG_TRACE_USR4(("[ecnr_sse_handler_sse_create_instance]: EXIT"));
    pr_debug("EXIT with err = %d \n", err);
    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_compare_version
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_sse_handler_compare_version(int* version)
{
#ifdef D_SSE_VERSION_CHECK_ON
    unsigned int i = 0;  //modified for lint fix

    pr_debug("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_sse_handler_compare_version]: ENTERED"));

    if(!m_sseState.pVersionArray)
        return ECNR_ERR_NULL_POINTER;

    /* compare all element one by one*/
    for (i = 0; i< m_sseState.iArrayLen; i++)
    {
        if(m_sseState.pVersionArray[i] != version[i])
        {
            pr_debug("sse_version[%d] = %d do not match with dataset_version[%d] = %d\n", i, m_sseState.pVersionArray[i],i,version[i]);
            ETG_TRACE_USR4(("[ecnr_sse_handler_compare_version]: sse_version[%d] = %d do not match with dataset_version[%d] = %d", i, m_sseState.pVersionArray[i],i,version[i]));
            return FALSE;
        }

        pr_debug("sse_version[%d] = %d equal to dataset_version[%d] = %d \n", i, m_sseState.pVersionArray[i],i, version[i]);
        ETG_TRACE_USR4(("[ecnr_sse_handler_compare_version]: sse_version[%d] = %d equal to dataset_version[%d] = %d", i, m_sseState.pVersionArray[i],i, version[i]));
    }
#endif

    ETG_TRACE_USR3(("[ecnr_sse_handler_compare_version]: EXIT"));
    return TRUE;
}


/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_create_session
*
* DESCRIPTION: THis function initialize the sse Engine with the passed ConfigId
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

int ecnr_sse_handler_create_session(unsigned short ConfigId)
{
    int err = 0;

    pr_debug("ENTERED, requested Configid -> %d \n",ConfigId);
    ETG_TRACE_USR3(("[ecnr_sse_handler_create_session]: ENTERED, requested Configid -> %d", ConfigId));

    /* the sse is already initailized with the requested configuration */
    if(m_sseState.SSEInitOk)
    {
        pr_warning("already initialized with Config %d \n",  m_CurrConfigId);
        ETG_TRACE_USR3(("[ecnr_sse_handler_create_session]: already initialized with config_id -> %d", m_CurrConfigId));
        return ECNR_ERR_OK;
    }

    /* initialization from sse and data structure*/
    err = ecnr_sse_handler_sse_create_instance();

    if (err)
    {
        pr_warning("no instance of SSE available");
        ETG_TRACE_USR3(("[ecnr_sse_handler_create_session]: no sse instance available"));
        return err;
    }

    if(!ConfigId)
    {
        pr_warning("ConfigId is 0, sse is not initialized");
        ETG_TRACE_USR3(("[ecnr_sse_handler_create_session]: config_id is 0, sse is not initialized"));
        return ECNR_ERR_OK;
    }

    if (ConfigId == m_CurrConfigId)
    {
        pr_warning("already initialized with the requested Config %d \n",  m_CurrConfigId);
        ETG_TRACE_USR3(("[ecnr_sse_handler_create_session]: already initialized with the requested config_id -> %d", m_CurrConfigId));
        return ECNR_ERR_OK;
    }

    /* set configuration */
    err = ecnr_sse_handler_set_configuration(ConfigId, FALSE);

    /* failure while initialization, destroy SSE instance */
    if (err)
    {
        ecnr_engine_handler_destroy(); //ecnr_sse_handler_destroy();
    }

    pr_debug("EXIT with err = %d \n", err);
    ETG_TRACE_USR3(("[ecnr_sse_handler_create_session]: EXIT"));

    if(err)
        ETG_TRACE_USR3(("[ecnr_sse_handler_create_session]: EXIT with error -> %d", err));

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_reset
*
* DESCRIPTION: THis function initialize the sse Engine with the passed ConfigId
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_sse_handler_reset (void)
{
    int status = (int)ECNR_ERR_OK;
    pr_debug("ENTERED\n");

#ifdef D_ECNR_LIB_SSE_3_17
    status = sseReset(m_sseState.pMain);
#elif  D_ECNR_LIB_SSE_4_03
    guint32  sse_reset_mode = 0xFFFFFFFF; /* full reset */
    status = sseReset(m_sseState.pMain, sse_reset_mode);
#endif

    pr_debug("EXIT\n");
    return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_destroy
*
* DESCRIPTION: THis function initialize the sse Engine with the passed ConfigId
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_sse_handler_destroy (void)
{
    int err = ECNR_ERR_OK;

    pr_message("ENTERED \n");
    ETG_TRACE_USR3(("[ecnr_sse_handler_destroy]: ENTERED"));

    if(!m_sseState.pMain)
    {
        pr_warning("the sse library is not instanced");
        return SSE_ERR_NOT_CREATED ;
    }

#if defined(D_ECNR_LIB_SSE_3_17) || defined(D_ECNR_LIB_SSE_4_03)
    err = sseDestroy(&m_sseState.pMain);
#endif

    m_sseState.pMain = NULL;
    m_sseState.SSEInitOk = FALSE;
    m_CurrConfigId = 0;


#ifdef D_SSE_TEST_PERFORMANCE
    ecnr_sse_handler_performance_test_terminate();
#endif

#ifdef SSE_TEST_STUB
    ecnr_sse_handler_test_stub_clear();
#endif

    ETG_TRACE_USR3(("[ecnr_sse_handler_destroy]: EXIT"));
    return err;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_check_operating_mode
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_sse_handler_check_operating_mode(void)
{
    int status = (int)ECNR_ERR_OK;

    stecnr_alsa_state* state = ecnr_alsa_get_instance();
    if(!state)
    {
        pr_debug("no alsa module instance available\n");
        status = ECNR_ERR_SERV_NOT_INITIALIZED;
        return status;
    }

#ifdef D_ECNR_LIB_SSE_3_17
    if(state->ecnr_op_mode == sseVoiceRecog)
    {
        if(state->ecnr_asr_mode == sseSpeechProbLSB)
        {
            state->ecnr_enable_asr_preprocessing = TRUE;
            pr_debug("enable ASR pre-processing\n");
        }
        else if(state->ecnr_asr_mode == sseSpeechProbAttached)
        {
            state->ecnr_enable_asr_preprocessing = FALSE;
            pr_critical("disable ASR pre-processing. the mode sseSpeechProbAttached is not supported\n");
            status = ECNR_ERR_INVALID_ASR_MODE;
        }
        else
        {
        /* do nothing */
        }
    }
    else
    {
        state->ecnr_enable_asr_preprocessing = FALSE;

        /* hands-free mode */
        state->ecnr_asr_mode = -1;
    }
#elif  D_ECNR_LIB_SSE_4_03
    if(state->ecnr_op_mode == ecnrVoiceRecog)
    {
        if(state->ecnr_asr_mode == ecnrLimitedInfo)
        {
            state->ecnr_enable_asr_preprocessing = TRUE;
            pr_message("limited meta information (to be used for VoCon ASR version < 4.11) \n");
        }
        else if(state->ecnr_asr_mode == ecnrEnhancedInfo)
        {
            state->ecnr_enable_asr_preprocessing = TRUE;
            pr_message("enhanced meta information (to be used for VoCon ASR version > 4.11) \n");
        }
        else
        {
        /* do nothing */
        }
    }
    else
    {
        state->ecnr_enable_asr_preprocessing = FALSE;

        /* hands-free mode */
        state->ecnr_asr_mode = -1;
    }
#endif

    return status;
}


/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_set_configuration
*
* DESCRIPTION: THis function initialize the sse Engine with the passed ConfigId
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_sse_handler_set_configuration (unsigned short ConfigId, gboolean test)
{
    int                          err = SSE_ERR_OK;
    const struct EcnrConfig      *pdata = NULL;
    guint8                       type = 0;
    gboolean                     isdefault = FALSE;

    pr_debug("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_sse_handler_set_configuration]: ENTERED"));

    /* Is the sse initialized ? */
    if(!m_sseState.SSEInitOk)
    {
        pr_warning("sse not initialized, call ecnr_initialize first\n");
        return SSE_ERR_MODULE_NOT_INIT;
    }

    /* the sse is already initialized with the requested configuration */
    else
    {
        if(ConfigId == m_CurrConfigId)
        {
            pr_warning("sse already initialized with Config %d\n", m_CurrConfigId);
            return ECNR_ERR_OK;
        }
    }

    pdata = ecnr_configuration_get_config_data(ConfigId);

    if (!pdata)
    {
        pr_warning("no valid configuration for configId : % d\n", ConfigId);
        return ECNR_ERR_CONFIG_NOT_AVAILABLE;
    }

    pr_debug("configId: %d, loc: %s\n", pdata->configId, pdata->bsd_str);
    ETG_TRACE_USR3(("[ecnr_sse_handler_set_configuration]: configId: %d, loc: %s", pdata->configId, pdata->bsd_str));

    /* Provide correct pointers, depending on use case*/
    type = ecnr_configuration_get_config_type(ConfigId);

    switch(type)
    {
        case CONFIG_TYPE_FILE:
        case CONFIG_TYPE_TABLE:
        case CONFIG_TYPE_KDS:
            /* return configuration file */
            err = ecnr_configuration_get_configuration(ConfigId, &m_sseState.ConfigData, &isdefault);
            break;
        case CONFIG_TYPE_SET_FCT:
            if(pdata->vsetFunction)
                err = pdata->vsetFunction();
            break;
        default:
            err = SSE_ERR_PARAM;
            break;
    }

    /*-------------------------------------------------------------------------*/
    /*       Create an instance of SSE                                         */
    /*-------------------------------------------------------------------------*/
    /* First stage of the initialization process. This must be done at the very
       beginning, before anything except for fetching the version number can be
       done with SSE. If SSEMemTab is initialized and contains at least one
       valid pointer to a memory buffer, only memory of this buffer is used
       within SSE (i.e. no internal memory allocation) */

    if(err == SSE_ERR_OK)
    {
        /* Initialize SSE */
        if(type == CONFIG_TYPE_SET_FCT)
        {
            err = ecnr_sse_handler_initialize(NULL, ConfigId);

            /* Initialize SSE engine */
            if((err == ECNR_ERR_OK) /* && (type == CONFIG_TYPE_SET_FCT)*/)
            {
                /* post parameter initialization ? */
                pr_message("set post-init parameter after ecnr initialization");
                ETG_TRACE_USR3(("[ecnr_sse_handler_set_configuration]: set post-init parameter after ecnr initialization"));

                if(pdata->vsetFunction_2)
                    err = (int)pdata->vsetFunction_2();
            }
            else
            {
                /* do nothing */
            }
        }
        else
        {
            /* no configuration file available */
            if (!m_sseState.ConfigData.ConfigBufferCnt)
            {
                err = ecnr_sse_handler_initialize(NULL, ConfigId);
            }
            /* initialize the sse with the configuration file */
            else
            {
                err = ecnr_sse_handler_initialize(&m_sseState.ConfigData, ConfigId);
                if(test)
                    return err;

                if(err == ECNR_ERR_PERIODE_SIZE_INVALID)
                {
                    (void)ecnr_sse_handler_sse_create_instance();

                    if(!isdefault)
                    {
                        err = ecnr_configuration_read_default_configuration(pdata, &m_sseState.ConfigData);
                        if(!err)
                        {
                            err = ecnr_sse_handler_initialize(&m_sseState.ConfigData, ConfigId);
                            isdefault = TRUE;
                        }
                    }
                    pr_debug("is default -> %d \n", isdefault);
                    ETG_TRACE_USR3(("[ecnr_sse_handler_set_configuration]: is default -> %d", isdefault));
                }
            }
        }

        if(err && (err != ECNR_ERR_PERIODE_SIZE_INVALID))
        {
             ecnr_critical(err, "ecnr_sse_handler_set_configuration\n");
             return err;
        }
        else
        {
             /*save current configuration and initialization status */
             m_CurrConfigId = ConfigId;
             pr_debug("save current configuration %d \n", m_CurrConfigId);

            if(isdefault)
            {
                ecnr_sse_handler_log_config_data(ConfigId, pdata, ecnr_log_type_init_from_default);
                ecnr_engine_load_default_data_set();
            }
            else
                ecnr_sse_handler_log_config_data(ConfigId, pdata, ecnr_log_type_init_from_dp);

            /* delete error */
            err =  ECNR_ERR_OK;
        }
    }

    ETG_TRACE_USR3(("[ecnr_sse_handler_set_configuration]: EXIT"));
    pr_debug("EXIT with error: %d\n",err);
    return err;
}



/*******************************************************************************
*
* FUNCTION: ecnr_sse_get_version
* DESCRIPTION: ends debug session
*
* PARAMETER: [check xml for details]
*
* RETURNVALUE: void
*
*******************************************************************************/
void ecnr_sse_get_version
(
        unsigned  int*   pArrayLen,
  const           int**  pVersionArray,
  const          char**  pVersionString,
  const          char**  pVersionComment
)
{
    unsigned int i = 0;
    const int* ver = (int*)NULL;

#if defined(D_ECNR_LIB_SSE_3_17) || defined(D_ECNR_LIB_SSE_4_03)
    /* retrieve sse version */
    sseGetVersion( pArrayLen, pVersionArray, pVersionString, pVersionComment);
#endif

    pr_message("---- sse library version elements (%d) ------ \n", *pArrayLen);
    ETG_TRACE_USR2(("[ecnr_sse_get_version]: ---- sse library version elements (%d) ------", *pArrayLen));

    ver = *pVersionArray;

    if(ver)
    {
        /* print sse version*/
        for (i = 0; i < *pArrayLen; i++)
        {
            pr_message("version[%d] = %d\n", i, ver[i]);
            ETG_TRACE_USR2(("[ecnr_sse_get_version]: version[%d] = %d", i, ver[i]));
        }
    }
    else
        pr_debug("pVersionArray(%p) is NULL\n", *pVersionArray);

    return;

}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_get_status_message
* DESCRIPTION:
*
* PARAMETER: [check xml for details]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_sse_handler_get_status_message( char* status_message)
{
  int status = (int)ECNR_ERR_OK;

#if   D_ECNR_LIB_SSE_3_17
   status_message = (char*)g_malloc0(ECNR_MAX_STRING_SIZE + 1);

  if(*status_message)
      status = sseGetErrorMessage(m_sseState.pMain, status_message);
  else
      status = ECNR_ERR_NULL_POINTER;

#elif D_ECNR_LIB_SSE_4_03
  status_message = (char*)g_malloc0(ECNR_MAX_STRING_SIZE + 1);

  if(*status_message)
      status = sseGetErrorMessage
        (
                m_sseState.pMain
               ,status_message
               ,ECNR_MAX_STRING_SIZE
        );
  else
      status = ECNR_ERR_NULL_POINTER;
#endif

  return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_get_current_config_id
* DESCRIPTION:
*
* PARAMETER: [check xml for details]
*
* RETURNVALUE: int
*
*******************************************************************************/
unsigned short ecnr_sse_get_current_config_id(void)
{
  return m_CurrConfigId;
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_handler_performance_enable
* DESCRIPTION:
*
* PARAMETER: state
*
* RETURNVALUE: void
*
*******************************************************************************/
void ecnr_sse_handler_performance_enable(gboolean state)
{
    m_ecnr_sse_handler_performance_enable = state;
}


/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_micin_count
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_micin_count(guint* sse_mic_in_cnt)
{
    gint  status = SSE_ERR_OK;
    guint   size = sizeof(guint32);

#ifdef D_ECNR_LIB_SSE_3_17
    size   = sizeof(guint16);
    status = ecnr_sse_handler_get_data(sse_MicInCnt, 0, &size, sse_mic_in_cnt);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_data(sse_MicInCnt, 0, &size, sse_mic_in_cnt);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_mic_proc_count
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_mic_proc_count(guint* sse_mic_proc_cnt)
{
    gint  status = SSE_ERR_OK;
    guint   size = sizeof(guint32);

#ifdef D_ECNR_LIB_SSE_3_17
    size   = sizeof(guint16);
    status = ecnr_sse_handler_get_data(sse_MicOutCnt, 0, &size, sse_mic_proc_cnt);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_data(sse_MicProcCnt, 0, &size, sse_mic_proc_cnt);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_refin_count
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_refin_count(guint* sse_ref_in_cnt)
{
    gint  status = SSE_ERR_OK;
    guint   size = sizeof(guint32);

#ifdef D_ECNR_LIB_SSE_3_17
    size   = sizeof(guint16);
    status = ecnr_sse_handler_get_data(sse_RefInCnt, 0, &size, sse_ref_in_cnt);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_data(sse_RefInCnt, 0, &size, sse_ref_in_cnt);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_recvin_count
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_recvin_count(guint* sse_recv_in_cnt)
{
    gint  status = SSE_ERR_OK;
    guint   size = sizeof(guint32);

#ifdef D_ECNR_LIB_SSE_3_17
    size   = sizeof(guint16);
    status = ecnr_sse_handler_get_data(sse_RecvInCnt, 0, &size, sse_recv_in_cnt);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_data(sse_RcvInCnt, 0, &size, sse_recv_in_cnt);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_frame_shift_in
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_frame_shift_in(guint* sse_frame_shift_in)
{
    gint  status = SSE_ERR_OK;
    guint   size = sizeof(guint32);

#ifdef D_ECNR_LIB_SSE_3_17
    size   = sizeof(guint16);
    status = ecnr_sse_handler_get_data(sse_FrameShift, 0, &size, sse_frame_shift_in);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_data(sse_FrameShift, 0, &size, sse_frame_shift_in);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_frame_shift_out
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_frame_shift_out(guint* sse_frame_shift_out)
{
    gint  status = SSE_ERR_OK;
    guint   size = sizeof(guint32);

#ifdef D_ECNR_LIB_SSE_3_17
    size   = sizeof(guint16);
    status = ecnr_sse_handler_get_data(sse_FrameShift, 0, &size, sse_frame_shift_out);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_data(sse_FrameShift, 0, &size, sse_frame_shift_out);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_recv_frame_shift_in
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_recv_frame_shift_in(guint* sse_recv_frame_shift_in)
{
    gint  status = SSE_ERR_OK;
    guint   size = sizeof(guint32);

#ifdef D_ECNR_LIB_SSE_3_17
    size   = sizeof(guint16);
    status = ecnr_sse_handler_get_data(sse_FrameShift, 0, &size, sse_recv_frame_shift_in);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_data(sse_FrameShift, 0, &size, sse_recv_frame_shift_in);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_recv_frame_shift_out
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_recv_frame_shift_out(guint* sse_recv_frame_shift_out)
{
    gint  status = SSE_ERR_OK;
    guint   size = sizeof(guint32);

#ifdef D_ECNR_LIB_SSE_3_17
    size   = sizeof(guint16);
    status = ecnr_sse_handler_get_data(sse_FrameShift, 0, &size, sse_recv_frame_shift_out);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_data(sse_FrameShift, 0, &size, sse_recv_frame_shift_out);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_ref_frame_shift_in
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_ref_frame_shift_in(guint* sse_ref_frame_shift_in)
{
    gint  status = SSE_ERR_OK;
    guint   size = sizeof(guint32);

#ifdef D_ECNR_LIB_SSE_3_17
    size   = sizeof(guint16);
    status = ecnr_sse_handler_get_data(sse_FrameShift, 0, &size, sse_ref_frame_shift_in);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_data(sse_FrameShift, 0, &size, sse_ref_frame_shift_in);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_sample_rate_in
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_sample_rate_in(guint* sse_sample_rate_in)
{
    gint  status = SSE_ERR_OK;
    guint   size = sizeof(guint32);

#ifdef D_ECNR_LIB_SSE_3_17
    size   = sizeof(guint16);
    status = ecnr_sse_handler_get_data(sse_SampleRate, 0, &size, sse_sample_rate_in);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_data(sse_SampleRate, 0, &size, sse_sample_rate_in);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_sample_rate_out
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_sample_rate_out(guint* sse_sample_rate_out)
{
    gint  status = SSE_ERR_OK;
    guint   size = sizeof(guint32);

#ifdef D_ECNR_LIB_SSE_3_17
    size   = sizeof(guint16);
    status = ecnr_sse_handler_get_data(sse_SampleRate, 0, &size, sse_sample_rate_out);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_data(sse_SampleRate, 0, &size, sse_sample_rate_out);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_recv_sample_rate_in
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_recv_sample_rate_in(guint* sse_recv_sample_rate_in)
{
    gint  status = SSE_ERR_OK;
    guint   size = sizeof(guint32);

#ifdef D_ECNR_LIB_SSE_3_17
    size   = sizeof(guint16);
    status = ecnr_sse_handler_get_data(sse_SampleRate, 0, &size, sse_recv_sample_rate_in);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_data(sse_SampleRate, 0, &size, sse_recv_sample_rate_in);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_recv_sample_rate_out
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_recv_sample_rate_out(guint* sse_recv_sample_rate_out)
{
    gint  status = SSE_ERR_OK;
    guint   size = sizeof(guint32);

#ifdef D_ECNR_LIB_SSE_3_17
    size   = sizeof(guint16);
    status = ecnr_sse_handler_get_data(sse_SampleRate, 0, &size, sse_recv_sample_rate_out);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_data(sse_SampleRate, 0, &size, sse_recv_sample_rate_out);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_ref_sample_rate_in
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_ref_sample_rate_in(guint* sse_ref_sample_rate_in)
{
    gint  status = SSE_ERR_OK;
    guint   size = sizeof(guint32);

#ifdef D_ECNR_LIB_SSE_3_17
    size   = sizeof(guint16);
    status = ecnr_sse_handler_get_data(sse_SampleRate, 0, &size, sse_ref_sample_rate_in);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_data(sse_SampleRate, 0, &size, sse_ref_sample_rate_in);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_audio_type_for_asr
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_audio_type_for_asr(guint32* psize, gchar** pdata)
{
    gint    status = SSE_ERR_OK;
    guint32   size = sizeof(guint32);

#ifdef D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_data(sse_AudioTypeForASR, 0, &size, NULL);
    if(status == SSE_ERR_SIZE_ONLY)
    {
        if(NULL == *pdata)
            *pdata = (gchar*) g_malloc0 (size * sizeof(gchar));
        status = ecnr_sse_handler_get_data(sse_AudioTypeForASR, 0, &size, *pdata);
    }
#elif  D_ECNR_LIB_SSE_4_03
    size = 40;
    if(NULL == *pdata)
        *pdata = (gchar*) g_malloc0 (size * sizeof(gchar));
    status = ecnr_sse_handler_get_data(sse_AudioTypeForASR, 0, &size, *pdata);

//  if(NULL != pdata)
//  {
//
//                      FILE            *tempFile = NULL;   /* pointer to config data file */
//                      tempFile = fopen("/var/opt/bosch/dynamic/audiomanager/ecnr/tempFile.txt", "w");
//                      if((int32_t)size != fwrite(pdata, 1, size, tempFile))
//                      {
//                          pr_warning("fail to write Audio type for ASR to a file\n");
//
//                      }
//                      fclose(tempFile);
//  }

#endif

    memcpy((void*)psize,&size,sizeof(guint32));

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_op_mode
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_op_mode(gint32* sse_op_mode)
{
    gint  status = SSE_ERR_OK;
    guint   size = sizeof(guint);

#ifdef D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_data(sse_OpMode, 0, &size, sse_op_mode);
#elif  D_ECNR_LIB_SSE_4_03
    *sse_op_mode = ecnrHandsFree;
    unsigned short config_id = ecnr_engine_get_current_config_id();

    pr_warning("config_id -> %d \n",config_id);

    switch(config_id)
    {
#ifdef SDS_DATASET_2
    case SDS_DATASET_2:
          *sse_op_mode = ecnrVoiceRecog; break;
#endif
    default: break;
    }

    pr_warning("sse_op_mode -> %d \n",*sse_op_mode);

    (void)size;
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_asr_mode
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_sse_handler_get_asr_mode(guint* sse_asr_mode)
{
    gint  status = SSE_ERR_OK;
    guint32 size = sizeof(guint32);

#ifdef D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_data(sse_ASRMode, 0, &size, sse_asr_mode);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_data(sse_SideInfoMode, 0, &size, sse_asr_mode);
#endif

    pr_warning("sse_asr_mode -> %d \n",*sse_asr_mode);

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_set_send_mute_switch
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int  ecnr_sse_handler_set_send_mute_switch(guchar sse_send_mute_switch)
{
    int  status = SSE_ERR_OK;

    pr_warning("sse_send_mute_switch -> %d \n",sse_send_mute_switch);

#ifdef D_ECNR_LIB_SSE_3_17
    int lswitch = (int)sseOff;

    if(sse_send_mute_switch)
        lswitch = (int)sseOn;

    status = ecnr_sse_handler_set_data(sse_SendMuteSwitch, 0, sizeof(lswitch),(void*) &lswitch );
#elif  D_ECNR_LIB_SSE_4_03

    tI32 lswitch = (tI32)0;

    if(sse_send_mute_switch)
        lswitch = (tI32)1;

    status = ecnr_sse_handler_set_data(sse_SndMuteActive, 0, sizeof(lswitch),(void*) &lswitch );
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_set_send_gain_delta
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int  ecnr_sse_handler_set_send_gain_delta(int sse_send_gain_delta_lvl)
{
    int  status = SSE_ERR_OK;

#ifdef D_ECNR_LIB_SSE_3_17
    gint8 level = (gint8)sse_send_gain_delta_lvl;
    status = ecnr_sse_handler_set_data( sse_SendGainDelta, 0, sizeof(level), (void*)&level);
#elif  D_ECNR_LIB_SSE_4_03
    /* sse_send_gain_delta_lvl always given in dB hence needs to multiply by 100 */
    tI32 level = (tI32)(sse_send_gain_delta_lvl * 100);
    /* sse_SndAgcFixedPostGainDelta accepts value range from -2000 to 2000 */
    status = ecnr_sse_handler_set_data( sse_SndAgcFixedPostGainDelta, 0, sizeof(level), (void*)&level);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_set_pass_through_mode
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int  ecnr_sse_handler_set_pass_through_mode(guchar sse_pass_through_mode)
{
    int  status = SSE_ERR_OK;

#ifdef D_ECNR_LIB_SSE_3_17
    int   lmode = (int)sseNoPassThrough;

    if(sse_pass_through_mode)
        lmode = (int)ssePassThroughCopy;

    pr_message("lmode -> (%d)\n", lmode );
    status = ecnr_sse_handler_set_data(sse_PassThroughMode, 0, sizeof(lmode), (void*)&lmode );
#elif  D_ECNR_LIB_SSE_4_03

    tI32 lmode = (tI32)1; /* sseNoBypass */

    if(sse_pass_through_mode)
        lmode = (tI32)4; /* sseMultiChannelBypass */

    pr_message("lmode -> (%d)\n", lmode );
    status = ecnr_sse_handler_set_data(sse_MultiChannelBypass, 0, sizeof(lmode), (void*)&lmode );
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_set_nr_comb_floor
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int  ecnr_sse_handler_set_nr_comb_floor(int sse_nr_comb_floor)
{
    int   status = SSE_ERR_OK;

#ifdef D_ECNR_LIB_SSE_3_17
    gint16 level = (gint16)sse_nr_comb_floor;

    status = ecnr_sse_handler_set_data(sse_NRCombFloor, 0, sizeof(level), (void*)&level);
#elif  D_ECNR_LIB_SSE_4_03
    status = SSE_ERR_NO_RESULT; //TODO: rsu2kor : workaround the NR comb floor in new library.
    (void)sse_nr_comb_floor;
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_set_echo_cancellation
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
extern int  ecnr_sse_handler_set_echo_cancellation(guchar sse_aec_state)
{
    int   status = SSE_ERR_OK;

#ifdef D_ECNR_LIB_SSE_3_17
    int lmode = (int)sseOff;

    if(sse_aec_state)
        lmode = (int)sseOn;

    pr_message("lmode -> (%d)\n", lmode );
    status = ecnr_sse_handler_set_data(sse_AECSwitch, 0, sizeof(lmode), (void*)&lmode);
#elif  D_ECNR_LIB_SSE_4_03
    status = SSE_ERR_NO_RESULT ;//TODO: rsu2kor : workaround the echo cancellation mode in new library.
    (void)sse_aec_state;
#endif

   return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_set_noise_reduction
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 19.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
extern int  ecnr_sse_handler_set_noise_reduction(guchar sse_nr_state)
{
    int   status = SSE_ERR_OK;

#ifdef D_ECNR_LIB_SSE_3_17
    int lmode = (int)sseOff;

    if(sse_nr_state)
        lmode = (int)sseOn;

    pr_message("lmode -> (%d)\n", lmode );
    status = ecnr_sse_handler_set_data(sse_NRSwitch, 0, sizeof(lmode),(void*) &lmode );
#elif  D_ECNR_LIB_SSE_4_03
    status = SSE_ERR_NO_RESULT; //TODO: rsu2kor : workaround the noise reduction mode in new library.
    (void)sse_nr_state;
#endif

   return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_clip_stats
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int  ecnr_sse_handler_get_clip_stats
(
              int  channel
    ,unsigned int  DataID
    ,unsigned int* ClipFrames
    ,unsigned int* ClipCount
             ,int* MaxAmp
             ,int* MinAmp
)
{
    int  status = (int)ECNR_ERR_OK;

#ifdef  D_ECNR_LIB_SSE_3_17
    tsseClipStats ClipStats = {0};
    unsigned int        len = sizeof(ClipStats);
    status = ecnr_sse_handler_get_data(DataID, channel, &len, (void*)&ClipStats);

    *ClipFrames = ClipStats.ClipFrames;
    *ClipCount  = ClipStats.ClipCount;
    *MaxAmp     = ClipStats.MaxAmp;
    *MinAmp     = ClipStats.MinAmp;

#elif  D_ECNR_LIB_SSE_4_03
    status = SSE_ERR_NO_RESULT; //TODO: rsu2kor : workaround the clip stats in new library.
    (void)channel;
    (void)DataID;
    (void)ClipFrames;
    (void)ClipCount;
    (void)MaxAmp;
    (void)MinAmp;
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_mic_in_clip_stats
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int  ecnr_sse_handler_get_mic_in_clip_stats
(
              int  Channel
    ,unsigned int* ClipFrames
    ,unsigned int* ClipCount
             ,int* MaxAmp
             ,int* MinAmp
)
{
    int  status = (int)ECNR_ERR_OK;

#ifdef D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_clip_stats
             (
                Channel
               ,sse_MicInClipStats
               ,ClipFrames
               ,ClipCount
               ,MaxAmp
               ,MinAmp
             );
#elif  D_ECNR_LIB_SSE_4_03
    status = SSE_ERR_NO_RESULT; //TODO: rsu2kor : workaround the clip stats in new library.
    (void)Channel;
    (void)ClipFrames;
    (void)ClipCount;
    (void)MaxAmp;
    (void)MinAmp;
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_ref_in_clip_stats
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int  ecnr_sse_handler_get_ref_in_clip_stats
(
              int  Channel
    ,unsigned int* ClipFrames
    ,unsigned int* ClipCount
             ,int* MaxAmp
             ,int* MinAmp
)
{
    int  status = (int)ECNR_ERR_OK;

#ifdef D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_clip_stats
             (
                Channel
               ,sse_RefInClipStats
               ,ClipFrames
               ,ClipCount
               ,MaxAmp
               ,MinAmp
             );
#elif  D_ECNR_LIB_SSE_4_03
    status = SSE_ERR_NO_RESULT; //TODO: rsu2kor : workaround the clip stats in new library.
    (void)Channel;
    (void)ClipFrames;
    (void)ClipCount;
    (void)MaxAmp;
    (void)MinAmp;
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_rcv_in_clip_stats
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int  ecnr_sse_handler_get_rcv_in_clip_stats
(
              int  Channel
    ,unsigned int* ClipFrames
    ,unsigned int* ClipCount
             ,int* MaxAmp
             ,int* MinAmp
)
{
    int  status = (int)ECNR_ERR_OK;

#ifdef D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_clip_stats
             (
                Channel
               ,sse_RecvInClipStats
               ,ClipFrames
               ,ClipCount
               ,MaxAmp
               ,MinAmp
             );
#elif  D_ECNR_LIB_SSE_4_03
    status = SSE_ERR_NO_RESULT; //TODO: rsu2kor : workaround the clip stats in new library.
    (void)Channel;
    (void)ClipFrames;
    (void)ClipCount;
    (void)MaxAmp;
    (void)MinAmp;
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_mic_out_clip_stats
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int  ecnr_sse_handler_get_mic_out_clip_stats
(
              int  Channel
    ,unsigned int* ClipFrames
    ,unsigned int* ClipCount
             ,int* MaxAmp
             ,int* MinAmp
)
{
    int  status = (int)ECNR_ERR_OK;

#ifdef D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_clip_stats
             (
                Channel
               ,sse_MicOutClipStats
               ,ClipFrames
               ,ClipCount
               ,MaxAmp
               ,MinAmp
             );
#elif  D_ECNR_LIB_SSE_4_03
    status = SSE_ERR_NO_RESULT; //TODO: rsu2kor : workaround the clip stats in new library.
    (void)Channel;
    (void)ClipFrames;
    (void)ClipCount;
    (void)MaxAmp;
    (void)MinAmp;
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_sse_handler_get_rcv_out_clip_stats
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int  ecnr_sse_handler_get_rcv_out_clip_stats
(
              int  Channel
    ,unsigned int* ClipFrames
    ,unsigned int* ClipCount
             ,int* MaxAmp
             ,int* MinAmp
)
{
    int  status = (int)ECNR_ERR_OK;

#ifdef D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_clip_stats
             (
                Channel
               ,sse_RecvOutClipStats
               ,ClipFrames
               ,ClipCount
               ,MaxAmp
               ,MinAmp
             );
#elif  D_ECNR_LIB_SSE_4_03
    status = SSE_ERR_NO_RESULT; //TODO: rsu2kor : workaround the clip stats in new library.
    (void)Channel;
    (void)ClipFrames;
    (void)ClipCount;
    (void)MaxAmp;
    (void)MinAmp;
#endif

    return status;
}


/*******************************************************************************
* FUNCTION     : ecnr_sse_handler_get_config_data_crc
*
* DESCRIPTION  : Retrieve CRC of currently loaded configuration data (.scd) from SSE
*              : note: Valid from SSE 4.x onwards
* PARAMETER    : reference received to be copied with CRC
* RETURNVALUE  : return status from SSE
* HISTORY      :
* Date         | Author                      | Modification
* 05.02.2020   | Sudeep Chandra R(RBEI/ECO4) | Initial Revision
*******************************************************************************/
int ecnr_sse_handler_get_config_data_crc(tI32* sse_config_data_crc)
{
    gint  status = SSE_ERR_OK;
    guint size   = sizeof(tI32);

#ifdef D_ECNR_LIB_SSE_3_17
    (void)sse_config_data_crc;
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_data(sse_ScdCrc, 0, &size, sse_config_data_crc);
#endif

    return status;
}

/******************************************************************************
*
*   SSE parameter set for AA VOICE WB for GEN3ARMMAKE/GEN4RCAR (2, 205)
*
******************************************************************************/

static ecnr_sse_parameter ecnr_param_set_aa_voice_wb_pre_init[] = {
#ifdef D_ECNR_LIB_SSE_3_17
    SSE_PARM_I32_PER_VAL(sse_OpMode,            sseHandsFree),
    SSE_PARM_UI16_PER_VAL(sse_AECFiltLen,       52),
    SSE_PARM_I32_PER_VAL(sse_MicMaxLatency,     30),
    SSE_PARM_I32_PER_VAL(sse_RefMaxLatency,     30),
    SSE_PARM_I32_PER_VAL(sse_BFSwitchCD,        sseOff),
    SSE_PARM_I32_PER_VAL(sse_MIXSwitch,         sseOff),
    SSE_PARM_I32_PER_VAL(sse_SpeakerLocMode,    sseSpeakerLocOff),
    SSE_PARM_I32_PER_VAL(sse_RECVAdaptEQSwitch, sseOff),
    SSE_PARM_UI16_PER_VAL(sse_MicInCnt,         1),
    SSE_PARM_UI16_PER_VAL(sse_RefInCnt,         1),
    SSE_PARM_UI16_PER_VAL(sse_RecvInCnt,        0),
    SSE_PARM_UI16_PER_VAL(sse_SampleRate,       16000),
    SSE_PARM_UI16_PER_VAL(sse_FrameShift,       128),
    SSE_PARM_UI16_PER_VAL(sse_MaxFreqProc,      8000),
    SSE_PARM_I32_PER_VAL(sse_NRSwitch,          sseOn),
    SSE_PARM_I32_PER_VAL(sse_RECVAgcSwitch,     sseOff),
    SSE_PARM_I32_PER_VAL(sse_RECVBweSwitch,     sseOff),
    SSE_PARM_I32_PER_VAL(sse_AECSwitch,         sseOn),
    SSE_PARM_I32_PER_VAL(sse_RECVDtfSwitch,     sseOff),
    SSE_PARM_I32_PER_VAL(sse_RAESwitch,         sseOn),
    SSE_PARM_I32_PER_VAL(sse_SendAGCSwitch,     sseOff),
    SSE_PARM_I32_PER_VAL(sse_WNDBSwitch,        sseOff),
    SSE_PARM_I32_PER_VAL(sse_SpPfSwitch,        sseOff),
    SSE_PARM_I32_PER_VAL(sse_SendEQSwitch,      sseDisabled),
    SSE_PARM_I32_PER_VAL(sse_RoadBumpSupSwitch, sseOff),
    SSE_PARM_I32_PER_VAL(sse_DIAGSwitch,        sseOn)
#elif   D_ECNR_LIB_SSE_4_03
    SSE_PARM_UI32_PER_VAL(sse_MicInCnt,         1),
    SSE_PARM_UI32_PER_VAL(sse_RefInCnt,         1),
    SSE_PARM_UI32_PER_VAL(sse_RcvInCnt,         0),
    SSE_PARM_UI32_PER_VAL(sse_SampleRate,       16000),
    SSE_PARM_UI32_PER_VAL(sse_FrameShift,       256)
#endif
};
#define ECNR_SSE_PARAM_SET_AA_VOICE_WB_PRE  ARRAYSIZE(ecnr_param_set_aa_voice_wb_pre_init)

static ecnr_sse_parameter ecnr_param_set_aa_voice_wb_post_init[] = {
#ifdef D_ECNR_LIB_SSE_3_17
    SSE_PARM_UI16_PER_VAL(sse_SendGain,    3500),
    SSE_PARM_I8_PER_VAL(sse_SendGainDelta, 0)
#elif  D_ECNR_LIB_SSE_4_03
    //TODO: rsu2kor : enable post initialization parameters for new library
#endif
};
#define ECNR_SSE_PARAM_SET_AA_VOICE_WB_POST  ARRAYSIZE(ecnr_param_set_aa_voice_wb_post_init)

int ecnr_sse_handler_set_config_aa_voice_wb_pre (void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_load_configuration_set
          (
                  ecnr_param_set_aa_voice_wb_pre_init
                 ,ECNR_SSE_PARAM_SET_AA_VOICE_WB_PRE
          );
    pr_message("set up paramter set ecnr_param_set_aap_vr_pre_init\n");
    ETG_TRACE_USR3(("[ecnr_sse_handler_set_config_aa_voice_wb_pre]: set up parameter set ecnr_param_set_aap_vr_pre_init"));

    return err;
}

int ecnr_sse_handler_set_config_aa_voice_wb_post(void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_load_configuration_set
          (
                  ecnr_param_set_aa_voice_wb_post_init
                 ,ECNR_SSE_PARAM_SET_AA_VOICE_WB_POST
          );
    pr_message("set up paramter set ecnr_param_set_aap_vr_post_init\n");
    ETG_TRACE_USR3(("[ecnr_sse_handler_set_config_aa_voice_wb_post]: set up parameter set ecnr_param_set_aap_vr_post_init"));

    return err;
}






/*******************************************************************************
*
*   SSE parameter set for BT PHONE NB for LSIM (1, 1)
*
*******************************************************************************/

static ecnr_sse_parameter ecnr_param_set_bt_phone_nb_lsim_pre_init[] = {
#ifdef D_ECNR_LIB_SSE_3_17
    SSE_PARM_UI16_PER_VAL(sse_MicInCnt,     1),
    SSE_PARM_UI16_PER_VAL(sse_RefInCnt,     0),
    SSE_PARM_UI16_PER_VAL(sse_RecvInCnt,    1),
    SSE_PARM_UI16_PER_VAL(sse_SampleRate,   8000),
    SSE_PARM_UI16_PER_VAL(sse_FrameShift,   64)
#elif  D_ECNR_LIB_SSE_4_03
    SSE_PARM_UI32_PER_VAL(sse_MicInCnt,         1),
    SSE_PARM_UI32_PER_VAL(sse_RefInCnt,         0),
    SSE_PARM_UI32_PER_VAL(sse_RcvInCnt,         1),
    SSE_PARM_UI32_PER_VAL(sse_SampleRate,       8000),
    SSE_PARM_UI32_PER_VAL(sse_FrameShift,       128)
#endif
};
#define ECNR_SSE_PARAM_SET_BT_PHONE_NB_LSIM_PRE  ARRAYSIZE(ecnr_param_set_bt_phone_nb_lsim_pre_init)


static ecnr_sse_parameter ecnr_param_set_bt_phone_nb_lsim_post_init[] = {
#ifdef D_ECNR_LIB_SSE_3_17
    SSE_PARM_UI16_PER_VAL(sse_SendGain,    3500),
    SSE_PARM_I8_PER_VAL(sse_SendGainDelta, 0)
#elif  D_ECNR_LIB_SSE_4_03
    //TODO: rsu2kor : enable post initialization parameters for new library
#endif
};
#define ECNR_SSE_PARAM_SET_INT_VR_LSIM_POST  ARRAYSIZE(ecnr_param_set_bt_phone_nb_lsim_post_init)

int ecnr_sse_handler_set_config_bt_phone_nb_lsim_pre(void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_load_configuration_set
          (
                  ecnr_param_set_bt_phone_nb_lsim_pre_init
                 ,ECNR_SSE_PARAM_SET_BT_PHONE_NB_LSIM_PRE
          );
    pr_message("set up paramter set ecnr_param_set_bt_phone_nb_lsim_pre_init\n");
    ETG_TRACE_USR3(("[ecnr_sse_handler_set_config_bt_phone_nb_lsim_pre]: set up parameter set ecnr_param_set_bt_phone_nb_lsim_pre_init"));

    return err;
}

int ecnr_sse_handler_set_config_bt_phone_nb_lsim_post(void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_load_configuration_set
          (
                  ecnr_param_set_bt_phone_nb_lsim_post_init
                 ,ECNR_SSE_PARAM_SET_INT_VR_LSIM_POST
          );
    pr_message("set up paramter set ecnr_param_set_bt_phone_nb_lsim_post_init\n");
    ETG_TRACE_USR3(("[ecnr_sse_handler_set_config_bt_phone_nb_lsim_post]: set up parameter set ecnr_param_set_bt_phone_nb_lsim_post_init"));

    return err;
}


/*******************************************************************************
*
*   SSE parameter set for BT PHONE WB for LSIM (1, 2)
*
*******************************************************************************/

static ecnr_sse_parameter ecnr_param_set_bt_phone_wb_lsim_pre_init[] = {
#ifdef D_ECNR_LIB_SSE_3_17
    SSE_PARM_UI16_PER_VAL(sse_MicInCnt,     1),
    SSE_PARM_UI16_PER_VAL(sse_RefInCnt,     0),
    SSE_PARM_UI16_PER_VAL(sse_RecvInCnt,    1),
    SSE_PARM_UI16_PER_VAL(sse_SampleRate,   16000),
    SSE_PARM_UI16_PER_VAL(sse_FrameShift,   128)
#elif  D_ECNR_LIB_SSE_4_03
    SSE_PARM_UI32_PER_VAL(sse_MicInCnt,         1),
    SSE_PARM_UI32_PER_VAL(sse_RefInCnt,         0),
    SSE_PARM_UI32_PER_VAL(sse_RcvInCnt,         1),
    SSE_PARM_UI32_PER_VAL(sse_SampleRate,       16000),
    SSE_PARM_UI32_PER_VAL(sse_FrameShift,       256)
#endif
};
#define ECNR_SSE_PARAM_SET_BT_PHONE_WB_LSIM_PRE  ARRAYSIZE(ecnr_param_set_bt_phone_wb_lsim_pre_init)


static ecnr_sse_parameter ecnr_param_set_bt_phone_wb_lsim_post_init[] = {
#ifdef D_ECNR_LIB_SSE_3_17
    SSE_PARM_UI16_PER_VAL(sse_SendGain,     3500),
    SSE_PARM_I8_PER_VAL(sse_SendGainDelta,  0)
#elif  D_ECNR_LIB_SSE_4_03
    //TODO: rsu2kor : enable post initialization parameters for new library
#endif
};
#define ECNR_SSE_PARAM_SET_BT_PHONE_WB_LSIM_POST  ARRAYSIZE(ecnr_param_set_bt_phone_wb_lsim_post_init)

int ecnr_sse_handler_set_config_bt_phone_wb_lsim_pre(void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_load_configuration_set
          (
                  ecnr_param_set_bt_phone_wb_lsim_pre_init
                 ,ECNR_SSE_PARAM_SET_BT_PHONE_WB_LSIM_PRE
          );
    pr_message("set up paramter set ecnr_param_set_bt_phone_wb_lsim_pre_init\n");
    ETG_TRACE_USR2(("[ecnr_sse_configuration_set_bt_phone_wb_lsim_pre]: set up parameter set ecnr_param_set_bt_phone_wb_lsim_pre_init"));

    return err;
}

int ecnr_sse_handler_set_config_bt_phone_wb_lsim_post(void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_load_configuration_set
          (
                  ecnr_param_set_bt_phone_wb_lsim_post_init
                 ,ECNR_SSE_PARAM_SET_BT_PHONE_WB_LSIM_POST
          );
    pr_message("set up paramter set ecnr_param_set_bt_phone_wb_lsim_post_init\n");
    ETG_TRACE_USR2(("[ecnr_sse_configuration_set_bt_phone_wb_lsim_post]: set up parameter set ecnr_param_set_bt_phone_wb_lsim_post_init"));

    return err;
}

/*******************************************************************************
*
*   SSE parameter set for CP PHONE NB  for LSIM (2, 201)
*
*******************************************************************************/

static ecnr_sse_parameter ecnr_param_set_cp_phone_nb_lsim_pre_init[] = {
#ifdef D_ECNR_LIB_SSE_3_17
    SSE_PARM_UI16_PER_VAL(sse_MicInCnt,     1),
    SSE_PARM_UI16_PER_VAL(sse_RefInCnt,     0),
    SSE_PARM_UI16_PER_VAL(sse_RecvInCnt,    1),
    SSE_PARM_UI16_PER_VAL(sse_SampleRate,   8000),
    SSE_PARM_UI16_PER_VAL(sse_FrameShift,   128)
#elif  D_ECNR_LIB_SSE_4_03
    SSE_PARM_UI32_PER_VAL(sse_MicInCnt,         1),
    SSE_PARM_UI32_PER_VAL(sse_RefInCnt,         0),
    SSE_PARM_UI32_PER_VAL(sse_RcvInCnt,         1),
    SSE_PARM_UI32_PER_VAL(sse_SampleRate,       8000),
    SSE_PARM_UI32_PER_VAL(sse_FrameShift,       256)
#endif
};
#define ECNR_SSE_PARAM_SET_CP_PHONE_NB_LSIM_PRE  ARRAYSIZE(ecnr_param_set_cp_phone_nb_lsim_pre_init)


static ecnr_sse_parameter ecnr_param_set_cp_phone_nb_lsim_post_init[] = {
#ifdef D_ECNR_LIB_SSE_3_17
    SSE_PARM_UI16_PER_VAL(sse_SendGain,      3500),
    SSE_PARM_I8_PER_VAL(sse_SendGainDelta,   0)
#elif  D_ECNR_LIB_SSE_4_03
    //TODO: rsu2kor : enable post initialization parameters for new library
#endif
};
#define ECNR_SSE_PARAM_SET_CP_PHONE_NB_LSIM_POST  ARRAYSIZE(ecnr_param_set_cp_phone_nb_lsim_post_init)

int ecnr_sse_handler_set_config_cp_phone_nb_lsim_pre(void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_load_configuration_set
          (
                  ecnr_param_set_cp_phone_nb_lsim_pre_init
                 ,ECNR_SSE_PARAM_SET_CP_PHONE_NB_LSIM_PRE
          );
    pr_message("set up paramter set ecnr_param_set_cp_phone_nb_lsim_pre_init\n");
    ETG_TRACE_USR2(("[ecnr_sse_handler_set_config_cp_phone_nb_lsim_pre]: set up parameter set ecnr_param_set_cp_phone_nb_lsim_pre_init"));

    return err;
}

int ecnr_sse_handler_set_config_cp_phone_nb_lsim_post(void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_load_configuration_set
          (
                  ecnr_param_set_cp_phone_nb_lsim_post_init
                 ,ECNR_SSE_PARAM_SET_CP_PHONE_NB_LSIM_POST
          );
    pr_message("set up paramter set ecnr_param_set_cp_phone_nb_lsim_post_init\n");
    ETG_TRACE_USR2(("[ecnr_sse_handler_set_config_cp_phone_nb_lsim_post]: set up parameter set ecnr_param_set_cp_phone_nb_lsim_post_init"));

    return err;
}

/*******************************************************************************
*
*   SSE parameter set for CP PHONE WB LSIM (2,202)
*
*******************************************************************************/

static ecnr_sse_parameter ecnr_param_set_cp_phone_wb_lsim_pre_init[] = {
#ifdef D_ECNR_LIB_SSE_3_17
    SSE_PARM_UI16_PER_VAL(sse_MicInCnt,     1),
    SSE_PARM_UI16_PER_VAL(sse_RefInCnt,     0),
    SSE_PARM_UI16_PER_VAL(sse_RecvInCnt,    1),
    SSE_PARM_UI16_PER_VAL(sse_SampleRate,   16000),
    SSE_PARM_UI16_PER_VAL(sse_FrameShift,   256)
#elif  D_ECNR_LIB_SSE_4_03
    SSE_PARM_UI32_PER_VAL(sse_MicInCnt,         1),
    SSE_PARM_UI32_PER_VAL(sse_RefInCnt,         0),
    SSE_PARM_UI32_PER_VAL(sse_RcvInCnt,         1),
    SSE_PARM_UI32_PER_VAL(sse_SampleRate,       16000),
    SSE_PARM_UI32_PER_VAL(sse_FrameShift,       256)
#endif
};
#define ECNR_SSE_PARAM_SET_CP_PHONE_WB_LSIM_PRE  ARRAYSIZE(ecnr_param_set_cp_phone_wb_lsim_pre_init)

static ecnr_sse_parameter ecnr_param_set_cp_phone_wb_lsim_post_init[] = {
#ifdef D_ECNR_LIB_SSE_3_17
    SSE_PARM_UI16_PER_VAL(sse_SendGain, 3500),
    SSE_PARM_I8_PER_VAL(sse_SendGainDelta, 0)
#elif  D_ECNR_LIB_SSE_4_03
    //TODO: rsu2kor : enable post initialization parameters for new library
#endif
};
#define ECNR_SSE_PARAM_SET_CP_PHONE_WB_LSIM_POST  ARRAYSIZE(ecnr_param_set_cp_phone_wb_lsim_post_init)

int ecnr_sse_handler_set_config_cp_phone_wb_lsim_pre(void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_load_configuration_set
          (
                  ecnr_param_set_cp_phone_wb_lsim_pre_init
                 ,ECNR_SSE_PARAM_SET_CP_PHONE_WB_LSIM_PRE
          );
    pr_message("set up paramter set ecnr_param_set_cp_phone_wb_lsim_pre_init\n");
    ETG_TRACE_USR2(("[ecnr_sse_handler_set_config_cp_phone_wb_lsim_pre]: set up parameter set ecnr_param_set_cp_phone_wb_lsim_pre_init"));

    return err;
}

int ecnr_sse_handler_set_config_cp_phone_wb_lsim_post(void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_load_configuration_set
          (
                  ecnr_param_set_cp_phone_wb_lsim_post_init
                 ,ECNR_SSE_PARAM_SET_CP_PHONE_WB_LSIM_POST
          );
    pr_message("set up paramter set ecnr_param_set_cp_phone_wb_lsim_post_init\n");
    ETG_TRACE_USR2(("[ecnr_sse_handler_set_config_cp_phone_wb_lsim_post]: set up parameter set ecnr_param_set_cp_phone_wb_lsim_post_init"));

    return err;
}

/******************************************************************************
*
*   SSE parameter set for CP VOICE WB for LSIM (2, 204)
*
******************************************************************************/

static ecnr_sse_parameter ecnr_param_set_cp_voice_wb_lsim_pre_init[] = {
#ifdef D_ECNR_LIB_SSE_3_17
    SSE_PARM_UI16_PER_VAL(sse_MicInCnt,     1),
    SSE_PARM_UI16_PER_VAL(sse_RefInCnt,     0),
    SSE_PARM_UI16_PER_VAL(sse_RecvInCnt,    1),
    SSE_PARM_UI16_PER_VAL(sse_SampleRate,   16000),
    SSE_PARM_UI16_PER_VAL(sse_FrameShift,   256)
#elif  D_ECNR_LIB_SSE_4_03
    SSE_PARM_UI32_PER_VAL(sse_MicInCnt,         1),
    SSE_PARM_UI32_PER_VAL(sse_RefInCnt,         0),
    SSE_PARM_UI32_PER_VAL(sse_RcvInCnt,         1),
    SSE_PARM_UI32_PER_VAL(sse_SampleRate,       16000),
    SSE_PARM_UI32_PER_VAL(sse_FrameShift,       256)
#endif
};
#define ECNR_SSE_PARAM_SET_CP_VOICE_WB_LSIM_PRE  ARRAYSIZE(ecnr_param_set_cp_voice_wb_lsim_pre_init)


static ecnr_sse_parameter ecnr_param_set_cp_voice_wb_lsim_post_init[] = {
#ifdef D_ECNR_LIB_SSE_3_17
    SSE_PARM_UI16_PER_VAL(sse_SendGain,    3500),
    SSE_PARM_I8_PER_VAL(sse_SendGainDelta, 0)
#elif  D_ECNR_LIB_SSE_4_03
    //TODO: rsu2kor : enable post initialization parameters for new library
#endif
};
#define ECNR_SSE_PARAM_SET_CP_VOICE_WB_LSIM_POST  ARRAYSIZE(ecnr_param_set_cp_voice_wb_lsim_post_init)

int ecnr_sse_handler_set_config_cp_voice_wb_lsim_pre(void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_load_configuration_set
          (
                  ecnr_param_set_cp_voice_wb_lsim_pre_init
                 ,ECNR_SSE_PARAM_SET_CP_VOICE_WB_LSIM_PRE
          );
    pr_message("set up paramter set ecnr_param_set_cp_voice_wb_lsim_pre_init\n");
    ETG_TRACE_USR2(("[ecnr_sse_handler_set_config_cp_voice_wb_lsim_pre]: set up parameter set ecnr_param_set_cp_voice_wb_lsim_pre_init"));

    return err;
}

int ecnr_sse_handler_set_config_cp_voice_wb_lsim_post(void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_load_configuration_set
          (
                  ecnr_param_set_cp_voice_wb_lsim_post_init
                 ,ECNR_SSE_PARAM_SET_CP_VOICE_WB_LSIM_POST
          );
    pr_message("set up paramter set ecnr_param_set_cp_voice_wb_lsim_post_init\n");
    ETG_TRACE_USR2(("[ecnr_sse_handler_set_config_cp_voice_wb_lsim_post]: set up parameter set ecnr_param_set_cp_voice_wb_lsim_post_init"));

    return err;
}

/******************************************************************************
*
*   SSE parameter set for AA VOICE WB for LSIM (2, 205)
*
******************************************************************************/

static ecnr_sse_parameter ecnr_param_set_aa_voice_wb_lsim_pre_init[] = {
#ifdef D_ECNR_LIB_SSE_3_17
    SSE_PARM_I32_PER_VAL(sse_OpMode,            sseHandsFree),
    SSE_PARM_UI16_PER_VAL(sse_AECFiltLen,       52),
    SSE_PARM_I32_PER_VAL(sse_MicMaxLatency,     30),
    SSE_PARM_I32_PER_VAL(sse_RefMaxLatency,     30),
    SSE_PARM_I32_PER_VAL(sse_BFSwitchCD,        sseOff),
    SSE_PARM_I32_PER_VAL(sse_MIXSwitch,         sseOff),
    SSE_PARM_I32_PER_VAL(sse_SpeakerLocMode,    sseSpeakerLocOff),
    SSE_PARM_I32_PER_VAL(sse_RECVAdaptEQSwitch, sseOff),
    SSE_PARM_UI16_PER_VAL(sse_MicInCnt,         1),
    SSE_PARM_UI16_PER_VAL(sse_RefInCnt,         0),
    SSE_PARM_UI16_PER_VAL(sse_RecvInCnt,        0),
    SSE_PARM_UI16_PER_VAL(sse_SampleRate,       16000),
    SSE_PARM_UI16_PER_VAL(sse_FrameShift,       128),
    SSE_PARM_UI16_PER_VAL(sse_MaxFreqProc,      8000),
    SSE_PARM_I32_PER_VAL(sse_NRSwitch,          sseOn),
    SSE_PARM_I32_PER_VAL(sse_RECVAgcSwitch,     sseOff),
    SSE_PARM_I32_PER_VAL(sse_RECVBweSwitch,     sseOff),
    SSE_PARM_I32_PER_VAL(sse_AECSwitch,         sseOn),
    SSE_PARM_I32_PER_VAL(sse_RECVDtfSwitch,     sseOff),
    SSE_PARM_I32_PER_VAL(sse_RAESwitch,         sseOn),
    SSE_PARM_I32_PER_VAL(sse_SendAGCSwitch,     sseOff),
    SSE_PARM_I32_PER_VAL(sse_WNDBSwitch,        sseOff),
    SSE_PARM_I32_PER_VAL(sse_SpPfSwitch,        sseOff),
    SSE_PARM_I32_PER_VAL(sse_SendEQSwitch,      sseDisabled),
    SSE_PARM_I32_PER_VAL(sse_RoadBumpSupSwitch, sseOff),
    SSE_PARM_I32_PER_VAL(sse_DIAGSwitch,        sseOn)
#elif   D_ECNR_LIB_SSE_4_03
    SSE_PARM_UI32_PER_VAL(sse_MicInCnt,         1),
    SSE_PARM_UI32_PER_VAL(sse_RefInCnt,         0),
    SSE_PARM_UI32_PER_VAL(sse_RcvInCnt,         0),
    SSE_PARM_UI32_PER_VAL(sse_SampleRate,       16000),
    SSE_PARM_UI32_PER_VAL(sse_FrameShift,       256)
#endif
};
#define ECNR_SSE_PARAM_SET_AA_VOICE_WB_LSIM_PRE  ARRAYSIZE(ecnr_param_set_aa_voice_wb_lsim_pre_init)

static ecnr_sse_parameter ecnr_param_set_aa_voice_wb_lsim_post_init[] = {
#ifdef D_ECNR_LIB_SSE_3_17
    SSE_PARM_UI16_PER_VAL(sse_SendGain,    3500),
    SSE_PARM_I8_PER_VAL(sse_SendGainDelta, 0)
#elif  D_ECNR_LIB_SSE_4_03
    //TODO: rsu2kor : enable post initialization parameters for new library
#endif
};
#define ECNR_SSE_PARAM_SET_AA_VOICE_WB_LSIM_POST  ARRAYSIZE(ecnr_param_set_aa_voice_wb_lsim_post_init)

int ecnr_sse_handler_set_config_aa_voice_wb_lsim_pre (void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_load_configuration_set
          (
                  ecnr_param_set_aa_voice_wb_lsim_pre_init
                 ,ECNR_SSE_PARAM_SET_AA_VOICE_WB_LSIM_PRE
          );
    pr_message("set up paramter set ecnr_param_set_aa_voice_wb_lsim_pre_init\n");
    ETG_TRACE_USR3(("[ecnr_sse_handler_set_config_aa_voice_wb_lsim_pre]: set up parameter set ecnr_param_set_aa_voice_wb_lsim_pre_init"));

    return err;
}

int ecnr_sse_handler_set_config_aa_voice_wb_lsim_post(void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_load_configuration_set
          (
                  ecnr_param_set_aa_voice_wb_lsim_post_init
                 ,ECNR_SSE_PARAM_SET_AA_VOICE_WB_LSIM_POST
          );
    pr_message("set up paramter set ecnr_param_set_aa_voice_wb_lsim_post_init\n");
    ETG_TRACE_USR3(("[ecnr_sse_handler_set_config_aa_voice_wb_lsim_post]: set up parameter set ecnr_param_set_aa_voice_wb_lsim_post_init"));

    return err;
}

/******************************************************************************
*
*   SSE parameter set for INT VOICE WB for LSIM (3,402)
*
******************************************************************************/

static ecnr_sse_parameter ecnr_param_set_int_voice_wb_lsim_pre_init[] = {
#ifdef D_ECNR_LIB_SSE_3_17
    SSE_PARM_UI16_PER_VAL(sse_MicInCnt,     1),
    SSE_PARM_UI16_PER_VAL(sse_RefInCnt,     0),
    SSE_PARM_UI16_PER_VAL(sse_RecvInCnt,    0),
    SSE_PARM_UI16_PER_VAL(sse_SampleRate,   16000),
    SSE_PARM_UI16_PER_VAL(sse_FrameShift,   160)
#elif  D_ECNR_LIB_SSE_4_03
    SSE_PARM_UI32_PER_VAL(sse_MicInCnt,         1),
    SSE_PARM_UI32_PER_VAL(sse_RefInCnt,         0),
    SSE_PARM_UI32_PER_VAL(sse_RcvInCnt,         0),
    SSE_PARM_UI32_PER_VAL(sse_SampleRate,       16000),
    SSE_PARM_UI32_PER_VAL(sse_FrameShift,       160)
#endif
};
#define ECNR_SSE_PARAM_SET_INT_VOICE_WB_PRE  ARRAYSIZE(ecnr_param_set_int_voice_wb_lsim_pre_init)

static ecnr_sse_parameter ecnr_param_set_int_voice_wb_lsim_post_init[] = {
#ifdef D_ECNR_LIB_SSE_3_17
    SSE_PARM_UI16_PER_VAL(sse_SendGain,    3500),
    SSE_PARM_I8_PER_VAL(sse_SendGainDelta, 0)
#elif  D_ECNR_LIB_SSE_4_03
    //TODO: rsu2kor : enable post initialization parameters for new library
#endif
};
#define ECNR_SSE_PARAM_SET_INT_VOICE_WB_POST  ARRAYSIZE(ecnr_param_set_int_voice_wb_lsim_post_init)

int ecnr_sse_handler_set_config_int_voice_wb_lsim_pre(void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_load_configuration_set
          (
                  ecnr_param_set_int_voice_wb_lsim_pre_init
                 ,ECNR_SSE_PARAM_SET_INT_VOICE_WB_PRE
          );
    pr_message("set up paramter set ecnr_param_set_sds_vr_lsim_pre_init\n");
    ETG_TRACE_USR3(("[ecnr_sse_handler_set_config_int_voice_wb_lsim_pre]: set up parameter set ecnr_param_set_sds_vr_lsim_pre_init"));

    return err;
}

int ecnr_sse_handler_set_config_int_voice_wb_lsim_post(void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_load_configuration_set
          (
                  ecnr_param_set_int_voice_wb_lsim_post_init
                 ,ECNR_SSE_PARAM_SET_INT_VOICE_WB_POST
          );
    pr_message("set up paramter set ecnr_param_set_sds_vr_lsim_post_init\n");
    ETG_TRACE_USR3(("[ecnr_sse_handler_set_config_int_voice_wb_lsim_post]: set up parameter set ecnr_param_set_sds_vr_lsim_post_init"));

    return err;
}

#endif //#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX

