/******************************************************************************
 * FILE        : ecnr-sse-debug-interface.c
 * PROJECT     : Gen3 and Gen4
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine:
 *               this module provides a tuning interface via TCP/IP for the
 *               tuning tool from NUANCE
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2013 RBCM GMBH
 * HISTORY     :
 * Date        | Author                 | Modification
 * 16.10.2013  | Patrick Rey            | initial version
 *             |                        | SSE 3.X engine tuning handler
 * 08.03.2019  | Ranjit Susal(RBEI/ECO) | SSE 4.X engine tuning handler
 *****************************************************************************/

#ifdef D_ECNR_LIB_SSE_3_17
#include "sse.h"
#include "sse_err.h"
#endif

#ifdef D_ECNR_LIB_SSE_4_03
#include "sse.h"
#include "sse_param.h"
#endif

#include "ecnr-types.h"
#include "ecnr-error.h"
#include "ecnr-common-defs.h"
#include "ecnr-object.h"
#include "ecnr-service.h"
#include "ecnr-alsa.h"
#include "ecnr-engine-handler.h"
#include "ecnr-sse-handler.h"
#include "ecnr-configuration.h"
#include "ecnr-ringbuffer.h"

#define ECNR_S_IMPORT_INTERFACE_DP
#include "ecnr-datapool-default.h"
#include "ecnr-configuration-data.h"

#include <pthread.h>
#include <semaphore.h>

#include <netdb.h>
#include <unistd.h>
#include <sys/socket.h>

#include <alsa/asoundlib.h>


/*******************************************************************************
              VERSION COMPILER
*******************************************************************************/



/*******************************************************************************
              DEFINES
*******************************************************************************/


/*******************************************************************************
              Constant definition
*******************************************************************************/

/*******************************************************************************
               SOCKET BASED VARIABLES
*******************************************************************************/

/* debug interface structure */
typedef struct _debug_interface
{
    tsRingbuffer*   pRxBuffer;              //Reception Buffer
    tsRingbuffer*   pTxBuffer;              //Transmission Buffer
    GThread *       tid_RxThread;
    gboolean        exitRxThread;
    int             iServerSocket;          //socket server
    int             iSocketfd;
    gboolean        clientConnected;
    sem_t*          phTxSema;
}sDebug_interface;


static tecnrTuneIO *pTuneIO;

sDebug_interface*   pDebug_interface    = NULL;       //debug interface pointer
static          gboolean IsDebugActive  = FALSE;
static char     SseDebugBufferIn[BUFFER_LENGTH];
static char     SseDebugBufferOut[BUFFER_LENGTH];

static gboolean isDebugReadEntered      = FALSE;
static gboolean isDebugWriteEntered     = FALSE;
static gboolean isTraceReadNullBuffer_1 = FALSE;
static gboolean isTraceReadNullBuffer_2 = FALSE;
static gboolean isTraceWriteNullBuffer  = FALSE;



/*******************************************************************************
               GLOBAL VARIABLES
*******************************************************************************/
int global = 0, recglobal= 0;

/*******************************************************************************
               STATIC FUNCTIONS
*******************************************************************************/




/*******************************************************************************
*
* FUNCTION: ecnr_sse_allocate_Debug_Buffer
* DESCRIPTION:
*
* PARAMETER: [check xml for details]
*
* RETURNVALUE: int
*
*******************************************************************************/
static int ecnr_sse_allocate_Debug_Buffer(void)
{
    int err = ECNR_ERR_OK;

    pr_debug("ENTERED");

    if(!pTuneIO)
    {

        pTuneIO = g_malloc0(sizeof(tecnrTuneIO));

        if(!pTuneIO)
        {
            err = ECNR_ERR_ALLOC;
            pr_debug(" g_malloc0 to pTuneIO failed");
        }
    }

    pr_debug("EXIT");
    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_sse_free_Debug_Buffer
* DESCRIPTION:
*
* PARAMETER: [check xml for details]
*
* RETURNVALUE: int
*
*******************************************************************************/
static int ecnr_sse_free_Debug_Buffer(void)
{
    int ret = ECNR_ERR_OK;

    pr_debug("ENTERED");

    if(pTuneIO)
    {
        if(pTuneIO)
            g_free(pTuneIO);
        pTuneIO=NULL;
    }

    pr_debug("EXIT");
    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_sse_Debug_RxThread
*
* DESCRIPTION: receive data thread function from socket
*
* PARAMETER: None
*
* RETURNVALUE: void.
*
*******************************************************************************/

static void* ecnr_sse_Debug_RxThread(void* pvArg)
{
    int err = ECNR_ERR_OK;
    (void)pvArg;
    char pcRxSocketBuffer[BUFFER_LENGTH];
    ssize_t iBytesReceived = 0;
    struct sockaddr_in cli_addr; /*lint 64 prio1*/
    socklen_t clilen;

    pr_debug("Entered");

    if (!pDebug_interface)
    {
        pr_debug( "pDebug_interface null pointer" );
        return NULL;
    }

    err = listen(pDebug_interface->iServerSocket,iALLOW_ONLY_ONE_SOCKET_CONNECTION );

    if(err)
    {
        pr_warning("socket listen failed with error: %d", err);
        return (void*)NULL;
    }

    while (FALSE == pDebug_interface->exitRxThread)
    {
        if (FALSE == pDebug_interface->clientConnected)
        {
            err = listen(pDebug_interface->iServerSocket,iALLOW_ONLY_ONE_SOCKET_CONNECTION );

            if(err)
            {
                pr_warning("socket listen failed with error: %d", err);
                pDebug_interface->exitRxThread = TRUE;
            }

            if (FALSE == pDebug_interface->exitRxThread)
            {
                pr_message("Waiting for connection on port %i\n", iIPPORT );

                /* accept(...) is non blocking*/
                clilen = sizeof(cli_addr);
                pDebug_interface->iSocketfd = accept(pDebug_interface->iServerSocket, (struct sockaddr *)&cli_addr, &clilen);

                if ( iINVALID_SOCKET == pDebug_interface->iSocketfd)
                {
                    pr_warning("socket accept() failed" );
                    pDebug_interface->exitRxThread = TRUE;
                }

                else
                {
                    pr_debug("accepted the connection here \n");
                    pDebug_interface->clientConnected = TRUE;
                }
            }
        }

        if ( TRUE == pDebug_interface->clientConnected)
        {
            /* recv(...) is blocking until some data from the client is available */
            iBytesReceived = recv( pDebug_interface->iSocketfd, pcRxSocketBuffer, BUFFER_LENGTH,0);
            if ( 0 >= iBytesReceived )
            {
                pr_warning( "recv() failed. Try new connection" );
                pDebug_interface->clientConnected = FALSE;
            }
            else
            {
                //recglobal+=iBytesReceived;
                pr_message("received %zd byte from Host PC ", iBytesReceived );
                ecnr_ring_buffer_data_insert(pDebug_interface->pRxBuffer,pcRxSocketBuffer,(int)iBytesReceived);
            }
        }
    }

    if(pDebug_interface->exitRxThread)
        pr_debug("exit thread");

    return pDebug_interface;
}




/*******************************************************************************
*
* FUNCTION: ecnr_sse_debug_interface_create
* DESCRIPTION: creates all buffers
*
* PARAMETER: None
*
* RETURNVALUE: int
*
*******************************************************************************/
static int ecnr_sse_debug_interface_create(void)
{
    int err = ECNR_ERR_OK;

    pr_debug("Entered");

    pDebug_interface = (sDebug_interface*)g_malloc0(sizeof(sDebug_interface));

    if(!pDebug_interface)
    {
        pr_debug("Error while allocation debug interface");
        return ECNR_ERR_ALLOC;
    }

    /* create the rx and tx ringbuffers */
    pDebug_interface->pRxBuffer = NULL;
    pDebug_interface->pTxBuffer = NULL;

    err = ecnr_ring_buffer_create( &(pDebug_interface->pRxBuffer), iRX_RINGBUFFER_SIZE );

    if (err )
    {
        pr_debug( "createRingbuffer() failed for rx");
        return (err);
    }

    err = ecnr_ring_buffer_create( &(pDebug_interface->pTxBuffer), iTX_RINGBUFFER_SIZE );

    if (err )
    {
        pr_debug( "createRingbuffer() failed for tx");
        return (err);
    }

    /* initialize debug interface data */
    pDebug_interface->iServerSocket = 0;
    pDebug_interface->iSocketfd = 0;
    pDebug_interface->exitRxThread = FALSE;
    pDebug_interface->tid_RxThread = (GThread *)NULL;
    pDebug_interface->clientConnected= FALSE;

    pr_debug("EXIT");
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_debug_interface_CreateServerSocket
* DESCRIPTION: creates server socket
*
* PARAMETER: None
*
* RETURNVALUE: int
*
*******************************************************************************/

static unsigned int  ecnr_sse_debug_interface_CreateServerSocket(void)
{
    struct protoent * psProtoTcp;
    struct sockaddr_in sIpAddr;
    int err = ECNR_ERR_OK;

    pr_debug("ENTERED\n");

    sIpAddr.sin_family = AF_INET;
    sIpAddr.sin_addr.s_addr = INADDR_ANY; /* or localhost adress  inet_addr("172.17.0.5");*/
    sIpAddr.sin_port = htons( (short)iIPPORT );
    memset(&sIpAddr.sin_zero, 0, sizeof(sIpAddr.sin_zero));

    psProtoTcp = getprotobyname("tcp");
    pDebug_interface->iServerSocket = socket(AF_INET, SOCK_STREAM, psProtoTcp->p_proto );

    if (pDebug_interface->iServerSocket==iINVALID_SOCKET)
    {
        pr_debug("socket() failed\n" );
        return ECNR_ERR_DGB_INIT_FAILED;
    }

    int iSocketOption = 1;
    err = setsockopt(pDebug_interface->iServerSocket, SOL_SOCKET, SO_REUSEADDR, &iSocketOption, (socklen_t)sizeof(int) );
    if (err!=0)
    {
        pr_debug("setsockopt() failed");
        return ECNR_ERR_DGB_INIT_FAILED;
    }

    err = bind(pDebug_interface->iServerSocket, (struct sockaddr *)&sIpAddr, sizeof(struct sockaddr));

    if (err != ECNR_ERR_OK)
    {
        pr_debug("bind failed" );
        return ECNR_ERR_DGB_INIT_FAILED;
    }

    pr_debug("EXIT");
    return ECNR_ERR_OK;
}


/*******************************************************************************
*
* FUNCTION: ecnr_sse_debug_interface_exit
* DESCRIPTION:
*
* PARAMETER: [check xml for details]
*
* RETURNVALUE: void
*
*******************************************************************************/

static void ecnr_sse_debug_interface_exit(void)
{
    pr_debug("Entered");

    pDebug_interface->exitRxThread = TRUE;

    sem_post(pDebug_interface->phTxSema);

    pr_debug("EXIT");
}


/*******************************************************************************
*
* FUNCTION: ecnr_sse_start_debug
* DESCRIPTION: starts debug session
*
* PARAMETER: [check xml for details]
*
* RETURNVALUE: gboolean
*
*******************************************************************************/
static int  ecnr_sse_start_debug( void)
{
    int err = ECNR_ERR_OK;
    sem_t sem;
    GError *error = NULL;


    pr_message("ENTERED\n");

    /* create debug interface data structure */
    err = ecnr_sse_debug_interface_create();

    if(err > 0)
    {
        pr_warning("failed to create debug interface with err: %d", err);
        return err;
    }

    /* create debug buffer */
    ecnr_sse_allocate_Debug_Buffer();

    /*create semaphore*/
    err = sem_init(&sem,0, 0);

    pr_debug("sem_init done");

    if ( ECNR_ERR_OK != err )
    {
        pr_debug("sem_init() for tx semaphore failed");
        return ECNR_ERR_DGB_INIT_FAILED;
    }

    pDebug_interface->phTxSema=&sem;

    if(pDebug_interface->pTxBuffer != NULL && pDebug_interface->pRxBuffer != NULL)
    {
        err = (int)ecnr_sse_debug_interface_CreateServerSocket();

        pr_debug("ecnr_sse_debug_interface_CreateServerSocket done");

        if(err)
        {
            pr_debug("Socket creation failed");
            return ECNR_ERR_DGB_INIT_FAILED;
        }

        pDebug_interface->tid_RxThread = g_thread_try_new ("ecnr_sse_tuning", ecnr_sse_Debug_RxThread, NULL, &error);
        if (!pDebug_interface->tid_RxThread)
        {
            pr_error("Can't create ecnr_sse_Debug_RxThread: %d", ECNR_ERR_DGB_INIT_FAILED);
            return ECNR_ERR_DGB_INIT_FAILED;
        }
    }
    else
    {
        pr_debug("Buffer creation failed");
        return ECNR_ERR_DGB_INIT_FAILED;
    }

    pr_debug("EXIT");
    return ECNR_ERR_OK;
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_debug_interface_start_session
* DESCRIPTION: starts debug session
*
* PARAMETER: [check xml for details]
*
* RETURNVALUE: gboolean
*
*******************************************************************************/

int  ecnr_sse_debug_interface_start_session( void)
{
   int err = 0;

   pr_debug("ENTERED");

   err = ecnr_sse_start_debug();

   pr_debug("EXIT");
   return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_sse_debug_interface_end_session
* DESCRIPTION: ends debug session
*
* PARAMETER: [check xml for details]
*
* RETURNVALUE: gboolean
*
*******************************************************************************/

int ecnr_sse_debug_interface_end_session( void)
{
    pr_message("Entered");

    if(!pDebug_interface)
    {
        pr_debug("debug interface not available");
        return ECNR_ERR_DGB_INIT_FAILED;
    }

    /* End threads*/
    ecnr_sse_debug_interface_exit();

    /* close server socket and a connected client socket */
    if ( TRUE == pDebug_interface->clientConnected )
    {
        shutdown( pDebug_interface->iSocketfd, 2 );
        close( pDebug_interface->iSocketfd );
    }

    shutdown( pDebug_interface->iServerSocket, 2 );
    close( pDebug_interface->iServerSocket );

    pr_debug("Socket closed");

    /* wait end threads*/
    if(pDebug_interface->tid_RxThread != NULL)
    {
        pr_debug("wait for Rxthread exit with handle %p", pDebug_interface->tid_RxThread);
        /* wait till the thread terminate */
        /*(gint)*/ g_thread_join(pDebug_interface->tid_RxThread);
        pDebug_interface->tid_RxThread = (GThread *)NULL;
    }

    pr_debug("all threads exit");

    /*Delete the Tx, Rx Buffer*/
    ecnr_ring_buffer_destroy(&pDebug_interface->pTxBuffer);
    ecnr_ring_buffer_destroy(&pDebug_interface->pRxBuffer);

    sem_close(pDebug_interface->phTxSema);
    sem_destroy(pDebug_interface->phTxSema);

    /* free debug buffer*/
    ecnr_sse_free_Debug_Buffer();

    if(pDebug_interface)
        g_free(pDebug_interface);
    pDebug_interface = NULL;

    pr_debug("EXIT");
    return ECNR_ERR_OK;
}




/*******************************************************************************
*
* FUNCTION: ecnr_sse_read_Debug_data
* DESCRIPTION: processes tcp ip data
*
* PARAMETER: None
*
* RETURNVALUE: int
*
*******************************************************************************/
void* ecnr_sse_read_Debug_data(void)
{
    int   iSseDebugInBytesAvailable = 0;

    if(!isDebugReadEntered)
    {
        pr_debug("Entered");
        isDebugReadEntered = TRUE;
    }

    if (!pTuneIO)
    {
        if(!isTraceReadNullBuffer_2)
        {
            pr_warning("NULL debugBuffer pointer");
            isTraceReadNullBuffer_2 = TRUE;
        }

        return (void*)NULL;
    }

    /* initialize trace variable here*/
    isTraceReadNullBuffer_2 = FALSE;

    if (!pDebug_interface)
    {
        pr_debug( "pDebug_interface null pointer" );
        return (void*)NULL;
    }

    /* thread work */

    if ( TRUE == pDebug_interface->clientConnected )
    {
        iSseDebugInBytesAvailable = ecnr_ring_buffer_data_extract(pDebug_interface->pRxBuffer, SseDebugBufferIn,iTX_SOCKET_BUFFER_MAXSIZE );

        if ( 0 == iSseDebugInBytesAvailable )
        {
            if(!isTraceReadNullBuffer_1)
            {
                pr_debug("no data received from Host PC : %d",iSseDebugInBytesAvailable);
                isTraceReadNullBuffer_1 = TRUE;
            }

            pTuneIO->InBuffer = NULL;
            pTuneIO->InBufferSize = 0;
            pTuneIO->InBufferUsedSize = 0;

            pTuneIO->OutBuffer = SseDebugBufferOut;
            pTuneIO->OutBufferSize = BUFFER_LENGTH;
            pTuneIO->OutBufferUsedSize = 0;
        }
        else if (iSseDebugInBytesAvailable > 0 )
        {
            pr_message("number of bytes received from Host PC is %d",iSseDebugInBytesAvailable);
            isTraceReadNullBuffer_1 = FALSE;

            pTuneIO->InBuffer = SseDebugBufferIn;
            pTuneIO->InBufferSize = BUFFER_LENGTH;
            pTuneIO->InBufferUsedSize = (unsigned int)iSseDebugInBytesAvailable;

            pTuneIO->OutBuffer = SseDebugBufferOut;
            pTuneIO->OutBufferSize = BUFFER_LENGTH;
        }
    }

    return pTuneIO;
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_Debug_Data_Tx
* DESCRIPTION: processes tcp ip data
*
* PARAMETER: None
*
* RETURNVALUE: int
*
*******************************************************************************/
static int ecnr_sse_Debug_Data_Tx(unsigned int nbTxbytes )
{
    char pcTxSocketBuffer[BUFFER_LENGTH];
    ssize_t iBytesTransmitted = 0;
    int iBytesRingbufferExtracted = 0;

    pr_debug("Entered");

    if (!pDebug_interface)
    {
        pr_debug( "pDebug_interface null pointer" );
        return -1;
    }

    if (pDebug_interface->clientConnected)
    {

        iBytesRingbufferExtracted =ecnr_ring_buffer_data_extract(pDebug_interface->pTxBuffer,pcTxSocketBuffer, (int)nbTxbytes); //iTX_SOCKET_BUFFER_MAXSIZE);

    //global+=iBytesRingbufferExtracted;
    //pr_debug("Number of bytes is %d",global);

        if ( 0 < iBytesRingbufferExtracted )
        {
            iBytesTransmitted = send( pDebug_interface->iSocketfd, pcTxSocketBuffer, (size_t)iBytesRingbufferExtracted, 0 );

            if ( -1 == iBytesTransmitted )
            {
                pr_warning( "socket send() failed" );
                return -1;
            }
            else
            {
                //pr_debug("number of send bytes to host PC is %d",iBytesTransmitted);
                if (iBytesTransmitted < iBytesRingbufferExtracted)
                {
                    pr_debug( "send() dropping %zd bytes\n", iBytesRingbufferExtracted - iBytesTransmitted );
                }
            }
        }
    }

    pr_message("EXIT with %zd bytes sended", iBytesTransmitted);

    return (int)iBytesTransmitted;
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_write_Debug_data
* DESCRIPTION: processes tcp ip data
*
* PARAMETER: None
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_sse_write_Debug_data(void* pTuneData)
{
    int err = ECNR_ERR_OK;
    unsigned int uiSseDebugOutBytesAvailable = 0;

    tecnrTuneIO *pData = (tecnrTuneIO*)pTuneData;

    if(!isDebugWriteEntered)
    {
        pr_debug("Entered");
        isDebugWriteEntered = TRUE;
    }

    if (!pData)
    {
        if(!isTraceWriteNullBuffer)
        {
            pr_warning("NULL debugBuffer pointer");
            isTraceWriteNullBuffer = TRUE;
        }
        return -1;
    }

    if (!pDebug_interface)
    {
        pr_debug( "pDebug_interface null pointer" );
        return -1;
    }

    uiSseDebugOutBytesAvailable = pData->OutBufferUsedSize;
    //memcpy(pcSseDebugBufferOut, pData->OutBuffer, uiSseDebugOutBytesAvailable);

    if (uiSseDebugOutBytesAvailable > 0)
    {
        pr_debug("the number of byte returned by the ecnr process is %d",uiSseDebugOutBytesAvailable);
        ecnr_ring_buffer_data_insert(pDebug_interface->pTxBuffer, pData->OutBuffer, (int)uiSseDebugOutBytesAvailable);// pcSseDebugBufferOut, uiSseDebugOutBytesAvailable);
        isTraceWriteNullBuffer = FALSE;

        /* transmit processed data to host*/
        err = ecnr_sse_Debug_Data_Tx(uiSseDebugOutBytesAvailable);
        if(err < 0)
        {
            return -1;
        }

        pTuneIO->OutBuffer = NULL;
        pTuneIO->OutBufferSize = 0;
        pTuneIO->OutBufferUsedSize = 0;

        pr_debug("the number of bytes send to host PC is %d",err);

    }

    //pr_debug("EXIT");
    return (int)uiSseDebugOutBytesAvailable;
}


/*******************************************************************************
*
* FUNCTION: ecnr_sse_debug_check_enable_file
* DESCRIPTION: processes tcp ip data
*
* PARAMETER: None
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_sse_debug_check_enable_file(void)
{
    FILE               *pFile = NULL;   /* pointer to configuration data file */
    gboolean           enable = TRUE;   /* internal error status */
    const char      *FileName = DEBUG_TIGGER_FILE;
    const char  *TuningEnable = TUNING_ENABLE_FILE;

    pr_debug("enable file is: %s \n",FileName);

    /* FileName is empty string or NULL pointer means we do not have a
     * configuration file (not really an error)
     */

    pFile = fopen(FileName, "r");

    if(NULL == pFile)
    {
        pr_debug("enable file not available in :  %s\n", FileName);
        enable = FALSE; /* opening file failed */
    }
    else
    {
        enable = TRUE;
        fclose(pFile);
        pr_debug("enable file is available in :  %s\n", FileName);
    }



    /* check in the second directory whether file is present */
    if(!enable)
    {
        pFile = fopen(TuningEnable, "r");

        if(NULL == pFile)
        {
            pr_debug("enable file not available in :  %s\n", TuningEnable);
            enable = FALSE; /* opening file failed */
        }
        else
        {
            enable = TRUE;
            fclose(pFile);
            pr_debug("enable file is available in :  %s\n", TuningEnable);
        }

    }
    pr_debug("EXIT\n ");

    return enable;
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_set_debug_trigger_file
* DESCRIPTION: processes tcp ip data
*
* PARAMETER: None
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_sse_set_debug_trigger_file(void)
{
    FILE            *pFile = NULL;   /* pointer to config data file */
    gboolean        ret = TRUE;        /* internal error status */
    const char       *FileName = TUNING_ENABLE_FILE; //DEBUG_TIGGER_FILE;

    pr_debug("enable file is: %s\n",FileName);

    /* File name is empty string or NULL pointer means we do not have a
     * config file (not really an error)
     */
    pFile = fopen(FileName, "w");

    if(NULL == pFile)
    {
        pr_warning("enable file not available in :  %s\n", FileName);
        ret = FALSE; /* opening file failed */
    }
    else
    {
        ret = TRUE;
        fclose(pFile);
        pr_warning("enable file is available in :  %s\n", FileName);

    }



    pr_debug("EXIT ");

    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_sse_remove_debug_trigger_file
* DESCRIPTION: processes tcp ip data
*
* PARAMETER: None
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_sse_remove_debug_trigger_file(void)
{
    gboolean        ret = TRUE;        /* internal error status */
    const char       *FileName = TUNING_ENABLE_FILE; //DEBUG_TIGGER_FILE;

    pr_debug("enable file is: %s",FileName);

    /* File name is empty string or NULL pointer means we do not have a
    * config file (not really an error)
    */

    ret = remove(FileName);

    if(!ret)
    {
        pr_warning("enable file not available in :  %s\n", FileName);
        ret = TRUE; /* opening file failed */
    }
    else
    {
        ret = FALSE;
        pr_warning("failure while removing the trigger file :  %s\n", FileName);

    }


    pr_debug("EXIT\n");
    return ret;
}
/*******************************************************************************
*
* FUNCTION: ecnr_sse_enable_debug_interface
* DESCRIPTION: processes tcp ip data
*
* PARAMETER: None
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_sse_enable_debug_interface(void)
{
    int err = ECNR_ERR_OK;

    /* initaiate debug interface*/
    err = ecnr_sse_debug_interface_start_session();

    /* setup trigger file and enable debug in alsa module*/
    if (!err)
    {
        /* set debug trigger file */
        if (ecnr_sse_set_debug_trigger_file())
        {
            err = ecnr_alsa_enable_sse_debug();
            if (!err)
            {
                IsDebugActive = TRUE;
            }
        }
    }

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_sse_enable_debug_interface_auto
* DESCRIPTION: processes tcp ip data
*
* PARAMETER: None
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_sse_enable_debug_interface_auto(void)
{
    int err = ECNR_ERR_OK;
    gboolean started = FALSE;

    pr_debug("ENTERED");

    /* create new instance of the sse*/
    ecnr_engine_handler_destroy(); //ecnr_sse_handler_destroy();

    #if !defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_LSIM)
    err = ecnr_engine_handler_create_session(TEST_DATASET_1); //ecnr_sse_handler_create_session(TEST_DATASET_1);
    #endif

    /* initiate debug interface*/
    err = ecnr_sse_debug_interface_start_session();

    /* process debug data */
    err = ecnr_alsa_enable_sse_debug();
    if (!err)
    {
        IsDebugActive = TRUE;
    }

    /* start audio */
    if(!err)
    {
        started = ecnr_alsa_start(ECNR_DEBUG_START_STOP_TIMEOUT);
        if (!started)
            err = ECNR_ERR_NO_RESULT;
    }

    pr_debug("Leave");

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_sse_disable_debug_interface
* DESCRIPTION: processes tcp ip data
*
* PARAMETER: None
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_sse_disable_debug_interface(void)
{
    int err = ECNR_ERR_OK;

    err = ecnr_sse_debug_interface_end_session();

    if (!err)
    {
        /* set debug trigger file */
        if (ecnr_sse_remove_debug_trigger_file())
        {
            err = ecnr_alsa_disable_sse_debug();
            IsDebugActive = FALSE;
        }
    }
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_sse_ist_debug_interface_Active
* DESCRIPTION: processes tcp ip data
*
* PARAMETER: None
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_sse_ist_debug_interface_Active(void)
{
    return IsDebugActive;
}



