/******************************************************************************
 * FILE        : ecnr-qwa-rcs-handler.c
 * PROJECT     : Gen3
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2014 RBCM GMBH
 * HISTORY     :
 * Date        | Author                 | Modification
 * 04.03.2014  | Patrick Rey            | initial version
 *             |                        | QWA RCS engine feature handling
 *****************************************************************************/

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX

#include <sched.h>
#include <glib/gi18n.h>
#include <glib-object.h>

#include <alsa/asoundlib.h>

#include "qwa.h"
#include "qwa_defs.h"
#include "qwa_err.h"
#include "qwa_rcs.h"
#include "qwa_rcs_err.h"

#include "ecnr-error.h"
#include "ecnr-common-defs.h"
#include "ecnr-object.h"
#include "ecnr-service.h"
#include "ecnr-alsa.h"
#include "ecnr-engine-handler.h"
#include "ecnr-qwa-handler.h"
#include "ecnr-qwa-rcs-handler.h"
#include "ecnr-configuration.h"

/*******************************************************************************
              STRUCTURE DEFINITION
*******************************************************************************/

#define QWA_RCS_ERROR(name) { name, #name }

struct QwaError {
  gint code;
  const gchar *str;
};


/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/


typedef struct _rcs_state
{
    gint rcs_thread_priority_default;
    gint rcs_enabled;
    guint rcs_server_port;
    tQwaRcs rcs_h;
    gboolean run;
    tQwaRcsInjectFlags rcsInjectFlags;
    tQwaRcsAudio rcsAudio;
    tQwaRcsAudio rcsInjectAudio;
    tQwaI16 *MicOutInject;
    tQwaI16 *RecvOutInject;
    guint rcs_buffer_size;
}rcs_state;

static rcs_state* state = NULL;


//static rcs_state *state = NULL;


/* table of QWA RCS error codes */
static const struct QwaError qwa_rcs_error_table[] = {
  QWA_RCS_ERROR(QWA_RCS_OK),
  QWA_RCS_ERROR(QWA_RCS_ERR_INV_POINTER),
  QWA_RCS_ERROR(QWA_RCS_ERR_SOCKET),
  QWA_RCS_ERROR(QWA_RCS_ERR_THREAD),
  QWA_RCS_ERROR(QWA_RCS_ERR_MUTEX),
  QWA_RCS_ERROR(QWA_RCS_ERR_SEMA),
  QWA_RCS_ERROR(QWA_RCS_ERR_INV_COMMAND),
  QWA_RCS_ERROR(QWA_RCS_ERR_INVALID_GET),
  QWA_RCS_ERROR(QWA_RCS_ERR_INVALID_SET),
  QWA_RCS_ERROR(QWA_RCS_ERR_BUSY),
  QWA_RCS_ERROR(QWA_RCS_ERR_INVALID_AUDIO_CHANNEL),
  QWA_RCS_ERROR(QWA_RCS_ERR_USER),
  QWA_RCS_ERROR(QWA_RCS_ERR_AUDIO_PATH_BUSY),
  QWA_RCS_ERROR(QWA_RCS_ERR_INVALID_AUDIO_SIZE),
  QWA_RCS_ERROR(QWA_RCS_ERR_AUDIO_INJECT_NOT_INIT),
  QWA_RCS_ERROR(QWA_RCS_ERR_INVALID_BUFFSIZE),
  QWA_RCS_ERROR(QWA_RCS_ERR_INVALID_BUFFTYPE),
  QWA_RCS_ERROR(QWA_RCS_ERR_INVALID_INJECT_CLIENT),
  QWA_RCS_ERROR(QWA_RCS_ERR_ALLOC),
  QWA_RCS_ERROR(QWA_RCS_ERR_AUDIO_PATH_UNAVAILABLE),
  QWA_RCS_ERROR(QWA_RCS_ERR_UNSPECIFIED),
};

#define QWA_RCS_NUM_ERROR_CODES ARRAYSIZE(qwa_rcs_error_table)



/*******************************************************************************
              FUNCTION DECLARATIONS
*******************************************************************************/


/*******************************************************************************
*
* FUNCTION: qwa_rcs_error_str
*
* DESCRIPTION: This function initialize the qwa-rcs Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

const gchar *qwa_rcs_error_str(tQwaInt code)
{
  guint i;
  const struct QwaError *err;
  static const gchar qwa_rcs_unknown_error[] = "Unknown QWA RCS error";

  for (i = 0, err = qwa_rcs_error_table; i < QWA_RCS_NUM_ERROR_CODES;
       i++, err++) {
    if (err->code == code)
      return err->str;
  }
  return qwa_rcs_unknown_error;
}



static int  ecnr_qwa_rcs_recover_from_init_err(unsigned short ecnr_cfg_id)
{
    int err = ECNR_ERR_OK;

    pr_debug("entered\n");

    err =  ecnr_qwa_free_binary_config(ecnr_cfg_id);
    if(err == ECNR_ERR_OK)
    {
        err = ecnr_qwa_handler_create_instance();
        if(err == ECNR_ERR_OK)
            err = ecnr_qwa_handler_set_configuration(ecnr_cfg_id);
    }

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_qwa_rcs_handler_process_signal
*
* DESCRIPTION: This function initialize the qwa-rcs Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static int ecnr_qwa_rcs_handler_process_signal(tQwaRcsSignal rcs_signal, gint * restart)
{
    int err = ECNR_ERR_OK;
    unsigned short ecnr_cfg_id;

    *restart = 0;

    switch (rcs_signal)
    {
        case qwaRcsNone:
            break;
        case qwaRcsReset:
            err = ecnr_qwa_handler_reset();
            break;
        case qwaRcsRestart:
        case qwaRcsRestartCfg:
        {
            pr_message("qwalife: restart requested");

            /*  retrieve the last configured devices */
            ecnr_alsa_disable_dev_configuration(NUM_ECNR_AUDIO);


            err = ecnr_qwa_handler_create_instance();
            if(err)
                return err;

            /* initiate configuration with last configuration id if available
               First proceed to the engine initialization using the configuration file loaded via rcs ,
               the last configuration Id will be used for the initialization if the loaded configuration leads to a failure.
            */
            ecnr_cfg_id = ecnr_engine_get_current_config_id();
            err = ecnr_qwa_handler_set_configuration(ecnr_cfg_id);
            if (err)
            {
                ecnr_critical(err, "qwaInitialize");
                //err = QWA_RCS_ERR_UNSPECIFIED;
                err = ecnr_qwa_rcs_recover_from_init_err(ecnr_cfg_id);
            }
            else
                *restart = 1;

            break;
        }
        case qwaRcsCfgAvailable:
        {

            pr_message("qwalife: configuration available\n");

            tQwaU8 *cfg;
            tQwaU16 size;

            size = QWA_RCS_MAX_BUFFER;
            cfg = g_malloc(size);
            err = ecnr_qwa_rcs_handler_get_binary((int)qwaRcsBufferConfig, cfg, &size);
            if (!err && size != 0)
            {
                /* store configuration file */
                ecnr_cfg_id = ecnr_engine_get_current_config_id();
                (void)ecnr_qwa_set_handler_binary_config(ecnr_cfg_id, cfg, size);

                pr_debug("store configuration file for cfg id (%d) \n", ecnr_cfg_id);
            }

            /* user has pushed qwalife button "Delete Target's Config"
               The configuration respective to the current CfgId can be deleted
            */

            if(size == 0)
            {
                ecnr_cfg_id = ecnr_engine_get_current_config_id();
                ecnr_qwa_free_binary_config(ecnr_cfg_id);
                pr_message("current configuration for cfgId(%d) was deleted in Target\n", ecnr_cfg_id);

                /* err can be reset, this is not a failure  */
                err = ECNR_ERR_OK;
            }

            if(cfg)
                g_free(cfg);
            cfg = NULL;

            break;
        }
        case qwaRcsConnect:
            pr_message("qwalife: Rcs connected\n");
            break;
        case qwaRcsDisconnect:
            pr_message("qwalife: Rcs disconnected\n");
            break;
        default:
            break;
    }

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_qwa_rcs_handler_process
*
* DESCRIPTION: This function initialize the rcs Engine
*
* PARAMETER: [check]
*
* RETURNVALUE: void
*
*******************************************************************************/
static int ecnr_qwa_rcs_handler_process(tQwaRcsSignal*  pSignal)
{
    gint    err = ECNR_ERR_OK;
    tQwa   qwa_handler = NULL;

    if(!state)
        return ECNR_ERR_NULL_POINTER;

    qwa_handler = (tQwa)ecnr_qwa_get_qwa_engine_handler();
    if(!qwa_handler)
        return QWA_ERR_NOT_CREATED;

    err = qwaRcsProcess(qwa_handler, state->rcs_h, pSignal, &state->rcsAudio);

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_qwa_rcs_handler_process_audio_with_debug_data
*
* DESCRIPTION: This function initialize the rcs Engine
*
* PARAMETER: [check]
*
* RETURNVALUE: void
*
*******************************************************************************/
int ecnr_qwa_rcs_handler_process_audio_with_debug_data(int* restart)
{
    int                  err = ECNR_ERR_OK;
    tQwaRcsSignal rcs_signal = qwaRcsNone;


    err = ecnr_qwa_rcs_handler_process(&rcs_signal);
    if(ECNR_ERR_OK != err)
        pr_debug("leave ecnr_qwa_rcs_handler_transfert_out_stream() with err (%d)", err);

    if (!err)
    {
        err = ecnr_qwa_rcs_handler_process_signal(rcs_signal, restart);
        if(*restart)
            pr_debug("restart RCS target was requested (%d)\n", *restart);
    }
    else
        pr_debug("leave ecnr_qwa_rcs_handler_process() with err (%d)", err);

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_qwa_rcs_handler_finalize
*
* DESCRIPTION: This function initialize the qwa-rcs Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
void ecnr_qwa_rcs_handler_finalize(void)
{
    int err = ECNR_ERR_OK;

    if (!state->rcs_h)
        return;

    err = ecnr_qwa_rcs_handler_audio_inject_destroy();
    if (err)
        rcs_critical(err, "qwaRcsAudioInjectDestroy");

    if (state->MicOutInject)
    {
        g_free(state->MicOutInject);
        state->MicOutInject = NULL;
    }

    if (state->RecvOutInject)
    {
        g_free(state->RecvOutInject);
        state->RecvOutInject = NULL;
    }
}


/*******************************************************************************
*
* FUNCTION: ecnr_qwa_rcs_handler_initialize
*
* DESCRIPTION: This function initialize the rcs Engine
*
* PARAMETER: [check]
*
* RETURNVALUE: void
*
*******************************************************************************/
int  ecnr_qwa_rcs_handler_initialize(void)
{
    int err = ECNR_ERR_OK;
    tQwaI16 **qwa_buffer;
    gint i;
    stecnr_alsa_state* paudio = (stecnr_alsa_state*)NULL;

    pr_debug("entered");

    if (!state)
        return ECNR_ERR_NULL_POINTER;

    /* retrieve pointer data container from alsa module */
    paudio = ecnr_alsa_get_instance();

    if(!paudio)
        return ECNR_ERR_NULL_POINTER;

    /* allocate the RCS audio injection buffers */
    memset(state->rcsInjectFlags, QWA_RCS_FLAG_NONE, sizeof(state->rcsInjectFlags));
    state->rcsInjectFlags[qwaRcsPathMicIn][0] = QWA_RCS_FLAG_INJECT;

    if (paudio->ecnr_mic_in_cnt > 1)
        state->rcsInjectFlags[qwaRcsPathMicIn][1]  = QWA_RCS_FLAG_INJECT;

    if (paudio->ecnr_recv_in_cnt > 0)
    {
        state->rcsInjectFlags[qwaRcsPathRecvIn][0] = QWA_RCS_FLAG_INJECT;
        if (paudio->RefIn)
            state->rcsInjectFlags[qwaRcsPathRefIn][0] = QWA_RCS_FLAG_INJECT;
    }

    state->rcsInjectFlags[qwaRcsPathMicOut][0] = QWA_RCS_FLAG_INJECT;
    state->MicOutInject = g_malloc0(paudio->ecnr_frame_shift_out * sizeof(tQwaI16));
    if (paudio->ecnr_recv_in_cnt > 0)
    {
        state->rcsInjectFlags[qwaRcsPathRecvOut][0] = QWA_RCS_FLAG_INJECT;
        state->RecvOutInject = g_malloc0(paudio->ecnr_recv_frame_shift_out * sizeof(tQwaI16));
    }

    err = ecnr_qwa_rcs_handler_audio_inject_create(); //state->rcs_h, state->rcs_buffer_size, state->rcsInjectFlags);
    if (err)
        rcs_critical(err, "qwaRcsAudioInjectCreate");

    memset(state->rcsInjectFlags, QWA_RCS_FLAG_NONE, sizeof(state->rcsInjectFlags));

    /* initialize the audio injection buffer pointers */
    memset(&state->rcsAudio, 0, sizeof(state->rcsAudio));
    state->rcsAudio.iFrameShift[qwaRcsPathMicIn] = paudio->ecnr_frame_shift_in;
    state->rcsAudio.iFrameShift[qwaRcsPathRecvIn] = paudio->ecnr_recv_frame_shift_in;
    state->rcsAudio.iFrameShift[qwaRcsPathRefIn] = paudio->ecnr_ref_frame_shift_in;
    state->rcsAudio.iFrameShift[qwaRcsPathMicOut] = paudio->ecnr_frame_shift_out;
    state->rcsAudio.iFrameShift[qwaRcsPathRecvOut] = paudio->ecnr_recv_frame_shift_out;

    memset(&state->rcsInjectAudio, 0, sizeof(state->rcsInjectAudio));
    state->rcsInjectAudio.iFrameShift[qwaRcsPathMicIn] = paudio->ecnr_frame_shift_in;
    state->rcsInjectAudio.iFrameShift[qwaRcsPathRecvIn] = paudio->ecnr_recv_frame_shift_in;
    state->rcsInjectAudio.iFrameShift[qwaRcsPathRefIn] = paudio->ecnr_ref_frame_shift_in;
    state->rcsInjectAudio.iFrameShift[qwaRcsPathMicOut] = paudio->ecnr_frame_shift_out;
    state->rcsInjectAudio.iFrameShift[qwaRcsPathRecvOut] = paudio->ecnr_recv_frame_shift_out;

    if (paudio->MicIn)
    {
        qwa_buffer = *paudio->MicIn;
        for (i = 0; i < QWA_MAX_AUDIO_CNT; i++)
        {
            state->rcsAudio.MicIn[i] = qwa_buffer[i];
            state->rcsInjectAudio.MicIn[i] = qwa_buffer[i];
            pr_debug("created buffer for rcsAudio.MicIn[%d] and rcsInjectAudio.MicIn[%d]\n",i);
        }
    }

    if (paudio->MicOut)
    {
        qwa_buffer = *paudio->MicOut;
        for (i = 0; i < QWA_MAX_AUDIO_CNT; i++)
        {
            state->rcsAudio.MicOut[i] = qwa_buffer[i];
            pr_debug("created buffer for rcsAudio.MicOut[%d] \n",i);
        }
        state->rcsInjectAudio.MicOut[0] = state->MicOutInject;
        pr_debug("created buffer for rcsInjectAudio.MicOut[0] \n");
    }

    if (paudio->RecvIn)
    {
        qwa_buffer = *paudio->RecvIn;
        for (i = 0; i < QWA_MAX_AUDIO_CNT; i++)
        {
            state->rcsAudio.RecvIn[i] = qwa_buffer[i];
            state->rcsInjectAudio.RecvIn[i] = qwa_buffer[i];
            pr_debug("created buffer for rcsAudio.RecvIn[%d] and rcsInjectAudio.RecvIn[%d]\n",i);
        }

    }

    if (paudio->RecvOut)
    {
        qwa_buffer = *paudio->RecvOut;
        for (i = 0; i < QWA_MAX_AUDIO_CNT; i++)
        {
            state->rcsAudio.RecvOut[i] = qwa_buffer[i];
            pr_debug("created buffer for rcsAudio.RecvOut[%d] \n",i);
        }
        state->rcsInjectAudio.RecvOut[0] = state->RecvOutInject;
        pr_debug("created buffer for rcsInjectAudio.RecvOut[0] \n");
    }

    if (paudio->RefIn)
    {
        qwa_buffer = *paudio->RefIn;
        for (i = 0; i < QWA_MAX_AUDIO_CNT; i++)
        {
            state->rcsAudio.RefIn[i] = qwa_buffer[i];
            state->rcsInjectAudio.RefIn[i] = qwa_buffer[i];
            pr_debug("created buffer for rcsAudio.RefIn[%d] and rcsInjectAudio.RefIn[%d]\n",i);
        }
    }

    return err;
}




/*******************************************************************************
*
* FUNCTION: ecnr_qwa_rcs_handler_transfert_out_stream
*
* DESCRIPTION: This function initialize the rcs Engine
*
* PARAMETER: [check]
*
* RETURNVALUE: void
*
*******************************************************************************/
int ecnr_qwa_rcs_handler_transfert_out_stream(void)
{
    gint err = ECNR_ERR_OK;
    stecnr_alsa_state* paudio = (stecnr_alsa_state*)NULL;

    /* retrieve pointer data container from alsa module */
    paudio = ecnr_alsa_get_instance();

    if(!paudio)
        return ECNR_ERR_NULL_POINTER;

    /*
     * If audio injection is active for any output
     * signals, then overwrite the filter output
     * with the injected audio.
     */
    if (state->rcsInjectFlags[qwaRcsPathMicOut][0])
    {
        memcpy(state->rcsAudio.MicOut[0], state->rcsInjectAudio.MicOut[0], paudio->ecnr_frame_shift_out *sizeof(tQwaI16));
    }
    if (state->rcsInjectFlags[qwaRcsPathRecvOut][0])
    {
        memcpy(state->rcsAudio.RecvOut[0], state->rcsInjectAudio.RecvOut[0], paudio->ecnr_recv_frame_shift_out *sizeof(tQwaI16));
    }

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_qwa_rcs_handler_audio_inject_create
*
* DESCRIPTION: This function initialize the rcs Engine
*
* PARAMETER: [check]
*
* RETURNVALUE: void
*
*******************************************************************************/
int ecnr_qwa_rcs_handler_audio_inject_create(void)
{
    gint err = ECNR_ERR_OK;

    pr_debug("entered");

    if(!state)
        return ECNR_ERR_NULL_POINTER;

    err = qwaRcsAudioInjectCreate(state->rcs_h, state->rcs_buffer_size, state->rcsInjectFlags);
    if(!err)
        pr_debug("rcs audio injection was created \n");

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_qwa_rcs_handler_create
*
* DESCRIPTION: This function initialize the rcs Engine
*
* PARAMETER: [check]
*
* RETURNVALUE: void
*
*******************************************************************************/
int ecnr_qwa_rcs_handler_create(void)
{
    gint err = ECNR_ERR_OK;

    pr_debug("entered");

    if(!state)
        return ECNR_ERR_NULL_POINTER;

    err = qwaRcsCreate(&state->rcs_h,state->rcs_server_port);

    if(!err)
        pr_debug("rcs was created with server port number (%d)\n",state->rcs_server_port);

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_qwa_rcs_handler_destroy
*
* DESCRIPTION: This function initialize the rcs Engine
*
* PARAMETER: [check]
*
* RETURNVALUE: void
*
*******************************************************************************/
int ecnr_qwa_rcs_handler_destroy(void)
{
    gint err = ECNR_ERR_OK;

    pr_debug("entered");

    if(!state)
        return ECNR_ERR_NULL_POINTER;

    err = qwaRcsDestroy(&state->rcs_h);

    if(!err)
        pr_debug("rcs was destroyed, server port number (%d) socket is removed\n", state->rcs_server_port);

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_qwa_rcs_handler_process_init
*
* DESCRIPTION: This function initialize the rcs Engine
*
* PARAMETER: [check]
*
* RETURNVALUE: void
*
*******************************************************************************/
int ecnr_qwa_rcs_handler_process_init(void)
{
    gint err = ECNR_ERR_OK;
    tQwa   qwa_handler = NULL;

    pr_debug("entered");

    if(!state)
        return ECNR_ERR_NULL_POINTER;

    qwa_handler = (tQwa)ecnr_qwa_get_qwa_engine_handler();
    if(!qwa_handler)
        return QWA_ERR_NOT_CREATED;


    err = qwaRcsProcessInit(qwa_handler, state->rcs_h);

    if(!err)
        pr_debug("rcs process was initialized \n");

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_qwa_rcs_handler_audio_inject_destroy
*
* DESCRIPTION: This function initialize the rcs Engine
*
* PARAMETER: [check]
*
* RETURNVALUE: void
*
*******************************************************************************/
int ecnr_qwa_rcs_handler_audio_inject_destroy(void)
{
    gint err = ECNR_ERR_OK;

    pr_debug("entered");

    if(!state)
        return ECNR_ERR_NULL_POINTER;

    qwaRcsAudioInjectDestroy(state->rcs_h);

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_qwa_rcs_handler_process_audio_inject
*
* DESCRIPTION: This function initialize the rcs Engine
*
* PARAMETER: [check]
*
* RETURNVALUE: void
*
*******************************************************************************/
int ecnr_qwa_rcs_handler_process_audio_inject(void)
{
    gint err = ECNR_ERR_OK;

    //pr_debug("entered");

    if(!state)
        return ECNR_ERR_NULL_POINTER;

    err = qwaRcsProcessAudioInject(state->rcs_h,state->rcsInjectFlags,&state->rcsInjectAudio);

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_qwa_rcs_handler_get_binary
*
* DESCRIPTION: This function initialize the rcs Engine
*
* PARAMETER: [check]
*
* RETURNVALUE: void
*
*******************************************************************************/
int ecnr_qwa_rcs_handler_get_binary
(
               int   buffType,
    unsigned  char*  data,
    unsigned short*  pusBufSize
)
{
    gint err = ECNR_ERR_OK;
    tQwa   qwa_handler = NULL;

    pr_debug("entered");

    if(!state)
        return ECNR_ERR_NULL_POINTER;

    qwa_handler = (tQwa)ecnr_qwa_get_qwa_engine_handler();
    if(!qwa_handler)
    {
        pr_debug("qwa-rcs not created\n");
        return QWA_ERR_NOT_CREATED;
    }

    err = qwaRcsGetBinary(state->rcs_h, (tQwaRcsBufferType)buffType, (tQwaU8*)data, (tQwaU16*)pusBufSize );

    return err;
}


/****************************************************/
/*                                                  */
/* static / private functions ecnr-qwa-rcs-handler  */
/*                                                  */
/****************************************************/

static void ecnr_qwa_rcs_handler_set_scheduling_priority(guint thread_priority)
{
    gint err;
    struct sched_param sched_param;

    err = sched_getparam(0, &sched_param);
    if (err < 0)
    {
        pr_warning("sched_getparam error %d", err);
        return;
    }

    sched_param.sched_priority = thread_priority;
    err = sched_setscheduler(0, SCHED_RR, &sched_param);

    if (!err)
    {
        pr_debug("Priority of QWA RCS thread set to SCHED_RR %d", sched_param.sched_priority);
    }
    else
    {
        pr_warning("sched_setscheduler error %d when attempting to set " "priority SCHED_RR %d", err, sched_param.sched_priority);
    }
}

static gpointer ecnr_qwa_rcs_handler_creation_thread(gpointer data)
{
    gint err = ECNR_ERR_OK;

    ecnr_qwa_rcs_handler_set_scheduling_priority(state->rcs_thread_priority_default);
    err = ecnr_qwa_rcs_handler_create();
    if(err)
        pr_warning("rcs handler creation failed with err(%d)\n", err);

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)data;
    return NULL;
}


/*****************************************/
/* public interface ecnr-qwa-rcs-handler */
/*****************************************/

int ecnr_qwa_rcs_handler_session_create(void)
{
    gint err = ECNR_ERR_OK;
    GError *error = NULL;

    pr_debug("entered");

    state = g_malloc0(sizeof(*state));
    state->rcs_thread_priority_default = 64;
    state->rcs_server_port = 2012;
    state->rcs_h = NULL;
    state->run = FALSE;
    state->rcs_buffer_size = 0x4000;


    /* set up rcs client */
    if(state->rcs_h == NULL)
    {
        GThread *thread;

        thread = g_thread_try_new ("rcs", ecnr_qwa_rcs_handler_creation_thread, NULL, &error);
        if (!thread)
        {
            pr_warning("Can't create ecnr_qwa_rcs_handler_creation_thread\n");
        }
        else
        {
            err = (gint) g_thread_join(thread);
            thread = (GThread*)NULL;
        }
    }
    else
        err = QWA_RCS_ERR_BUSY;

    if(err == ECNR_ERR_OK)
    {
        state->run = TRUE;
        pr_message("QWALive client is active");
    }
    return err;
}


gint ecnr_qwa_rcs_handler_session_destroy(void)
{
    gint err;

    pr_debug("");

    if (state->run)
        return QWA_ERR_ALREADY_INIT;

    err = ecnr_qwa_rcs_handler_destroy();

    /* free all buffer */
    ecnr_qwa_rcs_handler_finalize();
    state->run = FALSE;
    if(state)
        g_free(state);

    return err;
}


gboolean ecnr_qwa_rcs_get_active_state(void)
{
    gboolean ret = FALSE;

    if(state)
    {
        ret = state->run;
        pr_message("current qwa debug state is %d",ret);
    }

    return ret;
}


#endif // VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
