/******************************************************************************
 * FILE        : ecnr_ioctrl.c
 * PROJECT     : Gen3 and Gen4
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2015 RBCM GMBH
 * HISTORY     :
 * Date        | Author                 | Modification
 * 13.07.2015  | Patrick Rey            | initial version for Gen3
 *             |                        | GPIO feature handling
 *****************************************************************************/



/*********************************************************************************/
/*                         include                                               */
/*********************************************************************************/
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>
#include <glib/gi18n.h>
#include <glib-object.h>
#include <glib-unix.h> //new

#include <glib.h>
#include "bpcl.h"
#include "ecnr-common-defs.h"
#include "ecnr-service.h"
#include "ecnr-ioctrl.h"


/* Needed for Trace */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define SPM_UNIT_TESTS // solve warning
#include "ecnr-trace.h"
#include "etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ECNR
#include "trcGenProj/Header/ecnr-ioctrl.c.trc.h"
#endif

//#define D_RESTORE_STATE_SUPPORT

/*********************************************************************************/
/*                         define                                                */
/*                                                                               */
/* --IOCTRLs & Device names                                                      */
/*      #include "include/osioctrl.h"                                            */
/*                                                                               */
/* -> OSAL_EN_HF_VR_MODE                                                         */
/*********************************************************************************/


#ifdef D_RESTORE_STATE_SUPPORT
//Below macro's are unused in the source file due to that observed lint errors.
//To solve lint errors below Macro's are commented. In future if below macro's are usable, please uncomment it and use it.
//#define ECNR_IO_ACTIVE (gboolean)TRUE
#define ECNR_IO_INACTIVE (gboolean)FALSE
#endif




/*********************************************************************************/
/*                         macro                                                 */
/*********************************************************************************/

/*********************************************************************************/
/*                         static variable                                       */
/*********************************************************************************/


static  guint32 m_u32LastErrorCode = 0 ;         // Last error code.
static  OSAL_tIODescriptor m_gpiohandle = OSAL_ERROR;

#ifdef D_RESTORE_STATE_SUPPORT
static ecnr_ioctrl_cfg* m_ecnr_ioctrl_restore;
#endif
/*******************************************************************************
*
* FUNCTION: ecnr_ioctrl_set_GPIO
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static gboolean ecnr_ioctrl_set_gpio(gboolean state, OSAL_enGpioPins egpio)
{
    OSAL_trGPIOData    rGPIOData         = { 0 };
    gboolean bret = TRUE;

    pr_message("enter to vSetGPIO: egpio=%d, state=%d", (int)egpio, (int)state);
    ETG_TRACE_USR3(("[ecnr_ioctrl_set_gpio]: ENTERED with gpio[%d], state[%d]", (int)egpio, (int)state));
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_INF4CV_LINUX
    if(egpio <= 0)
    {
        bret = FALSE;
        pr_message("egpio=%d hence not required to set gpio for usecase", (int)egpio);
        ETG_TRACE_USR3(("[ecnr_ioctrl_set_gpio]: ENTERED egpio=%d hence not required to set gpio for usecase", (int)egpio));
        return bret; 
    }
#endif

    rGPIOData.tId = (OSAL_tGPIODevID) egpio;
    if(m_gpiohandle != OSAL_ERROR)
    {
        /*
        Based on the assumption "hf-vr-mode" pin is always configured towards output direction from platform,
        hence explicitly setting output direction call is removed from ecnr
        */
        if(state == TRUE)
        {
            if(OSAL_ERROR == OSAL_s32IOControl(m_gpiohandle, OSAL_C_32_IOCTRL_GPIO_SET_ACTIVE_STATE, (intptr_t)&rGPIOData))
            {
                bret = FALSE;
                pr_warning("failed to open control gpio pin(%d)\n", egpio);
                ETG_TRACE_ERR(("[ecnr_ioctrl_set_gpio]: failed to open control gpio pin gpio[%d]", (int)egpio));
            }
        }
        else
        {
            if(OSAL_ERROR == OSAL_s32IOControl(m_gpiohandle, OSAL_C_32_IOCTRL_GPIO_SET_INACTIVE_STATE, (intptr_t)&rGPIOData))
            {
                bret = FALSE;
                pr_warning("failed to open control gpio pin(%d)\n", egpio);
                ETG_TRACE_ERR(("[ecnr_ioctrl_set_gpio]: failed to open control gpio pin gpio[%d]", (int)egpio));
            }
        }
    }
    else
    {
        bret = FALSE;
        pr_warning("io device is not open, should be open before usage");
    }

    return bret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_ioctrl_open
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/

static gboolean ecnr_ioctrl_open(void)
{
    gboolean ret = FALSE;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_ioctrl_open]: ENTERED"));

    // Open the EOL Device.
    m_gpiohandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE);


    if (OSAL_ERROR == m_gpiohandle)
    {
        m_u32LastErrorCode =  OSAL_u32ErrorCode();
        pr_critical("failed to open OSAL_C_STRING_DEVICE_DIAG_EOL device with err:%d\n", m_u32LastErrorCode);
        ETG_TRACE_FATAL(("[ecnr_ioctrl_open]: failed to open osal device[%s]", OSAL_C_STRING_DEVICE_GPIO));
    }
    else
    {
        ETG_TRACE_USR3(("[ecnr_ioctrl_open]: OSAL_C_STRING_DEVICE_DIAG_EOL device successfully open"));
        pr_debug("OSAL_C_STRING_DEVICE_DIAG_EOL device successfully open\n");
        ret = TRUE;
    }


    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_ioctrl_close
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static gboolean ecnr_ioctrl_close(void)
{
    gboolean ret = TRUE;
    pr_debug("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_ioctrl_close]: ENTERED"));


    if ((OSAL_ERROR != m_gpiohandle) && (OSAL_ERROR == OSAL_s32IOClose(m_gpiohandle)))
    {
        ret = FALSE;
        pr_critical("close osal IO device failed with error: %d \n", OSAL_u32ErrorCode());
    }
    m_gpiohandle =  OSAL_ERROR;

    return ret;
}




/*******************************************************************************
*
* FUNCTION: ecnr_diag_eol_get_last_Error_code
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
guint32  ecnr_ioctrl_get_last_Error_code(void)
{
    pr_message("osal IO read error : %d \n", m_u32LastErrorCode);
    return m_u32LastErrorCode;
}




/*******************************************************************************
*
* FUNCTION: ecnr_ioctrl_create
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
gboolean ecnr_ioctrl_create(void)
{
    gboolean bret = TRUE;
#ifdef D_RESTORE_STATE_SUPPORT
    gboolean state = FALSE;
    OSAL_enGpioPins egpio;
#endif

    pr_message("ENTERED");

    if(m_gpiohandle == OSAL_ERROR)
        bret = ecnr_ioctrl_open();


#ifdef D_RESTORE_STATE_SUPPORT
    if(bret)
    {
        if(m_ecnr_ioctrl_restore)
        {
            state =  m_ecnr_ioctrl_restore->state;
            egpio =  m_ecnr_ioctrl_restore->egpio;
        }
        else
        {
            /* retrieve value from config */
            state =  ECNR_IO_INACTIVE;
            egpio =  OSAL_EN_HF_VR_MODE;
        }

        /*search io control configuration*/
        bret = ecnr_ioctrl_set_gpio(state, egpio);
    }
#endif

    pr_message("EXIT");

    return bret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_ioctrl_finalize
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
gboolean ecnr_ioctrl_finalize(void)
{
    gboolean bret = TRUE;

    /* close io device */
    ecnr_ioctrl_close();

#ifdef D_RESTORE_STATE_SUPPORT
    if(m_ecnr_ioctrl_restore)
        g_free(m_ecnr_ioctrl_restore);
#endif

    return bret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_ioctrl_set_pin_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
gboolean ecnr_ioctrl_set_pin_state(int egpio, gboolean state)
{
    gboolean bret = TRUE;
    OSAL_enGpioPins iodev = (OSAL_enGpioPins)egpio;

    pr_debug("ENTERED");
    ETG_TRACE_USR3(("[ecnr_ioctrl_set_pin_state]: ENTERED with gpio[%d] and state[%d]",egpio, state));

    if(m_gpiohandle == OSAL_ERROR)
        bret = ecnr_ioctrl_open();

    if(bret)
        bret = ecnr_ioctrl_set_gpio(state, iodev);
    else
        return FALSE;

#ifdef D_RESTORE_STATE_SUPPORT
    if(bret)
    {
        pr_message("set GPIO(%d) to state(%d)\n", egpio, state);

        if(m_ecnr_ioctrl_restore)
        {
            g_free(m_ecnr_ioctrl_restore);
            m_ecnr_ioctrl_restore = NULL;
        }

        m_ecnr_ioctrl_restore = g_malloc0(sizeof(ecnr_ioctrl_cfg));

        if(m_ecnr_ioctrl_restore)
        {
            m_ecnr_ioctrl_restore->restore = TRUE;
            m_ecnr_ioctrl_restore->state = state;
            m_ecnr_ioctrl_restore->egpio = (OSAL_enGpioPins)egpio;
        }
    }
#endif


    ETG_TRACE_USR4(("[ecnr_ioctrl_set_pin_state]: EXIT with RetCode[%d]", bret));
    return bret;

}



