/******************************************************************************
 * FILE        : ecnr-error.c
 * PROJECT     : Gen3 and Gen4
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2019 RBEI
 * HISTORY     :
 * Date        | Author                 | Modification
 * 15.05.2019  | RBEI/ECO Ranjit Susal  | initial version
 *             |                        | error codes handling
 *****************************************************************************/

/* C standard library */
#include <stdio.h>
#include <unistd.h>
#include <string.h>

/* glib */
#include <glib.h>
#include <gmodule.h>

#ifdef D_ECNR_LIB_QWA
#include "qwa.h"
#include "qwa_defs.h"
#include "qwa_err.h"
#include "qwa_rcs.h"
#include "qwa_rcs_err.h"
#endif

#ifdef D_ECNR_LIB_SSE_3_17
#include "sse.h"
#include "sse_err.h"
#endif

#ifdef D_ECNR_LIB_SSE_4_03
#include "sse.h"
#include "sse_param.h"
#endif

#include "ecnr-error.h"
#include "ecnr-common-defs.h"


#define ECNR_ERROR(name) { name, #name }


/* ECNR ENGINE and ECNR error codes string conversion table  */
static const struct EcnrError ECNR_ERROR_table[] = {

#ifdef D_ECNR_LIB_QWA
    /* QWA specific error*/
    ECNR_ERROR(QWA_OK),
    ECNR_ERROR(QWA_NO_RESULT),
    ECNR_ERROR(QWA_SIZE_ONLY),
    ECNR_ERROR(QWA_ERR_ALLOC),
    ECNR_ERROR(QWA_ERR_NULL_POINTER),
    ECNR_ERROR(QWA_ERR_NOT_INIT),
    ECNR_ERROR(QWA_ERR_NOT_CREATED),
    ECNR_ERROR(QWA_ERR_ALREADY_INIT),
    ECNR_ERROR(QWA_ERR_INVALID_HANDLE),
    ECNR_ERROR(QWA_ERR_DESTRUCTION_FAILED),
    ECNR_ERROR(QWA_ERR_RESET_FAILED),
    ECNR_ERROR(QWA_ERR_ILLEGAL_SET_PARAM),
    ECNR_ERROR(QWA_ERR_ILLEGAL_SET_BEFORE_INIT),
    ECNR_ERROR(QWA_ERR_ILLEGAL_SET_AFTER_INIT),
    ECNR_ERROR(QWA_ERR_INVALID_ID),
    ECNR_ERROR(QWA_ERR_INVALID_OPERATION_MODE),
    ECNR_ERROR(QWA_ERR_INVALID_SIZE),
    ECNR_ERROR(QWA_ERR_PARAM),
    ECNR_ERROR(QWA_ERR_INVALID_SAMPLE_RATE),
    ECNR_ERROR(QWA_ERR_INVALID_FRAME_SHIFT),
    ECNR_ERROR(QWA_ERR_INVALID_CHANNEL_CNT),
    ECNR_ERROR(QWA_ERR_INVALID_FFT_LENGTH),
    ECNR_ERROR(QWA_ERR_INVALID_FILTER_LENGTH),
    ECNR_ERROR(QWA_ERR_MODULE_NOT_AVAILABLE),
    ECNR_ERROR(QWA_ERR_INVALID_ENDIANNESS),
    ECNR_ERROR(QWA_ERR_AEC_FILTER_LEN),
    ECNR_ERROR(QWA_ERR_CFG_INVALID_IDSTR),
    ECNR_ERROR(QWA_ERR_CFG_INVALID_FORMAT),
    ECNR_ERROR(QWA_ERR_CFG_INVALID_HEADER),
    ECNR_ERROR(QWA_ERR_CFG_PARAM_NOT_INITIALIZED),
    ECNR_ERROR(QWA_ERR_CFG_PARAM_INVALID_STATE),
    ECNR_ERROR(QWA_ERR_CFG_UNKNOWN_PARAM),
    ECNR_ERROR(QWA_ERR_CFG_INVALID_CHECKSUM),
    ECNR_ERROR(QWA_ERR_CFG_INVALID_SIZE),
    ECNR_ERROR(QWA_ERR_CFG_INVALID_VERSION),
    ECNR_ERROR(QWA_ERR_DIAGNOSTIC_ACTIVE),
    ECNR_ERROR(QWA_ERR_DIAGNOSTIC_NOT_READY),
#endif

#ifdef D_ECNR_LIB_SSE_3_17
    /* SSE 3.17 specific error*/
    ECNR_ERROR(SSE_ERR_OK),
    ECNR_ERROR(SSE_ERR_NO_RESULT),
    ECNR_ERROR(SSE_ERR_SIZE_ONLY),
    ECNR_ERROR(SSE_ERR_ALLOC),
    ECNR_ERROR(SSE_ERR_NULL_POINTER),
    ECNR_ERROR(SSE_ERR_MODULE_NOT_INIT),
    ECNR_ERROR(SSE_ERR_NOT_CREATED),
    ECNR_ERROR(SSE_ERR_MODULE_ALREADY_INIT),
    ECNR_ERROR(SSE_ERR_INVALID_MAIN_STRUCT),
    ECNR_ERROR(SSE_ERR_DEINIT_FAILED),
    ECNR_ERROR(SSE_ERR_ILLEGAL_SET_PARAM),
    ECNR_ERROR(SSE_ERR_ILLEGAL_SET_PARAM_BEFORE_INIT),
    ECNR_ERROR(SSE_ERR_ILLEGAL_SET_PARAM_AFTER_INIT),
    ECNR_ERROR(SSE_ERR_INVALID_PARAM_ID),
    ECNR_ERROR(SSE_ERR_INVALID_CHANNEL_NUMBER),
    ECNR_ERROR(SSE_ERR_INVALID_OPERATION_MODE),
    ECNR_ERROR(SSE_ERR_INVALID_SIZE),
    ECNR_ERROR(SSE_ERR_PARAM),
    ECNR_ERROR(SSE_ERR_PARAM_CNT),
    ECNR_ERROR(SSE_ERR_SAMPLE_RATE),
    ECNR_ERROR(SSE_ERR_FRAMESHIFT),
    ECNR_ERROR(SSE_ERR_INVALID_CHAN_CNT),
    ECNR_ERROR(SSE_ERR_FFT_LENGTH),
    ECNR_ERROR(SSE_ERR_INVALID_FILTER_LEN),
    ECNR_ERROR(SSE_ERR_NULL_FILTER),
    ECNR_ERROR(SSE_ERR_PARAMETER_TABLE),
    ECNR_ERROR(SSE_ERR_NOT_IMP),
    ECNR_ERROR(SSE_ERR_SUBMODULE_DISABLED),
    ECNR_ERROR(SSE_ERR_INVALID_SET_PARAM_ENDIANNESS),
    ECNR_ERROR(SSE_ERR_INIT_MMGR),
    ECNR_ERROR(SSE_ERR_INVALID_EXT_MEM_TYPE),
    ECNR_ERROR(SSE_ERR_INVALID_OPT_LEVEL),
    ECNR_ERROR(SSE_ERR_DBGIO_MSG_QUEUE),
    ECNR_ERROR(SSE_ERR_DBGIO_MSG_TYPE),
    ECNR_ERROR(SSE_ERR_DBGIO_OUTBUF_SIZE),
    ECNR_ERROR(SSE_ERR_INSUFFICIENT_SCRATCH),
    ECNR_ERROR(SSE_ERR_DBGIO_INST_ID),
    ECNR_ERROR(SSE_ERR_INTERNAL_MSG_BUF_TOO_SMALL),
    ECNR_ERROR(SSE_ERR_DBGIO_MSG_FORMAT),
    ECNR_ERROR(SSE_ERR_ASR_POST_PROC),
    ECNR_ERROR(SSE_ERR_INVALID_DATASET_ID),
    ECNR_ERROR(SSE_ERR_INVALID_MODULE_ID),
    ECNR_ERROR(SSE_ERR_INTERNAL_CALLBACK_ERROR),
    ECNR_ERROR(SSE_ERR_INVALID_DATA_CTRL),
    ECNR_ERROR(SSE_ERR_BSD_INVALID_IDENT),
    ECNR_ERROR(SSE_ERR_BSD_INVALID_FORMAT),
    ECNR_ERROR(SSE_ERR_BSD_INVALID_STRUCT_VERSION),
    ECNR_ERROR(SSE_ERR_BSD_INVALID_STRUCT_FORMAT),
    ECNR_ERROR(SSE_ERR_BSD_INVALID_PARAM),
    ECNR_ERROR(SSE_ERR_BSD_INVALID_BUFFER_PARAM),
    ECNR_ERROR(SSE_ERR_BSD_MAX_BSD_CNT_EXCEEDED),
    ECNR_ERROR(SSE_ERR_CONFIG_BSD_MISMATCH),
    ECNR_ERROR(SSE_ERR_BSD_CHECKSUM),
    ECNR_ERROR(SSE_ERR_BSD_READ),
    ECNR_ERROR(SSE_ERR_BSD_WRITE),
    ECNR_ERROR(SSE_ERR_BSD_EOF),
    ECNR_ERROR(SSE_ERR_BSD_VERSION),
    ECNR_ERROR(SSE_ERR_BSD_INVALID_DIM),
    ECNR_ERROR(SSE_ERR_BSD_UIO),
    ECNR_ERROR(SSE_ERR_BSD_INVALID_CONTEXT),
    ECNR_ERROR(SSE_ERR_BSD_DECOMPRESS),
    ECNR_ERROR(SSE_ERR_INVALID_BACKBAND),
    ECNR_ERROR(SSE_ERR_NUM_REF_CHAN),
    ECNR_ERROR(SSE_ERR_AEC_FILTER_LEN),
    ECNR_ERROR(SSE_ERR_LIMITER_DELAY),
    ECNR_ERROR(SSE_ERR_LIMITER_THRESHOLD),
    ECNR_ERROR(SSE_ERR_LIMITER_SUSTAIN_MAX),
    ECNR_ERROR(SSE_ERR_LIMITER_RELEASE_TIME),
    ECNR_ERROR(SSE_ERR_TIMER_NOT_CREATED),
    ECNR_ERROR(SSE_ERR_INVALID_TIMER_HANDLE),
    ECNR_ERROR(SSE_ERR_TIMER_HANDLE_NOT_UNIQUE),
    ECNR_ERROR(SSE_ERR_DIAGNOSTIC_ACTIVE),
    ECNR_ERROR(SSE_ERR_DIAGNOSTIC_NOT_READY),
#endif

#ifdef D_ECNR_LIB_SSE_4_03
    /* SSE 4.03 specific error*/
    ECNR_ERROR(SSE_ERR_OK),
    ECNR_ERROR(SSE_ERR_NO_RESULT),
    ECNR_ERROR(SSE_ERR_OBSOLETE_PARAM),
    ECNR_ERROR(SSE_ERR_ALLOC),
    ECNR_ERROR(SSE_ERR_NULL_POINTER_ARG),
    ECNR_ERROR(SSE_ERR_INVALID_EXT_MEM),
    ECNR_ERROR(SSE_ERR_MODULE_NOT_INIT),
    ECNR_ERROR(SSE_ERR_NOT_CREATED),
    ECNR_ERROR(SSE_ERR_NOT_INITIALIZED),
    ECNR_ERROR(SSE_ERR_ALREADY_INIT),
    ECNR_ERROR(SSE_ERR_INVALID_MAIN_STRUCT),
    ECNR_ERROR(SSE_ERR_SCD_FORMAT),
    ECNR_ERROR(SSE_ERR_SCD_MAX_CNT_EXCEEDED),
    ECNR_ERROR(SSE_ERR_INVALID_PARAM_ID),
    ECNR_ERROR(SSE_ERR_INVALID_CHANNEL_NUMBER),
    ECNR_ERROR(SSE_ERR_PARAM_OUT_OF_RANGE),
    ECNR_ERROR(SSE_ERR_PARAM_ILLEGAL_ACCESS),
    ECNR_ERROR(SSE_ERR_INVALID_SIZE),
    ECNR_ERROR(SSE_ERR_PARAM),
    ECNR_ERROR(SSE_ERR_NOT_AVAILABLE),
    ECNR_ERROR(SSE_ERR_SUBMODULE_DISABLED),
    ECNR_ERROR(SSE_ERR_AUDIO_BUFFER_TOO_SMALL),
    ECNR_ERROR(SSE_ERR_TOO_FEW_AUDIO_BUFFERS),
    ECNR_ERROR(SSE_ERR_MEM_ALIGNMENT),
    ECNR_ERROR(SSE_ERR_INVALID_ARG),
    ECNR_ERROR(SSE_ERR_INVALID_DNN_VERSION),
    ECNR_ERROR(SSE_ERR_MISSING_DNN_PARAM),
    ECNR_ERROR(SSE_ERR_TUNEIO_BUFFERS),
#endif

    /* ECNR specific error*/
    ECNR_ERROR(ECNR_ERR_INVALID_SIZE),
    ECNR_ERROR(ECNR_ERR_INVALID_ASR_MODE),
    ECNR_ERROR(ECNR_ERR_CONFIG_FILE_OPEN_FAILED),
    ECNR_ERROR(ECNR_ERR_CONFIG_FILE_EMPTY),
    ECNR_ERROR(ECNR_ERR_ALLOC),
    ECNR_ERROR(ECNR_ERR_CONFIG_FILE_READING_FAILED),
    ECNR_ERROR(ECNR_ERR_NOT_CREATED),
    ECNR_ERROR(ECNR_ERR_CONFIG_NOT_AVAILABLE),
    ECNR_ERROR(ECNR_ERR_NULL_POINTER),
    ECNR_ERROR(ECNR_ERR_DGB_NO_DATA),
    ECNR_ERROR(ECNR_ERR_APPID_NOT_AVAILABLE),
    ECNR_ERROR(ECNR_ERR_CONFID_NOT_AVAILABLE),
    ECNR_ERROR(ECNR_ERR_APPID_INSTANCE_ALREADY_ACTIVE),
    ECNR_ERROR(ECNR_ERR_APPID_INSTANCE_NOT_AVAILABLE),
    ECNR_ERROR(ECNR_ERR_NO_INSTANCE_AVAILABLE),
    ECNR_ERROR(ECNR_ERR_APPID_CONFIGID_NOT_MATCHING),
    ECNR_ERROR(ECNR_ERR_REQ_APPID_NOT_ACTIVE),
    ECNR_ERROR(ECNR_ERR_OBJECT_FI_NOT_SUPPORTED),
    ECNR_ERROR(ECNR_ERR_DEV_ID_NOT_AVAILABLE),
    ECNR_ERROR(ECNR_ERR_DEV_CHNID_NOT_MATCHING),
    ECNR_ERROR(ECNR_ERR_OBJECT_FI_AUDIO_IS_RUNNING),
    ECNR_ERROR(ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE),
    ECNR_ERROR(ECNR_ERR_TBL_IDX_NB_CFG_VALUE_TRANSGRESSION),
    ECNR_ERROR(ECNR_ERR_PERIODE_SIZE_INVALID),
    ECNR_ERROR(ECNR_ERR_INVALID_INPUT),

    /* Error configuration module */
    ECNR_ERROR(ECNR_ERR_DP_ERROR_SIZE_OF_DATAPOOL),

    /* Error debug interface*/
    ECNR_ERROR(ECNR_ERR_DGB_INIT_FAILED),
    ECNR_ERROR(ECNR_ERR_DEBUG_ALREADY_ACTIVE),
    ECNR_ERROR(ECNR_ERR_DEBUG_MODER_NOT_SUPPORTED),
    ECNR_ERROR(ECNR_ERR_DGB_SET_RMV_TRIGGER_FILE_FAILED),
    ECNR_ERROR(ECNR_ERR_NO_AUDIO_STREAMING),
    ECNR_ERROR(ECNR_ERR_AUDIO_THREAD_ALREADY_ACTIVE),

    /* ALSA Module*/
    ECNR_ERROR(ECNR_ERR_ALSA_NO_INSTANCE),

    /* RTS module */
    ECNR_ERROR(ECNR_ERR_RTS_NOT_SUPPORTED),

    /* QWA RCS error  group*/
    ECNR_ERROR(ECNR_QWA_RCS_ERR_INV_POINTER),
    ECNR_ERROR(ECNR_QWA_RCS_ERR_INVALID_BUFFSIZE),

    /* SERVICE module */
    ECNR_ERROR(ECNR_ERR_SERV_NOT_INITIALIZED),
    ECNR_ERROR(ECNR_ERR_SERV_ASR_TYPE_UNSUPPORTED),

};

#define ECNR_NUM_ERROR_CODES ARRAYSIZE(ECNR_ERROR_table)


/******************************************************************************
 * FUNCTION     : ecnr_error_str
 * DESCRIPTION  : search for corresponding string from the error code
 * PARAMETER    : ECNR Engine or ECNR error code
 * RETURN VALUE : pointer to a error code string
 * HISTORY      :
 * Date         | Author                    | Modification
 * 21.08.2013   | Patrick Rey(CM-AI/PJ-G33) | Initial Revision
 *****************************************************************************/
const gchar *ecnr_error_str(int code)
{
    guint i;
    const struct EcnrError *err;
    static const gchar ecnr_unknown_error[] = "Unknown ECNR error";

    for (i = 0, err = ECNR_ERROR_table; i < ECNR_NUM_ERROR_CODES; i++, err++) {
        if (err->code == code)
            return err->str;
    }
    return ecnr_unknown_error;
}

/******************************************************************************
 * FUNCTION     : ECNR_object_error_quark
 * DESCRIPTION  : error string prefix
 * PARAMETER    : void
 * RETURN VALUE : quark with error string prefix
 * HISTORY      :
 * Date         | Author                    | Modification
 * 21.08.2013   | Patrick Rey(CM-AI/PJ-G33) | Initial Revision
 *****************************************************************************/
GQuark ECNR_object_error_quark(void)
{
    static GQuark quark = 0;

    if (!quark)
        quark = g_quark_from_static_string("ecnr-error");

    return quark;
}

/******************************************************************************
 * FUNCTION     : report_ecnr_error
 * DESCRIPTION  : convert a ECNR status code to a DBus error, A non-zero status
 *                status code indicates an error
 * PARAMETER    : status -> error code
 *              : dbus_msg -> string message
 *              : error -> error code and error string are set to this.
 * RETURN VALUE : TRUE if no error occurred, or FALSE if an error occurred
 * HISTORY      :
 * Date         | Author                    | Modification
 * 21.08.2013   | Patrick Rey(CM-AI/PJ-G33) | Initial Revision
 *****************************************************************************/
gboolean report_ecnr_error
(
             int   status,
    const  gchar*  dbus_msg,
          GError** error
)
{
    gboolean ret = TRUE;

    if (status != ECNR_ERR_OK) {
        ecnr_warning(status, "%s", dbus_msg);

        g_set_error(error, ECNR_object_error_quark(), status,
          "%s (%d): %s", ecnr_error_str(status), status,
          dbus_msg);

        ret = FALSE;
    }

    return ret;
}
