/******************************************************************************
 * FILE        : ecnr-engine-handler.c
 * PROJECT     : Gen3 and Gen4
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2014 RBCM GMBH
 * HISTORY     :
 * Date        | Author                 | Modification
 * 04.04.2014  | Patrick Rey            | initial version
 * 08.03.2019  | Ranjit Susal(RBEI/ECO) | extended set/get functionality
 *****************************************************************************/

#include "ecnr-error.h"
#include "ecnr-common-defs.h"
#include "ecnr-object.h"
#include "ecnr-service.h"
#include "ecnr-alsa.h"
#include "ecnr-sse-handler.h"
#include "ecnr-qwa-handler.h"
#include "ecnr-qwa-rcs-handler.h"
#include "ecnr-configuration.h"


#define ECNR_S_IMPORT_INTERFACE_DP
#include "ecnr-datapool-default.h"
#include "ecnr-configuration-data.h"

/* Needed for Trace */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define SPM_UNIT_TESTS // solve warning
#include "ecnr-trace.h"
#include "etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ECNR_ENGINE
#include "trcGenProj/Header/ecnr-engine-handler.c.trc.h"
#endif


/*******************************************************************************
              DEFINES
*******************************************************************************/


/*******************************************************************************
              STRUCTURE DEFINITION
*******************************************************************************/

/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/


/*******************************************************************************
              STATIC FUNCTION / VARIABLES  DECLARATIONS
*******************************************************************************/
static gboolean         m_ecnr_engine_init_state                = FALSE;
static gboolean         m_ecnr_engine_default_data_set_loaded   = FALSE;
static unsigned short   m_ecnr_engine_active_configuration_id   = 0;



/*******************************************************************************
*
* FUNCTION: ecnr_engine_handler_destroy
*
* DESCRIPTION: THis function initialize the sse Engine with the passed ConfigId
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
void ecnr_engine_load_default_data_set(void)
{

    pr_message("ENTERED\n");

    m_ecnr_engine_default_data_set_loaded = TRUE;

    return;
}



/*******************************************************************************
*
* FUNCTION: ecnr_engine_handler_process_audio_with_debug_data
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int ecnr_engine_handler_process_audio_with_debug_data (stecnr_alsa_state* state, gint* restart)
{
    int err = ECNR_ERR_OK;
    int txdata = 0;

    /* initialize*/
    *restart = 0;
    ETG_TRACE_USR4(("[ecnr_engine_handler_process_audio_with_debug_data]: ENTERED"));

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    err = ecnr_qwa_rcs_handler_process_audio_inject();
    if(!err)
        err = ecnr_qwa_handler_audio_in(state);
    else
        pr_debug("leave ecnr_qwa_rcs_handler_process_audio_inject() with err (%d)", err);
    if(!err)
        err = ecnr_qwa_handler_audio_out(state);
    else
        pr_debug("leave ecnr_qwa_handler_audio_in() with err (%d)", err);
    if (!err)
        err = ecnr_qwa_rcs_handler_transfert_out_stream();
    else
        pr_debug("leave ecnr_qwa_handler_audio_out() with err (%d)", err);

    if(!err)
        ecnr_qwa_rcs_handler_process_audio_with_debug_data(restart);
    else
        pr_debug("leave ecnr_qwa_rcs_handler_process_audio_with_debug_data() with err (%d)", err);
#else

    state->pTuneIO = (tecnrTuneIO*)ecnr_sse_read_Debug_data();

    if(state->pTuneIO->InBufferUsedSize)
    {
        pr_debug("process %d Debug data \n", state->pTuneIO->InBufferUsedSize);
    }

    err = ecnr_sse_handler_process_audio_with_debug_data(state);
    if (!err)
    {
        if(state->pTuneIO->OutBufferSize != BUFFER_LENGTH)
        {
            pr_debug("returned Debug data to send -> %d \n", state->pTuneIO->OutBufferSize);
        }
        txdata = ecnr_sse_write_Debug_data((void*)state->pTuneIO);
    }

    if(txdata)
    {
        pr_debug("write Debug data to send to the host -> %d \n", txdata);
    }
#endif

    ETG_TRACE_USR4(("[ecnr_engine_handler_process_audio_with_debug_data]: EXIT"));
    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_engine_handler_process_audio
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int ecnr_engine_handler_process_audio (stecnr_alsa_state* pstate)
{
    int err = ECNR_ERR_OK;

    ETG_TRACE_USR4(("[ecnr_engine_handler_process_audio]: ENTERED"));

    #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    err = ecnr_qwa_handler_audio_in(pstate);
    if(!err)
        err = ecnr_qwa_handler_audio_out(pstate);
    #else
    err = ecnr_sse_handler_process_audio(pstate);
    #endif

    ETG_TRACE_USR4(("[ecnr_engine_handler_process_audio]: EXIT"));
    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_engine_handler_create_session
*
* DESCRIPTION: THis function initialize the sse Engine with the passed ConfigId
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

int ecnr_engine_handler_create_session(unsigned short ConfigId)
{
    int err = 0;

    pr_debug("the requested configuration id is %d\n", ConfigId);
    ETG_TRACE_USR3(("[ecnr_engine_handler_create_session]: the requested configuration id is %d\n", ConfigId));

    if(!ConfigId)
    {
        pr_warning("invalid configuration id: %d, engine is not initialized\n",ConfigId);
        return ECNR_ERR_OK;
    }

    if(m_ecnr_engine_init_state || ConfigId == m_ecnr_engine_active_configuration_id)
    {
        pr_warning("the engine is already initialized with the requested configuration id: %d \n",  m_ecnr_engine_active_configuration_id);
        return ECNR_ERR_OK;
    }

    #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    err = ecnr_qwa_handler_create_session(ConfigId);
    #else
    err = ecnr_sse_handler_create_session(ConfigId);
    #endif

    if (err)
    {
        pr_critical("failed while engine initialization\n");
        return err;
    }
    else
    {
        pr_message("new emgine instance created with config_id -> %d\n", ConfigId );
        ETG_TRACE_USR3(("[ecnr_engine_handler_create_session]: new emgine instance created with config_id -> %d\n", ConfigId));
        m_ecnr_engine_init_state = TRUE;
        m_ecnr_engine_active_configuration_id = ConfigId;
    }

    ETG_TRACE_USR3(("[ecnr_engine_handler_create_session]: the requested configuration id is %d\n", ConfigId));
    pr_debug("EXIT with err = %d\n", err);

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_engine_handler_destroy
*
* DESCRIPTION: THis function initialize the sse Engine with the passed ConfigId
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_engine_handler_destroy(void)
{
    int err = 0;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_engine_handler_destroy]: ENTERED"));

    #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    err = ecnr_qwa_handler_destroy();
    #else //
    err = ecnr_sse_handler_destroy();
    #endif


    if(m_ecnr_engine_default_data_set_loaded)
    {
        /* remove the bsd file from the rfs for the last active configuration */
        const struct EcnrConfig *pdata = NULL;
        pdata = ecnr_configuration_get_config_data(m_ecnr_engine_active_configuration_id);
        if(pdata)
        {
            /* renove bsd file */
            (void)ecnr_configuration_remove_bsd_file(pdata->bsd_str);
        }
    }

    m_ecnr_engine_init_state = FALSE;
    m_ecnr_engine_active_configuration_id = 0;
    m_ecnr_engine_default_data_set_loaded = FALSE;

    pr_message("EXIT, remove engine instance, err %d \n", err);
    ETG_TRACE_USR3(("[ecnr_engine_handler_destroy]: EXIT"));

    return err;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_check_operating_mode
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
extern int ecnr_engine_handler_check_operating_mode(void)
{
    int status = (int)ECNR_ERR_OK;

    stecnr_alsa_state* state = ecnr_alsa_get_instance();
    if(!state)
    {
        pr_debug("no alsa module instance available\n");
        status = ECNR_ERR_SERV_NOT_INITIALIZED;
        return status;
    }

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_check_operating_mode();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_check_operating_mode();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_check_operating_mode();
#endif

    /* emit always ASR mode */
    (void)ecnr_object_emit_signal_ecnr_asr_mode_enable((gint)state->ecnr_enable_asr_preprocessing);

    pr_debug("operating mode: %d , ASR mode: %d \n", state->ecnr_op_mode, state->ecnr_asr_mode);

    return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_engine_handler_set_configuration
*
* DESCRIPTION: THis function initialize the sse Engine with the passed ConfigId
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_engine_handler_set_configuration (unsigned short ConfigId)
{
    int err = 0;

    pr_debug("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_engine_handler_set_configuration]: ENTERED"));

    #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    err = ecnr_qwa_handler_set_configuration(ConfigId);
    #else //
    err = ecnr_sse_handler_set_configuration(ConfigId, FALSE);
    #endif

    ETG_TRACE_USR3(("[ecnr_engine_handler_set_configuration]: EXIT"));
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_engine_handler_ecnr_engine_reset
*
* DESCRIPTION: THis function reset the ECNR Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_engine_handler_ecnr_engine_reset(void)
{
    int   status = (int)ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_reset();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_reset();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_reset();
#endif

    return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_engine_get_current_config_id
*
* DESCRIPTION: THis function initialize the sse Engine with the passed ConfigId
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
unsigned short ecnr_engine_get_current_config_id(void)
{
    pr_debug("active configuration id is: %d\n",m_ecnr_engine_active_configuration_id);
    ETG_TRACE_USR3(("[ecnr_engine_get_current_config_id]: active configuration id is: %d\n",m_ecnr_engine_active_configuration_id));
    return m_ecnr_engine_active_configuration_id;
}

/*******************************************************************************
*
* FUNCTION: ecnr_engine_start_debug_session
*
* DESCRIPTION: THis function initialize the sse Engine with the passed ConfigId
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_engine_start_debug_session(void)
{
    int err = 0;

    pr_debug("ENTERED\n");
    ETG_TRACE_USR4(("[ecnr_engine_start_debug_session]: ENTERED"));

    #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    err = ecnr_qwa_rcs_handler_session_create();
    #else
    err = ecnr_sse_debug_interface_start_session();
    if(!err)
    {
        err = ecnr_alsa_enable_sse_debug();
    }
    #endif
    ETG_TRACE_USR4(("[ecnr_engine_start_debug_session]: EXIT"));
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_engine_end_debug_session
*
* DESCRIPTION: THis function initialize the sse Engine with the passed ConfigId
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_engine_end_debug_session(void)
{
    int err = 0;

    pr_debug("entered\n");

    #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    err = ecnr_qwa_rcs_handler_session_destroy() ;
    #else
    err = ecnr_sse_debug_interface_end_session();
    #endif

    err = ecnr_alsa_disable_sse_debug();

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_engine_is_debug_session_active
*
* DESCRIPTION: THis function initialize the sse Engine with the passed ConfigId
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_engine_is_debug_session_active(void)
{
    gboolean state = FALSE;

    #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    state = ecnr_qwa_rcs_get_active_state();
    #else
    state = ecnr_alsa_is_sse_debug_active();
    #endif
    return state;
}

/*******************************************************************************
*
* FUNCTION: ecnr_engine_clean_on_audio_end
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_engine_clean_on_audio_end(void)
{
    int err = ECNR_ERR_OK;

    pr_debug("entered\n");

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    unsigned short ecnr_cfg_id;
    /* rcs handling */

    //err = ecnr_qwa_free_binary_config();
    ecnr_cfg_id = ecnr_engine_get_current_config_id();
    err =  ecnr_qwa_free_binary_config(ecnr_cfg_id);
#endif

    return err;
}

void ecnr_engine_get_version(unsigned int* pArrayLen,
        const int**         pVersionArray,
        const char**        pVersionString,
        const char**        pVersionComment)
{

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
//TODO: retrive QWA version from QWA library.
/*
  unsigned int pqwa_VersionArrayLen = 4;
    const tQwaU16      pqwa_VersionArray[4] = {0};
    const tQwaChar*    pqwa_VersionString = NULL;
    const tQwaChar*    pqwa_VersionComment = NULL;

  ecnr_qwa_get_version ( pqwa_VersionArray, &pqwa_VersionString, &pqwa_VersionComment);
*/
    unsigned int pqwa_VersionArrayLen = 0;
    pArrayLen = &pqwa_VersionArrayLen;
    *pVersionArray = NULL ; //(const int*)pqwa_VersionArray;
    *pVersionString = NULL ;//(const char*)pqwa_VersionString;
    *pVersionComment =  NULL; //(const char*)pqwa_VersionComment;

#else

    unsigned int psse_VersionArrayLen = 0;
    const int *psse_VersionArray = NULL;
    const char *psse_VersionString = NULL;
    const char *psse_VersionComment = NULL;

    ecnr_sse_get_version( &psse_VersionArrayLen, &psse_VersionArray, &psse_VersionString, &psse_VersionComment);

    *pArrayLen = psse_VersionArrayLen;
    *pVersionArray = psse_VersionArray;
    *pVersionString = psse_VersionString;
    *pVersionComment = psse_VersionComment;

#endif

}

/*******************************************************************************
*
* FUNCTION: ecnr_engine_handler_performance_enable
*
* DESCRIPTION:
*
* PARAMETER: state
*
* RETURNVALUE: void
*
*******************************************************************************/
void ecnr_engine_handler_performance_enable(gboolean state)
{
    ETG_TRACE_USR3(("[ecnr_engine_handler_performance_enable]: ENTERED"));
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    //TODO: LSIM SDS VR configuration
    ETG_TRACE_USR3(("[ecnr_engine_handler_performance_enable]: not supportwd"));
#else
    ecnr_sse_handler_performance_enable(state);
#endif
    ETG_TRACE_USR3(("[ecnr_engine_handler_performance_enable]: EXIT"));
    return;

}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_micin_count
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_micin_count(guint* ecnr_mic_in_cnt)
{
    gint   status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_micin_count(ecnr_mic_in_cnt);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_micin_count(ecnr_mic_in_cnt);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_micin_count(ecnr_mic_in_cnt);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_mic_proc_count
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 28.08.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_mic_proc_count(guint* ecnr_mic_proc_cnt)
{
    gint   status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_mic_proc_count(ecnr_mic_proc_cnt);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_mic_proc_count(ecnr_mic_proc_cnt);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_mic_proc_count(ecnr_mic_proc_cnt);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_refin_count
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_refin_count(guint* ecnr_ref_in_cnt)
{
    gint   status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_refin_count(ecnr_ref_in_cnt);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_refin_count(ecnr_ref_in_cnt);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_refin_count(ecnr_ref_in_cnt);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_rcvin_count
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_recvin_count(guint* ecnr_recv_in_cnt)
{
    gint   status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_recvin_count(ecnr_recv_in_cnt);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_recvin_count(ecnr_recv_in_cnt);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_recvin_count(ecnr_recv_in_cnt);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_rcvin_count
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_frame_shift_in(guint* ecnr_frame_shift_in)
{
    gint   status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_frame_shift_in(ecnr_frame_shift_in);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_frame_shift_in(ecnr_frame_shift_in);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_frame_shift_in(ecnr_frame_shift_in);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_frame_shift_out
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_frame_shift_out(guint* ecnr_frame_shift_out)
{
    gint   status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_frame_shift_out(ecnr_frame_shift_out);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_frame_shift_out(ecnr_frame_shift_out);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_frame_shift_out(ecnr_frame_shift_out);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_recv_frame_shift_in
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_recv_frame_shift_in(guint* ecnr_recv_frame_shift_in)
{
    gint   status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_recv_frame_shift_in(ecnr_recv_frame_shift_in);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_recv_frame_shift_in(ecnr_recv_frame_shift_in);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_recv_frame_shift_in(ecnr_recv_frame_shift_in);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_recv_frame_shift_out
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_recv_frame_shift_out(guint* ecnr_recv_frame_shift_out)
{
    gint   status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_recv_frame_shift_out(ecnr_recv_frame_shift_out);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_recv_frame_shift_out(ecnr_recv_frame_shift_out);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_recv_frame_shift_out(ecnr_recv_frame_shift_out);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_ref_frame_shift_in
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_ref_frame_shift_in(guint* ecnr_ref_frame_shift_in)
{
    gint   status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_ref_frame_shift_in(ecnr_ref_frame_shift_in);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_ref_frame_shift_in(ecnr_ref_frame_shift_in);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_ref_frame_shift_in(ecnr_ref_frame_shift_in);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_sample_rate_in
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_sample_rate_in(guint* ecnr_sample_rate_in)
{
    gint   status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_sample_rate_in(ecnr_sample_rate_in);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_sample_rate_in(ecnr_sample_rate_in);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_sample_rate_in(ecnr_sample_rate_in);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_sample_rate_out
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_sample_rate_out(guint* ecnr_sample_rate_out)
{
    gint   status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_sample_rate_out(ecnr_sample_rate_out);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_sample_rate_out(ecnr_sample_rate_out);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_sample_rate_out(ecnr_sample_rate_out);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_recv_sample_rate_in
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_recv_sample_rate_in(guint* ecnr_recv_sample_rate_in)
{
    gint   status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_recv_sample_rate_in(ecnr_recv_sample_rate_in);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_recv_sample_rate_in(ecnr_recv_sample_rate_in);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_recv_sample_rate_in(ecnr_recv_sample_rate_in);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_recv_sample_rate_out
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_recv_sample_rate_out(guint* ecnr_recv_sample_rate_out)
{
    gint   status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_recv_sample_rate_out(ecnr_recv_sample_rate_out);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_recv_sample_rate_out(ecnr_recv_sample_rate_out);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_recv_sample_rate_out(ecnr_recv_sample_rate_out);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_ref_sample_rate_in
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_ref_sample_rate_in(guint* ecnr_ref_sample_rate_in)
{
    gint   status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_ref_sample_rate_in(ecnr_ref_sample_rate_in);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_ref_sample_rate_in(ecnr_ref_sample_rate_in);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_ref_sample_rate_in(ecnr_ref_sample_rate_in);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_audio_type_for_asr
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_audio_type_for_asr(guint32* psize, gchar** pdata)
{
    gint  status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_audio_type_for_asr(psize,pdata);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_audio_type_for_asr(psize,pdata);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_audio_type_for_asr(psize,pdata);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_op_mode
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_op_mode(gint32* ecnr_op_mode)
{
    gint  status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_op_mode(ecnr_op_mode);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_op_mode(ecnr_op_mode);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_op_mode(ecnr_op_mode);
#endif

    pr_warning("ecnr_op_mode -> %d \n",*ecnr_op_mode);

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_asr_mode
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_asr_mode(guint* ecnr_asr_mode)
{
    gint  status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_asr_mode(ecnr_asr_mode);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_asr_mode(ecnr_asr_mode);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_asr_mode(ecnr_asr_mode);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_status_message
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
gint ecnr_engine_handler_get_status_message(char** status_message)
{
    gint  status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_status_message(*status_message);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_status_message(*status_message);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_status_message(*status_message);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_clip_stats
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int  ecnr_engine_handler_get_clip_stats
(
              int  Channel
    ,unsigned int  DataID
    ,unsigned int* ClipFrames
    ,unsigned int* ClipCount
             ,int* MaxAmp
             ,int* MinAmp
)
{
    int  status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_clip_stats(Channel,DataID,ClipFrames,ClipCount,MaxAmp,MinAmp);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_clip_stats(Channel,DataID,ClipFrames,ClipCount,MaxAmp,MinAmp);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_clip_stats(Channel,DataID,ClipFrames,ClipCount,MaxAmp,MinAmp);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_mic_in_clip_stats
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int  ecnr_engine_handler_get_mic_in_clip_stats
(
              int  Channel
    ,unsigned int* ClipFrames
    ,unsigned int* ClipCount
             ,int* MaxAmp
             ,int* MinAmp
)
{
    int  status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_mic_in_clip_stats(Channel,ClipFrames,ClipCount,MaxAmp,MinAmp);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_mic_in_clip_stats(Channel,ClipFrames,ClipCount,MaxAmp,MinAmp);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_mic_in_clip_stats(Channel,ClipFrames,ClipCount,MaxAmp,MinAmp);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_ref_in_clip_stats
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int  ecnr_engine_handler_get_ref_in_clip_stats
(
              int  Channel
    ,unsigned int* ClipFrames
    ,unsigned int* ClipCount
             ,int* MaxAmp
             ,int* MinAmp
)
{
    int  status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_ref_in_clip_stats(Channel,ClipFrames,ClipCount,MaxAmp,MinAmp);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_ref_in_clip_stats(Channel,ClipFrames,ClipCount,MaxAmp,MinAmp);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_ref_in_clip_stats(Channel,ClipFrames,ClipCount,MaxAmp,MinAmp);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_rcv_in_clip_stats
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int  ecnr_engine_handler_get_rcv_in_clip_stats
(
              int  Channel
    ,unsigned int* ClipFrames
    ,unsigned int* ClipCount
             ,int* MaxAmp
             ,int* MinAmp
)
{
    int  status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_rcv_in_clip_stats(Channel,ClipFrames,ClipCount,MaxAmp,MinAmp);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_rcv_in_clip_stats(Channel,ClipFrames,ClipCount,MaxAmp,MinAmp);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_rcv_in_clip_stats(Channel,ClipFrames,ClipCount,MaxAmp,MinAmp);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_mic_out_clip_stats
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int  ecnr_engine_handler_get_mic_out_clip_stats
(
              int  Channel
    ,unsigned int* ClipFrames
    ,unsigned int* ClipCount
             ,int* MaxAmp
             ,int* MinAmp
)
{
    int  status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_mic_out_clip_stats(Channel,ClipFrames,ClipCount,MaxAmp,MinAmp);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_mic_out_clip_stats(Channel,ClipFrames,ClipCount,MaxAmp,MinAmp);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_mic_out_clip_stats(Channel,ClipFrames,ClipCount,MaxAmp,MinAmp);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_get_rcv_out_clip_stats
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int  ecnr_engine_handler_get_rcv_out_clip_stats
(
              int  Channel
    ,unsigned int* ClipFrames
    ,unsigned int* ClipCount
             ,int* MaxAmp
             ,int* MinAmp
)
{
    int  status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_get_rcv_out_clip_stats(Channel,ClipFrames,ClipCount,MaxAmp,MinAmp);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_get_rcv_out_clip_stats(Channel,ClipFrames,ClipCount,MaxAmp,MinAmp);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_get_rcv_out_clip_stats(Channel,ClipFrames,ClipCount,MaxAmp,MinAmp);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_send_mute_switch
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_send_mute_switch(guchar ecnr_send_mute_switch)
{
    int  status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_send_mute_switch(ecnr_send_mute_switch);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_send_mute_switch(ecnr_send_mute_switch);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_send_mute_switch(ecnr_send_mute_switch);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_send_gain_delta
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_send_gain_delta(int ecnr_send_gain_delta_lvl)
{
    int  status = (int)ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_send_gain_delta(ecnr_send_gain_delta_lvl);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_send_gain_delta(ecnr_send_gain_delta_lvl);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_send_gain_delta(ecnr_send_gain_delta_lvl);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_pass_through_mode
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_pass_through_mode(guchar ecnr_pass_through_mode)
{
    int  status = (int)ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_pass_through_mode(ecnr_pass_through_mode);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_pass_through_mode(ecnr_pass_through_mode);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_pass_through_mode(ecnr_pass_through_mode);
#endif

    return status;
}


/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_nr_comb_floor
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_nr_comb_floor(int ecnr_nr_comb_floor)
{
    int  status = (int)ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_nr_comb_floor(ecnr_nr_comb_floor);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_nr_comb_floor(ecnr_nr_comb_floor);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_nr_comb_floor(ecnr_nr_comb_floor);
#endif

    return status;
}


/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_echo_cancellation
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_echo_cancellation(guchar ecnr_aec_state)
{
    int  status = (int)ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_echo_cancellation(ecnr_aec_state);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_echo_cancellation(ecnr_aec_state);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_echo_cancellation(ecnr_aec_state);
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_noise_reduction
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_noise_reduction(guchar ecnr_nr_state)
{
    int  status = (int)ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_noise_reduction(ecnr_nr_state);
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_noise_reduction(ecnr_nr_state);
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_noise_reduction(ecnr_nr_state);
#endif

    return status;
}


/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_config_aa_voice_wb_pre
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_config_aa_voice_wb_pre(void)
{
    int status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_config_aa_voice_wb_lsim_pre();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_config_aa_voice_wb_pre();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_config_aa_voice_wb_pre();
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_config_aa_voice_wb_post
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_config_aa_voice_wb_post(void)
{
    int status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_config_aa_voice_wb_lsim_post();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_config_aa_voice_wb_post();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_config_aa_voice_wb_post();
#endif

    return status;
}













/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_config_bt_phone_nb_lsim_pre
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_config_bt_phone_nb_lsim_pre(void)
{
    int status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_config_bt_phone_nb_lsim_pre();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_config_bt_phone_nb_lsim_pre();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_config_bt_phone_nb_lsim_pre();
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_config_bt_phone_nb_lsim_post
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_config_bt_phone_nb_lsim_post(void)
{
    int status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_config_bt_phone_nb_lsim_post();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_config_bt_phone_nb_lsim_post();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_config_bt_phone_nb_lsim_post();
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_config_bt_phone_wb_lsim_pre
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_config_bt_phone_wb_lsim_pre(void)
{
    int status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_config_bt_phone_wb_lsim_pre();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_config_bt_phone_wb_lsim_pre();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_config_bt_phone_wb_lsim_pre();
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_config_bt_phone_wb_lsim_post
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_config_bt_phone_wb_lsim_post(void)
{
    int status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_config_bt_phone_wb_lsim_post();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_config_bt_phone_wb_lsim_post();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_config_bt_phone_wb_lsim_post();
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_config_aa_voice_wb_lsim_pre
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_config_aa_voice_wb_lsim_pre(void)
{
    int status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_config_aa_voice_wb_lsim_pre();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_config_aa_voice_wb_lsim_pre();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_config_aa_voice_wb_lsim_pre();
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_config_aa_voice_wb_lsim_post
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_config_aa_voice_wb_lsim_post(void)
{
    int status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_config_aa_voice_wb_lsim_post();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_config_aa_voice_wb_lsim_post();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_config_aa_voice_wb_lsim_post();
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_config_cp_voice_wb_lsim_pre
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_config_cp_voice_wb_lsim_pre(void)
{
    int status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_config_cp_voice_wb_lsim_pre();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_config_cp_voice_wb_lsim_pre();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_config_cp_voice_wb_lsim_pre();
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_config_cp_voice_wb_lsim_post
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_config_cp_voice_wb_lsim_post(void)
{
    int status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_config_cp_voice_wb_lsim_post();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_config_cp_voice_wb_lsim_post();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_config_cp_voice_wb_lsim_post();
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_config_cp_phone_nb_lsim_pre
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_config_cp_phone_nb_lsim_pre(void)
{
    int status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_config_cp_phone_nb_lsim_pre();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_config_cp_phone_nb_lsim_pre();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_config_cp_phone_nb_lsim_pre();
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_config_cp_phone_nb_lsim_post
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_config_cp_phone_nb_lsim_post(void)
{
    int status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_config_cp_phone_nb_lsim_post();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_config_cp_phone_nb_lsim_post();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_config_cp_phone_nb_lsim_post();
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_config_cp_phone_wb_lsim_pre
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_config_cp_phone_wb_lsim_pre(void)
{
    int status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_config_cp_phone_wb_lsim_pre();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_config_cp_phone_wb_lsim_pre();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_config_cp_phone_wb_lsim_pre();
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_config_cp_phone_wb_lsim_post
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_config_cp_phone_wb_lsim_post(void)
{
    int status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_config_cp_phone_wb_lsim_post();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_config_cp_phone_wb_lsim_post();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_config_cp_phone_wb_lsim_post();
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_config_int_voice_wb_lsim_pre
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_config_int_voice_wb_lsim_pre(void)
{
    int status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_config_int_voice_wb_lsim_pre();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_config_int_voice_wb_lsim_pre();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_config_int_voice_wb_lsim_pre();
#endif

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_engine_handler_set_config_int_voice_wb_lsim_post
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                 | Modification
 * 08.03.2019   | Ranjit Susal(RBEI/ECO) | Initial Revision
 *****************************************************************************/
int ecnr_engine_handler_set_config_int_voice_wb_lsim_post(void)
{
    int status = ECNR_ERR_OK;

#ifdef D_ECNR_LIB_QWA
    status = ecnr_qwa_handler_set_config_int_voice_wb_lsim_post();
#elif  D_ECNR_LIB_SSE_3_17
    status = ecnr_sse_handler_set_config_int_voice_wb_lsim_post();
#elif  D_ECNR_LIB_SSE_4_03
    status = ecnr_sse_handler_set_config_int_voice_wb_lsim_post();
#endif

    return status;
}
