/******************************************************************************
 * FILE        : ecnr-diag-eol-access.c
 * PROJECT     : Gen3
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2013 RBCM GMBH
 * HISTORY     :
 * Date        | Author                 | Modification
 * 06.08.2013  | BSOT Andrea Bueter     | initial version
 * 03.05.2014  | Patrick Rey            | diagnosis EOL feature handling
 *****************************************************************************/




#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX

/*********************************************************************************/
/*                         include                                               */
/*********************************************************************************/
#include <glib.h>
#include <glib-unix.h>
#include <glib/gi18n.h>
#include <glib-object.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include "qwa.h"
#include "qwa_defs.h"
#include "qwa_err.h"
#include "qwa_rcs.h"
#include "qwa_rcs_err.h"

#include "bpcl.h"

#include "ecnr-error.h"
#include "ecnr-common-defs.h"
#include "ecnr-service.h"
#include "ecnr-alsa.h"
#include "ecnr-diag-eol-access.h"
#include "ecnr-qwa-handler.h"

/*********************************************************************************/
/*                         define                                                */
/*********************************************************************************/

/*********************************************************************************/
/*                         macro                                                 */
/*********************************************************************************/

/*********************************************************************************/
/*                         static variable                                       */
/*********************************************************************************/

static  OSAL_tIODescriptor m_ioEolDesc = OSAL_ERROR;  // EOL file descriptor.
static  guint32 m_u32LastErrorCode = 0 ;         // Last error code.


#define D_SAFE_LOOP_MAX 300


/*******************************************************************************
*
* FUNCTION: ecnr_diag_eol_access_byte_to_ushort
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static guint16 ecnr_diag_eol_access_byte_to_ushort(const guint8 u8valB1, const guint8 u8valB0)
{
  guint16 u16val;
  u16val = u8valB1 << 8;
  u16val |= u8valB0;

  return u16val;
}

/*******************************************************************************
*
* FUNCTION: ecnr_diag_eol_access_byte_to_short
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static gint16 ecnr_diag_eol_access_byte_to_short(const guint8 u8valB1, const guint8 u8valB0)
{
  gint16 i16val;
  i16val = u8valB1 << 8;
  i16val |= u8valB0;

  return i16val;
}

static int ecnr_diag_eol_access_set_i16_param(struct ecnr_qwa_parameter* pd, guint8* peol)
{
    gint   err = ECNR_ERR_OK;
    gint16    val = 0;
    guint8 *au8Data = NULL;
    guint8 u8val = 0;

    pr_debug("ENTERED\n");

    if(!pd || !peol)
        return ECNR_ERR_NULL_POINTER;

    au8Data = peol;

    if(pd->eol_type == ECNR_EOL_TYPE_I16)
    {
        if(pd->param_type == ECNR_PARAM_SET_CONST_VALUE)
        {
            (void)memcpy( (void*)&val, pd->ecnr_qwa_pointer_set_value, sizeof(gint16));
            pr_message("qwa parameter %s(%d) as i16 eol constant set to :%d\n", pd->ecnr_qwa_param_name, pd->ecnr_qwa_param_id,val);
        }
        else
        {
            val = ecnr_diag_eol_access_byte_to_short(au8Data[pd->ecnr_qwa_array_start_offset + 1], au8Data[pd->ecnr_qwa_array_start_offset]);
            pr_message("qwa parameter %s(%d) with offset(%d) set to :%d\n", pd->ecnr_qwa_param_name, pd->ecnr_qwa_param_id,pd->ecnr_qwa_array_start_offset, val);
        }
    }
    else if(pd->eol_type == ECNR_EOL_TYPE_U8)
    {
        if(pd->param_type == ECNR_PARAM_SET_CONST_VALUE)
        {
            (void)memcpy( (void*)&u8val, pd->ecnr_qwa_pointer_set_value, sizeof(guint8));
            pr_message("qwa parameter %s(%d) as u8 eol constant set to :%d\n", pd->ecnr_qwa_param_name, pd->ecnr_qwa_param_id,u8val);
        }
        else
        {
            u8val = (gint16)au8Data[pd->ecnr_qwa_array_start_offset];
            pr_message("qwa parameter %s(%d) with offset(%d) set to :%d\n", pd->ecnr_qwa_param_name, pd->ecnr_qwa_param_id,pd->ecnr_qwa_array_start_offset, u8val);
        }
        val = (gint16)u8val;
    }
    err = ecnr_qwa_handler_set_data(pd->ecnr_qwa_param_id,0, sizeof(gint16),(void*)&val);

    return err;
}

static int ecnr_diag_eol_access_set_i32_param(struct ecnr_qwa_parameter* pd, guint8* peol)
{
    gint   err = ECNR_ERR_OK;
    gint32    val = 0;
    guint8 *au8Data = NULL;
    guint8  u8val = 0;
    guint16 u16val = 0;

    pr_debug("ENTERED\n");

    if(!pd || !peol)
        return ECNR_ERR_NULL_POINTER;

    au8Data = peol;

    if(pd->eol_type == ECNR_EOL_TYPE_U8)
    {
        if(pd->param_type == ECNR_PARAM_SET_CONST_VALUE)
        {
            (void)memcpy( (void*)&u8val, pd->ecnr_qwa_pointer_set_value, sizeof(guint8));
            pr_message("qwa parameter %s(%d) as u8 eol constant set to :%d\n", pd->ecnr_qwa_param_name, pd->ecnr_qwa_param_id,u8val);
        }
        else
        {
            u8val = (guint16)au8Data[pd->ecnr_qwa_array_start_offset];
            pr_message("qwa parameter %s(%d) with offset(%d) set to :%d\n", pd->ecnr_qwa_param_name, pd->ecnr_qwa_param_id,pd->ecnr_qwa_array_start_offset, u8val);
        }
        val = (gint32)u8val;
    }
    else if(pd->eol_type == ECNR_EOL_TYPE_U16)
    {
        if(pd->param_type == ECNR_PARAM_SET_CONST_VALUE)
        {
            (void)memcpy( (void*)&u16val, pd->ecnr_qwa_pointer_set_value, sizeof(guint16));
            pr_message("qwa parameter %s(%d) as u16 eol constant set to :%d\n", pd->ecnr_qwa_param_name, pd->ecnr_qwa_param_id,u16val);
        }
        else
        {
            u16val = ecnr_diag_eol_access_byte_to_ushort(au8Data[pd->ecnr_qwa_array_start_offset + 1], au8Data[pd->ecnr_qwa_array_start_offset]);
            pr_message("qwa parameter %s(%d) with offset(%d) set to :%d\n", pd->ecnr_qwa_param_name, pd->ecnr_qwa_param_id,pd->ecnr_qwa_array_start_offset, u16val);
        }
        val = (gint32)u16val;
    }
    else
    {
        pr_debug("no implementation available\n");
        return ECNR_ERR_OK;
    }
    err = ecnr_qwa_handler_set_data(pd->ecnr_qwa_param_id,0, sizeof(gint32),(void*)&val);

    return err;
}




static int ecnr_diag_eol_access_set_u16_param(struct ecnr_qwa_parameter* pd, guint8* peol)
{
    gint   err = ECNR_ERR_OK;
    guint16    val = 0;
    guint8 *au8Data = NULL;
    guint8 u8val = 0;

    pr_debug("ENTERED\n");

    if(!pd || !peol)
        return ECNR_ERR_NULL_POINTER;

    au8Data = peol;

    if(pd->eol_type == ECNR_EOL_TYPE_U16)
    {
        if(pd->param_type == ECNR_PARAM_SET_CONST_VALUE)
        {
            (void)memcpy( (void*)&val, pd->ecnr_qwa_pointer_set_value, sizeof(guint16));
            pr_message("qwa parameter %s(%d) as u16 eol constant set to :%d\n", pd->ecnr_qwa_param_name, pd->ecnr_qwa_param_id,val);
        }
        else
        {
            val = ecnr_diag_eol_access_byte_to_ushort(au8Data[pd->ecnr_qwa_array_start_offset + 1], au8Data[pd->ecnr_qwa_array_start_offset]);
            pr_message("qwa parameter %s(%d) with offset(%d) set to :%d\n", pd->ecnr_qwa_param_name, pd->ecnr_qwa_param_id,pd->ecnr_qwa_array_start_offset, val);
        }
    }
    else if(pd->eol_type == ECNR_EOL_TYPE_U8)
    {
        if(pd->param_type == ECNR_PARAM_SET_CONST_VALUE)
        {
            (void)memcpy( (void*)&u8val, pd->ecnr_qwa_pointer_set_value, sizeof(guint8));
            pr_message("qwa parameter %s(%d) as u8 eol constant set to :%d\n", pd->ecnr_qwa_param_name, pd->ecnr_qwa_param_id,u8val);
        }
        else
        {
            u8val = (guint16)au8Data[pd->ecnr_qwa_array_start_offset];
            pr_message("qwa parameter %s(%d) with offset(%d) set to :%d\n", pd->ecnr_qwa_param_name, pd->ecnr_qwa_param_id,pd->ecnr_qwa_array_start_offset, u8val);
        }
        val = (guint16)u8val;
    }
    err = ecnr_qwa_handler_set_data(pd->ecnr_qwa_param_id,0, sizeof(guint16),(void*)&val);

    return err;
}

static int ecnr_diag_eol_access_set_select_param(struct ecnr_qwa_parameter* pd, guint8* peol)
{
    int err = ECNR_ERR_OK;
    guint8 *au8Data = NULL;
    gint val = 0;
    gint32 qwaSelect= (gint32)qwaOff;
    gboolean bDoSet = TRUE;

    pr_debug("ENTERED\n");

    if(!pd || !peol)
        return ECNR_ERR_NULL_POINTER;

    au8Data = peol;

    /* set switch parameter */
    if(pd->param_type == ECNR_PARAM_SET_CONST_VALUE)
    {
        (void)memcpy( (void*)&val, pd->ecnr_qwa_pointer_set_value, sizeof(gint));
    }
    else
    {
        val = (gint)au8Data[pd->ecnr_qwa_array_start_offset];
    }

    if(   pd->ecnr_qwa_param_id == qwaOpMode
        ||pd->ecnr_qwa_param_id == qwaWNDBMode
    )
    {

        if (val == 0)
        {
            if(pd->ecnr_qwa_param_id == qwaOpMode)
                qwaSelect= (gint32)qwaHandsFree;
            else if(pd->ecnr_qwa_param_id == qwaWNDBMode)
                qwaSelect= (gint32)qwaSingleChannel;
        }
        else if (val == 1)
        {
            if(pd->ecnr_qwa_param_id == qwaOpMode)
                qwaSelect= (gint32)qwaVoiceRecog;
            else if(pd->ecnr_qwa_param_id == qwaWNDBMode)
                qwaSelect= (gint32)qwaMultiChannel;

        }
        else if (val == 2 && pd->ecnr_qwa_param_id == qwaWNDBMode)
        {
            qwaSelect= (gint32)qwaCombined;
        }
        else
        {
            pr_warning("%s(%d) value read not correct\n", pd->ecnr_qwa_param_name, val);
            bDoSet = FALSE;
        }
    }
    else
    {
        if (val)
        {
            qwaSelect = (gint32)qwaOn;
        }
        else
        {
            qwaSelect = (gint32)qwaOff;
        }
    }

    if(bDoSet)
    {
        err = ecnr_qwa_handler_set_data(pd->ecnr_qwa_param_id,0, sizeof(gint32),(void*)&qwaSelect);
        if(pd->param_type == ECNR_PARAM_SET_CONST_VALUE)
        {
            pr_message("qwa parameter %s(%d) as constant value set to :%d\n", pd->ecnr_qwa_param_name, pd->ecnr_qwa_param_id,qwaSelect);
        }
        else
        {
            pr_message("qwa parameter %s(%d) with offset(%d) set to :%d\n", pd->ecnr_qwa_param_name, pd->ecnr_qwa_param_id,pd->ecnr_qwa_array_start_offset, qwaSelect);
        }
    }

    return err;
}


static int ecnr_diag_eol_access_set_array_i16_param(struct ecnr_qwa_parameter* pd, guint8* peol)
{
    int err = ECNR_ERR_OK;
    guint8 *au8Data = NULL;
 //   int val = 0;
    gint16* pi16 = (gint16*)NULL;
    guint16 indexCnt = 0;

    pr_debug("ENTERED\n");

    if(!pd || !peol)
        return ECNR_ERR_NULL_POINTER;

    au8Data = peol;

    /* set qwaRecvAGCAmpTgts parameter */
    if(pd->ecnr_qwa_param_length == 0)
        return ECNR_ERR_CONFIG_EOL_IO_ERROR;

    pi16 = g_malloc0(pd->ecnr_qwa_param_length * sizeof(gint16));
    if(!pi16)
        return ECNR_ERR_NULL_POINTER;

    for(indexCnt = 0; indexCnt < pd->ecnr_qwa_param_length; indexCnt++)
    {
        pi16[indexCnt] = ecnr_diag_eol_access_byte_to_short(au8Data[pd->ecnr_qwa_array_start_offset + (1 + (2 * indexCnt))], \
                                                             au8Data[pd->ecnr_qwa_array_start_offset + 2 * indexCnt]);
        pr_message("qwa parameter %s(%d) with offset(%d)set val[%d] from %d to :%d\n", pd->ecnr_qwa_param_name,pd->ecnr_qwa_param_id, pd->ecnr_qwa_array_start_offset,indexCnt,pd->ecnr_qwa_param_length, pi16[indexCnt]);
    }
    err = ecnr_qwa_handler_set_data(pd->ecnr_qwa_param_id,0, (pd->ecnr_qwa_param_length * sizeof(gint16)),(void*)pi16);


    g_free(pi16);
    pi16 = NULL;

    pr_debug("EXIT with error(%d) \n", err);
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_diag_eol_access_set_param
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_diag_eol_access_set_param(struct ecnr_qwa_parameter *pparam, guint8* pdata)
{
    int err = ECNR_ERR_OK;
    struct ecnr_qwa_parameter* pp = (struct ecnr_qwa_parameter*)NULL;
    int i = 0;
    gboolean process = TRUE;
    int loop = 0;
    const gchar *str_invalid_param = "qwaInvalidParam";


    pr_debug("ENTERED\n");

    if(!pparam)
    {
        pr_debug("reference to the eol configuration table is NULL\n");
        return ECNR_ERR_NULL_POINTER;
    }

    if(!pdata)
    {
        pr_debug("eol table not available\n");
        return ECNR_ERR_NULL_POINTER;
    }


    /* go though all parameter */

    while (process)
    {
        /*walk through all structur element  */
        pp = &pparam[i];

        if(pp->ecnr_qwa_param_name)
            pr_debug("set paramter %s\n",pp->ecnr_qwa_param_name);

        /* warning lint:Info 774: prio1, replace code "if (pp->ecnr_qwa_param_name == "qwaInvalidParam")" */
        if (!g_strcmp0(str_invalid_param, pp->ecnr_qwa_param_name))
        {
            pr_debug("last parameter in the configuration table\n");
            process = FALSE;
        }
        else
        {
            switch(pp->ecnr_qwa_param_type)
            {
                case ECNR_TYPE_SELECT:
                    err = ecnr_diag_eol_access_set_select_param(pp, pdata);
                    break;
                case ECNR_TYPE_U16:
                    err = ecnr_diag_eol_access_set_u16_param(pp, pdata);
                    break;
                case ECNR_TYPE_I16:
                    err = ecnr_diag_eol_access_set_i16_param(pp, pdata);
                    break;
                case ECNR_TYPE_I32:
                    err = ecnr_diag_eol_access_set_i32_param(pp, pdata);
                    break;
                case ECNR_TYPE_ARRAY_U16:
                case ECNR_TYPE_ARRAY_I16:
                    err = ecnr_diag_eol_access_set_array_i16_param(pp, pdata);
                     break;
                case ECNR_TYPE_U32:
                default:
                    break;
            }
            if(err)
            {
                pr_message("failed to set parameter: %d with error: %d\n",pp->ecnr_qwa_param_id, err);
                if(err == QWA_ERR_NOT_INIT) //  || err == QWA_ERR_PARAM)
                {
                    pr_message("try to set a parameter to a module which is not initialized, parameter:%d with error:%d\n",pp->ecnr_qwa_param_id, err);

                    /* ignore error*/
                    err = ECNR_ERR_OK;
                }
                else
                    goto error_out;
            }
        }

        i++;
        if( ++loop  > D_SAFE_LOOP_MAX )
            process = FALSE;
    }

error_out:

    pr_debug("EXIT with error(%d) \n", err);
    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_diag_eol_open
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
gboolean ecnr_diag_eol_open(void)
{
    gboolean ret = FALSE;

    pr_debug("ENTERED\n");

    // Open the EOL Device.
    m_ioEolDesc =  OSAL_IOOpen(OSAL_C_STRING_DEVICE_DIAG_EOL, OSAL_EN_READWRITE);

    if (OSAL_ERROR == m_ioEolDesc)
    {
        m_u32LastErrorCode =  OSAL_u32ErrorCode();
        pr_critical("failed to open OSAL_C_STRING_DEVICE_DIAG_EOL device with err:%d\n", m_u32LastErrorCode);
    }
    else
    {
        pr_debug("OSAL_C_STRING_DEVICE_DIAG_EOL device successfully open\n");
        ret = TRUE;
    }
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_diag_eol_close
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
gboolean ecnr_diag_eol_close(void)
{
    gboolean ret = TRUE;
    pr_debug("ENTERED\n");

    if ((OSAL_ERROR != m_ioEolDesc) && (OSAL_ERROR == OSAL_s32IOClose(m_ioEolDesc)))
    {
        ret = FALSE;
        pr_critical("close osal IO device failed with error: %d \n", OSAL_u32ErrorCode());
    }
    m_ioEolDesc =  OSAL_ERROR;

    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_diag_eol_Read
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
gboolean ecnr_diag_eol_Read(const guint8 cu8Table, const guint16 cu16Offset, guint8* rfu8Value)
{
    gboolean ret = FALSE;

    pr_debug("ENTERED\n");

    if (OSAL_ERROR != m_ioEolDesc)
    {
        pr_debug("read from eoltable:%d, lenght:%d, offset:%d \n", cu8Table, sizeof(tU8),  cu16Offset);

        OSAL_trDiagEOLEntry oDiagEolEntry;
        oDiagEolEntry.u8Table         =  cu8Table;
        oDiagEolEntry.u16Offset       =  cu16Offset;
        oDiagEolEntry.u16EntryLength  =  sizeof(tU8);
        oDiagEolEntry.pu8EntryData    =  rfu8Value;

        if (OSAL_ERROR != OSAL_s32IORead(m_ioEolDesc, (tS8*)&oDiagEolEntry,(tS32)sizeof(oDiagEolEntry)))
        {
            pr_debug("read from osal device successfully\n");
            ret  =  TRUE;
        }
        else
        {
            m_u32LastErrorCode =  OSAL_u32ErrorCode();
            pr_warning("osal IO read error : %d \n",  m_u32LastErrorCode);
        }
    }

    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_diag_eol_Read_qwa_values
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
gboolean ecnr_diag_eol_Read_qwa_values(const guint8 cu8Table, const guint16 cu16Offset, tU16 cu16buflen, guint8* rfu8Value)
{
    gboolean ret = FALSE;


    pr_debug("ENTERED\n");

    if (OSAL_ERROR != m_ioEolDesc)
    {

        pr_debug("read from eoltable:%d, lenght:%d, offset:%d \n", cu8Table, cu16buflen * sizeof(tU8), cu16Offset);

        OSAL_trDiagEOLEntry oDiagEolEntry;
        oDiagEolEntry.u8Table         =  cu8Table;
        oDiagEolEntry.u16Offset       =  cu16Offset;
        oDiagEolEntry.u16EntryLength  =  cu16buflen * sizeof(tU8);
        oDiagEolEntry.pu8EntryData    =  &rfu8Value[0];


        if (OSAL_ERROR != OSAL_s32IORead(m_ioEolDesc, (tS8*)&oDiagEolEntry,(tS32)sizeof(oDiagEolEntry)))
        {
            pr_debug("read from osal device successfully\n");
            ret = TRUE;
        }
        else
        {
            m_u32LastErrorCode = OSAL_u32ErrorCode();
            pr_warning("osal IO read error : %d \n",  m_u32LastErrorCode);
        }
    }

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_diag_eol_get_last_Error_code
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
guint32  ecnr_diag_eol_get_last_Error_code(void)
{
    pr_message("osal IO read error : %d \n",  m_u32LastErrorCode);
    return m_u32LastErrorCode;
}

/*******************************************************************************
*
* FUNCTION: ecnr_diag_eol_is_open
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
gboolean ecnr_diag_eol_is_open(void)
{
    if(OSAL_ERROR != m_ioEolDesc)
    {
        pr_debug("osal IO device is open\n");
        return TRUE;
    }
    else
    {
        pr_debug("osal IO device not open\n");
        return FALSE;
    }
}




#endif // #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
