/******************************************************************************
 * FILE        : ecnr-configuration.c
 * PROJECT     : Gen3 and Gen4
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2013 RBCM GMBH
 * HISTORY     :
 * Date        | Author                 | Modification
 * 20.08.2013  | Patrick Rey            | initial version
 *             |                        | configuration features handling
 *****************************************************************************/



/*******************************************************************************
              DEFINES
*******************************************************************************/
//#define D_KDS_SUPPORT



/*******************************************************************************
              GENERATED CODE HEADERS
*******************************************************************************/

#include "ecnr-types.h"
#include "ecnr-error.h"
#include "ecnr-common-defs.h"
#include "ecnr-object.h"
#include "ecnr-service.h"
#include "ecnr-alsa.h"
#include "ecnr-sse-handler.h"
#include "ecnr-configuration.h"
#include "ecnr-datapool-access.h"

#define ECNR_S_IMPORT_INTERFACE_DP
#include "ecnr-datapool-default.h"
#include "ecnr-configuration-data.h"

#ifdef D_KDS_SUPPORT
#define OSAL_IO_GEN2
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#include "osalio_public.h"


#define KDS_CFG_HEADER_SIZE     (1+2+1+2+1+3+2)     // NOTE: See KDSDataHeader but do NOT use sizeof(KDSDataHeader) for this!
                                                    // The data in KDS is packed while the compiler inserts fillbytes to achieve word alignment of the structure's members!
#define KDS_CFG_START         11
#define KDS_CFG_BLOCK_LEN     230
#define KDS_CFG_LEN           (2058 + 1 - 11)
#define KDS_MAX_BSD_LEN       (KDS_CFG_LEN - KDS_CFG_HEADER_SIZE)
#endif


/* Needed for Trace */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "ecnr-trace.h"
#include "etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ECNR_CONFIG
#include "trcGenProj/Header/ecnr-configuration.c.trc.h"
#endif


/*******************************************************************************
              STRUCTURE DEFINITION
*******************************************************************************/

#ifdef D_KDS_SUPPORT
typedef struct
{
    tU8     InternalMajorVersion;   // Major version number, incremented with each incompatible update of Nuance's SSE lib
    tU16    InternalMinorVersion;   // Minor version number, incremented with each compatible update of Nuance's SSE lib
    tU8     DataSetId;              // Just a counter, incremented for each set of data
    tU16    CarId;                  // Type of car, this data set was made for
    tU8     Interieur;              // ...
    tU8     LibMajorVersion;        // Major version number of Nuance library, used to create this data set
    tU16    LibMinorVersion;        // Minor version number of Nuance library, used to create this data set
    tU16    BSDSize;                // Size of BSD file in bytes
} KDSDataHeader;
#endif
/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/

#ifdef D_KDS_SUPPORT
static KDSDataHeader    KDSDataHeaderVoice, KDSDataHeaderPhone;
static const char       NoBSDInfoStr[] = "BSD info not available";
static const tU8        InternalMajorVersion = 2;   // Used to determine compatibility with BSD in KDS, see InternalMajorVersion in KDSDataHeader
static const tU16       InternalMinorVersion = 1;   // Used to determine compatibility with BSD in KDS, see InternalMinorVersion in KDSDataHeader
#endif

/*******************************************************************************
              HELPER FUNCTIONS
*******************************************************************************/
#define ECNR_NUM_CONFIG ARRAYSIZE(ECNR_CONFIG_table)

/*******************************************************************************
              BSD CONFIGURATION TABLE
*******************************************************************************/

static gboolean bConfigPrinted = FALSE;
static gboolean bNewdpBTPhoneNB = FALSE;
static gboolean bNewdpBTPhoneWB = FALSE;
static gboolean bNewdpBTVoiceNB = FALSE;
static gboolean bNewdpBTVoiceWB = FALSE;
static gboolean bNewdpBTSiriNB = FALSE;
static gboolean bNewdpBTSiriWB = FALSE;
static gboolean bNewdpCPVoiceNB = FALSE;
static gboolean bNewdpCPVoiceWB = FALSE;
static gboolean bNewdpCPVoiceSWB = FALSE;
static gboolean bNewdpCPPhoneNB = FALSE;
static gboolean bNewdpCPPhoneWB = FALSE;
static gboolean bNewdpCPPhoneSWB = FALSE;
static gboolean bNewdpCPFTimeSWB = FALSE;
static gboolean bNewdpAAVoiceNB = FALSE;
static gboolean bNewdpAAVoiceWB = FALSE;
static gboolean bNewdpAAVoiceSWB = FALSE;
static gboolean bNewdpAAPhoneNB = FALSE;
static gboolean bNewdpAAPhoneWB = FALSE;
static gboolean bNewdpAAPhoneSWB = FALSE;
static gboolean bNewdpINTVoiceNB = FALSE;
static gboolean bNewdpINTVoiceWB = FALSE;
static gboolean bNewdpWUWVoiceWB = FALSE;
static gboolean bNewdpVPAVoiceWB = FALSE;




/*********************************************************************************/
/*                         static function                                       */
/*********************************************************************************/

/*******************************************************************************
*
* FUNCTION: ecnr_get_config_data
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
void ecnr_configuration_print_user_config(void)
{
    int ct = 0;
    char* appIdstr = (char*)NULL;

    while (ct < ECNR_CFG_ID_TABLE_SIZE)
    {
        /* which app ?*/
        switch(ECNR_USER_CONF_table[ct].EcnrAppId)
        {
            case ECNR_APPID_PHONE:
                appIdstr = g_strdup("ECNR_APPID_PHONE");
                break;
            case ECNR_APPID_SPI:
                appIdstr = g_strdup("ECNR_APPID_SPI");
                break;
            case ECNR_APPID_SDS:
                appIdstr = g_strdup("ECNR_APPID_SDS");
                break;
            case ECNR_APPID_TEST:
                appIdstr = g_strdup("ECNR_APPID_TEST");
                break;
            case ECNR_APPID_WEB:
                appIdstr = g_strdup("ECNR_APPID_WEB");
                break;
            case ECNR_APPID_NA:
            default:
                break;
        }

        if(appIdstr)
        {
            pr_message("%s(%d), CFGId(%d), ECNRId(%d)\n",
                        appIdstr,
                        ECNR_USER_CONF_table[ct].EcnrAppId,
                        ECNR_USER_CONF_table[ct].EcnrAppConfId,
                        ECNR_USER_CONF_table[ct].EcnrConfId);


            ct = ct + 1;
            g_free(appIdstr);
            appIdstr = (char*)NULL;
        }
    }
    pr_message("Number of  User Configuration : %d \n", ECNR_CFG_ID_TABLE_SIZE);
}



/*******************************************************************************
*
* FUNCTION: ecnr_get_config_data
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
void ecnr_configuration_print_config_data(void)
{
    int i;
    const struct EcnrConfig *pdata;

    /* print ecnr configuration*/
    for (i = 0, pdata = ECNR_CONFIG_table; i < (int)ECNR_NUM_CONFIG; i++, pdata++)
    {
        pr_message("configId: %d, bsd: %s, opeth: %s\n", pdata->configId, pdata->bsd_str, pdata->opeth_str);
    }

    /* print user configuration */
    ecnr_configuration_print_user_config();

    pr_message("Number of  Valid Configuration : %lu \n", ECNR_NUM_CONFIG);
}


/*******************************************************************************
*
* FUNCTION: ecnr_configuration_get_ecnr_config_Id
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
unsigned short ecnr_configuration_get_ecnr_config_Id(unsigned short configId)
{
    unsigned short ct = 0;
    unsigned short ret = 0;

    pr_debug("ENTERED");

    while (ct < ECNR_CFG_ID_TABLE_SIZE)
    {
        if (ECNR_USER_CONF_table[ct].EcnrAppConfId == configId)
        {
            ret = ECNR_USER_CONF_table[ct].EcnrConfId;
            ct = ECNR_CFG_ID_TABLE_SIZE + 1;
        }
        ct = (unsigned short)(ct + 1);
    }

    pr_debug("User Id :%d, ecnr CfgId: %d\n", configId, ret);

   return ret;
}

/* default sync data */
#ifndef D_CREATE_ALWAYS_DYN_SYNC_DATA
static guint32  devprop[1] = {0};
#endif
static gchar*   inNodev[1] = {"NoDev"};
static gchar*   outNodev[1] = {"NoDev"};

/* sync data dynamic structure */
static guint32* pprop   = (guint32*)NULL;
static gchar**  pIndev  = (gchar**)NULL;
static gchar**  pOutdev = (gchar**)NULL;



#define MASK_IN_DEV_NB 0x0F
#define MASK_OUT_DEV_NB 0xF0
#define MASK_FEATURE_DIPO_STRING_SELECT 0x01


static gint16 ecnr_configuration_get_in_dev_list(gchar*** ecnr_in_dev, unsigned short config_id, gboolean* isSyncDataMatching )
{
    gint i = 0;
    gint in_dev_nb = 0;
    gint nbdev = 0;
    const struct ecnr_id_spec_feature *p = (struct ecnr_id_spec_feature*)NULL;
    struct sync_stream_descriptor *ps =  (struct sync_stream_descriptor*)NULL;

    if( config_id == 0)
        return ECNR_ERR_CONFID_NOT_AVAILABLE;

    pr_debug("in_dev_list for config_id(%d)\n", config_id);

    for( i = 0, p = ECNR_FEATURE_CONF_table; i < FEATURE_CFG_MAX; p++, i++)
    {
        if(p->config_id == config_id)
        {

            /* no sync data available */
            if(!p->feat_cfg_table.is_sync_infos)
            {

                *ecnr_in_dev = inNodev;
                return 1;
            }

            ps = (struct sync_stream_descriptor*)&(p->feat_cfg_table.str_sync_desc);

            *ecnr_in_dev = ps->in_adev;

            in_dev_nb = (gint)( MASK_IN_DEV_NB & (guint8)(ps->prop_adev[0]));

            if(*isSyncDataMatching && pprop )
            {
                pr_message("set input device list matching with current tuning \n");

                nbdev = in_dev_nb;
                pr_message("nb of input devices(%d)\n", nbdev);

                pIndev = g_malloc0( (size_t)in_dev_nb * sizeof(gchar*));


                GString *devstr, *devlist, *str1, *str2;

                do
                {
                    pr_message("device string for device(%d)\n", nbdev);
                    guint8 is_dipo_select = (MASK_FEATURE_DIPO_STRING_SELECT & (guint8)(pprop[0] >> 16));

                    devlist = g_string_new (NULL);

                    if(is_dipo_select)
                        str1 = g_string_new("device=AdevSPIInOutExt");
                    else
                        str1 = g_string_new("AdevSPIInOutExt");

                    str2 = g_string_new(NULL);

                    /* calculated with the assumption the asr output SR is 24 khz */
                    guint buffer_size = pprop[3 + (nbdev - 1) * 5] * 24 * pprop[2 + (nbdev - 1) * 5]; //(periode_size_ms * (pdv->rate / 1000 )) * buffer_nb
                    g_string_printf (str2, "%s:%d", g_strdup(str1->str), buffer_size);
                    g_string_append(devlist, g_strdup(str2->str));


                    if(is_dipo_select)
                    {
                        pr_message("string dipo select feature is set (%d)\n", (guint8)(pprop[0] >> 16));

                        devstr = g_string_new (NULL);
                        g_string_printf (devstr, " %s=%d", "periodMs",pprop[3 + (nbdev - 1) * 5]);
                        g_string_append(devlist, g_strdup(devstr->str));
                        g_string_free(devstr, TRUE);

                        devstr = g_string_new (NULL);
                        g_string_printf (devstr, " %s=%d", "sampleRate",pprop[1 + (nbdev - 1) * 5]);
                        g_string_append(devlist, g_strdup(devstr->str));
                        g_string_free(devstr, TRUE);

                        devstr = g_string_new (NULL);
                        g_string_printf (devstr, " %s=%d", "bufferPeriods",pprop[2 + (nbdev - 1) * 5]);
                        g_string_append(devlist, g_strdup(devstr->str));
                        g_string_free(devstr, TRUE);

                        devstr = g_string_new (NULL);
                        g_string_printf (devstr, " %s=%d", "silenceMs",pprop[5 + (nbdev - 1) * 5]);
                        g_string_append(devlist, g_strdup(devstr->str));
                        g_string_free(devstr, TRUE);

                        devstr = g_string_new (NULL);
                        g_string_printf (devstr, " %s=%d", "inittoutms",pprop[4 + (nbdev - 1) * 5]);
                        g_string_append(devlist, g_strdup(devstr->str));

                        g_string_free(devstr, TRUE);
                    }

                    if(nbdev)
                    {
                        pIndev[nbdev - 1] = g_strdup(devlist->str);
                        pr_message("setup device string pIndev[%d](%s) for dev(%d)\n", (nbdev - 1), pIndev[nbdev - 1], nbdev);
                    }
                    g_string_free(devlist, TRUE);
                    g_string_free(str1, TRUE);
                    g_string_free(str2, TRUE);

                    if(nbdev)
                        nbdev--;

                }while(nbdev);

                *ecnr_in_dev =  pIndev;
            }

            pr_debug("nb of input devices(%d)\n", in_dev_nb);

            i = FEATURE_CFG_MAX;
        }
    }

    return (gint16)in_dev_nb;
}

static gint ecnr_configuration_get_out_dev_list(gchar*** ecnr_out_dev, unsigned short config_id, gboolean* isSyncDataMatching  )
{
    gint i = 0;
    gint out_dev_nb = 0;
    gint nbdev = 0;
    const struct ecnr_id_spec_feature *p = (struct ecnr_id_spec_feature*)NULL;
    struct sync_stream_descriptor *ps =  (struct sync_stream_descriptor*)NULL;

    if( config_id == 0)
        return ECNR_ERR_CONFID_NOT_AVAILABLE;

    pr_debug("out_dev_list for config_id(%d)\n", config_id);

    for( i = 0, p = ECNR_FEATURE_CONF_table; i < FEATURE_CFG_MAX; p++, i++)
    {
        if(p->config_id == config_id)
        {

            /* no sync data available */
            if(!p->feat_cfg_table.is_sync_infos)
            {

                *ecnr_out_dev = outNodev;
                return 1;
            }

            ps = (struct sync_stream_descriptor*)&(p->feat_cfg_table.str_sync_desc);

            *ecnr_out_dev = ps->out_adev;

            out_dev_nb =  (gint) ((MASK_OUT_DEV_NB & (guint8)(ps->prop_adev[0])) >> 4);//  MASK_OUT_DEV_NB & ((guint8)(ps->prop_adev[0]));

            if(*isSyncDataMatching && pprop )
            {
                pr_message("set output device list matching with current tuning \n");

                nbdev = out_dev_nb;

                pOutdev = g_malloc0( (size_t)nbdev * sizeof(gchar*));
                pr_message("nb of output devices(%d)\n", nbdev);

                GString *devstr, *devlist, *str1, *str2;

                do
                {
                    pr_message("device string for device(%d)\n", nbdev);

                    guint8 is_dipo_select = (MASK_FEATURE_DIPO_STRING_SELECT & (guint8)(pprop[0] >> 16));
                    devlist = g_string_new (NULL);

                    if(is_dipo_select)
                        str1 = g_string_new("device=AdevSPIInOutExt");
                    else
                        str1 = g_string_new("AdevSPIInOutExt");

                    str2 = g_string_new(NULL);

                    /* calculated with the assumption the asr output SR is 24 khz */
                    guint buffer_size = pprop[3 + (nbdev - 1) * 5] * 24 * pprop[2 + (nbdev - 1) * 5]; //(periode_size_ms * (pdv->rate / 1000 )) * buffer_nb
                    g_string_printf (str2, "%s:%d", g_strdup(str1->str), buffer_size);
                    g_string_append(devlist, g_strdup(str2->str));


                    if(is_dipo_select)
                    {
                        pr_message("string dipo select feature is set (%d)\n", (guint8)(pprop[0] >> 16));

                        devstr = g_string_new (NULL);
                        g_string_printf (devstr, " %s=%d", "periodMs",pprop[3 + (nbdev - 1) * 5]);
                        g_string_append(devlist, g_strdup(devstr->str));
                        g_string_free(devstr, TRUE);

                        devstr = g_string_new (NULL);
                        g_string_printf (devstr, " %s=%d", "sampleRate",pprop[1 + (nbdev - 1) * 5]);
                        g_string_append(devlist, g_strdup(devstr->str));
                        g_string_free(devstr, TRUE);

                        devstr = g_string_new (NULL);
                        g_string_printf (devstr, " %s=%d", "bufferPeriods",pprop[2 + (nbdev - 1) * 5]);
                        g_string_append(devlist, g_strdup(devstr->str));
                        g_string_free(devstr, TRUE);

                        devstr = g_string_new (NULL);
                        g_string_printf (devstr, " %s=%d", "silenceMs",pprop[5 + (nbdev - 1) * 5]);
                        g_string_append(devlist, g_strdup(devstr->str));
                        g_string_free(devstr, TRUE);

                        devstr = g_string_new (NULL);
                        g_string_printf (devstr, " %s=%d", "inittoutms",pprop[4 + (nbdev - 1) * 5]);
                        g_string_append(devlist, g_strdup(devstr->str));

                        g_string_free(devstr, TRUE);
                    }

                    if(nbdev)
                    {
                        pOutdev[nbdev - 1] = g_strdup(devlist->str);
                        pr_message("setup device string pOutdev[%d](%s) for dev(%d)\n", (nbdev - 1), pOutdev[nbdev - 1], nbdev);
                    }
                    g_string_free(devlist, TRUE);
                    g_string_free(str1, TRUE);
                    g_string_free(str2, TRUE);

                    if(nbdev)
                        nbdev--;

                }while(nbdev);

                *ecnr_out_dev =  pOutdev;
            }
            i = FEATURE_CFG_MAX;
        }
    }

    return out_dev_nb;
}

static gint ecnr_configuration_get_dev_properties(guint32**  ecnr_device_properties, unsigned short config_id, gboolean* isSyncDataMatching  )
{
    gint i = 0;
    gint dev_prop_nb = 0;
    gint out_dev_nb = 0;
    gint in_dev_nb = 0;
    const struct ecnr_id_spec_feature   *p  = (struct ecnr_id_spec_feature*)NULL;
    struct sync_stream_descriptor       *ps = (struct sync_stream_descriptor*)NULL;
    struct ecnr_feature_cfg             *pc = (struct ecnr_feature_cfg*)NULL;

    if( config_id == 0)
  {
    ETG_TRACE_ERR(("[ecnr_configuration_get_dev_properties]: ECNR_ERR_CONFID_NOT_AVAILABLE"));
        return 0;
  }

    pr_debug("property_dev_list for config_id(%d)\n", config_id);

    for( i = 0, p = ECNR_FEATURE_CONF_table; i < FEATURE_CFG_MAX; p++, i++)
    {
        if(p->config_id == config_id)
        {

#ifndef D_CREATE_ALWAYS_DYN_SYNC_DATA
            /* no sync data available */
            if(!p->feat_cfg_table.is_sync_infos)
            {

                *ecnr_device_properties = devprop;
                return 1;
            }
#endif

            ps = (struct sync_stream_descriptor*)&(p->feat_cfg_table.str_sync_desc);

            *ecnr_device_properties = ps->prop_adev;

            out_dev_nb = (gint) ((MASK_OUT_DEV_NB & (guint8)(ps->prop_adev[0])) >> 4);
            in_dev_nb  = (gint) (MASK_IN_DEV_NB & (guint8)(ps->prop_adev[0]));

            dev_prop_nb = 1 + (out_dev_nb + in_dev_nb) * 5 ;

            guint frame_size   = ecnr_alsa_get_frame_shift_in();
            guint sample_rate  = ecnr_alsa_get_sample_rate_in();
      guint periode_size = 0;

      if( sample_rate != 0) //To solve coverity issue 194572
      {
         periode_size = (1000 * frame_size) / sample_rate;
      }
      else
      {
        ETG_TRACE_ERR(("[ecnr_configuration_get_dev_properties]: ECNR_ERR_PERIODE_SIZE_INVALID, Invalid Sample rate"));
        return 0;
      }
            pr_debug("periode_size(%d) = (1000 * frame_size(%d)) / sample_rate(%d)\n", periode_size, frame_size, sample_rate);

#ifndef D_CREATE_ALWAYS_DYN_SYNC_DATA
            if( periode_size !=  ps->prop_adev[3])
            {
#endif
                pr_message("set device properties matching with current tuning \n");

                pc = (struct ecnr_feature_cfg*)&p->feat_cfg_table;

                gint nbdev = 0;

                nbdev = dev_prop_nb;
                nbdev -= 1;
                nbdev /= 5;

                if(nbdev)
                {
                    pprop = g_malloc0( (size_t)dev_prop_nb * sizeof(guint32));

                    /* device properties info */
                    pprop[0] = ps->prop_adev[0];

                    do
                    {
                        /*Device n: period time (ms)*/
                        pprop[ 3 + (nbdev - 1) * 5 ] = periode_size;

                        /*Device n: sample rate (Hz))*/
                        pprop[ 1 + (nbdev - 1) * 5 ] = ps->prop_adev[1 + (nbdev - 1) * 5 ]; // sample_rate;

                        if(pc->is_preFill)
                        {
                            if(pc->prefill_value > periode_size)
                            {
                                /*Device n: nb period buffer*/
                                pprop[ 2 + (nbdev - 1) * 5 ] = 3;
                                /*Device n: silence (ms)*/
                                pprop[ 5 + (nbdev - 1) * 5 ] = 0;
                            }
                            else
                            {
                                /*Device n: nb period buffer*/
                                pprop[ 2 + (nbdev - 1) * 5 ] = 2;
                                /*Device n: silence (ms)*/
                                pprop[ 5 + (nbdev - 1) * 5 ] = pc->prefill_value;
                            }
                        }

                        /*Device n: timeout (ms)*/
                        pprop[ 4 + (nbdev - 1) * 5 ] = ps->prop_adev[4 + (nbdev - 1) * 5 ];

                        nbdev--;
                    }while(nbdev);

                    *isSyncDataMatching = TRUE;

                    *ecnr_device_properties = pprop;
                }
#ifndef D_CREATE_ALWAYS_DYN_SYNC_DATA
            }
#endif


            pr_debug("nb of poperties(%d)\n", dev_prop_nb);

            i = FEATURE_CFG_MAX;
        }
    }

    return dev_prop_nb;
}


/*******************************************************************************
*
* FUNCTION: ecnr_configuration_sync_data_clear
* DESCRIPTION: retrive the input alsa device list
*
*
* PARAMETER: ecnr AppID and ConfigID
*
* RETURNVALUE: pointer to the array of character pointers
*
*******************************************************************************/
void ecnr_configuration_sync_data_clear(void)
{
    gint out_dev_nb, in_dev_nb, i;

    pr_message("enter\n");

    if(pprop)
    {
        if(pprop[0])
        {
            out_dev_nb =  (gint) ((MASK_OUT_DEV_NB & (guint8)(pprop[0])) >> 4); //MASK_OUT_DEV_NB && ((guint8)(pprop[0]));
            in_dev_nb =   (gint) ( MASK_IN_DEV_NB & ((guint8)(pprop[0])));

            for(i = 0; i < in_dev_nb; i++)
            {
                if(pIndev[i])
                {
                    g_free(pIndev[i]);
                    pIndev[i] = NULL;
                }
            }

            g_free(pIndev);
            pIndev = NULL;

            for(i = 0; i < out_dev_nb; i++)
            {
                if(pOutdev[i])
                {
                    g_free(pOutdev[i]);
                    pOutdev[i] = NULL;
                }
            }

            g_free(pOutdev);
            pOutdev = NULL;


            g_free(pprop);
            pprop = NULL;
        }
    }

    return;
}


/*******************************************************************************
*
* FUNCTION: ecnr_configuration_get_in_devices
* DESCRIPTION: retrive the input alsa device list
*
*
* PARAMETER: ecnr AppID and ConfigID
*
* RETURNVALUE: pointer to the array of character pointers
*
*******************************************************************************/
gchar* ecnr_configuration_get_in_devices(guchar ecnrAppId, gint ecnrConfigurationId, gboolean* isSyncDataMatching )
{
    gint    nInDev   = 0;
    gchar**    ecnr_in_dev  = NULL;
    gchar*    ret = NULL;
    gint     i;

    pr_debug("ENTERED with appid(%d) and cfgid (%d)\n", ecnrAppId, ecnrConfigurationId);


    nInDev   = ecnr_configuration_get_in_dev_list(&ecnr_in_dev, ecnr_configuration_get_ecnr_config_Id((unsigned short)ecnrConfigurationId), isSyncDataMatching);

    /* transfer input device's list */
    if(nInDev && ecnr_in_dev)
    {
        GStrv valInDev = g_new(gchar *, (nInDev+1));
        for (i = 0; i < nInDev; i++)
        {
            valInDev[i] = g_strdup((gchar *) ecnr_in_dev[i]);
            pr_debug("valInDev[%d] -> (%s)",i,valInDev[i]);
        }
        valInDev[nInDev] = NULL;
        ret = (gchar *)valInDev;
    }
    else
    {
        pr_debug("one of both condition is not satisfied nInDev(%d) > 0 && ecnr_in_dev(%p) != NULL", nInDev, ecnr_in_dev);
    }

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_configuration_get_out_devices
* DESCRIPTION: retrieve the output alsa device list
*
*
* PARAMETER: ecnr AppID and ConfigID
*
* RETURNVALUE: pointer to the array of character pointers
*
*******************************************************************************/
gchar* ecnr_configuration_get_out_devices(guchar ecnrAppId, gint ecnrConfigurationId, gboolean* isSyncDataMatching )
{
    gint    nOutDev   = 0;
    gchar**    ecnr_out_dev  = NULL;
    gchar*    ret = NULL;
    gint     i;

    pr_debug("ENTERED with appid(%d) and cfgid (%d)\n", ecnrAppId, ecnrConfigurationId);

    nOutDev  = ecnr_configuration_get_out_dev_list(&ecnr_out_dev, ecnr_configuration_get_ecnr_config_Id((unsigned short)ecnrConfigurationId), isSyncDataMatching);

    if(nOutDev && ecnr_out_dev)
    {
        GStrv valOutDev = g_new(gchar *, (nOutDev+1));
        for (i = 0; i < nOutDev; i++)
        {
            valOutDev[i] = g_strdup((gchar *) ecnr_out_dev[i]);
            pr_debug("valOutDev[%d] -> (%s)",i,valOutDev[i]);
        }
        valOutDev[nOutDev] = NULL;
        ret = (gchar *)valOutDev ;
    }
    else
    {
        pr_debug("one of both condition is not satisfied nOutDev(%d) > 0 && ecnr_out_dev(%p) != NULL", nOutDev, ecnr_out_dev);
    }

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_configuration_get_device_properties
* DESCRIPTION: retrive the alsa device properties list
*
*
* PARAMETER: ecnr AppID and ConfigID
*
* RETURNVALUE: pointer to the array of integer pointers
*
*******************************************************************************/
GArray* ecnr_configuration_get_device_properties(guchar ecnrAppId, gint ecnrConfigurationId, gboolean* isSyncDataMatching )
{
    gint      nDevProp = 0;
    guint32*    ecnr_device_properties = NULL;
    GArray*    ret = NULL;
    guint32     val;
    gint        i=0;

    pr_debug("ENTERED with appid(%d) and cfgid (%d)\n", ecnrAppId, ecnrConfigurationId);

    nDevProp = ecnr_configuration_get_dev_properties(&ecnr_device_properties, ecnr_configuration_get_ecnr_config_Id((unsigned short)ecnrConfigurationId), isSyncDataMatching);

    if (nDevProp && ecnr_device_properties)
    {
        GArray* valDevProp = g_array_sized_new(FALSE, FALSE, sizeof(guint32), (guint)nDevProp);
        for (i = 0; i < nDevProp; i++)
        {
            val = ecnr_device_properties[i];
            g_array_append_val(valDevProp, val);

            pr_debug("Device_properties[%i] = %d", i, val);
        }
        ret = valDevProp;
    }
    else
    {
        pr_debug("one of both condition is not satisfied nDevProp(%d) > 0 && ecnr_device_properties(%p) != NULL", nDevProp, ecnr_device_properties);
    }

    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_configuration_get_dp
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
enum eDataPool ecnr_configuration_get_dp(unsigned short configId)
{
    unsigned short i = 0;
    enum eDataPool ret = ECNR_DP_NOT_DEF;
    const struct EcnrConfig *pdata;

    pr_debug("ENTERED");

    for (i = 0, pdata = ECNR_CONFIG_table;  i < (int)ECNR_NUM_CONFIG; i++, pdata++)
    {
        if (pdata->configId == configId)
        {
            ret = pdata->EcnrdPBaseKey;
        }
    }

    pr_debug("dp: %d, ecnr CfgId: %d\n", configId, ret);

    return ret;
}

#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX  //for lint fix /* lint warning 528 */
static int ecnr_configuration_remove_datapool_file(const char* str)
{
    int    ret = ECNR_ERR_OK;        /* internal error status */
    const char  *FileName = str;

    pr_debug("ENTERED\n");

    if (remove(FileName) != 0)
    {
        pr_warning("error deleting file (%s)\n", FileName);
        ret = ECNR_ERR_NO_FILE_AVAILABLE;
    }
    else
        pr_debug("file (%s) successfully deleted\n", FileName);

    return ret;
}


static void ecnr_configuration_new_dp_release(enum eDataPool dpBaseKey)
{
    pr_debug("release notification for dp element %d\n", dpBaseKey);

    switch (dpBaseKey)
    {
        case ECNR_DP_BT_PHONE_NB:
            bNewdpBTPhoneNB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_BT_PHONE_NB);
            break;
        case ECNR_DP_BT_PHONE_WB:
            bNewdpBTPhoneWB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_BT_PHONE_WB);
            break;
        case ECNR_DP_BT_VOICE_NB:
            bNewdpBTVoiceNB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_BT_VOICE_NB);
            break;
        case ECNR_DP_BT_VOICE_WB:
            bNewdpBTVoiceWB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_BT_VOICE_WB);
            break;
        case ECNR_DP_BT_SIRI_NB:
            bNewdpBTSiriNB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_BT_SIRI_NB);
            break;
        case ECNR_DP_BT_SIRI_WB:
            bNewdpBTSiriWB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_BT_SIRI_WB);
            break;
        case ECNR_DP_CP_VOICE_NB:
            bNewdpCPVoiceNB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_CP_VOICE_NB);
            break;
        case ECNR_DP_CP_VOICE_WB:
            bNewdpCPVoiceWB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_CP_VOICE_WB);
            break;
        case ECNR_DP_CP_VOICE_SWB:
            bNewdpCPVoiceSWB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_CP_VOICE_SWB);
            break;
        case ECNR_DP_CP_PHONE_NB:
            bNewdpCPPhoneNB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_CP_PHONE_NB);
            break;
        case ECNR_DP_CP_PHONE_WB:
            bNewdpCPPhoneWB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_CP_PHONE_WB);
            break;
        case ECNR_DP_CP_PHONE_SWB:
            bNewdpCPPhoneSWB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_CP_PHONE_SWB);
            break;
        case ECNR_DP_CP_FTIME_SWB:
            bNewdpCPFTimeSWB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_CP_FTIME_SWB);
            break;
        case ECNR_DP_AA_VOICE_NB:
            bNewdpAAVoiceNB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_AA_VOICE_NB);
            break;
        case ECNR_DP_AA_VOICE_WB:
            bNewdpAAVoiceWB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_AA_VOICE_WB);
            break;
        case ECNR_DP_AA_VOICE_SWB:
            bNewdpAAVoiceSWB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_AA_VOICE_SWB);
            break;
        case ECNR_DP_AA_PHONE_NB:
            bNewdpAAPhoneNB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_AA_PHONE_NB);
            break;
        case ECNR_DP_AA_PHONE_WB:
            bNewdpAAPhoneWB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_AA_PHONE_WB);
            break;
        case ECNR_DP_AA_PHONE_SWB:
            bNewdpAAPhoneSWB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_AA_PHONE_SWB);
            break;
        case ECNR_DP_INT_VOICE_NB:
            bNewdpINTVoiceNB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_INT_VOICE_NB);
            break;
        case ECNR_DP_INT_VOICE_WB:
            bNewdpINTVoiceWB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_INT_VOICE_WB);
            break;
        case ECNR_DP_WUW_VOICE_WB:
            bNewdpWUWVoiceWB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_WUW_VOICE_WB);
            break;
        case ECNR_DP_VPA_VOICE_WB:
            bNewdpVPAVoiceWB = FALSE;
            ecnr_configuration_remove_datapool_file(NEW_DP_FILE_VPA_VOICE_WB);
            break;
        default:
            break;
    }
}

/*******************************************************************************
*
* FUNCTION: ecnr_configuration_set_crc
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static gboolean ecnr_configuration_set_crc(enum eDataPool dpid, guint16 crc)
{
    gboolean ret = FALSE;
    int err = ECNR_ERR_OK;
    const  char* str = NULL;
    FILE *crc_file;
    size_t  wdata = 0;

    pr_debug("ENTERED\n");

    switch(dpid)
    {
        case ECNR_DP_BT_PHONE_NB:
            str = CRC_DP_FILE_BT_PHONE_NB;
            break;
        case ECNR_DP_BT_PHONE_WB:
            str = CRC_DP_FILE_BT_PHONE_WB;
            break;
        case ECNR_DP_BT_VOICE_NB:
            str = CRC_DP_FILE_BT_VOICE_NB;
            break;
        case ECNR_DP_BT_VOICE_WB:
            str = CRC_DP_FILE_BT_VOICE_WB;
            break;
        case ECNR_DP_BT_SIRI_NB:
            str = CRC_DP_FILE_BT_SIRI_NB;
            break;
        case ECNR_DP_BT_SIRI_WB:
            str = CRC_DP_FILE_BT_SIRI_WB;
            break;
        case ECNR_DP_CP_VOICE_NB:
            str = CRC_DP_FILE_CP_VOICE_NB;
            break;
        case ECNR_DP_CP_VOICE_WB:
            str = CRC_DP_FILE_CP_VOICE_WB;
            break;
        case ECNR_DP_CP_VOICE_SWB:
            str = CRC_DP_FILE_CP_VOICE_SWB;
            break;
        case ECNR_DP_CP_PHONE_NB:
            str = CRC_DP_FILE_CP_PHONE_NB;
            break;
        case ECNR_DP_CP_PHONE_WB:
            str = CRC_DP_FILE_CP_PHONE_WB;
            break;
        case ECNR_DP_CP_PHONE_SWB:
            str = CRC_DP_FILE_CP_PHONE_SWB;
            break;
        case ECNR_DP_CP_FTIME_SWB:
            str = CRC_DP_FILE_CP_FTIME_SWB;
            break;
        case ECNR_DP_AA_VOICE_NB:
            str = CRC_DP_FILE_AA_VOICE_NB;
            break;
        case ECNR_DP_AA_VOICE_WB:
            str = CRC_DP_FILE_AA_VOICE_WB;
            break;
        case ECNR_DP_AA_VOICE_SWB:
            str = CRC_DP_FILE_AA_VOICE_SWB;
            break;
        case ECNR_DP_AA_PHONE_NB:
            str = CRC_DP_FILE_AA_PHONE_NB;
            break;
        case ECNR_DP_AA_PHONE_WB:
            str = CRC_DP_FILE_AA_PHONE_WB;
            break;
        case ECNR_DP_AA_PHONE_SWB:
            str = CRC_DP_FILE_AA_PHONE_SWB;
            break;
        case ECNR_DP_INT_VOICE_NB:
            str = CRC_DP_FILE_INT_VOICE_NB;
            break;
        case ECNR_DP_INT_VOICE_WB:
            str = CRC_DP_FILE_INT_VOICE_WB;
            break;
        case ECNR_DP_WUW_VOICE_WB:
            str = CRC_DP_FILE_WUW_VOICE_WB;
            break;
        case ECNR_DP_VPA_VOICE_WB:
            str = CRC_DP_FILE_VPA_VOICE_WB;
            break;
        default:
            break;
    }

    if(!str)
    {
        pr_debug("an unknown datapool Id(%d) was passed\n", dpid);
        return FALSE;
    }

    pr_debug("set crc(0x%x) from datapool(%d) in crc file(%s)\n", crc, dpid, str);

    /* first remove crc file if exist*/
    pr_debug("remove crc file(%s)\n", str);

    err = remove(str);
    if(err)
    {
        pr_warning(" no crc file in rfs :  %s\n", str);
    }

    /* create crc file */
    crc_file = fopen (str,"a+");

    if(crc_file)
    {
        wdata = fwrite(&crc , sizeof(char), 2 , crc_file);
        pr_debug("crc(0x%x) with size(%lu) was written in file(%s)\n",crc,wdata,str);
        fclose(crc_file);
        ret = TRUE;
    }
    else
        pr_debug("no file(%s) available\n", str);
      return ret;
}
#endif  //for lint fix /* lint warning 528 */

static int ecnr_configuration_create_bsd_files_from_dp(const struct EcnrConfig* data)
{
    int     err = ECNR_ERR_OK;
#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    guint16 dataset_size = 0;
    unsigned char*   dpdata = (unsigned char*)NULL;
    FILE*   fd_bsd;   /* pointer to config data file */
    size_t  wdata = 0;
    int     version[4]={0};
    guint16 size = 0;
    int     pos= 0;
    guint16 crc = 0;
#endif
    pr_debug("ENTERED ");
    ETG_TRACE_USR3(("[ecnr_configuration_create_bsd_files_from_dp]: ENTERED"));

    if(!data)
        return ECNR_ERR_NULL_POINTER;


    /* read dp element */

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    /* warning lint:Info 774: prio1, change code dpdata = (char*)NULL; */
    pr_debug("no datapool available\n");

    err = ECNR_ERR_DP_NO_ELEMENT_AVAILABLE;

    pr_debug("EXIT\n");
    return err;
#else
    dpdata = ecnr_datapool_access_get_bsd_dataset(data->EcnrdPBaseKey, &dataset_size, version, &crc);


    if(!dpdata)
    {
        pr_debug("no datapool available\n");
        ETG_TRACE_USR3(("[ecnr_configuration_create_bsd_files_from_dp]:no datapool available"));
        return ECNR_ERR_DP_NO_ELEMENT_AVAILABLE;
    }
    else
    {
        /* compare sse version in dataset from datapool with the one of the current sse version  */
        if(FALSE == ecnr_sse_handler_compare_version(version))
        {
            pr_critical("the dp data set coded version do not match with current engine version\n");
            ETG_TRACE_USR3(("[ecnr_configuration_create_bsd_files_from_dp]:the dp data set coded version do not match with current engine version"));
            return ECNR_ERR_VERSION_SSE_DP_DO_NOT_MATCH;
        }

        /* create a bsd file with dp element */
        fd_bsd = fopen (data->bsd_str,"a+");

        /* retrieve start position data set in datapool */
        pos = ecnr_datapool_access_get_config_row_data_start_pos(dpdata, &size);

        if (NULL != fd_bsd)
        {
            if(dataset_size > 0 && dataset_size < D_ECNR_USE_MAX_DP_SIZE)
               wdata = fwrite(&dpdata[pos], (size_t)1, dataset_size, fd_bsd);
            if(wdata == 0)
            {
               pr_debug("fails to create a bsd file -> %s with dp element %d of size -> %lu\n", data->bsd_str, data->EcnrdPBaseKey, wdata);
               ETG_TRACE_USR3(("[ecnr_configuration_create_bsd_files_from_dp]: with dp element %d of size -> %d, the creation of a bsd file -> %s has failed ", data->EcnrdPBaseKey, wdata ,data->bsd_str));
            }
            else
            {
                pr_debug("create bsd file %s with dp element %d from size : %lu\n",data->bsd_str, data->EcnrdPBaseKey, wdata);
                ETG_TRACE_USR3(("[ecnr_configuration_create_bsd_files_from_dp]: dp element -> %d with size -> %d used to create a bsd file %s ", data->EcnrdPBaseKey, wdata ,data->bsd_str));
            }
            fclose(fd_bsd);
        }
    }

    /* dp was updated, release dp notification */
    ecnr_configuration_new_dp_release(data->EcnrdPBaseKey);

    /* dp new, save crc */
    ecnr_configuration_set_crc(data->EcnrdPBaseKey, crc);
#endif
    pr_debug("EXIT\n");

    return err;
}

static int ecnr_configuration_create_bsd_files_from_data(const struct EcnrConfig* data)
{
    int     err = ECNR_ERR_OK;
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    pr_debug("data pool not supported\n");
    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)data;
#else
    struct tcconfigData* pd = (struct tcconfigData*)NULL;
    FILE*   fd_bsd;   /* pointer to config data file */
    size_t  wdata;
    guint16 size = 0;
    int pos= 0;

    pr_debug("ENTERED ");
    ETG_TRACE_USR3(("[ecnr_configuration_create_bsd_files_from_data]: ENTERED"));

    /* read dp element */

    if(!data)
        return ECNR_ERR_NULL_POINTER;

    pd = (struct tcconfigData*)&data->configData;

    /* create a bsd file with dp element */
    if(!data->bsd_str)
        return ECNR_ERR_NULL_POINTER;

    fd_bsd = fopen (data->bsd_str,"a+");

#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    pos = ecnr_datapool_access_get_config_row_data_start_pos(pd->data, &size);
#endif

    if (NULL != fd_bsd)
    {
        wdata = fwrite(&(pd->data[pos]), (size_t)1, size, fd_bsd);//pd->size, fd_bsd);
        pr_debug("create bsd file %s with data table from size : %lu\n",data->bsd_str,  wdata);
        ETG_TRACE_USR3(("[ecnr_configuration_create_bsd_files_from_dp]: %d bytes writen in the file -> %s ", wdata ,data->bsd_str));

        fclose(fd_bsd);
    }
#endif
    return err;
}

/*static*/ int ecnr_configuration_remove_bsd_file(const char* str)
{
    int err = ECNR_ERR_OK;

    /* File name is empty string or NULL pointer means we do not have a
     * config file (not really an error)
     */
    if((NULL != str) && ('\0' != str[0]))
    {

        pr_debug("remove bsd file: %s\n",str);
        ETG_TRACE_USR3(("[ecnr_configuration_remove_bsd_file]: remove bsd file bsd_file -> %s", str));

        err = remove(str);

        if(err)
        {
            pr_warning(" no bsd file found ->  %s\n", str);
            ETG_TRACE_USR3(("[ecnr_configuration_remove_bsd_file]: no bsd file found -> %s", str));
            err =  ECNR_ERR_OK; //ECNR_ERR_NO_CONFIG_FILE_AVAILABLE;

        }
    }
    else
    {
        pr_warning("no bsd file configuration table\n");
        err =  ECNR_ERR_NO_CONFIG_FILE_AVAILABLE ;
    }

    pr_debug("EXIT\n");

    return err;
}

/*static*/ int ecnr_configuration_remove_crc_log_file(const char* str)
{
    int err = ECNR_ERR_OK;

    /* File name is empty string or NULL pointer means we do not have a
     * config file (not really an error)
     */
    if((NULL != str) && ('\0' != str[0]))
    {

        pr_debug("remove bsd file: %s\n",str);
        ETG_TRACE_USR3(("[ecnr_configuration_remove_bsd_file]: remove bsd file bsd_file -> %s", str));

        err = remove(str);

        if(err)
        {
            pr_warning(" no bsd file found ->  %s\n", str);
            ETG_TRACE_USR3(("[ecnr_configuration_remove_bsd_file]: no bsd file found -> %s", str));
            err =  ECNR_ERR_OK; //ECNR_ERR_NO_CONFIG_FILE_AVAILABLE;

        }
    }
    else
    {
        pr_warning("no bsd file configuration table\n");
        err =  ECNR_ERR_NO_CONFIG_FILE_AVAILABLE ;
    }

    pr_debug("EXIT\n");

    return err;
}

static gboolean ecnr_configuration_is_new_datapool_file(const char* str)
{
    gboolean ret = FALSE;
    const char       *FileName = str;
    FILE*   fd;              /* pointer to config data file */


    pr_debug("ENTERED\n");

    /* create a file to indicate new datapool are available */
    fd = fopen (FileName,"r");

    if (NULL != fd)
    {
        fclose(fd);
        pr_debug("datapool file(%s) available\n", FileName);
        ret = TRUE;
    }

    pr_message("%s\n", ret? "new datapool file was found": "no new data pool file ");

    return ret;
}

#ifdef D_DP_NOTIFICATION_SUPPORT
static gboolean ecnr_configuration_create_new_datapool_file(const char* str)
{
    gboolean ret = FALSE;
    const char       *FileName = str;
    FILE*   fd;              /* pointer to config data file */


    pr_debug("ENTERED\n");

    /* create a file to indicate new datapool are available */
    fd = fopen (FileName,"a+");

    if (NULL != fd)
    {
        fclose(fd);
        pr_debug("create datapool file %s\n",FileName);
        ret = TRUE;
    }

    return ret;
}
#endif




static gboolean ecnr_configuration_is_new_dp_available(enum eDataPool dpBaseKey)
{
    pr_debug("ENTERED\n");

    gboolean bnewdp = FALSE;

    switch (dpBaseKey)
    {
        case ECNR_DP_BT_PHONE_NB:
            bnewdp = bNewdpBTPhoneNB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_BT_PHONE_NB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_BT_PHONE_WB:
            bnewdp = bNewdpBTPhoneWB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_BT_PHONE_WB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_BT_VOICE_NB:
            bnewdp = bNewdpBTVoiceNB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_BT_VOICE_NB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_BT_VOICE_WB:
            bnewdp = bNewdpBTVoiceWB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_BT_VOICE_WB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_BT_SIRI_NB:
            bnewdp = bNewdpBTSiriNB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_BT_SIRI_NB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_BT_SIRI_WB:
            bnewdp = bNewdpBTSiriWB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_BT_SIRI_WB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_CP_VOICE_NB:
            bnewdp = bNewdpCPVoiceNB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_CP_VOICE_NB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_CP_VOICE_WB:
            bnewdp = bNewdpCPVoiceWB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_CP_VOICE_WB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_CP_VOICE_SWB:
            bnewdp = bNewdpCPVoiceSWB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_CP_VOICE_SWB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_CP_PHONE_NB:
            bnewdp = bNewdpCPPhoneNB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_CP_PHONE_NB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_CP_PHONE_WB:
            bnewdp = bNewdpCPPhoneWB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_CP_PHONE_WB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_CP_PHONE_SWB:
            bnewdp = bNewdpCPPhoneSWB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_CP_PHONE_SWB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_CP_FTIME_SWB:
            bnewdp = bNewdpCPFTimeSWB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_CP_FTIME_SWB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_AA_VOICE_NB:
            bnewdp = bNewdpAAVoiceNB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_AA_VOICE_NB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_AA_VOICE_WB:
            bnewdp = bNewdpAAVoiceWB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_AA_VOICE_WB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_AA_VOICE_SWB:
            bnewdp = bNewdpCPVoiceSWB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_AA_VOICE_SWB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_AA_PHONE_NB:
            bnewdp = bNewdpAAPhoneNB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_AA_PHONE_NB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_AA_PHONE_WB:
            bnewdp = bNewdpAAPhoneWB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_AA_PHONE_WB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_AA_PHONE_SWB:
            bnewdp = bNewdpAAPhoneSWB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_AA_PHONE_SWB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_INT_VOICE_NB:
            bnewdp = bNewdpINTVoiceNB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_INT_VOICE_NB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_INT_VOICE_WB:
            bnewdp = bNewdpINTVoiceWB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_INT_VOICE_WB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_WUW_VOICE_WB:
            bnewdp = bNewdpINTVoiceWB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_WUW_VOICE_WB))
                bnewdp = TRUE;
            break;
        case ECNR_DP_VPA_VOICE_WB:
            bnewdp = bNewdpVPAVoiceWB;
            if(!bnewdp && ecnr_configuration_is_new_datapool_file(NEW_DP_FILE_VPA_VOICE_WB))
                bnewdp = TRUE;
            break;
        default:
            break;
    }

    pr_message("dp_%d %s\n", dpBaseKey, (bnewdp? "new data pool available": "remains unchanged"));

    return bnewdp;
}


int ecnr_configuration_read_default_configuration(const struct EcnrConfig* pdata, tecnrConfigData *pConfigData)
{
    int     err = ECNR_ERR_OK;
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    pr_debug("not supported\n");
    //to remove prio2 lint warning (Info 715: Symbol not referenced)
       (void*)pdata;
       (void*)pConfigData;
#else

    struct tcconfigData* pd = (struct tcconfigData*)NULL;
    guint16 size = 0;
    int     pos= 0;

    pr_debug("ENTERED\n");

    if(!pConfigData)
    {
        pr_debug("invalid pointer pConfigData \n");
        return ECNR_ERR_NULL_POINTER;
    }

    if(!pdata)
    {
        pr_debug("invalid default dataset, invalid pointer \"pdata\" \n");
        return ECNR_ERR_NO_DEFAULT_CONFIGURATION;
    }

    pd = (struct tcconfigData*)&pdata->configData;

    /* retrieve SSE data set */
    if(pd)
    {
        pos = ecnr_datapool_access_get_config_row_data_start_pos(pd->data,&size);// dpdata, &size);

        if(!size)
        {
            pr_debug("invalid default dataset with size->%d, data_ptr->%s\n",size, pd->data);
            return ECNR_ERR_NO_DEFAULT_CONFIGURATION;
        }
    }
    else
    {
        pr_debug("invalid default dataset, invalid pointer \"pdata->configData\" \n");
        return ECNR_ERR_NO_DEFAULT_CONFIGURATION;
    }

#ifdef D_ECNR_LIB_SSE_3_17

    char*   pcb = g_memdup(&pd->data[pos], size);

    if(pcb == NULL)
    {
        pr_debug("invalid returned memory space\n");
        return ECNR_ERR_ALLOC;
    }
    else
    {
        pConfigData->ConfigBuffer[pConfigData->ConfigBufferCnt] = pcb;
        pConfigData->ConfigBufferSize[pConfigData->ConfigBufferCnt] = size;
        if((pConfigData->ConfigBufferCnt + 1) < ECNR_MAX_CONFIG_DATA_CNT)
            pConfigData->ConfigBufferCnt++;
        else
        {
            pr_debug("the max count of bsd buffer was reached, current bsd buffer count is-> %d\n", pConfigData->ConfigBufferCnt );
        }

        pr_debug("add new bsd buffer of size -> %d, current bsd buffer count is-> %d\n", size, pConfigData->ConfigBufferCnt );
    }

#elif  D_ECNR_LIB_SSE_4_03

    int32_t* i32pcb = g_memdup(&pd->data[pos], size);

    if(i32pcb == NULL)
    {
        pr_debug("invalid returned memory space\n");
        return ECNR_ERR_ALLOC;
    }
    else
    {
        pConfigData->i32ConfigBuffer[pConfigData->ConfigBufferCnt] = i32pcb;
        pConfigData->ConfigBufferSize[pConfigData->ConfigBufferCnt] = size;
        if((pConfigData->ConfigBufferCnt + 1) < ECNR_MAX_CONFIG_DATA_CNT)
            pConfigData->ConfigBufferCnt++;
        else
        {
            pr_debug("the max count of bsd buffer was reached, current bsd buffer count is-> %d\n", pConfigData->ConfigBufferCnt );
        }

        pr_debug("add new bsd buffer of size -> %d, current bsd buffer count is-> %d\n", size, pConfigData->ConfigBufferCnt );
    }

#endif



#endif

    pr_debug("EXIT\n");
    return err;
}


#define LINE_SIZE_MAX 20

/*******************************************************************************
*
* FUNCTION: ecnr_configuration_get_crc
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
gboolean ecnr_configuration_get_crc(enum eDataPool dpid, guint16* crc)
{
    gboolean ret = FALSE;
    const  char* str = NULL;
    FILE *crc_file;
    char data[LINE_SIZE_MAX];    /* Record read from the file.    */
    guint16 crc_in = 0;
    size_t crc_size = 0;

    switch(dpid)
    {
        case ECNR_DP_BT_PHONE_NB:
            str = CRC_DP_FILE_BT_PHONE_NB;
            break;
        case ECNR_DP_BT_PHONE_WB:
            str = CRC_DP_FILE_BT_PHONE_WB;
            break;
        case ECNR_DP_BT_VOICE_NB:
            str = CRC_DP_FILE_BT_VOICE_NB;
            break;
        case ECNR_DP_BT_VOICE_WB:
            str = CRC_DP_FILE_BT_VOICE_WB;
            break;
        case ECNR_DP_BT_SIRI_NB:
            str = CRC_DP_FILE_BT_SIRI_NB;
            break;
        case ECNR_DP_BT_SIRI_WB:
            str = CRC_DP_FILE_BT_SIRI_WB;
            break;
        case ECNR_DP_CP_VOICE_NB:
            str = CRC_DP_FILE_CP_VOICE_NB;
            break;
        case ECNR_DP_CP_VOICE_WB:
            str = CRC_DP_FILE_CP_VOICE_WB;
            break;
        case ECNR_DP_CP_VOICE_SWB:
            str = CRC_DP_FILE_CP_VOICE_SWB;
            break;
        case ECNR_DP_CP_PHONE_NB:
            str = CRC_DP_FILE_CP_PHONE_NB;
            break;
        case ECNR_DP_CP_PHONE_WB:
            str = CRC_DP_FILE_CP_PHONE_WB;
            break;
        case ECNR_DP_CP_PHONE_SWB:
            str = CRC_DP_FILE_CP_PHONE_SWB;
            break;
        case ECNR_DP_CP_FTIME_SWB:
            str = CRC_DP_FILE_CP_FTIME_SWB;
            break;
        case ECNR_DP_AA_VOICE_NB:
            str = CRC_DP_FILE_AA_VOICE_NB;
            break;
        case ECNR_DP_AA_VOICE_WB:
            str = CRC_DP_FILE_AA_VOICE_WB;
            break;
        case ECNR_DP_AA_VOICE_SWB:
            str = CRC_DP_FILE_AA_VOICE_SWB;
            break;
        case ECNR_DP_AA_PHONE_NB:
            str = CRC_DP_FILE_AA_PHONE_NB;
            break;
        case ECNR_DP_AA_PHONE_WB:
            str = CRC_DP_FILE_AA_PHONE_WB;
            break;
        case ECNR_DP_AA_PHONE_SWB:
            str = CRC_DP_FILE_AA_PHONE_SWB;
            break;
        case ECNR_DP_INT_VOICE_NB:
            str = CRC_DP_FILE_INT_VOICE_NB;
            break;
        case ECNR_DP_INT_VOICE_WB:
            str = CRC_DP_FILE_INT_VOICE_WB;
            break;
        case ECNR_DP_WUW_VOICE_WB:
            str = CRC_DP_FILE_WUW_VOICE_WB;
            break;
        case ECNR_DP_VPA_VOICE_WB:
            str = CRC_DP_FILE_VPA_VOICE_WB;
            break;
        default:
        break;
    }

    if(!str)
    {
        pr_debug("an unknown dpid(%d) was passed\n",dpid);
        return FALSE;
    }

    crc_file = fopen(str, "r");

    if(crc_file)
    {
        crc_size = fread(data, 1, 2, crc_file);
        (void*)memcpy((void*)&crc_in, (void*)data, 2);

        pr_debug("read from crc file(%s), crc_size(%lu) bytes, byte[0](%d), byte[1](%d)\n", str, crc_size, data[0], data[1]);
        pr_debug("crc(0x%x) was readen from file(%s)\n", crc_in, str);

        *crc = crc_in;
        fclose(crc_file);
        ret = TRUE;
    }
    else
        pr_debug("no file(%s) available\n", str);

    return ret;
}


int ecnr_configuration_remove_crc_file(const char* str)
{
    int err = ECNR_ERR_OK;

    /* File name is empty string or NULL pointer means we do not have a
     * config file (not really an error)
     */
    if((NULL != str) && ('\0' != str[0]))
    {

        pr_debug("remove bsd file: %s\n",str);
        ETG_TRACE_USR3(("[ecnr_configuration_remove_crc_file]: remove crc file -> %s", str));

        err = remove(str);

        if(err)
        {
            pr_warning(" no crc file found ->  %s\n", str);
            ETG_TRACE_USR3(("[ecnr_configuration_remove_crc_file]: no crc file found -> %s", str));
            err =  ECNR_ERR_OK;
        }
    }
    else
    {
        pr_warning("no crc file available\n");
        err =  ECNR_ERR_NO_CONFIG_FILE_AVAILABLE ;
    }

    pr_debug("EXIT\n");

    return err;
}



/*******************************************************************************
*
* FUNCTION: ecnr_configuration_compare_crc_dp_file
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static gboolean ecnr_configuration_compare_crc_dp_file(enum eDataPool dpid)
{
    gboolean    ret = FALSE;
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    pr_debug("datapool not support\n");
    //to remove prio2 lint warning (Info 715: Symbol not referenced)
            (void*)dpid;
#else
    guint16     crc = 0;

    pr_debug("ENTERED\n");

    if(ecnr_configuration_get_crc(dpid, &crc))
    {
            /* if the last stored datapool crc differs to the current on,
                - a datapool download was done, the ecnr was not informed
                - the power supply got lost before the ecnr was notified from any download
                - in the case the crc differ from each other than a notification is initiated.
            */
        ret = ecnr_datapool_access_compare_crc(dpid, crc);

    }
#endif
    return ret;
}



static gboolean ecnr_configuration_is_bsd_file(const char* str)
{
    gboolean ret = FALSE;
    const char       *FileName = str;
    FILE*   fd;              /* pointer to config data file */


    pr_debug("ENTERED\n");

    /* create a file to indicate new datapool are available */
    fd = fopen (FileName,"r");

    if (NULL != fd)
    {
        fclose(fd);
        pr_debug("datapool file(%s) available\n", FileName);
        ret = TRUE;
    }

    return ret;
}


static const char* ecnr_configutation_get_new_datapool_file_name(enum eDataPool dpid)
{
    const  char* str = NULL;


    switch(dpid)
    {
        case ECNR_DP_BT_PHONE_NB:
            str = NEW_DP_FILE_BT_PHONE_NB;
            break;
        case ECNR_DP_BT_PHONE_WB:
            str = NEW_DP_FILE_BT_PHONE_WB;
            break;
        case ECNR_DP_BT_VOICE_NB:
            str = NEW_DP_FILE_BT_VOICE_NB;
            break;
        case ECNR_DP_BT_VOICE_WB:
            str = NEW_DP_FILE_BT_VOICE_WB;
            break;
        case ECNR_DP_BT_SIRI_NB:
            str = NEW_DP_FILE_BT_SIRI_NB;
            break;
        case ECNR_DP_BT_SIRI_WB:
            str = NEW_DP_FILE_BT_SIRI_WB;
            break;
        case ECNR_DP_CP_VOICE_NB:
            str = NEW_DP_FILE_CP_VOICE_NB;
            break;
        case ECNR_DP_CP_VOICE_WB:
            str = NEW_DP_FILE_CP_VOICE_WB;
            break;
        case ECNR_DP_CP_VOICE_SWB:
            str = NEW_DP_FILE_CP_VOICE_SWB;
            break;
        case ECNR_DP_CP_PHONE_NB:
            str = NEW_DP_FILE_CP_PHONE_NB;
            break;
        case ECNR_DP_CP_PHONE_WB:
            str = NEW_DP_FILE_CP_PHONE_WB;
            break;
        case ECNR_DP_CP_PHONE_SWB:
            str = NEW_DP_FILE_CP_PHONE_SWB;
            break;
        case ECNR_DP_CP_FTIME_SWB:
            str = NEW_DP_FILE_CP_FTIME_SWB;
            break;
        case ECNR_DP_AA_VOICE_NB:
            str = NEW_DP_FILE_AA_VOICE_NB;
            break;
        case ECNR_DP_AA_VOICE_WB:
            str = NEW_DP_FILE_AA_VOICE_WB;
            break;
        case ECNR_DP_AA_VOICE_SWB:
            str = NEW_DP_FILE_AA_VOICE_SWB;
            break;
        case ECNR_DP_AA_PHONE_NB:
            str = NEW_DP_FILE_AA_PHONE_NB;
            break;
        case ECNR_DP_AA_PHONE_WB:
            str = NEW_DP_FILE_AA_PHONE_WB;
            break;
        case ECNR_DP_AA_PHONE_SWB:
            str = NEW_DP_FILE_AA_PHONE_SWB;
            break;
        case ECNR_DP_INT_VOICE_NB:
            str = NEW_DP_FILE_INT_VOICE_NB;
            break;
        case ECNR_DP_INT_VOICE_WB:
            str = NEW_DP_FILE_INT_VOICE_WB;
            break;
        case ECNR_DP_WUW_VOICE_WB:
            str = NEW_DP_FILE_WUW_VOICE_WB;
            break;
        case ECNR_DP_VPA_VOICE_WB:
            str = NEW_DP_FILE_VPA_VOICE_WB;
            break;
        default:
            break;
    }

    return str;
}


/*static*/ void ecnr_configuration_set_new_datapool(enum eDataPool dpid)
{

    pr_debug("set new datapool(%d) to %d\n", dpid, TRUE);

    switch (dpid)
    {
        case ECNR_DP_BT_PHONE_NB:
            bNewdpBTPhoneNB = TRUE;
            break;
        case ECNR_DP_BT_PHONE_WB:
            bNewdpBTPhoneWB = TRUE;
            break;
        case ECNR_DP_BT_VOICE_NB:
            bNewdpBTVoiceNB = TRUE;
            break;
        case ECNR_DP_BT_VOICE_WB:
            bNewdpBTVoiceWB = TRUE;
            break;
        case ECNR_DP_BT_SIRI_NB:
            bNewdpBTSiriNB = TRUE;
            break;
        case ECNR_DP_BT_SIRI_WB:
            bNewdpBTSiriWB = TRUE;
            break;
        case ECNR_DP_CP_VOICE_NB:
            bNewdpCPVoiceNB = TRUE;
            break;
        case ECNR_DP_CP_VOICE_WB:
            bNewdpCPVoiceWB = TRUE;
            break;
        case ECNR_DP_CP_VOICE_SWB:
            bNewdpCPVoiceSWB = TRUE;
            break;
        case ECNR_DP_CP_PHONE_NB:
            bNewdpCPPhoneNB = TRUE;
            break;
        case ECNR_DP_CP_PHONE_WB:
            bNewdpCPPhoneWB = TRUE;
            break;
        case ECNR_DP_CP_PHONE_SWB:
            bNewdpCPPhoneSWB = TRUE;
            break;
        case ECNR_DP_CP_FTIME_SWB:
            bNewdpCPFTimeSWB = TRUE;
            break;
        case ECNR_DP_AA_VOICE_NB:
            bNewdpAAVoiceNB = TRUE;
            break;
        case ECNR_DP_AA_VOICE_WB:
            bNewdpAAVoiceWB = TRUE;
            break;
        case ECNR_DP_AA_VOICE_SWB:
            bNewdpAAVoiceSWB = TRUE;
            break;
        case ECNR_DP_AA_PHONE_NB:
            bNewdpAAPhoneNB = TRUE;
            break;
        case ECNR_DP_AA_PHONE_WB:
            bNewdpAAPhoneWB = TRUE;
            break;
        case ECNR_DP_AA_PHONE_SWB:
            bNewdpAAPhoneSWB = TRUE;
            break;
        case ECNR_DP_INT_VOICE_NB:
            bNewdpINTVoiceNB = TRUE;
            break;
        case ECNR_DP_INT_VOICE_WB:
            bNewdpINTVoiceWB = TRUE;
            break;
        case ECNR_DP_WUW_VOICE_WB:
            bNewdpWUWVoiceWB = TRUE;
            break;
        default:
            break;
    }

    return;
}



/*******************************************************************************
*
* FUNCTION: ecnr_configuration_check_config_files
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static int ecnr_configuration_check_config_files(void)
{
    int err = ECNR_ERR_OK;
    int i = 0;
    const  char* str = NULL;
    int loopctrl = 0;
    enum eDataPool dpid = (enum eDataPool)ECNR_DP_NOT_DEF;

    pr_debug("ENTERED\n");

    if(ECNR_CONFIG_FILE_table[0].file == NULL)
    {
        pr_debug("Product without configuration file");
        return ECNR_ERR_OK;
    }

    /* do not remove if tuning mode is active*/
    if(ecnr_sse_debug_check_enable_file())
    {
        pr_debug("tuning mode active the BSD/SCD file should not be removed\n");
        return ECNR_ERR_OK;
    }

    loopctrl = D_SAFE_MAX_LOOP_ECNR_CONFIGURATION;

    while (ECNR_CONFIG_FILE_table[i].file != NULL && loopctrl)
    {
        pr_debug("dpkey(%d), bsdfile(%s), removePowerOn(%d)\n",
                  ECNR_CONFIG_FILE_table[i].EcnrdPBaseKey,
                  ECNR_CONFIG_FILE_table[i].file,
                  ECNR_CONFIG_FILE_table[i].delete_on_powerOn);

        dpid = ECNR_CONFIG_FILE_table[i].EcnrdPBaseKey;

        /* is a bsd file still available, then check further for next condition */
        if(ecnr_configuration_is_bsd_file(ECNR_CONFIG_FILE_table[i].file))
        {
#ifdef D_HANDLE_FILE_REMOVE_POWER_ON
            if(ECNR_CONFIG_FILE_table[i].delete_on_powerOn)
            {
                err = ecnr_configuration_remove_bsd_file(ECNR_CONFIG_FILE_table[i].file);
                if(!err)
                    pr_debug("file(%s) was removed\n", ECNR_CONFIG_FILE_table[i].file);
            }
            else
            {
#endif
            str = ecnr_configutation_get_new_datapool_file_name(dpid);

            /* check whether no new datapool trigger file is available */
            if(!ecnr_configuration_is_new_datapool_file(str))
            {
                /* compare CRC from current bsd file and datapool
                   if not legal, the datpool element was updated, so trigger a new
                   datapool notification
                */
                if(!ecnr_configuration_compare_crc_dp_file(dpid))
                {
                    ecnr_configuration_set_new_datapool(dpid);
                }
            }
#ifdef D_HANDLE_FILE_REMOVE_POWER_ON
            }
#endif
        }
        i++;
        loopctrl--;
    }

    if(!loopctrl)
        pr_warning("the device configuration contains some incompatible values\n");

    return err;
}



#ifdef D_SUPPORT_GLOBAL_CONF_STRUCTURE

static struct EcnrConfigFile_e*  ecnr_configuration_get_dp_infos(struct EcnrConfig_e* c_this, enum eDataPool EcnrdPBaseKey)
{
    int loopctrl = 100;
    int count = 0;
    struct EcnrConfigFile_e*  ps = (struct EcnrConfigFile_e*)NULL;

    pr_message("enter\n");

    if(c_this)
    {
        ps = c_this->EcnrdpCfg;

        if(ps)
        {
            while (ps[count].bsd_str!= NULL && loopctrl)
            {
                if(ps[count].EcnrdPBaseKey == EcnrdPBaseKey)
                {
                    pr_message("configuration element associated to the dpid(%d)\n",  ps[count].EcnrdPBaseKey);
                    return &ps[count];
                }

                ++count;
                loopctrl--;
            }

        }
        else
        {
            pr_message("NULL pointer EcnrdpCfg[%d]\n", count);
            return (struct EcnrConfigFile_e*)NULL;
        }
    }

    pr_message("final, no configuration found for dpid(%d)\n", EcnrdPBaseKey);

    return (struct EcnrConfigFile_e*)NULL;
}


#define ECNR_NUM_CONFIG_e ARRAYSIZE(ECNR_CONFIG)

#endif // D_SUPPORT_GLOBAL_CONF_STRUCTUR


gboolean ecnr_configuration_get_crc_from_current_active_dataset(enum eDataPool dpid, guint16* crc)
{
  int err = ECNR_ERR_OK;
#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
  unsigned char*   dpdata = (unsigned char*)NULL;
    struct  ecnr_dp_data_struct pd;
    int   dpsize = 0;
    int    cfg_dpsize = 0;
    guint   i = 0;

    pr_message("ENTERED\n");


    err    = ecnr_configuration_get_dp_size(dpid, &dpsize);

    dpdata = ecnr_datapool_access_get_bsd_dataset(dpid, NULL,  NULL, NULL);

    if(dpsize)
        cfg_dpsize = dpsize;

    if(!dpdata)
    {
        pr_message("Datapool(%d) is empty, default datapool is retrieved \n", dpid );

        /* search default dataset */
        for( i = 0 ; i < ECNR_NUM_CONFIG ; i++)
        {
            if(ECNR_CONFIG_table[i].EcnrdPBaseKey == dpid)
            {
                dpdata = ECNR_CONFIG_table[i].configData.data;
                cfg_dpsize = (int)ECNR_CONFIG_table[i].configData.size;
                pr_message("found default Datapool with size(%d)\n", cfg_dpsize );
                break;
            }
        }
    }

    if(dpdata)
    {
        if((cfg_dpsize - D_NB_CHAR_CRC_DP))
        {
            memmove((void*)&pd.crc, &dpdata[cfg_dpsize - D_NB_CHAR_CRC_DP] , sizeof(guint16));
            pr_message("cfg_dpkey(%d), cfg_dpsize(%d), dp_crc(0x%x)\n", dpid, cfg_dpsize, pd.crc);
            *crc = pd.crc;
        }
    }
    else
    {
        pr_message("Datapool(%d) is empty, default datapool is not available \n", dpid );
    }
#endif // VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    //to remove prio2 lint warning (Info 715: Symbol not referenced)
            (void*)crc;
            (void*)dpid;
#endif
    return err;
}
/*****************************************************************************
* FUNCTION:    ecnr_configuration_print_all_dp_infos
* PARAMETER:    none
* RETURNVALUE:  "0" on success  or "non-zero" value in case of error
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Patrick Rey
******************************************************************************/
int ecnr_configuration_print_all_dp_infos(void)
{
    int err = ECNR_ERR_OK;
 #ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    unsigned char*   dpdata = (unsigned char*)NULL;
    int loopctrl        = 0;
    enum eDataPool dpid = (enum eDataPool)ECNR_DP_NOT_DEF;
    struct ecnr_dp_data_struct pd;
    int dpsize          = 0;
    int                 select_io_pin;
    gboolean            select_logical_level;
    int cfg_dpsize      = 0;
    int count           = 0;
    guint i             = 0;

    pr_message("ENTERED\n");

    memset( &pd, 0, sizeof(struct ecnr_dp_data_struct) );

    loopctrl = D_SAFE_MAX_LOOP_ECNR_CONFIGURATION;


    while (ECNR_CONFIG_FILE_table[count].file != NULL && loopctrl)
    {

        dpid    = ECNR_CONFIG_FILE_table[count].EcnrdPBaseKey;
        dpsize  = (int)ECNR_CONFIG_FILE_table[count].dpsize;
        select_io_pin = ECNR_CONFIG_FILE_table[count].select_io_pin;
        select_logical_level = ECNR_CONFIG_FILE_table[count].select_logical_level;


        dpdata = ecnr_datapool_access_get_bsd_dataset(dpid, NULL,  NULL, NULL);

        if(!dpdata)
        {
            pr_message("Datapool(%d) is empty, default datapool is retrieved \n", dpid );

            /* search default data set */
            for( i = 0 ; i < ECNR_NUM_CONFIG ; i++)
            {
                if(ECNR_CONFIG_table[i].EcnrdPBaseKey == dpid)
                {
                    dpdata = ECNR_CONFIG_table[i].configData.data;
                    cfg_dpsize = (int)ECNR_CONFIG_table[i].configData.size;
                    pr_message("found default Datapool with size(%d)\n", cfg_dpsize );
                    break;
                }
            }
        }

        if(dpdata)
        {
            /* retrieve header information*/
            memmove((void*)&pd.hinfos, &dpdata[0] , sizeof(struct ecnr_dp_header_infos));
            pd.Data = &dpdata[D_NB_CHAR_VERSION + D_NB_CHAR_SIZE_DP];

            #if 0
            if((dpsize + D_DP_ALL_INFO_ITEMS_SIZE) -(1 + D_NB_CHAR_CRC_DP))
                memmove((void*)&pd.crc, &dpdata[(dpsize + D_DP_ALL_INFO_ITEMS_SIZE) - (D_NB_CHAR_CRC_DP)] , sizeof(guint16));
            #endif
            if((dpsize - D_NB_CHAR_CRC_DP))
                memmove((void*)&pd.crc, &dpdata[dpsize - D_NB_CHAR_CRC_DP] , sizeof(guint16));
            pr_message("cfg_dpkey(%d), cfg_dpsize(%d), dp_bsd_file_legnth(%d), dp_crc(0x%x), io_sel(%d), io_lv(%d)\n", dpid, dpsize, pd.hinfos.bsd_file_length, pd.crc, select_io_pin, select_logical_level);

            for (i = 0; i < 4; i++)
            {
                pr_message("dp_sse_version[%d] = %d\n", i, pd.hinfos.sse_lib_version[i]);
            }
        }
        else
        {
            pr_message("Datapool(%d) is empty, default datapool is not available \n", dpid );
        }

        ++count;
        loopctrl--;
    }
    pr_message("the target is configured with %d datapool\n", count);


#ifdef D_SUPPORT_GLOBAL_CONF_STRUCTURE
    /* extended */
    /* search default dataset */
    int configId = 0;
    int j = 0;
    struct EcnrConfigFile_e* pcfg;

    for( j = 0 ; j < ECNR_NUM_CONFIG_e ; j++)
    {
        configId = ECNR_CONFIG_table_e.EcnrCfg[j].configId;
        dpid     = ECNR_CONFIG_table_e.EcnrCfg[j].EcnrdPBaseKey;

        pr_message("configuration(%d) extension with ConfigId(%d), dpId(%d)\n", j, configId, dpid );

        // search all datapool
        pcfg = ecnr_configuration_get_dp_infos(&ECNR_CONFIG_table_e, dpid);

        if(pcfg)
        {
            dpdata = pcfg->configData.data;
            dpsize = pcfg->configData.size;


            /* retrieve header infos*/
            memmove((void*)&pd.hinfos, &dpdata[0] , sizeof(struct ecnr_dp_header_infos));
            pd.Data = &dpdata[D_NB_CHAR_VERSION + D_NB_CHAR_SIZE_DP];

            if((dpsize - D_NB_CHAR_CRC_DP))
                memmove((void*)&pd.crc, &dpdata[dpsize - D_NB_CHAR_CRC_DP] , sizeof(guint16));

            pr_message("cfg_dpkey(%d), cfg_dpsize(%d), dp_bsd_file_legnth(%d), dp_crc(0x%x)\n", dpid, dpsize, pd.hinfos.bsd_file_length, pd.crc);

            for (i = 0; i < 4; i++)
            {
                pr_message("dp_sse_version[%d] = %d\n", i, pd.hinfos.sse_lib_version[i]);
            }
        }
        else
        {
            pr_message("Datapool(%d) is empty, default datapool is not available \n", dpid );
        }

    }

#endif // D_SUPPORT_GLOBAL_CONF_STRUCTURE
#endif // VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_configuration_check_AppId
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_configuration_get_dp_size(enum eDataPool dpkey, int* dpsize)
{
    int err = ECNR_ERR_OK;
    int i = 0;
    int loopctrl = 0;
    loopctrl = D_SAFE_MAX_LOOP_ECNR_CONFIGURATION;


    while (ECNR_CONFIG_FILE_table[i].file != NULL && loopctrl)
    {
        if(ECNR_CONFIG_FILE_table[i].EcnrdPBaseKey == dpkey)
        {
            if(ECNR_CONFIG_FILE_table[i].dpsize == 0)
            {
                /* dpsize is null, set a default value*/
                *dpsize = 2000;
                pr_warning("size of datapool(%d) is zero\n", dpkey);
                return ECNR_ERR_DP_ERROR_SIZE_OF_DATAPOOL;
            }
            else
            {
                *dpsize = (int)ECNR_CONFIG_FILE_table[i].dpsize;
                pr_debug("size of datapool(%d) is %d\n", dpkey, *dpsize);
                return ECNR_ERR_OK;
            }
        }
        i++;
        loopctrl--;
    }


    return err;
}



#ifdef D_DP_NOTIFICATION_SUPPORT
static void ecnr_configuration_notify_new_configuration_phone_nb(void)
{

    pr_message("new configuration for phone narrow band available");
    bNewdpBTPhoneNB = TRUE;

    /* create a file to indicate a new datapool is available */
    ecnr_configuration_create_new_datapool_file(NEW_DP_FILE_BT_PHONE_NB);

}

static void ecnr_configuration_notify_new_configuration_phone_wb(void)
{

    pr_message("new configuration for phone wide band available");
    bNewdpBTPhoneWB = TRUE;

    /* create a file to indicate a new datapool is available */
    ecnr_configuration_create_new_datapool_file(NEW_DP_FILE_BT_PHONE_WB);

}

static void ecnr_configuration_notify_new_configuration_voice_nb(void)
{

    pr_message("new configuration for voice narrow band available");
    bNewdpBTVoiceNB = TRUE;

    /* create a file to indicate a new datapool is available */
    ecnr_configuration_create_new_datapool_file(NEW_DP_FILE_BT_VOICE_NB);

}

static void ecnr_configuration_notify_new_configuration_voice_wb(void)
{

    pr_message("new configuration for voice wide band available");
    bNewdpBTVoiceWB = TRUE;

    /* create a file to indicate a new datapool is available */
    ecnr_configuration_create_new_datapool_file(NEW_DP_FILE_BT_VOICE_WB);
}

static void ecnr_configuration_notify_new_configuration_cp_phone_nb(void)
{

    pr_message("new configuration for cp phone narrow band available");
    bNewdpCPPhoneNB = TRUE;

    /* create a file to indicate a new datapool is available */
    ecnr_configuration_create_new_datapool_file(NEW_DP_FILE_CP_PHONE_NB);
}

static void ecnr_configuration_notify_new_configuration_cp_phone_wb(void)
{

    pr_message("new configuration for cp phone wide band available");
    bNewdpCPPhoneWB = TRUE;

    /* create a file to indicate a new datapool is available */
    ecnr_configuration_create_new_datapool_file(NEW_DP_FILE_CP_PHONE_WB);
}

static void ecnr_configuration_notify_new_configuration_cp_voice_wb(void)
{

    pr_message("new configuration for cp siri wide band available");
    bNewdpCPVoiceWB = TRUE;

    /* create a file to indicate a new datapool is available */
    ecnr_configuration_create_new_datapool_file(NEW_DP_FILE_CP_VOICE_WB);
}
#endif

/*******************************************************************************
*
* FUNCTION: ecnr_configuration_check_AppId
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_configuration_check_AppId(unsigned char AppId)
{
    unsigned short ct = 0;
    int ret = ECNR_ERR_APPID_NOT_AVAILABLE;

    pr_debug("Check AppId: %d\n", AppId );

    if(AppId == ECNR_APPID_ADMIN)
        return ECNR_ERR_OK;

    if(AppId != ECNR_APPID_NA)
    {
        while (ct < ECNR_CFG_ID_TABLE_SIZE)
        {
            if(ECNR_USER_CONF_table[ct].EcnrAppId == AppId)
            {
                ret = ECNR_ERR_OK;
                ct = ECNR_CFG_ID_TABLE_SIZE + 1;
            }
            ct = (unsigned short)(ct + 1);
        }
    }
    pr_debug("EXIT\n");

    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_configuration_check_ConfigId
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/

int ecnr_configuration_check_ConfigId(short configId)
{
    unsigned short ct = 0;
    int ret = ECNR_ERR_CONFID_NOT_AVAILABLE;

    pr_debug("Check ConfigId: %d\n", configId );

    while (ct < ECNR_CFG_ID_TABLE_SIZE)
    {
        if(ECNR_USER_CONF_table[ct].EcnrAppConfId == configId)
        {
            ret = ECNR_ERR_OK;
            ct = ECNR_CFG_ID_TABLE_SIZE + 1;
        }
        ct = (unsigned short)(ct + 1);
    }

    pr_debug("EXIT\n");

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_configuration_check_match_AppId_and_configId
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_configuration_check_match_AppId_and_configId(unsigned short configId, unsigned char AppId)
{
    unsigned short ct = 0;
    int ret = ECNR_ERR_APPID_CONFIGID_NOT_MATCHING;

    pr_debug("ENTERED");

    if(AppId == ECNR_APPID_ADMIN)
        return ECNR_ERR_OK;

    while (ct < ECNR_CFG_ID_TABLE_SIZE)
    {
        if(ECNR_USER_CONF_table[ct].EcnrAppConfId == configId)
        {
            if (ECNR_USER_CONF_table[ct].EcnrAppId == AppId)
            {
                pr_debug("AppId :%d, and ConfigId: %d are matching together with element [%d] \n", AppId, configId, ct);
                return ECNR_ERR_OK;
            }
        }
        ct = (unsigned short)(ct + 1);
    }

    pr_debug("AppId :%d, and ConfigId: %d are not matching together within ECNR_USER_CONF_table[%d] elements \n", AppId, configId, ct);

    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_configuration_get_config_data
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/

const struct EcnrConfig *ecnr_configuration_get_config_data(unsigned short configId)
{
    guint i;
    const struct EcnrConfig *configdata;
    unsigned short ecnrCfgId;

    pr_debug("ENTERED");

#ifndef D_INCREASE_ECNR_START_UP_PERFORMANCE
    if(!bConfigPrinted)
    {
        ecnr_configuration_print_config_data();
        bConfigPrinted = TRUE;
    }
#endif
    /* retrieve ECNR configuration Id from mapping table user Id <-> ECNR Id*/
    ecnrCfgId = ecnr_configuration_get_ecnr_config_Id(configId);


    for (i = 0, configdata = ECNR_CONFIG_table; i < ECNR_NUM_CONFIG; i++, configdata++)
    {

        if (configdata->configId == ecnrCfgId)
        {
            pr_debug("ecnrCfgId: %d, loc: %s\n", configdata->configId, configdata->bsd_str);
            ETG_TRACE_USR3(("[ecnr_configuration_get_config_data]: ecnr_cfg_id -> %d, bsd_file -> %s", configdata->configId, configdata->bsd_str));
            return configdata;
        }
    }

    return (struct EcnrConfig*)NULL;
}


/*******************************************************************************
*
* FUNCTION: ecnr_get_config_data
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/

guint8 ecnr_configuration_get_config_class(unsigned short configId)
{
    const struct EcnrConfig *pdata;
    guint8 class = 0;

    pr_debug("ENTERED");

    pdata = ecnr_configuration_get_config_data(configId);


    if(!pdata)
        return class;

    pr_debug("extract class for configId: %d\n", pdata->configId);

    class = GET_U8_CONFIG_CLASS(pdata->configId );

    pr_debug("config class is : %d", class);

    return class;
}

/*******************************************************************************
*
* FUNCTION: ecnr_get_config_data
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/

guint8 ecnr_configuration_get_config_type(unsigned short configId)
{
    const struct EcnrConfig *pdata;
    guint8 type = 0;

    pr_debug("ENTERED");

    pdata = ecnr_configuration_get_config_data(configId);


    if(!pdata)
        return type;

    pr_debug("extract type for configId: %d\n", pdata->configId);

    type = GET_U8_CONFIG_TYPE(pdata->configId );
    pr_debug("config type is : %d", type);

    return type;
}

/*******************************************************************************
*
* FUNCTION: ecnr_get_config_data
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/

guint8 ecnr_configuration_get_config_instance(unsigned short configId)
{
    const struct EcnrConfig *pdata;
    guint8 instance = 0;

    pr_debug("ENTERED");

    pdata = ecnr_configuration_get_config_data(configId);

    if(!pdata)
        return instance;

    pr_debug("extract instance for configId: %d\n", pdata->configId);

    instance = GET_U8_CONFIG_INSTANCE(pdata->configId );

    pr_debug("config instance is ; %d", instance);

    return instance;
}

/*******************************************************************************
*
* FUNCTION: ecnr_configuration_get_config_feature
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/

guint8 ecnr_configuration_get_config_feature(unsigned short configId)
{
   const struct EcnrConfig *pdata;
   guint8 feature = 0;

   pr_debug("ENTERED");

   pdata = ecnr_configuration_get_config_data(configId);

   if(!pdata)
      return feature;

   pr_debug("extract feature for configId: %d\n", pdata->configId);

   feature = GET_U8_CONFIG_FEATURE(pdata->configId );

   pr_debug("config feature is : %d", feature);

   return feature;
}


/*******************************************************************************
*
* FUNCTION: ecnr_configuration_read_configuration_bsd_file
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static int ecnr_configuration_read_configuration_bsd_file(const char *FileName, tecnrConfigData *ConfigData)
{
    FILE            *pFile = NULL;   /* pointer to config data file */
    int             iError = 0;      /* internal error status */
    unsigned int    uDataSize = 0;   /* Config data buffer size */

    pr_debug("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_configuration_read_configuration_bsd_file]: ENTERED"));

    /* check for NULL pointer */
    if( NULL == ConfigData)
    {
        pr_warning("configdata struct pointer NULL");
        return ECNR_ERR_NULL_POINTER;
    }

    /* File name is empty string or NULL pointer means we do not have a
    * configuration file (not really an error)
    */
    if((NULL != FileName) && ('\0' != FileName[0]))
    {
        pFile = fopen(FileName, "rb");

        if(NULL == pFile)
        {
            pr_warning("fails to open the configuration file -> %s", FileName);
            ETG_TRACE_USR3(("[ecnr_configuration_read_configuration_bsd_file]: fails to open the configuration file -> %s", FileName));
            iError = ECNR_ERR_NULL_POINTER; /* opening file failed */
        }
        else
        {
            /* find out the length of the file
            */
            fseek(pFile, 0L, SEEK_END);
            uDataSize = (unsigned int)ftell(pFile);

#ifdef  D_ECNR_LIB_SSE_4_03
            unsigned int tempsize = (unsigned int)(uDataSize / 4);
#endif

            fseek(pFile, 0L, SEEK_SET); /* go back to the beginning */

            if (uDataSize==0)
            {
                pr_warning("reading from file failed -> empty file");
                ETG_TRACE_USR3(("[ecnr_configuration_read_configuration_bsd_file]: reading from file failed -> empty file"));
                iError = ECNR_ERR_CONFIG_FILE_READING_FAILED; /* memory allocation failed */
            }
            else
            {
                /* allocate memory for configuration data buffer */
                int n = ConfigData->ConfigBufferCnt;


#ifdef  D_ECNR_LIB_SSE_3_17
                ConfigData->ConfigBuffer[n] = (char *)g_malloc0(uDataSize);
                if(NULL == ConfigData->ConfigBuffer[n])
                {
                    pr_warning("fail to read the configuration file, memory allocation failed");
                    ETG_TRACE_USR3(("[ecnr_configuration_read_configuration_bsd_file]: fail to read the configuration file, memory allocation failed"));
                    iError = ECNR_ERR_CONFIG_FILE_READING_FAILED;
                }
                else
                {
                    /* initialize configuration data structure and read configuration data from file */
                    ConfigData->ConfigBufferSize[n] = uDataSize; /* size of configuration buffer */

                    if(uDataSize != (unsigned int) fread(ConfigData->ConfigBuffer[n], 1, (size_t)uDataSize, pFile))
                    {
                        pr_warning("fail to read the configuration file");
                        ETG_TRACE_USR3(("[ecnr_configuration_read_configuration_bsd_file]: fail to read the configuration file"));
                        iError = ECNR_ERR_CONFIG_FILE_READING_FAILED;
                    }
                    else
                        ConfigData->ConfigBufferCnt++; /* count of config buffers */
                }
#elif   D_ECNR_LIB_SSE_4_03
                ConfigData->i32ConfigBuffer[n] = (int32_t*)g_malloc0(uDataSize);
                if(NULL == ConfigData->i32ConfigBuffer[n])

                {
                    pr_warning("fail to read the configuration file, memory allocation failed");
                    ETG_TRACE_USR3(("[ecnr_configuration_read_configuration_bsd_file]: fail to read the configuration file, memory allocation failed"));
                    iError = ECNR_ERR_CONFIG_FILE_READING_FAILED;
                }
                else
                {
                    /* initialize configuration data structure and read configuration data from file */
                    ConfigData->ConfigBufferSize[n] = uDataSize; /* size of configuration buffer */

                    if(tempsize != (unsigned int)fread(ConfigData->i32ConfigBuffer[n], 4, (size_t)tempsize, pFile))
                    {
                        pr_warning("fail to read the configuration file");
                        ETG_TRACE_USR3(("[ecnr_configuration_read_configuration_bsd_file]: fail to read the configuration file"));
                        iError = ECNR_ERR_CONFIG_FILE_READING_FAILED;
                    }
                    else
                        ConfigData->ConfigBufferCnt++; /* count of config buffers */

//                    FILE            *tempFile = NULL;   /* pointer to config data file */
//                    tempFile = fopen("/var/opt/bosch/dynamic/audiomanager/ecnr/tempFile.txt", "w");
//                    if((int32_t)tempsize != fwrite(ConfigData->i32ConfigBuffer[n], 4, tempsize, tempFile))
//                    {
//                        pr_warning("fail to write the configuration file");
//
//                    }
//                    fclose(tempFile);

                }
#endif
            }
        }
    }
    else
    {
        pr_warning("Filename is NULL");
        iError = ECNR_ERR_CONFIG_FILE_EMPTY; /* memory allocation failed */
    }

    /* close file */
    if(NULL != pFile)
    {
        fclose(pFile);
    }

    pr_debug("EXIT with error: %d",iError );
    ETG_TRACE_USR3(("[ecnr_configuration_read_configuration_bsd_file]: EXIT with error -> %d ", iError));

  return iError;
}

#ifdef D_KDS_SUPPORT
/*******************************************************************************
*
* FUNCTION: KDSRead
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static int ecnr_configuration_kds_Read(guint8 *Buffer, guint16 Length, OSAL_tIODescriptor FDKDS, tsKDSEntry *Entry, guint16 StartOffset, guint16 BaseKey, guint16 BlockLen, guint16 MaxLen)
{
    int     count = 0;
    int     err = OSAL_E_NOERROR;

    pr_debug("ENTERED");

    if(StartOffset + Length > MaxLen)
        err = OSAL_E_INVALIDVALUE;

    if(err == OSAL_E_NOERROR && Length > 0)
    {
        guint16    key = StartOffset / BlockLen;
        guint16    offset = StartOffset - key * BlockLen;
        guint16    len = offset + Length;
        key += BaseKey;

        if(len > BlockLen)
            len = BlockLen;

        if(key != Entry->u16Entry)
        {
            Entry->u16Entry = key;
            Entry->u16EntryLength = len;
            Entry->u16EntryFlags = M_KDS_ENTRY_FLAG_NONE;

            tS32    result = OSAL_s32IORead(FDKDS, (tS8 *)Entry, sizeof(tsKDSEntry));
            if(result == OSAL_ERROR)
            {
                err = OSAL_u32ErrorCode();
                pr_debug("osal error: %d",err );
            }
        }

        if(err == OSAL_E_NOERROR)
        {
            count = len - offset;
            memcpy(Buffer, Entry->au8EntryData + offset, count);

#if 0
            if(len < Length)
            {
                int     result = KDSRead(Buffer + count, Length - count, FDKDS, Entry, StartOffset + count, BaseKey, BlockLen, MaxLen);
                if(result < 0)
                    err = OSAL_E_UNKNOWN;
                else
                    count += result;
            }
#endif
    }

  }

    pr_debug("EXIT");

    if(err == OSAL_E_NOERROR)
        return count;

    return -1;
}

/*******************************************************************************
*
* FUNCTION: ecnr_configuration_read_configuration_kds
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/

int ecnr_configuration_read_configuration_kds(int kdsKey, tecnrConfigData *ConfigData)
{
    OSAL_tIODescriptor  fdkds;
    KDSDataHeader       *header = NULL;
    guint32             err = OSAL_E_NOERROR;
    tsKDSEntry          entry;
    guint8              hdrbuff[KDS_CFG_HEADER_SIZE];
    gint32              result;
    int                 n = 0;

    pr_debug("ENTERED");

    if (!kdsKey)
    {
        pr_message("invalid kds key: %d\n", kdsKey);
        return ECNR_ERR_CONFID_NOT_AVAILABLE;
    }

    if(!ConfigData)
    {
        pr_critical("invalid configuration data structure \n");
        return ECNR_ERR_NULL_POINTER;
    }


    fdkds = OSAL_IOOpen(OSAL_C_STRING_DEVICE_KDS, OSAL_EN_READONLY);
    if(fdkds == OSAL_ERROR)
    {
        err = OSAL_u32ErrorCode();
        pr_debug("faile to eopen osal device OSAL_C_STRING_DEVICE_KDS with error: %d \n", err );
        return ECNR_ERR_FAILED;
    }

    // Read header from KDS

    header = &KDSDataHeaderPhone;
    entry.u16Entry = 0xffff;

    /* read kds header data */
    result = ecnr_configuration_kds_Read((guint8 *)hdrbuff,
                            KDS_CFG_HEADER_SIZE,
                            fdkds,
                            &entry,
                            KDS_CFG_START,
                            kdsKey,
                            KDS_CFG_BLOCK_LEN,
                            (KDS_CFG_LEN + KDS_CFG_START));
    if(result < 0)
        err = OSAL_E_UNKNOWN;
    else
    {
        int     i = 0;
        header->InternalMajorVersion=hdrbuff[i];                  i++;
        header->InternalMinorVersion=*((guint16 *)&hdrbuff[i]);  i+=2;
        header->DataSetId=hdrbuff[i];                            i++;
        header->CarId=*((guint16 *)&hdrbuff[i]);                 i+=2;
        header->Interieur=hdrbuff[i];                            i++;
        header->LibMajorVersion=hdrbuff[i];                      i++;
        header->LibMinorVersion=*((guint16 *)&hdrbuff[i]);       i+=2;
        header->BSDSize=*((guint16 *)&hdrbuff[i]);               i+=2;

        if(header->InternalMajorVersion != InternalMajorVersion)
        {
            err = OSAL_E_INVALIDVALUE;

        }
        else if(header->BSDSize == 0 || header->BSDSize > KDS_MAX_BSD_LEN)
        {
            err = OSAL_E_INVALIDVALUE;
        }
        else
        {
            n = ConfigData->ConfigBufferCnt;

            ConfigData->ConfigBufferSize[n] = header->BSDSize;
            ConfigData->ConfigBuffer[n] = (char *)g_malloc0(header->BSDSize);

            if(ConfigData->ConfigBuffer[n] == NULL)
                err = OSAL_E_NOSPACE;
            else
            {
                guint16    readpos = KDS_CFG_START + KDS_CFG_HEADER_SIZE;
                result = ecnr_configuration_kds_Read((tU8 *)ConfigData->ConfigBuffer[n],
                                                            header->BSDSize,
                                                            fdkds,
                                                            &entry,
                                                            readpos,
                                                            kdsKey,
                                                            KDS_CFG_BLOCK_LEN,
                                                            (KDS_CFG_LEN + KDS_CFG_START));
                if(result < 0)
                    err = OSAL_E_UNKNOWN;
            }
        }
    }

    if(err == OSAL_E_NOERROR)
        ConfigData->ConfigBufferCnt++;
    else
    {
        if(ConfigData->ConfigBuffer[n])
            free(ConfigData->ConfigBuffer[n]);
        ConfigData->ConfigBuffer[n] = NULL;
    }

    (tVoid)OSAL_s32IOClose(fdkds);

    pr_debug("EXIT");

    return err;
}

#endif // D_KDS_SUPPORT



/*******************************************************************************
*
* FUNCTION: ecnr_configuration_free_Configuration_Data
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
void ecnr_configuration_free_Configuration_Data(tecnrConfigData *ConfigData)
{
    unsigned short  i;

    pr_debug("ENTERED");

    if(ConfigData != NULL)
    {
        for(i=0; i<ConfigData->ConfigBufferCnt; i++)
        {
            if(ConfigData->ConfigBuffer[i] != NULL)
            {
                free(ConfigData->ConfigBuffer[i]);
                ConfigData->ConfigBuffer[i] = NULL;

                /*new*/
                ConfigData->ConfigBufferSize[i] = 0;
            }
#ifdef D_ECNR_LIB_SSE_4_03
            if(ConfigData->i32ConfigBuffer[i] != NULL)
            {
                free(ConfigData->i32ConfigBuffer[i]);
                ConfigData->i32ConfigBuffer[i] = NULL;

                /*new*/
                ConfigData->ConfigBufferSize[i] = 0;
            }
#endif
        }

        ConfigData->ConfigBufferCnt = 0;
    }

    pr_debug("EXIT");
}


/*******************************************************************************
*
* FUNCTION: ecnr_configuration_read_configuration_dp
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static int ecnr_configuration_read_configuration_dp(enum eDataPool dpKey, tecnrConfigData *ConfigData)
{
    int         err = ECNR_ERR_OK;
#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    guint32     dataset_size = 0;
    unsigned char*       dpdata = (unsigned char*)NULL;
    guint16     size = 0;
    int         pos= 0;
    int         version[4]={0};
    guint16     crc = 0;
#endif

    pr_message("setup ecnr configuration directly from datpool element %d \n", dpKey);

    if((dpKey == ECNR_DP_NOT_DEF) || (dpKey >= ECNR_DP_LAST))
    {
        pr_message("Invalide dp element %d", dpKey);
        return ECNR_ERR_DP_INVALID_ELEMENT;
    }

  /* read dp element */
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    /* warning lint:Info 774: prio1, change code dpdata = (char*)NULL; */
    pr_message("No dp configuration available");
    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)ConfigData;
    return ECNR_ERR_DP_NO_ELEMENT_AVAILABLE;
#else
    dpdata = ecnr_datapool_access_get_bsd_dataset(dpKey, (guint16*)&dataset_size, version, &crc );

    if (!dpdata)
    {
        pr_message("No dp configuration available");
        return ECNR_ERR_DP_NO_ELEMENT_AVAILABLE;
    }

    if(!ConfigData)
    {
        pr_warning("invalid configuration container");
        return ECNR_ERR_NULL_POINTER;
    }

    /* compare SSE version in dataset from datapool with the one of the current SSE version  */
    if(FALSE == ecnr_sse_handler_compare_version(version))
        return ECNR_ERR_VERSION_SSE_DP_DO_NOT_MATCH;

#ifdef D_ECNR_LIB_SSE_3_17
    ConfigData->ConfigBuffer[ConfigData->ConfigBufferCnt] = (char *)g_malloc0(dataset_size);

    if(ConfigData->ConfigBuffer[ConfigData->ConfigBufferCnt] == NULL)
    {
        pr_warning("allocation for SSE configuration buffer failed\n");
        return ECNR_ERR_NULL_POINTER;
    }

    ConfigData->ConfigBufferSize[ConfigData->ConfigBufferCnt] = dataset_size;

    pos = ecnr_datapool_access_get_config_row_data_start_pos(dpdata, &size);

    /* copy DP element data in SSE configuration buffer */
    memcpy(ConfigData->ConfigBuffer[ConfigData->ConfigBufferCnt], &dpdata[pos], dataset_size);
    ConfigData->ConfigBufferCnt++;
#elif   D_ECNR_LIB_SSE_4_03

    ConfigData->i32ConfigBuffer[ConfigData->ConfigBufferCnt] = (int32_t*)g_malloc0(dataset_size);

    if(ConfigData->i32ConfigBuffer[ConfigData->ConfigBufferCnt] == NULL)
    {
        pr_warning("allocation for SSE configuration buffer failed\n");
        return ECNR_ERR_NULL_POINTER;
    }

    ConfigData->ConfigBufferSize[ConfigData->ConfigBufferCnt] = dataset_size;

    pos = ecnr_datapool_access_get_config_row_data_start_pos(dpdata, &size);

    /* copy DP element data in SSE configuration buffer */
    memcpy(ConfigData->i32ConfigBuffer[ConfigData->ConfigBufferCnt], &dpdata[pos], dataset_size);
    ConfigData->ConfigBufferCnt++;
#endif

#endif

    pr_debug("EXIT\n");
    return err;
}



/*******************************************************************************
*
* FUNCTION: ecnr_configuration_create_bsd_file_from_configuration_data_container
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static int ecnr_configuration_create_bsd_file_from_configuration_data_container(guint8 datatype, const struct EcnrConfig* data)
{
    int     err = ECNR_ERR_OK;
    guint8  dt =  datatype;

    pr_debug("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_configuration_create_bsd_file_from_configuration_data_container]: ENTERED"));

    if(!data)
        return ECNR_ERR_NULL_POINTER;

    if (dt == CONFIG_TYPE_FILE)
    {
        /* return data from dp if available */
        if(data->EcnrdPBaseKey)
            dt = CONFIG_TYPE_DP;
        else
            dt = CONFIG_TYPE_TABLE;
    }

    /* select data container to create bsd file */
    switch(dt)
    {
        case CONFIG_TYPE_TABLE:
            pr_message("load default dataset\n");
            ETG_TRACE_USR3(("[ecnr_configuration_create_bsd_file_from_configuration_data_container]: load default dataset"));
            err = ecnr_configuration_create_bsd_files_from_data(data);
            break;
        case CONFIG_TYPE_DP:
            /* create bsd file with the new data pool
             * the present file is first removed, a new is create..
             */
            pr_message("load dataset out of datapool element\n");
            ETG_TRACE_USR3(("[ecnr_configuration_create_bsd_file_from_configuration_data_container]: load dataset out of datapool element"));
            err = ecnr_configuration_create_bsd_files_from_dp(data);
            break;
        case CONFIG_TYPE_SET_FCT:
        case CONFIG_TYPE_KDS:
        case CONFIG_TYPE_FILE:
        default:
            pr_warning("no configuration available");
            err = ECNR_ERR_PARAM;
            break;
    }

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_configuration_get_configuration
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_configuration_get_configuration
(
    unsigned short  ConfigId,
    tecnrConfigData* pConfigData,
    gboolean*       isdefault
)
{
    int err = (int)ECNR_ERR_OK;
    const struct EcnrConfig *pdata = NULL;

    pr_debug("Requested Configid is %d \n",ConfigId);


    /* retrieve configuration data for the requested configuration ID */
    pdata = ecnr_configuration_get_config_data(ConfigId);

    if (!pdata)
    {
        pr_warning("no valid configuration for configId : % d", ConfigId);
        return ECNR_ERR_CONFIG_NOT_AVAILABLE;
    }


#ifdef D_INCREASE_ECNR_START_UP_PERFORMANCE
    err = ecnr_configuration_read_configuration_dp(pdata->EcnrdPBaseKey, pConfigData);
    if(err)
        goto end;
    pr_message("configuration build with D_INCREASE_ECNR_START_UP_PERFORMANCE switch\n");
    return err;
#else
    if(ecnr_service_is_use_default_dp())
    {
        pr_message("load default dp mode is set\n");

        err = ecnr_configuration_remove_bsd_file(pdata->bsd_str);
        if(!err)
        {
            err = ecnr_configuration_create_bsd_file_from_configuration_data_container(CONFIG_TYPE_TABLE, pdata);
            if(!err)
                *isdefault = TRUE;
        }
    }


    /* first check whether a new configuration was notify */
    /* new data pool available */
    if(   (err && ecnr_service_is_use_default_dp())                       // -> the try to load the default dataset has failed
        || ecnr_configuration_is_new_dp_available(pdata->EcnrdPBaseKey))  // -> new datapool was notified
    {
        /* create bsd file with the new data pool
         * the present file ist first removed, a new is create..
         */
        err = ecnr_configuration_remove_bsd_file(pdata->bsd_str);
        if(!err)
        {
            err = ecnr_configuration_create_bsd_files_from_dp(pdata);

            pr_debug("create bsd file configuration from dp err=%d \n",err);
        }
    }

    /* always try to read the configuration data from a bsd file  */
    if(pdata->bsd_str) /* read from bsd file is configured */
    {
        err = ecnr_configuration_read_configuration_bsd_file(pdata->bsd_str, pConfigData);
        if(!err)
        {
            pr_debug("configuration data was returned from the bsd file, err->%d \n",err);
            return err;
        }
        else
        {
            /* error while trying to read from file
             * a bsd file is missing, the file should be created on the base of data of
             * one of the configured data container. Hence, the configuration is reading from the
             * newly created file
             */

            err = ecnr_configuration_create_bsd_file_from_configuration_data_container(ecnr_configuration_get_config_type(ConfigId), pdata);

            /* read from dp failed :: recover from default dataset
             */
            if(err)
            {
                err = ecnr_configuration_create_bsd_file_from_configuration_data_container(CONFIG_TYPE_TABLE, pdata);
                if(!err)
                     *isdefault = TRUE;
            }

            if(!err)
            {
                err = ecnr_configuration_read_configuration_bsd_file(pdata->bsd_str, pConfigData);
                if(!err)
                {
                    pr_debug("configuration data was returned from bsd file \n");
                    return err;
                }
            }
        }
    }


    /* configuration should be return from the different data containers
     * return configuration type and go though the case handling
     */
    switch(ecnr_configuration_get_config_type(ConfigId))
    {
        case CONFIG_TYPE_FILE:
            err = ecnr_configuration_read_configuration_bsd_file(pdata->bsd_str, pConfigData);
            break;
        case CONFIG_TYPE_TABLE:
            err = ecnr_configuration_read_default_configuration(pdata, pConfigData);
            break;
        case CONFIG_TYPE_KDS:
        #ifdef D_KDS_SUPPORT
            err = ecnr_configuration_read_configuration_kds(pdata->EcnrKdsBaseKey, pConfigData);
        #else
            err = ECNR_ERR_OK;
        #endif
            break;
        case CONFIG_TYPE_DP:
            err = ecnr_configuration_read_configuration_dp(pdata->EcnrdPBaseKey, pConfigData);
            break;
        case CONFIG_TYPE_SET_FCT:
        default:
            pr_warning("no configuration available");
            err = ECNR_ERR_PARAM;
            break;
    }
#endif

goto end; /* added to suppress compiler warning just above end: label */
end:

    /* retrieve default configuration if the read from data source failed */
    if(err)
    {
        pr_message("configuration data read has failed with error: %d, try retrieve default configuration", err);
        err = ecnr_configuration_read_default_configuration(pdata, pConfigData);
        if(err == ECNR_ERR_NO_DEFAULT_CONFIGURATION)
        {
            pr_debug("no default configuration available\n");
        }
        else
        {
            *isdefault = TRUE;
        }
    }

    pr_debug("EXIT\n");
    return err;
}



/*******************************************************************************
*
* FUNCTION: ecnr_configuration_init
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_configuration_init(void)
{
    int    err = ECNR_ERR_OK;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_configuration_init]: ENTERED"));


#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    pr_debug("datapool not supported\n");
    pr_debug("EXIT\n");

    return err;
#endif // VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX

    bNewdpBTPhoneNB  = FALSE;
    bNewdpBTPhoneWB  = FALSE;
    bNewdpBTVoiceNB  = FALSE;
    bNewdpBTVoiceWB  = FALSE;
    bNewdpBTSiriNB   = FALSE;
    bNewdpBTSiriWB   = FALSE;
    bNewdpCPVoiceNB  = FALSE;
    bNewdpCPVoiceWB  = FALSE;
    bNewdpCPVoiceSWB = FALSE;
    bNewdpCPPhoneNB  = FALSE;
    bNewdpCPPhoneWB  = FALSE;
    bNewdpCPPhoneSWB = FALSE;
    bNewdpCPFTimeSWB = FALSE;
    bNewdpAAVoiceNB  = FALSE;
    bNewdpAAVoiceWB  = FALSE;
    bNewdpAAVoiceSWB = FALSE;
    bNewdpAAPhoneNB  = FALSE;
    bNewdpAAPhoneWB  = FALSE;
    bNewdpAAPhoneSWB = FALSE;
    bNewdpINTVoiceNB = FALSE;
    bNewdpINTVoiceWB = FALSE;
    bNewdpWUWVoiceWB = FALSE;

    /* walk though the project configuration
     * registry the DP element for change notification
     */

    ecnr_datapool_access_init();

#ifdef D_DP_NOTIFICATION_SUPPORT
    ecnr_datapool_access_reg_notification(ecnr_configuration_notify_new_configuration_phone_nb, ECNR_DP_BT_PHONE_NB);
    ecnr_datapool_access_reg_notification(ecnr_configuration_notify_new_configuration_phone_wb, ECNR_DP_BT_PHONE_WB);
    ecnr_datapool_access_reg_notification(ecnr_configuration_notify_new_configuration_voice_nb, ECNR_DP_BT_VOICE_NB);
    ecnr_datapool_access_reg_notification(ecnr_configuration_notify_new_configuration_voice_wb, ECNR_DP_BT_VOICE_WB);
    ecnr_datapool_access_reg_notification(ecnr_configuration_notify_new_configuration_cp_phone_nb, ECNR_DP_CP_PHONE_NB);
    ecnr_datapool_access_reg_notification(ecnr_configuration_notify_new_configuration_cp_phone_wb, ECNR_DP_CP_PHONE_WB);
    ecnr_datapool_access_reg_notification(ecnr_configuration_notify_new_configuration_cp_voice_wb, ECNR_DP_CP_VOICE_WB);
#endif //D_DP_NOTIFICATION_SUPPORT

    /* depend on product configuration, the configuration file will be removed from RFS */
#ifndef D_INCREASE_ECNR_START_UP_PERFORMANCE
    ecnr_configuration_check_config_files();
#endif
    pr_message("EXIT\n");
    ETG_TRACE_USR3(("[ecnr_configuration_init]: EXIT"));

    return err;
}


#ifdef D_DP_NOTIFICATION_SUPPORT
/* test functions */
int ecnr_configuration_stub_notify_dp_change(void)
{
    int    err = ECNR_ERR_OK;

    pr_debug("ENTERED\n");

    ecnr_configuration_notify_new_configuration_phone_nb();
    ecnr_configuration_notify_new_configuration_phone_wb();
    ecnr_configuration_notify_new_configuration_voice_nb();
    ecnr_configuration_notify_new_configuration_voice_wb();
    ecnr_configuration_notify_new_configuration_cp_phone_nb();
    ecnr_configuration_notify_new_configuration_cp_phone_wb();

    pr_debug("EXIT\n");

    return err;
}
#endif

static const struct ecnr_rts_dev_cfg* ecnr_configuration_search_config_table_input(enum ecnr_audio ecnr_dev_id)
{
    guint i;
    const struct ecnr_rts_dev_cfg* p0 = (struct ecnr_rts_dev_cfg*)NULL;
    const struct ecnr_rts_dev_cfg* p1 = (struct ecnr_rts_dev_cfg*)NULL;

    for (i = 0 ; i < DEV_CFG_MAX ; i++)
    {
        p0 = &ECNR_RTS_DEV_CONF_table[i];

        /* retrieve the device configuration with the default device
         * if there are  multiple definition
         */
        if(    p1
            && p0->rts_dev == ecnr_dev_id
            && p0->is_default_dev)
        {
            pr_debug("retrieve default device setting for devId:%d", ecnr_dev_id);
            p1 = &ECNR_RTS_DEV_CONF_table[i];
            return p1;
        }

        if (p0->rts_dev == ecnr_dev_id)
        {
            p1 = &ECNR_RTS_DEV_CONF_table[i];
        }
    }

    if (!p1)
    {
        pr_debug("no configuration available with dev_id %d\n",ecnr_dev_id);
    }

    return p1;
}


/*******************************************************************************
*
* FUNCTION: ecnr_configuration_get_default_dev_name
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
gchar *ecnr_configuration_get_default_dev_name(enum ecnr_audio ecnr_dev_id, gboolean* is_default_dev)
{
    gchar* dev = (gchar*)NULL;
    const struct ecnr_rts_dev_cfg* p = (struct ecnr_rts_dev_cfg*)NULL;

    pr_debug("dev Id: %d\n",ecnr_dev_id);

    if(ecnr_dev_id >= NUM_ECNR_AUDIO)
    {
        pr_warning("dev Id out of range (max = %d) : %d\n", NUM_ECNR_AUDIO, ecnr_dev_id);
        return dev;
    }

    p = ecnr_configuration_search_config_table_input(ecnr_dev_id);

    if(!p)
    {
        pr_debug("no configuration input for dev Id : %d\n",ecnr_dev_id);
        return dev;
    }

    if(p->adev_name)
    {
        dev = g_strdup((char*)p->adev_name);
        *is_default_dev = p->is_default_dev;
    }
    else
        *is_default_dev = FALSE;

    return dev;

}

/*******************************************************************************
*
* FUNCTION: ecnr_configuration_get_default_num_channels
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
unsigned int ecnr_configuration_get_default_num_channels(enum ecnr_audio ecnr_dev_id)
{
    const struct ecnr_rts_dev_cfg* p = (struct ecnr_rts_dev_cfg*)NULL;

    if(ecnr_dev_id >= NUM_ECNR_AUDIO)
    {
        pr_warning("dev Id out of range (max = %d) : %d\n", NUM_ECNR_AUDIO, ecnr_dev_id);
        return 0;
    }

    p = ecnr_configuration_search_config_table_input(ecnr_dev_id);

    if(!p)
    {
        pr_debug("no configuration input for dev Id : %d\n",ecnr_dev_id);
        return 0;
    }

    return p->num_channel;

}

/*******************************************************************************
*
* FUNCTION: ecnr_configuration_get_default_active_channel
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
unsigned int ecnr_configuration_get_default_active_channel(enum ecnr_audio ecnr_dev_id)
{
    const struct ecnr_rts_dev_cfg* p = (struct ecnr_rts_dev_cfg*)NULL;

    if(ecnr_dev_id >= NUM_ECNR_AUDIO)
    {
        pr_warning("dev Id out of range (max = %d) : %d\n", NUM_ECNR_AUDIO, ecnr_dev_id);
        return 0;
    }

    p = ecnr_configuration_search_config_table_input(ecnr_dev_id);

    if(!p)
    {
        pr_debug("no configuration input for dev Id : %d\n",ecnr_dev_id);
        return 0;
    }

    return p->active_channel;

}

/*******************************************************************************
*
* FUNCTION: ecnr_configuration_get_default_dev_open_order
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
unsigned int ecnr_configuration_get_default_dev_open_order(enum ecnr_audio ecnr_dev_id)
{
    const struct ecnr_rts_dev_cfg* p = (struct ecnr_rts_dev_cfg*)NULL;

    if(ecnr_dev_id >= NUM_ECNR_AUDIO)
    {
        pr_warning("dev Id out of range (max = %d) : %d\n", NUM_ECNR_AUDIO, ecnr_dev_id);
        return 0;
    }

    p = ecnr_configuration_search_config_table_input(ecnr_dev_id);

    if(!p)
    {
        pr_debug("no configuration input for dev Id : %d\n",ecnr_dev_id);
        return 0;
    }

    return p->dev_open_order;
}


static  snd_pcm_stream_t ecnr_configuration_get_stream_dir(enum ecnr_audio dev_ch)
{
    snd_pcm_stream_t stream = SND_PCM_STREAM_PLAYBACK;


    switch(dev_ch)
    {
        case ECNR_MIC_IN0:
        case ECNR_MIC_IN1:
        case ECNR_MULTI_CHAN_IN_MIC_0:
        case ECNR_MULTI_CHAN_IN_MIC_1:
        case ECNR_REF_IN0:
        case ECNR_REF_IN1:
        case ECNR_MULTI_CHAN_IN_REF_0:
        case ECNR_MULTI_CHAN_IN_REF_1:
        case ECNR_MULTI_CHAN_IN_DEV:
        case ECNR_RECV_IN:
            stream = SND_PCM_STREAM_CAPTURE;
            break;

        case ECNR_MIC_OUT:
        case ECNR_RECV_OUT:
            stream = SND_PCM_STREAM_PLAYBACK;
            break;
        default:
            break;
    }

    pr_debug("stream direction(%d) for device channel(%d)\n", stream, dev_ch);

    return stream;
}






/*******************************************************************************
*
* FUNCTION: ecnr_configuration_get_dev_id
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
enum ecnr_audio ecnr_configuration_get_dev_id(gchar * dev, snd_pcm_stream_t sdir)
{
    guint i;
    const struct ecnr_rts_dev_cfg* p = (struct ecnr_rts_dev_cfg*)NULL;
    snd_pcm_stream_t dir = SND_PCM_STREAM_PLAYBACK;

    pr_debug("device %s with stream direction(%d)\n", dev , sdir);

    for (i = 0 ; i < DEV_CFG_MAX ; i++)
    {
        p =  &ECNR_RTS_DEV_CONF_table[i];
        if (!g_strcmp0((char*)p->adev_name, dev))
        {
            dir = ecnr_configuration_get_stream_dir(p->rts_dev);
            if( sdir == dir )
            {
                pr_debug("set %s device %s to channel (%d)\n",(dir == SND_PCM_STREAM_CAPTURE)? "capture":"playback",dev, p->rts_dev );
                return p->rts_dev;
            }
         }
    }

    pr_warning("no configuration available\n");

    return NUM_ECNR_AUDIO;
}



/*******************************************************************************
*
* FUNCTION: ecnr_configuration_check_match_dev_chid
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_configuration_check_match_dev_chid(enum ecnr_audio dev_id, gchar * dev)
{
    guint i;
    const struct ecnr_rts_dev_cfg* p = (struct ecnr_rts_dev_cfg*)NULL;

    pr_debug("search configuration device (%s) to ecnr audio channel(%d)\n", dev , dev_id);

    for (i = 0 ; i < DEV_CFG_MAX ; i++)
    {
        p =  &ECNR_RTS_DEV_CONF_table[i];
        if (   !g_strcmp0((char*)p->adev_name, dev)
             && dev_id == p->rts_dev
            )
        {
            pr_debug("found configuration device (%s) in ecnr channel (%d)\n",dev, dev_id);
            return ECNR_ERR_OK;
        }
    }
    pr_warning("no configuration available\n");

    return ECNR_ERR_DEV_CHNID_NOT_MATCHING;
}


/*******************************************************************************
*
* FUNCTION: ecnr_configuration_retrieve_all_dp_size
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_configuration_retrieve_all_dp_size(void)
{
    int err = ECNR_ERR_OK;
#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    guint16 size = 0;
    const struct EcnrConfigFile* pd = (struct EcnrConfigFile*)NULL;

    //while (ECNR_CONFIG_FILE_table[ecnr_dp_dyn_nb].file != NULL && (ecnr_dp_dyn_nb < NB_ELEM_MAX) )
    for(pd = ECNR_CONFIG_FILE_table; pd->file != NULL; pd++)
    {
        pr_debug("retrieved dyn size of dp(%d)\n", pd->EcnrdPBaseKey);

        size = (guint16) ecnr_datapool_access_get_bsd_dataset_size(pd->EcnrdPBaseKey);

        if(size)
            ecnr_datapool_access_set_dp_size_dyn(pd->EcnrdPBaseKey, (int)size);
        else
            ecnr_datapool_access_set_dp_size_dyn(pd->EcnrdPBaseKey, (int)0);

    }
#endif
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_configuration_get_prefill
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
guint ecnr_configuration_get_prefill(unsigned short config_id)
{
#if (FEATURE_CFG_MAX == 0)
    pr_debug("No feature setting available\n");
    return ECNR_ERR_OK;
#else
    const struct ecnr_id_spec_feature   *p = (struct ecnr_id_spec_feature*)NULL;
    struct ecnr_feature_cfg             *pfc = (struct ecnr_feature_cfg*)NULL;
    int                                 i = 0;
    guint                               config_prefill_value = 0;

    pr_debug("retrieve configured prefill for the config id -> %d\n", config_id );

    if(config_id == 0)
        return ECNR_ERR_CONFID_NOT_AVAILABLE;


    for( i = 0; i < FEATURE_CFG_MAX; i++)
    {
        p = (struct ecnr_id_spec_feature*)&ECNR_FEATURE_CONF_table[i];

        if(p->config_id == config_id)
        {
            pr_debug("configuratino match for the config id -> %d\n", config_id);
            pfc = (struct ecnr_feature_cfg*)&(p->feat_cfg_table);
            if(pfc->is_preFill)
            {
                if(!pfc->prefill_value)
                    return ECNR_ERR_VALUE_RANGE_FAILURE;

                /* enable setting of pre-fill  */
                pr_debug("configured prefill value -> %d\n", pfc->prefill_value);

                config_prefill_value = pfc->prefill_value;
                break;
            }
            else
                pr_debug("set prefill feature is disabled\n");
        }
    }

    return config_prefill_value;
#endif
}

/*******************************************************************************
*
* FUNCTION: ecnr_configuration_get_period
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
#ifdef D_SUPPORT_PERIOD_PARAMETER
guint ecnr_configuration_get_period(unsigned short config_id)
{
#if (FEATURE_CFG_MAX == 0)
    pr_debug("No feature setting available\n");
    return ECNR_ERR_OK;
#else
    const struct ecnr_id_spec_feature   *p = (struct ecnr_id_spec_feature*)NULL;
    struct ecnr_feature_cfg             *pfc = (struct ecnr_feature_cfg*)NULL;
    int                                 i = 0;
    guint                               config_period_value = 0;

    pr_debug("retrieve configured period for the config id -> %d\n", config_id );

    if(config_id == 0)
        return ECNR_ERR_CONFID_NOT_AVAILABLE;


    for( i = 0; i < FEATURE_CFG_MAX; i++)
    {
        p = (struct ecnr_id_spec_feature*)&ECNR_FEATURE_CONF_table[i];

        if(p->config_id == config_id)
        {
            pr_debug("configuration matches with the config id -> %d\n", config_id);
            pfc = (struct ecnr_feature_cfg*)&(p->feat_cfg_table);


            if(!pfc->period_value)
                return ECNR_ERR_VALUE_RANGE_FAILURE;

            /* enable setting of pre-fill  */
            pr_debug("configured period value -> %d\n", pfc->period_value);

            config_period_value = pfc->period_value;

            break;
        }
    }

    return config_period_value;
#endif
}
#endif
/***********************************************************************
 *  definition test function for unitest suites
 ***********************************************************************/


#if  !defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX) &&  !defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_LSIM)

gboolean ecnr_configuration_ut_s2_1_1(void)
{
    pr_message("START_UNITEST_S2_1_1");


    gboolean bRet_2_1_1 = FALSE;

    /* test ECNR_DP_BT_PHONE_NB datapool */
    ecnr_configuration_set_crc(ECNR_DP_BT_PHONE_NB,0xFFFF);
    bRet_2_1_1 = ecnr_configuration_compare_crc_dp_file(ECNR_DP_BT_PHONE_NB);
    g_assert(bRet_2_1_1 == FALSE);

    gboolean bRet_2_1_2 = FALSE;
    guint16 crc_2_1_2 = 0x0000;

    bRet_2_1_2 = ecnr_configuration_get_crc(ECNR_DP_BT_PHONE_NB, &crc_2_1_2);
    pr_message("value crc_2_1_2(0x%x) of dp(ECNR_DP_BT_PHONE_NB)", crc_2_1_2);
    g_assert(bRet_2_1_2 == TRUE);
    g_assert(crc_2_1_2 == 0xFFFF);

    const struct EcnrConfig *pdata = NULL;
    gboolean bRet_2_1_3 = FALSE;
    pdata = ecnr_configuration_get_config_data(PHONE_DATASET_1);
    (void)ecnr_configuration_create_bsd_files_from_dp(pdata);
    bRet_2_1_3 = ecnr_configuration_compare_crc_dp_file(ECNR_DP_BT_PHONE_NB);
    g_assert(bRet_2_1_3 == TRUE);

    gboolean bRet_2_1_4 = FALSE;
    guint16 crc_2_1_4 = 0x0000;

    bRet_2_1_4 = ecnr_configuration_get_crc(ECNR_DP_BT_PHONE_NB, &crc_2_1_4);
    pr_message("value crc_2_1_4(0x%x) of dp(ECNR_DP_BT_PHONE_NB)", crc_2_1_4);
    g_assert(bRet_2_1_4 == TRUE);
    g_assert(crc_2_1_4 != 0x0000);

    pr_message("SUCCESS_UNITEST_S2_1_1");

    return TRUE;

}


gboolean ecnr_configuration_ut_s2_1_2(void)
{
    pr_message("START_UNITEST_S2_1_2");


    gboolean bRet_2_1_1 = FALSE;

    /* test ECNR_DP_BT_PHONE_WB datapool */
    ecnr_configuration_set_crc(ECNR_DP_BT_PHONE_WB,0xFFFF);
    bRet_2_1_1 = ecnr_configuration_compare_crc_dp_file(ECNR_DP_BT_PHONE_WB);
    g_assert(bRet_2_1_1 == FALSE);

    gboolean bRet_2_1_2 = FALSE;
    guint16 crc_2_1_2 = 0x0000;

    bRet_2_1_2 = ecnr_configuration_get_crc(ECNR_DP_BT_PHONE_WB, &crc_2_1_2);
    pr_message("value crc_2_1_2(0x%x) of dp(ECNR_DP_BT_PHONE_WB)", crc_2_1_2);
    g_assert(bRet_2_1_2 == TRUE);
    g_assert(crc_2_1_2 == 0xFFFF);

    const struct EcnrConfig *pdata = NULL;
    gboolean bRet_2_1_3 = FALSE;
    pdata = ecnr_configuration_get_config_data(PHONE_DATASET_2);
    (void)ecnr_configuration_create_bsd_files_from_dp(pdata);
    bRet_2_1_3 = ecnr_configuration_compare_crc_dp_file(ECNR_DP_BT_PHONE_WB);
    g_assert(bRet_2_1_3 == TRUE);

    gboolean bRet_2_1_4 = FALSE;
    guint16 crc_2_1_4 = 0x0000;

    bRet_2_1_4 = ecnr_configuration_get_crc(ECNR_DP_BT_PHONE_WB, &crc_2_1_4);
    pr_message("value crc_2_1_4(0x%x) of dp(ECNR_DP_BT_PHONE_WB)", crc_2_1_4);
    g_assert(bRet_2_1_4 == TRUE);
    g_assert(crc_2_1_4 != 0x0000);

    pr_message("SUCCESS_UNITEST_S2_1_2");

    return TRUE;

}

gboolean ecnr_configuration_ut_s2_1_3(void)
{
    pr_message("START_UNITEST_S2_1_3");


    gboolean bRet_2_1_1 = FALSE;

    /* test ECNR_DP_BT_VOICE_NB datapool */
    ecnr_configuration_set_crc(ECNR_DP_BT_VOICE_NB,0xFFFF);
    bRet_2_1_1 = ecnr_configuration_compare_crc_dp_file(ECNR_DP_BT_VOICE_NB);
    g_assert(bRet_2_1_1 == FALSE);

    gboolean bRet_2_1_2 = FALSE;
    guint16 crc_2_1_2 = 0x0000;

    bRet_2_1_2 = ecnr_configuration_get_crc(ECNR_DP_BT_VOICE_NB, &crc_2_1_2);
    pr_message("value crc_2_1_2(0x%x) of dp(ECNR_DP_BT_VOICE_NB)", crc_2_1_2);
    g_assert(bRet_2_1_2 == TRUE);
    g_assert(crc_2_1_2 == 0xFFFF);

    const struct EcnrConfig *pdata = NULL;
    gboolean bRet_2_1_3 = FALSE;
    pdata = ecnr_configuration_get_config_data(PHONE_DATASET_3);
    (void)ecnr_configuration_create_bsd_files_from_dp(pdata);
    bRet_2_1_3 = ecnr_configuration_compare_crc_dp_file(ECNR_DP_BT_VOICE_NB);
    g_assert(bRet_2_1_3 == TRUE);

    gboolean bRet_2_1_4 = FALSE;
    guint16 crc_2_1_4 = 0x0000;

    bRet_2_1_4 = ecnr_configuration_get_crc(ECNR_DP_BT_VOICE_NB, &crc_2_1_4);
    pr_message("value crc_2_1_4(0x%x) of dp(ECNR_DP_BT_VOICE_NB)", crc_2_1_4);
    g_assert(bRet_2_1_4 == TRUE);
    g_assert(crc_2_1_4 != 0x0000);

    pr_message("SUCCESS_UNITEST_S2_1_3");

    return TRUE;

}

gboolean ecnr_configuration_ut_s2_1_4(void)
{
    pr_message("START_UNITEST_S2_1_4");


    gboolean bRet_2_1_1 = FALSE;

    /* test ECNR_DP_BT_VOICE_WB datapool */
    ecnr_configuration_set_crc(ECNR_DP_BT_VOICE_WB,0xFFFF);
    bRet_2_1_1 = ecnr_configuration_compare_crc_dp_file(ECNR_DP_BT_VOICE_WB);
    g_assert(bRet_2_1_1 == FALSE);

    gboolean bRet_2_1_2 = FALSE;
    guint16 crc_2_1_2 = 0x0000;

    bRet_2_1_2 = ecnr_configuration_get_crc(ECNR_DP_BT_VOICE_WB, &crc_2_1_2);
    pr_message("value crc_2_1_2(0x%x) of dp(ECNR_DP_BT_VOICE_WB)", crc_2_1_2);
    g_assert(bRet_2_1_2 == TRUE);
    g_assert(crc_2_1_2 == 0xFFFF);

    const struct EcnrConfig *pdata = NULL;
    gboolean bRet_2_1_3 = FALSE;
    pdata = ecnr_configuration_get_config_data(PHONE_DATASET_4);
    (void)ecnr_configuration_create_bsd_files_from_dp(pdata);
    bRet_2_1_3 = ecnr_configuration_compare_crc_dp_file(ECNR_DP_BT_VOICE_WB);
    g_assert(bRet_2_1_3 == TRUE);

    gboolean bRet_2_1_4 = FALSE;
    guint16 crc_2_1_4 = 0x0000;

    bRet_2_1_4 = ecnr_configuration_get_crc(ECNR_DP_BT_VOICE_WB, &crc_2_1_4);
    pr_message("value crc_2_1_4(0x%x) of dp(ECNR_DP_BT_VOICE_WB)", crc_2_1_4);
    g_assert(bRet_2_1_4 == TRUE);
    g_assert(crc_2_1_4 != 0x0000);

    pr_message("SUCCESS_UNITEST_S2_1_4");

    return TRUE;

}

gboolean ecnr_configuration_ut_s2_1_5(void)
{
    pr_message("START_UNITEST_S2_1_5");


    gboolean bRet_2_1_1 = FALSE;

    /* test ECNR_DP_CP_PHONE_NB datapool */
    ecnr_configuration_set_crc(ECNR_DP_CP_PHONE_NB,0xFFFF);
    bRet_2_1_1 = ecnr_configuration_compare_crc_dp_file(ECNR_DP_CP_PHONE_NB);
    g_assert(bRet_2_1_1 == FALSE);

    gboolean bRet_2_1_2 = FALSE;
    guint16 crc_2_1_2 = 0x0000;

    bRet_2_1_2 = ecnr_configuration_get_crc(ECNR_DP_CP_PHONE_NB, &crc_2_1_2);
    pr_message("value crc_2_1_2(0x%x) of dp(ECNR_DP_CP_PHONE_NB)", crc_2_1_2);
    g_assert(bRet_2_1_2 == TRUE);
    g_assert(crc_2_1_2 == 0xFFFF);

    const struct EcnrConfig *pdata = NULL;
    gboolean bRet_2_1_3 = FALSE;
    pdata = ecnr_configuration_get_config_data(SPI_DATASET_1);
    (void)ecnr_configuration_create_bsd_files_from_dp(pdata);
    bRet_2_1_3 = ecnr_configuration_compare_crc_dp_file(ECNR_DP_CP_PHONE_NB);
    g_assert(bRet_2_1_3 == TRUE);

    gboolean bRet_2_1_4 = FALSE;
    guint16 crc_2_1_4 = 0x0000;

    bRet_2_1_4 = ecnr_configuration_get_crc(ECNR_DP_CP_PHONE_NB, &crc_2_1_4);
    pr_message("value crc_2_1_4(0x%x) of dp(ECNR_DP_CP_PHONE_NB)", crc_2_1_4);
    g_assert(bRet_2_1_4 == TRUE);
    g_assert(crc_2_1_4 != 0x0000);

    pr_message("SUCCESS_UNITEST_S2_1_5");

    return TRUE;

}

gboolean ecnr_configuration_ut_s2_1_6(void)
{
    pr_message("START_UNITEST_S2_1_6");


    gboolean bRet_2_1_1 = FALSE;

    /* test ECNR_DP_CP_PHONE_WB datapool */
    ecnr_configuration_set_crc(ECNR_DP_CP_PHONE_WB,0xFFFF);
    bRet_2_1_1 = ecnr_configuration_compare_crc_dp_file(ECNR_DP_CP_PHONE_WB);
    g_assert(bRet_2_1_1 == FALSE);

    gboolean bRet_2_1_2 = FALSE;
    guint16 crc_2_1_2 = 0x0000;

    bRet_2_1_2 = ecnr_configuration_get_crc(ECNR_DP_CP_PHONE_WB, &crc_2_1_2);
    pr_message("value crc_2_1_2(0x%x) of dp(ECNR_DP_CP_PHONE_WB)", crc_2_1_2);
    g_assert(bRet_2_1_2 == TRUE);
    g_assert(crc_2_1_2 == 0xFFFF);

    const struct EcnrConfig *pdata = NULL;
    gboolean bRet_2_1_3 = FALSE;
    pdata = ecnr_configuration_get_config_data(SPI_DATASET_2);
    (void)ecnr_configuration_create_bsd_files_from_dp(pdata);
    bRet_2_1_3 = ecnr_configuration_compare_crc_dp_file(ECNR_DP_CP_PHONE_WB);
    g_assert(bRet_2_1_3 == TRUE);

    gboolean bRet_2_1_4 = FALSE;
    guint16 crc_2_1_4 = 0x0000;

    bRet_2_1_4 = ecnr_configuration_get_crc(ECNR_DP_CP_PHONE_WB, &crc_2_1_4);
    pr_message("value crc_2_1_4(0x%x) of dp(ECNR_DP_CP_PHONE_WB)", crc_2_1_4);
    g_assert(bRet_2_1_4 == TRUE);
    g_assert(crc_2_1_4 != 0x0000);

    pr_message("SUCCESS_UNITEST_S2_1_6");

    return TRUE;

}

gboolean ecnr_configuration_ut_s2_1_7(void)
{
    pr_message("START_UNITEST_S2_1_7");


    gboolean bRet_2_1_1 = FALSE;

    /* test ECNR_DP_CP_VOICE_WB datapool */
    ecnr_configuration_set_crc(ECNR_DP_CP_VOICE_WB,0xFFFF);
    bRet_2_1_1 = ecnr_configuration_compare_crc_dp_file(ECNR_DP_CP_VOICE_WB);
    g_assert(bRet_2_1_1 == FALSE);

    gboolean bRet_2_1_2 = FALSE;
    guint16 crc_2_1_2 = 0x0000;

    bRet_2_1_2 = ecnr_configuration_get_crc(ECNR_DP_CP_VOICE_WB, &crc_2_1_2);
    pr_message("value crc_2_1_2(0x%x) of dp(ECNR_DP_CP_VOICE_WB)", crc_2_1_2);
    g_assert(bRet_2_1_2 == TRUE);
    g_assert(crc_2_1_2 == 0xFFFF);

    const struct EcnrConfig *pdata = NULL;
    gboolean bRet_2_1_3 = FALSE;
    pdata = ecnr_configuration_get_config_data(SPI_DATASET_4);
    (void)ecnr_configuration_create_bsd_files_from_dp(pdata);
    bRet_2_1_3 = ecnr_configuration_compare_crc_dp_file(ECNR_DP_CP_VOICE_WB);
    g_assert(bRet_2_1_3 == TRUE);

    gboolean bRet_2_1_4 = FALSE;
    guint16 crc_2_1_4 = 0x0000;

    bRet_2_1_4 = ecnr_configuration_get_crc(ECNR_DP_CP_VOICE_WB, &crc_2_1_4);
    pr_message("value crc_2_1_4(0x%x) of dp(ECNR_DP_CP_VOICE_WB)", crc_2_1_4);
    g_assert(bRet_2_1_4 == TRUE);
    g_assert(crc_2_1_4 != 0x0000);

    pr_message("SUCCESS_UNITEST_S2_1_7");

    return TRUE;

}
#endif // #ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
