/******************************************************************************
 * FILE        : ecnr-audio-rts-handler.c
 * PROJECT     : Gen3 and Gen4
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2014 RBCM GMBH
 * HISTORY     :
 * Date        | Author                 | Modification
 * 06.01.2014  | Patrick Rey            | initial version
 *             |                        | RTS library handling features
 *****************************************************************************/




/*******************************************************************************
              DEFINES
*******************************************************************************/


/*******************************************************************************
              GENERATED CODE HEADERS
*******************************************************************************/

#include <glib-unix.h>
#include <glib/gi18n.h>
#include <glib-object.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include <alsa/asoundlib.h>

#include "ecnr-error.h"
#include "ecnr-common-defs.h"
#include "ecnr-service.h"
#include "ecnr-alsa.h"
#include "ecnr-audio-recording.h"
#include "ecnr-audio-rts-handler.h"



/* Needed for Trace */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define SPM_UNIT_TESTS // solve warning
#include "ecnr-trace.h"
#include "etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ECNR_RTS
#include "trcGenProj/Header/ecnr-audio-rts-handler.c.trc.h"
#endif

/*******************************************************************************
              STRUCTURE DEFINITION
*******************************************************************************/

/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/

static  tRTS            rts_handle = NULL;
static  int             rts_state = 0;
static  trts_cfg*       rts_cfg = (trts_cfg*)NULL ;
static  int             rts_read_cnt;
static  struct timespec ecnr_audio_rts_last_ok_read_time = {0};
static  int             rts_write_cnt;
static  gboolean        rts_read_err_occured = FALSE;
static  gboolean        rts_write_err_occured = FALSE;



/**********************************************
 * RTS TEST FUNCTIONS - TOP
 ***********************************************/
#define D_NB_TEST_STEP_MAX      100
#define S_ECNR_RTS_TEST "/var/opt/bosch/dynamic/audiomanager/ecnr/ecnr_rts_performance_read.txt"

typedef struct _rts_time_test
{
 /* guint32 fct_call_exec_time;
    struct timespec fct_call_entry_time;
    struct timespec fct_call_exit_time; */ //remove comment on usage of struct variables.
    struct timespec fct_call_complete_time;
}rts_time_test;


static rts_time_test* ptstub = (rts_time_test*)NULL;
static gint ecnr_audio_rts_test_next_step = 0;
static gint ecnr_audio_rts_nb_step = 0;
static GString *test_fct_name;
static gint ecnr_audio_rts_handler_read_perform_avail = 0;
static gboolean m_ecnr_audio_rts_handler_performance_enable = FALSE;



static void ecnr_audio_rts_handler_performance_test_init(gchar* fct_name_to_test, gint nb_test_step)
{

    if(fct_name_to_test)
    {
        pr_message("TEST ENTER for function %s with %d step\n", fct_name_to_test, nb_test_step);
    }

    if(!ptstub)
    {
        if(!nb_test_step)
            ecnr_audio_rts_nb_step = D_NB_TEST_STEP_MAX;
        else
            ecnr_audio_rts_nb_step = nb_test_step;

        ptstub = g_malloc0( (size_t)ecnr_audio_rts_nb_step * sizeof(rts_time_test));
    }

    test_fct_name = g_string_new (NULL);
    if(fct_name_to_test)
        g_string_append(test_fct_name, fct_name_to_test);
    ecnr_audio_rts_test_next_step = 0;

    return;
}

static void ecnr_audio_rts_handler_performance_test_terminate(void)
{
    rts_time_test* pd = (rts_time_test*)NULL;
    rts_time_test* pn = (rts_time_test*)NULL;
    gint i = 0;
    glong exec_time = 0;
    int ret;


    /* create a bsd file with dp element */
    FILE*   fd_bsd;   /* pointer to config data file */


    ret =  remove(S_ECNR_RTS_TEST);
    if (ret != ECNR_ERR_OK)
    {
        pr_debug("ecnr_rts_performance_read.txt removal failed = %d", ret);
    }

    fd_bsd = fopen (S_ECNR_RTS_TEST,"a+");

    if(test_fct_name->str)
        pr_message("TEST EXIT for function %s with %d step\n", test_fct_name->str, ecnr_audio_rts_test_next_step);

    /* evaluate all result and write in a file */
    if(!ptstub   || !ecnr_audio_rts_test_next_step)
    {
        pr_message("No Result available\n");
        if (NULL != fd_bsd)
        {
            fclose(fd_bsd);
        }
        return;
    }

    for(i = 0; i < ecnr_audio_rts_test_next_step; i++)
    {

        if(i == ecnr_audio_rts_test_next_step)
        {
            pr_message("terminate test with %d step \n", ecnr_audio_rts_test_next_step);
            break;
        }
        pd = &ptstub[i];
        pn = pd;


        if(fd_bsd)
        {
            fprintf (fd_bsd,"%d %s COMPLETE time : [%05u,%.9lu]\n", (i+1), (test_fct_name->str)? test_fct_name->str :"NULL", (unsigned int)pd->fct_call_complete_time.tv_sec, (unsigned long int)pd->fct_call_complete_time.tv_nsec);
        }

        if( (i + 2) <  ecnr_audio_rts_test_next_step)
        {
            pn++;

            if(pd->fct_call_complete_time.tv_sec == pn->fct_call_complete_time.tv_sec)
            {
                exec_time = pn->fct_call_complete_time.tv_nsec - pd->fct_call_complete_time.tv_nsec;
            }
            else /* next second*/
            {
                exec_time = (1000000000 - pd->fct_call_complete_time.tv_nsec) + pn->fct_call_complete_time.tv_nsec;
            }

            if(fd_bsd)
                fprintf (fd_bsd,"%d %s call EXEC time : [%lu]\n", (i+1), (test_fct_name->str)? test_fct_name->str :"NULL", exec_time);
        }
    }

    if (NULL != fd_bsd)
    {
        fclose(fd_bsd);
    }

    g_string_free(test_fct_name, TRUE);
    if(ptstub)
    {
        g_free(ptstub);
        ptstub = NULL;
    }

    return;
}

inline static void ecnr_audio_rts_handler_performance_read_complet_timestamp(void)
{

    if(!ptstub  || (ecnr_audio_rts_test_next_step >= ecnr_audio_rts_nb_step))
        return;

    rts_time_test* p = &ptstub[ecnr_audio_rts_test_next_step++];

    clock_gettime(CLOCK_MONOTONIC, &p->fct_call_complete_time);
    return;
}

void ecnr_audio_rts_handler_performance_enable(gboolean state)
{
    m_ecnr_audio_rts_handler_performance_enable = state;
}


/**********************************************
 * RTS TEST FUNCTIONS - END
 ***********************************************/



static gboolean ecnr_audio_rts_set_buffer_size_dev_param(trts_cfg* pdata, const char* searchStr)
{
    gboolean update = FALSE;
    trts_cfgadev*  pdv = (trts_cfgadev*)NULL;
    guint i = 0 ; //modified for lint fix lint574


    if(!pdata || !searchStr)
        return FALSE;

    if(!rts_cfg)
        rts_cfg = g_malloc0(sizeof(trts_cfg));

    if(rts_cfg)
    {
        GString *str1, *str2;


        for(i = 0, pdv = pdata->adevs; i < pdata->num_adevs; pdv++, i++)
        {
            if(pdv->pcmname)
            {
                if(g_strrstr(pdv->pcmname, searchStr))
                {

                    guint buffer_nb = 2;
                    guint buffer_size = 0;
                    guint periode_size_ms = 0;


                    pr_message("pdv->period_frames(%d), pdv->rate(%d) , pdata->prefill_ms(%d)\n", pdv->period_frames, pdv->rate, pdata->prefill_ms);

                    periode_size_ms = (1000 * pdv->period_frames) / pdv->rate;

                    if(pdata->prefill_ms > periode_size_ms )
                        buffer_nb = 3;

                    /* calculated with the assumption the asr output SR is 24 khz */
                    //buffer_size = (periode_size_ms * (pdv->rate / 1000 )) * buffer_nb;
                    buffer_size = periode_size_ms * 24 * buffer_nb;

                    pr_message("buffer_nb(%d), buffer_size(%d) , periode_size_ms(%d)\n", buffer_nb, buffer_size, periode_size_ms);

                    /* found search device */


                    str1 = g_string_new(searchStr);
                    str2 = g_string_new(NULL);

                    //g_string_append(str1, "Ext");
                    g_string_printf (str2, "%s:%d", g_strdup(str1->str), buffer_size);

                    pr_message("change device name (%s) -> (%s)\n", pdv->pcmname, str2->str);

                    g_free((gpointer)pdv->pcmname);
                    pdv->pcmname = g_strdup(str2->str);

                    g_string_free(str1, TRUE);
                    g_string_free(str2, TRUE);

                    update = TRUE;

                }
            }
        }

        /* copy device configuration to rts local container */
        if(update)
            memcpy(rts_cfg, pdata,sizeof(trts_cfg));


    }

    return update;
}



/*************************
RTS log
**************************/

#ifdef D_ENABLE_RTS_LOG_CB

#define ECNR_AUDIO_RTS_LOG_LIMIT 200

static void ecnr_audio_rts_log_cb(const char* msg)
{
    if(!msg)
    {
        /* do nothing */
        return;
    }

    /* split message if bigger as ECNR_AUDIO_RTS_LOG_LIMIT  */
  gulong logLength = strlen(msg);
    GString *logmsg;

    while (logLength > ECNR_AUDIO_RTS_LOG_LIMIT)
    {
        int actualLimit = ECNR_AUDIO_RTS_LOG_LIMIT - 1;
        for (; actualLimit >= 0; actualLimit--)
        {
            if ((msg[actualLimit] == 0x0a) || msg[actualLimit] == 0x0d)
                break;
        }

        logmsg = g_string_new (NULL);

        if(actualLimit < 0)
            actualLimit = ECNR_AUDIO_RTS_LOG_LIMIT;

        g_string_append_len(logmsg, msg, actualLimit);

        ETG_TRACE_USR4(("[RTS_LIB_MSG]: %s", logmsg->str ));

        logLength = logLength - (gulong)actualLimit;

        msg = msg + actualLimit;
        g_string_free(logmsg, TRUE);
    }


    ETG_TRACE_USR4(("[RTS_LIB_MSG]: %s", msg ));

    return;
}
#endif



/*******************************************************************************
*
* FUNCTION: rts_dbug_config_data
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: void
*
*******************************************************************************/
void ecnr_audio_rts_handler_print_config_data(void)
{
    guint i =0;
    trts_cfgadev* d;
    trts_cfgstream* s;

    pr_debug("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_audio_rts_handler_print_config_data]: ENTERED"));

    if(rts_cfg)
    {
        pr_message("device num = %d, stream num = %d, prefill level= %d ",rts_cfg->num_adevs,rts_cfg->num_streams, rts_cfg->prefill_ms);
        ETG_TRACE_USR3(("[ecnr_audio_rts_handler_print_config_data]: device num = %d, stream num = %d, prefill level= %d ",rts_cfg->num_adevs,rts_cfg->num_streams, rts_cfg->prefill_ms));

        if(rts_cfg->adevs)
        {
            for (i=0 ; i < rts_cfg->num_adevs ; i++)
            {
                d = (trts_cfgadev*)&rts_cfg->adevs[i];

                #ifdef D_SET_INVIDUAL_PREFILL_PER_DEV
                pr_message("adevs[%d] :: dev name:%s, dir: %d, rate: %d, period_frames: %d, format:%d, startup_tout:%d, extra_periods:%d\n",
                                    i, d->pcmname,d->dir,d->rate,d->period_frames,d->format, d->startup_tout, d->extra_periods);

                #else
                pr_message("adevs[%d] :: dev name:%s, dir: %d, rate: %d, period_frames: %d, format:%d, startup_tout:%d\n",
                                    i, d->pcmname,d->dir,d->rate,d->period_frames,d->format, d->startup_tout);
                #endif


                ETG_TRACE_USR3(("[ecnr_audio_rts_handler_print_config_data]: adevs[%d] -> dir: %d, rate: %d, period_frames: %d, format:%d, startup_tout:%d -> \"%s\"",
                                    i,d->dir,d->rate,d->period_frames,d->format, d->startup_tout,d->pcmname));
            }

            for (i=0; i < rts_cfg->num_streams;i++)
            {
                s=&rts_cfg->streams[i];
                pr_message("streams[%d] :: adev Idx: %d, channel: %d", i, s->adevidx,s->channel);
                ETG_TRACE_USR3(("[ecnr_audio_rts_handler_print_config_data]: streams[%d] :: adev Idx: %d, channel: %d", i, s->adevidx,s->channel));
            }
        }
        else
        {
            pr_message("No rts device configuration available \n");
            ETG_TRACE_USR3(("[ecnr_audio_rts_handler_print_config_data]: No rts device configuration available \n"));
        }
    }
    else
    {
        pr_message("No rts configuration available \n");
        ETG_TRACE_USR3(("[ecnr_audio_rts_handler_print_config_data]: No rts configuration available \n"));
    }

    pr_debug("EXIT\n");
    ETG_TRACE_USR3(("[ecnr_audio_rts_handler_print_config_data]: EXIT"));
}



/*******************************************************************************
*
* FUNCTION: ecnr_audio_rts_handler_create
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: void
*
*******************************************************************************/
int ecnr_audio_rts_handler_create(stecnr_alsa_state* state, gboolean log)
{
    int err = ECNR_ERR_OK;
    gboolean update = FALSE;

    pr_debug("ENTERED");
    ETG_TRACE_USR3(("[ecnr_audio_rts_handler_create]: ENTERED"));

    if(!state)
    {
        pr_critical("state structure pointer not initialized\n");
        return ECNR_ERR_NULL_POINTER;
    }

    if(!state->rstdata)//st)
    {
        pr_warning("no structure rts structure initialized\n");
        ETG_TRACE_USR3(("[ecnr_audio_rts_handler_create]: no structure rts structure initialized"));
        return  ECNR_ERR_NULL_POINTER;
    }

    g_mutex_lock(&state->lock);
    if(rts_state == 1)
    {
        pr_warning("rts instance is already active");
        ETG_TRACE_USR4(("[ecnr_audio_rts_handler_create]: rts instance is already active"));
        err = ECNR_ERR_RTS_STATE_ERROR;
        goto error_out;
    }

    /* new : first free previous allocated memory if ecnr_audio_rts_handler_destroyed was not
       call prior to ecnr_audio_rts_handler_create
    */
    if(rts_cfg)
    {
        pr_warning("free rts data structur prior to rts_create call");
        g_free(rts_cfg);
        rts_cfg = NULL;
    }

    /* copy configuration structure internally */
    rts_cfg = g_malloc0(sizeof(trts_cfg));

    if(rts_cfg)
    {
        /* set Buffer size parameter by spi device */
        update = ecnr_audio_rts_set_buffer_size_dev_param(state->rstdata,"AdevECNRInOutExt");

        if(!update)
            memcpy(rts_cfg, state->rstdata,sizeof(trts_cfg));
    }
    else
    {
        err = ECNR_ERR_NULL_POINTER;
        goto error_out;
    }
    /* print in debug mode of the configuration data */
    if(log == TRUE)
        ecnr_audio_rts_handler_print_config_data();


    pr_message("call rts_create now\n");
    ETG_TRACE_USR3(("[ecnr_audio_rts_handler_create]: call rts_create now"));


    /* register rts log callback */
#ifdef D_ENABLE_RTS_LOG_CB
    rts_cfg->log_callback = ecnr_audio_rts_log_cb;
    ETG_TRACE_USR3(("[ecnr_audio_rts_handler_create]: registry RTS library Loggger callback function"));
#endif
    err = rts_create(&rts_handle,rts_cfg);
    if(!err)
    {
        rts_state = 1;
        rts_read_cnt = 0;
        rts_write_cnt = 0;
        rts_read_err_occured = FALSE;
        rts_write_err_occured = FALSE;

        if(ecnr_audio_rts_handler_read_perform_avail)
        {
            ecnr_audio_rts_handler_performance_test_init("rts_read", 40000);
        }

        pr_message("handle after create rts_handle(%p)\n",rts_handle);
        ETG_TRACE_USR4(("[ecnr_audio_rts_handler_create]: handle after create rts_handle(%p)",rts_handle));
    }
    else
    {
        pr_critical("fail to create rts instance with err : %d\n", err);
        ETG_TRACE_USR3(("[ecnr_audio_rts_handler_create]: fail to create rts instance with err : %d", err));
    }

error_out:
    g_mutex_unlock(&state->lock);
    pr_debug("EXIT");
    ETG_TRACE_USR4(("[ecnr_audio_rts_handler_create]: EXIT"));
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_audio_rts_handler_destroy
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: int
*
*******************************************************************************/
extern int ecnr_audio_rts_handler_destroy(stecnr_alsa_state* state)
{
    int err = ECNR_ERR_OK;

    pr_debug("ENTERED");
    ETG_TRACE_USR3(("[ecnr_audio_rts_handler_destroy]: ENTERED"));


    if(!state)
    {
        pr_critical("state structure pointer not initialized\n");
        ETG_TRACE_ERR(("[ecnr_audio_rts_handler_destroy]: unvalide alsa instance"));
        return ECNR_ERR_NULL_POINTER;
    }

    g_mutex_lock(&state->lock);

    if(rts_state == 1)
    {
        if(rts_handle != NULL)
        {
            pr_debug("rts_destroy with handle(%p)\n",rts_handle);

            err = rts_destroy(rts_handle);
            if(err)
                pr_critical("rts_destroy failed with error(%d)\n", err);
        }
        else
            pr_warning("rts handle is NULL, rts_destroy is not called\n");

        /* always free data structure, the behavior in the case the rts_destroy failed
           is impredicable. Maybe the device handle are still owned by alsa library?
        */
        if(rts_cfg)
        {
            g_free(rts_cfg);
            rts_cfg = NULL;
        }

        rts_state = 0;
        rts_handle = NULL;

    if(ecnr_audio_rts_handler_read_perform_avail)
      ecnr_audio_rts_handler_performance_test_terminate();
    }
    else
    {
        pr_debug("no rts instance available");
        err = ECNR_ERR_NULL_POINTER;
    }

    g_mutex_unlock(&state->lock);
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_audio_rts_handler_read
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_audio_rts_handler_read(void * buffer[])
{
    int err = ECNR_ERR_OK;
    //pr_debug("ENTERED");

    if(rts_state == 1)
    {
        if(m_ecnr_audio_rts_handler_performance_enable)
        {
            struct timespec ts_call, ts_ret;
            glong exec_time = 0;

            clock_gettime(CLOCK_MONOTONIC, &ts_call);
            err = rts_read(rts_handle, buffer);
            clock_gettime(CLOCK_MONOTONIC, &ts_ret);

            if(ts_call.tv_sec == ts_ret.tv_sec)
            {
                exec_time = ts_ret.tv_nsec - ts_call.tv_nsec;
            }
            else /* next second*/
            {
                exec_time = (1000000000 - ts_call.tv_nsec) + ts_ret.tv_nsec;
            }

            ETG_TRACE_USR4(("[ecnr_audio_rts_handler_read]: rts_read EXEC time : [%ld]", exec_time));
        }
        else
            err = rts_read(rts_handle, buffer);
    }
    else
        err = ECNR_ERR_RTS_STATE_ERROR;

    /* increment read counter*/
    rts_read_cnt++;

    if(!err)
        ETG_TRACE_USR4(("[ecnr_audio_rts_handler_read]: read successfully, current read count -> %d", rts_read_cnt));


    if(ecnr_audio_rts_handler_read_perform_avail)
        ecnr_audio_rts_handler_performance_read_complet_timestamp();

    if(err)
    {
        if(!rts_read_err_occured)
        {
            struct timespec err_read;
            clock_gettime(CLOCK_MONOTONIC, &err_read);

            pr_warning("audio capture fails with err = %d after %d reads  \n", err, rts_read_cnt);
            ETG_TRACE_USR3(("[ecnr_audio_rts_handler_read]: audio playback fails with err = %d after %d reads ", err, rts_read_cnt));
            ETG_TRACE_USR3(("[ecnr_audio_rts_handler_read]: ecnr_read_fails[%d,%d]", (unsigned int)err_read.tv_sec,
                                                                                        (unsigned long int)err_read.tv_nsec));
            ETG_TRACE_USR3(("[ecnr_audio_rts_handler_read]: ecnr_last_ok_read[%d,%d]", (unsigned int)ecnr_audio_rts_last_ok_read_time.tv_sec,
                                                                                          (unsigned long int)ecnr_audio_rts_last_ok_read_time.tv_nsec));
            rts_read_err_occured = TRUE;
        }
    }
    else
    {
        /* reset read error flag */
        rts_read_err_occured = FALSE;

        /* recording */
        /* write audio frame recording file  */
        if(mic_rec_desc.init)
        {
            if(mic_rec_desc.in_rec_active)
            {
                pr_debug("recording mic_in\n");
                (void)ecnr_audio_recording_mic_cap_stream_rec_file_write();
            }
        }

        if(rcv_rec_desc.init)
        {
            if(rcv_rec_desc.in_rec_active)
            {
                pr_debug("recording mic_in\n");
                (void)ecnr_audio_recording_rcv_cap_stream_rec_file_write();
            }
        }

        /* store time last ok read */
        clock_gettime(CLOCK_MONOTONIC, &ecnr_audio_rts_last_ok_read_time);

    }

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_audio_rts_handler_write
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_audio_rts_handler_write(void * buffer[])
{
    int err = ECNR_ERR_OK;
    //pr_debug("ENTERED");

    if(rts_state == 1)
    {
        if(m_ecnr_audio_rts_handler_performance_enable)
        {
            struct timespec ts_call, ts_ret;
            glong exec_time = 0;

            clock_gettime(CLOCK_MONOTONIC, &ts_call);
            err = rts_write(rts_handle, buffer);
            clock_gettime(CLOCK_MONOTONIC, &ts_ret);

            if(ts_call.tv_sec == ts_ret.tv_sec)
            {
                exec_time = ts_ret.tv_nsec - ts_call.tv_nsec;
            }
            else /* next second*/
            {
                exec_time = (1000000000 - ts_call.tv_nsec) + ts_ret.tv_nsec;
            }

            ETG_TRACE_USR4(("[ecnr_audio_rts_handler_write]: rts_write EXEC time : [%ld]\n", exec_time));
        }
        else
            err = rts_write(rts_handle, buffer);
    }
    else
        err = ECNR_ERR_RTS_STATE_ERROR;

    /* increment write counter*/
    rts_write_cnt++;

    if(!err)
        ETG_TRACE_USR4(("[ecnr_audio_rts_handler_write]: write successfully, current write count -> %d", rts_write_cnt));

    if(err)
    {
        if(!rts_write_err_occured)
        {
            pr_warning("audio playback fails with err = %d after %d writes  \n", err, rts_write_cnt);
            ETG_TRACE_USR3(("[ecnr_audio_rts_handler_write]: write fails with err = %d after %d writes ", err, rts_write_cnt));
            rts_write_err_occured = TRUE;
        }
    }
    else
    {
        /* reset write error flag */
        rts_write_err_occured = FALSE;

        /* recording */
        /* write audio frame recording file  */
        if(mic_rec_desc.init)
        {
            if(mic_rec_desc.out_rec_active)
            {
                pr_debug("recording mic_out\n");
                (void)ecnr_audio_recording_mic_pb_stream_rec_file_write();
            }
        }

        if(rcv_rec_desc.init)
        {
            if(rcv_rec_desc.out_rec_active)
            {
                pr_debug("recording mic_out\n");
                (void)ecnr_audio_recording_rcv_pb_stream_rec_file_write();
            }
        }
    }


    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_audio_rts_handler_recover
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_audio_rts_handler_recover(stecnr_alsa_state* state)
{
    int err = ECNR_ERR_OK;

    pr_debug("ENTERED");
    ETG_TRACE_USR3(("[ecnr_audio_rts_handler_recover]: ENTERED"));

    if(!state)
    {
        pr_critical("state structure pointer not initialized\n");
        return ECNR_ERR_NULL_POINTER;
    }


    if(rts_handle != NULL)
    {
        pr_debug("rts_recover with handle(%p)\n", rts_handle);

#ifndef D_SUPPORT_FAST_RECOVER
        err = ecnr_audio_rts_handler_destroy(state);
        if(err == ECNR_ERR_OK)
            err = ecnr_audio_rts_handler_create(state, FALSE);
        if(err)
        {
            pr_critical("rts creation failed with err(%d)\n", err);
        }
#else
        g_mutex_lock(&state->lock);
        err = rts_recover(rts_handle);
        if(err)
        {
            pr_critical("rts_recover failed with error(%d)\n", err);
            ETG_TRACE_ERR(("[ecnr_audio_rts_handler_recover]: rts_recover failed with error(%d)", err));
        }

        g_mutex_unlock(&state->lock);
#endif
    }
    else
    {
        pr_warning("rts handle is NULL, rts_recover is not called\n");
    }

    ETG_TRACE_USR3(("[ecnr_audio_rts_handler_recover]: EXIT"));
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_get_config_data
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_audio_rts_get_state(stecnr_alsa_state* state)
{
    int curr_state = 0;

    if(state)
    {
        g_mutex_lock(&state->lock);
        pr_debug("rts handler state: %d", rts_state);
        curr_state = rts_state;
        g_mutex_unlock(&state->lock);
    }
    else
        pr_warning("alsa audio strreaming instance structure is not initialized\n");

    return curr_state;
}


/*******************************************************************************
*
* FUNCTION: ecnr_audio_rts_handler_statistic_get
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_audio_rts_handler_statistic_get(trts_stat *stat)
{
    int err = ECNR_ERR_OK;

    if(rts_state == 1)
        err = rts_statistic(rts_handle, stat);
    else
        err = ECNR_ERR_RTS_STATE_ERROR;

    pr_message("number of rts write call: %d \n",rts_write_cnt);
    pr_message("number of rts read call: %d \n",rts_read_cnt);

    ETG_TRACE_USR3(("[ecnr_audio_rts_handler_statistic_get]: number of rts write call: %d",rts_write_cnt));
    ETG_TRACE_USR3(("[ecnr_audio_rts_handler_statistic_get]: number of rts read call: %d",rts_read_cnt));

    /*new*/
    if(stat)
    {
        pr_message("number of xrun: %d \n",stat->num_xruns);
        ETG_TRACE_USR3(("[ecnr_audio_rts_handler_statistic_get]: number of xrun: %d",stat->num_xruns));
    }

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_audio_rts_handler_clear_xrun
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_audio_rts_handler_clear_xrun(void)
{
    int err = ECNR_ERR_OK;
    trts_stat stat= {0};

    if(rts_handle != NULL)
    {
        stat.clear = 1;
        err = rts_statistic(rts_handle, &stat);
        if(err)
            pr_warning("fails to clear rts statistic\n");
    }
    else
        pr_warning("invalid rts handle, the rts_statistic fct is not called\n");

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_audio_rts_handler_streaming_state_get
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_audio_rts_handler_streaming_rw_state_get(int* wrount, int* rdcount)
{
    int err = ECNR_ERR_OK;

    *wrount = rts_write_cnt;
    *rdcount = rts_read_cnt;

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_audio_rts_handler_statistic_get
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_audio_rts_handler_get_xrun(unsigned int *xrun)
{
    int err = ECNR_ERR_OK;
    trts_stat stat= {0};

    if(!xrun)
        return ECNR_ERR_NULL_POINTER;

    if(rts_state == 1)
        err = rts_statistic(rts_handle, &stat);
    else
        err = ECNR_ERR_RTS_STATE_ERROR;

    /* return cuurent xrun count*/
    *xrun = stat.num_xruns;

    if(stat.num_xruns)
    {
        pr_message("current xrun count-> %d \n",stat.num_xruns);
        ETG_TRACE_USR3(("[ecnr_audio_rts_handler_get_xrun]: current xrun count-> %d \n",stat.num_xruns));
    }

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_audio_rts_handler_init
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int  ecnr_audio_rts_handler_init(void)
{

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_audio_rts_handler_init]: ENTERED"));

    ecnr_audio_recording_init();


    /* performance test */
    const char* sRec_rts_read_write = getenv("ECNR_RTS_PERFORMANCE_AVAILABLE");

    if(sRec_rts_read_write)
        pr_message("FEAT ECNR_RTS_PERFORMANCE_AVAILABLE = %s\n", sRec_rts_read_write);

    if (!g_strcmp0(sRec_rts_read_write, "1"))  // record read stream only
    {
        ecnr_audio_rts_handler_read_perform_avail = 1;
    }
    else
    {
        ecnr_audio_rts_handler_read_perform_avail = 0;
    }

    pr_message("EXIT\n");
    ETG_TRACE_USR3(("[ecnr_audio_rts_handler_init]: EXIT"));

    return ECNR_ERR_OK;
}
