/******************************************************************************
 * FILE        : ecnr-audio-recording.c
 * PROJECT     : Gen3 and Gen4
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2017 RBCM GMBH
 * HISTORY     :
 * Date        | Author                 | Modification
 * 10.11.2017  | Patrick Rey            | initial version
 *             |                        | audio stream recording features
 *****************************************************************************/



/*******************************************************************************
              DEFINES
*******************************************************************************/


/*******************************************************************************
              GENERATED CODE HEADERS
*******************************************************************************/


#include <glib-unix.h>
#include <glib/gi18n.h>
#include <glib-object.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include <alsa/asoundlib.h>

#include "ecnr-error.h"
#include "ecnr-common-defs.h"
#include "ecnr-service.h"
#include "ecnr-alsa.h"
#include "ecnr-audio-recording.h"
#include "ecnr-audio-rts-handler.h"


/* Needed for Trace */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define SPM_UNIT_TESTS // solve warning
#include "ecnr-trace.h"
#include "etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ECNR
#include "trcGenProj/Header/ecnr-audio-recording.c.trc.h"
#endif

/*******************************************************************************
              STRUCTURE DEFINITION
*******************************************************************************/

#define D_AUDPROC_RAM_REC_SIZE_MAX 6000000
static const char      *D_MICRO_RECORD_FILE_OUT_PCM     = "/tmp/ecnr_mic_record_out";
static const char      *D_MICRO_RECORD_FILE_IN_PCM      = "/tmp/ecnr_mic_record_in";
static const char      *D_REF_RECORD_FILE_IN_PCM        = "/tmp/ecnr_ref_record_in";
static const char      *D_RCV_RECORD_FILE_OUT_PCM       = "/tmp/ecnr_rcv_record_out";
static const char      *D_RCV_RECORD_FILE_IN_PCM        = "/tmp/ecnr_rcv_record_in";
static char*           sRec_stream = (char*)NULL;
static int             ecnr_audio_recording_export_rec_idx = 0;

typedef struct _wave_header
{
    unsigned int    num_samples;
    unsigned long   num_bytes;
    unsigned int    sample_rate;
    unsigned int    bytes_per_sample;
    unsigned int    num_channels;
}wave_header;



/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/

record_desc mic_rec_desc = {0};
record_desc rcv_rec_desc = {0};
record_desc ref_rec_desc = {0};





static int ecnr_audio_rts_mic_rec_file_open(void)
{
    int err = ECNR_ERR_OK;

    pr_message("ENTERED\n");


    mic_rec_desc.rdata = 0;
    mic_rec_desc.wdata = 0;
    mic_rec_desc.rpos = 0;
    mic_rec_desc.wr_data_out_ptr  = 0;
    mic_rec_desc.wr_data_in_ptr    = 0;
    ref_rec_desc.wr_data_in_ptr    = 0;
    mic_rec_desc.start_stop_rec_in = TRUE;
    mic_rec_desc.start_stop_rec_out = TRUE;
    ref_rec_desc.start_stop_rec_in = TRUE;

    return err;
}

static int ecnr_audio_rts_rcv_rec_file_open(void)
{
    int err = ECNR_ERR_OK;

    pr_message("ENTERED\n");


    rcv_rec_desc.rdata = 0;
    rcv_rec_desc.wdata = 0;
    rcv_rec_desc.rpos = 0;
    rcv_rec_desc.wr_data_out_ptr  = 0;
    rcv_rec_desc.wr_data_in_ptr    = 0;
    rcv_rec_desc.start_stop_rec_in = TRUE;
    rcv_rec_desc.start_stop_rec_out = TRUE;

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_audio_recording_new
* DESCRIPTION:module constructor
*
*
* PARAMETER: None.
*
* RETURNVALUE: void
*
*******************************************************************************/
void ecnr_audio_recording_new(void)
{
    pr_message("ENTERED\n");
    ecnr_audio_recording_export_rec_idx = 0;
    return;
}




/*******************************************************************************
*
* FUNCTION: ecnr_audio_recording_init
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: void
*
*******************************************************************************/
void ecnr_audio_recording_init(void)
{
    pr_message("ENTERED\n");


    if(   !mic_rec_desc.init
        &&!rcv_rec_desc.init
        &&!ref_rec_desc.init)
    {

        /* initialize module member variable  */
        memset(&mic_rec_desc,0 ,sizeof(record_desc));
        memset(&rcv_rec_desc,0 ,sizeof(record_desc));
        memset(&ref_rec_desc,0 ,sizeof(record_desc));

        sRec_stream = getenv("ECNR_REC_STREAM_TO_FILE");

        if(sRec_stream)
            pr_message("FEAT ECNR_REC_STREAM_TO_FILE = %s\n", sRec_stream);

        if (!g_strcmp0(sRec_stream, "1"))  // record playback stream only
        {
            mic_rec_desc.out_rec_active = 1;
            mic_rec_desc.in_rec_active = 0;
            ref_rec_desc.out_rec_active = 1;
            ref_rec_desc.in_rec_active = 0;
            mic_rec_desc.start_stop_rec_out = TRUE;
        }
        else if (!g_strcmp0(sRec_stream, "2"))  // record capture stream only
        {
            mic_rec_desc.out_rec_active = 0;
            mic_rec_desc.in_rec_active = 1;
            ref_rec_desc.out_rec_active = 0;
            ref_rec_desc.in_rec_active = 1;
            mic_rec_desc.start_stop_rec_out = TRUE;
        }
        else if (!g_strcmp0(sRec_stream, "3"))  // record capture and playback
        {
            mic_rec_desc.out_rec_active = 1;
            mic_rec_desc.in_rec_active = 1;
            ref_rec_desc.out_rec_active = 1;
            ref_rec_desc.in_rec_active = 1;
            mic_rec_desc.start_stop_rec_out = TRUE;
        }
        else if (!g_strcmp0(sRec_stream, "4"))  // record playback stream only
        {
            ecnr_audio_rts_rcv_rec_file_open(); ;
            rcv_rec_desc.out_rec_active = 1;
            rcv_rec_desc.in_rec_active = 0;
        }
        else if (!g_strcmp0(sRec_stream, "5"))  // record capture stream only
        {
            ecnr_audio_rts_rcv_rec_file_open();
            rcv_rec_desc.out_rec_active = 0;
            rcv_rec_desc.in_rec_active = 1;
        }
        else if (!g_strcmp0(sRec_stream, "6"))  // record capture and playback
        {
            ecnr_audio_rts_rcv_rec_file_open(); ;
            rcv_rec_desc.out_rec_active = 1;
            rcv_rec_desc.in_rec_active = 1;
        }
        else if (!g_strcmp0(sRec_stream, "7"))  // record capture and playback
        {
            ecnr_audio_rts_mic_rec_file_open();
            mic_rec_desc.out_rec_active = 1;
            mic_rec_desc.in_rec_active = 1;
            ref_rec_desc.out_rec_active = 1;
            ref_rec_desc.in_rec_active = 1;
            ecnr_audio_rts_rcv_rec_file_open();
            rcv_rec_desc.out_rec_active = 1;
            rcv_rec_desc.in_rec_active = 1;
        }
        else
        {
            /* no defined */

        }
    }

    ecnr_audio_recording_export_rec_idx++;

    return;
}

void ecnr_audio_recording_next_record(void)
{
    ecnr_audio_recording_export_rec_idx++;
    return;
}

int  ecnr_audio_recording_is_rec_active(enum ecnr_audio ecnr_id)
{
    int state = 0;
    pr_debug("ENTERED\n");

    switch(ecnr_id)
    {
        case ECNR_MIC_OUT:
        {
            state = mic_rec_desc.out_rec_active;
            break;
        }
        case ECNR_MULTI_CHAN_IN_DEV:
        case ECNR_MIC_IN0:
        case ECNR_MIC_IN1:
        case ECNR_REF_IN0:
        case ECNR_REF_IN1:
        case ECNR_MULTI_CHAN_IN_REF_0:
        case ECNR_MULTI_CHAN_IN_REF_1:
        {
            state = mic_rec_desc.in_rec_active;
            break;
        }
        case ECNR_RECV_IN:
        {
            state = rcv_rec_desc.in_rec_active;
            break;
        }
        case ECNR_RECV_OUT:
        {
            state = rcv_rec_desc.out_rec_active;
            break;
        }
        default:
            break;
    }

    return state;
}

void  ecnr_audio_recording_enable_mic_rec(void)
{

    pr_message("mic recording is enabled \n");
    ETG_TRACE_USR1(("[ecnr_audio_recording_enable_mic_rec]: mic recording is enabled"));
    mic_rec_desc.init = 1;
}

void  ecnr_audio_recording_enable_rcv_rec(void)
{

    pr_message("rcv recording is enabled \n");
    ETG_TRACE_USR1(("[ecnr_audio_recording_enable_rcv_rec]: rcv recording is enabled"));
    rcv_rec_desc.init = 1;
}

void ecnr_audio_recording_stop(void)
{
#if 0
    if (  !g_strcmp0(sRec_stream, "1")
        ||!g_strcmp0(sRec_stream, "2")
        ||!g_strcmp0(sRec_stream, "3")
        ||!g_strcmp0(sRec_stream, "4")
        ||!g_strcmp0(sRec_stream, "5")
        ||!g_strcmp0(sRec_stream, "6")
        ||!g_strcmp0(sRec_stream, "7"))  // record playback stream only
    {
        /*voice sources*/
        mic_rec_desc.out_rec_active = 0;
        mic_rec_desc.in_rec_active = 0;
    }
    else
    {
        /* do nothing */
    }
#endif
        mic_rec_desc.out_rec_active = 0;
        mic_rec_desc.in_rec_active = 0;
        ref_rec_desc.out_rec_active = 0;
        ref_rec_desc.in_rec_active = 0;
        rcv_rec_desc.out_rec_active = 0;
        rcv_rec_desc.in_rec_active = 0;
    return;
}

void ecnr_audio_recording_restart(void)
{
    if (!g_strcmp0(sRec_stream, "1"))  // record playback stream only
    {
        mic_rec_desc.out_rec_active = 1;
        mic_rec_desc.in_rec_active = 0;
        ref_rec_desc.out_rec_active = 1;
        ref_rec_desc.in_rec_active = 0;
        mic_rec_desc.start_stop_rec_out = TRUE;
    }
    else if (!g_strcmp0(sRec_stream, "2"))  // record capture stream only
    {
        mic_rec_desc.out_rec_active = 0;
        mic_rec_desc.in_rec_active = 1;
        ref_rec_desc.out_rec_active = 0;
        ref_rec_desc.in_rec_active = 1;
        mic_rec_desc.start_stop_rec_out = TRUE;
    }
    else if (!g_strcmp0(sRec_stream, "3"))  // record capture and playback
    {
        mic_rec_desc.out_rec_active = 1;
        mic_rec_desc.in_rec_active = 1;
        ref_rec_desc.out_rec_active = 1;
        ref_rec_desc.in_rec_active = 1;
        mic_rec_desc.start_stop_rec_out = TRUE;
    }
    else if (!g_strcmp0(sRec_stream, "4"))  // record playback stream only
    {
        ecnr_audio_rts_rcv_rec_file_open(); ;
        rcv_rec_desc.out_rec_active = 1;
        rcv_rec_desc.in_rec_active = 0;
    }
    else if (!g_strcmp0(sRec_stream, "5"))  // record capture stream only
    {
        ecnr_audio_rts_rcv_rec_file_open();
        rcv_rec_desc.out_rec_active = 0;
        rcv_rec_desc.in_rec_active = 1;
    }
    else if (!g_strcmp0(sRec_stream, "6"))  // record capture and playback
    {
        ecnr_audio_rts_rcv_rec_file_open(); ;
        rcv_rec_desc.out_rec_active = 1;
        rcv_rec_desc.in_rec_active = 1;
    }
    else if (!g_strcmp0(sRec_stream, "7"))  // record capture and playback
    {
        ecnr_audio_rts_mic_rec_file_open(); ;
        mic_rec_desc.out_rec_active = 1;
        mic_rec_desc.in_rec_active = 1;
        ref_rec_desc.out_rec_active = 1;
        ref_rec_desc.in_rec_active = 1;
        ecnr_audio_rts_rcv_rec_file_open(); ;
        rcv_rec_desc.out_rec_active = 1;
        rcv_rec_desc.in_rec_active = 1;
    }
    else
    {
        /* no defined */

    }
    return;
}
/*******************************************************************************
*
* FUNCTION: ecnr_audio_recording_mic_rec_file_reopen
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: void
*
*******************************************************************************/
int ecnr_audio_recording_mic_rec_file_reopen(void)
{
    int err = ECNR_ERR_OK;
    unsigned int i = 0;

    pr_debug("ENTERED\n");

    if(!mic_rec_desc.init)
        return ECNR_ERR_OK;


    stecnr_alsa_state*  paudio = ecnr_alsa_get_instance();
    if(!paudio)
        return  ECNR_ERR_OK;

    pr_message("mic recording is enabled \n");
    ETG_TRACE_USR1(("[ecnr_audio_recording_mic_rec_file_reopen]: mic recording is enabled"));

    /* store first in a RAM Buffer */
    if(mic_rec_desc.in_rec_active)
    {
        if(paudio->ecnr_mic_in_cnt)
        {
            if(mic_rec_desc.AudRamBufChannels_in)
            {
                // free first the channel buffer
                for (i = 0; i < paudio->ecnr_mic_in_cnt ; i++ )
                {
                    if(mic_rec_desc.AudRamBufChannels_in[i])
                    {
                        g_free(mic_rec_desc.AudRamBufChannels_in[i]);
                        mic_rec_desc.AudRamBufChannels_in[i] = (guint8*)NULL;
                    }
                }
                g_free(mic_rec_desc.AudRamBufChannels_in);
                mic_rec_desc.AudRamBufChannels_in = (guint8**)NULL;
            }

            /* create buffer */

            pr_message("create %d ram record input buffer\n", paudio->ecnr_mic_in_cnt );
            mic_rec_desc.AudRamBufChannels_in = (guint8**)g_malloc0(paudio->ecnr_mic_in_cnt);

            if(mic_rec_desc.AudRamBufChannels_in)
            {
                // create channel buffer
                for (i = 0; i < paudio->ecnr_mic_in_cnt ; i++ )
                {
                    mic_rec_desc.AudRamBufChannels_in[i] = (guint8*)g_malloc0(D_AUDPROC_RAM_REC_SIZE_MAX);
                    if(mic_rec_desc.AudRamBufChannels_in[i])
                        pr_message("create ram record buffer channel_%d from %d bytes", i, D_AUDPROC_RAM_REC_SIZE_MAX );
                }
            }

            mic_rec_desc.init = 1;
            pr_message("mic_in recording is activated\n");
            ETG_TRACE_USR1(("[ecnr_audio_recording_mic_rec_file_reopen]: mic_in recording is activated"));
        }

        if(paudio->ecnr_ref_in_cnt)
        {
            if(ref_rec_desc.AudRamBufChannels_in)
            {
                // free first the channel buffer
                for (i = 0; i < paudio->ecnr_ref_in_cnt ; i++ )
                {
                    if(ref_rec_desc.AudRamBufChannels_in[i])
                    {
                        g_free(ref_rec_desc.AudRamBufChannels_in[i]);
                        ref_rec_desc.AudRamBufChannels_in[i] = (guint8*)NULL;
                    }
                }
                g_free(ref_rec_desc.AudRamBufChannels_in);
                ref_rec_desc.AudRamBufChannels_in = (guint8**)NULL;
            }

            /* create buffer */

            pr_message("create %d ram record input buffer\n", paudio->ecnr_ref_in_cnt );
            ref_rec_desc.AudRamBufChannels_in = (guint8**)g_malloc0(paudio->ecnr_ref_in_cnt);

            if(ref_rec_desc.AudRamBufChannels_in)
            {
                // create channel buffer
                for (i = 0; i < paudio->ecnr_ref_in_cnt ; i++ )
                {
                    ref_rec_desc.AudRamBufChannels_in[i] = (guint8*)g_malloc0(D_AUDPROC_RAM_REC_SIZE_MAX);
                    if(ref_rec_desc.AudRamBufChannels_in[i])
                        pr_message("create ram record buffer channel_%d from %d bytes", i, D_AUDPROC_RAM_REC_SIZE_MAX );
                }
            }

            ref_rec_desc.init = 1;
            pr_message("ref_in recording is activated\n");
            ETG_TRACE_USR1(("[ecnr_audio_recording_mic_rec_file_reopen]: ref_in recording is activated"));
        }
    }


    if(mic_rec_desc.out_rec_active)
    {
        if(mic_rec_desc.AudRamBufChannels_out)
        {
            // free firts the channel buffer
            for (i = 0; i < 1 ; i++ )
            {
                if(mic_rec_desc.AudRamBufChannels_out[i])
                {
                    g_free(mic_rec_desc.AudRamBufChannels_out[i]);
                    mic_rec_desc.AudRamBufChannels_out[i] = (guint8*)NULL;
                }
            }
            g_free(mic_rec_desc.AudRamBufChannels_out);
            mic_rec_desc.AudRamBufChannels_out = (guint8**)NULL;
        }

        /* create buffer */

        pr_message("create %d ram record output buffer\n", 1 );
        mic_rec_desc.AudRamBufChannels_out = (guint8**)g_malloc0(1 * sizeof(guint8*));

        if(mic_rec_desc.AudRamBufChannels_out)
        {
            // free firts the channel buffer
            for (i = 0; i < 1 ; i++ )
            {
                mic_rec_desc.AudRamBufChannels_out[i] = (guint8*)g_malloc0(D_AUDPROC_RAM_REC_SIZE_MAX);
                if(mic_rec_desc.AudRamBufChannels_out[i])
                    pr_message("create ram record buffer channel_%d from %d bytes\n", i, D_AUDPROC_RAM_REC_SIZE_MAX );
            }
        }

        mic_rec_desc.init = 1;
        pr_message("mic_out recording is activated\n");
        ETG_TRACE_USR1(("[ecnr_audio_recording_mic_rec_file_reopen]: mic_out recording is activated"));
    }

    if(mic_rec_desc.init)
    {
        mic_rec_desc.rdata = 0;
        mic_rec_desc.wdata = 0;
        mic_rec_desc.rpos = 0;
        mic_rec_desc.wr_data_out_ptr  = 0;
        mic_rec_desc.wr_data_in_ptr    = 0;
        ref_rec_desc.wr_data_in_ptr    = 0;
        mic_rec_desc.start_stop_rec_in = TRUE;
        mic_rec_desc.start_stop_rec_out = TRUE;
        ref_rec_desc.start_stop_rec_in = TRUE;

        /* store start record time */
        clock_gettime(CLOCK_MONOTONIC, &mic_rec_desc.rec_in_time_start_record);
        clock_gettime(CLOCK_MONOTONIC, &mic_rec_desc.rec_out_time_start_record);
        clock_gettime(CLOCK_MONOTONIC, &ref_rec_desc.rec_in_time_start_record);


        mic_rec_desc.rec_in_dev = ecnr_alsa_get_dev_name(ECNR_MULTI_CHAN_IN_DEV);
        mic_rec_desc.rec_out_dev = ecnr_alsa_get_dev_name(ECNR_MIC_OUT);
        ref_rec_desc.rec_in_dev = ecnr_alsa_get_dev_name(ECNR_MULTI_CHAN_IN_DEV);

    }

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_audio_recording_rcv_rec_file_reopen
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: void
*
*******************************************************************************/
int ecnr_audio_recording_rcv_rec_file_reopen(void)
{
    int err = ECNR_ERR_OK;
    unsigned int i = 0;

    pr_debug("ENTERED\n");

    if(!rcv_rec_desc.init)
        return ECNR_ERR_OK;


    stecnr_alsa_state*  paudio = ecnr_alsa_get_instance();
    if(!paudio)
        return  ECNR_ERR_OK;

    pr_message("rcv recording is enabled \n");
    ETG_TRACE_USR1(("[ecnr_audio_recording_rcv_rec_file_reopen]: rcv recording is enabled"));

    /* store first in a RAM Buffer */
    if(rcv_rec_desc.in_rec_active)
    {
        if(paudio->ecnr_recv_in_cnt)
        {
            if(rcv_rec_desc.AudRamBufChannels_in)
            {
                // free first the channel buffer
                for (i = 0; i < paudio->ecnr_recv_in_cnt ; i++ )
                {
                    if(rcv_rec_desc.AudRamBufChannels_in[i])
                    {
                        g_free(rcv_rec_desc.AudRamBufChannels_in[i]);
                        rcv_rec_desc.AudRamBufChannels_in[i] = (guint8*)NULL;
                    }
                }
                g_free(rcv_rec_desc.AudRamBufChannels_in);
                rcv_rec_desc.AudRamBufChannels_in = (guint8**)NULL;
            }

            /* create buffer */

            pr_message("create %d ram record input buffer\n", paudio->ecnr_recv_in_cnt );
            rcv_rec_desc.AudRamBufChannels_in = (guint8**)g_malloc0(paudio->ecnr_recv_in_cnt);

            if(rcv_rec_desc.AudRamBufChannels_in)
            {
                // create channel buffer
                for (i = 0; i < paudio->ecnr_recv_in_cnt ; i++ )
                {
                    rcv_rec_desc.AudRamBufChannels_in[i] = (guint8*)g_malloc0(D_AUDPROC_RAM_REC_SIZE_MAX);
                    if(rcv_rec_desc.AudRamBufChannels_in[i])
                        pr_message("create ram record buffer channel_%d from %d bytes", i, D_AUDPROC_RAM_REC_SIZE_MAX );
                }
            }

            rcv_rec_desc.init = 1;
            pr_message("rcv_in recording is activated\n");
            ETG_TRACE_USR1(("[ecnr_audio_recording_rcv_rec_file_reopen]: rcv_in recording is activated"));
        }

    }


    if(rcv_rec_desc.out_rec_active)
    {
        if(paudio->ecnr_recv_in_cnt)
        {
            if(rcv_rec_desc.AudRamBufChannels_out)
            {
                // free firts the channel buffer
                for (i = 0; i < 1 ; i++ )
                {
                    if(rcv_rec_desc.AudRamBufChannels_out[i])
                    {
                        g_free(rcv_rec_desc.AudRamBufChannels_out[i]);
                        rcv_rec_desc.AudRamBufChannels_out[i] = (guint8*)NULL;
                    }
                }
                g_free(rcv_rec_desc.AudRamBufChannels_out);
                rcv_rec_desc.AudRamBufChannels_out = (guint8**)NULL;
            }

            /* create buffer */

            pr_message("create %d ram record output buffer\n", 1 );
            rcv_rec_desc.AudRamBufChannels_out = (guint8**)g_malloc0(1 * sizeof(guint8*));

            if(rcv_rec_desc.AudRamBufChannels_out)
            {
                // free firts the channel buffer
                for (i = 0; i < 1 ; i++ )
                {
                    rcv_rec_desc.AudRamBufChannels_out[i] = (guint8*)g_malloc0(D_AUDPROC_RAM_REC_SIZE_MAX);
                    if(rcv_rec_desc.AudRamBufChannels_out[i])
                        pr_message("create ram record buffer channel_%d from %d bytes\n", i, D_AUDPROC_RAM_REC_SIZE_MAX );
                }
            }

            rcv_rec_desc.init = 1;
            pr_message("rcv_out recording is activated\n");
            ETG_TRACE_USR1(("[ecnr_audio_recording_rcv_rec_file_reopen]: rcv_out recording is activated"));

        }
    }

    if(rcv_rec_desc.init)
    {
        rcv_rec_desc.rdata = 0;
        rcv_rec_desc.wdata = 0;
        rcv_rec_desc.rpos = 0;
        rcv_rec_desc.wr_data_out_ptr  = 0;
        rcv_rec_desc.wr_data_in_ptr    = 0;
        rcv_rec_desc.start_stop_rec_in = TRUE;
        rcv_rec_desc.start_stop_rec_out = TRUE;

        /* store start record time */
        clock_gettime(CLOCK_MONOTONIC, &rcv_rec_desc.rec_in_time_start_record);
        clock_gettime(CLOCK_MONOTONIC, &rcv_rec_desc.rec_out_time_start_record);

        rcv_rec_desc.rec_in_dev = ecnr_alsa_get_dev_name(ECNR_RECV_IN);
        rcv_rec_desc.rec_out_dev = ecnr_alsa_get_dev_name(ECNR_RECV_OUT);
    }

    return err;
}

int ecnr_audio_recording_mic_pb_stream_rec_file_write(void)
{
    int             err = ECNR_ERR_OK;
    unsigned int    i = 0;
    short**         ecnr_buffer;
    guint16         data_size = 0;

    pr_debug("ENTERED\n");


    stecnr_alsa_state*  paudio = ecnr_alsa_get_instance();
    if(!paudio)
    {
        pr_critical("invalide alsa instance\n");
        return ECNR_ERR_OK;
    }

    if(!paudio->ecnr_mic_in_cnt)
        return ECNR_ERR_OK;

    data_size = (guint16)(2 * paudio->ecnr_frame_shift_out);

    if(!mic_rec_desc.start_stop_rec_out)
    {
        pr_debug("max size (%d) of the Ram Buffer was exceeded \n", D_AUDPROC_RAM_REC_SIZE_MAX);
    }
    else
    {
        if((mic_rec_desc.wr_data_out_ptr + data_size) >=  D_AUDPROC_RAM_REC_SIZE_MAX)
        {
            /* store start record time */
            clock_gettime(CLOCK_MONOTONIC, &mic_rec_desc.rec_out_time_stop_record);
            mic_rec_desc.start_stop_rec_out = FALSE;
        }
        else
        {
            ecnr_buffer = *paudio->MicOut;

            guint8* dest = mic_rec_desc.AudRamBufChannels_out[0];
            guint8* src =  (guint8*)ecnr_buffer[0];

            if(dest && src)
                memcpy((void*)&dest[mic_rec_desc.wr_data_out_ptr], (void*)src, data_size);
            else
                pr_warning("audio channel_buffer_%d is invalid \n", i);

            mic_rec_desc.wr_data_out_ptr += data_size;
            pr_debug("write %d_bytes in the %d Ram channel_Buffer with current size:%d\n", data_size, 1, mic_rec_desc.wr_data_out_ptr);
        }
    }

    return err;
}

int ecnr_audio_recording_rcv_pb_stream_rec_file_write(void)
{
    int             err = ECNR_ERR_OK;
    unsigned int    i = 0;
    short**         ecnr_buffer;
    guint16         data_size = 0;

    pr_debug("ENTERED\n");


    stecnr_alsa_state*  paudio = ecnr_alsa_get_instance();
    if(!paudio)
    {
        pr_critical("invalide alsa instance\n");
        return ECNR_ERR_OK;
    }

    if(!paudio->ecnr_recv_in_cnt)
        return ECNR_ERR_OK;


    data_size = (guint16)(2 * paudio->ecnr_recv_frame_shift_out);

    if(!rcv_rec_desc.start_stop_rec_out)
    {
        pr_debug("max size (%d) of the Ram Buffer was exceeded \n", D_AUDPROC_RAM_REC_SIZE_MAX);
    }
    else
    {
        if((rcv_rec_desc.wr_data_out_ptr + data_size) >=  D_AUDPROC_RAM_REC_SIZE_MAX)
        {
            /* store start record time */
            clock_gettime(CLOCK_MONOTONIC, &rcv_rec_desc.rec_out_time_stop_record);
            rcv_rec_desc.start_stop_rec_out = FALSE;
        }
        else
        {
            ecnr_buffer = *paudio->RecvOut;

            guint8* dest = rcv_rec_desc.AudRamBufChannels_out[0];
            guint8* src =  (guint8*)ecnr_buffer[0];

            if(dest && src)
                memcpy((void*)&dest[rcv_rec_desc.wr_data_out_ptr], (void*)src, data_size);
            else
                pr_warning("audio channel_buffer_%d is invalid \n", i);

            rcv_rec_desc.wr_data_out_ptr += data_size;
            pr_debug("write %d_bytes in the %d Ram channel_Buffer with current size:%d\n", data_size, 1, rcv_rec_desc.wr_data_out_ptr);
        }
    }

    return err;
}


int ecnr_audio_recording_mic_cap_stream_rec_file_write(void)
{
    int err = ECNR_ERR_OK;
    unsigned int i = 0;
    short **ecnr_buffer;

    pr_debug("ENTERED\n");


    stecnr_alsa_state*  paudio = ecnr_alsa_get_instance();
    if(!paudio)
    {
        pr_critical("invalide alsa instance\n");
        return ECNR_ERR_OK;
    }


    guint16             data_size = (guint16)(2 * paudio->ecnr_frame_shift_in);


    if(!mic_rec_desc.start_stop_rec_in)
    {
        pr_debug("max size (%d) of the Ram Buffer was exceeded \n", D_AUDPROC_RAM_REC_SIZE_MAX);
    }
    else
    {
        if((mic_rec_desc.wr_data_in_ptr + data_size) >=  D_AUDPROC_RAM_REC_SIZE_MAX)
        {
            /* store start record time */
            clock_gettime(CLOCK_MONOTONIC, &mic_rec_desc.rec_in_time_stop_record);
            mic_rec_desc.start_stop_rec_in = FALSE;
        }
        else
        {
            ecnr_buffer = *paudio->MicIn;
            for (i = 0; i < paudio->ecnr_mic_in_cnt ; i++ )
            {
                guint8* dest = mic_rec_desc.AudRamBufChannels_in[i];
                guint8* src =  (guint8*)ecnr_buffer[i];

                if(dest && src)
                    memcpy((void*)&dest[mic_rec_desc.wr_data_in_ptr], (void*)src, data_size);
                else
                    pr_warning("audio channel_buffer_%d is invalid \n", i);
            }
            mic_rec_desc.wr_data_in_ptr += data_size;
            pr_debug("write %d_bytes in the %d Ram channel_Buffer with current size:%d\n", data_size, paudio->ecnr_mic_in_cnt, mic_rec_desc.wr_data_in_ptr);
        }
    }

    if(!paudio->ecnr_ref_in_cnt)
        return ECNR_ERR_OK;

    if(!ref_rec_desc.start_stop_rec_in)
    {
        pr_debug("max size (%d) of the Ram Buffer was exceeded \n", D_AUDPROC_RAM_REC_SIZE_MAX);
    }
    else
    {
        if((ref_rec_desc.wr_data_in_ptr + data_size) >=  D_AUDPROC_RAM_REC_SIZE_MAX)
        {
            /* store start record time */
            clock_gettime(CLOCK_MONOTONIC, &ref_rec_desc.rec_in_time_stop_record);
            ref_rec_desc.start_stop_rec_in = FALSE;
        }
        else
        {
            ecnr_buffer = *paudio->RefIn;
            for (i = 0; i < paudio->ecnr_ref_in_cnt ; i++ )
            {
                guint8* dest = ref_rec_desc.AudRamBufChannels_in[i];
                guint8* src =  (guint8*)ecnr_buffer[i];

                if(dest && src)
                    memcpy((void*)&dest[ref_rec_desc.wr_data_in_ptr], (void*)src, data_size);
                else
                {
                    pr_warning("audio channel_buffer_%d is invalid \n", i);
                }
            }
            ref_rec_desc.wr_data_in_ptr += data_size;
            pr_debug("write %d_bytes in the %d Ram channel_Buffer with current size:%d\n", data_size, paudio->ecnr_ref_in_cnt, ref_rec_desc.wr_data_in_ptr);
        }
    }

    return err;
}


int ecnr_audio_recording_rcv_cap_stream_rec_file_write(void)
{
    int err = ECNR_ERR_OK;
    unsigned int i = 0;
    short **ecnr_buffer;

    pr_debug("ENTERED\n");


    stecnr_alsa_state*  paudio = ecnr_alsa_get_instance();
    if(!paudio)
    {
        pr_critical("invalide alsa instance\n");
        return  ECNR_ERR_OK;
    }

    if(!paudio->ecnr_recv_in_cnt)
        return  ECNR_ERR_OK;

    guint16             data_size = (guint16)(2 * paudio->ecnr_recv_frame_shift_in);

    if(!rcv_rec_desc.start_stop_rec_in)
    {
        pr_debug("max size (%d) of the Ram Buffer was exceeded \n", D_AUDPROC_RAM_REC_SIZE_MAX);
    }
    else
    {
        if((rcv_rec_desc.wr_data_in_ptr + data_size) >=  D_AUDPROC_RAM_REC_SIZE_MAX)
        {
            /* store start record time */
            clock_gettime(CLOCK_MONOTONIC, &rcv_rec_desc.rec_in_time_stop_record);
            rcv_rec_desc.start_stop_rec_in = FALSE;
        }
        else
        {
            ecnr_buffer = *paudio->RecvIn;
            for (i = 0; i < paudio->ecnr_recv_in_cnt ; i++ )
            {
                guint8* dest = rcv_rec_desc.AudRamBufChannels_in[i];
                guint8* src =  (guint8*)ecnr_buffer[i];

                if(dest && src)
                    memcpy((void*)&dest[rcv_rec_desc.wr_data_in_ptr], (void*)src, data_size);
                else
                    pr_warning("audio channel_buffer_%d is invalid \n", i);
            }
            rcv_rec_desc.wr_data_in_ptr += data_size;
            pr_debug("write %d_bytes in the %d Ram channel_Buffer with current size:%d\n", data_size, paudio->ecnr_recv_in_cnt, rcv_rec_desc.wr_data_in_ptr);
        }
    }
    return err;
}

static void ecnr_audio_rts_write_little_endian(unsigned int word, int num_bytes, FILE *wav_file)
{
    unsigned buf;
    while(num_bytes>0)
    {
        buf = word & 0xff;
        pr_debug("write byte_%d:%d",num_bytes, buf);
        fwrite(&buf, 1,1, wav_file);
        num_bytes--;
        word >>= 8;
    }
}

/* information about the WAV file format from
    http://ccrma.stanford.edu/courses/422/projects/WaveFormat/
 */

static void ecnr_audio_rts_write_wav(char* filename, wave_header* wav_infos, guint8** data)
{
    FILE* wav_file;
    unsigned int sample_rate, num_channels, bytes_per_sample, byte_rate ;
    unsigned int i;    /* counter for samples */
    size_t wdata;
    unsigned int wr_pt = 0;
    unsigned int num_samples;

    pr_message("write audio sample in wav file :\"%s\"\n",filename );
    ETG_TRACE_USR1(("[ecnr_audio_rts_write_wav]: write audio sample in wav file :\"%s\"", filename));


    if(!filename || !wav_infos || !data)
        return;


    num_channels        = wav_infos->num_channels;   /* monoaural */
    pr_debug("wav_infos->num_channels: %d", wav_infos->num_channels);
    bytes_per_sample    = wav_infos->bytes_per_sample;
    pr_debug("wav_infos->bytes_per_sample: %d", wav_infos->bytes_per_sample);
    sample_rate         = wav_infos->sample_rate;
    pr_debug("wav_infos->sample_rate: %d", wav_infos->sample_rate);
    num_samples         = wav_infos->num_samples;
    pr_debug("wav_infos->num_samples: %d", wav_infos->num_samples);
    byte_rate = sample_rate * num_channels * bytes_per_sample;
    pr_debug("byte_rate: %d", byte_rate);


    wav_file = fopen(filename, "w");
    if(!wav_file)
    {
        pr_warning("fails to open wav aoutput record file\n");
        return;
    }


    /* write RIFF header */
    fwrite("RIFF", 1, 4, wav_file);
    ecnr_audio_rts_write_little_endian(36 + bytes_per_sample * num_samples * num_channels, 4, wav_file);
    fwrite("WAVE", 1, 4, wav_file);

    /* write fmt  subchunk */
    fwrite("fmt ", 1, 4, wav_file);
    ecnr_audio_rts_write_little_endian(16, 4, wav_file);   /* SubChunk1Size is 16 */
    ecnr_audio_rts_write_little_endian(1, 2, wav_file);    /* PCM is format 1 */
    ecnr_audio_rts_write_little_endian(num_channels, 2, wav_file);
    ecnr_audio_rts_write_little_endian(sample_rate, 4, wav_file);
    ecnr_audio_rts_write_little_endian(byte_rate, 4, wav_file);
    ecnr_audio_rts_write_little_endian( num_channels * bytes_per_sample, 2, wav_file);  /* block align */
    ecnr_audio_rts_write_little_endian( 8* bytes_per_sample, 2, wav_file);  /* bits/sample */

    /* write data subchunk */
    fwrite("data", 1, 4, wav_file);
    ecnr_audio_rts_write_little_endian(bytes_per_sample * num_samples * num_channels, 4, wav_file);

    wr_pt = 0;

    pr_debug("wav_infos->num_bytes: %lu", wav_infos->num_bytes);

    while(wr_pt < wav_infos->num_bytes)
    {
        for (i = 0; i < num_channels ; i++ )
        {
            guint8* audio = data[i];
            if(audio)
            {
                wdata = fwrite((void*)&audio[wr_pt], (size_t)1, bytes_per_sample, wav_file);
                pr_debug("write count is %lu",wdata);
            }
            else
                pr_warning("audio channel_buffer_%d is invalid \n", i);
        }
        wr_pt += bytes_per_sample;
    }

    fclose(wav_file);

}

#if 0 /* TODO: this function is not yet used, enable this when in need */
static void ecnr_audio_rts_create_wav_file(char* filename, wave_header* wav_infos)
{
    FILE* wav_file;
    unsigned int sample_rate, num_channels, bytes_per_sample, byte_rate ;
    unsigned int num_samples;

    pr_message("write audio sample in wav file :\"%s\"\n",filename );
    ETG_TRACE_USR1(("[ecnr_audio_rts_write_wav]: write audio sample in wav file :\"%s\"", filename));


    if(!filename || !wav_infos)
        return;


    num_channels        = wav_infos->num_channels;   /* monoaural */
    pr_debug("wav_infos->num_channels: %d", wav_infos->num_channels);
    bytes_per_sample    = wav_infos->bytes_per_sample;
    pr_debug("wav_infos->bytes_per_sample: %d", wav_infos->bytes_per_sample);
    sample_rate         = wav_infos->sample_rate;
    pr_debug("wav_infos->sample_rate: %d", wav_infos->sample_rate);
    num_samples         = wav_infos->num_samples;
    pr_debug("wav_infos->num_samples: %d", wav_infos->num_samples);
    byte_rate = sample_rate * num_channels * bytes_per_sample;
    pr_debug("byte_rate: %d", byte_rate);


    wav_file = fopen(filename, "w");
    if(!wav_file)
    {
        pr_warning("fails to open WAV output record file\n");
        return;
    }


    /* write RIFF header */
    fwrite("RIFF", 1, 4, wav_file);
    ecnr_audio_rts_write_little_endian(36 + bytes_per_sample * num_samples * num_channels, 4, wav_file);
    fwrite("WAVE", 1, 4, wav_file);

    /* write fmt  subchunk */
    fwrite("fmt ", 1, 4, wav_file);
    ecnr_audio_rts_write_little_endian(16, 4, wav_file);   /* SubChunk1Size is 16 */
    ecnr_audio_rts_write_little_endian(1, 2, wav_file);    /* PCM is format 1 */
    ecnr_audio_rts_write_little_endian(num_channels, 2, wav_file);
    ecnr_audio_rts_write_little_endian(sample_rate, 4, wav_file);
    ecnr_audio_rts_write_little_endian(byte_rate, 4, wav_file);
    ecnr_audio_rts_write_little_endian( num_channels * bytes_per_sample, 2, wav_file);  /* block align */
    ecnr_audio_rts_write_little_endian( 8* bytes_per_sample, 2, wav_file);  /* bits/sample */

    /* write data sub chunk */
    fwrite("data", 1, 4, wav_file);
    ecnr_audio_rts_write_little_endian(bytes_per_sample * num_samples * num_channels, 4, wav_file);
    fclose(wav_file);
}
#endif

static void ecnr_audio_recording_export_cap_dev_rec(enum ecnr_audio ecnr_id, record_desc* rec, const char* file_prefix)
{
    gchar*          pfound = NULL;
    gchar           key[] = ":";
    wave_header     wav_infos = {0};
    //guint16         data_size = 0;

    pr_debug("ENTERED\n");


    stecnr_alsa_state*  paudio = ecnr_alsa_get_instance();
    if(!paudio)
    {
        pr_critical("invalide alsa instance\n");
        return;
    }

    if(!rec || !file_prefix)
        return;

    if(rec->in_rec_active)
    {

        if(!rec->rec_in_dev)
            return;

        /* wave info's structure */
        wav_infos.num_bytes            = rec->wr_data_in_ptr ;
        if(ecnr_id == ECNR_MULTI_CHAN_IN_DEV)
            wav_infos.sample_rate      = paudio->ecnr_sample_rate_in; /*fix me*/
        else if(ecnr_id == ECNR_RECV_IN)
            wav_infos.sample_rate      = paudio->ecnr_recv_sample_rate_in; /*fix me*/
        else if((ecnr_id == ECNR_REF_IN0) || (ecnr_id == ECNR_REF_IN1))
            wav_infos.sample_rate      = paudio->ecnr_ref_sample_rate_in; /*fix me*/
        else
            wav_infos.sample_rate      =  16000;

        wav_infos.bytes_per_sample     = 2;

        if(ecnr_id == ECNR_MULTI_CHAN_IN_DEV)
            wav_infos.num_channels     = paudio->ecnr_mic_in_cnt; /*fix me*/
        else if(ecnr_id == ECNR_RECV_IN)
            wav_infos.num_channels     = paudio->ecnr_recv_in_cnt;
        else if((ecnr_id == ECNR_REF_IN0) || (ecnr_id == ECNR_REF_IN1))
            wav_infos.num_channels     = paudio->ecnr_ref_in_cnt; /*fix me*/
        else
            wav_infos.num_channels     = 1;

        wav_infos.num_samples          = (unsigned int)(wav_infos.num_bytes / wav_infos.bytes_per_sample) ;

        pr_message("write all recorded audio samples in files\n");
        ETG_TRACE_USR1(("[ecnr_audio_recording_export_cap_dev_rec]: write all recorded audio samples in files"));

        if(rec->start_stop_rec_in)
        {
            clock_gettime(CLOCK_MONOTONIC, &rec->rec_in_time_stop_record);
            rec->start_stop_rec_in = FALSE;
        }

        memset(rec->rec_file_name,0,300);


        sprintf(rec->rec_file_name, "%s_%s_%dch_%dkHz_%s_LE_%d.wav",  file_prefix,
                                                                       rec->rec_in_dev,
                                                                       wav_infos.num_channels,
                                                                       (wav_infos.sample_rate / 1000),
                                                                       "S16",
                                                                       ecnr_audio_recording_export_rec_idx);


        pr_debug("record file:%s\n",rec->rec_file_name);

        /* replace ":" character with "_"*/
        pfound = strpbrk (rec->rec_file_name, key);
        while (pfound != NULL)
        {
            *pfound = 0x5F;  /* "_" */
            pfound = strpbrk (pfound+1,key);
        }

        pr_message("************** print wav header infos ***************\n");
        pr_message("wav_infos.num_bytes         : %lu\n", wav_infos.num_bytes );
        pr_message("wav_infos.sample_rate       : %d\n", wav_infos.sample_rate );
        pr_message("wav_infos.bytes_per_sample  : %d\n", wav_infos.bytes_per_sample );
        pr_message("wav_infos.num_channels      : %d\n", wav_infos.num_channels );
        pr_message("wav_infos.num_samples       : %d\n", wav_infos.num_samples );
        pr_message("************** print wav header infos ***************\n");

        /* create wave file */
        ecnr_audio_rts_write_wav(rec->rec_file_name, &wav_infos, rec->AudRamBufChannels_in);
    }

    return;
}

static void ecnr_audio_recording_export_pb_dev_rec(enum ecnr_audio ecnr_id, record_desc* rec, const char* file_prefix)
{
    gchar*          pfound = NULL;
    gchar           key[] = ":";
    wave_header     wav_infos = {0};
    //guint16         data_size = 0;

    pr_debug("ENTERED\n");


    stecnr_alsa_state*  paudio = ecnr_alsa_get_instance();
    if(!paudio)
    {
        pr_critical("invalide alsa instance\n");
        return;
    }

    if(!rec || !file_prefix)
        return;

    if(rec->out_rec_active)
    {

        if(!rec->rec_out_dev)
            return;

        pr_message("write all recorded audio samples in files\n");
        ETG_TRACE_USR1(("[ecnr_audio_recording_export_pb_dev_rec]: write all recorded audio samples in files"));

        if(rec->start_stop_rec_out)
        {
            clock_gettime(CLOCK_MONOTONIC, &rec->rec_out_time_stop_record);
            rec->start_stop_rec_out = FALSE;
        }

        memset(rec->rec_file_name,0,300);

        /* wave infos structure */
        wav_infos.num_bytes            = rec->wr_data_out_ptr ;
        if(ecnr_id == ECNR_MIC_OUT)
            wav_infos.sample_rate      = paudio->ecnr_sample_rate_out; /*fix me*/
        else if(ecnr_id == ECNR_RECV_OUT)
            wav_infos.sample_rate      = paudio->ecnr_recv_sample_rate_out; /*fix me*/
        else
            wav_infos.sample_rate      = 16000;
        wav_infos.bytes_per_sample     = 2;
        wav_infos.num_channels         = 1;
        wav_infos.num_samples          = (unsigned int)(wav_infos.num_bytes / wav_infos.bytes_per_sample );


        sprintf(rec->rec_file_name, "%s_%s_%dch_%dkHz_%s_LE_%d.wav",  file_prefix,
                                                                       rec->rec_out_dev,
                                                                       wav_infos.num_channels,
                                                                       (wav_infos.sample_rate / 1000),
                                                                       "S16",
                                                                       ecnr_audio_recording_export_rec_idx);

        pr_debug("record file:%s\n",rec->rec_file_name);

        /* replace ":" charactere with "_"*/
        pfound = strpbrk (rec->rec_file_name, key);
        while (pfound != NULL)
        {
            *pfound = 0x5F;  /* "_" */
            pfound = strpbrk (pfound+1,key);
        }

        pr_message("************** print wav header infos ***************\n");
        pr_message("wav_infos.num_bytes         : %lu\n", wav_infos.num_bytes );
        pr_message("wav_infos.sample_rate       : %d\n", wav_infos.sample_rate );
        pr_message("wav_infos.bytes_per_sample  : %d\n", wav_infos.bytes_per_sample );
        pr_message("wav_infos.num_channels      : %d\n", wav_infos.num_channels );
        pr_message("wav_infos.num_samples       : %d\n", wav_infos.num_samples );
        pr_message("************** print wav header infos ***************\n");

        /* create wave file */
        ecnr_audio_rts_write_wav(rec->rec_file_name, &wav_infos, rec->AudRamBufChannels_out);
    }

    return;
}

void ecnr_audio_recording_export_rec(enum ecnr_audio ecnr_id)
{
    pr_message("request recording export for channel -> %d \n", ecnr_id);

    switch(ecnr_id)
    {
        case ECNR_MIC_OUT:
        {
            ecnr_audio_recording_export_pb_dev_rec(ecnr_id, &mic_rec_desc, D_MICRO_RECORD_FILE_OUT_PCM);
            break;
        }
        case ECNR_REF_IN0:
        case ECNR_REF_IN1:
        case ECNR_MULTI_CHAN_IN_REF_0:
        case ECNR_MULTI_CHAN_IN_REF_1:
        {
            ecnr_audio_recording_export_cap_dev_rec(ecnr_id, &ref_rec_desc, D_REF_RECORD_FILE_IN_PCM);
            break;
        }
        case ECNR_MULTI_CHAN_IN_DEV:
        case ECNR_MIC_IN0:
        case ECNR_MIC_IN1:
        {
            ecnr_audio_recording_export_cap_dev_rec(ecnr_id, &mic_rec_desc, D_MICRO_RECORD_FILE_IN_PCM);
            break;
        }
        case ECNR_RECV_IN:
        {
            ecnr_audio_recording_export_cap_dev_rec(ecnr_id, &rcv_rec_desc, D_RCV_RECORD_FILE_IN_PCM);
            break;
        }
        case ECNR_RECV_OUT:
        {
            ecnr_audio_recording_export_pb_dev_rec(ecnr_id, &rcv_rec_desc, D_RCV_RECORD_FILE_OUT_PCM);
            break;
        }
        default:
            break;
    }

    return;
}

