/* =======================================================================================
 *  Project          SSE (Speech Signal Enhancement)
 *  (c) Copyright    2004-2010
 *  Company          Nuance Communications
 *                   All rights reserved
 *  Secrecy Level    STRICTLY CONFIDENTIAL
 * --------------------------------------------------------------------------------------*/
/**
 *  @internal
 *  @file            sse_err.h
 *
 *  SSE specific error codes.
 *
 *  This header file contains all SSE specific error codes.
 */
/* =====================================================================================*/

/** @addtogroup ERROR Error Codes
  * @{*/

#ifndef __SSE_ERR_H__
#define __SSE_ERR_H__

#ifdef __cplusplus
 extern "C" {
#endif


/*----------------------------------------------------------------------------*/
/*              Non-fatal error codes (0x0000 - 0x000F)                       */
/*----------------------------------------------------------------------------*/

    /** No error. */
#define SSE_ERR_OK                          0x0000

    /** Result not ready.
        Returned by sseAudioOut() when audio data is not ready for output
        (sseAudioOut is called out of sequence with sseAudioIn()). */
#define SSE_ERR_NO_RESULT                   0x0001

    /** Size information only is valid.
        Returned by the get data functions when a null is input for the data
        output pointer, and valid size information is returned for the
        specified data ID.  */
#define SSE_ERR_SIZE_ONLY                   0x0002

/*----------------------------------------------------------------------------*/
/*              General error codes (0x0100 - 0x01FF)                         */
/*----------------------------------------------------------------------------*/

    /** A memory allocation error has occurred. */
#define SSE_ERR_ALLOC                       0x0100

    /** An illegal null pointer function argument has been used. */
#define SSE_ERR_NULL_POINTER                0x0101

    /** Handle not initialized.
        Returned by sseProcess() if @c pMain is null.
        Otherwise indicates an attempt to access functionality that is not
        initialized (@a sseOff at initialization).
        In addition, submodule functions will return this error if called
        when not initialized. */
#define SSE_ERR_MODULE_NOT_INIT             0x0102

    /** Handle not created.
        Returned by sseInitialize() if sseCreate() is not called successfully
        first. */
#define SSE_ERR_NOT_CREATED                 0x0103

    /** Handle already initialized.
        Returned by sseCreate() if @c Main points to a non-null value.
        Returned by sseInitialize() if @c pMain points to a fully-initialized
        @a tsseMain structure (sseInitialize has been called twice).
        In addition, submodule initialization functions will return this error
        if called incorrectly. */
#define SSE_ERR_MODULE_ALREADY_INIT         0x0104

    /** Invalid SSE handle.
        Returned by the SSE API functions if SSE has not been properly initialized before. */
#define SSE_ERR_INVALID_MAIN_STRUCT  0x0105

    /** Deinitialization failure.
        Returned by sseDestroy() if an error occurs during destruction. */
#define SSE_ERR_DEINIT_FAILED               0x0106

    /** Illegal set request.
        Returned by the set functions if the specified data ID is
        read-only or the requested parameter change is not allowed. */
#define SSE_ERR_ILLEGAL_SET_PARAM           0x0108

    /** Illegal set request before initialization.
        Returned by the set data functions when the specified data ID can only
        be set after sseInitialize() has been called. */
#define SSE_ERR_ILLEGAL_SET_PARAM_BEFORE_INIT 0x0109

    /** Illegal set request after initialization.
        Returned by the set data functions when the specified data ID must be
        set before sseInitialize() is called. */
#define SSE_ERR_ILLEGAL_SET_PARAM_AFTER_INIT 0x010A

    /** Invalid data ID.
        Returned by the set/get data functions when the specified data ID
        is invalid. */
#define SSE_ERR_INVALID_PARAM_ID            0x010B

    /** Invalid channel number.
        Returned by the set/get or callback functions when the channel
        number is invalid for the specified data ID. The channel
        number should be between 0 and the maximum of valid channels
        for which the specified parameter exists, minus 1. */
#define SSE_ERR_INVALID_CHANNEL_NUMBER      0x010C

    /** Invalid operation mode for set request.
        Returned by the set functions if the requested parameter change
        is illegal given the current operation mode.
        Also returned if a set data function is called between
        sseAudioIn() and sseAudioOut(). */
#define SSE_ERR_INVALID_OPERATION_MODE      0x010D

    /** Invalid size information.
        Returned by the set/get functions when the size is invalid for
        the specified data ID, or if internal scratch buffers are too small for the
        requested data access.
        Also returned by #ssePrepareDataForASR when the provided buffers are too small. */
#define SSE_ERR_INVALID_SIZE                0x010E

    /** Invalid parameter.
        Returned at initialization if there is an error in the configuration
        parameters.
        Returned by the set data functions if there was an error in the data
        for the specified data ID.
        Also returned by sseSetDataString() if there was an error parsing
        the string information. */
#define SSE_ERR_PARAM                       0x010F

    /** Invalid number of parameters.
        Returned by sseSetDataString() when the number of parsed parameters
        conflicts with the expected value. */
#define SSE_ERR_PARAM_CNT                   0x0110

    /** Invalid sample rate. */
#define SSE_ERR_SAMPLE_RATE                 0x0111

    /** Invalid frame shift. */
#define SSE_ERR_FRAMESHIFT                  0x0112

    /** Invalid number of input or output channels. */
#define SSE_ERR_INVALID_CHAN_CNT            0x0113

    /** Invalid FFT length. */
#define SSE_ERR_FFT_LENGTH                  0x0114

    /** Invalid filter length. */
#define SSE_ERR_INVALID_FILTER_LEN          0x0115

    /** Error loading prototype or other filter coefficients. */
#define SSE_ERR_NULL_FILTER                 0x0116

    /** Internal parameter table error. */
#define SSE_ERR_PARAMETER_TABLE             0x0117

    /** The requested functionality is not implemented. */
#define SSE_ERR_NOT_IMP                     0x0118

    /** The requested functionality is disabled by a preprocessor switch. */
#define SSE_ERR_SUBMODULE_DISABLED          0x0119

    /** Invalid endianness for data used with @c sse_PostConfig. */
#define SSE_ERR_INVALID_SET_PARAM_ENDIANNESS 0x011A

    /** Initialization of memory manager failed. */
#define SSE_ERR_INIT_MMGR                   0x011B

/** Invalid external memory type.
    sseCreate() only accepts values of type #tsseExtMem. */
#define SSE_ERR_INVALID_EXT_MEM_TYPE        0x011C

/** Invalid optimization level
    sseCreate() accepts only optimization levels of type #tsseMemOptLevel. */
#define SSE_ERR_INVALID_OPT_LEVEL           0x011D

/** Error handling the debug I/O message queue. Maybe the initialization failed. */
#define SSE_ERR_DBGIO_MSG_QUEUE             0x011E

/** Invalid message type in the debug input buffer or message queue. */
#define SSE_ERR_DBGIO_MSG_TYPE              0x011F

/** Debug I/O output buffer too small. */
#define SSE_ERR_DBGIO_OUTBUF_SIZE           0x0120

/** Internal scratch buffer is too small. */
#define SSE_ERR_INSUFFICIENT_SCRATCH        0x0121

/** Invalid instance Id in the debug input buffer or message queue. Different
    instance Ids have probably been used for registering callbacks with Debug I/O. */
#define SSE_ERR_DBGIO_INST_ID               0x0122

/** A debug I/O message does not fit into the internal message buffer. */
#define SSE_ERR_INTERNAL_MSG_BUF_TOO_SMALL  0x0123

/** Invalid debug I/O message format. */
#define SSE_ERR_DBGIO_MSG_FORMAT            0x0124

/** All ASRModes except for sseAudioOnly require a post processing with ssePrepareDataForASR
    after each sseProcess call. This error indicates that ssePrepareDataForASR has not been called
    after the previous sseProcess call. It is also returned by ssePrepareDataForASR when it is
    called twice on the same audio frame. */
#define SSE_ERR_ASR_POST_PROC               0x0125

/** General error with the debug I/O submodule. */
#define SSE_ERR_DBGIO                       0x0126

/** Buffer overflow with signal injection. */
#define SSE_ERR_SIGIN_OVERFLOW              0x0127


/*----------------------------------------------------------------------------*/
/*              Callback / DataExchange error codes (0x0200 - 0x02FF)         */
/*----------------------------------------------------------------------------*/

    /** The data set ID used in a callback label is invalid. */
#define SSE_ERR_INVALID_DATASET_ID          0x0200

    /** The module ID used in a callback label is invalid. */
#define SSE_ERR_INVALID_MODULE_ID           0x0201

    /** Unspecified error when preparing the callback data. It could be an unexpected NULL
        pointer, an invalid data ID, or data dimension. */
#define SSE_ERR_INTERNAL_CALLBACK_ERROR     0x0202

    /** Invalid data control word (used for preprocessing of the callback data). */
#define SSE_ERR_INVALID_DATA_CTRL           0x0203



/*----------------------------------------------------------------------------*/
/*              BSD error codes (0x0300 - 0x03FF)                             */
/*----------------------------------------------------------------------------*/

    /** Invalid bsd uni_io ident code. */
#define SSE_ERR_BSD_INVALID_IDENT           0x0300

    /** Invalid bsd data type. */
#define SSE_ERR_BSD_INVALID_FORMAT          0x0301

    /** Invalid bsd structure version. */
#define SSE_ERR_BSD_INVALID_STRUCT_VERSION  0x0302

    /** Error in bsd structure definition. */
#define SSE_ERR_BSD_INVALID_STRUCT_FORMAT   0x0303

    /** Improperly formatted bsd data or
        (internal) invalid bsd function argument. */
#define SSE_ERR_BSD_INVALID_PARAM           0x0304

    /** Invalid pointer or buffer size specified in @a tsseBsd or
        @a tsseConfig. */
#define SSE_ERR_BSD_INVALID_BUFFER_PARAM    0x0305

    /** Too many buffers (max is 3) specified in @a tsseBsd or
        @a tsseConfig. */
#define SSE_ERR_BSD_MAX_BSD_CNT_EXCEEDED    0x0306

    /** Mismatch in configuration information.
        Returned by sseInitialize() when there is a mismatch in the bsd
        data, tsseConfig data and/or configuration specified by sseSetData()
        for any of sample rate, frame shift or mic count. */
#define SSE_ERR_CONFIG_BSD_MISMATCH         0x0307

    /** Bsd checksum error. */
#define SSE_ERR_BSD_CHECKSUM                0x0308

    /** Error during bsd read. */
#define SSE_ERR_BSD_READ                    0x0309

    /** Error during bsd write. */
#define SSE_ERR_BSD_WRITE                   0x030A

    /** EOF occurred while reading bsd header. */
#define SSE_ERR_BSD_EOF                     0x030B

    /** Invalid version in bsd header. */
#define SSE_ERR_BSD_VERSION                 0x030C

    /** Invalid number of dimensions for bsd parameter. */
#define SSE_ERR_BSD_INVALID_DIM             0x030D

    /** Error occurred during UIO processing. */
#define SSE_ERR_BSD_UIO                     0x030E

    /** Invalid context for bsd processing call (internal). */
#define SSE_ERR_BSD_INVALID_CONTEXT         0x030F

    /** Error during bsd decompression. */
#define SSE_ERR_BSD_DECOMPRESS              0x0310

/*----------------------------------------------------------------------------*/
/*              Acoustic echo cancellation error codes (0x0400 - 0x040F)      */
/*----------------------------------------------------------------------------*/

    /** Invalid data for shadow subband parameters. */
#define SSE_ERR_INVALID_BACKBAND            0x0400

    /** Invalid number of reference channels: Must be equal to 1. */
#define SSE_ERR_NUM_REF_CHAN                0x0401

    /** Invalid AEC filter length. */
#define SSE_ERR_AEC_FILTER_LEN              0x0402

/*----------------------------------------------------------------------------*/
/*              Limiter initialization error codes (0x0500 - 0x050F)          */
/*----------------------------------------------------------------------------*/

    /** Invalid limiter delay: Must be larger than 0 and less than 100. */
#define SSE_ERR_LIMITER_DELAY               0x0500

    /** Invalid limiter threshold: Must be between 1 and +32767. */
#define SSE_ERR_LIMITER_THRESHOLD           0x0501

    /** Invalid limiter maximum sustain time: Must be at least 0 milliseconds. */
#define SSE_ERR_LIMITER_SUSTAIN_MAX         0x0502

    /** Invalid limiter release time: Must be at least 1 millisecond. */
#define SSE_ERR_LIMITER_RELEASE_TIME        0x0503

/*----------------------------------------------------------------------------*/
/*              Timer (profiling) error codes (0x0600 - 0x06FF)               */
/*----------------------------------------------------------------------------*/

    /** The specified timer has not been created. */
#define SSE_ERR_TIMER_NOT_CREATED           0x0600

    /** The handle for the specified timer is not valid. */
#define SSE_ERR_INVALID_TIMER_HANDLE        0x0601

    /** The handle for the timer is not unique. */
#define SSE_ERR_TIMER_HANDLE_NOT_UNIQUE     0x0602

/*----------------------------------------------------------------------------*/
/*              Diagnostic error codes (0x0700 - 0x07FF)                      */
/*----------------------------------------------------------------------------*/

    /** Diagnostic activity in progress.  Wait for current test to complete
        or stop current test before starting new test. */
#define SSE_ERR_DIAGNOSTIC_ACTIVE           0x0700

    /** Diagnostic result data is not ready.  Test may have not been started
        or may currently be in progress.  Also returned if diagnostic module
        is switched off. */
#define SSE_ERR_DIAGNOSTIC_NOT_READY        0x0701

#ifdef __cplusplus
 }
#endif

#endif /* __SSE_ERR_H__ */
/**@}*/
