/******************************************************************************
 * FILE        : ecnr-types.h
 * PROJECT     : Gen3 and Gen4
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2019 RBEI
 * HISTORY     :
 * Date        | Author                 | Modification
 * 15.05.2019  | RBEI/ECO Ranjit Susal  | initial version
 *             |                        | ECNR specific data type handling
 *****************************************************************************/
 
#ifndef __ECNR_TYPES_H__
#define __ECNR_TYPES_H__

#include <stdint.h>


#define ECNR_MAX_CONFIG_DATA_CNT  3 /* this value is taken from SSE_MAX_CONFIG_DATA_CNT */
#define ECNR_MAX_AUDIO_CNT        4 /* this value is taken from SSE_MAX_AUDIO_CNT */
#define ECNR_MAX_CONFIG_COMMENT_LENGTH    100 /*this value is maximum additional comment length retreived from SSE for a .scd file */

typedef int8_t            tI8;
typedef int32_t           tI32;



/** Structure used to input configuration information with two-stage initialization */
typedef struct tecnrConfigData
{
   /** Number of configuration data buffers to load, must be <= ECNR_MAX_CONFIG_DATA_CNT   */
   unsigned short int ConfigBufferCnt;

   /** Array of pointers to configuration data buffers that contain parameters
       for the configuration of the SSE module. Unused pointers must be NULL     */
   char* ConfigBuffer[ECNR_MAX_CONFIG_DATA_CNT];
#ifdef D_ECNR_LIB_SSE_4_03
   int32_t* i32ConfigBuffer[ECNR_MAX_CONFIG_DATA_CNT];
#endif
   /** Array of BSD buffer sizes in bytes                                        */
   unsigned int ConfigBufferSize[ECNR_MAX_CONFIG_DATA_CNT];
} tecnrConfigData;



/** Data structure for input/output data from/to the tuning tool */
typedef struct tecnrTuneIO
{
   unsigned int    InBufferSize;      /**< Overall size of the input buffer in bytes                */
   unsigned int    InBufferUsedSize;  /**< Number of bytes actually being used in the input buffer  */
   void*           InBuffer;          /**< Pointer to the input buffer                              */
   unsigned int    OutBufferSize;     /**< Overall size of the output buffer in bytes               */
   unsigned int    OutBufferUsedSize; /**< Number of bytes actually being used in the output buffer */
   void*           OutBuffer;         /**< Pointer to the output buffer                             */
} tecnrTuneIO;


/** Pointer array for audio input / output buffers used in
 *  sseProcess().-> SSE engine
 *  qwaAudioIn().-> QWA engine
 *  */
typedef short int *tecnrAudio[ECNR_MAX_AUDIO_CNT];


/**  Switch for the operation mode (hands-free and voice recognition).     */
typedef enum tecnrOperationMode
{
   ecnrHandsFree       = 1,           /**< hands-free mode                                */
   ecnrVoiceRecog      = 2,           /**< voice recognition mode                         */
   ecnrPassThrough     = 4,           /**< \if CREATE_INTERNAL_DOC
                                            pass through mode: copy data from input to
                                            output buffer without processing (legacy)
                                         \endif */
   ecnrPassThroughHF   = 5,           /**< \if CREATE_INTERNAL_DOC
                                            pass through mode: perform hands-free
                                            processing, but copy data from input to output (legacy)
                                         \endif */
   ecnrPassThroughVR   = 6,           /**< \if CREATE_INTERNAL_DOC
                                            pass through mode: perform voice recognition
                                            processing, but copy data from input to output (legacy)
                                         \endif */
   ECNR_ENFORCE_32BIT_05 = 0x7FFFFFFF /* force compiler to use four bytes for this enum */

} tecnrOperationMode;


/**  information provided for ASR below enumeration applicable only for (SSE 4.x library)*/
typedef enum tecnrSideInfoMode
{
   ecnrNoMetaInfo       = 1,    /**  No meta information                         */
   ecnrLimitedInfo      = 2,    /**  Limited meta information is provided
                                     (to be used for VoCon ASR version < 4.11)   */
   ecnrEnhancedInfo     = 3,    /**  Enhanced meta information is provided
                                     (recommended for VoCon ASR version >= 4.11) */
} tecnrSideInfoMode;







#endif /* __ECNR_TYPES_H__ */
