/******************************************************************************
 * FILE        : ecnr-alsa.h
 * PROJECT     : Gen3 and Gen4
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2013 RBCM GMBH
 * HISTORY     :
 * Date        | Author                 | Modification
 * 21.08.2013  | Patrick Rey            | initial version for Gen3
 *             |                        | porting from QWA ALSA module.
 *****************************************************************************/

#ifndef __ECNR_ALSA_H__
#define __ECNR_ALSA_H__


#include <alsa/asoundlib.h>


/***************************************/
/* section include engines definitions */
/***************************************/

#include "ecnr-types.h"
#include "ecnr-common-defs.h"



/*******************************************/
/* section include ADIT RTS API definition */
/*******************************************/
#include"rts.h"

#define NUM_ECNR_CH_MAX 3
#define NUM_ECNR_MAX_RTS_DEV_ORDER 6
#define NUM_ECNR_MAX_RTS_STREAM 10
#define RTS_DEFAULT_SILENCE_PREFILL 8

enum ecnr_audio_rts_adev_order {
  ECNR_ADEV_RTS_1,
  ECNR_ADEV_RTS_2,
  ECNR_ADEV_RTS_3,
  ECNR_ADEV_RTS_4,
  ECNR_ADEV_RTS_5,
  ECNR_ADEV_RTS_6,
  NUM_ECNR_RTS_AUDIO_DEV
};

enum ecnr_audio_rts_stream {
  ECNR_STREAM_RTS_1,
  ECNR_STREAM_RTS_2,
  ECNR_STREAM_RTS_3,
  ECNR_STREAM_RTS_4,
  ECNR_STREAM_RTS_5,
  ECNR_STREAM_RTS_6,
  NUM_ECNR_RTS_AUDIO_STREAM
};


#define NUM_ALSA_INPUTS  5  /* inputs: MicIn0, MicIn1, RecvIn, RefIn0, RefIn1 */
#define NUM_ALSA_OUTPUTS 3  /* outputs: MicOut, RecvOut, MicOutDup */

struct ecnr_alsa_stream {
  gchar *dev_name;
  snd_pcm_stream_t dir;
  guint num_channels;
  guint rate;    /* actual rate */
  snd_pcm_uframes_t period_size;  /* actual period size */
  guint dev_order; /* order to open the device, required by RTS */
  guint dev_order_fixed;
  guint dev_idx;
  guint dev_active_ch_nb;
  guint dev_active_chans[NUM_ECNR_CH_MAX];
};

struct ecnr_alsa_buffer {
  short **stream_buffer;
  struct ecnr_alsa_stream *stream;
  enum ecnr_audio ecnr_id;
  guint stream_channel;
  gulong num_samples;      /* number of samples in ECNR buffer */
};

typedef struct _asr_mode_data
{
      gboolean  isASRMode;
     short int* pi16InputBuffer;
  unsigned int  u32InputBufferSize;
     short int* pi16OutputBuffer;
  unsigned int  u32OutputBufferSize;
  unsigned int* pu32WrittenElemCnt;
}asr_mode_data;


typedef struct _stecnr_alsa_state {

  /* alsa streaming thread element*/
  GThread   *thread;
  GMutex    lock;
  GCond     update;
  gboolean  athread_active;
  guint32   athread_create_count;
  guint32   athread_terminate_count;
  guint32   devices_restart_after_xrun_count;

  /*
  * The following members are shared between threads and are locked by
  * the above mutex.
  */

  gboolean  run;
  gboolean  req_stop;
  gboolean  exit;
  gboolean  initialized;  /* true if ALSA channels have been initialized */
  guint     retry;
  gboolean  writeSilence;
  guint     silenceWriteCount;
  gchar     *alsa_dev_name[NUM_ECNR_AUDIO];
  guint     num_channels[NUM_ECNR_AUDIO];
  guint     channel[NUM_ECNR_AUDIO];
  guint     dev_order[NUM_ECNR_AUDIO];
  gboolean  dev_configuration_ctrl[NUM_ECNR_AUDIO];
  guint     prefill;
  int       alsa_thread_priority;
  gboolean  ecnr_alsa_multi_chan_in_dev_active;
  gboolean  enable_dev_configuration;
  gboolean  enable_force_dev_fill_per_feature_setting;
  gboolean  ecnr_alsa_prefill_ctrl;  // user has changes prefill per ECNR method call, this control is set to false to disable default setting within code
  gboolean  ecnr_alsa_dev_order_ctrl;
  gboolean  req_statistic;
  gboolean  req_start_stop_audio;

  /*
  * The following members are not shared between threads and thus are not
  * locked by the mutex.
  */

  /* copies of some of the ECNR parameters for quick reference */
  guint     ecnr_mic_in_cnt;  /* number of MicIn inputs (1 or 2) */
  guint     ecnr_mic_proc_cnt; /* number of MicProc outputs (0 or 1) */
  guint     ecnr_recv_in_cnt;  /* number of RecvIn inputs (0 or 1) */
  guint     ecnr_ref_in_cnt;  /* number of RefIn inputs (0 or 1) */
  guint     ecnr_frame_shift_in;  /* samples per period for MicIn */
  guint     ecnr_frame_shift_out;  /* samples per period for MicOut */
  guint     ecnr_recv_frame_shift_in;  /* samples per period for RecvIn */
  guint     ecnr_recv_frame_shift_out;  /* samples per period for RecvOut */
  guint     ecnr_ref_frame_shift_in;  /* samples per period for RefIn */
  guint     ecnr_sample_rate_in;  /* sample rate for MicIn */
  guint     ecnr_sample_rate_out;  /* sample rate for MicOut */
  guint     ecnr_recv_sample_rate_in;  /* sample rate for RecvIn */
  guint     ecnr_recv_sample_rate_out;  /* sample rate for RecvOut */
  guint     ecnr_ref_sample_rate_in;  /* sample rate for RefIn */
  gint      ecnr_op_mode;
  gint      ecnr_asr_mode;
  gboolean  ecnr_enable_asr_preprocessing;
  gchar*    ecnr_asr_data_type;
  guint32   ecnr_asr_data_type_size;
  gboolean  ecnr_dup_mic_in_0;

  gboolean  ecnr_mic_lvl_watch;
  guint32   ecnr_mic_lvl;

  /* the ALSA input and output devices */
  struct ecnr_alsa_stream *in_stream[NUM_ALSA_INPUTS];
  struct ecnr_alsa_stream *out_stream[NUM_ALSA_OUTPUTS];

  /* descriptors for copying samples between ALSA and ECNR buffers */
  struct ecnr_alsa_buffer *in_buffer[NUM_ALSA_INPUTS];
  struct ecnr_alsa_buffer *out_buffer[NUM_ALSA_OUTPUTS];

  tecnrAudio *MicIn;
  tecnrAudio *RecvIn;
  tecnrAudio *RefIn;
  tecnrAudio *MicOut;
  tecnrAudio *RecvOut;
  short int *AsrPrepare;


  /**********************
     * RTS data structure
     *********************/
  trts_cfg  *rstdata;
  short int* rstbuf_i[NUM_ECNR_MAX_RTS_STREAM];
  short int* rstbuf_o[NUM_ECNR_MAX_RTS_STREAM];
  short int* rstSilence_o[NUM_ECNR_MAX_RTS_STREAM];
  /* ASR data structure*/
  asr_mode_data stAsr;


  /* ECNR tune engine data */
  tecnrTuneIO *pTuneIO;

  gboolean ecnr_tuning_active;

}stecnr_alsa_state;


/* ALSA module functions*/
extern gboolean             ecnr_alsa_new(void);
extern void                 ecnr_alsa_delete(void);
extern void                 ecnr_alsa_finalize(void);
extern void                 ecnr_alsa_initialize(void);
extern void                 ecnr_alsa_reinitialize(void);

/* ALSA audio functions */
extern gint                 ecnr_alsa_audio_thread_new (void);
extern gboolean             ecnr_alsa_start(glong timeout_ms);
extern gboolean             ecnr_alsa_stop(glong timeout_ms);
extern gboolean             ecnr_alsa_exit_wait(glong timeout_ms);
extern gboolean             ecnr_alsa_get_audio_run_state(void);
extern stecnr_alsa_state*   ecnr_alsa_get_instance(void);
extern gboolean             ecnr_alsa_audio_active_is(void);
extern void                 ecnr_alsa_audio_started(void);
extern void                 ecnr_alsa_voice_stream_change_state(ecnr_audio_state ecnr_current_state);

/* ALSA configuration functions */
extern void                 ecnr_alsa_set_dev_name(enum ecnr_audio ecnr_id, const gchar * name);
extern gchar*               ecnr_alsa_get_dev_name(enum ecnr_audio ecnr_id);
extern void                 ecnr_alsa_set_num_channels(enum ecnr_audio ecnr_id, guint num_channels);
extern guint                ecnr_alsa_get_num_channels(enum ecnr_audio ecnr_id);
extern void                 ecnr_alsa_set_channel(enum ecnr_audio ecnr_id, guint channel);
extern guint                ecnr_alsa_get_channel(enum ecnr_audio ecnr_id);
extern void                 ecnr_alsa_set_thread_priority(int thread_priority);
extern int                  ecnr_alsa_get_thread_priority(void);
extern int                  ecnr_alsa_set_mic_level_watch(gboolean watch_state);
extern gboolean             ecnr_alsa_get_mic_level_watch(void);
       int                  ecnr_alsa_mic_level_calculate(void);
extern guint32              ecnr_alsa_get_mic_level(void);
extern int                  ecnr_alsa_enable_sse_debug(void);
extern int                  ecnr_alsa_disable_sse_debug(void);
extern gboolean             ecnr_alsa_is_sse_debug_active(void);
extern int                  ecnr_alsa_print_dev_properties(void);
extern int                  ecnr_alsa_disable_dev_configuration(enum ecnr_audio ecnr_id);
extern int                  ecnr_alsa_enable_dev_configuration(enum ecnr_audio ecnr_id);
extern int                  ecnr_alsa_prefill_ctrl(gboolean prefill);
extern int                  ecnr_alsa_dev_order_ctrl(gboolean dev_order);
extern void                 ecnr_alsa_print_current_audio_state(void);
extern void                 ecnr_alsa_request_print_current_audio_state(void);
extern void                 ecnr_alsa_set_device_open_order(enum ecnr_audio ecnr_id, guint dev_order);
extern void                 ecnr_alsa_set_device_prefill_value(guint prefill);
extern void                 ecnr_alsa_set_mutli_channel_in_stream_mode(gboolean multi_chan);
extern guint                ecnr_alsa_get_frame_shift_in(void);
extern guint                ecnr_alsa_get_sample_rate_in(void);
extern gint                 ecnr_alsa_asr_mode_state(void);
extern gint                 ecnr_alsa_notify_asr_mode_state(void);
extern void                 ecnr_alsa_set_scheduling_priority(void);
extern gint                 ecnr_alsa_audio_initialize(void);





#endif
