//
// MessageTypes.cpp
//
//  Created on: Jan 22, 2015
//      Author: Martin Koch, Fa. ESE
//


#include "Node_if.h"


#define ETG_S_IMPORT_INTERFACE_GENERIC
#include <etg_if.h>


#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_GENIVI_CCA_NODE
#include "trcGenProj/Header/MessageTypes.cpp.trc.h"


namespace FIMessaging
{

   // ==========================================================================
   //
   //                              F I - M e s s a g e
   //
   // Transport-independent class for incoming messages,
   // interpretable with FI-based object
   //

   /* constructor */ FIMessage:: FIMessage (uint16_t serviceId, const tFIVersionInfo& version
         , uint16_t functionId, uint8_t opCode, uint16_t responseID
         , uint8_t* pPayLoad_, uint32_t length)
      : u16ServiceID(serviceId)
      , fiVersion(version)
      , u16FunctionID(functionId)
      , u8OpCode(opCode)
      , u16ResponseID(responseID)
      , pPayload(pPayLoad_)
      , u32PayloadLength(length)
   {

   }

   // --------------------------------------------------------------------------

   /* constructor */ FIMessage:: FIMessage (const FunctionInfo& fctInfo, uint16_t responseID
        , uint8_t* payload, uint32_t length)
      : u16ServiceID(fctInfo.service.u16ServiceID)
      , fiVersion(fctInfo.service.fiVersion)
      , u16FunctionID(fctInfo.u16FunctionID)
      , u8OpCode(fctInfo.u8OpCode)
      , u16ResponseID(responseID)
      , pPayload(payload)
      , u32PayloadLength(length)
   {

   }

   // --------------------------------------------------------------------------

   /* copy constructor */ FIMessage:: FIMessage (const FIMessage& rhs)
      : u16ServiceID(rhs.u16ServiceID)
      , fiVersion(rhs.fiVersion)
      , u16FunctionID(rhs.u16FunctionID)
      , u8OpCode(rhs.u8OpCode)
      , u16ResponseID(rhs.u16ResponseID)
      , pPayload(rhs.pPayload)
      , u32PayloadLength(rhs.u32PayloadLength)
   {

   }

   // --------------------------------------------------------------------------

   FIMessage FIMessage:: DeepCopy () const
   {
      FIMessage retVal(*this);

      // create copy of duplicate of current payload on heap
      retVal.pPayload = new uint8_t[u32PayloadLength];
      memcpy(retVal.pPayload, pPayload, u32PayloadLength);

      ETG_TRACE_USR4(("FIMessage::vDeepCopy: created %u bytes at %x for function %x.%x.%u"
            , u32PayloadLength, retVal.pPayload, u16ServiceID, u16FunctionID, u8OpCode))

      return retVal;
   }

   // --------------------------------------------------------------------------

   void FIMessage:: vDestroy ()
   {
      ETG_TRACE_USR4(("FIMessage::vDestroy(): destroyed %u bytes at %x for function %x.%x.%u"
            , u32PayloadLength, pPayload, u16ServiceID, u16FunctionID, u8OpCode))

      delete[] pPayload;
      pPayload = NULL;
      u32PayloadLength = 0;
   }

   // --------------------------------------------------------------------------

   bool FIMessage:: bPopulateFIData(fi_tclMessageBase& oFiMsg) const
   {
      // validate preconditions
      if (NULL == pPayload)
      {
         ETG_TRACE_FATAL(("FIMessage::bPopulateFIData - E R R O R : invalid pointer to payload"))
         return false;
      }
      else if ((u16ServiceID != oFiMsg.u16GetServiceID()) || (u16FunctionID != oFiMsg.u16GetFunctionID()) || (u8OpCode != oFiMsg.u8GetOpCode()))
      {
         ETG_TRACE_FATAL(("FIMessage::bPopulateFIData - E R R O R : FI data object does not match: is %x.%x.%x - required: %x.%x.%x"
               , oFiMsg.u16GetServiceID(), oFiMsg.u16GetFunctionID(), oFiMsg.u8GetOpCode()
               , u16ServiceID, u16FunctionID, u8OpCode))
         return false;
      }

      fi_tclTypeBase& oFiData = oFiMsg.rfoGetTypeBase();

      //create context to read a message from a given buffer
      fi_tclInContext oIn(pPayload, pPayload + u32PayloadLength, fiVersion.u16MajorVersion, oFiData.enGetByteOrder());

      // populate given FI object
      (void)oFiData.oRead(oIn);

      if (oIn.bIsValid())
         return true;
      else
      {
         ETG_TRACE_FATAL(("FIMessage::bPopulateFIData - E R R O R parsing FI data object %x.%x.%x"
               , oFiMsg.u16GetServiceID(), oFiMsg.u16GetFunctionID(), oFiMsg.u8GetOpCode()))
         return false;
      }
   }

   // ==========================================================================
   //
   //                   E m p t y - fi_tclMessage   class
   //
   // helper class to generate any sendable fi-object with binary payload
   //

   /* virtual destructor */ fi_tclEmptyMsg:: ~fi_tclEmptyMsg ()
   {

   }

   // --------------------------------------------------------------------------
   // fi_tclMessageBase implementation

   /* virtual */ tU16 fi_tclEmptyMsg:: u16GetServiceID () const
   {
      return _info.service.u16ServiceID;
   }

   // --------------------------------------------------------------------------

   /* virtual */ tU16 fi_tclEmptyMsg:: u16GetFunctionID () const
   {
      return _info.u16FunctionID;
   }

   // --------------------------------------------------------------------------

   /* virtual */ tU8  fi_tclEmptyMsg:: u8GetOpCode () const
   {
      return _info.u8OpCode;
   }

   // --------------------------------------------------------------------------

   /* virtual */ const fi_tclTypeBase& fi_tclEmptyMsg:: corfoGetTypeBase () const
   {
      return *this;
   }

   // --------------------------------------------------------------------------

   /* virtual */ fi_tclTypeBase& fi_tclEmptyMsg:: rfoGetTypeBase ()
   {
      return *this;
   }

   // --------------------------------------------------------------------------
   // fi_tclTypeBase implementation

   /* virtual */ tU32 fi_tclEmptyMsg:: u32GetSize (tU16 /* u16MajorVersion */) const
   {
      return 0;
   }

   // --------------------------------------------------------------------------

   /* virtual */ tS32 fi_tclEmptyMsg:: s32GetTypeId () const
   {
      return (tS32)((_info.u16FunctionID << 8) + _info.u8OpCode);
   }

   // --------------------------------------------------------------------------

   /* virtual */ fi_tclOutContext& fi_tclEmptyMsg:: oWrite (fi_tclOutContext& oOut) const
   {
      return oOut;
   }

   // --------------------------------------------------------------------------

   /* virtual */ fi_tclInContext& fi_tclEmptyMsg:: oRead (fi_tclInContext& oIn)
   {
      return oIn;
   }

   // ==========================================================================
   //
   //                 B i n a r y - fi_tclMessage   class
   //
   // helper class to generate any sendable fi-object with binary payload
   //

   /* virtual destructor */ fi_tclBinaryMsg:: ~fi_tclBinaryMsg ()
   {
      fi_tclBinaryMsg::vDestroy();
   }

   // --------------------------------------------------------------------------

   /* virtual */ tU32 fi_tclBinaryMsg:: u32GetSize (tU16 /* u16MajorVersion */) const
   {
      return _payloadSize;
   }

   // --------------------------------------------------------------------------

   /* virtual */ fi_tclOutContext& fi_tclBinaryMsg:: oWrite (fi_tclOutContext& oOut) const
   {
      if (_pPayload && _payloadSize)
         for (unsigned i = 0; i < _payloadSize; ++i)
            (void) oOut.oWrite(_pPayload[i]);

      return oOut;
   }

   // --------------------------------------------------------------------------

   /* virtual */ fi_tclInContext& fi_tclBinaryMsg:: oRead (fi_tclInContext& oIn)
   {
      if (_pLocalPayload)
      {
         delete[] _pLocalPayload;
         _pLocalPayload = NULL;
      }

      _payloadSize = oIn.u32GetFreeSize();
      _pLocalPayload = new tU8[_payloadSize];
      _pPayload = _pLocalPayload;
      for (unsigned i = 0; _pLocalPayload && (i < _payloadSize); ++i)
         (void) oIn.oRead(_pLocalPayload[i]);

      return oIn;
   }

   // --------------------------------------------------------------------------

   /* virtual */ tVoid fi_tclBinaryMsg:: vDestroy ()
   {
      if (_pLocalPayload)
      {
         delete[] _pLocalPayload;
         _pLocalPayload = NULL;
      }
      _pPayload = NULL;
   }

   // --------------------------------------------------------------------------

   /* virtual */ fi_tclIOContextBase::tenByteOrder fi_tclEmptyMsg:: enGetByteOrder() const
   {
      return fi_tclIOContextBase::EN_LITTLE;
   }

   // --------------------------------------------------------------------------
   //
   //                  E r r o r - fi_tclMessage   class
   //
   // helper class to generate any sendable error message fi-object
   //

   /* virtual destructor */ fi_tclErrorMsg:: ~fi_tclErrorMsg ()
   {

   }

   // --------------------------------------------------------------------------
   // fi_tclTypeBase implementation

   /* virtual */ tU32 fi_tclErrorMsg:: u32GetSize (tU16 /* u16MajorVersion */) const
   {
      return 2;
   }

   // --------------------------------------------------------------------------

   /* virtual */ fi_tclOutContext& fi_tclErrorMsg:: oWrite (fi_tclOutContext& oOut) const
   {
      if (oOut.u32GetFreeSize() < 2)
         // complain
         return oOut;

      tU8 lowByte  = (tU8)(u16ErrorCode & 0x00ff);
      tU8 highByte = (tU8)((u16ErrorCode & 0xff00) >> 8);
      (void) oOut.oWrite(lowByte)
                  .oWrite(highByte);

      return oOut;
   }

   // --------------------------------------------------------------------------

   /* virtual */ fi_tclInContext& fi_tclErrorMsg:: oRead (fi_tclInContext& oIn)
   {
      if (oIn.u32GetFreeSize() < 2)
         // complain
         return oIn;

      tU8 lowByte;
      tU8 highByte;
      (void) oIn.oRead(lowByte)
                 .oRead(highByte);
      u16ErrorCode = (tU16)((highByte << 8) + lowByte);

      return oIn;
   }

   // --------------------------------------------------------------------------

}  // namespace FIMessaging


