//
// Service.cpp
//
//  Created on: Jan 12, 2015
//      Author: Martin Koch, Fa. ESE
//

#include "CCA_Node.h"
#include "CCA_Service.h"
// - - - - - - - - - - - -

#include "CCA_MessageTrackerBase.h"

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include <etg_if.h>


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GENIVI_CCA_NODE
#include "trcGenProj/Header/CCA_Service.cpp.trc.h"


namespace FIMessaging  {  namespace CCA
{
   // ==========================================================================
   //
   //        private helper class   M e s s a g e - T r a  c k e r
   //

   class Service::MessageTracker : public MessageTrackerBase
   {
      public:

         inline /* constructor */ MessageTracker (const ServiceInfo& info)
            : MessageTrackerBase(info)  {  }
         virtual /* destructor */ ~MessageTracker ()  { }

         tU16 u16Append (const amt_tclServiceData& roInMsg);  // will return new u16ResponseID
         MessageInfo oRemoveEntry (uint16_t u16ResponseID, uint16_t u16FunctionID, uint8_t  u8OpCode);
   };

   // --------------------------------------------------------------------------

   tU16 Service::MessageTracker:: u16Append (const amt_tclServiceData& oInMsg)
   {
      // will return new u16ResponseID

      tU16 u16ResponseID;
      unsigned listSize;
      {
         Node::AutoLock lock;

         u16ResponseID = u16CreateResponseID();
         _list.push_back(PendingCall (MessageInfo(oInMsg, serviceInfo), u16ResponseID));
         listSize = (unsigned)_list.size();

      }  // end of AutoLock scope

      ETG_TRACE_USR4(("Service::MessageTracker::u16Append() - inserted ID %u for function %x.%x.%u from App %x; size is now %u"
            , u16ResponseID
            , oInMsg.u16GetServiceID(), oInMsg.u16GetFunctionID(), oInMsg.u8GetOpCode()
            , oInMsg.u16GetSourceAppID()
            , listSize))

      return u16ResponseID;
   }

   // --------------------------------------------------------------------------

   MessageInfo Service::MessageTracker:: oRemoveEntry (uint16_t u16ResponseID, uint16_t u16FunctionID, uint8_t  u8OpCode)
   {
      bool success = false;
      unsigned listSize = 0;
      MessageInfo retVal;

      {
         Node::AutoLock lock;

         for (Iterator it = _list.begin(); it != _list.end(); ++it)
            if (u16FunctionID != it->messageInfo.function.u16FunctionID)
               continue;
            else if (u16ResponseID != it->u16ResponseID)
               continue;
            else if (bResponseOpCodeMatchesRequest (u8OpCode, it->messageInfo.function.u8OpCode))
            {
               retVal = it->messageInfo;
               retVal.function.u8OpCode = u8OpCode;

               if ((u8OpCode != CCA_C_U8_OPCODE_METHODRESULTFIRST) && (u8OpCode != CCA_C_U8_OPCODE_METHODRESULTMIDDLE))
                  (void) _list.erase(it);  // don't remove these; we expect more to come

               success = true;
               listSize = (unsigned)_list.size();

               break;
            }

         // use this opportunity to get rid of old stuff
         vDropOverdues();

      }  // end of AutoLock scope

      if (success)
         ETG_TRACE_USR4(("Service::MessageTracker::RemoveEntry() - removed ID %u for function %x.%x.%u from App %x.%x; list size is now %u"
               , u16ResponseID
               , serviceInfo.u16ServiceID, u16FunctionID, u8OpCode
               , retVal.remoteApp.u16TargetAppID, retVal.remoteApp.u16TargetSubID
               , listSize))
      else
         ETG_TRACE_USR4(("Service::MessageTracker::RemoveEntry() - E R R O R : ID %u not found for function %x.%x.%u; list size is %u"
               , u16ResponseID
               , serviceInfo.u16ServiceID, u16FunctionID, u8OpCode
               , listSize))

      return retVal;
   }

   // ==========================================================================
   //
   //       private helper class   N o t i f i c a t i o n - T a b l e
   //

   class   Service:: NotificationTable
   {
      public:
         struct Item
         {
            uint16_t    u16FunctionID;
            // client registration parameters
            RemoteAppInfo remoteApp;
            // values from UpReg message
            uint16_t    u16CmdCounter;
            uint8_t     u8ACT;

            /* constructor */ Item (const amt_tclServiceData& oSvcData);
            bool /* comparison */ operator== (const Item& rhs);
            // use compiler-generated destructor and assignment operator
         };
         typedef std::vector<Item> Entries;
         typedef Entries::iterator  Iterator;

         void vAppend (const amt_tclServiceData& oSvcData);
         void vRemove (const amt_tclServiceData& oSvcData);
         void vRemoveAllEntriesForRegisterID (uint16_t u16RegID);
         void vGetEntriesForFID (uint16_t u16FunctionID, /* out */ Entries& externalTable);

      private:
         // member variables
         Entries  _table;
   };

   // --------------------------------------------------------------------------
   //  Notification table item

   /* constructor */ Service::NotificationTable::Item:: Item (const amt_tclServiceData& oUpRegMsg)
      : u16FunctionID(oUpRegMsg.u16GetFunctionID())
      , remoteApp(oUpRegMsg.u16GetSourceAppID(), oUpRegMsg.u16GetSourceSubID(), oUpRegMsg.u16GetRegisterID())
      , u16CmdCounter(oUpRegMsg.u16GetCmdCounter())
      , u8ACT(oUpRegMsg.u8GetACT())
   {

   }

   // --------------------------------------------------------------------------

   bool /* comparison */ Service::NotificationTable::Item:: operator== (const Item& rhs)
   {
      if (u16FunctionID != rhs.u16FunctionID)
         return false;
      else if (remoteApp.u16TargetAppID != rhs.remoteApp.u16TargetAppID)
         return false;
      else if (remoteApp.u16TargetSubID != rhs.remoteApp.u16TargetSubID)
         return false;
      else if (remoteApp.u16RegisterID != rhs.remoteApp.u16RegisterID)
         return false;
      // skip comparing command counter and ACT

      return true;
   }

   // --------------------------------------------------------------------------
   //  Notification table

   void Service::NotificationTable:: vAppend (const amt_tclServiceData& oUpRegMsg)
   {
      ETG_TRACE_USR2(("NotificationTable<Svc %x>::vAppend() - registering FID %x for application %x.%d"
            , oUpRegMsg.u16GetServiceID(), oUpRegMsg.u16GetFunctionID()
            , oUpRegMsg.u16GetSourceAppID(), oUpRegMsg.u16GetSourceSubID()))

      {  Node::AutoLock lock;

         _table.push_back(Item (oUpRegMsg));
      }  // end of AutoLock scope
   }

   // --------------------------------------------------------------------------

   void Service::NotificationTable:: vRemove (const amt_tclServiceData& oRelUpregMsg)
   {
      Item rhs(oRelUpregMsg);
      bool bSuccess = false;

      {  Node::AutoLock lock;

         for (Iterator it = _table.begin(); it != _table.end(); ++it)
            if ((*it) == rhs)  // compare FunctionID, AppID, SubID and RegisterID as defined in comparison operator
            {
               _table.erase(it);
               bSuccess = true;
               break;
            }

      }  // end of AutoLock scope

      if (bSuccess)
         ETG_TRACE_USR2(("NotificationTable<Svc %x>::vRemove() - removed registration for FID %x for application %x.%d"
               , oRelUpregMsg.u16GetServiceID(), rhs.u16FunctionID
               , rhs.remoteApp.u16TargetAppID, rhs.remoteApp.u16TargetSubID))
      else
         ETG_TRACE_ERR(("NotificationTable<Svc %x>::vRemove() - Warning : failed removing registration for FID %x for application %x.%d"
               , oRelUpregMsg.u16GetServiceID(), rhs.u16FunctionID
               , rhs.remoteApp.u16TargetAppID, rhs.remoteApp.u16TargetSubID))
   }

   // --------------------------------------------------------------------------

   void Service::NotificationTable:: vRemoveAllEntriesForRegisterID (uint16_t u16RegID)
   {
      unsigned count = 0;

      {  Node::AutoLock lock;

         for (Iterator it = _table.begin(); it != _table.end(); /* ++it */)
            if (it->remoteApp.u16RegisterID == u16RegID)
            {
               ++count;
               it = _table.erase(it);
            }
            else
               ++it;

      }  // end of AutoLock scope

      ETG_TRACE_USR2(("NotificationTable::vRemoveAllEntriesForRegisterID() - removing %u registrations for registration-ID %d"
            , count, u16RegID))
   }

   // --------------------------------------------------------------------------

   void Service::NotificationTable:: vGetEntriesForFID (uint16_t u16FunctionID, /* out */ Entries& externalTable)
   {
      {  Node::AutoLock lock;

         externalTable.reserve(_table.size());
         for (Iterator it = _table.begin(); it != _table.end(); ++it)
            if (it->u16FunctionID == u16FunctionID)
               externalTable.push_back(*it);

      }  // end of AutoLock scope

      ETG_TRACE_USR3(("NotificationTable::vGetEntriesForFID() - found %u registrations function-ID %x"
            , externalTable.size(), u16FunctionID))
   }

   // ==========================================================================
   //
   //           struct   M e m o r i z e d   R e g i s t e r   M s g
   //

   /* default constructor */ Service::tMemorizedRegisterMsg ::tMemorizedRegisterMsg()
   {
      // initialize with invalid
      memset(&byte[0], 0xff, ServiceRegisterSize);
   }

   // ==========================================================================
   //
   //                       class   C C A - S e r v i c e
   //

   /* constructor */ Service:: Service (Node& node, ITarget& externalTarget)
      : TargetBase(node, externalTarget)
      , _bServiceAvailable(false)
      , _pNotifications(new NotificationTable)
      , _pMessageTracker(new MessageTracker(serviceInfo))
   {
      ETG_TRACE_USR1(("Service constructor for Service %X with version {%u, %u}"
            , serviceInfo.u16ServiceID, serviceInfo.fiVersion.u16MajorVersion, serviceInfo.fiVersion.u16MinorVersion))
   }

   // --------------------------------------------------------------------------

   /* virtual destructor */ Service:: ~Service ()
   {
      ETG_TRACE_USR1(("Service<%X> shutting down ...", serviceInfo.u16ServiceID))

      delete _pMessageTracker;
      delete _pNotifications;
   }

   // --------------------------------------------------------------------------

   /* virtual */ void Service:: vOnNewAppState (tU32 u32NewAppState)
   {
      // callback function for incoming messages

      if (_u32AppState == u32NewAppState)
         return;

      tU32 u32LastAppState = _u32AppState;
      _u32AppState = u32NewAppState;

      ETG_TRACE_USR1(("Service<%x.%u> changing state: %u --> %u"
            , serviceInfo.u16ServiceID, u16OwnSubID
            , ETG_ENUM(ail_u32CCAState, u32LastAppState)
            , ETG_ENUM(ail_u32CCAState, u32NewAppState) ))

      if (u32LastAppState == u32NewAppState)
         return;  // unchanged
      else if (_bServiceAvailable == (AMT_C_U32_STATE_NORMAL == u32NewAppState))
         return;  // no change in availability

      // need to initialize if we are called from App::poAddTarget() after its connection to SPM
      if (    (AMT_C_U32_STATE_UNINITALIZED == u32LastAppState)
           && (AMT_C_U32_STATE_INITIALIZED != u32NewAppState) )
         _externalTarget.vSetAssociatedSender(this);

      if (AMT_C_U32_STATE_NORMAL == u32NewAppState)
      {
         _externalTarget.vOnConnect();
      }
      else
      {
         _externalTarget.vOnDisconnect();
         _node.vServiceAvailabilityChanged(serviceInfo.u16ServiceID, AMT_C_U8_SVCSTATE_NOT_AVAILABLE);
         _bServiceAvailable = false;
      }
   }

   // --------------------------------------------------------------------------

   /* virtual */ void Service:: vOnNewMessage (amt_tclServiceData& oSvcData)
   {
      // callback function for incoming messages (overrides TargetBase)

      // callback function for incoming messages
      ETG_TRACE_USR1(("Service(%4x.%d)::vOnNewMessage(): ServiceData message received for FunctionID 0x%x with OpCode %u"
                 , ETG_ENUM(ail_u16ServiceId, oSvcData.u16GetServiceID()), oSvcData.u16GetSourceSubID()
                 , oSvcData.u16GetFunctionID(), oSvcData.u8GetOpCode()))

      // validate preconditions
      if (oSvcData.u16GetServiceID() != serviceInfo.u16ServiceID)
      {
         ETG_TRACE_FATAL(("Service::vOnNewMessage - E R R O R : ServiceID of message does not match: is %x"
               , serviceInfo.u16ServiceID))
         return;
      }

      if ((NULL == _pNotifications) || (NULL == _pMessageTracker))
      {
         ETG_TRACE_FATAL(("Service::vOnNewMessage - E R R O R : invalid pointer _pNotifications = %x, _pMessageTracker %x"
               , _pNotifications, _pMessageTracker))
         return;
      }

      // handle property notification requests
      tU8 u8OpCode = oSvcData.u8GetOpCode();
      if (u8OpCode == CCA_C_U8_OPCODE_UPREG)
         _pNotifications->vAppend(oSvcData);
      else if (u8OpCode == CCA_C_U8_OPCODE_RELUPREG)
         _pNotifications->vRemove(oSvcData);

      uint16_t u16ResponseID = _pMessageTracker->u16Append(oSvcData);

      // transform and dispatch to external target
      uint32_t payloadOffset = AMT_C_U32_BASEMSG_ABSMSGSIZE + AMT_C_U32_SVCDATA_RELMSGSIZE; // skip 32 byte message header
      uint8_t* pPayload = oSvcData.pu8GetSharedMemBase() + payloadOffset;
      uint32_t length = oSvcData.u32GetSize();
      if (length >= payloadOffset)
         length -= payloadOffset;
      else
      {
         ETG_TRACE_FATAL(("Service(%4x.%d)::vOnNewMessage() - E R R O R : ServiceData message too short for FunctionID 0x%x with OpCode %u"
                    , ETG_ENUM(ail_u16ServiceId, oSvcData.u16GetServiceID()), oSvcData.u16GetSourceSubID()
                    , oSvcData.u16GetFunctionID(), oSvcData.u8GetOpCode()))
         return;
      }

      FIMessage fiMsg(serviceInfo.u16ServiceID, serviceInfo.fiVersion, oSvcData.u16GetFunctionID(), oSvcData.u8GetOpCode(), u16ResponseID
            , pPayload, length);
      _externalTarget.vOnNewMessage(fiMsg);
   }


   // --------------------------------------------------------------------------

   /* virtual */ uint16_t Service:: u16PostFICommand (const fi_tclMessageBase& fiMsg)
   {
      ETG_TRACE_FATAL(("Service<%x>::u16PostFICommand - E R R O R  sending command message %x.%u not allowed"
            , fiMsg.u16GetServiceID(), fiMsg.u16GetFunctionID(), fiMsg.u8GetOpCode() ))

      return 0xffff;
   }

   // --------------------------------------------------------------------------

   /* virtual */ void Service:: vSetServiceAvailable (bool bIsAvailable)
   {
      ETG_TRACE_USR1(("Service(%4x.%d)::vSetServiceAvailable(): Service now %s"
                 , ETG_ENUM(ail_u16ServiceId, serviceInfo.u16ServiceID), u16OwnSubID
                 , (bIsAvailable ? "available" : "unavailable")))

      tU8 u8Availability = (bIsAvailable ? AMT_C_U8_SVCSTATE_AVAILABLE : AMT_C_U8_SVCSTATE_NOT_AVAILABLE);
      _bServiceAvailable = bIsAvailable;
      _node.vServiceAvailabilityChanged(serviceInfo.u16ServiceID, u8Availability);
   }

   // --------------------------------------------------------------------------

   /* virtual */ bool Service:: bPostFIResult (const fi_tclMessageBase& fiMsg, uint16_t responseID)
   {
      ETG_TRACE_USR2(("Service<%x>::bPostFIResult() - sending result message %x.%x with responseID %d"
            , serviceInfo.u16ServiceID, fiMsg.u16GetFunctionID(), fiMsg.u8GetOpCode(), responseID))

      if (NULL == _pMessageTracker)
         return false;
      else if (serviceInfo.u16ServiceID != fiMsg.u16GetServiceID())
      {
         ETG_TRACE_FATAL(("Service<%x>::bPostFIResult() - E R R O R :  serviceID does not match for requested result message %x.%x.%x with responseID %d"
               , serviceInfo.u16ServiceID, fiMsg.u16GetServiceID(), fiMsg.u16GetFunctionID(), fiMsg.u8GetOpCode(), responseID))
          return false;
      }

      MessageInfo info = _pMessageTracker->oRemoveEntry(responseID, fiMsg.u16GetFunctionID(), fiMsg.u8GetOpCode());
      _node.vPostFIMessage(info, fiMsg);

      if (AMT_C_U16_APPID_INVALID == info.remoteApp.u16TargetAppID)
         return false;
      else if (AMT_C_U16_REGID_INVALID == info.remoteApp.u16RegisterID)
         return false;
      else
         return true;
   }

   // --------------------------------------------------------------------------

   /* virtual */ bool Service:: bNotifyClients (const fi_tclMessageBase& oStatusMsg)
   {
      if (NULL == _pNotifications)
         return false;

      // get a local copy of the notification table
      NotificationTable::Entries entries;
      _pNotifications->vGetEntriesForFID(oStatusMsg.u16GetFunctionID(), entries);

      ETG_TRACE_USR2(("Service<%x>::bNotifyClients() - sending property update message for FID %x to %u clients"
            , serviceInfo.u16ServiceID, oStatusMsg.u16GetFunctionID(), entries.size()))

      for (NotificationTable::Iterator it = entries.begin(); it != entries.end(); ++it)
      {
         FunctionInfo func(oStatusMsg, serviceInfo.fiVersion);
         MessageInfo info(it->remoteApp, func, it->u16CmdCounter, it->u8ACT);
         _node.vPostFIMessage(info, oStatusMsg);
      }

      return true;
   }

   // --------------------------------------------------------------------------

   // instance of list of postponed ServeiceRegister requests

   /* static */ std::list<Service::tMemorizedRegisterMsg> Service:: delayedRegisterMessages;

   // --------------------------------------------------------------------------

   /* static */ void Service:: vPostponeRegisterMsg (const amt_tclServiceRegister& regMsg)
   {
      // must be called under protection of App::AutoLock

      delayedRegisterMessages.push_back(tMemorizedRegisterMsg());
      memcpy(&(delayedRegisterMessages.back().byte[0]), regMsg.pu8GetSharedMemBase(), sizeof(tMemorizedRegisterMsg));

      ETG_TRACE_USR2(("vPostponeRegisterMsg() - successfully stored %u. ServiceRegister for %x from Application %x.%d"
            , delayedRegisterMessages.size()
            , regMsg.u16GetServiceID(), regMsg.u16GetSourceAppID(), regMsg.u16GetSourceSubID()))
   }

   // --------------------------------------------------------------------------

   void Service:: vGetPostponedRegisters (PendingRegistrationList& extPendingRegistrations)
   {
      // must be called under protection of App::AutoLock

      if (0 == delayedRegisterMessages.size())
         ETG_TRACE_USR4(("Service<%x>::vGetPostponedRegisters() - delayedRegisterMessages empty"
               , serviceInfo.u16ServiceID))

      PendingRegistrationList::iterator it;
      for (it = delayedRegisterMessages.begin(); it != delayedRegisterMessages.end() ; /* ++it */)
      {
         // extract ServiceID from stream as big endian
         tU16 u16ServiceID = (tU16)(it->byte[AMT_C_U32_SVSUNREGISTER_SERVICEID] + (it->byte[1 + AMT_C_U32_SVSUNREGISTER_SERVICEID] << 8));

         if (serviceInfo.u16ServiceID == u16ServiceID)
         {
            // copy message to list owned by caller
            extPendingRegistrations.push_back(*it);
            memcpy(&(extPendingRegistrations.back().byte[0]), &(it->byte[0]), sizeof(tMemorizedRegisterMsg));

            ETG_TRACE_USR4(("Service<%x>::vGetPostponedRegisters() - retrieved %u. ServiceRegister"
                  , serviceInfo.u16ServiceID, extPendingRegistrations.size()))

            // remove this entry from our static list
            it = delayedRegisterMessages.erase(it);
            continue;  // skip incrementing iterator
         }
         else
            ETG_TRACE_USR4(("Service<%x>::vGetPostponedRegisters() - ignoring ServiceRegister for Service %x"
                  , serviceInfo.u16ServiceID, u16ServiceID))

         ++it;
      }
   }

   // --------------------------------------------------------------------------

}  }  // namespace FIMessaging::CCA




