//
// CCA_Node.h
//
//  Created on: Dec 5, 2014
//      Author: Martin Koch, Fa. ESE
//
//  Oct. 2015
//   - remove intermediate Rx-thread owned by AIL
//   - extend Singleton to support one instance per Application-ID
//   - merge former CCA-App into CCA_Node
//

#ifndef CCA_NODE_H_
#define CCA_NODE_H_

#include "Node_if.h"


#define AIL_S_IMPORT_INTERFACE_GENERIC
#include <ail_if.h>  // implicitly links generic <amt_if.h>

#include <set>
#include <map>


namespace FIMessaging  { namespace CCA
{

   // --------------------------------------------------------------------------

   class TargetBase;
   class Client;

   typedef std::set<TargetBase*> tTargetList;

   // --------------------------------------------------------------------------
   //
   //                               C C A - L o c a l s
   //

   struct RemoteAppInfo
   {
      tU16 u16TargetAppID;
      tU16 u16TargetSubID;
      tU16 u16RegisterID;

      // some construction shorthands
      inline /* default constructor */ RemoteAppInfo ();
      inline /* copy constructor */ RemoteAppInfo (const RemoteAppInfo& rhs);
      inline /* constructor */ RemoteAppInfo (tU16 appID, tU16 subID, tU16 regID);
      inline /* constructor */ RemoteAppInfo (const amt_tclServiceData& oSvcData);
   };

   // --------------------------------------------------------------------------

   struct MessageInfo
   {
      RemoteAppInfo remoteApp;
      FunctionInfo  function;
      tU16 u16CmdCounter;
      tU8  u8ACT;

      // some construction shorthands
      inline /* default constructor */ MessageInfo ();
      inline /* copy constructor */ MessageInfo (const MessageInfo& rhs);
      inline /* constructor */ MessageInfo (const RemoteAppInfo& app, const FunctionInfo& func, tU16 cmdCounter, tU8 act);
      inline /* constructor */ MessageInfo (const amt_tclServiceData& oSvcData, const ServiceInfo& serviceInfo);
   };

   // --------------------------------------------------------------------------
   //
   //                       class   C C A - N o d e
   //

   class Node  : public INode, public ail_tclOneThreadAppInterface
   {
      public:
         static INode* pGetInstance (uint16_t u16ApplicationID, const char* pRegistry, IRxPollingThread& externalWorkerThread);
         virtual void vReleaseInstance ();

         using ail_tclAppAdapt::u16GetAppId;

         // Mutex-based helper class, synchronizing multi-threaded access to
         // dynamic variables in entire application
         class AutoLock
         {
            public:
               /* constructor */ AutoLock();
               /* destructor */ ~AutoLock();
            private:
               static pthread_mutex_t _mutex;
               OSAL_tMSecond _startTime;
         };

         // INode implementation
         virtual ISender* poAddTarget (ITarget& externalTarget);
         virtual void vRemoveTarget (ITarget& externalTarget);
         virtual bool bPollAndDispatchMessage ();

         // send outgoing message
         virtual void vPostFIMessage (MessageInfo info, const fi_tclMessageBase& fiMsg);

      private:
         explicit /* constructor */ Node (uint16_t u16ApplicationID, IRxPollingThread& externalPollingThread);
         /* destructor */ ~Node ();

         static void vOnQueueCallback (void* pArg);

         // Special handling for ServiceRegister for a Service which has not revealed itself yet
         bool bWantsUnrevealedService (amt_tclBaseMessage* poMessage);

         // AIL overrides
         virtual void vAppEntry ();
      protected:
         virtual tBool bGetServiceVersion (tU16 u16ServiceID, tU16& rfu16MajorVersion, tU16& rfu16MinorVersion, tU16& rfu16PatchVersion);
      public:
         virtual tBool bOnInit ();
         virtual tVoid vOnApplicationClose ();
         virtual tVoid vOnNewAppState (tU32 u32OldAppState, tU32 u32AppState);
         virtual tVoid vOnAsyncRegisterConfExt (tU16 u16RegisterId, tU16 u16ServerAppId, tU16 u16ServiceId, tU16 u16TargetSubId, tU8 u8AsyncRegisterConfStatus);
         virtual tVoid vOnServiceState (tU16 u16ServiceId, tU16 u16ServerId, tU16 u16RegisterId, tU8 u8ServiceState, tU16 u16SubId);
         virtual tVoid vOnNewMessage (amt_tclBaseMessage* poMessage);

         // member variables
      private:
         size_t _instanceCount;
         bool _bQueueCallbackInstalled;
         bool _bMyQueueCreated;
         bool _bBreakLoop;
         OSAL_tMQueueHandle _hMyInQueue;
         OSAL_tEventHandle _hEventHandle;
         tTargetList _targets;
         IRxPollingThread& _externalPollingThread;

         typedef std::map<tU16, Node*> InstanceMap_t;
         static InstanceMap_t _instances;
   };

   // --------------------------------------------------------------------------

   inline /* default constructor */ RemoteAppInfo:: RemoteAppInfo ()
      : u16TargetAppID(AMT_C_U16_APPID_INVALID)
      , u16TargetSubID(AMT_C_U16_SUBID_INVALID)
      , u16RegisterID(AMT_C_U16_REGID_INVALID)
   {  }

   inline /* copy constructor */ RemoteAppInfo:: RemoteAppInfo (const RemoteAppInfo& rhs)
      : u16TargetAppID(rhs.u16TargetAppID)
      , u16TargetSubID(rhs.u16TargetSubID)
      , u16RegisterID(rhs.u16RegisterID)
   {  }

   inline /* constructor */ RemoteAppInfo:: RemoteAppInfo (tU16 appID, tU16 subID, tU16 regID)
      : u16TargetAppID(appID)
      , u16TargetSubID(subID)
      , u16RegisterID(regID)
   {  }

   inline /* constructor */ RemoteAppInfo:: RemoteAppInfo (const amt_tclServiceData& oSvcData)
      : u16TargetAppID(oSvcData.u16GetSourceAppID())
      , u16TargetSubID(oSvcData.u16GetSourceSubID())
      , u16RegisterID(oSvcData.u16GetRegisterID())
   {  }

   // --------------------------------------------------------------------------

   inline /* default constructor */ MessageInfo:: MessageInfo ()
      : remoteApp()
      , function()
      , u16CmdCounter(0)
      , u8ACT(0)
   {  }

   inline /* copy constructor */ MessageInfo:: MessageInfo (const MessageInfo& rhs)
      : remoteApp(rhs.remoteApp)
      , function(rhs.function)
      , u16CmdCounter(rhs.u16CmdCounter)
      , u8ACT(rhs.u8ACT)
   {  }

   inline /* constructor */ MessageInfo:: MessageInfo (const RemoteAppInfo& app, const FunctionInfo& func, tU16 cmdCounter, tU8 act)
      : remoteApp(app)
      , function(func)
      , u16CmdCounter(cmdCounter)
      , u8ACT(act)
   {  }

   inline /* constructor */ MessageInfo:: MessageInfo (const amt_tclServiceData& oSvcData, const ServiceInfo& serviceInfo)
      : remoteApp(RemoteAppInfo(oSvcData))
      , function(FunctionInfo(serviceInfo, oSvcData.u16GetFunctionID(), oSvcData.u8GetOpCode()))
      , u16CmdCounter(oSvcData.u16GetCmdCounter())
      , u8ACT(oSvcData.u8GetACT())
   {  }

   // --------------------------------------------------------------------------

}  }  // namespace FIMessaging::CCA


#endif /* CCA_NODE_H_ */
