//
// CCA_Node.cpp
//
//  Created on: Dec 9, 2014
//      Author: Martin Koch, Fa. ESE
//


#include <stdint.h>

#include "CCA_Node.h"
// - - - - - - - - - - -

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include <etg_if.h>

#include <iostream>


#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_GENIVI_CCA_NODE
#include "trcGenProj/Header/CCA_Node.cpp.trc.h"


namespace FIMessaging  {  namespace CCA
{

   // --------------------------------------------------------------------------
   //
   //                        F a c t o r y - F u n t i o n
   //

   INode* pGetNodeInstance (uint16_t u16ApplicationID, const char* pRegistry, IRxPollingThread& externalWorkerThread)
   {
      // delegate to class-internal factory function
      return Node::pGetInstance(u16ApplicationID, pRegistry, externalWorkerThread);
   }

   // --------------------------------------------------------------------------

}  }  // namespace FIMessaging::CCA



#include "CCA_Client.h"
#include "CCA_Service.h"

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include <scd_if.h>

#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include <fi_msgfw_if.h>


// -----------------------------------------------------------------------------


namespace FIMessaging   {  namespace CCA
{

   // --------------------------------------------------------------------------
   //
   //                      C C A - N o d e   implementation
   //

   /* static */ Node::InstanceMap_t Node:: _instances;
   /* static */ pthread_mutex_t Node::AutoLock:: _mutex = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;

   // --------------------------------------------------------------------------

   /* static */ INode* Node:: pGetInstance (uint16_t u16AppID, const char* pRegistry, IRxPollingThread&  externalWorkerThread)
   {
      Node* retVal = NULL;
      {
         AutoLock lock;

         // initialize framework if this is the first call
         if (_instances.empty())
         {
            scd_init();

            NORMAL_M_ASSERT(amt_bInit() == TRUE);
         }

         // return matching instance if exists
         InstanceMap_t::iterator it = _instances.find(u16AppID);
         if (it != _instances.end())
         {
            retVal = (*it).second;
            retVal->_instanceCount++;
         }
      } // end of AutoLock scope

      if (retVal)
      {
         ETG_TRACE_USR1(("CCA_Node<%x>::pGetInstance() found existing instance at %08x, count = %u"
               , u16AppID, retVal, retVal->_instanceCount))
         return retVal;
      }

      // - - - - - - - - -

      // load registry snippet if given
      if (pRegistry)
      {
         OSAL_tIODescriptor  regHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_REGISTRY, OSAL_EN_READWRITE);
         if (regHandle != OSAL_ERROR )
         {
            if ((OSAL_s32IOControl(regHandle, OSAL_C_S32_IOCTRL_BUILD_REG, (intptr_t)pRegistry)) == OSAL_OK)
               ETG_TRACE_USR1(("CCA_App::pGetInstance() REGISTRY loaded: %s!", pRegistry))
            else
               ETG_TRACE_FATAL(("!!!!!!!!!! FAILED to load REGISTRY %s!", pRegistry))
         }
         OSAL_s32IOClose(regHandle);
      }

      // create new Application
      retVal = new Node(u16AppID, externalWorkerThread);
      ETG_TRACE_USR1(("CCA_Node<%x>::pGetInstance() created new instance at %08x, count = %u"
            , u16AppID, retVal, (retVal ? retVal->_instanceCount : 0)))
      return retVal;
   }

   // --------------------------------------------------------------------------

   /* virtual */ void Node:: vReleaseInstance ()
   {
      ETG_TRACE_USR1(("CCA_Node<%x>::vReleaseInstance() at %08x, count = %u"
            , u16AppId, this, _instanceCount))

      {
         AutoLock lock;

         if (--_instanceCount)
            return;  // other instances left

         // if corresponding entry is in map, remove it
         InstanceMap_t::iterator it = _instances.find(u16GetAppId());
         if ((it != _instances.end()) && ((*it).second == this))
            _instances.erase(it);

      } // end of AutoLock scope

      // once we reach here, no more instances are out
      delete this;
   }

   // --------------------------------------------------------------------------

   /* constructor */ Node:: Node (uint16_t u16AppID, IRxPollingThread& externalPollingThread)
      : INode()
      , ail_tclOneThreadAppInterface(AIL_C_U32_ENTRY_JUMP_STACKSIZE, AIL_C_U32_ENTRY_DESTROY_STACKSIZE)
      , _instanceCount(1)
      , _bQueueCallbackInstalled(false)
      , _bMyQueueCreated(false)
      , _bBreakLoop(false)
      , _hMyInQueue(OSAL_C_INVALID_HANDLE)
      , _hEventHandle(OSAL_C_INVALID_HANDLE)
      , _targets()
      , _externalPollingThread(externalPollingThread)
   {
      // add to instance map
      {
         AutoLock lock;

         _instances.insert(std::pair<tU16, Node*>(u16AppID, this));
      }  // end of AutoLock scope


      ETG_TRACE_USR1(("CCA_App instance created for AppID %x at %08x", u16AppID, this))

      // create queue
      _bMyQueueCreated = scd_bCreateQueue( u16AppID );
      ETG_TRACE_USR1(("_bMyQueueCreated() created = %u for AppID %x at %08x"
         , _bMyQueueCreated, u16AppID, this))

      // Initialize AIL for use with external thread
      OSAL_tThreadID hExternalDispatcherThreadID = OSAL_ThreadWhoAmI();
      tBool bSuccess = ail_tclOneThreadAppInterface::bInitInstance(0, u16AppID, hExternalDispatcherThreadID);
      if ( ! bSuccess)
      {
         ETG_TRACE_FATAL(("CCA_Node<%x> Constructor() - E R R O R :  bInitInstance() failed!", u16AppId))
      }

      // install callback for incoming messages
      tS32 s32Installed = OSAL_s32MessageQueueNotify(hMyInQueue, (OSAL_tpfCallback)vOnQueueCallback, this);
      if (s32Installed == OSAL_OK)
      {
         ETG_TRACE_COMP(("CCA_Node<%x> Constructor() - Install of message queue callback successful", u16AppId))
         _bQueueCallbackInstalled = true;
      }
      else
         ETG_TRACE_FATAL(("CCA_Node<%x> Constructor() - E R R O R :  could not install message queue callback (error: %s)"
            , u16AppId, OSAL_coszErrorText(OSAL_u32ErrorCode()) ))

      // process any message which is received before installing our callback function
      Node::bPollAndDispatchMessage();

   }

   // --------------------------------------------------------------------------

   /* destructor */ Node:: ~Node ()
   {
      // unregister queue callback function
      if (_bQueueCallbackInstalled)
      {
         // remove callback for incoming messages if installed
         if (OSAL_s32MessageQueueNotify(hMyInQueue, NULL, NULL) == OSAL_OK)
              ETG_TRACE_COMP(("CCA_Node<%x> Destructor() : Remove of message queue callback successfully"
                    , u16AppId ))
         else
              ETG_TRACE_ERRMEM(("CCA_Node<%x> Destructor() : could not remove message queue callback (error: %s)"
                    , u16AppId, OSAL_coszErrorText(OSAL_u32ErrorCode()) ))
      }

      // cleanup base class
      ail_tclOneThreadAppInterface::vDeinitInstance();

//      amt_bDeInit()
   }

   // --------------------------------------------------------------------------

   tVoid Node:: vAppEntry ()
   {
      // This function shall not be called, as it is for use with AIL-owned worker thread

      ETG_TRACE_SYS_MIN(("CCA_Node<%x> vAppEntry() - E R R O R : erroneously entered vAppEntry() - thread terminating ...", u16AppId ));

      // release all basic class objects && inform LPX of correct shutdown
      vTerminateMySelf ();
   }

   // --------------------------------------------------------------------------

   /* virtual */ bool Node:: bPollAndDispatchMessage ()
   {
      // process CCA messages

      ETG_TRACE_USR1(("CCA_Node<%x> bPollAndDispatchMessage() called from ThreadID = %u with hMyInQueue = %u"
         , u16AppId, OSAL_ThreadWhoAmI(), hMyInQueue ));

      amt_tclBaseMessage          oMsgObject;
      tU32                    u32Prio;
      bool  bContinueDispatching = true;
      bool  bMessageAvailable = true;

      // process all CCA messages
      while (bContinueDispatching && bMessageAvailable)
      {
         // get messages without wait
         bMessageAvailable = ail_bIpcMessageWait(u16AppId, hMyInQueue, &oMsgObject, &u32Prio, OSAL_C_TIMEOUT_NOBLOCKING);
         if (bMessageAvailable)
         {
             // special handling for ServiceRegister for a Service which has not revealed itself yet
             bool bNeedsSpecialHandling = bWantsUnrevealedService(&oMsgObject);

             if ( ! bNeedsSpecialHandling)
                 bContinueDispatching = poInternalDispatch->bDispatchCCAMessages(&oMsgObject, OSAL_C_INVALID_HANDLE, FALSE, u32Prio);
         }
         else
         {
            if (OSAL_u32ErrorCode() != OSAL_E_TIMEOUT)
            {
               ETG_TRACE_ERRMEM(("CCA_Node<%x>::  ail_bIpcMessageWait returns error for handle %08x an message (%s)"
                      , u16AppId, (tUInt)hMyInQueue, OSAL_coszErrorText(OSAL_u32ErrorCode()) ));
            }
         }
      }

      return bContinueDispatching;
   }

   // ==========================================================================

   bool Node:: bWantsUnrevealedService (amt_tclBaseMessage* poMessage)
   {
      // As we generally allow services to attach to CCA_Node after the Node is created
      // and has connected to the SPM, we must delay processing of ServiceRegister requests
      // until the service has attached itself to us.

      /* get type and dispatch */
      tU8   u8Type = poMessage->u8GetType();
      tU8   u8Context = poMessage->u8GetContext();
      if (    (u8Type != AMT_C_U8_CCAMSGTYPE_SVCREGISTER)
           || (u8Context != AMT_C_U8_CCACONTEXT_LOCAL)
           || (enInterfaceState != AIL_EN_N_APPLICATION_INITIALIZED) )
         return false;  // normal AIL processing is perfectly OK

      amt_tclServiceRegister  oServiceRegister(poMessage);
      tU16 u16ServiceId = oServiceRegister.u16GetServiceID();
      // check if we already have the requested service
      {
         AutoLock lock;

         for (tTargetList::iterator it = _targets.begin(); it != _targets.end(); ++it)
            if ((*it) && ((*it)->serviceInfo.u16ServiceID == u16ServiceId) && ((*it)->bIsService()) )
               return false;  // we already know this service, so let AIL do normal processing

         // not found - so memorize request for processing once the service attaches
         Service:: vPostponeRegisterMsg (oServiceRegister);

      }  // end of AutoLock scope

      poMessage->bDelete();
      return true;
   }

   // --------------------------------------------------------------------------

   /* virtual */ tBool Node:: bOnInit ()
   {
      {  AutoLock lock;

         for (tTargetList::iterator it = _targets.begin(); it != _targets.end(); ++it)
            (*it)->vOnNewAppState(AMT_C_U32_STATE_INITIALIZED);

      } // end of AutoLock scope

      return TRUE;
   }

   // --------------------------------------------------------------------------

   /* virtual */ tVoid Node:: vOnApplicationClose ()
   {

   }

   // --------------------------------------------------------------------------

   /* virtual */ tVoid Node:: vOnNewAppState (tU32 u32OldAppState, tU32 u32AppState)
   {
      vAppStateChanged(u32AppState);

      ETG_TRACE_USR1(("CCA_Node<%x>:: vOnNewAppState() - changing state: %u --> %u"
            , u16AppId, u32OldAppState, u32AppState ))

      {  AutoLock lock;

         for (tTargetList::iterator it = _targets.begin(); it != _targets.end(); ++it)
            (*it)->vOnNewAppState(u32AppState);

      } // end of AutoLock scope
   }

   // --------------------------------------------------------------------------

   /* virtual */  tVoid Node:: vOnServiceState (tU16 u16ServiceId, tU16 u16ServerId, tU16 u16RegisterId, tU8 u8ServiceState, tU16 u16SubId)
   {
      Client* pClient = NULL;
      {
         AutoLock lock;

         for (tTargetList::iterator it = _targets.begin(); it != _targets.end(); ++it)
            if ((*it) && ((*it)->serviceInfo.u16ServiceID == u16ServiceId) && ( ! ((*it)->bIsService())) )
            {
               pClient = static_cast<Client*>(*it);
               break;
            }
      }  // end of AutoLock scope

      if (pClient)
         pClient->vOnServiceState(u8ServiceState);
      else
         ETG_TRACE_FATAL(("CCA_Node<%x>:: vOnServiceState() no client for this service %x"
               , u16AppId, ETG_ENUM(ail_u16ServiceId, (tU16) u16ServiceId) ))

         // --- call the corresponding method of the base class to give AIL the chance to do the rest ---
         ail_tclAppInterfaceRestricted::vOnServiceState(u16ServiceId, u16ServerId, u16RegisterId, u8ServiceState, u16SubId);
   }

   // --------------------------------------------------------------------------

   /* virtual */ tVoid Node:: vOnAsyncRegisterConfExt (tU16 u16RegisterId
         , tU16 u16ServerAppId, tU16 u16ServiceId, tU16 u16TargetSubId, tU8 u8AsyncRegisterConfStatus)
   {
      ETG_TRACE_USR1(("CCA_Node<%x>:: vOnAsyncRegisterConfExt() received for ServiceID 0x%X with RegisterId %d and Status = %u"
            , u16AppId, u16ServiceId, u16RegisterId, u8AsyncRegisterConfStatus))

      Client* pClient = NULL;
      {
         AutoLock lock;

         for (tTargetList::iterator it = _targets.begin(); it != _targets.end(); ++it)
            if ((*it) && ((*it)->serviceInfo.u16ServiceID == u16ServiceId) && ( ! ((*it)->bIsService())) )
            {
               pClient = static_cast<Client*>(*it);
               break;
            }
      }  // end of AutoLock scope

      if (pClient)
         pClient->vOnAsyncRegisterConfExt(u16RegisterId, u16ServerAppId, u16ServiceId, u16TargetSubId, u8AsyncRegisterConfStatus);
      else
         ETG_TRACE_FATAL(("CCA_Node<%x>:: vOnAsyncRegisterConfExt() no client for this service %x"
               , u16AppId, ETG_ENUM(ail_u16ServiceId, (tU16) u16ServiceId) ))
   }

   // --------------------------------------------------------------------------

   /* virtual */ tVoid Node:: vOnNewMessage (amt_tclBaseMessage* pMsg)
   {
      if (NULL == pMsg)
      {
         ETG_TRACE_FATAL(("CCA_Node<%x>:: vOnNewMessage() - E R R O R : message pointer is NULL", u16AppId))
         return;
      }

      if (CCA_C_U8_TYPE_SVCDATA == pMsg->u8GetType())
      {
         amt_tclServiceData& oSvcData = static_cast<amt_tclServiceData&>(*pMsg);
         tU16 u16ServiceID = oSvcData.u16GetServiceID();

         TargetBase* pTarget = NULL;
         {
            AutoLock lock;

            for (tTargetList::iterator it = _targets.begin(); it != _targets.end(); ++it)
               if ((*it) && ((*it)->serviceInfo.u16ServiceID == u16ServiceID) )
               {
                  pTarget = *it;
                  break;
               }
         }  // end of AutoLock scope

         if (pTarget)
            pTarget->vOnNewMessage(oSvcData);
         else
            ETG_TRACE_FATAL(("CCA_Node<%x> :: vOnNewMessage() no target for ServiceID 0x%x"
                  , u16AppId,  ETG_ENUM(ail_u16ServiceId, (tU16) u16ServiceID) ))
      }
      else
      {
         ETG_TRACE_FATAL(("CCA_Node<%x> :: vOnNewMessage() - E R R O R : message is not ServiceData, but 0x%02X"
               , u16AppId, pMsg->u8GetType() ))
      }

      // --- always delete incoming message and free the resources ---
      if (pMsg->bIsValid() && (! pMsg->bDelete()))
      {
          ETG_TRACE_ERR(("CCA_Node<%x>::vOnNewMessage - E R R O R : message could not be deleted ... !? "
                , u16AppId))
      }
   }

   // --------------------------------------------------------------------------

   /* virtual */ tBool Node:: bGetServiceVersion (tU16 u16ServiceID, tU16& rfu16MajorVersion, tU16& rfu16MinorVersion, tU16& rfu16PatchVersion)
   {
      bool bFound = false;

      {
         AutoLock lock;

         for (tTargetList::iterator it = _targets.begin(); it != _targets.end(); ++it)
            if ((*it) && ((*it)->bIsService()) && ((*it)->serviceInfo.u16ServiceID == u16ServiceID) )
            {
               const tFIVersionInfo& v = (*it)->serviceInfo.fiVersion;
               rfu16MajorVersion = v.u16MajorVersion;
               rfu16MinorVersion = v.u16MinorVersion;
               rfu16PatchVersion = 0;  // not relevant

               bFound = true;
               break;
            }

      }  // end of AutoLock scope

      if (bFound)
         ETG_TRACE_USR4(("CCA_Node<%x>:: bGetServiceVersion() : returning version %u.%u for Service %x"
               , u16AppId, rfu16MajorVersion, rfu16MinorVersion, u16ServiceID ))
      else
         ETG_TRACE_FATAL(("CCA_Node<%x>::  bGetServiceVersion() - E R R O R : requested ServiceID %x unknown"
               , u16AppId, u16ServiceID ))

      return (bFound ? TRUE : FALSE);
   }

   // --------------------------------------------------------------------------

   /* static */ void Node:: vOnQueueCallback (void* pArg)
   {
      Node* pNode = (Node*)pArg;
      if (NULL == pArg)
      {
         ETG_TRACE_FATAL(("CCA_Node<%x>:: vOnQueueCallback() - E R R O R : NULL-pointer"
               , pNode->u16AppId))
         return;
      }

      ETG_TRACE_USR4(("CCA_Node<%x>::vOnQueueCallback() : ThreadID = %u"
            , pNode->u16AppId, OSAL_ThreadWhoAmI()))

      pNode->_externalPollingThread.vRxNotificationCallback();
   }

   // --------------------------------------------------------------------------

   ISender*  Node:: poAddTarget (ITarget& externalTarget)
   {
      tU16 serviceID = externalTarget.GetServiceInfo().u16ServiceID;
      tFIVersionInfo version = externalTarget.GetServiceInfo().fiVersion;

      if (externalTarget.bProvidesService())
      {
         ETG_TRACE_USR1(("CCA_Node<%x>:: adding Service for ID = 0x%X, providing version %u.%u"
               , u16AppId, serviceID, version.u16MajorVersion, version.u16MinorVersion))

         Service* pNewService = new Service(*this, externalTarget);
         if (NULL == pNewService)
            return pNewService;  // for Lint - new operator should throw exception instead of returning a NULL value

         Service::PendingRegistrationList pendings;
         {
            AutoLock lock;

            _targets.insert(pNewService);

            pNewService->vGetPostponedRegisters(pendings);
         }  // end of AutoLock scope

         pNewService->vOnNewAppState(u32LastConfirmedAppState);

         // process postponed ServiceRegister requests
         for (Service::PendingRegistrationList::iterator it = pendings.begin(); it != pendings.end(); ++it)
         {
            amt_tclServiceRegister regMsg;  // default constructor allocates shared memory
            memcpy(regMsg.pu8GetSharedMemBase(), &(it->byte[0]), sizeof(Service::tMemorizedRegisterMsg));

            // let AIL process now what has been suppressed before
            if (poInternalDispatch)
            {
               ETG_TRACE_USR2(("CCA_Node<%x>:: poAddTarget() - processing postponed ServiceRegister for %x from Application %x.%d"
                     , u16AppId, regMsg.u16GetServiceID(), regMsg.u16GetSourceAppID(), regMsg.u16GetSourceSubID()))

               (void) poInternalDispatch->ail_bHandleMsgServiceRegister (&regMsg);
            }
            else
               ETG_TRACE_FATAL(("CCA_Node<%x>:: poAddTarget() - E R R O R - invalid pointer poInternalDispatch: cannot process postponed ServiceRegister for %x from Application %x.%d"
                     , u16AppId, regMsg.u16GetServiceID(), regMsg.u16GetSourceAppID(), regMsg.u16GetSourceSubID()))

            (void)regMsg.bDelete();
         }
         pendings.clear();

         return pNewService;
      }
      else
      {
         ETG_TRACE_USR1(("CCA_Node<%x>:: adding Client for Service 0x%X, requesting version %u.%u"
               , u16AppId, serviceID, version.u16MajorVersion, version.u16MinorVersion))

         Client* pNewClient = new Client(*this, externalTarget);
         if (NULL == pNewClient)
            return pNewClient;  // for Lint - new operator should throw exception instead of returning a NULL value

         {  AutoLock lock;
            _targets.insert(pNewClient);
         }  // end of AutoLock scope

         pNewClient->vOnNewAppState(u32LastConfirmedAppState);

         return pNewClient;
      }
   }

   // --------------------------------------------------------------------------

   void Node:: vRemoveTarget (ITarget& externalTarget)
   {
      AutoLock lock;

      // locate target with matching external target pointer
      for (tTargetList::iterator it = _targets.begin(); it != _targets.end(); ++it)
         if (&((*it)->_externalTarget) == &externalTarget)
         {
            tU16 u16ServiceID = externalTarget.GetServiceInfo().u16ServiceID;
            if ((*it)->bIsService())
            {
               ETG_TRACE_USR1(("App: removing Service for ID = 0x%X", u16ServiceID))

               // t.b.d. disconnect service
            }
            else
            {
               ETG_TRACE_USR1(("App: removing Client for Service 0x%X", u16ServiceID))

               Client& client = static_cast<Client&>(**it);
               client.vDisconnectFromService();
            }

            delete *it;
            _targets.erase(it);
            return;
         }

      ETG_TRACE_FATAL(("CCA_Node<%x> - E R R O R : failed removing child for Service 0x%X"
            , u16AppId, externalTarget.GetServiceInfo().u16ServiceID))
      return;
   }

   // --------------------------------------------------------------------------

   /* virtual */ void Node:: vPostFIMessage (MessageInfo info, const fi_tclMessageBase& fiMsg)
   {
      // send outgoing message
      const FunctionInfo& func = info.function;
      const ServiceInfo&  service = func.service;
      const RemoteAppInfo& app = info.remoteApp;

      fi_tclVisitorMessage msg(fiMsg.corfoGetTypeBase(), service.fiVersion.u16MajorVersion);
      msg.vInitServiceData(u16AppId, app.u16TargetAppID, AMT_C_U8_CCAMSG_STREAMTYPE_NODATA, 0
            , app.u16RegisterID, info.u16CmdCounter
            , fiMsg.u16GetServiceID(), fiMsg.u16GetFunctionID(), fiMsg.u8GetOpCode()
            , /* u8ACT */ 0, /* u16SourceSub */ AMT_C_U16_DEFAULT_NULL, /* u16TargetSub */ AMT_C_U16_DEFAULT_NULL
            , /* u32Timestamp */ AMT_C_U32_DEFAULT_NULL);

      ail_tenCommunicationError enAilError = ail_tclAppInterfaceRestricted::enPostMessage(&msg, true);
      if (enAilError  != AIL_EN_N_NO_ERROR )
         ETG_TRACE_FATAL(("CA_Node<%x>:: bPostFIMessage() - E R R O R :  failed posting message %x.%x.%x to application %x - error = %x"
               , u16AppId, service.u16ServiceID, func.u16FunctionID, func.u8OpCode, app.u16TargetAppID
               , ETG_CENUM(ail_tenCommunicationError, enAilError) ))
      else
         ETG_TRACE_USR2(("CCA_Node<%x>:: bPostFIMessage() - successfully posted message %x.%x.%x to application %x"
               , u16AppId, service.u16ServiceID, func.u16FunctionID, func.u8OpCode, app.u16TargetAppID))
   }

   // ============================================================================
   //
   //                             A u t o L o c k
   //
   // Mutex-based helper class, synchronizing multi-threaded access
   // to dynamic variables in application
   //

   Node::AutoLock:: AutoLock ()
      : _startTime(OSAL_ClockGetElapsedTime())
   {
      int success = pthread_mutex_lock(&_mutex);
      ETG_TRACE_USR4(("CCA_Node:: AutoLock constructor - lock success = %d", success))
   }

   // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

   Node::AutoLock:: ~AutoLock()
   {
      int success = pthread_mutex_unlock(&_mutex);
      ETG_TRACE_USR4(("CCA_Node:: AutoLock destructor - unlock success = %d", success))

      OSAL_tMSecond endTime = OSAL_ClockGetElapsedTime();
      if((endTime - _startTime) > 1000) // AutoLock lasts longer than 1s
      {
         OSAL_tThreadID s32ThreadId = OSAL_ThreadWhoAmI();

         ETG_TRACE_FATAL(("CCA_Node:: AutoLock for Thread Id: 0x%X took longer than 1s, exactly %d ms"
            , s32ThreadId, (endTime - _startTime)))
      }
   }

   // --------------------------------------------------------------------------

}  }  // namespace FIMessaging::CCA


