/*******************************************************************************
* FILE : apl-main.c
*
* SW-COMPONENT :
*
* DESCRIPTION :
*
* AUTHOR : Patrick Rey (ESE)
*
* COPYRIGHT : (C)
*
* HISTORY : 06.08.2015
*
*
*
* date             item
* 08.07.2014       Initial version
*******************************************************************************/





/* c standard library */
#include <stdio.h>
#include <unistd.h>
#include <string.h>


/* glib */
#include <glib/gi18n.h>
#include <glib.h>
#include <alsa/asoundlib.h>
#include "apl.h"
#include "apl-main.h"
#include "apl-internal.h"
#include "apl-parameter-factory.h"
#include "apl-libArkamysAiviImx-handler.h"







/*******************************************************************************
              VERSION COMPILER
*******************************************************************************/


/*******************************************************************************
              DEFINES - MACROS
*******************************************************************************/


/*******************************************************************************
              Constant defintion
*******************************************************************************/



/*******************************************************************************
              STRUCTURE DEFINITION
*******************************************************************************/

/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/
//static const int apl_vers[4] ={1,1,0,01};
static guint8     apl_inst_cnt = 0;
/*static*/ tAplStCfg*   m_AplLib = (tAplStCfg*)NULL;
static GMutex        apllock;
static tAplU32    apl_main_rec_mode_select = 0;

#ifdef D_APL_SIM
static struct aplMainStruct* _my_apl = NULL;
#endif


int     log_level;
char     trace_buf[D_TRACE_BUF_SIZE];
tAplBool  log_journal_enable;

/*******************************************************************************
              STATIC FUNCTION  DECLARATIONS
*******************************************************************************/



/*******************************************************************************
*
* FUNCTION: apl_main_audio_transfert_process
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
LOCAL APL_STATUS apl_main_audio_transfert_process
(
    struct  aplMainStruct*          aplh,
    const   aplAudioSamples_desc*   outAudio
)
{
    //APL_STATUS              status = APL_ERR_OK;
    //aplAudioSamples_desc*   minAudio = (aplAudioSamples_desc*)NULL;

    pr_debug_data("ENTERED\n");

    if(!aplh)
        return APL_ERR_NULL_POINTER;

    if(!outAudio)
        return APL_ERR_NULL_POINTER ;

    aplh->status = APL_ERR_OK;

    /**************************************************
     *retrieve buffer description o the last module
     *************************************************/
    aplh->pm_last = apl_internal_last_module(aplh);


    if(aplh->pm_last)
        aplh->minAudio = aplh->pm_last->apl_module_get_buffer(aplh->pm_last->data);

    if(!aplh->minAudio)
    {
        pr_warning("buffer descriptor by last module is NULL\n");
        return APL_ERR_NULL_POINTER;
    }


    /****************************
     * start audio frame transfer
     ****************************/

    /* set buffer pointer, consider only channel 1 */
    if(   (aplh->minAudio->apl_audio_format == SND_PCM_FORMAT_S16_LE)
        &&(outAudio->apl_audio_format == SND_PCM_FORMAT_S16_LE))
    {

        aplh->status = apl_internal_s16AudioTransfert(aplh->minAudio, (aplAudioSamples_desc*)outAudio); //, FALSE);
    }
    else if(  (aplh->minAudio->apl_audio_format == SND_PCM_FORMAT_S16_LE)
            &&(outAudio->apl_audio_format == SND_PCM_FORMAT_S32_LE))
    {
        aplh->status = apl_internal_is16os32AudioTransfert(aplh->minAudio, outAudio);
    }
    else if(  (aplh->minAudio->apl_audio_format == SND_PCM_FORMAT_S32_LE)
            &&(outAudio->apl_audio_format == SND_PCM_FORMAT_S32_LE))
    {
        aplh->status = apl_internal_s32AudioTransfert(aplh->minAudio, (aplAudioSamples_desc*)outAudio); //, FALSE);
    }
    else if(  (aplh->minAudio->apl_audio_format == SND_PCM_FORMAT_S32_LE)
            &&(outAudio->apl_audio_format == SND_PCM_FORMAT_S16_LE))
    {
        pr_warning("there a lost of data, s32_in -> s16_out\n");
        aplh->status = apl_internal_is32os16AudioTransfert(aplh->minAudio, outAudio);
    }
    else
    {
        /* do nothing */
    }


    return aplh->status;
}

/*******************************************************************************
*
* FUNCTION: apl_main_audio_transfert_prepare
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
LOCAL APL_STATUS apl_main_audio_transfert_prepare
(
    struct aplMainStruct*        aplh
)
{
    APL_STATUS  status = APL_ERR_OK;

    pr_message("ENTERED\n");

    if(!aplh)
        return APL_ERR_NULL_POINTER;

    if(!aplh->instance)
        return APL_ERR_NOT_CREATED;


    /* do something */

    pr_debug("ENTERED\n");

    return status;
}

/*******************************************************************************
*
* FUNCTION: apl_main_module_chain_set_up
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
LOCAL APL_STATUS apl_main_module_chain_set_up
(
    struct aplMainStruct*        aplh
)
{
    APL_STATUS status = APL_ERR_OK;
    tAplU32 i;

    pr_message("ENTERED\n");

    if(!aplh)
        return APL_ERR_NULL_POINTER;

    if(!aplh->instance)
        return APL_ERR_NOT_CREATED;

    if(!aplh->mlistcon)
    {
        pr_debug("no connection list available\n");
        return APL_ERR_OK;
    }

    module_desc*    pNewlist= (module_desc*)NULL;
    module_desc**   mlist = g_malloc0(aplh->mlistsize * sizeof(module_desc*));

    /* store current module list */
    tAplU32 mcurrlistsize = apl_internal_module_list_count(aplh);
    module_desc**   mcurrlist = (module_desc**)NULL;
    if(mcurrlistsize)
    {
        mcurrlist = g_malloc0(mcurrlistsize * sizeof(module_desc*));
        if(mcurrlist)
            (void)apl_internal_get_module_list(aplh, &mcurrlist);
    }

    /* create new module list */
    if(mlist)
    {
        /* walk though module connection list */
        for( i = 0; i < aplh->mlistsize; i++)
        {
            pr_debug("set module(%d) with order(%d)\n", aplh->mlistcon[i], i);
            mlist[i] = apl_internal_find_module(aplh, aplh->mlistcon[i]);
            pr_debug("next module in the list index(%d) is : %d\n", i, mlist[i]->associated_switch);
        }

        /* attachment of new module list */
        (void)apl_internal_print_current_module_list(aplh);
        (void)apl_internal_attach_module_list(&pNewlist, &mlist, aplh->mlistsize);

        /* clean previous list */
        (void)apl_internal_print_current_module_list(aplh);
        (void)apl_internal_clean_prev_module_list(&mcurrlist, mcurrlistsize, &mlist, aplh->mlistsize);

        /* replace now ordered module list */
        aplh->mdesc = (module_desc*)NULL;
        aplh->mdesc = pNewlist;

        g_free(mlist);
    }

    if(mcurrlist)
        g_free(mcurrlist);

    pr_message("EXIT\n");

    return status;
}



/*******************************************************************************
*
* FUNCTION: sseCreate
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
EXPORT APL_STATUS aplNew(const tAplStCfg* Aplcfg)
{
    APL_STATUS status = APL_ERR_OK;

    if(!Aplcfg)
        return APL_ERR_MODULE_NOT_INIT;


    g_print("APL_MSG: aplNew ENTERED\n");

    if(m_AplLib)
        g_free(m_AplLib);

    m_AplLib = g_malloc0(sizeof(tAplStCfg));


    /* copy constructor*/
    if(m_AplLib)
    {
        m_AplLib->AplNotifyfct = Aplcfg->AplNotifyfct;
        m_AplLib->AplIsLoadArkamays = Aplcfg->AplIsLoadArkamays;
        m_AplLib->AplTracecb = Aplcfg->AplTracecb; /*new*/
    }

    /* module initialization */
    if(m_AplLib->AplIsLoadArkamays)
    {
        /* do arkamys initialize */
    }

    /********************************************
     * read environment variables related
     * to the apl library in set in this running process
     */
    const char* slog_level = getenv("APL_LOG_LEVEL");

  /* default setting */
  log_level = 1;
  log_journal_enable = aplFALSE;

    if(slog_level)
        g_print("APL_MSG: active APL_LOG_LEVEL = %s\n", slog_level);

    if (!g_strcmp0(slog_level, "1"))  // standard log -> message
    {
        log_level = 1;
    }
    else if (!g_strcmp0(slog_level, "2"))  // standard log -> dbug
    {
        log_level = 2;
    }
    else
    {
        /* default setting  */
        log_level = 1;
    }


    /* stream recorder initialization */
    const char* sStream_record = getenv("APL_STREAM_RECORDER");
    if(sStream_record)
        g_print("APL_MSG: state APL_STREAM_RECORDER = %s\n", sStream_record);

    if (!g_strcmp0(sStream_record, "1"))  // record In-Out only
    {
        apl_main_rec_mode_select = apl_internal_rec_init(1);
    }
    else if (!g_strcmp0(sStream_record, "2"))  // record In-Out only
    {
        apl_main_rec_mode_select = apl_internal_rec_init(2);
    }
    else if (!g_strcmp0(sStream_record, "3"))  // record In-Out only
    {
        apl_main_rec_mode_select = apl_internal_rec_init(3);
    }
    else if (!g_strcmp0(sStream_record, "4"))  // record all, In-out and modules
    {
        apl_main_rec_mode_select = apl_internal_rec_init(4);
    }
    else
    {
        apl_main_rec_mode_select = apl_internal_rec_init(0);
    }


    /* call constructor from other module */
    apl_internal_init();

    g_mutex_init(&apllock);


    return status;
}


/*******************************************************************************
*
* FUNCTION: sseCreate
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
EXPORT APL_STATUS aplDelete(void)
{
    APL_STATUS status = APL_ERR_OK;

    g_print("aplDelete:ENTERED\n");

    if(!m_AplLib)
        return APL_ERR_MODULE_NOT_INIT;

    if(m_AplLib->AplIsLoadArkamays)
    {
        /* unload arkamys*/
    }

    g_free(m_AplLib);
    m_AplLib = NULL;

    (void)apl_internal_log_finalize();

    /* call constructor from other module */
    apl_internal_finalize();

    g_mutex_clear(&apllock);


    return status;
}

/*******************************************************************************
*
* FUNCTION: sseCreate
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
EXPORT APL_STATUS aplCreate
(
    tApl* hApl
)
{
    APL_STATUS status = APL_ERR_OK;
    GString *str1;
    static struct aplMainStruct* my_apl = NULL;

    pr_message("ENTERED\n");


    apl_inst_cnt++;
    str1 = g_string_new(NULL);
    g_string_printf (str1, "%s_%d", "apl_lib_inst", apl_inst_cnt);


    g_mutex_lock(&apllock);

    /* create an apl instance */
    my_apl = g_malloc0(sizeof(struct aplMainStruct));

    if(!my_apl)
    {
        g_string_free(str1, aplTRUE);
        return APL_ERR_NULL_POINTER;
    }

    /* initialize new instance  */
    my_apl->instance = apl_inst_cnt;
    my_apl->name = g_strdup(str1->str);
    g_mutex_init(&my_apl->lock);

    pr_debug("this apl handel           :%p\n", my_apl);
    pr_debug("this apl instance         :%d\n", my_apl->instance);
    pr_debug("this apl instance name    :%s\n", my_apl->name );

    g_string_free(str1, aplTRUE);

    /* instance initialize*/
    my_apl->init = aplFALSE;
    my_apl->start = aplFALSE;
    my_apl->mdesc = (module_desc*)NULL;
    my_apl->apl_internal_rec_mode_select = 0;    /* audio recording switched off*/

    /* assigne new instance */
    *hApl = my_apl;

    g_mutex_unlock(&apllock);

    return status;
}

/*******************************************************************************
*
* FUNCTION: sseCreate
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
EXPORT APL_STATUS aplInitialize
(
    tApl           hApl,
    const tAplCfg* paplCfg
)
{
    APL_STATUS status = APL_ERR_OK;

    pr_debug("ENTERED\n");

    if(!hApl)
        return APL_ERR_NULL_POINTER;

    g_mutex_lock(&apllock);

    struct aplMainStruct* aplh = (struct aplMainStruct*)hApl;

    /* print current state */
    pr_debug("aplh = %p\n", aplh);
    pr_debug("aplh->init = %d\n", aplh->init);
    pr_debug("aplh->name = %s\n", aplh->name);

    if(aplh->init)
    {
        status = APL_ERR_MODULE_ALREADY_INIT;
    }

    if(!status)
    {
        module_desc* pm = aplh->mdesc;
        module_desc* pm_next = (module_desc*)NULL;

        /* assign data buffer in apl instance */
        status = aplSetConfig(aplh,paplCfg);

        /* setup audio recording in teh instance */
        aplh->apl_internal_rec_mode_select = apl_main_rec_mode_select;

        if(!status)
        {
            /* walk though the configured modules */
            while(pm)
            {
                pr_debug(" call prepare of module(%d)\n", pm->associated_switch);
                if(APL_ERR_OK == pm->apl_module_prepare((void*)aplh, pm->data))
                {
                    pr_debug("change state of module(%d) to apl_state_prepared\n", pm->associated_switch);
                    pm->state = apl_state_prepared;
                }
                /*search for next */
                pm_next = (module_desc*)pm->next;
                pm = pm_next;
            }

            apl_main_module_chain_set_up(aplh);
            apl_main_audio_transfert_prepare(aplh);

            /* fully initialized */
            aplh->init = aplTRUE;

            pr_debug("apl instance(%d) with name(%s) is initialized(%d) \n", aplh->instance, aplh->name, aplh->init);
        }
    }

    g_mutex_unlock(&apllock);

    return status;
}

/*******************************************************************************
*
* FUNCTION: sseCreate
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

EXPORT APL_STATUS aplSetConfig
(
    tApl           hApl,
    const tAplCfg* paplCfg
)

{
   APL_STATUS status = APL_ERR_OK;

    struct aplMainStruct* aplh = (struct aplMainStruct*)hApl;

    if(!aplh)
        return APL_ERR_NULL_POINTER;

    if(!aplh->instance)
        return APL_ERR_NOT_CREATED;

    /* save config data */
    if(paplCfg)
    {
        if(paplCfg->pData)
        {
            /**********************
             * - free buffer if still available
             * - create new buffer
             */
            if(aplh->file.pData)
            {
               g_free(aplh->file.pData);
               aplh->file.pData = NULL;
            }

            aplh->file.pData = g_malloc0(paplCfg->usSize * sizeof(tAplU8));
            if(aplh->file.pData)
                memcpy((void*)aplh->file.pData, paplCfg->pData, paplCfg->usSize * sizeof(tAplU8));
        }

        aplh->file.usSize = paplCfg->usSize;
    }
    return status;
}
/*******************************************************************************
*
* FUNCTION: aplDestroy
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

EXPORT APL_STATUS aplDestroy
(
    const tApl* hApl
)
{
    APL_STATUS status = APL_ERR_OK;

    pr_message("ENTERED\n");

    struct aplMainStruct* aplh = (struct aplMainStruct*)*hApl;

    if(!aplh)
    {
        pr_debug("apl not created\n");
        return APL_ERR_NULL_POINTER;
    }

    if(aplh->init)
        status = APL_ERR_MODULE_ALREADY_INIT;

    g_mutex_lock(&apllock);


    /* walk though all module */
    module_desc* pm = aplh->mdesc;
    module_desc* pm_next = (module_desc*)NULL;

    /* walk though the configured modules */
    while(pm)
    {
        status = pm->apl_module_finalize(pm->data); //aplh, inAudio, outAudio));

        if(status)
        {
            pr_warning("fails to finalize module(%d)\n", (pm->associated_switch));
                break;
        }

        /*search for next */
        pm_next = (module_desc*)pm->next;

        /* free module instance */
        g_free(pm);
        pm = NULL;

        /* point to next module */
        pm = pm_next;
    }


    /* release apl instance */
    if(aplh->name)
    {
        g_free(aplh->name);
        aplh->name = NULL;
    }

    if(aplh->mlistcon)
    {
        g_free(aplh->mlistcon);
        aplh->mlistcon = NULL;
    }

    g_mutex_clear(&aplh->lock);
    g_free(aplh);

    if(apl_inst_cnt)
        apl_inst_cnt--;

    pr_message("EXIT: apl instance was destroyed, current active apl instance count is: %d\n", apl_inst_cnt);

    g_mutex_unlock(&apllock);

    return status;
}


/*******************************************************************************
*
* FUNCTION: aplCommand
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

EXPORT APL_STATUS aplCommand
(
    const tApl* hApl,
    tAplCmd     cmd
)
{
    APL_STATUS status = APL_ERR_OK;

    pr_debug("ENTERED\n");

    struct aplMainStruct* aplh = (struct aplMainStruct*)*hApl;

    if(!aplh)
    {
        pr_debug("apl not created\n");
        return APL_ERR_NULL_POINTER;
    }


    g_mutex_lock(&apllock);


    /* walk though all module */
    module_desc* pm = aplh->mdesc;
    module_desc* pm_next = (module_desc*)NULL;

    /* walk though the configured modules */
    while(pm)
    {
        status = pm->apl_module_command(pm->data, cmd );

        if(status)
        {
            pr_warning("fails to apply command to module(%d)\n", (pm->associated_switch));
                break;
        }

        /*search for next */
        pm_next = (module_desc*)pm->next;

        /* point to next module */
        pm = pm_next;
    }

    /* reinitialize recording for next session */
    if(cmd == aplCmdResetRecording)
        apl_internal_rec_reinit();


    pr_debug("EXIT command(%d) was propagate through the apl instance_%d \n", cmd, aplh->instance);

    g_mutex_unlock(&apllock);

    return status;
}

/*******************************************************************************
*
* FUNCTION: aplProcessAudioEnt
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

EXPORT APL_STATUS aplProcessAudioEnt
(
    tApl                        hApl,
    const aplAudioSamples_desc* inAudio,
    const aplAudioSamples_desc* outAudio
)
{
    APL_STATUS              status = APL_ERR_OK;
    aplAudioSamples_desc*   minAudio = (aplAudioSamples_desc*)NULL;
    aplAudioSamples_desc*   moutAudio = (aplAudioSamples_desc*)NULL;

    pr_debug_data("ENTERED\n");


    if(!hApl)
        return APL_ERR_NULL_POINTER;

    if(!inAudio || !outAudio)
        return APL_ERR_NULL_POINTER;

    struct aplMainStruct* aplh = (struct aplMainStruct*)hApl;

    if(!aplh->instance)
        return APL_ERR_NOT_CREATED;

    if(!aplh->init)
        return APL_ERR_MODULE_NOT_INIT;

    /* walk though the module */
    module_desc* pm_prev = (module_desc*)NULL;;
    module_desc* pm = aplh->mdesc;
    module_desc* pm_next = (module_desc*)NULL;


    /**************************************
     * walk though all attached modules
     **************************************/

    tAplInt     NotifyParam;
    tAplAction  NotifyAct;

    while(pm)
    {
        if(pm->first_mod)
        {
            /* recording */
            status = apl_internal_stream_rec_file_write(1, pm, inAudio);

            /* processing */
            status = pm->apl_module_process(pm, inAudio);
        }
        else
        {
            pm_prev = (module_desc*)pm->prev;
            if(pm_prev)
            {
                minAudio = pm_prev->apl_module_get_buffer(pm_prev->data);

                /* recording */
                status = apl_internal_stream_rec_file_write(4, pm, inAudio);

                /* processing */
                status = pm->apl_module_process(pm, minAudio);
            }
            else
            {
                /*tbd*/
            }
        }

        /* check for pending notification */
        while(pm->apl_module_notification(pm->data, &NotifyParam, &NotifyAct))
        {
            pr_message("send notification to the application for the property -> %d \n", NotifyParam);
            if(m_AplLib)
            {
                if(m_AplLib->AplNotifyfct)
                {
                    pr_message("call notify cb for parameter(%d) \n",NotifyParam);
                    m_AplLib->AplNotifyfct((tAplU16)NotifyParam, NotifyAct); /*TODO, in Gen4*/
                }
            }
        }

        if(status)
        {
            pr_warning("fails to process audio from the module(%d)\n", pm->associated_switch);
            break;
        }

        /*search for next */
        pm_next = (module_desc*)pm->next;
        pm = pm_next;
    }


    /********************************************
     * recording last processed buffer
     ********************************************/
    module_desc*    pm_last = apl_internal_last_module(aplh);

    if(pm_last)
    {
        moutAudio = pm_last->apl_module_get_buffer(pm_last->data);

        /* recording */
        status = apl_internal_stream_rec_file_write(4, pm_last, moutAudio);
    }

    /********************************************
     * bypass input buffer to output buffer
     ********************************************/
    status = apl_main_audio_transfert_process(aplh, outAudio);


    /* recording */
    status = apl_internal_stream_rec_file_write(2, pm, outAudio);

    return status;
}



/*******************************************************************************
*
* FUNCTION: aplProcessAudioVoice
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

EXPORT APL_STATUS aplProcessAudioVoice
(
    tApl                        hApl,
    const aplAudioSamples_desc* inAudio,
    const aplAudioSamples_desc* outAudio
)
{
    APL_STATUS status = APL_ERR_OK;

    //pr_debug("ENTERED\n");

    if(!hApl)
        return APL_ERR_NULL_POINTER;

    if(!inAudio || !outAudio)
        return APL_ERR_NULL_POINTER;


    struct aplMainStruct* aplh = (struct aplMainStruct*)hApl;

    if(!aplh->instance)
        return APL_ERR_NOT_CREATED;

    if(!aplh->init)
        return APL_ERR_MODULE_NOT_INIT;

    /********************************************
     * walk through all configured modules
     ********************************************/
    module_desc*            pm_prev = (module_desc*)NULL;;
    module_desc*            pm = aplh->mdesc;
    module_desc*            pm_next = (module_desc*)NULL;
    tAplOpMode              opmode = APL_OM_UNDEF;
    aplAudioSamples_desc*   minAudio = (aplAudioSamples_desc*)NULL;

    while(pm)
    {
        if(pm->first_mod)
        {
            status = pm->apl_module_process(pm, inAudio);
        }
        else
        {
            pm_prev = (module_desc*)pm->prev;
            if(pm_prev)
            {
                minAudio = pm_prev->apl_module_get_buffer(pm_prev->data);
                status = pm->apl_module_process(pm, minAudio);
            }
            else
            {
                /*tbd*/
            }
        }

        /* store operating mode */
        opmode =  pm->aplopmode;

        if(status)
        {
            pr_warning("fails to process audio from the module(%d)\n", pm->associated_switch);
            break;
        }

        /*search for next */
        pm_next = (module_desc*)pm->next;
        pm = pm_next;
    }


    /********************************************
     * notify change to caller
     ********************************************/
    guint32     size = 4;
    tAplI32     IS_aplFctCtrl;

    g_mutex_lock(&apllock);
    (void)apl_internal_apl_get_data(aplh, aplMicLvlWatchStartStop, 0, &size, &IS_aplFctCtrl);
    g_mutex_unlock(&apllock);

    if(IS_aplFctCtrl == aplFctStart)
    {
        if(m_AplLib)
        {
            if(m_AplLib->AplNotifyfct)
            {
                //g_mutex_lock(&apllock);
                pr_debug("call notify cb for parameter(%d) \n",aplMicLvl);
                m_AplLib->AplNotifyfct(aplMicLvl, APL_ACT_GET_PARAM);

                tAplI32 REQ_aplFctCtrl = aplFctStop;

                g_mutex_lock(&apllock);
                (void)apl_internal_apl_set_data(aplh, aplMicLvlWatchStartStop, 0, sizeof(tAplI32),(void*) &REQ_aplFctCtrl );
                g_mutex_unlock(&apllock);
            }
        }
    }


    /********************************************
     * bypass input buffer to output buffer
     ********************************************/

    if(opmode == APL_OM_CONS_ONLY)
    {
        pr_debug("consume only operation mode is active\n");
        return APL_ERR_NO_RESULT;
    }

    status = apl_main_audio_transfert_process(aplh, outAudio);

    return status;
}

/*******************************************************************************
*
* FUNCTION: aplProcessAudioInfo
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

EXPORT APL_STATUS aplProcessAudioInfo
(
    tApl                        hApl,
    const aplAudioSamples_desc* inAudio,
    const aplAudioSamples_desc* outAudio
)
{
    APL_STATUS              status = APL_ERR_OK;
    aplAudioSamples_desc*   minAudio = (aplAudioSamples_desc*)NULL;
    aplAudioSamples_desc*   moutAudio = (aplAudioSamples_desc*)NULL;

    pr_debug("ENTERED\n");


    if(!hApl)
        return APL_ERR_NULL_POINTER;

    if(!inAudio || !outAudio)
        return APL_ERR_NULL_POINTER;

    struct aplMainStruct* aplh = (struct aplMainStruct*)hApl;

    if(!aplh->instance)
        return APL_ERR_NOT_CREATED;

    if(!aplh->init)
        return APL_ERR_MODULE_NOT_INIT;

    /* walk though the module */
    module_desc* pm_prev = (module_desc*)NULL;;
    module_desc* pm = aplh->mdesc;
    module_desc* pm_next = (module_desc*)NULL;


    /**************************************
     * walk though all attached modules
     **************************************/

    tAplInt     NotifyParam;
    tAplAction  NotifyAct;

    while(pm)
    {
        if(pm->first_mod)
        {
            /* recording */
            status = apl_internal_stream_rec_file_write(1, pm, inAudio);

            /* processing */
            status = pm->apl_module_process(pm, inAudio);
        }
        else
        {
            pm_prev = (module_desc*)pm->prev;
            if(pm_prev)
            {
                minAudio = pm_prev->apl_module_get_buffer(pm_prev->data);

                /* recording */
                status = apl_internal_stream_rec_file_write(4, pm, inAudio);

                /* processing */
                status = pm->apl_module_process(pm, minAudio);
            }
            else
            {
                /*tbd*/
            }
        }

        /* check for pending notification */
        while(pm->apl_module_notification(pm->data, &NotifyParam, &NotifyAct))
        {
            if(m_AplLib)
            {
                if(m_AplLib->AplNotifyfct)
                {
                    pr_debug("call notify cb for parameter(%d) \n",NotifyParam);
                    m_AplLib->AplNotifyfct((tAplU16)NotifyParam, NotifyAct); /*TODO, in Gen4*/
                }
            }
        }

        if(status)
        {
            pr_warning("fails to process audio from the module(%d)\n", pm->associated_switch);
            break;
        }

        /*search for next */
        pm_next = (module_desc*)pm->next;
        pm = pm_next;
    }


    /********************************************
     * recording last processed buffer
     ********************************************/
    module_desc*    pm_last = apl_internal_last_module(aplh);

    if(pm_last)
    {
        moutAudio = pm_last->apl_module_get_buffer(pm_last->data);

        /* recording */
        status = apl_internal_stream_rec_file_write(4, pm_last, moutAudio);
    }

    /********************************************
     * bypass input buffer to output buffer
     ********************************************/
    status = apl_main_audio_transfert_process(aplh, outAudio);


    /* recording */
    status = apl_internal_stream_rec_file_write(2, pm, outAudio);

    return status;
}

/*******************************************************************************
*
* FUNCTION: aplSetData
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

EXPORT APL_STATUS aplSetData
(
    tApl            hApl,
    tAplDataID      iDataID,
    tAplInt         iChannel,
    tAplU32         ulSize,
    const void*     pData
)
{
    APL_STATUS status;

    pr_debug("ENTERED\n");

    g_mutex_lock(&apllock);

    struct aplMainStruct* aplh = (struct aplMainStruct*)hApl;

    if(!aplh)
        return APL_ERR_NULL_POINTER;

    if(!aplh->instance)// != _my_apl->instance)
        return APL_ERR_NOT_CREATED;

    /* set parameter internally */

    status = apl_parameter_set_data(aplh, iDataID, iChannel, ulSize, pData);

    g_mutex_unlock(&apllock);

    return status;
}

/*******************************************************************************
*
* FUNCTION: aplGetData
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
EXPORT APL_STATUS aplGetData
(
    tApl       hApl,
    tAplDataID iDataID,
    tAplInt    iChannel,
    tAplU32*   pulSize,
    void*      pData
)
{
    APL_STATUS status;

    pr_debug("ENTERED\n");

    g_mutex_lock(&apllock);

    struct aplMainStruct* aplh = (struct aplMainStruct*)hApl;

    if(!aplh)
        return APL_ERR_NULL_POINTER;

    if(!aplh->instance)// != _my_apl->instance)
        return APL_ERR_NOT_CREATED;


    /* return internal parameter value */
    status = apl_parameter_get_data(aplh, iDataID, iChannel, pulSize, pData);


    if(*pulSize <= 4)
    {
        int ivalue = 0;
        memcpy((void*)&ivalue, pData, *pulSize);
        pr_debug("parameter: %d , size : %d, value: %d\n", iDataID, *pulSize, ivalue);
    }
    else
        pr_debug("parameter: %d , size : %d\n", iDataID, *pulSize);


    g_mutex_unlock(&apllock);
    return status;
}

#define NU_STR "NO_USED"
/*******************************************************************************
*
* FUNCTION: aplGetStatusMessage
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

EXPORT APL_STATUS aplGetStatusMessage
(
    const tApl hApl,
    tAplChar   szStatusMessage[APL_MAX_STRING_SIZE]
)
{
   APL_STATUS status = APL_ERR_OK;

   struct aplMainStruct* aplh = (struct aplMainStruct*)hApl;

   if(!aplh)
        return APL_ERR_NULL_POINTER;

   memcpy((void*)szStatusMessage, (void*)NU_STR,  8 * sizeof(tAplChar));

   return status;
}

/*******************************************************************************
*
* FUNCTION: aplGetVersion
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

EXPORT APL_STATUS aplGetVersion
(
    tAplI16*   pusVersionArray[4],
    tAplChar** pszVersionString,
    tAplChar** pszVersionComment
)
{
    APL_STATUS status = APL_ERR_OK;


    *pusVersionArray[0] = MAJOR;
    *pusVersionArray[1] = MINOR1;
    *pusVersionArray[2] = MINOR2;
    *pusVersionArray[3] = MINOR3;

    *pszVersionString = (tAplChar*)g_strdup(VERSION_STRING);
    *pszVersionComment = (tAplChar*)g_strdup(VERSION_COMMENT);

    return status;
}


/*******************************************************************************
*
* FUNCTION: aplModuleChainSetup
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS aplModuleChainSetup
(
    tApl                hApl,
    const tAplDataID*   mlist,
    tAplU32             size
)
{
    APL_STATUS status = APL_ERR_OK;

    pr_debug("ENTERED\n");

    struct aplMainStruct* aplh = (struct aplMainStruct*)hApl;

    if(!aplh)
        return APL_ERR_NULL_POINTER;

    if(!aplh->instance)
        return APL_ERR_NOT_CREATED;

    if(!mlist || !size)
         return APL_ERR_NULL_POINTER;

    if(aplh->mlistcon)
    {
        g_free(aplh->mlistcon);
        aplh->mlistcon = NULL;
    }

    aplh->mlistcon = g_malloc0(size * sizeof(tAplDataID));
    memcpy((void*)aplh->mlistcon, (void*)mlist, size * sizeof(tAplDataID));
    aplh->mlistsize = size;

    pr_debug("EXIT\n");

    return status;
}

/*******************************************************************************
*
* FUNCTION: aplModuleChainSetup
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

APL_STATUS aplAnalyzeSetup
(
    tAplAnalyze feature,
    tAplI32     value
)
{
    APL_STATUS status = APL_ERR_OK;

    pr_debug("ENTERED\n");

    switch(feature)
    {
        case  aplAnalyzeFeat_Ark_log:
            if(value >= 1)
                (void)apl_internal_log_init(apl_log_arkamys_calls);
            else if(value <= 0)
                (void)apl_internal_log_finalize();
            else
            {
                /* do nothing */
            }
        break;

        case  aplAnalyzeFeat_log_level:
            if(value > 3)
                value = 3;
            else if(value < 0)
                value = 0;
            log_level = (int)value;
            break;
        case  aplAnalyzeFeat_performance:
        {
            audproc_apl_lib_arkamys_aivi_imx_handler_set_debug_level(ARK_DBG_LVL_PERF, value);
            break;
        }

    case aplAnalyzeFeat_log_journal_enable:
      if(value)
        log_journal_enable = aplTRUE;
      else
        log_journal_enable = aplFALSE;

      break;
        default:
            break;
    }

    return status;
}
