#include <glib/gi18n.h>
#include <glib.h>
#include <alsa/asoundlib.h>
#include "apl.h"
#include "apl-main.h"
#include "apl-internal.h"
#include "apl-libArkamysAiviImx-handler.h"



/*******************************************************************************
              DEFINES
*******************************************************************************/


/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/

static ARKAMYS_HANDLE ark_hdl = (ARKAMYS_HANDLE)NULL;
static tAplBool    ark_stream_runing_state = aplFALSE;
static tAplBool    ark_dbg_perf = aplFALSE;


/*******************************************************************************
*
* FUNCTION: audproc_apl_lib_arkamys_aivi_imx_handler_Open
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS  audproc_apl_lib_arkamys_aivi_imx_handler_init(void)
{
    APL_STATUS error = APL_ERR_OK;

#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX

    /* log arkamys Calls */
    const char* slog_arkamys_calls = getenv("APL_LOG_ARKAMYS_CALLS");
    if(slog_arkamys_calls)
        g_print("APL_MSG: state APL_LOG_ARKAMYS_CALLS = %s\n", slog_arkamys_calls);

    if (!g_strcmp0(slog_arkamys_calls, "1"))  // standard log -> message
    {
        (void)apl_internal_log_init(apl_log_arkamys_calls);
    }
    else
    {
        /* no defined */

    }

    /* create arkamys library instance */
    if(!ark_hdl)
    {
        ARKAMYS_RETURN ark_err = ArkamysAiviImxOpen(&ark_hdl);

        (void)apl_internal_log_item(apl_log_arkamys_calls, "call ArkamysAiviImxOpen \n");
        if(ark_err)
        {
            pr_warning("ArkamysAiviImx library open fails with error: %d \n", ark_err);

            (void)apl_internal_log_item(apl_log_arkamys_calls, "call ArkamysAiviImxOpen fails with ERROR: %d\n", ark_err );
            error = (ARKAMYS_RETURN)ark_err;
        }
        else
            pr_message("ArkamysAiviImx library was open successfully\n");
    }
    else
        error = APL_ERR_MODULE_ALREADY_INIT;

#else
    pr_warning("not supported\n");
#endif
    return error;
}


/*******************************************************************************
*
* FUNCTION: audproc_apl_lib_arkamys_aivi_imx_handler_start
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS   audproc_apl_lib_arkamys_aivi_imx_handler_start(void)
{
#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX
    pr_message("ENTERED\n");

    (void)apl_internal_log_item(apl_log_arkamys_calls, "first call of ArkamysAiviImxProcess\n");

    ark_stream_runing_state = aplTRUE;
#else
    pr_warning("not supported\n");
#endif
    return APL_ERR_OK;
}

/*******************************************************************************
*
* FUNCTION: audproc_apl_lib_arkamys_aivi_imx_handler_stop
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS   audproc_apl_lib_arkamys_aivi_imx_handler_stop(int proc_count)
{
#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX
    pr_message("ENTERED with count of audio processing : %d\n", proc_count );

    (void)apl_internal_log_item(apl_log_arkamys_calls, "suspend call of ArkamysAiviImxProcess after %d trials \n", proc_count );

    ark_stream_runing_state = aplFALSE;
#else
    pr_warning("not supported\n");
#endif
    return APL_ERR_OK;
}

/*******************************************************************************
*
* FUNCTION: audproc_apl_lib_arkamys_aivi_imx_handler_finalize
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS  audproc_apl_lib_arkamys_aivi_imx_handler_finalize(void)
{
    APL_STATUS error = APL_ERR_OK;
#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX
    pr_message("ENTERED\n");

    if(ark_hdl)
    {
        /* close arkamys libaray instance */
        (void)ArkamysAiviImxClose(ark_hdl);

        ark_hdl = (ARKAMYS_HANDLE)NULL;
        (void)apl_internal_log_item(apl_log_arkamys_calls, "call ArkamysAiviImxClose \n");
    }
    else
    {
        pr_warning("the ArkamysAiviImx library is currently not open\n");
        error = APL_ERR_MODULE_NOT_INIT;
    }

#else
    pr_warning("not supported\n");
#endif
    return error;
}

/*******************************************************************************
*
* FUNCTION: audproc_apl_lib_arkamys_aivi_imx_handler_ImxProcess
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS  audproc_apl_lib_arkamys_aivi_imx_handler_ImxProcess
(
  int** pIn,
  int** pOut,
  int numberOfSamplesPerChannel
)
{
    APL_STATUS      error   = APL_ERR_OK;
    ARKAMYS_RETURN  ark_err = ARKAMYS_NO_ERROR;

#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX
    //pr_message("ENTERED\n")
    if(ark_hdl)
    {
        if(ark_dbg_perf)
        {
            struct timespec ts_call, ts_ret;
            glong exec_time = 0;

            clock_gettime(CLOCK_MONOTONIC, &ts_call);
            //(void)apl_internal_log_item(apl_log_arkamys_calls, "call ArkamysAiviImxProcess \n");
            ark_err = ArkamysAiviImxProcess(ark_hdl, pIn, pOut, numberOfSamplesPerChannel);
            //(void)apl_internal_log_item(apl_log_arkamys_calls, "end  ArkamysAiviImxProcess \n");
            clock_gettime(CLOCK_MONOTONIC, &ts_ret);

            if(ts_call.tv_sec == ts_ret.tv_sec)
            {
                exec_time = ts_ret.tv_nsec - ts_call.tv_nsec;
            }
            else /* next second*/
            {
                exec_time = (1000000000 - ts_call.tv_nsec) + ts_ret.tv_nsec;
            }

            pr_message("ArkamysAiviImxProcess EXEC time : [%ld]\n", exec_time);
        }
        else
            ark_err = ArkamysAiviImxProcess(ark_hdl, pIn, pOut, numberOfSamplesPerChannel);

        error = (ARKAMYS_RETURN)ark_err;

    }
    else
    {
        pr_warning("the ArkamysAiviImx library is currently not open\n");
        error = APL_ERR_MODULE_NOT_INIT;
    }

#else
    pr_warning("not supported\n");
#endif
    return error;
}

/*******************************************************************************
*
* FUNCTION: audproc_apl_lib_arkamys_aivi_imx_handler_Set
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS   audproc_apl_lib_arkamys_aivi_imx_handler_Set
(
  int blockId,
  int paramId,
  int size,
  unsigned char*  pData
)
{
  APL_STATUS error = APL_ERR_OK;

#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX

    if(ark_hdl)
    {
        struct timespec ts_call, ts_ret;
        glong exec_time = 0;

        clock_gettime(CLOCK_MONOTONIC, &ts_call);
        ARKAMYS_RETURN ark_err  = ArkamysAiviImxSet(ark_hdl, blockId, paramId, size, pData);
        clock_gettime(CLOCK_MONOTONIC, &ts_ret);

        if(ts_call.tv_sec == ts_ret.tv_sec)
        {
            exec_time = ts_ret.tv_nsec - ts_call.tv_nsec;
        }
        else /* next second*/
        {
            exec_time = (1000000000 - ts_call.tv_nsec) + ts_ret.tv_nsec;
        }

        if(size <= 4)
        {
            int idata = 0;
            memcpy((void*)&idata, pData, (size_t)size);
            pr_message("ArkamysAiviImxSet call with blockid:%d, paramid:%d, size:%d, data=%d,  EXEC time : [%ld]\n", blockId, paramId, size, idata, exec_time);

        }
        else
            pr_message("ArkamysAiviImxSet call with blockid:%d, paramid:%d, size:%d,  EXEC time : [%ld]\n", blockId, paramId, size, exec_time);

        (void)apl_internal_log_item(apl_log_arkamys_calls, "call ArkamysAiviImxSet: blockId=%d, arkparamid=%d, size=%d \n", blockId, paramId, size );

        /* log Arkamys library calls */
        (void)apl_internal_log_item_data(apl_log_arkamys_calls, size, pData, aplTRUE);
        if(ark_err)
        {
            pr_warning("ArkamysAiviImxSet call with blockid:%d, paramid:%d, size:%d fails with error: %d\n", blockId, paramId, size, ark_err );

            (void)apl_internal_log_item(apl_log_arkamys_calls, "call ArkamysAiviImxSet fails with ERROR: %d\n", ark_err );
            error = (ARKAMYS_RETURN)ark_err;
        }
        //else
            //(void)apl_internal_log_item(apl_log_arkamys_calls, "end ArkamysAiviImxSet: blockId=%d, arkparamid=%d, size=%d \n", blockId, paramId, size );
    }
    else
    {
        pr_warning("the ArkamysAiviImx library is currently not open\n");
        error = APL_ERR_MODULE_NOT_INIT;
    }
#else
    pr_warning("not supported\n");
#endif
    return error;
}


/*******************************************************************************
*
* FUNCTION: audproc_apl_lib_arkamys_aivi_imx_handler_Get
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS   audproc_apl_lib_arkamys_aivi_imx_handler_Get
(
  int blockId,
  int paramId,
  int size,
  unsigned char*  pData
)
{
    APL_STATUS error = APL_ERR_OK;

#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX
    if(ark_hdl)
    {
        pr_message("ArkamysAiviImxGet call with blockid:%d, paramid:%d, size:%d\n", blockId, paramId, size);
        ARKAMYS_RETURN ark_err  = ArkamysAiviImxGet(ark_hdl, blockId, paramId, size, pData);

        (void)apl_internal_log_item(apl_log_arkamys_calls, "call ArkamysAiviImxGet: blockId=%d, arkparamid=%d, size=%d \n", blockId, paramId, size );
        if(!ark_err)
        {
            (void)apl_internal_log_item_data(apl_log_arkamys_calls, size,(unsigned char*)pData, aplFALSE);
        }
        else
        {
            pr_warning("ArkamysAiviImxGet call with blockid:%d, paramid:%d, size:%d fails with error: %d\n", blockId, paramId, size, ark_err );

            (void)apl_internal_log_item(apl_log_arkamys_calls, "call ArkamysAiviImxGet fails with ERROR: %d\n", ark_err );
            error = (ARKAMYS_RETURN)ark_err;
        }
    }
    else
    {
        pr_warning("the ArkamysAiviImx library is currently not open\n");
        error = APL_ERR_MODULE_NOT_INIT;
    }
#else
    pr_warning("not supported\n");
#endif
    return error;
}

/*******************************************************************************
*
* FUNCTION: audproc_apl_lib_arkamys_aivi_imx_handler_Flush
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS   audproc_apl_lib_arkamys_aivi_imx_handler_Flush(void)
{
    APL_STATUS error = APL_ERR_OK;

#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX
    if(ark_hdl)
    {
        struct timespec ts_call, ts_ret;
        glong exec_time = 0;

        clock_gettime(CLOCK_MONOTONIC, &ts_call);
        ARKAMYS_RETURN ark_err  = ArkamysAiviImxFlush(ark_hdl);
        clock_gettime(CLOCK_MONOTONIC, &ts_ret);

        if(ts_call.tv_sec == ts_ret.tv_sec)
        {
            exec_time = ts_ret.tv_nsec - ts_call.tv_nsec;
        }
        else /* next second*/
        {
            exec_time = (1000000000 - ts_call.tv_nsec) + ts_ret.tv_nsec;
        }

        pr_message("ArkamysAiviImxFlush called,  EXEC time : [%ld]\n", exec_time);



        (void)apl_internal_log_item(apl_log_arkamys_calls, "call ArkamysAiviImxFlush \n");

        if(ark_err)
        {
            (void)apl_internal_log_item(apl_log_arkamys_calls, "call ArkamysAiviImxFlush fails with ERROR: %d\n", ark_err );
            error = (ARKAMYS_RETURN)ark_err;
        }
    }
    else
    {
        pr_warning("the ArkamysAiviImx library is currently not open\n");
        error = APL_ERR_MODULE_NOT_INIT;
    }
#else
    pr_warning("not supported\n");
#endif
    return error;
}


void  audproc_apl_lib_arkamys_aivi_imx_handler_set_debug_level
(
    tAplU8      feat_mask,
    tAplI32     value
)
{

    pr_message(" feature mask -> %d\n", feat_mask);

    if(feat_mask & ARK_DBG_LVL_PERF)
    {
        if(value == 1)
            ark_dbg_perf = aplTRUE;
        else if (value == 0)
             ark_dbg_perf = aplFALSE;
    }
    else
    {
        /* do nothing */
    }
    return ;
}


