#include <glib/gi18n.h>
#include <glib.h>
#include <alsa/asoundlib.h>
#include "apl-main.h"
#include "apl-internal.h"
#include "apl-parameter-factory.h"
#include "apl-micro-level.h"
#include "apl-pcm-copy.h"
#include "apl-volume.h"



/*******************************************************************************
              DEFINES
*******************************************************************************/
#define D_SWAP_DISABLED

//#define D_APL_LOG_FILE "/var/opt/bosch/dynamic/apl_log.txt"
#define D_APL_LOG_FILE "/tmp/apl_log.txt"
const char*  data_in  =  "DATA_IN";
const char*  data_out =  "DATA_OUT";


/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/
typedef struct _log_desc
{
    FILE*       log_file;
    size_t      wdata;
    guint8      init;
    tAplBool    islogArkamysCall;
    //GMutex      lock;
}log_desc;



typedef struct _chunkcfg
{
    tAplU32 chunksize;
    tAplU32 range;
}chunkcfg;


static GMutex log_desc_lock;
static log_desc* logdesc = (log_desc*)NULL;

#define CHUNK_SIZE_MIN 4
static  chunkcfg datachunkcfg[] =
{
    {CHUNK_SIZE_MIN,  20},
    {5, 50},
    {10, 100},
    {20, 200},
    {40, 400},
    {50, 2000},
    {100, 20000}
};
#define CHUNK_RANGE_MAX ARRAYSIZE(datachunkcfg)


void      apl_internal_init(void)
{

    //g_print("%s_probe_0_1\n", "apl_internal_init");
    g_mutex_init(&log_desc_lock);

    return;
}

void      apl_internal_finalize(void)
{
    //g_print("%s_probe_0_1\n", "apl_internal_finalize");

    g_mutex_lock(&log_desc_lock);
    if(logdesc)
    {
        g_free(logdesc);
        logdesc = NULL;
    }
    g_mutex_unlock(&log_desc_lock);
    g_mutex_clear(&log_desc_lock);

    return;
}

/*******************************************************************************
*
* FUNCTION: apl_internal_log_init
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS      apl_internal_log_init(apllogitem item)
{
    APL_STATUS status = APL_ERR_OK;

    pr_message("ENTERED\n");

    if(!logdesc)
    {

        /* alloc log structure */
        logdesc = g_malloc0(sizeof(log_desc));
        if(!logdesc)
        {
            return  APL_ERR_NULL_POINTER;
        }

        /* create output test file */
        remove(D_APL_LOG_FILE);
        logdesc->log_file = fopen (D_APL_LOG_FILE,"a+");

        if(!logdesc->log_file)
        {
            pr_debug("failure while trying to open file %s\n", D_APL_LOG_FILE);
            return APL_ERR_NULL_POINTER;
        }

        fseek(logdesc->log_file, 0, SEEK_SET);
        logdesc->wdata = 0;
        logdesc->init = 1;

        //g_print("%s_probe_1_0\n", "apl_internal_log_init");
    }

    if(item == apl_log_arkamys_calls)
    {
        g_mutex_lock(&log_desc_lock);
        logdesc->islogArkamysCall = aplFALSE;

        remove(D_APL_LOG_FILE);
        logdesc->log_file = fopen (D_APL_LOG_FILE,"a+");

        if(logdesc->log_file)
        {
            struct timespec ts;
            clock_gettime(CLOCK_MONOTONIC, &ts);

            fprintf(logdesc->log_file,"********* LOG OF ARKAMYS_CALLS IS ACTIVATED AT TIME: %05u,%lu ********* \n", (unsigned int)ts.tv_sec, (unsigned long int)ts.tv_nsec);
            fprintf(logdesc->log_file,"\n");
            fclose(logdesc->log_file);
            logdesc->log_file = NULL;
        }

        logdesc->islogArkamysCall = aplTRUE;
        g_mutex_unlock(&log_desc_lock);

        //g_print("%s_probe_2_0\n", "apl_internal_log_init");
    }


    return status;
}

/*******************************************************************************
*
* FUNCTION: apl_internal_log_finalize
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS      apl_internal_log_finalize(void)
{
    APL_STATUS status = APL_ERR_OK;

    pr_message("ENTERED\n");

    g_mutex_lock(&log_desc_lock);
    if(logdesc)
    {
        g_print("%s_probe_1_0\n", "apl_internal_log_finalize");

        logdesc->islogArkamysCall = aplFALSE;
        //fclose(logdesc->log_file);
        //g_free(logdesc);
        //logdesc = NULL;
    }
    g_mutex_unlock(&log_desc_lock);

    g_print("%s_probe_0_2\n", "apl_internal_log_finalize");

    return status;
}



/*******************************************************************************
*
* FUNCTION: apl_internal_log_item
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS      apl_internal_log_item(apllogitem item, gchar* format,...)
{
    APL_STATUS err = APL_ERR_OK;
    tAplBool dolog = aplFALSE;

    pr_debug("ENTERED\n");

    g_mutex_lock(&log_desc_lock);

    if(!logdesc)
    {
        g_mutex_unlock(&log_desc_lock);
        return APL_ERR_OK;
    }

    switch(item)
    {
        case apl_log_arkamys_calls:
            if(logdesc->islogArkamysCall)
                dolog = aplTRUE;
            break;
        default:
            dolog = aplFALSE;
            break;
    }

    if(!dolog)
    {
        g_mutex_unlock(&log_desc_lock);
        return APL_ERR_OK;
    }
    //g_mutex_lock(&log_desc_lock);

    /* start format log string */
    va_list args;
    gchar *string;

    va_start(args, format);
    string = g_strdup_vprintf(format, args);
    va_end(args);

    if(!string)
        err = APL_ERR_NULL_POINTER;

    if(!err)
    {
        logdesc->log_file = fopen (D_APL_LOG_FILE,"a+");

        if(!logdesc->log_file)
        {
            pr_debug("failure while trying to open file %s\n", D_APL_LOG_FILE);
            if(string)
                g_free(string);
            err = APL_ERR_NULL_POINTER;
        }

        if(!err)
        {
            struct timespec ts;
            clock_gettime(CLOCK_MONOTONIC, &ts);
            fprintf(logdesc->log_file,"ARKAMYS_CALLS_TIME[%05u,%lu]: %s", (unsigned int)ts.tv_sec, (unsigned long int)ts.tv_nsec, string);
            fclose(logdesc->log_file);

            if(string)
                g_free(string);
        }
    }

    g_mutex_unlock(&log_desc_lock);

    return err;
}


/*******************************************************************************
*
* FUNCTION: apl_internal_log_item_data
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS      apl_internal_log_item_data(apllogitem item, tAplI32 size, unsigned char* data, tAplBool datadir)
{
    APL_STATUS err = APL_ERR_OK;
    tAplBool dolog = aplFALSE;
    const char* datalabel = (const char*)NULL;

    pr_debug("ENTERED\n");

    if(datadir) /* set data buffer */
        datalabel = data_in;
    else  /* get data buffer */
        datalabel = data_out;

    g_mutex_lock(&log_desc_lock);

    if(!logdesc || !data)
    {
        g_mutex_unlock(&log_desc_lock);
        return APL_ERR_NULL_POINTER;
    }

//    if(!data)
//        return APL_ERR_NULL_POINTER;

    switch(item)
    {
        case apl_log_arkamys_calls:
            if(logdesc->islogArkamysCall)
                dolog = aplTRUE;
            break;
        default:
            dolog = aplFALSE;
            break;
    }

    if(!dolog)
    {
        g_mutex_unlock(&log_desc_lock);
        return APL_ERR_OK;
    }

    //g_mutex_lock(&log_desc_lock);

    logdesc->log_file = fopen (D_APL_LOG_FILE,"a+");

    if(!logdesc->log_file)
    {
        pr_debug("failure while trying to open file %s\n", D_APL_LOG_FILE);
        err = APL_ERR_NULL_POINTER;
    }


    if(!err)
    {
        tAplU32 i = 0;
        tAplU32 usize = (tAplU32)size;
        tAplU32 chunksize = 4;
        tAplU32 chunkpos = 0;
        chunkcfg* p = (chunkcfg*)NULL;

        for(i = 0, p = datachunkcfg ; i < CHUNK_RANGE_MAX ; i++, p++)
        {
            if(p->range > usize)
            {
                if(CHUNK_SIZE_MIN > usize)
                    chunksize = usize;
                else
                    chunksize = p->chunksize;
                break;
            }
        }

        pr_message("selected chunksize is : %d, size(%d), range_pos(%d)\n", chunksize, usize, i);

        if(usize == 1)
        {
            fprintf(logdesc->log_file,"%s[0]    :{%d}\n", datalabel, data[0]);
        }
        else
        {
            for(i = 0; i < usize ; i++)
            {
                if(chunkpos == 0)
                {
                    if((i + (chunksize -1)) < 10)
                        fprintf(logdesc->log_file,"%s[%d-%d]   :{%d,", datalabel, i, (i + (chunksize -1)), data[i]);
                    else if((i + (chunksize -1)) > 9)
                        fprintf(logdesc->log_file,"%s[%d-%d]  :{%d,", datalabel, i, (i + (chunksize -1)), data[i]);
                    else if((i + (chunksize -1)) > 99)
                        fprintf(logdesc->log_file,"%s[%d-%d] :{%d,", datalabel, i, (i + (chunksize -1)), data[i]);
                    else
                    {
                        /* do nothing */
                    }

                    chunkpos = chunksize;
                }
                else if((chunkpos == 1) || (i == (usize - 1)))
                    fprintf(logdesc->log_file," %d}\n", data[i]);
                else
                    fprintf(logdesc->log_file," %d,", data[i]);

                --chunkpos;
            }
        }

        fclose(logdesc->log_file);
        logdesc->log_file = NULL;
    }

    g_mutex_unlock(&log_desc_lock);

    return err;
}




/*******************************************************************************
*
* FUNCTION: apl_internal_find_module
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

module_desc*  apl_internal_find_module
(
    const struct aplMainStruct*     hApl,
    tAplI32                         module
)
{

    pr_debug("ENTERED\n");

    if(!hApl)
        return (module_desc*)NULL;

    module_desc* pm_search = (module_desc*)hApl->mdesc;

    while(pm_search)
    {
        if(pm_search->associated_switch == module)
        {
            pr_debug("instance module(%d) found\n", module);
            break;
        }

        /* next module instance */
        module_desc* pm =  pm_search;
        pm_search = pm->next;
    }

    return pm_search;
}

/*******************************************************************************
*
* FUNCTION: apl_internal_last_module
*
* DESCRIPTION: first stage of two-stage initialization of the SSE module
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

module_desc*  apl_internal_last_module
(
    const struct aplMainStruct*   aplh
)
{
    pr_debug_data("ENTERED\n");

    if(!aplh)
        return (module_desc*)NULL;

    module_desc* pm_search = (module_desc*)aplh->mdesc;

    while(pm_search)
    {
        if(pm_search->last_mod)
        {
            pr_debug_data("instance module(%d) is the last module \n", pm_search->associated_switch);
            break;
        }

        /* next module instance */
        module_desc*  pm =  pm_search;
        pm_search = pm->next;
    }

    return pm_search;
}

/*******************************************************************************
*
* FUNCTION: apl_internal_attach_module
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS apl_internal_attach_module(module_desc** module_chain, module_desc* module_attachment)
{
    gint status = APL_ERR_OK;

    pr_debug("try to attach the module(%d)\n", module_attachment->associated_switch);

    if(!*module_chain)
  {
        /* first module attachment */
        pr_debug("first module attachment for switch(%d)\n", module_attachment->associated_switch);
        module_attachment->first_mod = aplTRUE;
        module_attachment->last_mod  = aplTRUE;
        module_attachment->prev = (void*)NULL;
        //module_attachment->next = (void*)NULL;
        *module_chain = module_attachment;
  }
    else
  {
        /* next module attachment */
        module_attachment->last_mod = aplTRUE;
        module_attachment->next = (void*)NULL;

        module_desc* pm      = *module_chain;
        module_desc* pm_next = pm->next;

        /* walk though the configured modules */

        while(pm_next)
        {
            if(pm_next->associated_switch == module_attachment->associated_switch)
            {
                pr_debug("switch(%d) already attached\n", module_attachment->associated_switch);
                return APL_ERR_MODULE_ALREADY_ATTACHED;
            }

            /* first module should have the reference of the last one for fast search */
            if(pm->first_mod)
                pm->prev = (void*)module_attachment;

            /*search for next */
            pm = pm_next;
            pm_next = (module_desc*)pm->next;
        }

        /* attach new module  */
        pm->next = (void*)module_attachment;
        module_attachment->prev = pm;

        /* make the previous module no longer the last one */
        pm->last_mod = aplFALSE;

        pr_debug("attachment new module(%d)\n", module_attachment->associated_switch);
    }

    return status;
}


/*******************************************************************************
*
* FUNCTION: apl_internal_attach_module
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS      apl_internal_attach_module_list
(
    module_desc**   module_chain,
    module_desc***  module_list,
    tAplU32         list_size
)
{
    module_desc*    module_attachment = (module_desc*)NULL;
    module_desc*    pm      = (module_desc*)NULL;
    module_desc*    pm_first  = (module_desc*)NULL;
    tAplU32 i;


    pr_debug("ENTERED\n");

    if(*module_chain)
        return APL_ERR_MODULE_ALREADY_INIT;
    if(!*module_list)
        return APL_ERR_NULL_POINTER;

    module_desc**    mlist = *module_list;

    /* walk though module connection list */
    for( i = 0; i < list_size; i++)
    {
        pm     = module_attachment;
        module_attachment   = mlist[i];

        if(module_attachment)
        {
            if( i == 0)
            {
                pr_debug("first module(%d) attachment with order(%d)\n", module_attachment->associated_switch, i);

                module_attachment->first_mod = aplTRUE;
                module_attachment->last_mod  = aplTRUE;
                module_attachment->prev = (void*)NULL;
                *module_chain = module_attachment;
                pm_first = module_attachment;

            }
            else
            {
                pr_debug("set next module(%d) with order(%d)\n", module_attachment->associated_switch, i);
                module_attachment->first_mod = aplFALSE;
                module_attachment->next = (void*)NULL;
                if(pm)
                    pm->next = module_attachment;

                /* make the last module always the precedent of the first  */
                if(pm_first)
                    pm_first->prev = module_attachment;
                module_attachment->prev = pm;
            }
        }
    }

    if(list_size > 1)
    {
        if(module_attachment)
            module_attachment->last_mod = aplTRUE;

        /* make the previous module no longer the last one */
        if(pm_first)
            pm_first->last_mod = aplFALSE;
    }

    if(pm_first)
        pr_debug("module(%d) is the first module in the list\n", pm_first->associated_switch);
    if(module_attachment)
        pr_debug("module(%d) is the last module in the list\n", module_attachment->associated_switch);
    pr_debug("EXIT\n");

    return APL_ERR_OK;
}


/*******************************************************************************
*
* FUNCTION: apl_internal_clean_prev_module_list
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS      apl_internal_clean_prev_module_list
(
    module_desc***  curr_list,
    tAplU32         curr_list_size,
    module_desc***  new_list,
    tAplU32         new_list_size
)
{
    tAplU32         i,j;
    APL_STATUS      status = APL_ERR_OK;

    pr_debug("ENTERED\n");

    if(!*curr_list)
        return APL_ERR_NULL_POINTER;
    if(!*new_list)
        return APL_ERR_NULL_POINTER;

    module_desc**    mcurrlist = *curr_list;
    module_desc**    mnewlist =  *new_list;


    /* walk though the configured modules */
    tAplBool found = aplFALSE;
    for( i = 0; i < curr_list_size; i++)
    {
        module_desc* pm = mcurrlist[i];

        pr_debug("search module(%d) in the new list \n", pm->associated_switch );

        /*search module in the new list */
        for( j = 0; j < new_list_size; j++)
        {
            module_desc*    module_search = mnewlist[j];
            pr_debug("compare %d_module(%d) in the new list with search module(%d)\n", j, module_search->associated_switch, pm->associated_switch );

            if(pm->associated_switch == module_search->associated_switch)
            {
                pr_debug("found module(%d) in the new list\n", pm->associated_switch );
                found = aplTRUE;
                break;
            }
        }

        /* module not found in eth new list, module instance should be destructed */
        if(!found)
        {
            pr_debug("module(%d) was not found in the new list, clean instance\n", pm->associated_switch );

            status = pm->apl_module_finalize(pm->data);
            if(status)
            {
                pr_warning("fails to finalize module(%d)\n", (pm->associated_switch));
                break;
            }

            /* free module instance */
            g_free(pm);
            pm = NULL;

        }

        found = aplFALSE;
    }

    pr_debug("EXIT\n");

    return APL_ERR_OK;
}


/*******************************************************************************
*
* FUNCTION: apl_internal_print_current_module_list
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS      apl_internal_print_current_module_list
(
 const struct aplMainStruct*   aplh
)
{
    pr_debug("ENTERED\n");

    if(!aplh)
        return APL_ERR_NULL_POINTER;

    module_desc* pm = (module_desc*)aplh->mdesc;
    guint i = 0;

    while(pm)
    {

        pr_debug("%d_module(%d) in the current list \n", i++, pm->associated_switch);

        /* next module instance */
        module_desc*  pm_next =  pm;
        pm = pm_next->next;
    }

    return APL_ERR_OK;
}


/*******************************************************************************
*
* FUNCTION: apl_internal_module_list_count
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
tAplU32      apl_internal_module_list_count
(
    const struct aplMainStruct*   aplh
)
{
    pr_debug("ENTERED\n");

    if(!aplh)
        return 0;

    module_desc*    pm = (module_desc*)aplh->mdesc;
    tAplU32         count = 0;

    while(pm)
    {
        count++;

        /* next module instance */
        module_desc*  pm_next =  pm;
        pm = pm_next->next;
    }

    pr_debug(" count of configured module in the current list is %d \n", count);

    return count;
}

/*******************************************************************************
*
* FUNCTION: apl_internal_get_module_list
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
tAplU32      apl_internal_get_module_list
(
    const struct aplMainStruct*     aplh,
    module_desc***                  module_list
)
{
    pr_debug("ENTERED\n");

    if(!aplh)
        return 0;

    module_desc*    pm = (module_desc*)aplh->mdesc;
    tAplU32         count = 0;
    module_desc**   mlist = *module_list;


    while(pm)
    {
        mlist[count++] = pm;

        /* next module instance */
        module_desc*  pm_next =  pm;
        pm = pm_next->next;
    }

    pr_debug(" count of configured module in the current list is %d \n", count);

    return count;
}

/*******************************************************************************
*
* FUNCTION: apl_internal_apl_get_data
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS apl_internal_apl_get_data
(
    struct aplMainStruct*   hApl,
    const tAplDataID        DataID,
    tAplInt                 iChannel,
    unsigned int*           pSize,
    void*                   pData
)
{
    APL_STATUS            status = APL_ERR_OK;

    pr_debug("ENTERED");

    if(!hApl)
        return APL_ERR_INVALID_MAIN_STRUCT;

//    g_mutex_lock(&hApl->lock);

    /* extract module Id */
    tAplU8 u8Moduleid = (tAplU8)(DataID >> 8) ;

    pr_debug("extract groupeId(%d)out of dataId(0x%x)\n", u8Moduleid, DataID);

    switch(u8Moduleid)
    {
        case aplGeneralGrp:
            status = apl_parameter_general_get(hApl, DataID, iChannel, pSize, pData);
            break;
        case aplModuleMicLvl:
            status = apl_parameter_Mic_Lvl_get(hApl, DataID, iChannel, pSize, pData);
            break;
        case aplModuleArkamys:
            status = apl_parameter_Arkamys_get(hApl, DataID, iChannel, pSize, pData);
            break;
        case aplModulePcmCopy:
            status = apl_parameter_PcmCopy_get(hApl, DataID, iChannel, pSize, pData);
            break;
        case aplModuleVolume:
            status = apl_paramteer_Volume_get(hApl, DataID, iChannel, pSize, pData);
            break;
        default:
            break;
    }

//    g_mutex_unlock(&hApl->lock);

    return status;
}


/*******************************************************************************
*
* FUNCTION: apl_internal_apl_set_data
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS apl_internal_apl_set_data
(
    struct aplMainStruct*   hApl,
    const tAplDataID        DataID,
    const int               iChannel,
    const unsigned int      iSize,
    const void*             pData
)
{
    APL_STATUS              status = APL_ERR_OK;

    pr_debug("ENTERED");

    if(!hApl)
        return APL_ERR_INVALID_MAIN_STRUCT;

//    g_mutex_lock(&hApl->lock);

    /* extract module Id */
    tAplU8  u8Moduleid = (tAplU8)(DataID >> 8) ;

    pr_debug("module_id(%d) associated to data_id(0x%x)\n", u8Moduleid, DataID);

    switch(u8Moduleid)
    {
        case aplGeneralGrp:
            status = apl_parameter_general_set(hApl, DataID, iChannel, iSize, pData);
            break;
        case aplModuleMicLvl:
            status = apl_parameter_Mic_Lvl_set(hApl, DataID, iChannel, iSize, pData);
            break;
        case aplModuleArkamys:
            status = apl_parameter_Arkamys_set(hApl, DataID, iChannel, iSize, pData);
            break;
        case aplModulePcmCopy:
            status = apl_parameter_PcmCopy_set(hApl, DataID, iChannel, iSize, pData);
            break;
        case aplModuleVolume:
            status = apl_parameter_Volume_set(hApl, DataID, iChannel, iSize, pData);
            break;
        default:
            break;
    }

//    g_mutex_unlock(&hApl->lock);

    return status;
}

/*******************************************************************************
*
* FUNCTION: apl_internal_s16AudioTransfert
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS apl_internal_s16AudioTransfert
(
    aplAudioSamples_desc*           inAudio,
    aplAudioSamples_desc*           outAudio
)
{


    pr_debug_data("ENTERED\n");

    if(!inAudio)
        return APL_ERR_NULL_POINTER ;

    if(!outAudio)
        return APL_ERR_NULL_POINTER ;

    inAudio->s16ibuf = inAudio->s16audiosamples[0];

    if(!inAudio->s16ibuf)
    {
        pr_warning("No input data stream buffer available \n");
        return APL_ERR_NULL_POINTER;
    }

    inAudio->err = APL_ERR_OK;

    if(outAudio->apl_period_size <  inAudio->apl_period_size)
    {
        pr_critical("mismatch in size out_stream(%d) and in_stream(%d) \n", outAudio->apl_period_size, inAudio->apl_period_size);
        return APL_ERR_AUDIO_BUFFER_MISMATCH;
    }

    /* in stream and out stream are aligned */
    if(   (outAudio->apl_stream_nb_ch == inAudio->apl_stream_nb_ch)
        ||(outAudio->apl_stream_nb_ch > inAudio->apl_stream_nb_ch ))
    {
       inAudio->stream_cnt =  inAudio->apl_stream_nb_ch;
    }
    else if(outAudio->apl_stream_nb_ch < inAudio->apl_stream_nb_ch)
    {
       inAudio->stream_cnt =  outAudio->apl_stream_nb_ch;
       pr_warning(" mismatch nb channel out_stream(%d) and in_stream(%d)\n",outAudio->apl_stream_nb_ch, inAudio->apl_stream_nb_ch );
    }

    for( inAudio->i = 0; inAudio->i < inAudio->stream_cnt ; inAudio->i++)
    {
        if(   (outAudio->apl_period_size ==  inAudio->apl_period_size)
            ||(outAudio->apl_period_size >  inAudio->apl_period_size))
        {
            inAudio->sample_cnt = inAudio->apl_period_size;

#ifndef D_SWAP_DISABLED
            if(inAudio->apl_can_swap)
            {
                pr_debug_data ("swap stream buffer[%d]\n", inAudio->i);

                inAudio->s16ibuf = inAudio->s16audiosamples[inAudio->i];
                outAudio->s16audiosamples[inAudio->i] = inAudio->s16ibuf;
            }
            else
            {
#endif
                pr_debug_data("copy %d s16 sample to out stream s16 buffer[%d]\n", inAudio->sample_cnt, inAudio->i);

                inAudio->s16ibuf = inAudio->s16audiosamples[inAudio->i];
                inAudio->s16obuf = outAudio->s16audiosamples[inAudio->i];
                memcpy((void*)inAudio->s16obuf, (void*)inAudio->s16ibuf, inAudio->sample_cnt * sizeof(short int));
#ifndef D_SWAP_DISABLED
            }
#endif
        }
        else
        {
            pr_critical(" mismatch between nb of in_stream(%d) and out_strem(%d) samples \n", inAudio->apl_period_size, outAudio->apl_period_size);
        }
    }

    return inAudio->err;
}


/*******************************************************************************
*
* FUNCTION: apl_internal_is16os32AudioTransfert
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

APL_STATUS apl_internal_is16os32AudioTransfert
(
    aplAudioSamples_desc*         inAudio,
    const aplAudioSamples_desc*   outAudio
)
{

    pr_debug_data("ENTERED\n");

    if(!inAudio)
        return APL_ERR_NULL_POINTER ;

    if(!outAudio)
        return APL_ERR_NULL_POINTER ;


    inAudio->s16ibuf = inAudio->s16audiosamples[0];
    if(!inAudio->s16ibuf)
    {
        pr_warning("No input data stream buffer available \n");
        return APL_ERR_NULL_POINTER;
    }

    inAudio->err = APL_ERR_OK;

    if(outAudio->apl_period_size <  inAudio->apl_period_size)
    {
        pr_critical("mismatch in size out_stream(%d) and in_stream(%d) \n", outAudio->apl_period_size, inAudio->apl_period_size);
        return APL_ERR_AUDIO_BUFFER_MISMATCH;
    }

    /* in stream and out stream are aligned */
    if(   (outAudio->apl_stream_nb_ch == inAudio->apl_stream_nb_ch)
        ||(outAudio->apl_stream_nb_ch > inAudio->apl_stream_nb_ch ))
    {
       inAudio->stream_cnt =  inAudio->apl_stream_nb_ch;
    }
    else if(outAudio->apl_stream_nb_ch < inAudio->apl_stream_nb_ch)
    {
       inAudio->stream_cnt =  outAudio->apl_stream_nb_ch;
       pr_warning(" mismatch nb channel out_stream(%d) and in_stream(%d)\n",outAudio->apl_stream_nb_ch, inAudio->apl_stream_nb_ch );
    }

    for( inAudio->i = 0; inAudio->i < inAudio->stream_cnt ; inAudio->i++)
    {
        if(   (outAudio->apl_period_size ==  inAudio->apl_period_size)
            ||(outAudio->apl_period_size >  inAudio->apl_period_size))
        {
            inAudio->sample_cnt = inAudio->apl_period_size;

            pr_debug_data("copy %d s16 sample to stream s32 buffer[%d]\n", inAudio->sample_cnt, inAudio->i);

            inAudio->s16ibuf = inAudio->s16audiosamples[inAudio->i];
            inAudio->s32obuf = outAudio->s32audiosamples[inAudio->i];
            for (inAudio->j = 0; inAudio->j <inAudio->sample_cnt; inAudio->j++)
                *inAudio->s32obuf++ = *inAudio->s16ibuf++;
        }
        else
        {
            pr_critical(" mismatch between nb of in_stream(%d) and out_strem(%d) samples \n", inAudio->apl_period_size, outAudio->apl_period_size);
        }
    }

    return inAudio->err;
}
/*******************************************************************************
*
* FUNCTION: apl_internal_s32AudioTransfert
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

APL_STATUS apl_internal_s32AudioTransfert
(
    aplAudioSamples_desc*   inAudio,
    aplAudioSamples_desc*   outAudio
)
{

    pr_debug_data("ENTERED\n");

    if(!inAudio)
        return APL_ERR_NULL_POINTER ;

    if(!outAudio)
        return APL_ERR_NULL_POINTER ;


    inAudio->s32ibuf = inAudio->s32audiosamples[0];
    if(!inAudio->s32ibuf)
    {
        pr_warning("No input data stream buffer available \n");
        return APL_ERR_NULL_POINTER;
    }

    inAudio->err = APL_ERR_OK;

    if(outAudio->apl_period_size <  inAudio->apl_period_size)
    {
        pr_critical("mismatch in size out_stream(%d) and in_stream(%d) \n", outAudio->apl_period_size, inAudio->apl_period_size);
        return APL_ERR_AUDIO_BUFFER_MISMATCH;
    }

    /* in stream and out stream are aligned */
    if(   (outAudio->apl_stream_nb_ch == inAudio->apl_stream_nb_ch)
        ||(outAudio->apl_stream_nb_ch > inAudio->apl_stream_nb_ch ))
    {
       inAudio->stream_cnt =  inAudio->apl_stream_nb_ch;
    }
    else if(outAudio->apl_stream_nb_ch < inAudio->apl_stream_nb_ch)
    {
       inAudio->stream_cnt =  outAudio->apl_stream_nb_ch;
       pr_warning(" mismatch nb channel out_stream(%d) and in_stream(%d)\n",outAudio->apl_stream_nb_ch, inAudio->apl_stream_nb_ch );
    }

    for( inAudio->i = 0; inAudio->i < inAudio->stream_cnt ; inAudio->i++)
    {
        if(   (outAudio->apl_period_size ==  inAudio->apl_period_size)
            ||(outAudio->apl_period_size >  inAudio->apl_period_size))
        {
            inAudio->sample_cnt = inAudio->apl_period_size;
#ifndef D_SWAP_DISABLED
            if(inAudio->apl_can_swap)
            {
                pr_debug_data ("swap stream buffer[%d]\n", inAudio->i);

                inAudio->s32ibuf = inAudio->s32audiosamples[inAudio->i];
                outAudio->s32audiosamples[inAudio->i] = inAudio->s32ibuf;
            }
            else
            {
#endif
                pr_debug_data("copy %d s32 sample to out stream s32 buffer[%d]\n", inAudio->sample_cnt, inAudio->i);

                inAudio->s32ibuf = inAudio->s32audiosamples[inAudio->i];
                inAudio->s32obuf = outAudio->s32audiosamples[inAudio->i];
                memcpy((void*)inAudio->s32obuf, (void*)inAudio->s32ibuf, inAudio->sample_cnt * sizeof(long int));
#ifndef D_SWAP_DISABLED
            }
#endif
        }
        else
        {
            pr_critical(" mismatch between nb of in_stream(%d) and out_strem(%d) samples \n", inAudio->apl_period_size, outAudio->apl_period_size);
        }
    }

    return inAudio->err;
}


/*******************************************************************************
*
* FUNCTION: apl_internal_is32os16AudioTransfert
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

APL_STATUS apl_internal_is32os16AudioTransfert
(
    aplAudioSamples_desc*           inAudio,
    const aplAudioSamples_desc*     outAudio
)
{

    pr_debug_data("ENTERED\n");

    if(!inAudio)
        return APL_ERR_NULL_POINTER ;

    if(!outAudio)
        return APL_ERR_NULL_POINTER ;


    if(!inAudio->s32audiosamples[0])
    {
        pr_warning("inAudio->s32audiosamples[0] is NULL \n");
        return APL_ERR_NULL_POINTER;
    }


    inAudio->s32ibuf = inAudio->s32audiosamples[0];
    if(!inAudio->s32ibuf)
    {
        pr_warning("No input data stream buffer available \n");
        return APL_ERR_NULL_POINTER;
    }

    inAudio->err = APL_ERR_OK;

    /* in stream and out stream are aligned */
    if(   (outAudio->apl_stream_nb_ch == inAudio->apl_stream_nb_ch)
        ||(outAudio->apl_stream_nb_ch > inAudio->apl_stream_nb_ch ))
    {
       inAudio->stream_cnt =  inAudio->apl_stream_nb_ch;
    }
    else if(outAudio->apl_stream_nb_ch < inAudio->apl_stream_nb_ch)
    {
       inAudio->stream_cnt =  outAudio->apl_stream_nb_ch;
       pr_warning(" mismatch nb channel out_stream(%d) and in_stream(%d)\n",outAudio->apl_stream_nb_ch, inAudio->apl_stream_nb_ch );
    }

    for( inAudio->i = 0; inAudio->i < inAudio->stream_cnt ; inAudio->i++)
    {
        if(   (outAudio->apl_period_size ==  inAudio->apl_period_size)
            ||(outAudio->apl_period_size >  inAudio->apl_period_size))
        {
            inAudio->sample_cnt = inAudio->apl_period_size;

            pr_debug_data("copy %d s32 sample to stream s16 buffer[%d]\n", inAudio->sample_cnt, inAudio->i);

            inAudio->s32ibuf = inAudio->s32audiosamples[inAudio->i];
            inAudio->s16obuf = outAudio->s16audiosamples[inAudio->i];
            for (inAudio->j = 0; inAudio->j < inAudio->sample_cnt; inAudio->j++)
                *inAudio->s16obuf++ = (short int)*inAudio->s32ibuf++;
        }
        else
        {
            pr_critical(" mismatch between nb of in_stream(%d) and out_strem(%d) samples \n", inAudio->apl_period_size, outAudio->apl_period_size);
        }
    }

    return inAudio->err;
}



/*******************************************************************************
*
* FUNCTION: apl_internal_print_module_desc
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

APL_STATUS  apl_internal_print_module_desc(const aplAudioSamples_desc* buf_desc, const char* buffer_name)
{
    pr_debug("ENTERED\n");

    if(!buf_desc)
        return APL_ERR_NULL_POINTER;

    if(!buffer_name)
        return APL_ERR_NULL_POINTER;

    /* initialize stream buffer */
    pr_debug("-------------------descriptor of buffer : %s --------------", buffer_name);
    pr_debug("this buffer stream_nb_ch  = %d", buf_desc->apl_stream_nb_ch);
    pr_debug("this buffer audio_format  = %d", buf_desc->apl_audio_format);
    pr_debug("this buffer sample_rate   = %d", buf_desc->apl_sample_rate);
    pr_debug("this buffer period_size   = %d", buf_desc->apl_period_size);

    return APL_ERR_OK;
}

/****************************************
 *  Stream recording API functions
 ****************************************/

static const char        *D_APL_RECORD_PROCESS_FILE_IN_PCM      = "/tmp/apl_process_rec_in.pcm";
static const char        *D_APL_RECORD_PROCESS_FILE_OUT_PCM     = "/tmp/apl_process_rec_out.pcm";
static aplrecord_desc  apl_process_rec_desc = {0};
unsigned int          apl_internal_rec_mode_select = 0;


/*******************************************************************************
*
* FUNCTION: apl_internal_rec_init
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
tAplU32 apl_internal_rec_init(unsigned int mode_select)
{

    pr_message("ENTERED, select recording mode = %d\n", mode_select);

    /* open input test file */
    if(mode_select == 1 || mode_select == 3 || mode_select == 4 )
    {
        remove(D_APL_RECORD_PROCESS_FILE_IN_PCM);
        apl_process_rec_desc.file_in_pcm = fopen (D_APL_RECORD_PROCESS_FILE_IN_PCM,"a");

        if(!apl_process_rec_desc.file_in_pcm)
            return 0;

        fseek(apl_process_rec_desc.file_in_pcm, 0, SEEK_SET);

    }

    if(mode_select == 2 || mode_select == 3 || mode_select == 4 )
    {

        /* create output test file */
        remove(D_APL_RECORD_PROCESS_FILE_OUT_PCM);
        apl_process_rec_desc.file_out_pcm = fopen (D_APL_RECORD_PROCESS_FILE_OUT_PCM,"a");
        if(!apl_process_rec_desc.file_out_pcm)
            return 0;

        fseek(apl_process_rec_desc.file_out_pcm, 0, SEEK_SET);

    }
    else
    {
        /* do nothing */
    }

    apl_process_rec_desc.wdata = 0;
    apl_internal_rec_mode_select =   mode_select;

    return apl_internal_rec_mode_select;
}



/*******************************************************************************
*
* FUNCTION: apl_internal_rec_reinit
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
void apl_internal_rec_reinit(void)
{

    pr_message("ENTERED\n");

    if(!apl_internal_rec_mode_select)
        return;

    /* open input test file */
    if(apl_internal_rec_mode_select == 1 || apl_internal_rec_mode_select == 3 || apl_internal_rec_mode_select == 4 )
    {
        /* close */
        remove(D_APL_RECORD_PROCESS_FILE_IN_PCM);
        if(apl_process_rec_desc.file_in_pcm)
        {
            fclose(apl_process_rec_desc.file_in_pcm);
            apl_process_rec_desc.file_in_pcm = NULL;
        }

        /*recreate */
        apl_process_rec_desc.file_in_pcm = fopen (D_APL_RECORD_PROCESS_FILE_IN_PCM,"a");

        if(!apl_process_rec_desc.file_in_pcm)
            return;

        fseek(apl_process_rec_desc.file_in_pcm, 0, SEEK_SET);
    }

    if(apl_internal_rec_mode_select == 2 || apl_internal_rec_mode_select == 3 || apl_internal_rec_mode_select == 4  )
    {

        /* close */
        remove(D_APL_RECORD_PROCESS_FILE_OUT_PCM);
        if(apl_process_rec_desc.file_out_pcm)
        {
            fclose(apl_process_rec_desc.file_out_pcm);
            apl_process_rec_desc.file_out_pcm = NULL;
        }

        /* recreate output test file */
        apl_process_rec_desc.file_out_pcm = fopen (D_APL_RECORD_PROCESS_FILE_OUT_PCM,"a");
        if(!apl_process_rec_desc.file_out_pcm)
            return ;

        fseek(apl_process_rec_desc.file_out_pcm, 0, SEEK_SET);
    }
    else
    {
        /* do nothing */
    }

    apl_process_rec_desc.wdata = 0;


    return;
}
/*******************************************************************************
*
* FUNCTION: apl_internal_stream_rec_file_write
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
APL_STATUS apl_internal_stream_rec_file_write(unsigned int mode_select,  module_desc* mod, const aplAudioSamples_desc* RecAudio)
{
    tAplUInt   data_size = 0;
    short int*  s16buf = (short int*)NULL;
    long int*  s32buf = (long int*)NULL;

    if(!apl_internal_rec_mode_select || !mode_select)
        return APL_ERR_OK;

    pr_debug("ENTERED\n");


    if(RecAudio->apl_audio_format == SND_PCM_FORMAT_S16_LE)
    {
        s16buf = RecAudio->s16audiosamples[0];
        data_size = RecAudio->apl_period_size * 2;
    }
    else if(RecAudio->apl_audio_format == SND_PCM_FORMAT_S32_LE)
    {
        s32buf = RecAudio->s32audiosamples[0];
        data_size = RecAudio->apl_period_size * 4;
    }
    else
    {
        pr_warning("data format(%d) not supported for recording\n",RecAudio->apl_audio_format );
    }

    if(!data_size)
    {

        pr_warning("number of byte to copy is NULL \n");
        return APL_ERR_OK;
    }


    if(mode_select == 1) /* process_in*/
    {
        if(!apl_process_rec_desc.file_in_pcm)
            return APL_ERR_NULL_POINTER;

        pr_debug_data("recording of input buffer\n");
        if(RecAudio->apl_audio_format == SND_PCM_FORMAT_S16_LE)
        {
            if(!s16buf)
            {
                pr_warning("NULL POINTER audio daten buffer\n");
                return APL_ERR_NULL_POINTER;
            }
            else
                apl_process_rec_desc.wdata = fwrite((void*)s16buf, (size_t)1, data_size, apl_process_rec_desc.file_in_pcm);
        }
        else if(RecAudio->apl_audio_format == SND_PCM_FORMAT_S32_LE)
        {
            if(!s32buf)
            {
                pr_warning("NULL POINTER audio daten buffer\n");
                return APL_ERR_NULL_POINTER;
            }
            else
                apl_process_rec_desc.wdata = fwrite((void*)s32buf, (size_t)1, data_size, apl_process_rec_desc.file_in_pcm);
        }
        else
        {
            pr_warning("data format(%d) not supported for recording\n",RecAudio->apl_audio_format );
        }

    }
    else if(mode_select == 2) /* process_out*/
    {
        if(!apl_process_rec_desc.file_out_pcm)
            return APL_ERR_NULL_POINTER;
        pr_debug("recording of output buffer\n");

        if(RecAudio->apl_audio_format == SND_PCM_FORMAT_S16_LE)
        {
            if(!s16buf)
            {
                pr_warning("NULL POINTER audio daten buffer\n");
                return APL_ERR_NULL_POINTER;
            }
            else
                apl_process_rec_desc.wdata = fwrite((void*)s16buf, (size_t)1, data_size, apl_process_rec_desc.file_out_pcm);
        }
        else if(RecAudio->apl_audio_format == SND_PCM_FORMAT_S32_LE)
        {
            if(!s32buf)
            {
                pr_warning("NULL POINTER audio daten buffer\n");
                return APL_ERR_NULL_POINTER;
            }
            else
                apl_process_rec_desc.wdata = fwrite((void*)s32buf, (size_t)1, data_size, apl_process_rec_desc.file_out_pcm);
        }
        else
        {
            pr_warning("data format(%d) not supported for recording\n",RecAudio->apl_audio_format );
        }
    }
    (void)mod;
    return APL_ERR_OK;
}


