/*******************************************************************************
* FILE : apl.h
*
* SW-COMPONENT : Audio processing library
*
* DESCRIPTION :
*
* AUTHOR : Patrick Rey
*
* COPYRIGHT :
*
* HISTORY : 06.08.2015
* Initial version
*******************************************************************************/



#ifndef __APL_H__
#define __APL_H__

#ifdef __cplusplus
extern "C" {
#endif

/* version information */
#define MAJOR  01
#define MINOR1 01
#define MINOR2 00
#define MINOR3 00


#define VERSION_STRING "apl version for auditorium"
#define VERSION_COMMENT "apl new"

/****************************************************************************
 *                 APL  Constant definitions
 ****************************************************************************/
/** Maximum number of audio channels */
#define APL_MAX_AUDIO_CNT       4

/** Maximum length of a string in the apl module */
#define APL_MAX_STRING_SIZE   200

/** True */
#define aplTRUE                 1
/** False */
#define aplFALSE                0

#define APL_MAX_STREAM_CH       5

/************ Switch state  ************* */
#define aplSwitchOFF            0x00
#define aplSwitchON             0x01
#define aplSwitchREADY          0x03


#define aplFctStop              0
#define aplFctStart             1
#define aplFctRestart           2

typedef enum _tAplCmd
{
    aplCmdUndef = 0,
    aplCmdStartAudProc,
    aplCmdStopAudProc,
    aplCmdFlush,
  aplCmdResetRecording,
  aplCmdDeleteAllRequest
}tAplCmd;



/* operation mode */

typedef enum _tAplOpMode
{
    APL_OM_UNDEF = 0,
    APL_OM_CONS_ONLY,       // module is only consumer, consumes the input buffer an process a result without output data buffer
    APL_OM_PROD_ONLY,       // module is only producer, produce audio data only, e.g a signal generator
    APL_OM_PROD_CONS,       // module consumes the input data buffers and a produces after processing output data buffers, default Operation mode from modules
    APL_OM_ASYNC_CONS_ONLY, // same as OM_CONS_ONLY but run in a own thread context apart from the main thread
    APL_OM_ASYNC_PROD_ONLY  // same as OM_PROD_ONLY but run in a own thread context apart from the main thread
}tAplOpMode;

/* analyze feature */
typedef  enum _tAplAnalyze
{
    aplAnalyzeFeat_Ark_log,
    aplAnalyzeFeat_log_level,
    aplAnalyzeFeat_performance,
    aplAnalyzeFeat_log_journal_enable
}tAplAnalyze;


/****************************************************************************
 *                      APL Data type definition
 ****************************************************************************/

typedef size_t                 tAplSize;
typedef signed char            tAplChar;
typedef int                    tAplBool;
typedef unsigned char          tAplU8;
typedef signed char            tAplI8;
typedef unsigned int           tAplU32;
typedef int                    tAplI32;
typedef unsigned int           tAplUInt;
typedef int                    tAplInt;
typedef unsigned short         tAplU16;
typedef short                  tAplI16;

/****************************************************************************
 *                      Function return types
 ****************************************************************************/

#ifndef APL_STATUS
/** Return type for APL API calls that have a result status */
#define APL_STATUS  tAplInt
/** Return type for APL API calls that do not return a value */
#define APL_VOID    void
#endif

typedef enum _tAplAction
{
    APL_ACT_UNDEF     = 0,
    APL_ACT_GET_PARAM ,
    APL_ACT_SET_PARAM ,
    APL_ACT_RESET_PARAM,
    APL_ACT_SYNC_PARAM
}tAplAction;






/** Handle to apl library returned by aplCreate() */
typedef void* tApl;

/** Audio pointer array used with aplProcess() */
typedef short int* tAplAudio[APL_MAX_AUDIO_CNT];

/** Data identification parameter used with aplSetData() and aplGetData() */
typedef tAplU16 tAplDataID;

typedef struct _tAplStCfg
{
   void(*AplNotifyfct)(tAplDataID /*param_Id*/, tAplAction /* param_action */);
   tAplBool AplIsLoadArkamays;    /* set to TRUE if Arkamys library has to be loaded */
   void(*AplTracecb)(tAplU8 /*trace_lvl*/, char* /* trace_string */);
}tAplStCfg;


typedef struct _aplAudioSamples_desc
{
    gint                apl_aceess_state;          /*0 means wr_acess = rd_acess, >0 overrun (wr_acess > rd_access), <0 underrun (wr_acess < rd_access) */
    unsigned int        apl_stream_nb_ch;                   /*1->mono stream, 2->stereo stream, >2 -> multichannel stream  */
    unsigned int        apl_sample_rate;          /*any, e.g. 8000-192000*/
    unsigned int        apl_period_size;          /*e.g. 64/128/256/512*/
    snd_pcm_format_t    apl_audio_format;              /*format S16, S32*/
    snd_pcm_stream_t    apl_audio_dir;                      /* capture-> in buffer, playback-> out buffer */
    short int*          s16audiosamples[APL_MAX_STREAM_CH];  /* array 16 bits frame buffer pointer */
    long  int*          s32audiosamples[APL_MAX_STREAM_CH]; /* array 32 bits frame buffer pointer */
    gboolean            apl_can_swap;
    gboolean            apl_is_buf_full;
    guint               apl_read_chunk_nb;
    gboolean            apl_read_has_bigger_period;
    guint               apl_write_pos;
    guint               apl_read_pos;
    guint               apl_fifo_current_sample_count;
    guint               apl_fifo_size_max;
    gboolean            apl_fifo_has_frame_avail;
    guint               apl_buffer_mode;



    /**************************
     * function stack variables
     */
    unsigned int        i;
    unsigned int        j;
    short int*          s16ibuf;
    short int*          s16obuf;
    long int*           s32ibuf;
    long int*           s32obuf;
    unsigned int        sample_cnt;
    unsigned int        stream_cnt;
    tAplBool            isBinding;
    APL_STATUS          err;
    unsigned int        ich;
    unsigned int        och;

}aplAudioSamples_desc;


typedef struct _tAplBinding
{
    tAplU8 in;
    tAplU8 out;
}tAplBinding;


typedef struct _tAplBindingMatrix
{
    tAplU8  inchannelnb;
    tAplU8  outchannelnb;
    tAplU8  bindingnb;
    tAplBinding* audbind;
}tAplBindingMatrix;

#define REQ_TYPE_UNDEF              0x0000
#define REQ_TYPE_SET_REQ            0x0001
#define REQ_TYPE_GET_REQ            0x0002
#define REQ_TYPE_SETGET             0x0004
#define REQ_TYPE_READ_DATA_REQ      0x0008
#define REQ_TYPE_CMD_FLUSH          0x0010
#define REQ_TYPE_SET_RESULT         0x0020
#define REQ_TYPE_GET_RESULT         0x0040
#define REQ_TYPE_NOTIFY             0x0080
#define REQ_TYPE_ALL                0xFFFF



typedef struct _ArkParamList ArkParamList;


typedef struct _ReqItem
{
    tAplU32         RequestedParameter;
    tAplU8          ReqType;
    ArkParamList*   pparam;
    void*           data;
}ReqItem;

typedef struct _SetGetParamReqList
{
    ReqItem* ReqList;
    volatile tAplU8   ReqCnt;
    /*new*/
    ReqItem* WaitReqList;
    volatile tAplU8   WaitReqCnt;
}SetGetParamReqList;

typedef struct _NotifyItem
{
    tAplU32     NotifyParameter;
    tAplAction  ActType;
}NotifyItem;

typedef struct _NotificationTable
{
    NotifyItem*   NotList;
    tAplU8        NotCnt;
}NotificationTable;

typedef enum _aplmodstate /* module state */
{
    apl_state_undefined     = 0,    /* set by destroy module_initialize */
    apl_state_initialized   = 1,    /* set by destroy module_initialize */
    apl_state_eol_loaded    = 2,    /* set by destroy module_initialize */
    apl_state_setup         = 3,    /* set by destroy module_initialize */
    apl_state_prepared      = 4,    /* set next to module_prepare call, means ready to start  */
    apl_state_stop          = 5,    /* set next to module_prepare call, means ready to start  */
    apl_state_running       = 6,    /* all functions are running in the module and available  */
    apl_state_running_nop   = 7,    /* running with no audio processing, loopback audioonly, setgetof properties are available */
}aplmodstate;



/****************************************************************************
 *                      Structure types
 ****************************************************************************/


/** Binary information structure. Used with aplInitialize() to load a binary
    configuration.  Will overwrite any configuration settings previously set
    using aplSetData(). */
typedef struct tAplCfg
{
   tAplU16   usSize;                 /**< Binary configuration size */
   tAplU8*   pData;                  /**< Pointer to binary configuration data */
} tAplCfg;

/****************************************************************************
 *                       Define aplSet/GetData identification labels/strings
 ****************************************************************************/


/* Labels used by the aplSetData() and aplGetData() functions to access extended data

   RO - Read only
   WB - Data can only be written before initialization, read anytime
   WA - Data can only be written after initialization, read anytime
   RW - Data can be written, read anytime
   WO - Data can only be written after initialization, never read */

/********************** audio streaming description range : 0-255  ********************/
#define     aplGeneralGrp                   0x0000
#define     aplAllparam                     0x0001
#define     aplFrameShiftIn                 0x0010       /* WB,(tAplU16)*/
#define     aplSampleRateIn                 0x0011       /* WB,(tAplU16)*/
#define     aplChannelCntIn                 0x0012       /* WB,(tAplU16)*/
#define     aplAudioPcmFormatIn             0x0013       /* WB,(tAplU16)*/
#define     aplFrameShiftOut                0x0020       /* WB,(tAplU16)*/
#define     aplSampleRateOut                0x0021       /* WB,(tAplU16)*/
#define     aplChannelCntOut                0x0022       /* WB,(tAplU16)*/
#define     aplAudioPcmFormatOut            0x0023       /* WB,(tAplU16)*/
#define     aplLoopbackActivate             0x0030       /* WB,(tAplU8)*/
#define     aplAudioMode                    0x0031       /* WB,(tAplU8)*/
#define     aplAdminUpdateParamOnProcess    0x8000       /* RW,(tAplU16)*/

/******************** module 0xMMxx : 0x7M00 - 0x7MFF  (7M) module parameter specific range  ******************************************/

/* micro level measurement, range 0x100 - 0x1FF  */
#define     aplModuleMicLvl         0x0001
#define     aplMicLvlWatchSwitch    0x0100       /* WB,(tAplI32)*/
#define     aplMicLvlWatchRefInt    0x0101       /* RW,(tAplU16), time in ms , in the range from 200ms to 65535 ms */
#define     aplMicLvl               0x0102       /* WB,(tAplI16)*/
#define     aplMicLvlWatchStartStop 0x0103       /* WB,(tAplI32)*/
#define     aplMicLvlOperationMode  0x0104       /* WB,(tAplI32)*/



/**************************************************************
 * Arkamys module , range 0x200 - 0x2FF                       *
 * update 02.03.2016:
    - based on spec AIVI_Arkamys_iMx_API_v1.2
    - Add aplArkamysAvcActivate,
          aplArkamysSdeqActivate
          aplArkamysBassExciterActivate
          aplArkamysByPassActivate
          aplArkamysSetData
          aplArkamysGetData
          aplArkamysEolData
          aplArkamysInputLevelDeviation
          aplArkamysCurrentSpeed
          aplArkamysCurrentVolumeStep
          aplArkamysLevelLoss
          aplArkamysLibraryVersion
          aplArkamysAmbienceSelect
 */

#define     aplModuleArkamys                    0x0002
#define     aplArkamysSwitch                    0x0200       /* WB,(tAplI32)*/
#define     aplArkamysAvcActivate               0x0201       /* WB,(tAplI32)*/
#define     aplArkamysSdeqActivate              0x0202       /* WB,(tAplI32)*/
#define     aplArkamysBassExciterActivate       0x0203       /* WB,(tAplI32)*/
#define     aplArkamysReverbActivate            0x0204       /* WB,(tAplI32)*/
#define     aplArkamysByPassActivate            0x0205       /* WB,(tAplI32)*/
#define     aplArkamysSetData                   0x0206       /* WB,(tAplI32)*/
#define     aplArkamysGetData                   0x0207       /* WB,(tAplI32)*/
#define     aplArkamysSetGetData                0x0208       /* WB,(tAplI32)*/
#define     aplArkamysReadData                  0x0209       /* WB,(tAplI32)*/
#define     aplArkamysEolData                   0x020A       /* WB,(tAplI32)*/
#define     aplArkamysInputLevelDeviation       0x020B       /* WB,(tAplI32)*/
#define     aplArkamysCurrentSpeed              0x020C       /* WB,(tAplI32)*/
#define     aplArkamysCurrentVolumeStep         0x020D       /* WB,(tAplI32)*/
#define     aplArkamysLevelLoss                 0x020E       /* WB,(tAplI32)*/
#define     aplArkamysLibraryVersion            0x020F       /* WB,(tAplI32)*/
#define     aplArkamysAmbienceSelect            0x0210       /* WB,(tAplI32[2])*/
#define     aplArkamysCurrVolAttenuation        0x0211       /* WB,(tAplI32)*/
#define     aplArkamysFrameShiftIn              0x0250       /* WB,(tAplU16)*/
#define     aplArkamysSampleRateIn              0x0251       /* WB,(tAplU16)*/
#define     aplArkamysChannelCntIn              0x0252       /* WB,(tAplU16)*/
#define     aplArkamysAudioPcmFormatIn          0x0253       /* WB,(tAplU16)*/
#define     aplArkamysFrameShiftOut             0x0254       /* WB,(tAplU16)*/
#define     aplArkamysSampleRateOut             0x0255       /* WB,(tAplU16)*/
#define     aplArkamysChannelCntOut             0x0256       /* WB,(tAplU16)*/
#define     aplArkamysAudioPcmFormatOut         0x0257       /* WB,(tAplU16)*/
#define     aplArkamysGetLastError              0x0258       /* WB,(tAplI32)*/
#define     aplArkamysResetLastError            0x0259       /* WB,(tAplI32)*/
#define     aplArkamysSwitchState               0x025A       /* WB,(tAplI32)*/
#define     aplArkamysLibraryState              0x025B       /* WB,(tAplI32)*/
#define     aplArkamysFlush                     0x025C       /* WB,(tAplI32)*/
#define     aplArkamysWriteMultiFrame           0x025D       /* WB,(tAplI32)*/





/* pcm copy module, range 0x300 - 0x3FF */
#define     aplModulePcmCopy                0x0003
#define     aplPcmCopySwitch                0x0300       /* WB,(tAplI32)*/
#define     aplAudioChannelBinding          0x0301       /* WB,(tAplI32)*/
#define     aplInRoutingChannelCnt          0x0302       /* WB,(tAplU8)*/
#define     aplOutRoutingChannelCnt         0x0303       /* WB,(tAplU8)*/
#define     aplNbBinding                    0x0304       /* WB,(tAplU8)*/
#define     aplPcmCopyFrameShiftIn          0x0305       /* WB,(tAplU16)*/
#define     aplPcmCopySampleRateIn          0x0306       /* WB,(tAplU16)*/
#define     aplPcmCopyChannelCntIn          0x0307       /* WB,(tAplU16)*/
#define     aplPcmCopyAudioPcmFormatIn      0x0308       /* WB,(tAplU16)*/
#define     aplPcmCopyFrameShiftOut         0x0309       /* WB,(tAplU16)*/
#define     aplPcmCopyampleRateOut          0x0310       /* WB,(tAplU16)*/
#define     aplPcmCopyChannelCntOut         0x0311       /* WB,(tAplU16)*/
#define     aplPcmCopyAudioPcmFormatOut     0x0312       /* WB,(tAplU16)*/

/* volume module */
#define     aplModuleVolume                 0x0004
#define     aplvolumeSwitch                 0x0400       /* WB,(tAplI32)*/


/****************************************************************************
 *                       Non-fatal error codes (0x0000 - 0x000F)
 ****************************************************************************/
    /** No error. */
#define APL_ERR_OK                          0x0000

    /** Result not ready.
        Returned by aplProcess() when audio data is not ready for output
     */
#define APL_ERR_NO_RESULT                   0x0001

    /** Size information only is valid.
        Returned by the get data functions when a null is input for the data
        output pointer, and valid size information is returned for the
        specified data ID.  */
#define APL_ERR_SIZE_ONLY                   0x0002


/****************************************************************************
 *                        General error codes (0x0100 - 0x01FF)
 ****************************************************************************/


    /** A memory allocation error has occurred. */
#define APL_ERR_ALLOC                       0x0100

    /** An illegal null pointer function argument has been used. */
#define APL_ERR_NULL_POINTER                0x0101

    /** Handle not initialized.
        Returned by sseProcess() if @c pMain is null.
        Otherwise indicates an attempt to access functionality that is not
        initialized .
        In addition, submodule functions will return this error if called
        when not initialized. */
#define APL_ERR_MODULE_NOT_INIT             0x0102

    /** Handle not created.
        Returned by aplInitialize() if aplCreate() is not called successfully
        first. */
#define APL_ERR_NOT_CREATED                 0x0103

    /** Handle already initialized.
        Returned by aplCreate() if Main points to a non-null value.
        Returned by aplInitialize() if pMain points to a fully-initialized
        aplMainStruct structure (aplInitialize has been called twice).
        In addition, submodule initialization functions will return this error
        if called incorrectly. */
#define APL_ERR_MODULE_ALREADY_INIT         0x0104


    /** Invalid APL handle.
        Returned by the APL API functions if APL has not been properly initialized before. */
#define APL_ERR_INVALID_MAIN_STRUCT  0x0105

    /** Deinitialization failure.
        Returned by sseDestroy() if an error occurs during destruction. */
#define APL_ERR_DEINIT_FAILED               0x0106

    /** Illegal set request.
        Returned by the set functions if the specified data ID is
        read-only or the requested parameter change is not allowed. */
#define APL_ERR_ILLEGAL_SET_PARAM           0x0108

    /** Illegal set request before initialization.
        Returned by the set data functions when the specified data ID can only
        be set after aplInitialize() has been called. */
#define APL_ERR_ILLEGAL_SET_PARAM_BEFORE_INIT 0x0109

    /** Illegal set request after initialization.
        Returned by the set data functions when the specified data ID must be
        set before aplInitialize() is called. */
#define APL_ERR_ILLEGAL_SET_PARAM_AFTER_INIT 0x010A

    /** Invalid data ID.
        Returned by the set/get data functions when the specified data ID
        is invalid. */
#define APL_ERR_INVALID_PARAM_ID            0x010B

    /** Invalid channel number.
        Returned by the set/get or callback functions when the channel
        number is invalid for the specified data ID. The channel
        number should be between 0 and the maximum of valid channels
        for which the specified parameter exists, minus 1. */
#define APL_ERR_INVALID_CHANNEL_NUMBER      0x010C

    /** Invalid operation mode for set request.
        Returned by the set functions if the requested parameter change
        is illegal given the current operation mode.
        Also returned if a set data function is called between
        aplProcess(). */
#define APL_ERR_INVALID_OPERATION_MODE      0x010D

    /** Invalid size information.
        Returned by the set/get functions when the size is invalid for
        the specified data ID, or if internal scratch buffers are too small for the
        requested data access. */
#define APL_ERR_INVALID_SIZE                0x010E

    /** Invalid parameter.
        Returned at initialization if there is an error in the configuration
        parameters.
        Returned by the set data functions if there was an error in the data
        for the specified data ID.*/

#define APL_ERR_PARAM                       0x010F

    /** Invalid number of parameters.*/
#define APL_ERR_PARAM_CNT                   0x0110

    /** Invalid sample rate. */
#define APL_ERR_SAMPLE_RATE                 0x0111

    /** Invalid frame shift. */
#define APL_ERR_FRAMESHIFT                  0x0112


    /** Error loading prototype or other filter coefficients. */
#define APL_ERR_NULL_FILTER                 0x0116

    /** The requested functionality is not implemented. */
#define APL_ERR_NOT_IMP                     0x0118

    /** The requested functionality is disabled by a preprocessor switch. */
#define APL_ERR_SUBMODULE_DISABLED          0x0119

#define APL_ERR_MODULE_ALREADY_ATTACHED     0x0120
#define APL_ERR_MODULE_NOT_INITIALIZED      0x0121
#define APL_ERR_MODULE_NOT_DATA             0x0122
#define APL_ERR_AUDIO_BUFFER_MISMATCH       0x0123
#define APL_ERR_AUDIO_STREAM_FORMAT_MISMATCH         0x0124

    /** An illegal null pointer function argument has been used. */
#define APL_ERR_NO_ERR_DATA_AVAILABLE       0x0125

    /** Invalid parameter size.*/
#define APL_ERR_PARAM_SIZE                  0x0126

#define APL_ERR_MODULE_ARKAMYS_CREATE       0x0201
#define APL_ERR_MODULE_ARKAMYS_INVALID_WRT_CMD_NB       0x0202

#define APL_ERR_INVALID_REQ_MASK            0x0210


    /** APL status which are not error  */
#define APL_REQ_NOTIFY_PROPERTY_CHANGE      0xF001

/****************************************************************************
 *                        APL API
 ****************************************************************************/

/* library module API */
APL_STATUS aplNew(const tAplStCfg* Aplcfg);
APL_STATUS aplDelete(void);
APL_STATUS aplAnalyzeSetup(tAplAnalyze feature, tAplI32 value);


/* library instance API functions  */
APL_STATUS aplCreate(tApl* hApl);
APL_STATUS aplInitialize(tApl  hApl, const tAplCfg* paplCfg);
APL_STATUS aplSetConfig(tApl   hApl, const tAplCfg* paplCfg);
APL_STATUS aplDestroy(const tApl* hApl);
APL_STATUS aplCommand(const tApl* hApl, tAplCmd cmd);
APL_STATUS aplProcessAudioEnt(tApl  hApl, const aplAudioSamples_desc*  inAudio, const aplAudioSamples_desc*   outAudio);
APL_STATUS aplProcessAudioVoice(tApl  hApl, const aplAudioSamples_desc*  inAudio, const aplAudioSamples_desc*   outAudio);
APL_STATUS aplProcessAudioInfo(tApl  hApl, const aplAudioSamples_desc*  inAudio, const aplAudioSamples_desc*   outAudio);
APL_STATUS aplSetData(tApl hApl,tAplDataID iDataID, tAplInt iChannel, tAplU32 ulSize, const void* pData);
APL_STATUS aplGetData(tApl hApl,tAplDataID iDataID, tAplInt iChannel, tAplU32* pulSize, void* pData);
APL_STATUS aplModuleChainSetup(tApl hApl,const tAplDataID* mlist, tAplU32 size);
APL_STATUS aplGetStatusMessage(const tApl hApl, tAplChar szStatusMessage[APL_MAX_STRING_SIZE]);
APL_STATUS aplGetVersion( tAplI16*   pusVersionArray[4],  tAplChar** pszVersionString,  tAplChar** pszVersionComment);


#ifdef __cplusplus
}
#endif

#endif // __APL_H__
