/*
 * audproc-stream-buffer.c
 *
 * Author : Patrick Rey
 * Date: 24.01.2019
 * description:
 */

#include <glib/gi18n.h>
#include <glib-object.h>
#include <sched.h>

#include "audproc-common-defs.h"
#include "audproc-object.h"
#include "audproc-apl-handler.h"
#include "audproc-alsa.h"
#include "audproc-audio-rts-handler.h"
#include "audproc-service.h"
#include "audproc-configuration.h"


/* Needed for Trace */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define SPM_UNIT_TESTS // solve warning
#include "audproc-audioprocess-trace.h"
#include "etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AUDPROC_ALSA
#include "trcGenProj/Header/audproc-stream-buffer.c.trc.h"
#endif

/*******************************************************************************
              Constant defintion
*******************************************************************************/

#define S_AUDPROC_STREAM_BUFFER_EXPORT      "/tmp/audproc_stream_fifo_proc_buf" //.txt"
#define D_AUDPROC_STREAM_BUFFER_RAM_REC_SIZE_MAX 6000000

/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/

static GMutex   audproc_stream_buffer_lock_l;

#if 0
static long  int*   s32audiosamplesBufRecCh0; //[APL_MAX_STREAM_CH]



static void audproc_stream_buffer_init_rec(aplAudioSamples_desc* this_Buf)
{

    if(s32audiosamplesBufRecCh0)
    {
        g_free(s32audiosamplesBufRecCh0);
        s32audiosamplesBufRecCh0 = (long  int*)NULL;
    }

    /* create recording buffer */
    s32audiosamplesBufRecCh0 = g_malloc0(D_AUDPROC_STREAM_BUFFER_RAM_REC_SIZE_MAX * sizeof(long  int));

 (void)this_Buf;
}
#endif

/*******************************************************************************
*
* FUNCTION: audproc_stream_buffer_init
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int       audproc_stream_buffer_init(void)
{
    gint         err = AUDPROC_ERR_OK;

    g_mutex_init(&audproc_stream_buffer_lock_l);

    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_stream_buffer_destroy
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int       audproc_stream_buffer_finalize(void)
{
    gint         err = AUDPROC_ERR_OK;

    g_mutex_clear(&audproc_stream_buffer_lock_l);

    return err;
}


/*******************************************************************************
*
* FUNCTION: audproc_stream_buffer_desc_init
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int       audproc_stream_buffer_desc_init(aplAudioSamples_desc* this_buf, const aplAudioSamples_desc* init_buf)
{
    gint         err = AUDPROC_ERR_OK;


    this_buf->apl_stream_nb_ch               = init_buf->apl_stream_nb_ch;
    this_buf->apl_period_size                = init_buf->apl_period_size;
    this_buf->apl_audio_format               = init_buf->apl_audio_format;
    this_buf->apl_audio_dir                  = init_buf->apl_audio_dir;
    this_buf->apl_read_has_bigger_period     = init_buf->apl_read_has_bigger_period;
    this_buf->apl_fifo_current_sample_count  = init_buf->apl_fifo_current_sample_count;
    this_buf->apl_fifo_has_frame_avail       = init_buf->apl_fifo_has_frame_avail;


    //audproc_stream_buffer_init_rec(this_Buf);

    return err;
}



/*******************************************************************************
*
* FUNCTION: audproc_stream_buffer_desc_reinit
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int       audproc_stream_buffer_desc_reinit(aplAudioSamples_desc* this_buf)
{
    gint         err = AUDPROC_ERR_OK;

    ETG_TRACE_USR3(("[audproc_stream_buffer_desc_reinit]: ENTERED"));

    if(!this_buf)
        return AUDPROC_ERR_NULL_POINTER;


    /*******************************
     * renit stream buffer descriptor
     *
     *  - retrieve configured input device frame size
     *  - retrieve sample rate input device
     */


    /* configure read audio pipeline */

    /* configure pre_process audio  pipeline */

    g_mutex_lock(&audproc_stream_buffer_lock_l);

    if(this_buf->apl_read_has_bigger_period)
    {
        this_buf->apl_fifo_current_sample_count  = 0;
        this_buf->apl_fifo_has_frame_avail = FALSE;

        ETG_TRACE_USR3(("[audproc_stream_buffer_desc_reinit]: FIFO Buffer was reinitialized"));
    }

    g_mutex_unlock(&audproc_stream_buffer_lock_l);

    ETG_TRACE_USR3(("[audproc_stream_buffer_desc_reinit]: EXIT"));


    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_stream_buffer_add_new_frame
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int       audproc_stream_buffer_add_new_frame(aplAudioSamples_desc* this_buf, aplAudioSamples_desc* in_buf )
{
    gint    err = AUDPROC_ERR_OK;
    unsigned int j =0;

    ETG_TRACE_USR4(("[audproc_stream_buffer_add_new_frame]: ENTERED" ));

    g_mutex_lock(&audproc_stream_buffer_lock_l);

    if((this_buf->apl_fifo_current_sample_count + in_buf->apl_period_size) <= this_buf->apl_fifo_size_max)
    {

        if(in_buf->apl_audio_format == SND_PCM_FORMAT_S16_LE)
        {
            short int*  s16ibuf = (short int*)NULL;
            short int*  s16obuf = (short int*)NULL;

            for( j = 0; j < this_buf->apl_stream_nb_ch ; j++)
            {
                s16ibuf = in_buf->s16audiosamples[j];
                s16obuf = this_buf->s16audiosamples[j]; /*clean*/
                s16obuf += this_buf->apl_fifo_current_sample_count;
                memcpy((void*)s16obuf, (void*)s16ibuf, in_buf->apl_period_size * sizeof(short int));
                ETG_TRACE_USR4(("[audproc_stream_buffer_add_new_frame]:copy sample in FIFO, current sample count -> %d", this_buf->apl_fifo_current_sample_count));
            }
        }
        else if(in_buf->apl_audio_format == SND_PCM_FORMAT_S32_LE || in_buf->apl_audio_format == SND_PCM_FORMAT_S24_LE)
        {
            long int*  s32ibuf = (long int*)NULL;
            long int*  s32obuf = (long int*)NULL;

            for( j = 0; j < this_buf->apl_stream_nb_ch ; j++)
            {
                s32ibuf = in_buf->s32audiosamples[j];
                s32obuf = this_buf->s32audiosamples[j]; /*clean*/
                s32obuf += this_buf->apl_fifo_current_sample_count;
                memmove((void*)s32obuf, (void*)s32ibuf, in_buf->apl_period_size * sizeof(long int));

                ETG_TRACE_USR4(("[audproc_stream_buffer_add_new_frame]:copy %d sample in FIFO ch %d at position %d", in_buf->apl_period_size,\
                                                                                                                 j,\
                                                                                                                 this_buf->apl_fifo_current_sample_count));
            }
        }
        else
        {
            /* do nothing */
            pr_warning("invalide this_buf->apl_audio_format = %d", in_buf->apl_audio_format);
        }

        this_buf->apl_fifo_current_sample_count += in_buf->apl_period_size;

        ETG_TRACE_USR4(("[audproc_stream_buffer_add_new_frame]:current sample count in FIFO -> %d, max size is -> %d", this_buf->apl_fifo_current_sample_count,\
                                                                                                                this_buf->apl_fifo_size_max));

    }
    else
        ETG_TRACE_USR4(("[audproc_stream_buffer_add_new_frame]: fifo already full"));

    g_mutex_unlock(&audproc_stream_buffer_lock_l);

    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_stream_buffer_frame_is_consumed
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int       audproc_stream_buffer_frame_is_consumed(aplAudioSamples_desc* this_buf)
{
    gint        err = AUDPROC_ERR_OK;
    guint       period_cnt = 1;
    unsigned int j =0;

    ETG_TRACE_USR4(("[audproc_stream_buffer_frame_is_consumed]: ENTERED" ));

    if(!this_buf)
    {
        ETG_TRACE_USR4(("[audproc_stream_buffer_frame_is_consumed]: invalid buffer descriptor" ));
        return AUDPROC_ERR_NO_INSTANCE_AVAILABLE;
    }

    g_mutex_lock(&audproc_stream_buffer_lock_l);

    if(this_buf->apl_read_has_bigger_period)
    {

        ETG_TRACE_USR4(("[audproc_stream_buffer_frame_is_consumed]: this is a FIFO buffer, remove a period frame and sfift to left aligned" ));

        /* calcule remaining FIFO sample count and shift aligned to left side */
        this_buf->apl_fifo_current_sample_count -= this_buf->apl_period_size;

        if(this_buf->apl_audio_format == SND_PCM_FORMAT_S16_LE)
        {
            short int*  s16ibuf = (short int*)NULL;
            short int*  s16obuf = (short int*)NULL;

            /* depend on the format */
            for(j = 0; j < this_buf->apl_stream_nb_ch ; j++)
            {
                s16ibuf = this_buf->s16audiosamples[j];
                s16ibuf += this_buf->apl_period_size; //apl_fifo_current_sample_count;
                s16obuf = this_buf->s16audiosamples[j];
                memmove((void*)s16obuf, (void*)s16ibuf, this_buf->apl_fifo_current_sample_count * sizeof(short int));

                ETG_TRACE_USR4(("[audproc_stream_buffer_frame_is_consumed]: shift %d samples in the FIFO left aligned", this_buf->apl_fifo_current_sample_count ));
            }
        }
        else if(this_buf->apl_audio_format == SND_PCM_FORMAT_S32_LE || this_buf->apl_audio_format == SND_PCM_FORMAT_S24_LE)
        {
            long int*  s32ibuf = (long int*)NULL;
            long int*  s32obuf = (long int*)NULL;

            /* depend on the format */
            for(j = 0; j < this_buf->apl_stream_nb_ch ; j++)
            {
                ETG_TRACE_USR4(("[audproc_stream_buffer_frame_is_consumed]: shift samples in the FIFO channel %d", j));

                s32ibuf = this_buf->s32audiosamples[j];
                s32ibuf += this_buf->apl_period_size; //this_buf->apl_fifo_current_sample_count;
                s32obuf = this_buf->s32audiosamples[j];
                memmove((void*)s32obuf, (void*)s32ibuf, this_buf->apl_fifo_current_sample_count * sizeof(long int));

                ETG_TRACE_USR4(("[audproc_stream_buffer_frame_is_consumed]: shift %d samples in the FIFO left aligned", this_buf->apl_fifo_current_sample_count ));
            }
        }

        ETG_TRACE_USR4(("[audproc_stream_buffer_frame_is_consumed]: remaining samples in the FIFO -> %d", this_buf->apl_fifo_current_sample_count ));


        /* the fifo has still an available full period frame for processing  */
        guint period_cnt = this_buf->apl_fifo_current_sample_count / this_buf->apl_period_size;

        if(period_cnt)
        {
            this_buf->apl_fifo_has_frame_avail = TRUE;
            ETG_TRACE_USR4(("[audproc_stream_buffer_frame_is_consumed]: a full frame period is still available in the buffer"));
        }
        else
            this_buf->apl_fifo_has_frame_avail = FALSE;
    }
    else
    {
        ETG_TRACE_USR4(("[audproc_stream_buffer_frame_is_consumed]: the buffer descriptor is not a Fifo, there is no action required"));
        this_buf->apl_fifo_has_frame_avail = FALSE;
    }


    g_mutex_unlock(&audproc_stream_buffer_lock_l);

    ETG_TRACE_USR4(("[audproc_stream_buffer_frame_is_consumed]: EXIT"));

    (void)period_cnt;

    return err;
}


/*******************************************************************************
*
* FUNCTION: audproc_stream_buffer_lock
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int       audproc_stream_buffer_lock(aplAudioSamples_desc* this_buf)
{
    gint         err = AUDPROC_ERR_OK;
    (void)this_buf;
    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_stream_buffer_unlock
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int       audproc_stream_buffer_unlock(aplAudioSamples_desc* this_buf)
{
    gint         err = AUDPROC_ERR_OK;

   (void)this_buf;
    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_stream_is_frame_pending_avail
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
extern gboolean  audproc_stream_buffer_is_frame_pending_avail(aplAudioSamples_desc* this_buf)
{
    gboolean         avail = FALSE;

    g_mutex_lock(&audproc_stream_buffer_lock_l);
    avail = this_buf->apl_fifo_has_frame_avail;
    g_mutex_unlock(&audproc_stream_buffer_lock_l);

    ETG_TRACE_USR4(("[audproc_stream_buffer_is_frame_pending_avail]: avail -> %d", avail));

    return avail;
}

