/*******************************************************************************
* FILE : audproc-apl-handler.c
*
* SW-COMPONENT :
*
* DESCRIPTION : This module provides an adapatation layer for the service
                to the SSE Engine from Nuance
*
* AUTHOR : Patrick Rey (ESE)
*
* COPYRIGHT : (C)
*
* HISTORY : 07.08.2015
*
*
* Initial version
*******************************************************************************/

#include <glib/gi18n.h>
#include <glib-object.h>
#include <sched.h>

#include "audproc-common-defs.h"
#include "audproc-object.h"
#include "audproc-apl-handler.h"
#include "audproc-alsa.h"
#include "audproc-service.h"
#include "audproc-configuration.h"


/* Needed for Trace */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define SPM_UNIT_TESTS // solve warning
#include "audproc-audioprocess-trace.h"
#include "etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AUDPROC_APL
#include "trcGenProj/Header/audproc-apl-handler.c.trc.h"
#endif
/*******************************************************************************
              VERSION COMPILER
*******************************************************************************/


/*******************************************************************************
              DEFINES
*******************************************************************************/
#define APL_NB_MAX_INST 5
//#define D_AUDPROC_APL_HANDLER_TEST

/*******************************************************************************
              Constant defintion
*******************************************************************************/


/*******************************************************************************
              STRUCTURE DEFINITION
*******************************************************************************/

/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/

extern GMainLoop *audproc_main_loop;


static GSource*     mic_lvl_time_source;
static AplState**   m_aplInst;
static guint8       m_aplInstcnt;
static GMutex       m_audproc_apl_lock;


/*******************************************************************************
              STATIC FUNCTION  DEFINITION
*******************************************************************************/

static int  audproc_apl_handler_get_instance_index(const gchar* instance_name)
{
    AplState*   p = (AplState*)NULL;
    int         i = 0;


    pr_debug("ENTERED: instance name: %s\n", (instance_name? instance_name: NULL));

    if(!instance_name)
    {
        pr_debug("instance name not available\n");
        return -1;
    }

    if(!m_aplInstcnt)
    {
        pr_debug("no instance available\n");
        return -1;
    }

    if(m_aplInst)
    {
        for(i = 0; i < APL_NB_MAX_INST; i++) //APL_NB_MAX_INST; i++)
        {
            p = m_aplInst[i];
            if(p)
            {
                if(p->instance_name)
                {
                    if(!g_strcmp0(p->instance_name, instance_name))
                    {
                        pr_debug("active instance found for instance name: %s \n", instance_name);
                        return i;
                    }
                }
            }
        }
    }

    pr_debug("no active instance found for instance name: %s \n", instance_name);
    return -1;
}


static int  audproc_apl_handler_destroy_this_instance(const gchar* instance_name)
{
    AplState*   p = (AplState*)NULL;
    int         idx = 0;


    pr_message("ENTERED: instance name: %s\n", (instance_name? instance_name: NULL));
    ETG_TRACE_USR3(("[audproc_apl_handler_destroy_this_instance]: ENTERED: instance name: %s", (instance_name? instance_name: NULL)));

    if(!instance_name)
    {
        pr_debug("instance name not available\n");
        return AUDPROC_ERR_APL_NO_INSTANCE;
    }

    /* first remove all open apl instances if any */
    if(m_aplInst)
    {
        idx  = audproc_apl_handler_get_instance_index(instance_name); //p = audproc_apl_handler_get_instance(instance_name);
        if(    idx < 0
            || idx >= (int)APL_NB_MAX_INST)
            return AUDPROC_ERR_APL_NO_INSTANCE;

        p = m_aplInst[idx];
        if(p)
        {
            if(p->Aplh)
                (void)aplDestroy(&p->Aplh);
            if(p->instance_name)
            {
                /* release instance */
                pr_message("apl instance %s was destroyed\n", (p->instance_name? p->instance_name: NULL));
                ETG_TRACE_USR3(("[audproc_apl_handler_destroy_this_instance]: apl instance %s was destroyed", (p->instance_name? p->instance_name: NULL)));

                g_free(p->instance_name);
                p->instance_name = NULL;
            }

            g_free(m_aplInst[idx]);
            m_aplInst[idx] = NULL;
            p = NULL;
        }
    }

    ETG_TRACE_USR3(("[audproc_apl_handler_destroy_this_instance]: EXIT"));
    pr_message("EXIT\n");
    return AUDPROC_ERR_OK;
}


static int  audproc_apl_handler_destroy_this_instance_1(AplState*   p)
{
    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_apl_handler_destroy_this_instance_1]: ENTERED"));

    if(!p)
    {
        pr_debug("apl instance not available\n");
        return AUDPROC_ERR_APL_NO_INSTANCE;
    }

    if(p->Aplh)
        (void)aplDestroy(&p->Aplh);
    if(p->instance_name)
    {
        /* release instance */
        pr_message("apl instance %s was destroyed\n", (p->instance_name? p->instance_name: NULL));
        ETG_TRACE_USR3(("[audproc_apl_handler_destroy_this_instance_1]: apl instance %s was destroyed", (p->instance_name? p->instance_name: NULL)));
        g_free(p->instance_name);
        p->instance_name = NULL;
    }
    ETG_TRACE_USR3(("[audproc_apl_handler_destroy_this_instance_1]: EXIT"));
    pr_message("EXIT\n");
    return AUDPROC_ERR_OK;
}


static void  audproc_apl_handler_close_resource(void)
{
    AplState*   p = (AplState*)NULL;
    int         i = 0;
    int         err = AUDPROC_ERR_OK;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_apl_handler_close_resource]: ENTERED"));

    /* first remove all open apl instances if any */
    if(m_aplInst)
    {
        for(i = 0; i < APL_NB_MAX_INST; i++)
        {
            p = m_aplInst[i];

            pr_message("Go to next apl instance[%d]\n", i);
            if(p)
            {
                pr_message("try to close apl instance[%d]\n", i);
                if(p->instance_name)
                {
                    //err = audproc_apl_handler_destroy_this_instance(p->instance_name);
                    err = audproc_apl_handler_destroy_this_instance_1(p);
                    if(!err)
                    {
                        if(m_aplInstcnt > 0)
                            --m_aplInstcnt;
                        pr_message("instance[%d] is closed, current instance count[%d] \n", i, m_aplInstcnt);
                        ETG_TRACE_USR3(("[audproc_apl_handler_close_resource]: instance[%d] is closed, current instance count[%d]", i, m_aplInstcnt));

                        g_free(m_aplInst[i]);
                        m_aplInst[i] = NULL;
                    }
                }
                p = NULL;
            }
        }
        g_free(m_aplInst);
        m_aplInst = NULL;


        ETG_TRACE_USR3(("[audproc_apl_handler_close_resource]: all apl instances freed"));
        pr_message("all apl instances freed\n");
    }

    ETG_TRACE_USR3(("[audproc_apl_handler_close_resource]: EXIT"));
    pr_message("EXIT\n");

    return;
}

static int  audproc_apl_handler_command_this_instance(const gchar* instance_name, tAplCmd  cmd)
{
    AplState*   p = (AplState*)NULL;
    int         idx = 0;


    pr_message("ENTERED: instance name: %s with command(%d)\n", (instance_name? instance_name: NULL), cmd);
    ETG_TRACE_USR3(("[audproc_apl_handler_command_this_instance]: ENTERED"));

    /* first remove all open apl instances if any */
    if(m_aplInst)
    {
        idx  = audproc_apl_handler_get_instance_index(instance_name); //p = audproc_apl_handler_get_instance(instance_name);
        if(    idx < 0
            || idx >= (int)APL_NB_MAX_INST)
            return AUDPROC_ERR_APL_NO_INSTANCE;

        p = m_aplInst[idx];
        if(p)
        {
            if(p-> AplInitState == APL_INIT_FINALIZE_INST)
            {
                (void)aplCommand(&p->Aplh, cmd);
                if(p->instance_name)
                {
                    pr_debug("command(%d) to apl instance %s was send\n", cmd, (p->instance_name? p->instance_name: NULL));
                }
            }
        }
    }

    ETG_TRACE_USR3(("[audproc_apl_handler_command_this_instance]: EXIT"));
    return AUDPROC_ERR_OK;
}

static AplState*  audproc_apl_handler_set_new_instance(const gchar* instance_name)
{
    AplState*   p = (AplState*)NULL;
    int         i = 0;


    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_apl_handler_set_new_instance]: ENTERED"));

    /* first remove all open apl instances if any */
    if(m_aplInst)
    {
        for(i = 0; i < APL_NB_MAX_INST; i++)
        {
            p = m_aplInst[i];
            if(!p)
            {
                p = g_malloc0(sizeof(AplState));
                if(p)
                {
                    p->instance_name = g_strdup(instance_name);
                    p->ref_cnt = 1;
                    m_aplInst[i] = p;

                    if(254 == m_aplInstcnt)   /* fix me : do not let the counter overlapping */
                       m_aplInstcnt = 0;

                    /* new instance */
                    m_aplInstcnt++;

                    pr_message("a new instance(%s) was created, current instance count is:%d \n", instance_name, m_aplInstcnt);

                    return p;
                }
            }
            else
            {
                if(!g_strcmp0(p->instance_name, instance_name))
                {
                    /*increment the ref count */
                    //p->ref_cnt++;

                    pr_message("this instance(%s) exist already, the current ref count is:%d \n", instance_name, p->ref_cnt);
                    return p;
                }
            }
        }
    }

    ETG_TRACE_USR3(("[audproc_apl_handler_set_new_instance]: no active instance found for instance name: %s", instance_name));
    pr_message("no active instance found for instance name: %s \n", instance_name);
    return (AplState*)NULL;
}


static AplState*  audproc_apl_handler_get_this_instance(const gchar* instance_name)
{

    pr_debug("ENTERE\n");

    AplState*   p = (AplState*)NULL;

    if(m_aplInst)
    {
        int         i = 0;
        for(i = 0; i < APL_NB_MAX_INST; i++)
        {
            p = m_aplInst[i];
            if(p)
            {
                if(p->instance_name)
                {
                    if(!g_strcmp0(p->instance_name, instance_name))
                    {
                        pr_debug("active instance found for instance name: %s \n", instance_name);
                        return p;
                    }
                }
            }
        }
    }

    pr_debug("no active instance found for instance name: %s \n", instance_name);

    return(AplState*)NULL;
}

static void AplTracecb(tAplU8 trace_lvl, char* trace_string)
{
    if(!trace_string)
    {
        ETG_TRACE_ERR(("[AUDPROC_LIB_ERR]: AplTracecb: invalid log descriptor"));
        pr_warning("invalid log descriptor\n");
        return;
    }
    if(trace_lvl >= 1)  /* pr_message*/
    {
        ETG_TRACE_USR4(("[AUDPROC_LIB_MSG]: %s", trace_string ));
    }
    else if(trace_lvl >= 2) /* pr_debug*/
    {
        ETG_TRACE_USR4(("[AUDPROC_LIB_DBG]: %s", trace_string ));
    }
    else if(trace_lvl >= 3) /* pr_debug_data*/
    {
        ETG_TRACE_USR4(("[AUDPROC_LIB_DBG]: %s", trace_string ));
    }
    else if(trace_lvl >= 4) /* pr_warning*/
    {
        ETG_TRACE_ERR(("[AUDPROC_LIB_WARN]: %s", trace_string ));
    }
    else
    {
        pr_warning("not support trace level \n");
    }
    return;
}


/*******************************************************************************
              FUNCTION  DECLARATION
*******************************************************************************/

/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_process_audio_ent
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/

int audproc_apl_handler_process_audio_ent
(
    AplState*           aplinst,
     aplAudioSamples_desc* InAudio,
    const aplAudioSamples_desc* OutAudio
)
{
    int   err = APL_ERR_NOT_CREATED;

    pr_debug("ENTERED\n");
    ETG_TRACE_USR4(("[audproc_apl_handler_process_audio_ent]: ENTERED" ));

    if(!aplinst)
        return AUDPROC_ERR_NO_INSTANCE_AVAILABLE;


    if(aplinst->Aplh)
    {

             /*------ signal processing function of the apl module -*/
            err =  aplProcessAudioEnt(aplinst->Aplh, InAudio, OutAudio);

            if(!err)
                err = audproc_stream_buffer_frame_is_consumed(InAudio);

    }
    else
    {
        pr_warning("invalid apl instance \n");
        ETG_TRACE_USR2(("[audproc_apl_handler_process_audio_ent]: invalid apl instance"));
    }

    ETG_TRACE_USR4(("[audproc_apl_handler_process_audio_ent]: EXIT"));
    return err;
}

#if 0
static int aplProcessAudioInfo_s16
(
    audproc_alsa_state*             inst,
    const aplAudioSamples_desc*     inAudio,
    aplAudioSamples_desc*           outAudio,
    gboolean                        can_swap
)
{
    unsigned int    i;
    short int*       s16ibuf;
    short int*       s16obuf;
    unsigned int    sample_cnt = 0;
    unsigned int    stream_cnt = 0;


    pr_debug("ENTERED\n");

    if(!inAudio)
        return AUDPROC_ERR_NULL_POINTER;

    if(!outAudio)
        return AUDPROC_ERR_NULL_POINTER ;



    s16ibuf = inAudio->s16audiosamples[0];
    if(!s16ibuf)
    {
        pr_debug("No input data stream buffer available \n");
        return AUDPROC_ERR_NULL_POINTER;
    }

    if(outAudio->apl_period_size <  inAudio->apl_period_size)
    {
        pr_critical("mismatch in size out_stream(%d) and in_stream(%d) \n", outAudio->apl_period_size, inAudio->apl_period_size);
        return AUDPROC_ERR_AUDIO_BUFFER_MISMATCH;
    }


    /* in stream and out stream are aligned */
    if(   (outAudio->apl_stream_nb_ch == inAudio->apl_stream_nb_ch)
        ||(outAudio->apl_stream_nb_ch > inAudio->apl_stream_nb_ch ))
    {
       stream_cnt =  inAudio->apl_stream_nb_ch;
    }
    else if(outAudio->apl_stream_nb_ch < inAudio->apl_stream_nb_ch)
    {
       stream_cnt =  outAudio->apl_stream_nb_ch;
       pr_warning(" mismatch nb channel out_stream(%d) and in_stream(%d)\n",outAudio->apl_stream_nb_ch, inAudio->apl_stream_nb_ch );
    }

    for( i = 0; i < stream_cnt ; i++)
    {
        if(   (outAudio->apl_period_size ==  inAudio->apl_period_size)
            ||(outAudio->apl_period_size >  inAudio->apl_period_size))
        {
            unsigned int ich = i;
            unsigned int och = i;


            sample_cnt = inAudio->apl_period_size;

            if(can_swap)
            {
                pr_debug ("swap stream buffer[%d]\n", ich);

                s16ibuf = inAudio->s16audiosamples[ich];
                outAudio->s16audiosamples[och] = s16ibuf;
            }
            else
            {
                pr_debug("copy %d s16 sample from in_stream_s16_buffer[%d] to out_stream_s16_buffer[%d]\n", sample_cnt, ich, och);

                s16ibuf = inAudio->s16audiosamples[ich];
                s16obuf = outAudio->s16audiosamples[och];
                memcpy((void*)s16obuf, (void*)s16ibuf, sample_cnt * sizeof(long int));
            }
        }
        else
        {
            pr_critical(" mismatch between nb of in_stream(%d) and out_strem(%d) samples \n", inAudio->apl_period_size, outAudio->apl_period_size);
        }
    }

  /* now write data in eth loop device */
  //(void)audproc_audio_rts_handler_Loop_write(inst, (void**)outAudio->s32audiosamples);

    //To remove Gen4 Prio2 Compiler Warning
    (void)inst;

    return AUDPROC_ERR_OK;
}



static int aplProcessAudioInfo_s32
(
    audproc_alsa_state*             inst,
    const aplAudioSamples_desc*     inAudio,
    aplAudioSamples_desc*           outAudio,
    gboolean                        can_swap
)
{
    unsigned int    i;
    long int*       s32ibuf;
    long int*       s32obuf;
    unsigned int    sample_cnt = 0;
    unsigned int    stream_cnt = 0;


    pr_debug("ENTERED\n");

    if(!inAudio)
        return AUDPROC_ERR_NULL_POINTER;

    if(!outAudio)
        return AUDPROC_ERR_NULL_POINTER ;



    s32ibuf = inAudio->s32audiosamples[0];
    if(!s32ibuf)
    {
        pr_debug("No input data stream buffer available \n");
        return AUDPROC_ERR_NULL_POINTER;
    }

    if(outAudio->apl_period_size <  inAudio->apl_period_size)
    {
        pr_critical("mismatch in size out_stream(%d) and in_stream(%d) \n", outAudio->apl_period_size, inAudio->apl_period_size);
        return AUDPROC_ERR_AUDIO_BUFFER_MISMATCH;
    }


    /* in stream and out stream are aligned */
    if(   (outAudio->apl_stream_nb_ch == inAudio->apl_stream_nb_ch)
        ||(outAudio->apl_stream_nb_ch > inAudio->apl_stream_nb_ch ))
    {
       stream_cnt =  inAudio->apl_stream_nb_ch;
    }
    else if(outAudio->apl_stream_nb_ch < inAudio->apl_stream_nb_ch)
    {
       stream_cnt =  outAudio->apl_stream_nb_ch;
       pr_warning(" mismatch nb channel out_stream(%d) and in_stream(%d)\n",outAudio->apl_stream_nb_ch, inAudio->apl_stream_nb_ch );
    }

    for( i = 0; i < stream_cnt ; i++)
    {
        if(   (outAudio->apl_period_size ==  inAudio->apl_period_size)
            ||(outAudio->apl_period_size >  inAudio->apl_period_size))
        {
            unsigned int ich = i;
            unsigned int och = i;


            sample_cnt = inAudio->apl_period_size;

            if(can_swap)
            {
                pr_debug ("swap stream buffer[%d]\n", ich);

                s32ibuf = inAudio->s32audiosamples[ich];
                outAudio->s32audiosamples[och] = s32ibuf;
            }
            else
            {
                pr_debug("copy %d s32 sample from in_stream_s32_buffer[%d] to out_stream_s32_buffer[%d]\n", sample_cnt, ich, och);

                s32ibuf = inAudio->s32audiosamples[ich];
                s32obuf = outAudio->s32audiosamples[och];
                memcpy((void*)s32obuf, (void*)s32ibuf, sample_cnt * sizeof(long int));
            }
        }
        else
        {
            pr_critical(" mismatch between nb of in_stream(%d) and out_strem(%d) samples \n", inAudio->apl_period_size, outAudio->apl_period_size);
        }
    }

  /* now write data in eth loop device */
  //(void)audproc_audio_rts_handler_Loop_write(inst, (void**)outAudio->s32audiosamples);

    //To remove Gen4 Prio2 Compiler Warning
    (void)inst;

    return AUDPROC_ERR_OK;
}


/*******************************************************************************
*
* FUNCTION: aplProcessAudioInfo_
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static int aplProcessAudioInfo_
(
    audproc_alsa_state*             aplinst,
    const aplAudioSamples_desc*     inAudio,
    aplAudioSamples_desc*           outAudio,
    gboolean                        can_swap
)
{

    int err     = AUDPROC_ERR_OK;


    pr_debug("ENTERED\n");


    if(!inAudio)
        return AUDPROC_ERR_NULL_POINTER;

    if(!outAudio)
        return AUDPROC_ERR_NULL_POINTER ;


    if(   (inAudio->apl_audio_format == SND_PCM_FORMAT_S16_LE)
        &&(outAudio->apl_audio_format == SND_PCM_FORMAT_S16_LE))
    {
        err = aplProcessAudioInfo_s16(aplinst, inAudio, outAudio, FALSE);
    }
    else if(  (inAudio->apl_audio_format == SND_PCM_FORMAT_S32_LE)
            &&(outAudio->apl_audio_format == SND_PCM_FORMAT_S32_LE))
    {
        err = aplProcessAudioInfo_s32(aplinst, inAudio, outAudio, FALSE);
    }
    else
    {
        /* do nothing */
        pr_warning("inAudio->apl_audio_format = %d, mdata->buf_desc->apl_audio_format = %d \n", inAudio->apl_audio_format, outAudio->apl_audio_format);
    }


    //To remove Gen4 Prio2 Compiler Warning
    (void)can_swap;
    (void)err;

    return AUDPROC_ERR_OK;
}

#endif
/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_process_audio_info
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/

int audproc_apl_handler_process_audio_info
(
    AplState*                   aplinst,
    const aplAudioSamples_desc* InAudio,
    const aplAudioSamples_desc* OutAudio
)
{

    if(!aplinst)
       return AUDPROC_ERR_NO_INSTANCE_AVAILABLE;

    if(aplinst->Aplh)
    {
        /*------ signal processing function of the apl module -*/
        //return  aplProcessAudioInfo_(aplinst, InAudio, OutAudio, FALSE);
        return  aplProcessAudioInfo(aplinst->Aplh, InAudio, OutAudio); //, FALSE);
    }
    else
    {
        pr_warning("no apl valid instance \n");
    }

    return APL_ERR_NOT_CREATED;
}


/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_process_audio_voice
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/

int audproc_apl_handler_process_audio_voice
(
    AplState*                   aplinst,
    const aplAudioSamples_desc* InAudio,
    const aplAudioSamples_desc* OutAudio
)
{

    if(!aplinst)
       return AUDPROC_ERR_NO_INSTANCE_AVAILABLE;

    if(aplinst->Aplh)
    {
        /*------ signal processing function of the apl module -*/
        return  aplProcessAudioVoice(aplinst->Aplh, InAudio, OutAudio);
    }
    else
    {
        pr_warning("no apl valid instance \n");
    }

    return APL_ERR_NOT_CREATED;
}

/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_create
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: apl error
*
*******************************************************************************/
int audproc_apl_handler_create(AplState* aplinst)
{
    int err = AUDPROC_ERR_OK;

    pr_debug("Entered \n");

    if(!aplinst)
       return AUDPROC_ERR_NO_INSTANCE_AVAILABLE;

    if(aplinst->Aplh)
        return AUDPROC_ERR_OK;

    err = aplCreate(&aplinst->Aplh);

    if (!aplinst->Aplh)
        pr_critical("apl instance creation failed \n");

    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_get_data
*
* DESCRIPTION: To retrieve the data set
*
* PARAMETER: [IN] Id of the data to be retrieved
*
* RETURNVALUE: int.
*
*******************************************************************************/
int audproc_apl_handler_get_data
(
    AplState*           aplinst,
    const tAplDataID    DataID,
    const unsigned int  iChannel,
    unsigned int*       pSize,
    void*               pData
)
{
    int err = AUDPROC_ERR_OK;
    gboolean    docallgetData = TRUE;

    pr_debug("ENTERED");


    if(    (DataID != aplAudioMode)
        && !aplinst)
        return AUDPROC_ERR_NO_INSTANCE_AVAILABLE;

    if(DataID  == aplAudioMode)
        docallgetData = FALSE;
    else
    {
        if(!aplinst->Aplh)
        {
            pr_warning("no APL instance available");
            err = APL_ERR_NOT_CREATED ;
        }
    }

    if(!err)
    {
        /* here take care of parameter specifical handling  */
        switch(DataID)
        {
            case aplAudioMode:
            {
                /* */
                gint IS_AudioMode;

                IS_AudioMode =  audproc_configuration_get_audio_mode();

                memcpy(pData, (void*)&(IS_AudioMode), *pSize);

                break;
            }
            default:
                break;

        }

        /* Get parameter */
        if(docallgetData)
            err = aplGetData(aplinst->Aplh, DataID, (tAplInt)iChannel, pSize, pData );

        if(*pSize <= 4)
        {
            int ivalue = 0;
            memcpy((void*)&ivalue, pData, *pSize);
            pr_debug("parameter: %d , size : %d, value: %d\n", DataID, *pSize, ivalue);
        }
        else
            pr_debug("parameter: %d , size : %d\n", DataID, *pSize );
    }
    else
    {
        pr_warning("no APL instance available");
        err = APL_ERR_NOT_CREATED ;
    }

    pr_debug("EXIT");
    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_set_data
*
* DESCRIPTION: To set data in teh sse Engine
*
* PARAMETER: [IN] Id of the data to be retrieved
*
* RETURNVALUE: int.
*
*******************************************************************************/
int audproc_apl_handler_set_data
(
    AplState*           aplinst,
    const tAplDataID    DataID,
    const unsigned int  iChannel,
    unsigned int        iSize,
    void*               pData
)
{
    int         err = AUDPROC_ERR_OK;
    gboolean    docallsetData = TRUE;

    ETG_TRACE_USR3(("[audproc_apl_handler_set_data}: DataId:%d, Channel:%d, Size:%d", DataID, iChannel, iSize));
    pr_debug("ENTERED: DataId:%d, Channel:%d, Size:%d", DataID, iChannel, iSize);

    if(!aplinst)
    {
        pr_warning("apl instance is NULL\n");
        return AUDPROC_ERR_NO_INSTANCE_AVAILABLE;
    }

    if(!pData)
    {
        pr_warning("no data available\n");
        return AUDPROC_ERR_NULL_POINTER;
    }

    if(aplinst->Aplh)
    {
        /* here take care of parameter specifical handling  */
        switch(DataID)
        {
            case aplMicLvlWatchRefInt:
            {
                tAplU32    IS_MicLvlInt = 0;
                tAplU16  REQ_MicLvlInt = 0;
                guint32  size = 2;

                memcpy((void*)&(REQ_MicLvlInt), pData , iSize);

                /* retrieve current parameter value */
                err = audproc_apl_handler_get_data(aplinst, aplMicLvlWatchRefInt, 0, &size, (void*)&IS_MicLvlInt);
                if(!err)
                {
                    if(IS_MicLvlInt != REQ_MicLvlInt)
                    {
                        pr_debug("change MiclvlInt value from %d to %d\n", IS_MicLvlInt, REQ_MicLvlInt);

                        /* destroy time out resource automatically */
                        tAplI32 REQ_aplFctCtrl = aplFctRestart;
                        err = aplSetData(aplinst->Aplh, aplMicLvlWatchStartStop, 0, sizeof(tAplI32),(void*) &REQ_aplFctCtrl );
                    }
                }
                break;
            }
            case aplAudioMode:
            {
                /* */
                guint32 REQ_AudioMode;

                memcpy((void*)&(REQ_AudioMode), pData , iSize);

                switch((guint8)REQ_AudioMode)
                {
                    case D_AUDIO_MODE_CLASSIC:
                        err = audproc_configuration_set_audio_mode_classic();
                        break;
                    case D_AUDIO_MODE_AUDITORIUM:
                        err = audproc_configuration_set_audio_mode_auditorium();
                        break;
                    case D_AUDIO_MODE_PREMIUM:
                        err = audproc_configuration_set_audio_mode_premium();
                        break;
                    case D_AUDIO_MODE_EOL_RELOAD:
                        err = audproc_configuration_set_audio_mode_eol_reload();
                        break;
                    default:
                        pr_warning("audioMode(%d) not supported\n", (guint8)REQ_AudioMode);
                        break;
                }

                docallsetData = FALSE;

                break;
            }
            case aplLoopbackActivate:
            {
                /* */
                guint32 REQ_LoopbackAct;

                memcpy((void*)&(REQ_LoopbackAct), pData , iSize);

                if(REQ_LoopbackAct)
                    audproc_service_loopback_activate(TRUE);
                else
                    audproc_service_loopback_activate(FALSE);

                docallsetData = FALSE;

                break;
            }
            default:
                break;
        }

        /* set parameter */
        if(docallsetData)
        {
            err = aplSetData(aplinst->Aplh, DataID, (tAplInt)iChannel, iSize, pData );
            ETG_TRACE_USR3(("[audproc_apl_handler_set_data}: DataId ->%d, Channel->%d, Size->%d", DataID, iChannel, iSize));
        }
    }
    else
    {
        pr_debug("no apl valid instance");
        err = APL_ERR_NOT_CREATED ;
    }

    if(err)
    {
        pr_debug("aplSetData for parameter:%d  is failed with error: %d \n", DataID, err);
    }

    ETG_TRACE_USR3(("[audproc_apl_handler_set_data}: EXIT"));
    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_initialize
*
* DESCRIPTION: To set data in the apl Engine
*
* PARAMETER: [IN] Id of the data to be retrieved
*
* RETURNVALUE: int.
*
*******************************************************************************/
int audproc_apl_handler_initialize
(
    const gchar* instance_name,
    const tAplCfg*  paplCfg
)
{
    int err = AUDPROC_ERR_OK;
    AplState* pa =(AplState*)NULL;
    const gchar*  iname = (gchar*)NULL;


    pr_debug("ENTERED: instance name:%s\n", (instance_name? instance_name: NULL));
    ETG_TRACE_USR3(("[audproc_apl_handler_initialize]: ENTERED"));

    /* get instance if available otherwise create a new one */
    if(!instance_name)
        iname = APL_INSTANCE_STR_ALL_FCTS;
    else
        iname = instance_name;

    pa = audproc_apl_handler_get_this_instance(iname);

    if(!pa)
    {
        pa = audproc_apl_handler_set_new_instance(iname); //instance_name);
        if(!pa)
            return AUDPROC_ERR_ALLOC;

        /* create a new apl instance */
        err = aplCreate(&pa->Aplh);
        if(err != APL_ERR_OK)
        {
            audproc_critical(err, "audproc_apl_handler_create_instance");
            return err;
        }
    }

    if(pa->Aplh)
    {
        if(!paplCfg)
        {
            pr_debug("apl initialization w/o configuration data");
            ETG_TRACE_USR3(("[audproc_apl_handler_initialize]: apl initialization w/o configuration data"));
            err = aplInitialize(pa->Aplh, NULL);
        }
        else
        {
            pr_debug("apl initialization with configuration data");
            ETG_TRACE_USR3(("[audproc_apl_handler_initialize]: apl initialization with configuration data"));
            err = aplInitialize(pa->Aplh, paplCfg);
        }

        if(!err)
            pa->AplInitState = APL_INIT_FINALIZE_INST;
    }
    else
    {
        pr_warning("no apl instance available");
        ETG_TRACE_USR1(("[audproc_apl_handler_initialize]: no apl instance available"));
        err = APL_ERR_NOT_CREATED ;
    }

    pr_debug("EXIT with instance state: %d", pa->AplInitState);
    ETG_TRACE_USR3(("[audproc_apl_handler_initialize]: EXIT"));

    return err;
}



/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_is_instance
*
* DESCRIPTION: To set data in the apl Engine
*
* PARAMETER: [IN] Id of the data to be retrieved
*
* RETURNVALUE: int.
*
*******************************************************************************/
int audproc_apl_handler_is_instance
(
    const gchar* instance_name
)
{
    AplState* pa =(AplState*)NULL;

    pr_debug("ENTERED: instance name:%s\n", (instance_name? instance_name: NULL));

    /* get instance if available otherwise create a new one */
    if(!instance_name)
        return APL_NO_INST;

    pa = audproc_apl_handler_get_this_instance(instance_name);

    if(!pa)
        return APL_NO_INST;

    /* return instance state  */
    pr_debug("instance state is: %d", pa->AplInitState);

    return pa->AplInitState;
}




/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_create_instance
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

AplState*  audproc_apl_handler_create_instance
(
    const gchar*    instance_name,
    int             mode
)
{
    int         err = AUDPROC_ERR_OK;
    AplState*   p   = (AplState*)NULL;

    pr_debug("ENTERED: instance name:%s, mode:%d\n", (instance_name? instance_name: NULL), mode);
    ETG_TRACE_USR3(("[audproc_apl_handler_create_instance]: ENTERED, instance name:%s", (instance_name? instance_name: NULL)));


    /* Reinitialize every time
     * We need to do this, because the application will configure SSE after each init.
     * This is required because we can't receive messages if the audio route is not configured.
     * Therefore we have to provide a well defined status after each call to init.
     *
     * Delete a previous instance of apl
     * If the previous instance was already deleted or there was no previous
     * instance, m_sseState.pMain will be NULL which is a valid input for sseDestroy()
     */

    p = audproc_apl_handler_get_this_instance(instance_name);

    if(p)
    {
        if(mode & APL_DESTROY_PREVIOUS_INST )
        {
            if(p->Aplh)
            {
                err = aplDestroy(&p->Aplh);
                if(err != APL_ERR_OK)
                {
                    pr_debug("failed to apl Destroy");
                    return (AplState*)NULL; //APL_ERR_ALLOC;
                }
            }
        }
        else
        {
            /*increment the ref count */
            p->ref_cnt++;

            pr_message("this instance(%s) exist already, the current ref count is:%d \n", instance_name, p->ref_cnt);
            ETG_TRACE_USR3(("[audproc_apl_handler_create_instance]: this instance(%s) exist already", instance_name));
            //pr_debug("instance still exist and no destroy and recreate is required per configuration, instance state is:%d\n", p->AplInitState);
            return  p;
        }
    }

    /* Create an instance of apl
     */

    p = audproc_apl_handler_set_new_instance(instance_name);

    if(!p)
    {
        audproc_critical(err, "audproc_apl_handler_create_instance");
        return (AplState*)NULL;
    }


    if(!p->Aplh)
    {
        err = aplCreate(&p->Aplh);
        if(err != APL_ERR_OK)
        {
            audproc_critical(err, "audproc_apl_handler_create_instance");
            return (AplState*)NULL; //err;
        }
        else
        {
            /* If initialisation was successful, store it in m_sseState.SSEInitOk.
             * Otherwise leave it as FALSE so we know that the setup failed inside the callback function
             */
            pr_debug("new instance (%s) from apl was created", p->instance_name);
            ETG_TRACE_USR3(("[audproc_apl_handler_create_instance]: new instance (%s) from apl was created",  p->instance_name));

            if(mode & APL_INIT_PREPARE_INST )
                p->AplInitState = APL_INIT_PREPARE_INST;
            else
                p->AplInitState = APL_INIT_FINALIZE_INST; //AplInitOk = TRUE;
        }
    }

    pr_debug("EXIT with instance state: %d\n", p->AplInitState);
    ETG_TRACE_USR3(("[audproc_apl_handler_create_instance]: EXIT"));
    return p;
}



/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_destroy
*
* DESCRIPTION: THis function initialize the sse Engine with the passed ConfigId
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int audproc_apl_handler_destroy (const gchar* instance_name)
{
    int         err = AUDPROC_ERR_OK;
    AplState*   p = (AplState*)NULL;

    pr_message("ENTERED");
    ETG_TRACE_USR3(("[audproc_apl_handler_destroy]: ENTERED"));

    if(!instance_name)
        return AUDPROC_ERR_NULL_POINTER;

    p = audproc_apl_handler_get_this_instance(instance_name);
    if(!p)
    {
        pr_warning("no apl library instance available for name(%s)\n", instance_name);
        ETG_TRACE_ERR(("[audproc_apl_handler_destroy]: no apl library instance available for name(%s)", instance_name));
        return APL_ERR_NOT_CREATED ;
    }

    if(p->ref_cnt > 1)
    {
        /* discrement ref count by one */
        p->ref_cnt--;
        pr_message("the apl instance(%s) count is dicreased by one, current count is: %d \n",instance_name, p->ref_cnt );
    }
    else
    {
        /* now destroy this instance */
        err = audproc_apl_handler_destroy_this_instance(instance_name);
    }

    ETG_TRACE_USR3(("[audproc_apl_handler_destroy]: EXIT"));
    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_reset
*
* DESCRIPTION: THis function initialize the sse Engine with the passed ConfigId
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int audproc_apl_handler_command (const gchar* instance_name, tAplCmd cmd)
{
    int err = AUDPROC_ERR_OK;

    g_mutex_lock(&m_audproc_apl_lock);

    pr_message("requested command(%d)\n", cmd);

    if(!instance_name)
        err =  AUDPROC_ERR_NULL_POINTER;

    if(!err)
    {
        AplState*   p = audproc_apl_handler_get_this_instance(instance_name);
        if(!p)
        {
            pr_warning("no apl library instance available for name(%s)\n", instance_name);
            err = APL_ERR_NOT_CREATED ;
        }
    }

    if(!err)
        err =  audproc_apl_handler_command_this_instance(instance_name, cmd);

    g_mutex_unlock(&m_audproc_apl_lock);

    return err;
}




/*******************************************************************************
*
* FUNCTION: audproc_apl_get_version
* DESCRIPTION: ends debug session
*
* PARAMETER: [check xml for details]
*
* RETURNVALUE: void
*
*******************************************************************************/
void audproc_apl_handler_get_version
(
    const int**         pVersionArray,
    const char**        pVersionString,
    const char**        pVersionComment
)
{

    /* lint 715 */
    *pVersionArray      =  (int*)NULL;
    *pVersionString     =  (char*)NULL;
    *pVersionComment    =  (char*)NULL;
    /* lint 715 */

    return;
}

/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_get_Status_Message
* DESCRIPTION:
*
* PARAMETER: [check xml for details]
*
* RETURNVALUE: int
*
*******************************************************************************/
int audproc_apl_handler_get_Status_Message( char* pStatusMessage)
{

    /* lint 756*/
    pStatusMessage = (char*)NULL;
    (void)pStatusMessage;
    //err = aplGetStatusMessage(m_aplState.Aplh, pStatusMessage);

    return AUDPROC_ERR_OK;
}



/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_init
* DESCRIPTION:
*
* PARAMETER: [check xml for details]
*
* RETURNVALUE: int
*
*******************************************************************************/
int  audproc_apl_handler_init(tAplStCfg* cfg)
{
    APL_STATUS err = APL_ERR_OK;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_apl_handler_init]: ENTERED"));

    /* registry log cb function for ttfis tracing*/
    cfg->AplTracecb = AplTracecb;

    /* apl lib creation */
    err = aplNew(cfg);

    /* first remove all open apl instances if any */
    audproc_apl_handler_close_resource();

    /* setup structure*/
    if(!m_aplInst)
    {
        m_aplInst = g_malloc0(APL_NB_MAX_INST * sizeof(AplState*));
        pr_message("create a new apl handler instance\n");
        ETG_TRACE_USR3(("[audproc_apl_handler_init]: create a new apl handler instance"));
    }

    /*initialize mutex */
    g_mutex_init(&m_audproc_apl_lock);
    m_aplInstcnt = 0;

    ETG_TRACE_USR3(("[audproc_apl_handler_init]: EXIT"));
    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_finalize
* DESCRIPTION:
*
* PARAMETER: [check xml for details]
*
* RETURNVALUE: int
*
*******************************************************************************/
void  audproc_apl_handler_finalize(void)
{
    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_apl_handler_finalize]: ENTERED"));

    /* remove all apl associated resources*/
    audproc_apl_handler_close_resource();

    /* free GSource
     * We did an attach() with the GSource, so we need to
     * destroy() it
     */
#if 0 /* both calls lead sometime to a reset */
     if(mic_lvl_time_source)
        g_source_destroy(mic_lvl_time_source);

    /* We need to unref() the GSource to end the last reference
     * we got
     */
    if(mic_lvl_time_source)
        g_source_unref(mic_lvl_time_source);
#endif


    /* now delete / close the apl library instance */
    (void)audproc_apl_handler_delete();


    g_mutex_clear(&m_audproc_apl_lock);
    ETG_TRACE_USR3(("[audproc_apl_handler_finalize]: EXIT"));

    return;
}


/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_delete
* DESCRIPTION:
*
* PARAMETER: [check xml for details]
*
* RETURNVALUE: int
*
*******************************************************************************/
void  audproc_apl_handler_delete(void)
{
    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_apl_handler_delete]: ENTERED"));

    /* apl lib finalize */
    (void)aplDelete();


    return;
}


/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_get_instance
* DESCRIPTION:
*
* PARAMETER: [check xml for details]
*
* RETURNVALUE: int
*
*******************************************************************************/

AplState*  audproc_apl_handler_get_instance(const gchar* instance_name)
{

    g_mutex_lock(&m_audproc_apl_lock);

    AplState*   p = (AplState*)NULL;

    pr_debug("ENTERED: instance name: %s\n", (instance_name? instance_name: NULL));

    if(!m_aplInstcnt)
    {
        pr_debug("no instance available\n");
    }
    else if(!instance_name)
    {
        pr_warning("pointer requested instance name is NULL\n");
    }
    else
        p = audproc_apl_handler_get_this_instance(instance_name);

    g_mutex_unlock(&m_audproc_apl_lock);

    if(!p)
        pr_debug("no active instance found for instance name: %s \n", instance_name);

    return p;
}


/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_trigger_mic_level_cb
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/

static gboolean  audproc_apl_handler_trigger_mic_level_cb(void)
{

    pr_debug("ENTERED");

    AplState*   aplinst = audproc_apl_handler_get_this_instance(APL_INSTANCE_STR_MICRO_LEVEL_MONITORING_FCTS);
    guint32     size;

    if(!aplinst)
    {
        pr_debug("no apl instance available for the source AUD_PROC_SRC_MICIN \n");
        return FALSE;
    }

    size = 4;
    tAplI32     IS_aplswitch;
    (void)audproc_apl_handler_get_data(aplinst, aplMicLvlWatchSwitch, 0, &size, (void*)&IS_aplswitch);

    if(IS_aplswitch == aplSwitchOFF)
    {
        pr_debug("Terminate micro level monitoring\n");
        return FALSE;
    }

    size = 4;
    tAplI32     IS_aplFctCtrl;
    (void)audproc_apl_handler_get_data(aplinst, aplMicLvlWatchStartStop, 0, &size, (void*)&IS_aplFctCtrl);

    if(IS_aplFctCtrl == aplFctStop)
    {
        tAplI32 REQ_aplFctCtrl = aplFctStart;
        (void)audproc_apl_handler_set_data(aplinst, aplMicLvlWatchStartStop, 0, sizeof(tAplI32),(void*) &REQ_aplFctCtrl );
        return TRUE;
    }
    else if(IS_aplFctCtrl == aplFctRestart)
    {
        pr_debug("restart time out with new value\n");
        return FALSE;
    }
    else
    {
        /* do nothing */
    }

    return TRUE;
}

/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_miclvl_end_notify_cb
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static void  audproc_apl_handler_miclvl_end_notify_cb(void)
{

    pr_debug("ENTERED");

    tAplU32     MicLvlInt = 0;
    guint32     size;
    gint        err = AUDPROC_ERR_OK;
    tAplI32     aplswitch = 0;

    AplState*    aplinst = audproc_apl_handler_get_this_instance(APL_INSTANCE_STR_MICRO_LEVEL_MONITORING_FCTS);

    if(!aplinst)
    {
        pr_debug("no apl instance available for the source AUD_PROC_SRC_MICIN\n");
        return;
    }


    size = 4;
    (void)audproc_apl_handler_get_data(aplinst, aplMicLvlWatchSwitch, 0, &size, (void*)&aplswitch);

    if(aplswitch == aplSwitchOFF)
    {
        pr_debug("Terminate micro level monitoring\n");
        return;
    }

    /* retrieve current parameter value */
    size = 2;
    err = audproc_apl_handler_get_data(aplinst, aplMicLvlWatchRefInt, 0, &size, (void*)&MicLvlInt);

    if(!err)
        audproc_apl_handler_set_miclvl_timeout((guint16)MicLvlInt);

    return;
}


/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_set_miclvl_timeout
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void audproc_apl_handler_set_miclvl_timeout(guint16 timeout_ms)
{

    pr_debug("ENTERED: timeout(%d)\n", timeout_ms);

    tAplI32     aplswitch = 0;
    AplState*   aplinst = audproc_apl_handler_get_this_instance(APL_INSTANCE_STR_MICRO_LEVEL_MONITORING_FCTS);

    if(!aplinst)
    {
        pr_debug("fail while creating a new apl instance for the source AUD_PROC_SRC_MICIN \n");
        return;
    }

    if(timeout_ms < D_MIC_LEVEL_TIME_OUT_INTERVAL_MIN)
    {
        pr_debug(" try to set an  interval time out (%d) out of range, should be > 100ms\n",timeout_ms );
        timeout_ms = D_MIC_LEVEL_TIME_OUT_INTERVAL_MIN;
    }

    mic_lvl_time_source = g_timeout_source_new ((guint)timeout_ms);
    g_source_set_priority (mic_lvl_time_source, G_PRIORITY_DEFAULT);
    g_source_set_callback (mic_lvl_time_source, (GSourceFunc)audproc_apl_handler_trigger_mic_level_cb, NULL, (GDestroyNotify)audproc_apl_handler_miclvl_end_notify_cb);

    GMainContext* context = g_main_loop_get_context(audproc_main_loop);
    g_source_attach(mic_lvl_time_source, context);
    g_source_unref(mic_lvl_time_source);

    aplswitch = aplTRUE;
    (void)audproc_apl_handler_set_data(aplinst, aplMicLvlWatchStartStop, 0, sizeof(tAplI32),(void*) &aplswitch );

    /* make resource while CB ended automatically restart */
    aplinst->mic_lvl_cb_return = TRUE;

    return;
}


/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_chain_set_up
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int audproc_apl_handler_chain_set_up
(
    AplState*           aplinst,
    const tAplDataID*   chain_order,
    int                 nb_module
)
{
    int err = AUDPROC_ERR_OK;

    pr_message("ENTERED\n");


    if(!aplinst)
    {
        pr_debug("no apl instance available\n");
        return err;
    }

    err = aplModuleChainSetup(aplinst->Aplh, chain_order, (tAplU32)nb_module);

    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_analyze_setup
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int    audproc_apl_handler_analyze_setup
(
    tAplAnalyze feature,
    tAplI32 value
)
{
    int err = AUDPROC_ERR_OK;

    pr_message("ENTERED\n");

    err = aplAnalyzeSetup(feature,value);

    return err;
}


/*******************************************************************************
*
* FUNCTION: audproc_apl_handler_analyze_setup
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int audproc_apl_handler_get_ark_lib_ver
(
    void*         pData
)
{
    int             err = AUDPROC_ERR_OK;
    gint32          aplswitch = aplSwitchON;
    unsigned char   *eol_data = (unsigned char*)NULL;
    guint32         size = 0;
    AplState*       aplinst = (AplState*)NULL;


    if(!pData)
        return AUDPROC_ERR_NULL_POINTER;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_apl_handler_get_ark_lib_ver]: ENTERED"));

    aplinst = audproc_apl_handler_get_instance(APL_INSTANCE_STR_ARKAMYS_FCTS);

    if(!aplinst)
    {
        aplinst = audproc_apl_handler_create_instance(APL_INSTANCE_STR_ARKAMYS_FCTS, APL_INIT_PREPARE_INST);

        if(aplinst)
        {
            err = audproc_apl_handler_set_data(aplinst, aplArkamysSwitch, 0, sizeof(tAplI32),(void*) &aplswitch );

            /* finalize apl initialization */
            err = audproc_apl_handler_initialize(APL_INSTANCE_STR_ARKAMYS_FCTS, NULL) ;

#ifdef D_APL_HANDLER_FCT_GET_ARK_LIB_VER_LOAD_EOL_DATA
            if(!err)
            {

                eol_data = audproc_datapool_access_get_eol_dataset(&size);

                if(size && eol_data)
                {
                    err = audproc_apl_handler_set_data(aplinst, aplArkamysEolData, 0, size,(void*)eol_data );
                }
            }
#endif
            size = 12;
            err = audproc_apl_handler_get_data(aplinst, aplArkamysLibraryVersion, 0, &size, pData); //current_imx_ark_lib_vers);
            if(err)
                ETG_TRACE_USR1(("[audproc_apl_handler_get_ark_lib_ver]: fail to retrieve the arkamys lib version with err-> %d", err));

            err = audproc_apl_handler_destroy(APL_INSTANCE_STR_ARKAMYS_FCTS);
        }
    }

    ETG_TRACE_USR3(("[audproc_apl_handler_get_ark_lib_ver]: EXIT"));

    (void)eol_data;
    return err;

}
