/*
 * audproc-alsa.c
 *
 * Author : Patrick Rey
 * Date: 07.08.2015
 * description:
 */

#include <glib/gi18n.h>
#include <glib-object.h>
#include <sched.h>

#include "audproc-common-defs.h"
#include "audproc-object.h"
#include "audproc-apl-handler.h"
#include "audproc-alsa.h"
#include "audproc-audio-rts-handler.h"
#include "audproc-service.h"
#include "audproc-configuration.h"


/* Needed for Trace */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define SPM_UNIT_TESTS // solve warning
#include "audproc-audioprocess-trace.h"
#include "etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AUDPROC_ALSA
#include "trcGenProj/Header/audproc-alsa.c.trc.h"
#endif

/*******************************************************************************
              Constant defintion
*******************************************************************************/
#define RTS_START_TIMEOUT_MS 50 //1000 //50 //1000
//#define D_MIN_R_W_COUNT_BEFORE_SIGNAL_READ 2
#define D_MONITOR_SILENCE_INJECTION

/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/
static GMutex   audproc_alsa_silence_lock;
static gboolean audproc_silence_detect = FALSE;
static int      silence_detect_cnt = 0;
static int      silence_detect_release_cnt = 0;
#define D_SILENCE_DETECT_THRES          24 //4
#define D_SILENCE_DETECT_RELEASE_THRES  12  //3

/*******************************************************************************
*
* FUNCTION: audproc_alsa_set_thread_priority
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void audproc_alsa_set_thread_priority(audproc_alsa_state* state, guint thread_priority)
{
    if(!state)
        return;

    g_mutex_lock(&state->lock);
    state->alsa_thread_priority = thread_priority;
    g_mutex_unlock(&state->lock);
}


/*******************************************************************************
*
* FUNCTION: audproc_alsa_get_thread_priority
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
guint audproc_alsa_get_thread_priority(audproc_alsa_state* state)
{
    guint thread_priority;

    if(!state)
        return 0;

    g_mutex_lock(&state->lock);
    thread_priority = state->alsa_thread_priority;
    g_mutex_unlock(&state->lock);

    return thread_priority;
}



/*******************************************************************************
*
* FUNCTION: audproc_alsa_get_audio_run_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: run state
*
*******************************************************************************/
gboolean audproc_alsa_get_audio_run_state(audproc_alsa_state* state)
{
    gboolean run = FALSE;

    if(!state)
        return FALSE;

    g_mutex_lock(&state->lock);
    run = state->run;
    g_mutex_unlock(&state->lock);

    return run;
}


/*******************************************************************************
*
* FUNCTION: audproc_alsa_set_audio_thread_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*               true : alsa thread is active
*               false: alsa thread was terminated
*******************************************************************************/
gboolean audproc_alsa_set_audio_thread_state(gboolean thread_state, audproc_alsa_state* state)
{
    gboolean retstate = FALSE;


    if(state)
    {
        g_mutex_lock(&state->lock);
        state->athread_active = thread_state;

        /* release loopback when streaming thread is termintated  */
        if(!thread_state)
        {
            /* handling for loopback device */
            state->select_loopback = FALSE;

            /* reset alsa instance state variables to force termination of a pending StartAudio
               request  */
               state->initialized = FALSE;
               state->run = FALSE;

            /*  signal send to wake up from call of g_cond_wait_until()
                within the audproc_alsa_start_ent_instance() function */
            g_cond_broadcast(&state->update);
        }

        pr_debug("current alsa audio thread state(%d)\n", state->athread_active);
        g_mutex_unlock(&state->lock);
    }


    return retstate;
}

/*******************************************************************************
*
* FUNCTION: audproc_alsa_get_audio_thread_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*               true : alsa thread is active
*               false: alsa thread was terminated
*******************************************************************************/
static gboolean audproc_alsa_get_audio_thread_state(audproc_alsa_state* state)
{
    gboolean retstate = FALSE;

    if(state)
    {
        g_mutex_lock(&state->lock);
        retstate = state->athread_active;
        pr_debug("current alsa audio thread state is %d",retstate);
        g_mutex_unlock(&state->lock);
    }

    return retstate;
}


/*******************************************************************************
*
* FUNCTION: audproc_alsa_streaming_initialize
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int audproc_alsa_streaming_initialize(audproc_alsa_state* state)//, audproc_alsa_stream_cfg* data)
{
    gint         err = AUDPROC_ERR_OK;
    guint        in_ch_cnt, out_ch_cnt;
    audproc_alsa_stream_cfg* data = (audproc_alsa_stream_cfg*)NULL;


    pr_message("ENTER");
    ETG_TRACE_USR3(("[audproc_alsa_streaming_initialize]: ENTERED"));

    if(!state)
        return AUDPROC_ERR_NULL_POINTER;

    if(!state->audproc_stream_data)
        return AUDPROC_ERR_NULL_POINTER;

    data = state->audproc_stream_data;
    if (!data)
        return AUDPROC_ERR_NULL_POINTER;

    /*******************************
     * allocate stream buffer
     *
     *  - retrieve configured input device frame size
     *  - retrieve sample rate input device
     */


    guint inframe_size = data->indev.audproc_period_frames;
    guint insr         = data->indev.audproc_rate;


    /* configure read audio pipeline */
    state->InAudio.apl_stream_nb_ch = data->indev.audproc_nb_ch;
    state->InAudio.apl_sample_rate  = data->indev.audproc_rate;
    state->InAudio.apl_period_size  = data->indev.audproc_period_frames;
    state->InAudio.apl_audio_format = data->indev.audproc_format;
    state->InAudio.apl_audio_dir    = SND_PCM_STREAM_CAPTURE;

#if 0
    /* configure pre_process audio  pipeline */
    memset(&desc,0, sizeof(aplAudioSamples_desc));

    desc.apl_stream_nb_ch = data->indev.audproc_nb_ch;
    desc.apl_sample_rate  = data->indev.audproc_rate;
    desc.apl_period_size  = data->proc_frame_size; // InAudio.apl_period_size;
    desc.apl_audio_format = data->indev.audproc_format;
    desc.apl_audio_dir    = SND_PCM_STREAM_CAPTURE;
    desc.apl_read_has_bigger_period = FALSE;
    desc.apl_fifo_current_sample_count    = 0;
    desc.apl_fifo_has_frame_avail = FALSE;

    (void)audproc_stream_buffer_desc_init(&state->PreProcess, &desc);
#endif

    state->PreProcess.apl_stream_nb_ch = data->indev.audproc_nb_ch;
    state->PreProcess.apl_sample_rate  = data->indev.audproc_rate;
    state->PreProcess.apl_period_size  = data->proc_frame_size; // InAudio.apl_period_size;
    state->PreProcess.apl_audio_format = data->indev.audproc_format;
    state->PreProcess.apl_audio_dir    = SND_PCM_STREAM_CAPTURE;
    state->PreProcess.apl_read_has_bigger_period = FALSE;
    state->PreProcess.apl_fifo_current_sample_count    = 0;
    state->PreProcess.apl_fifo_has_frame_avail = FALSE;


    /* configures capture stream buffer*/
    for(in_ch_cnt = 0; in_ch_cnt < data->indev.audproc_nb_ch; in_ch_cnt++)
    {
        if(data->indev.audproc_format == SND_PCM_FORMAT_S16_LE)
        {
            if(!state->InAudio.s16audiosamples[in_ch_cnt])
            {
                state->InAudio.s16audiosamples[in_ch_cnt] = g_malloc0(inframe_size * sizeof(short));
                pr_debug("allocate read buffer for dev_%s_ch_%d with frame size(%d), sample rate(%d), format(%d)\n",    data->indev.audproc_dev_name,\
                                                                                                                        in_ch_cnt,\
                                                                                                                        inframe_size,\
                                                                                                                        insr, \
                                                                                                                        data->indev.audproc_format);

                ETG_TRACE_USR3(("[audproc_alsa_streaming_initialize]: in_period -> %d, proc_period->%d",state->InAudio.apl_period_size, state->PreProcess.apl_period_size));

                /* process and input stream same property, can_swap is active*/
                if(state->InAudio.apl_period_size  == state->PreProcess.apl_period_size)
                {
                    state->PreProcess.s16audiosamples[in_ch_cnt] = state->InAudio.s16audiosamples[in_ch_cnt];
                    state->PreProcess.apl_can_swap = TRUE;
                }
                else if(state->InAudio.apl_period_size  > state->PreProcess.apl_period_size)
                {
                    ETG_TRACE_USR3(("[audproc_alsa_streaming_initialize]: in_period > proc_period"));

                    state->PreProcess.apl_can_swap = FALSE;
                    state->OutAudio.apl_read_pos = 0;

                    if(!(state->InAudio.apl_period_size  % state->PreProcess.apl_period_size))
                    {
                        ETG_TRACE_USR3(("[audproc_alsa_streaming_initialize]: period size alignement"));

                        state->PreProcess.s16audiosamples[in_ch_cnt] = g_malloc0(state->PreProcess.apl_period_size * sizeof(short));

                        if(state->PreProcess.apl_period_size)
                            state->PreProcess.apl_read_chunk_nb = state->InAudio.apl_period_size / state->PreProcess.apl_period_size;
                        else
                            pr_critical("invalid frame size of PreProcess Buffer\n");
                    }
                    else
                    {
                        ETG_TRACE_USR3(("[audproc_alsa_streaming_initialize]: period size non alignement, in_period-> %d, proc_period -> %d", state->InAudio.apl_period_size , state->PreProcess.apl_period_size ));
                        guint mult = (state->InAudio.apl_period_size  / state->PreProcess.apl_period_size) + 1;


                        state->PreProcess.s16audiosamples[in_ch_cnt] = g_malloc0(mult * (state->PreProcess.apl_period_size * sizeof(short)));
                        state->PreProcess.apl_read_has_bigger_period = TRUE;
                        state->PreProcess.apl_fifo_size_max = mult * state->PreProcess.apl_period_size;

                        ETG_TRACE_USR3(("[audproc_alsa_streaming_initialize]: FIFO mode activated, a fifo was created with size -> %d", state->PreProcess.apl_fifo_size_max ));
                    }
                }
                else if(state->InAudio.apl_period_size  < state->PreProcess.apl_period_size)
                {
                    state->PreProcess.s16audiosamples[in_ch_cnt] = g_malloc0(state->PreProcess.apl_period_size * sizeof(short));
                    state->PreProcess.apl_can_swap = FALSE;
                    state->PreProcess.apl_write_pos = 0;
                }
            }
            else
            {
                pr_debug("stream buffer already available for dev_%s_ch_%d with frame size(%d), sample rate(%d), format(%d)\n",    data->indev.audproc_dev_name,\
                                                                                                                        in_ch_cnt,\
                                                                                                                        inframe_size,\
                                                                                                                        insr, \
                                                                                                                        data->indev.audproc_format);
            }
        }
        else if(data->indev.audproc_format == SND_PCM_FORMAT_S32_LE || data->indev.audproc_format == SND_PCM_FORMAT_S24_LE)
        {
            if(!state->InAudio.s32audiosamples[in_ch_cnt])
            {
                state->InAudio.s32audiosamples[in_ch_cnt] = g_malloc0(inframe_size * sizeof(long int));
                pr_debug("allocate read buffer for dev_%s_ch_%d with frame size(%d), sample rate(%d), format(%d)\n",    data->indev.audproc_dev_name,\
                                                                                                                        in_ch_cnt,\
                                                                                                                        inframe_size,\
                                                                                                                        insr, \
                                                                                                                        data->indev.audproc_format);

                ETG_TRACE_USR3(("[audproc_alsa_streaming_initialize]: in_period -> %d, proc_period->%d",state->InAudio.apl_period_size, state->PreProcess.apl_period_size));

                /* process and input stream same property, can_swap is active*/
                if(state->InAudio.apl_period_size  == state->PreProcess.apl_period_size)
                {
                    state->PreProcess.s32audiosamples[in_ch_cnt] = state->InAudio.s32audiosamples[in_ch_cnt];
                    state->PreProcess.apl_can_swap = TRUE;
                }
                else if(state->InAudio.apl_period_size  > state->PreProcess.apl_period_size)
                {
                    state->PreProcess.apl_can_swap = FALSE;
                    state->OutAudio.apl_read_pos = 0;

                    if(!(state->InAudio.apl_period_size  % state->PreProcess.apl_period_size))
                    {
                        ETG_TRACE_USR3(("[audproc_alsa_streaming_initialize]: period size alignement"));

                        state->PreProcess.s32audiosamples[in_ch_cnt] = g_malloc0(state->PreProcess.apl_period_size * sizeof(long int));

                        if(state->PreProcess.apl_period_size)
                            state->PreProcess.apl_read_chunk_nb = state->InAudio.apl_period_size / state->PreProcess.apl_period_size;
                        else
                            pr_critical("invalid frame size of PreProcess Buffer\n");
                    }
                    else
                    {
                        ETG_TRACE_USR3(("[audproc_alsa_streaming_initialize]: period size non alignement, in_period-> %d, proc_period -> %d", state->InAudio.apl_period_size , state->PreProcess.apl_period_size ));

                        guint mult = (state->InAudio.apl_period_size  / state->PreProcess.apl_period_size) + 1;


                        state->PreProcess.s32audiosamples[in_ch_cnt] = g_malloc0(mult * (state->PreProcess.apl_period_size * sizeof(long int)));
                        state->PreProcess.apl_read_has_bigger_period = TRUE;
                        state->PreProcess.apl_fifo_size_max = mult * state->PreProcess.apl_period_size;
                        state->PreProcess.apl_write_pos = 0;
                        state->PreProcess.apl_read_pos = 0;

                        ETG_TRACE_USR3(("[audproc_alsa_streaming_initialize]: FIFO mode activated, a fifo was created with size -> %d", state->PreProcess.apl_fifo_size_max ));
                    }
                }
                else if(state->InAudio.apl_period_size  < state->PreProcess.apl_period_size)
                {
                    state->PreProcess.s32audiosamples[in_ch_cnt] = g_malloc0(state->PreProcess.apl_period_size * sizeof(long int));
                    state->PreProcess.apl_can_swap = FALSE;
                    state->PreProcess.apl_write_pos = 0;
                }
            }
            else
            {
                pr_debug("stream buffer already available for dev_%s_ch_%d with frame size(%d), sample rate(%d), format(%d)\n",    data->indev.audproc_dev_name,\
                                                                                                                        in_ch_cnt,\
                                                                                                                        inframe_size,\
                                                                                                                        insr, \
                                                                                                                        data->indev.audproc_format);
            }
        }
        else
        {
            /* do nothing */
        }
    }

    guint outframe_size = data->outdev.audproc_period_frames;
    guint outsr         = data->outdev.audproc_rate;


    /* configure write audio pipeline */
    state->OutAudio.apl_stream_nb_ch = data->outdev.audproc_nb_ch;
    state->OutAudio.apl_sample_rate  = data->outdev.audproc_rate;
    state->OutAudio.apl_period_size  = data->outdev.audproc_period_frames;
    state->OutAudio.apl_audio_format = data->outdev.audproc_format;
    state->OutAudio.apl_audio_dir    = SND_PCM_STREAM_PLAYBACK;
#ifdef D_PREPROCESS_SILENCE_PRIOR_TO_STABILE_RUN
    state->PreSilence.apl_stream_nb_ch = data->outdev.audproc_nb_ch;
    state->PreSilence.apl_sample_rate  = data->outdev.audproc_rate;
    state->PreSilence.apl_period_size  = data->outdev.audproc_period_frames;
    state->PreSilence.apl_audio_format = data->outdev.audproc_format;
    state->PreSilence.apl_audio_dir    = SND_PCM_STREAM_PLAYBACK;
#endif
    /* configure post-process audio pipeline */
    state->PostProcess.apl_stream_nb_ch = data->outdev.audproc_nb_ch;
    state->PostProcess.apl_sample_rate  = data->outdev.audproc_rate;
    state->PostProcess.apl_period_size  = data->proc_frame_size; //state->PreProcess.apl_period_size;
    state->PostProcess.apl_audio_format = data->outdev.audproc_format;
    state->PostProcess.apl_audio_dir    = SND_PCM_STREAM_PLAYBACK;

#ifdef D_SUPPORT_LOOPBACK_DEVICE_ENABLE
    state->lpAudio.apl_stream_nb_ch = data->outdev.audproc_nb_ch;
    state->lpAudio.apl_sample_rate  = data->outdev.audproc_rate;
    state->lpAudio.apl_period_size  = data->outdev.audproc_period_frames ;
    state->lpAudio.apl_audio_format = data->outdev.audproc_format;
    state->lpAudio.apl_audio_dir    = SND_PCM_STREAM_PLAYBACK;
#endif

    /* configures playback stream buffer*/
    for(out_ch_cnt = 0; out_ch_cnt < data->outdev.audproc_nb_ch; out_ch_cnt++)
    {
        if(data->outdev.audproc_format == SND_PCM_FORMAT_S16_LE)
        {
            if(!state->OutAudio.s16audiosamples[out_ch_cnt])
            {
                state->OutAudio.s16audiosamples[out_ch_cnt] = g_malloc0(outframe_size * sizeof(short));
                #ifdef D_PREPROCESS_SILENCE_PRIOR_TO_STABILE_RUN
                state->PreSilence.s16audiosamples[out_ch_cnt] = g_malloc0(outframe_size * sizeof(short));
                #endif
                pr_debug("allocate write buffer for dev_%s_ch_%d with frame size(%d), sample rate(%d), format(%d)\n",   data->outdev.audproc_dev_name,\
                                                                                                                        out_ch_cnt,\
                                                                                                                        outframe_size,\
                                                                                                                        outsr, \

                                                                                                                       data->outdev.audproc_format);
                #ifdef D_SUPPORT_LOOPBACK_DEVICE_ENABLE
                state->lpAudio.s16audiosamples[out_ch_cnt] = g_malloc0(outframe_size * sizeof(short));
                #endif

                /* process and input stream same property, can_swap is active*/
                if(state->PostProcess.apl_period_size  == state->OutAudio.apl_period_size)
                {
                    state->PostProcess.s16audiosamples[out_ch_cnt] = state->OutAudio.s16audiosamples[out_ch_cnt];
                    state->PostProcess.apl_can_swap = TRUE;
                }
                else
                {
                    state->PostProcess.s16audiosamples[out_ch_cnt] = g_malloc0(state->PostProcess.apl_period_size * sizeof(short));
                    state->PostProcess.apl_can_swap = FALSE;
                    state->OutAudio.apl_write_pos = 0;
                    state->OutAudio.apl_is_buf_full = FALSE;
                }
            }
            else
            {
                pr_debug("stream buffer already available for dev_%s_ch_%d with frame size(%d), sample rate(%d), format(%d)\n",   data->outdev.audproc_dev_name,\
                                                                                                                        out_ch_cnt,\
                                                                                                                        outframe_size,\
                                                                                                                        outsr, \
                                                                                                                        data->outdev.audproc_format);
            }

        }
        else if(data->indev.audproc_format == SND_PCM_FORMAT_S32_LE || data->indev.audproc_format == SND_PCM_FORMAT_S24_LE)
        {
            if(!state->OutAudio.s32audiosamples[out_ch_cnt])
            {
                state->OutAudio.s32audiosamples[out_ch_cnt] = g_malloc0(outframe_size * sizeof(long));
                #ifdef D_PREPROCESS_SILENCE_PRIOR_TO_STABILE_RUN
                state->PreSilence.s32audiosamples[out_ch_cnt] = g_malloc0(outframe_size * sizeof(long));
                #endif
                pr_debug("allocate write buffer for dev_%s_ch_%d with frame size(%d), sample rate(%d), format(%d)\n",   data->outdev.audproc_dev_name,\
                                                                                                                        out_ch_cnt,\
                                                                                                                        outframe_size,\
                                                                                                                        outsr, \
                                                                                                                        data->outdev.audproc_format);

                #ifdef D_SUPPORT_LOOPBACK_DEVICE_ENABLE
                state->lpAudio.s32audiosamples[out_ch_cnt] = g_malloc0(outframe_size * sizeof(long));
                #endif

                /* process and input stream same property, can_swap is active*/
                if(state->PostProcess.apl_period_size  == state->OutAudio.apl_period_size)
                {
                    state->PostProcess.s32audiosamples[out_ch_cnt] = state->OutAudio.s32audiosamples[out_ch_cnt];
                    state->PostProcess.apl_can_swap = TRUE;
                }
                else
                {
                    state->PostProcess.s32audiosamples[out_ch_cnt] = g_malloc0(state->PostProcess.apl_period_size * sizeof(long));
                    state->PostProcess.apl_can_swap = FALSE;
                    state->OutAudio.apl_write_pos = 0;
                    state->OutAudio.apl_is_buf_full = FALSE;
                    pr_debug("state->OutAudio.apl_is_buf_full = %d\n",state->OutAudio.apl_is_buf_full )  ;
                }
            }
            else
            {
                pr_debug("stream buffer already available for dev_%s_ch_%d with frame size(%d), sample rate(%d), format(%d)\n",   data->outdev.audproc_dev_name,\
                                                                                                                                  out_ch_cnt,\
                                                                                                                                  outframe_size,\
                                                                                                                                  outsr, \
                                                                                                                                  data->outdev.audproc_format);
            }

        }
    }

    pr_message("EXIT");
    ETG_TRACE_USR3(("[audproc_alsa_streaming_initialize]: EXIT"));

    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_alsa_set_scheduling_priority
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void audproc_alsa_set_scheduling_priority(audproc_alsa_state* state)
{
    gint err;
    struct sched_param sched_param;

    err = sched_getparam(0, &sched_param);
    if (err < 0)
    {
        pr_warning("sched_getparam error %d", err);
        return;
    }

    sched_param.sched_priority = (int)audproc_alsa_get_thread_priority(state);
    err = sched_setscheduler(0, SCHED_FIFO, &sched_param);
    if (!err)
    {
        pr_message("Priority of audproc_alsa_streaming_thread set to SCHED_FIFO %d", sched_param.sched_priority);
    }
    else
    {
        pr_warning("sched_setscheduler error %d when attempting to set priority SCHED_FIFO %d", err, sched_param.sched_priority);
    }
}





/*******************************************************************************
*
* FUNCTION: audproc_alsa_init
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void audproc_alsa_init(void)
{
    pr_debug("Entered");

    g_mutex_init(&audproc_alsa_silence_lock);
    audproc_stream_buffer_init();
}


/*******************************************************************************
*
* FUNCTION: audproc_alsa_init
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void audproc_alsa_finalize(void)
{
    pr_debug("Entered");

    g_mutex_clear(&audproc_alsa_silence_lock);
    audproc_stream_buffer_finalize();

}


/******************************************************
 *  Multi-instance API
 */


/*******************************************************************************
*
* FUNCTION: audproc_alsa_init_instance
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
audproc_alsa_state*  audproc_alsa_init_instance(audproc_alsa_stream_cfg* data)
{
    audproc_alsa_state* state = (audproc_alsa_state*)NULL;
    guint               nb_ch_max = 0;
    guint               in_ch_cnt = 0;
    guint               out_ch_cnt = 0;
    guint               lp_ch_cnt = 0;
    gint                err = AUDPROC_ERR_OK;
    gboolean            streamCfgAvailable = FALSE;


    pr_message("ENTERED");

    if(!data)
    {
        pr_warning("no valid streaming data\n");
        return (audproc_alsa_state*)NULL;
    }

    state = g_malloc0(sizeof(audproc_alsa_state));
    g_mutex_init(&state->lock);
    g_cond_init(&state->update);
    state->thread = NULL;
    state->rts_handle = NULL;

    /* initiate audio thread state und action */
    state->exit = FALSE;
    state->run = FALSE;
    state->initialized = FALSE;
    state->prepared = FALSE;
    state->Req_start_audio = FALSE;
    state->isReadWriteErr = FALSE;
    state->isWriteErr = FALSE;
    state->isReadErr = FALSE;
    state->select_loopback = FALSE;
    state->audproc_can_process = FALSE;
    state->audproc_can_write = FALSE;
    state->audproc_can_read = TRUE;
    state->dosilence    = FALSE;
    state->wrtsilence   = FALSE;

    /* thread local data */
    state->err = AUDPROC_ERR_OK;
    state->last_err = AUDPROC_ERR_OK;
    state->apl_err = AUDPROC_ERR_OK;
    state->rts_err = AUDPROC_ERR_OK;
    state->audio_started = 0;
    state->retry = 0 ;
    state->aplinst = (AplState*)NULL;
    state->aplStr = (gchar*)NULL;
    state->rts_pb_desc = (rts_inst_desc*)NULL;
    state->rts_cap_desc = (rts_inst_desc*)NULL;
    state->rts_lp_desc = (rts_inst_desc*)NULL;


    /* create rts data structure*/
    state->rtsdata = g_malloc0(sizeof(trts_cfg));

    /* here the audio device are configured  */
    state->adevs = g_malloc0( ADEV_NUM_MAX * sizeof(trts_cfgadev_imp));

    state->adevs[ADEV_CAPTURE_IN].pcmname           = g_strdup(data->indev.audproc_dev_name);
    state->adevs[ADEV_CAPTURE_IN].dir               = data->indev.audproc_dir; // SND_PCM_STREAM_CAPTURE
    state->adevs[ADEV_CAPTURE_IN].rate              = data->indev.audproc_rate; // 16000;
    state->adevs[ADEV_CAPTURE_IN].period_frames     = data->indev.audproc_period_frames; //128;  /* 8ms period */
    //state->adevs[ADEV_PLAYBACK_OUT].format        = data->outdev.audproc_format; //SND_PCM_FORMAT_S32_LE;
    if(data->indev.audproc_format!=SND_PCM_FORMAT_UNKNOWN)
        state->adevs[ADEV_CAPTURE_IN].format        = data->indev.audproc_format; //SND_PCM_FORMAT_S32_LE;
    else
    {
        ETG_TRACE_ERR(("[audproc_alsa_init_instance]: replace the invalid configured format ->%d with a defaut value ->%d\n", data->indev.audproc_format, SND_PCM_FORMAT_S32_LE));
        state->adevs[ADEV_CAPTURE_IN].format        = SND_PCM_FORMAT_S32_LE;
    }
    state->adevs[ADEV_CAPTURE_IN].startup_tout      = data->indev.audproc_startup_tout_ms; //100; /* 100 ms timeout*/
    state->adevs[ADEV_CAPTURE_IN].stream_tout       = data->indev.audproc_stream_tout_ms; //100; /* 100 ms timeout*/
    state->cap_prefill_ms                           = data->indev.audproc_prefill_ms;

    state->adevs[ADEV_PLAYBACK_OUT].pcmname         = g_strdup(data->outdev.audproc_dev_name); //"AdevAuditoriumOut");
    state->adevs[ADEV_PLAYBACK_OUT].dir             = data->outdev.audproc_dir; //SND_PCM_STREAM_PLAYBACK;
    state->adevs[ADEV_PLAYBACK_OUT].rate            = data->outdev.audproc_rate;//16000;
    state->adevs[ADEV_PLAYBACK_OUT].period_frames   = data->outdev.audproc_period_frames; //128;  /* 8ms period */
    if(data->outdev.audproc_format!=SND_PCM_FORMAT_UNKNOWN)
        state->adevs[ADEV_PLAYBACK_OUT].format        = data->outdev.audproc_format; //SND_PCM_FORMAT_S32_LE;
    else
    {
        ETG_TRACE_ERR(("[audproc_alsa_init_instance]: replace the invalid configured format ->%d with a defaut value ->%d\n", data->outdev.audproc_format, SND_PCM_FORMAT_S32_LE));
        state->adevs[ADEV_PLAYBACK_OUT].format        = SND_PCM_FORMAT_S32_LE;
    }
//    state->adevs[ADEV_PLAYBACK_OUT].format          = data->outdev.audproc_format; //SND_PCM_FORMAT_S32_LE;
    state->adevs[ADEV_PLAYBACK_OUT].startup_tout    = data->outdev.audproc_startup_tout_ms; //100; /* 100 ms timeout*/
    state->adevs[ADEV_PLAYBACK_OUT].stream_tout     = data->outdev.audproc_stream_tout_ms; //100; /* 100 ms timeout*/
    state->pb_prefill_ms                            = data->outdev.audproc_prefill_ms;

    state->adevs[ADEV_LOOPBACK_OUT].pcmname         = g_strdup(data->lpdev.audproc_dev_name); //"AdevAuditoriumOut");
    state->adevs[ADEV_LOOPBACK_OUT].dir             = data->lpdev.audproc_dir; //SND_PCM_STREAM_PLAYBACK;
    state->adevs[ADEV_LOOPBACK_OUT].rate            = data->lpdev.audproc_rate;//16000;
    state->adevs[ADEV_LOOPBACK_OUT].period_frames   = data->lpdev.audproc_period_frames; //128;  /* 8ms period */
    if(data->lpdev.audproc_format!=SND_PCM_FORMAT_UNKNOWN)
        state->adevs[ADEV_LOOPBACK_OUT].format        = data->lpdev.audproc_format; //SND_PCM_FORMAT_S32_LE;
    else
    {
        ETG_TRACE_ERR(("[audproc_alsa_init_instance]: replace the invalid configured format ->%d with a defaut value ->%d\n", data->lpdev.audproc_format, SND_PCM_FORMAT_S32_LE));
        state->adevs[ADEV_LOOPBACK_OUT].format        = SND_PCM_FORMAT_S32_LE;
    }
    //state->adevs[ADEV_LOOPBACK_OUT].format          = data->lpdev.audproc_format; //SND_PCM_FORMAT_S32_LE;

    state->adevs[ADEV_LOOPBACK_OUT].startup_tout    = data->lpdev.audproc_startup_tout_ms; //100; /* 100 ms timeout*/
    state->adevs[ADEV_LOOPBACK_OUT].stream_tout     = data->lpdev.audproc_stream_tout_ms; //100; /* 100 ms timeout*/
    state->lp_prefill_ms                            = data->lpdev.audproc_prefill_ms;

    state->audproc_in_ch_cnt                        = data->indev.audproc_nb_ch;
    state->audproc_out_ch_cnt                       = data->outdev.audproc_nb_ch;

    /* initialization stream list */
    nb_ch_max =   data->indev.audproc_nb_ch + data->outdev.audproc_nb_ch + data->lpdev.audproc_nb_ch;


    pr_debug("nb of stream channel(%d)\n", nb_ch_max);
    state->streamdefs = g_malloc0( nb_ch_max * sizeof(trts_cfgstream));

    /* configures capture streams*/
    if(data->indev.audproc_nb_ch)
    {
        for(in_ch_cnt = 0; in_ch_cnt < data->indev.audproc_nb_ch; in_ch_cnt++)
        {
            state->streamdefs[in_ch_cnt].adevidx = ADEV_CAPTURE_IN;
            if(data->indev.audproc_nb_ch == 1)
                state->streamdefs[in_ch_cnt].channel = (guint)*data->indev.audproc_str_ch_map; //in_ch_cnt;
            else
                state->streamdefs[in_ch_cnt].channel = data->indev.audproc_str_ch_map[in_ch_cnt];
            pr_debug("set for capture stream[%d], devidx(%d), channel(%d)\n", (in_ch_cnt),\
                                                                               state->streamdefs[in_ch_cnt].adevidx,\
                                                                               state->streamdefs[in_ch_cnt].channel);
        }

        streamCfgAvailable = TRUE;
    }


    /* configures playback streams*/
    if(data->outdev.audproc_nb_ch)
    {
        for(out_ch_cnt = 0; out_ch_cnt < data->outdev.audproc_nb_ch; out_ch_cnt++)
        {
            state->streamdefs[in_ch_cnt + out_ch_cnt].adevidx = ADEV_PLAYBACK_OUT;
            if(data->outdev.audproc_nb_ch == 1)
                state->streamdefs[in_ch_cnt + out_ch_cnt].channel = (guint)*data->outdev.audproc_str_ch_map; //in_ch_cnt;
            else
                state->streamdefs[in_ch_cnt + out_ch_cnt].channel = data->outdev.audproc_str_ch_map[out_ch_cnt];
            //state->streamdefs[in_ch_cnt + out_ch_cnt].channel = out_ch_cnt;

            pr_debug("set for playback stream[%d], devidx(%d), channel(%d)\n", (in_ch_cnt + out_ch_cnt),\
                                                                                state->streamdefs[in_ch_cnt + out_ch_cnt].adevidx,\
                                                                                state->streamdefs[in_ch_cnt + out_ch_cnt].channel);
        }

        streamCfgAvailable = TRUE;
    }

    /* configures loopback streams*/
    if(data->lpdev.audproc_nb_ch)
    {
        for(lp_ch_cnt = 0; lp_ch_cnt < data->lpdev.audproc_nb_ch; lp_ch_cnt++)
        {
            state->streamdefs[in_ch_cnt + out_ch_cnt + lp_ch_cnt].adevidx = ADEV_LOOPBACK_OUT;
            if(data->lpdev.audproc_nb_ch == 1)
                state->streamdefs[in_ch_cnt + out_ch_cnt + lp_ch_cnt].channel = (guint)*data->lpdev.audproc_str_ch_map; //in_ch_cnt;
            else
                state->streamdefs[in_ch_cnt + out_ch_cnt + lp_ch_cnt].channel = data->lpdev.audproc_str_ch_map[lp_ch_cnt];


            pr_debug("set for loopback stream[%d], devidx(%d), channel(%d)\n", (in_ch_cnt + out_ch_cnt + lp_ch_cnt),\
                                                                                state->streamdefs[in_ch_cnt + out_ch_cnt + lp_ch_cnt].adevidx,\
                                                                                state->streamdefs[in_ch_cnt + out_ch_cnt + lp_ch_cnt].channel);
        }

        streamCfgAvailable = TRUE;
    }

    if(!streamCfgAvailable)
        goto error_out;


    /* initiate rts configuration structure */
    state->rtsdata->features = 0;
    state->rtsdata->adevs = (trts_cfgadev*)state->adevs;
    state->rtsdata->num_adevs = ADEV_NUM_MAX;
    state->rtsdata->streams = state->streamdefs;
    state->rtsdata->num_streams = nb_ch_max;//sizeof(state->streamdefs)/sizeof(trts_cfgstream);
    state->rtsdata->prefill_ms = data->prefill_ms;

    pr_debug("set num_stream(%d), num_adevs(%d), prefill_ms(%d)\n", state->rtsdata->num_streams, state->rtsdata->num_adevs, state->rtsdata->prefill_ms);


    /* store audio streaming configuration data container and return this alsa  instance */
    state->audproc_stream_data = data;

    err = audproc_audio_rts_handler_create_new_inst_desc((audproc_alsa_state*)state
                                                          ,data->indev.audproc_dev_name
                                                          ,D_CONV_CAPTURE_DEVICE
                                                          ,data->indev.audproc_prefill_ms
                                                          ,data->audproc_rts_log_cb);
    if(!err)
    {
        if(data->indev.audproc_create_on_start == AUD_PROC_DEVICE_OPEN_AT_STR_START_CLOSE_AT_STR_END)
        {
            (void)audproc_audio_rts_handler_close((audproc_alsa_state*)state, data->indev.audproc_dev_name, D_CONV_CAPTURE_DEVICE);
        }
    }

    if(!err)
    {
        err = audproc_audio_rts_handler_create_new_inst_desc((audproc_alsa_state*)state
                                                             ,data->outdev.audproc_dev_name
                                                             ,D_CONV_PLAYBACK_DEVICE
                                                             ,data->outdev.audproc_prefill_ms
                                                             ,data->audproc_rts_log_cb);
        if(!err)
        {
            if(data->outdev.audproc_create_on_start == AUD_PROC_DEVICE_OPEN_AT_STR_START_CLOSE_AT_STR_END)
            {
                (void)audproc_audio_rts_handler_close((audproc_alsa_state*)state, data->outdev.audproc_dev_name, D_CONV_PLAYBACK_DEVICE);
            }
        }
    }

    #ifdef D_SUPPORT_LOOPBACK_DEVICE_ENABLE
    if(!err)
    {
        err = audproc_audio_rts_handler_create_new_inst_desc((audproc_alsa_state*)state
                                                             ,data->lpdev.audproc_dev_name
                                                             ,D_CONV_LOOPBACK_DEVICE
                                                             ,data->outdev.audproc_prefill_ms
                                                             ,data->audproc_rts_log_cb);
        if(!err)
        {
            if(data->lpdev.audproc_create_on_start == AUD_PROC_DEVICE_OPEN_AT_STR_START_CLOSE_AT_STR_END)
                (void)audproc_audio_rts_handler_close((audproc_alsa_state*)state, data->lpdev.audproc_dev_name, D_CONV_LOOPBACK_DEVICE);
        }
    }
    #endif

    /* setup stream buffer  */
    if(!err)
    {
        err = audproc_alsa_streaming_initialize((audproc_alsa_state*)state);
    }
    else
        goto error_out; /*lint Info 801: prio2: Use of goto is deprecated*/


    /* set default scheduling priority for the ALSA thread */
    audproc_alsa_set_thread_priority(state, (guint)(sched_get_priority_max(SCHED_FIFO) - 1));

    #ifdef D_PREPROCESS_SILENCE_PRIOR_TO_STABILE_RUN
    if(
    #ifndef D_PREPROCESS_SILENCE_ENT_SRC_ONLY
        (data->source_id == AUD_PROC_SRC_VOICE1)
        ||
    #endif
        (data->source_id == AUD_PROC_SRC_ENT1)
        ||(data->source_id == AUD_PROC_SRC_ENT2)
        ||(data->source_id == AUD_PROC_SRC_ENT3)
        ||(data->source_id == AUD_PROC_SRC_ENT4)
        ||(data->source_id == AUD_PROC_SRC_ENT5)
        ||(data->source_id == AUD_PROC_SRC_ENT6)
        ||(data->source_id == AUD_PROC_SRC_ENT7)
        ||(data->source_id == AUD_PROC_SRC_ENT8)
        ||(data->source_id == AUD_PROC_SRC_ENT9)
        ||(data->source_id == AUD_PROC_SRC_ENT10)
        ||(data->source_id == AUD_PROC_SRC_ENT11)
        ||(data->source_id == AUD_PROC_SRC_ENT12)
        ||(data->source_id == AUD_PROC_SRC_ENT13)
        ||(data->source_id == AUD_PROC_SRC_ENT14)
        ||(data->source_id == AUD_PROC_SRC_ENT15)
        ||(data->source_id == AUD_PROC_SRC_ENT16)
        ||(data->source_id == AUD_PROC_SRC_ENT17))
    {
        pr_message("pre_silence process is activated\n");
        state->dosilence    = TRUE;
        state->wrtsilence   = TRUE;
        state->silence_wrt_cnt_after_recover = 0;
        state->recov_success_write_count = 0;
        ETG_TRACE_USR3(("[audproc_alsa_init_instance]: pre_silence process is activated"));
    }
    #else
    state->dosilence    = FALSE;
    state->wrtsilence   = FALSE;
    #endif

    return state;

error_out:
    pr_warning("rts library configuration fails\n");
    (void)audproc_alsa_finalize_intance(state); //, data);
    if(state)
    {
        g_free(state);
        state = NULL;
    }

    return NULL;
}


/*******************************************************************************
*
* FUNCTION: audproc_alsa_finalize_intance
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
gint audproc_alsa_finalize_intance(audproc_alsa_state* state)
{
    int     err = AUDPROC_ERR_OK;
    guint   in_ch_cnt, out_ch_cnt;

    pr_message("ENTERED");

    if(!state)
        return AUDPROC_ERR_NULL_POINTER;


    /* wait a start audio request is terminated */
    while(state->Req_start_audio)
        usleep(1000);


    /* free audio caprure buffers */
    for(in_ch_cnt = 0; in_ch_cnt < state->audproc_in_ch_cnt; in_ch_cnt++)
    {
        if(state->InAudio.s16audiosamples[in_ch_cnt])
        {
            g_free(state->InAudio.s16audiosamples[in_ch_cnt]);
            state->InAudio.s16audiosamples[in_ch_cnt] = NULL;
        }
        if(state->InAudio.s32audiosamples[in_ch_cnt])
        {
            g_free(state->InAudio.s32audiosamples[in_ch_cnt]);
            state->InAudio.s32audiosamples[in_ch_cnt] = NULL;
        }
        if(state->PreProcess.s32audiosamples[in_ch_cnt])
        {
            if(!state->PreProcess.apl_can_swap)
                g_free(state->PreProcess.s32audiosamples[in_ch_cnt]);
            state->PreProcess.s32audiosamples[in_ch_cnt] = NULL;
        }
    }

    /* free audio playback buffers*/
    for(out_ch_cnt = 0; out_ch_cnt < state->audproc_out_ch_cnt; out_ch_cnt++)
    {
        /* stream buffer playback main senk deivce */
        if(state->OutAudio.s16audiosamples[out_ch_cnt])
        {
            g_free(state->OutAudio.s16audiosamples[out_ch_cnt]);
            state->OutAudio.s16audiosamples[out_ch_cnt] = NULL;
        }
        if(state->OutAudio.s32audiosamples[out_ch_cnt])
        {
            g_free(state->OutAudio.s32audiosamples[out_ch_cnt]);
            state->OutAudio.s32audiosamples[out_ch_cnt] = NULL;
        }
        if(state->PreSilence.s16audiosamples[out_ch_cnt])
        {
            g_free(state->PreSilence.s16audiosamples[out_ch_cnt]);
            state->PreSilence.s16audiosamples[out_ch_cnt] = NULL;
        }
        if(state->PreSilence.s32audiosamples[out_ch_cnt])
        {
            g_free(state->PreSilence.s32audiosamples[out_ch_cnt]);
            state->PreSilence.s32audiosamples[out_ch_cnt] = NULL;
        }
        if(state->PostProcess.s32audiosamples[in_ch_cnt])
        {
            if(!state->PostProcess.apl_can_swap)
                g_free(state->PostProcess.s32audiosamples[in_ch_cnt]);
            state->PostProcess.s32audiosamples[in_ch_cnt] = NULL;
        }

#ifdef D_SUPPORT_LOOPBACK_DEVICE_ENABLE
        /* stream buffer loopback senk device */
        if(state->lpAudio.s16audiosamples[out_ch_cnt])
        {
            g_free(state->lpAudio.s16audiosamples[out_ch_cnt]);
            state->lpAudio.s16audiosamples[out_ch_cnt] = NULL;
        }

        if(state->lpAudio.s32audiosamples[out_ch_cnt])
        {
            g_free(state->lpAudio.s32audiosamples[out_ch_cnt]);
            state->lpAudio.s32audiosamples[out_ch_cnt] = NULL;
        }
#endif
    }

    if(state->adevs[ADEV_CAPTURE_IN].pcmname)
    {
        g_free((gpointer)state->adevs[ADEV_CAPTURE_IN].pcmname);  /* lint Warning 605: prio2: Increase in pointer capability (arg. no. 1)*/
        state->adevs[ADEV_CAPTURE_IN].pcmname = NULL;
    }
    if(state->adevs[ADEV_PLAYBACK_OUT].pcmname)
    {
        g_free((gpointer)state->adevs[ADEV_PLAYBACK_OUT].pcmname); /* lint Warning 605: prio2: Increase in pointer capability (arg. no. 1)*/
        state->adevs[ADEV_PLAYBACK_OUT].pcmname = NULL;
    }

    if(state->adevs)
    {
        g_free(state->adevs);
        state->adevs = NULL;
    }

    if(state->streamdefs)
    {
        g_free(state->streamdefs);
        state->streamdefs = NULL;
    }

    if(state->rtsdata)
    {
        g_free(state->rtsdata);
        state->rtsdata = NULL;
    }

    g_mutex_clear(&state->lock);
    g_cond_clear(&state->update);

    pr_message("audio streaming related resources are freed");

    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_alsa_exit_instance
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void    audproc_alsa_exit_instance(audproc_alsa_state* state)
{
    pr_debug("entered\n");


    /* audio thread was terminated */
    if(!audproc_alsa_get_audio_thread_state(state))
        return;

    g_mutex_lock(&state->lock);

    if (state->exit == FALSE)
    {
        pr_message("set exit state to TRUE \n");
        state->exit = TRUE;
        g_cond_broadcast(&state->update);
    }
    else
    {
        pr_warning("thread still terminated");
        audproc_alsa_print_current_audio_state(state);
    }

    /* wait till thread terminated */
    (gpointer) g_thread_join(state->thread);

    g_mutex_unlock(&state->lock);

    return;
}


/*******************************************************************************
*
* FUNCTION: audproc_alsa_start_ent_instance
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
gint    audproc_alsa_start_ent_instance(glong timeout_ms, audproc_alsa_state* state) //
{
    int                         status = AUDPROC_ERR_OK;
    audproc_alsa_stream_cfg*    data = (audproc_alsa_stream_cfg*)NULL;


    pr_message("ENTERED");

    if(!state)
        return AUDPROC_ERR_NULL_POINTER;

    data = state->audproc_stream_data;

    if(!data)
        return AUDPROC_ERR_NULL_POINTER;

    /* start thread */
    status = audproc_alsa_create_ent_streaming_thread(state);

    if(status)
    {
        pr_debug("fails to create streaming thread\n");
        return status;
    }

    /* initiate audio streaaming start, change run state to True
       and send signal to audio streaming thread to trigger streaming
       while release from asleep state (wait)
    */
    g_mutex_lock(&state->lock);
    if (state->run == FALSE)
    {
        pr_debug("set run state to TRUE \n");
        state->run = TRUE;
        state->Req_start_audio = TRUE;
        g_cond_broadcast(&state->update);
    }
    g_mutex_unlock(&state->lock);


    if(!timeout_ms)
    {
        state->Req_start_audio = FALSE;
        pr_message("EXIT");
        return AUDPROC_ERR_OK;
    }


    gint64 tout = (gint64)(timeout_ms) ;
    gint64 end_time = g_get_monotonic_time () + tout * G_TIME_SPAN_MILLISECOND;

    g_mutex_lock(&state->lock);
    /* wait the audio streaming thread has started streaming and
       updated state initialized to true after at least two R/W
       was processed successfully
    */
    while (state->run && !state->initialized)
    {
        if (!g_cond_wait_until(&state->update, &state->lock, end_time))
        {
            // timeout has passed.
            pr_warning("start time_out:%ld_ms has expired", timeout_ms);
            //g_mutex_unlock (&state->lock);
            break;
        }
    }

    /* streaming was started successfully return TRUE */
    if (state->run && state->initialized)
    {
        status = AUDPROC_ERR_OK;

        /* reset count of device restart due to xrun detecttion */
        state->devices_restart_after_xrun_count = 0;
    }

    state->Req_start_audio = FALSE;

    g_mutex_unlock(&state->lock);

    pr_message("EXIT");

    return status;
}

/*******************************************************************************
*
* FUNCTION: audproc_alsa_start_voice_instance
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
gint    audproc_alsa_start_voice_instance(glong timeout_ms, audproc_alsa_state* state) //
{
    int                         status = AUDPROC_ERR_OK;
    audproc_alsa_stream_cfg*    data = (audproc_alsa_stream_cfg*)NULL;


    pr_message("ENTERED\n");

    if(!state)
        return AUDPROC_ERR_NULL_POINTER;

    data = state->audproc_stream_data;

    if(!data)
        return AUDPROC_ERR_NULL_POINTER;

    /* start thread */
    status = audproc_alsa_create_voice_streaming_thread(state);

    if(status)
    {
        pr_debug("fails to create streaming thread\n");
        return status;
    }

    /* initiate audio streaaming start, change run state to True
       and send signal to audio streaming thread to trigger streaming
       while release from asleep state (wait)
    */
    g_mutex_lock(&state->lock);
    if (state->run == FALSE)
    {
        pr_message("set run state to TRUE \n");
        state->run = TRUE;
        state->Req_start_audio = TRUE;
        g_cond_broadcast(&state->update);
    }
    g_mutex_unlock(&state->lock);

    if(!timeout_ms)
    {
        state->Req_start_audio = FALSE;
        pr_message("EXIT");
        return AUDPROC_ERR_OK;
    }

    gint64 tout = (gint64)(timeout_ms) ;
    gint64 end_time = g_get_monotonic_time () + tout * G_TIME_SPAN_MILLISECOND;

    g_mutex_lock(&state->lock);
    /* wait the audio streaming thread has started streaming and
       updated state initialized to true after at least two R/W
       was processed successfully
    */
    while (state->run && !state->initialized)
    {
        if (!g_cond_wait_until (&state->update, &state->lock, end_time))
        {
            // timeout has passed.
            pr_warning("start time_out:%ld_ms has expired", timeout_ms);
            //g_mutex_unlock (&state->lock);
            break;
        }
    }

    /* streaming was started successfully return TRUE */
    if (state->run && state->initialized)
    {
        status = AUDPROC_ERR_OK;

        /* reset count of device restart due to xrun detecttion */
        state->devices_restart_after_xrun_count = 0;
    }

    state->Req_start_audio = FALSE;

    g_mutex_unlock(&state->lock);



    pr_message("EXIT with audio state run(%d), initialized(%d)\n", state->run, state->initialized );

    return status;
}


/*******************************************************************************
*
* FUNCTION: audproc_alsa_start_info_instance
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
gint    audproc_alsa_start_info_instance(glong timeout_ms, audproc_alsa_state* state) //
{
    int                         status = AUDPROC_ERR_OK;
    audproc_alsa_stream_cfg*    data = (audproc_alsa_stream_cfg*)NULL;


    pr_message("ENTERED\n");

    if(!state)
        return AUDPROC_ERR_NULL_POINTER;

    data = state->audproc_stream_data;

    if(!data)
        return AUDPROC_ERR_NULL_POINTER;

    /* start thread */
    status = audproc_alsa_create_info_streaming_thread(state);

    if(status)
    {
        pr_debug("fails to create streaming thread\n");
        return status;
    }

    /* initiate audio streaaming start, change run state to True
       and send signal to audio streaming thread to trigger streaming
       while release from asleep state (wait)
    */
    g_mutex_lock(&state->lock);
    if (state->run == FALSE)
    {
        pr_debug("set run state to TRUE \n");
        state->run = TRUE;
        state->Req_start_audio = TRUE;
        g_cond_broadcast(&state->update);
    }
    g_mutex_unlock(&state->lock);

    if(!timeout_ms)
    {
        state->Req_start_audio = FALSE;
        pr_message("EXIT");
        return AUDPROC_ERR_OK;
    }

    gint64 tout = (gint64)(timeout_ms) ;
    gint64 end_time = g_get_monotonic_time () + tout * G_TIME_SPAN_MILLISECOND;

    g_mutex_lock(&state->lock);
    /* wait the audio streaming thread has started streaming and
       updated state initialized to true after at least two R/W
       was processed successfully
    */
    while (state->run && !state->initialized)
    {
        if (!g_cond_wait_until (&state->update, &state->lock, end_time))
        {
            // timeout has passed.
            pr_warning("start time_out:%ld_ms has expired", timeout_ms);
            break;
        }
    }

    /* streaming was started successfully return TRUE */
    if (state->run && state->initialized)
    {
        status = AUDPROC_ERR_OK;

        /* reset count of device restart due to xrun detecttion */
        state->devices_restart_after_xrun_count = 0;
    }

    state->Req_start_audio = FALSE;

    g_mutex_unlock(&state->lock);



    pr_message("EXIT with audio state run(%d), initialized(%d)\n", state->run, state->initialized );

    return status;
}



/*******************************************************************************
*
* FUNCTION: audproc_alsa_stop_instance
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
gint    audproc_alsa_stop_instance(audproc_alsa_state* state)
{

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_alsa_stop_instance]:ENTERED"));

    if(!state)
    {
        pr_message("EXIT: the alsa module instance was alreday deleted\n");
        return AUDPROC_ERR_NULL_POINTER;
    }

    g_mutex_lock(&state->lock);


    state->run = FALSE;

    // exit audio thread every
    state->exit = TRUE;
    g_cond_broadcast(&state->update);
    g_mutex_unlock(&state->lock);

    /* wait till thread terminated */
    pr_message("wait of the termination of the streaming thread \n");
    if(state->thread)
        ETG_TRACE_USR3(("[audproc_alsa_stop_instance]:wait of the termination of the streaming thread with tid-> %p ", state->thread ));

    (gpointer) g_thread_join(state->thread);

    g_mutex_lock(&state->lock);

    /* thread was unref by thread_join, the thread struct is nulled hier*/
    state->thread = NULL;
    state->exit = FALSE;
    state->initialized = FALSE;
    state->run = FALSE;

    g_mutex_unlock(&state->lock);

    pr_message("EXIT: print current streaming states run -> %d, initialized -> %d, exit -> %d\n", state->run, state->initialized, state->exit);
    ETG_TRACE_USR3(("[audproc_alsa_stop_instance]:EXIT"));

    return AUDPROC_ERR_OK;
}


/*******************************************************************************
*
* FUNCTION: audproc_alsa_print_current_audio_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void audproc_alsa_print_current_audio_state(audproc_alsa_state* state)
{

    int err = AUDPROC_ERR_OK;
    gint     sstate = 0;
    guint32  size = 4;

    AplState* pi = audproc_apl_handler_get_instance(APL_INSTANCE_STR_ARKAMYS_FCTS);
    err = audproc_apl_handler_get_data(pi, aplArkamysSwitchState, 0, &size, (void*)&sstate);
    if(err)
    {
        sstate = apl_state_undefined;
    }

    pr_message("---------- current audio streaming state : top -----\n");
    pr_message("state->run                              : %d\n", state->run);
    pr_message("state->exit                             : %d\n", state->exit);
    pr_message("state->initialized                      : %d\n", state->initialized);
    pr_message("state->athread_active                   : %d\n", state->athread_active);
    pr_message("capture device                          : %s\n", state->adevs[ADEV_CAPTURE_IN].pcmname? state->adevs[ADEV_CAPTURE_IN].pcmname: NULL);
    pr_message("capture playback                        : %s\n", state->adevs[ADEV_PLAYBACK_OUT].pcmname? state->adevs[ADEV_PLAYBACK_OUT].pcmname: NULL);
    pr_message("state->rts_read_cnt                     : %d\n", state->rts_read_cnt);
    pr_message("state->rts_write_cnt                    : %d\n", state->rts_write_cnt);
    pr_message("state->rts_xrun_pb_count                : %d\n", state->rts_xrun_pb_count);
    pr_message("state->rts_xrun_cap_count               : %d\n", state->rts_xrun_cap_count);
    pr_message("state->rts_read_err_occured             : %d\n", state->rts_read_err_occured);
    pr_message("state->rts_write_err_occured            : %d\n", state->rts_read_err_occured);
    pr_message("state->err                              : %d\n", state->err);
    pr_message("state->apl_err                          : %d\n", state->apl_err);
    pr_message("state->isReadWriteErr                   : %d\n", state->isReadWriteErr);
    pr_message("state->isWriteErr                       : %d\n", state->isWriteErr);
    pr_message("state->isReadErr                        : %d\n", state->isReadErr);
    pr_message("state->recov_success_write_count        : %d\n", state->recov_success_write_count);
    pr_message("state->silence_wrt_cnt_after_recover    : %d\n", state->silence_wrt_cnt_after_recover);
    pr_message("state->dosilence                        : %d\n", state->dosilence);
    pr_message("state->wrtsilence                       : %d\n", state->wrtsilence);
    pr_message("state->ArkamysSwitchState               : %d\n", sstate);
    pr_message("---------- current audio streaming state : bottom  --------\n");

    return;
}

#ifdef D_SUPPORT_LOOPBACK_DEVICE_ENABLE
static int audproc_alsa_loopback_process
(
    audproc_alsa_state*             inst,
    const aplAudioSamples_desc*     inAudio,
    aplAudioSamples_desc*           outAudio,
    gboolean                        can_swap
)
{
    unsigned int    i;
    long int*       s32ibuf;
    long int*       s32obuf;
    unsigned int    sample_cnt = 0;
    unsigned int    stream_cnt = 0;


    pr_debug("ENTERED\n");

    if(!inAudio)
        return AUDPROC_ERR_NULL_POINTER;

    if(!outAudio)
        return AUDPROC_ERR_NULL_POINTER ;



    s32ibuf = inAudio->s32audiosamples[0];
    if(!s32ibuf)
    {
        pr_debug("No input data stream buffer available \n");
        return AUDPROC_ERR_NULL_POINTER;
    }

    if(outAudio->apl_period_size <  inAudio->apl_period_size)
    {
        pr_critical("mismatch in size out_stream(%d) and in_stream(%d) \n", outAudio->apl_period_size, inAudio->apl_period_size);
        return AUDPROC_ERR_AUDIO_BUFFER_MISMATCH;
    }


    /* in stream and out stream are aligned */
    if(   (outAudio->apl_stream_nb_ch == inAudio->apl_stream_nb_ch)
        ||(outAudio->apl_stream_nb_ch > inAudio->apl_stream_nb_ch ))
    {
       stream_cnt =  inAudio->apl_stream_nb_ch;
    }
    else if(outAudio->apl_stream_nb_ch < inAudio->apl_stream_nb_ch)
    {
       stream_cnt =  outAudio->apl_stream_nb_ch;
       pr_warning(" mismatch nb channel out_stream(%d) and in_stream(%d)\n",outAudio->apl_stream_nb_ch, inAudio->apl_stream_nb_ch );
    }

    for( i = 0; i < stream_cnt ; i++)
    {
        if(   (outAudio->apl_period_size ==  inAudio->apl_period_size)
            ||(outAudio->apl_period_size >  inAudio->apl_period_size))
        {
            unsigned int ich = i;
            unsigned int och = i;


            sample_cnt = inAudio->apl_period_size;

            if(can_swap)
            {
                pr_debug ("swap stream buffer[%d]\n", ich);

                s32ibuf = inAudio->s32audiosamples[ich];
                outAudio->s32audiosamples[och] = s32ibuf;
            }
            else
            {
                pr_debug("copy %d s32 sample from in_stream_s32_buffer[%d] to out_stream_s32_buffer[%d]\n", sample_cnt, ich, och);

                s32ibuf = inAudio->s32audiosamples[ich];
                s32obuf = outAudio->s32audiosamples[och];
                memcpy((void*)s32obuf, (void*)s32ibuf, sample_cnt * sizeof(long int));
            }
        }
        else
        {
            pr_critical(" mismatch between nb of in_stream(%d) and out_strem(%d) samples \n", inAudio->apl_period_size, outAudio->apl_period_size);
        }
    }

    /* now write data in eth loop device */
    (void)audproc_audio_rts_handler_Loop_write(inst, (void**)outAudio->s32audiosamples);

    return AUDPROC_ERR_OK;
}
#endif

#ifdef D_MONITOR_SILENCE_INJECTION
static int audproc_monitor_silence_injection
(
    const aplAudioSamples_desc*     inAudio
)
{
    //unsigned int                    i, j;
    long int*                       s32ibuf;
    unsigned int                    sample_cnt = 0;
    //unsigned int                    stream_cnt = 0;
    const gchar*                    str = (gchar*)NULL;
    guint8                          src = 0;
    struct audproc_instance_object* act_obj = (struct audproc_instance_object*)NULL;
    int                             err = AUDPROC_ERR_OK;
    gboolean                        silence_detected = FALSE;


    pr_debug("ENTERED\n");

    if(!inAudio)
        return AUDPROC_ERR_NULL_POINTER;

    s32ibuf = inAudio->s32audiosamples[0];
    if(!s32ibuf)
    {
        pr_debug("No input data stream buffer available \n");
        return AUDPROC_ERR_NULL_POINTER;
    }

    //stream_cnt =  inAudio->apl_stream_nb_ch;
    sample_cnt = inAudio->apl_period_size;

    /* check only the channel 0*/
    s32ibuf = inAudio->s32audiosamples[0];

    if(s32ibuf[0] == 0)
    {
        if(s32ibuf[(unsigned int)(sample_cnt - 1)] == 0)
        {
            if(s32ibuf[(unsigned int)((sample_cnt - 1) / 2)] == 0)
            {
                /* silence frame is high probable
                 * perform action here
                 * send flush the arkamys library
                */

                silence_detected = TRUE;
                silence_detect_release_cnt = 0;

                if((++silence_detect_cnt > D_SILENCE_DETECT_THRES) && !audproc_silence_detect)
                {
                    act_obj = audproc_service_get_active_type_src(AUD_PROC_SRC_TYPE_ENT);
                    if(act_obj)
                    {
                        err = audproc_configuration_look_up_src_table(act_obj->audproc_this_src, &src, TRUE);
                        if(!err)
                        {
                            str = audproc_configuration_get_inst_name(src);
                            if(str)
                            {
                                err = audproc_apl_handler_command(str, aplCmdFlush);
                                if(!err)
                                {
                                    pr_message("silence frame was detected, send memory flush coammnd to the arkamys library \n");
                                }
                                else
                                {
                                    pr_warning("fails with error -> %d", err);
                                }
                            }
                        }
                    }
                    audproc_silence_detect = TRUE;
                }
            }
        }
    }

    if(!silence_detected && audproc_silence_detect)
    {
        if(++silence_detect_release_cnt > D_SILENCE_DETECT_RELEASE_THRES)
        {
            pr_message("release silence detecttion\n");
            audproc_silence_detect = FALSE;
            silence_detect_cnt = 0;
            silence_detect_release_cnt = 0;
        }
    }

    return AUDPROC_ERR_OK;
}
#endif

/*******************************************************************************
*
* FUNCTION: audproc_alsa_read_all
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint audproc_alsa_read_all
(
    audproc_alsa_state*     state,
    audproc_source_type     src_type
)
{

    if(state->InAudio.apl_audio_format == SND_PCM_FORMAT_S16_LE)
        state->err_r_s = audproc_audio_rts_handler_read(state, (void**)state->InAudio.s16audiosamples, src_type);
    else if(state->InAudio.apl_audio_format == SND_PCM_FORMAT_S32_LE || state->InAudio.apl_audio_format == SND_PCM_FORMAT_S24_LE)
        state->err_r_s = audproc_audio_rts_handler_read(state, (void**)state->InAudio.s32audiosamples, src_type);
    else
    {
        /* do nothing */
    }

    if(state->err_r_s)
    {
        pr_debug("rts read returned with error:%d \n", state->err_r_s);
    }
    else
    {

#ifdef D_SUPPORT_LOOPBACK_DEVICE_ENABLE
        /* loopback */
        if(state->select_loopback)
        {
            (void)audproc_alsa_loopback_process(state,
                        &state->InAudio,
                        &state->lpAudio,
                        FALSE);
        }
#endif

#ifdef D_MONITOR_SILENCE_INJECTION
        if(AUD_PROC_SRC_TYPE_ENT == src_type)
            state->err_r_s = audproc_monitor_silence_injection(&state->InAudio);
#endif
    }
    return state->err_r_s;
}


/*******************************************************************************
*
* FUNCTION: audproc_alsa_write_all
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint audproc_alsa_write_all
(
    audproc_alsa_state*     state,
    audproc_source_type     src_type
)
{

    state->err_w_s = AUDPROC_ERR_OK;

    if(state->wrtsilence)
    {
        if(state->PreSilence.apl_audio_format == SND_PCM_FORMAT_S16_LE)
            state->err_w_s = audproc_audio_rts_handler_write(state, (void**)state->PreSilence.s16audiosamples, src_type, TRUE);
        else if(state->PreSilence.apl_audio_format == SND_PCM_FORMAT_S32_LE)
            state->err_w_s = audproc_audio_rts_handler_write(state, (void**)state->PreSilence.s32audiosamples, src_type, TRUE);
        else
        {
            /* do nothing */
            pr_warning("state->PreSilence.apl_audio_format = %d", state->PreSilence.apl_audio_format);
        }
        return state->err_w_s;
    }

    else if(    state->OutAudio.apl_is_buf_full
        || (src_type == AUD_PROC_SRC_TYPE_VOICE)
        || (src_type == AUD_PROC_SRC_TYPE_INFO))
    {
        if(state->OutAudio.apl_audio_format == SND_PCM_FORMAT_S16_LE)
            state->err_w_s = audproc_audio_rts_handler_write(state, (void**)state->OutAudio.s16audiosamples, src_type, FALSE);
        else if(state->InAudio.apl_audio_format == SND_PCM_FORMAT_S32_LE || state->InAudio.apl_audio_format == SND_PCM_FORMAT_S24_LE)
            state->err_w_s = audproc_audio_rts_handler_write(state, (void**)state->OutAudio.s32audiosamples, src_type, FALSE);
        else
        {
            /* do nothing */
            pr_warning("state->OutAudio.apl_audio_format = %d", state->OutAudio.apl_audio_format);
        }

        state->OutAudio.apl_is_buf_full = FALSE;
        state->OutAudio.apl_write_pos   = 0;

        return state->err_w_s;
    }
    else
    {
        /* do nothing */
    }



    if(state->PostProcess.apl_period_size  > state->OutAudio.apl_period_size)
    {
        /* todo: calculate modulo, if remainder not NULL appropriate handling should be takien care */
        if(state->OutAudio.apl_period_size)
            state->nb_wrt_chunk_s =   state->PostProcess.apl_period_size / state->OutAudio.apl_period_size;
        else
            state->nb_wrt_chunk_s = 1;
    }
    else
        state->nb_wrt_chunk_s = 1;


    for( state->i_s = 0; state->i_s < state->nb_wrt_chunk_s; state->i_s++)
    {
        pr_debug("write %d_chunk with %d_samples \n", (state->i_s + 1), state->OutAudio.apl_period_size );

        if(state->OutAudio.apl_audio_format == SND_PCM_FORMAT_S16_LE)
        {
            if(!state->PostProcess.apl_can_swap)
            {
                for( state->j_s = 0; state->j_s < state->OutAudio.apl_stream_nb_ch ; state->j_s++)
                {
                    state->s16ibuf_s = state->PostProcess.s16audiosamples[state->j_s];
                    state->s16ibuf_s += (state->i_s * state->OutAudio.apl_period_size);
                    state->s16obuf_s = state->OutAudio.s16audiosamples[state->j_s]; /*clean*/
                    memcpy((void*)state->s16obuf_s, (void*)state->s16ibuf_s, state->OutAudio.apl_period_size * sizeof(short int));
                }
            }

            state->err_w_s = audproc_audio_rts_handler_write(state, (void**)state->OutAudio.s16audiosamples, src_type, FALSE);
        }
        else if(state->InAudio.apl_audio_format == SND_PCM_FORMAT_S32_LE || state->InAudio.apl_audio_format == SND_PCM_FORMAT_S24_LE)
        {
            if(!state->PostProcess.apl_can_swap)
            {
                for( state->j_s = 0; state->j_s < state->OutAudio.apl_stream_nb_ch ; state->j_s++)
                {
                    state->s32ibuf_s = state->PostProcess.s32audiosamples[state->j_s];
                    state->s32ibuf_s += (state->i_s * state->OutAudio.apl_period_size);
                    state->s32obuf_s = state->OutAudio.s32audiosamples[state->j_s]; /*clean*/
                    memcpy((void*)state->s32obuf_s, (void*)state->s32ibuf_s, state->OutAudio.apl_period_size * sizeof(long int));
                }
            }

            state->err_w_s = audproc_audio_rts_handler_write(state, (void**)state->OutAudio.s32audiosamples, src_type, FALSE);
        }
        else
        {
            /* do nothing */
            pr_warning("state->OutAudio.apl_audio_format = %d", state->OutAudio.apl_audio_format);
        }
    }

    if(state->err_w_s)
        pr_message("rts write returned with error:%d \n",state->err_w_s);

    return state->err_w_s;
}


/*******************************************************************************
*
* FUNCTION: audproc_alsa_process_prepare
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gboolean audproc_alsa_process_prepare(audproc_alsa_state* state)
{

    pr_debug("ENTER\n");
    ETG_TRACE_USR4(("[audproc_alsa_process_prepare]:ENTERED"));

    if(state->PreProcess.apl_can_swap)
        return TRUE;

#ifdef D_ONLY_FOR_DEBUG
    pr_message("process dtreaming prepare\n");
    ETG_TRACE_USR3(("[audproc_alsa_process_prepare]:process dtreaming prepare"));
#endif
    if(state->PreProcess.apl_read_has_bigger_period)
    {
        ETG_TRACE_USR4(("[audproc_alsa_process_prepare]:process FIFO prepare with current sample count -> %d", state->PreProcess.apl_fifo_current_sample_count));

        (void)audproc_stream_buffer_add_new_frame(&state->PreProcess, &state->InAudio);
        state->avail_pp_s = TRUE;
    }

    else if(state->PreProcess.apl_read_chunk_nb)
    {
        pr_debug("read %d_samples from the InAudio buffers at position :%d \n", state->PreProcess.apl_period_size,  state->PreProcess.apl_read_pos );

        if(state->InAudio.apl_audio_format == SND_PCM_FORMAT_S16_LE)
        {
            for( state->j_s = 0; state->j_s < state->PreProcess.apl_stream_nb_ch ; state->j_s++)
            {
                state->s16ibuf_s = state->InAudio.s16audiosamples[state->j_s];
                state->s16ibuf_s += state->PreProcess.apl_read_pos; //(i * state->PreProcess.apl_period_size);
                state->s16obuf_s = state->PreProcess.s16audiosamples[state->j_s]; /*clean*/
                memcpy((void*)state->s16obuf_s, (void*)state->s16ibuf_s, state->PreProcess.apl_period_size * sizeof(short int));
            }
        }
        else if(state->InAudio.apl_audio_format == SND_PCM_FORMAT_S32_LE || state->InAudio.apl_audio_format == SND_PCM_FORMAT_S24_LE)
        {
            for( state->j_s = 0; state->j_s < state->PreProcess.apl_stream_nb_ch ; state->j_s++)
            {
                state->s32ibuf_s = state->InAudio.s32audiosamples[state->j_s];
                state->s32ibuf_s += state->PreProcess.apl_read_pos; //(i * state->PreProcess.apl_period_size);
                state->s32obuf_s = state->PreProcess.s32audiosamples[state->j_s]; /*clean*/
                //memcpy((void*)state->s32ibuf_s, (void*)state->s32ibuf_s, state->PreProcess.apl_period_size * sizeof(long int));
                /* Used memmove instead memcpy to resolve coverity issue */
                memmove((void*)state->s32obuf_s, (void*)state->s32ibuf_s, state->PreProcess.apl_period_size * sizeof(long int));
            }
        }
        else
        {
            /* do nothing */
            pr_warning("invalide state->PreProcess.apl_audio_format = %d", state->OutAudio.apl_audio_format);
        }


        state->PreProcess.apl_read_pos += state->PreProcess.apl_period_size;
        state->PreProcess.apl_read_chunk_nb--;

        pr_debug("remaining chunks: %d\n",  state->PreProcess.apl_read_chunk_nb);

        if(!state->PreProcess.apl_read_chunk_nb)
        {
            state->PreProcess.apl_read_pos = 0;
            state->audproc_can_read = TRUE;

            if(state->InAudio.apl_period_size  > state->PreProcess.apl_period_size)
            {
                if(state->PreProcess.apl_period_size)
                    state->PreProcess.apl_read_chunk_nb = state->InAudio.apl_period_size / state->PreProcess.apl_period_size;
                else
                    pr_critical("invalid frame size of PreProcess Buffer\n");
            }
            else
                pr_warning("here awaits frame of the PreProcess Buffer higher as the on of InAudio \n");
        }
        else
            state->audproc_can_read = FALSE;

        state->avail_pp_s = TRUE;
    }//if(state->PreProcess.apl_read_chunk_nb)
    else if(state->InAudio.apl_period_size  < state->PreProcess.apl_period_size)
    {
        if(state->InAudio.apl_audio_format == SND_PCM_FORMAT_S16_LE)
        {
            //state->avail_pp_s = TRUE;

            for( state->j_s = 0; state->j_s < state->InAudio.apl_stream_nb_ch ; state->j_s++)
            {
                state->s16ibuf_s = state->InAudio.s16audiosamples[state->j_s];
                state->s16obuf_s = state->PreProcess.s16audiosamples[state->j_s]; /*clean*/
                state->s16obuf_s += state->PreProcess.apl_write_pos;

                if((state->PreProcess.apl_write_pos + state->InAudio.apl_period_size) > state->PreProcess.apl_period_size)
                {
                    pr_critical("size of the output buffer will exceed, the input buffer should be an integer divider of the output buffer\n");
                    return FALSE;
                    //state->avail_pp_s = FALSE;
                    //break;
                }
                else
                    memcpy((void*)state->s16obuf_s, (void*)state->s16ibuf_s, state->InAudio.apl_period_size * sizeof(short int));
            }


            //if(state->avail_pp_s)
            //{
                state->PreProcess.apl_write_pos += state->InAudio.apl_period_size;
                if(state->PreProcess.apl_write_pos >= state->PreProcess.apl_period_size )
                {
                    state->PreProcess.apl_write_pos = 0;
                    state->avail_pp_s = TRUE;
                }
                else
                {
                    pr_debug("read next chunk\n");
                    state->avail_pp_s = FALSE;
                }
            //}
        }
        else if(state->InAudio.apl_audio_format == SND_PCM_FORMAT_S32_LE || state->InAudio.apl_audio_format == SND_PCM_FORMAT_S24_LE)
        {
            state->avail_pp_s = TRUE;

            for( state->j_s = 0; state->j_s < state->InAudio.apl_stream_nb_ch ; state->j_s++)
            {
                state->s32ibuf_s = state->InAudio.s32audiosamples[state->j_s];
                state->s32obuf_s = state->PreProcess.s32audiosamples[state->j_s]; /*clean*/
                state->s32obuf_s += state->PreProcess.apl_write_pos;

                if((state->PreProcess.apl_write_pos + state->InAudio.apl_period_size) > state->PreProcess.apl_period_size)
                {
                    pr_critical("size of the output buffer will exceed, the input buffer should be an integer divider of the output buffer\n");
                    return FALSE;
                    //state->avail_pp_s = FALSE;
                    //break;
                }
                else
                {
                    pr_debug("copy %d frames in channel_%d, current write buffer position %d, buffer size is %d \n",  state->InAudio.apl_period_size, state->j_s, state->PreProcess.apl_write_pos, state->PreProcess.apl_period_size);
//                    memcpy((void*)state->s32ibuf_s, (void*)state->s32ibuf_s, state->InAudio.apl_period_size * sizeof(long int));
                    /* Used memmove instead memcpy to resolve coverity issue */
                    memmove((void*)state->s32obuf_s, (void*)state->s32ibuf_s, state->InAudio.apl_period_size * sizeof(long int));

                }
            }

            //if(state->avail_pp_s)
            //{
                state->PreProcess.apl_write_pos += state->InAudio.apl_period_size;
                if(state->PreProcess.apl_write_pos >= state->PreProcess.apl_period_size )
                {
                    state->PreProcess.apl_write_pos = 0;
                    state->avail_pp_s = TRUE;
                    pr_debug(" allow to process\n");
                }
                else
                {
                    pr_debug("read next chunk\n");
                    state->avail_pp_s = FALSE;
                }
            //}
        }
    }
#ifdef D_ONLY_FOR_DEBUG
    pr_message("exit\n");
    ETG_TRACE_USR3(("[audproc_alsa_process_prepare]:exit"));
#endif
    ETG_TRACE_USR4(("[audproc_alsa_process_prepare]:EXIT"));
    return state->avail_pp_s;
}

/*******************************************************************************
*
* FUNCTION: audproc_alsa_write_prepare
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gboolean audproc_alsa_write_prepare(audproc_alsa_state* state)
{

    pr_debug("ENTER\n");

    if(     state->PostProcess.apl_can_swap
        || (state->PostProcess.apl_period_size  > state->OutAudio.apl_period_size))
        return TRUE;

#ifdef D_ONLY_FOR_DEBUG
    pr_message("process write prepare\n");
    ETG_TRACE_USR3(("[audproc_alsa_write_prepare]:process write prepare"));
#endif


    if(state->PostProcess.apl_period_size  < state->OutAudio.apl_period_size)
    {
        if(state->OutAudio.apl_audio_format == SND_PCM_FORMAT_S16_LE)
        {
            if(!state->OutAudio.apl_is_buf_full)
            {
                state->avail_wp_s = TRUE;
                for( state->j_s = 0; state->j_s < state->OutAudio.apl_stream_nb_ch ; state->j_s++)
                {
                    state->s16ibuf_s = state->PostProcess.s16audiosamples[state->j_s];
                    state->s16obuf_s = state->OutAudio.s16audiosamples[state->j_s]; /*clean*/
                    state->s16obuf_s += state->OutAudio.apl_write_pos;

                    if((state->OutAudio.apl_write_pos + state->PostProcess.apl_period_size) > state->OutAudio.apl_period_size)
                    {
                        pr_critical("size of the output buffer will exceed, the input buffer should be an integer divider of the output buffer\n");
                        return FALSE;
                        //state->avail_wp_s = FALSE;
                        //break;
                    }
                    else
                        memcpy((void*)state->s16obuf_s, (void*)state->s16ibuf_s, state->PostProcess.apl_period_size * sizeof(short int));
                }

                //if(state->avail_wp_s)
                //{
                    state->OutAudio.apl_write_pos += state->PostProcess.apl_period_size;
                    if(state->OutAudio.apl_write_pos >= state->OutAudio.apl_period_size )
                    {
                        state->OutAudio.apl_is_buf_full = TRUE;
                        state->avail_wp_s = TRUE;
                    }
                    else
                    {
                        pr_debug("read next chunk\n");
                        return FALSE;
                        //state->avail_wp_s = FALSE;
                    }
                //}
            }
        }
        else if(state->InAudio.apl_audio_format == SND_PCM_FORMAT_S32_LE || state->InAudio.apl_audio_format == SND_PCM_FORMAT_S24_LE)
        {
            if(!state->OutAudio.apl_is_buf_full)
            {
                for( state->j_s = 0; state->j_s < state->OutAudio.apl_stream_nb_ch ; state->j_s++)
                {
                    state->s32ibuf_s = state->PostProcess.s32audiosamples[state->j_s];
                    state->s32obuf_s = state->OutAudio.s32audiosamples[state->j_s]; /*clean*/
                    state->s32obuf_s += state->OutAudio.apl_write_pos;

                    if((state->OutAudio.apl_write_pos + state->PostProcess.apl_period_size) > state->OutAudio.apl_period_size)
                    {
                        pr_critical("size of the output buffer will exceed, the input buffer should be an integer divider of the output buffer\n");
                        return FALSE;
                        //state->avail_wp_s = FALSE;
                        //break;
                    }
                    else
                    {
                        pr_debug("copy %d frames in channel_%d, current write buffer position %d, buffer size is %d \n",  state->PostProcess.apl_period_size, state->j_s, state->OutAudio.apl_write_pos, state->OutAudio.apl_period_size);
                        memcpy((void*)state->s32obuf_s, (void*)state->s32ibuf_s, state->PostProcess.apl_period_size * sizeof(long int));
                    }
                }

                //if(state->avail_wp_s)
                //{
                    state->OutAudio.apl_write_pos += state->PostProcess.apl_period_size;
                    if(state->OutAudio.apl_write_pos >= state->OutAudio.apl_period_size )
                    {
                        state->OutAudio.apl_is_buf_full = TRUE;
                        state->avail_wp_s = TRUE;
                        pr_debug(" allow rts write, state->OutAudio.apl_is_buf_full = %d\n", state->OutAudio.apl_is_buf_full);
                    }
                    else
                    {
                        pr_debug("read next chunk\n");
                        return FALSE;
                        //state->avail_wp_s = FALSE;
                    }
                //}
            }
        }
    }


#ifdef D_ONLY_FOR_DEBUG
    pr_message("exit\n");
    ETG_TRACE_USR3(("[audproc_alsa_process_prepare]: exit"));
#endif
    return state->avail_wp_s;
}



#ifdef D_WRITE_SILENCE_DURING_PAUSE_TRACK_CHANGE

static GThread*     audproc_alsa_silence_thread_hdl;
static gboolean     audproc_alsa_silence_proc_active = FALSE;

static gboolean audproc_alsa_silence_proc_act_is(audproc_alsa_state*  state)
{
    gboolean status = FALSE;

    if(!state)
        return status;

    g_mutex_lock(&audproc_alsa_silence_lock);
    status = audproc_alsa_silence_proc_active;
    ETG_TRACE_USR4(("[audproc_alsa_silence_proc_act_is]: silence %s",  status? "ACTIVE" : "STOPPED"));
    g_mutex_unlock(&audproc_alsa_silence_lock);
    return status;
}



static gpointer audproc_alsa_write_silence_thread(gpointer data)
{
    gint                            err;
    struct sched_param              sched_param;
    audproc_alsa_state*             state = (audproc_alsa_state*)NULL;
    struct audproc_instance_object* obj = (struct audproc_instance_object*)NULL;

    ETG_TRACE_USR3(("[audproc_alsa_write_silence_thread]: ENTERED"));
    pr_message("ENTERED\n");

    if(!data)
    {
        pr_warning("invalide data pointer\n");
        return NULL;
    }

    state = (audproc_alsa_state*)data;

    /*set RT prio */
    err = sched_getparam(0, &sched_param);
    if (err < 0)
    {
        pr_warning("sched_getparam error %d", err);
        return NULL;
    }

    sched_param.sched_priority = sched_get_priority_max(SCHED_FIFO) - 1;

    err = sched_setscheduler(0, SCHED_FIFO, &sched_param);
    if (err)
        pr_warning("sched_setscheduler error %d when attempting to set priority SCHED_FIFO %d", err, sched_param.sched_priority);

    state->wrtsilence = TRUE;
    int wr_count = 0, wr_err_count = 0;

    while(audproc_alsa_silence_proc_active)
    {
        /*start write silence to the sink */
        if(wr_count == 0)
        {
            pr_message("silence process entered\n");
            ETG_TRACE_USR3(("[audproc_alsa_write_silence_thread]: silence process entered"));
        }

        obj = audproc_service_get_active_type_src(AUD_PROC_SRC_TYPE_ENT);
        if(obj)
        {
            if(obj->audproc_audio_instance)
            {
                err = audproc_alsa_write_all(obj->audproc_audio_instance, AUD_PROC_SRC_TYPE_ENT);
                if(err)
                {
                    wr_err_count++;

                    if(err == -22)
                    {
                        err = audproc_audio_rts_handler_recreate(obj->audproc_audio_instance, FALSE, D_CONV_PLAYBACK_DEVICE);
                    }
                    else
                        err = audproc_audio_rts_handler_recover(obj->audproc_audio_instance, D_CONV_PLAYBACK_DEVICE);
                }
                wr_count++;
            }
        }

        if((wr_count % 4) == 0)
        {
            pr_message("current silence write count -> %d \n", wr_count);
            ETG_TRACE_USR4(("[audproc_alsa_write_silence_thread]: current silence write count -> %d", wr_count));
        }
    }

    ETG_TRACE_USR3(("[audproc_alsa_write_silence_thread]: terminate after %d -> silence frame and %d -> write error", wr_count, wr_err_count));
    pr_message("terminate after %d -> silence frame and %d -> write error \n", wr_count, wr_err_count);
    //To resolve Gen4 Prio1 Compiler Warnings
    (void)data;
    return NULL;
}

void audproc_alsa_process_silence_start(gpointer data)
{
    GError *error;

    ETG_TRACE_USR3(("[audproc_alsa_process_silence_start]: ENTERED"));
    pr_message("ENTERED\n");

    audproc_alsa_silence_thread_hdl = g_thread_try_new ("silence_thread", audproc_alsa_write_silence_thread, data, &error);
    if(!audproc_alsa_silence_thread_hdl)
    {
        ETG_TRACE_ERR(("[audproc_alsa_process_silence_start]: Fails to create audproc_alsa_write_silence_thread "));
    }
    else
    {

        g_mutex_lock(&audproc_alsa_silence_lock);
        audproc_alsa_silence_proc_active = TRUE;
        g_mutex_unlock(&audproc_alsa_silence_lock);

        ETG_TRACE_USR3(("[audproc_alsa_process_silence_start]: silence process started"));
    }

    return;
}

void audproc_alsa_process_silence_terminated(void)
{

    ETG_TRACE_USR3(("[audproc_alsa_process_silence_terminated]: ENTERED"));
    pr_message("ENTERED\n");

    if(!audproc_alsa_silence_thread_hdl)
    {
        pr_message("EXIT: the silence thread was already terminated\n");
        ETG_TRACE_USR3(("[audproc_alsa_process_silence_terminated]: EXIT: the silence thread was already terminated"));
        return;
    }

    g_mutex_lock(&audproc_alsa_silence_lock);
    audproc_alsa_silence_proc_active = FALSE;
    g_mutex_unlock(&audproc_alsa_silence_lock);

    pr_message("wait termination of the write silence thread\n");
    ETG_TRACE_USR3(("[audproc_alsa_process_silence_terminated]: wait termination of the write silence thread"));

    g_thread_join(audproc_alsa_silence_thread_hdl);
    audproc_alsa_silence_thread_hdl = (GThread*)NULL;

    ETG_TRACE_USR3(("[audproc_alsa_process_silence_terminated]: EXIT"));
    pr_message("EXIT\n");
}

#endif





/*******************************************************************************
*
* FUNCTION: audproc_alsa_ent_streaming_thread
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gpointer audproc_alsa_ent_streaming_thread(gpointer data)
{

    audproc_alsa_state*             state = (audproc_alsa_state*)NULL;


    pr_message("ENTERED");

    /* if data non Null, than multi-instance interface is used */
    if(data)
    {
        state = (audproc_alsa_state*)data;
        state->aplStr = state->audproc_stream_data->audcfg.audproc_apl_inst_name;

        /* retrieve apl instance name */
        state->aplinst = audproc_apl_handler_get_instance(state->aplStr);

        if(!state->aplinst)
        {
            pr_critical("no APL instance available, processing cannot be continued\n");
            return NULL;
        }
    }
    else
    {
        pr_critical("data pointer is NULL?\n");
        return NULL;
    }

    audproc_alsa_set_scheduling_priority(state); //--> this the problem root cause

    pr_debug("instance state exit(%d), run(%d), initialized(%d), prepared(%d)\n", state->exit,
                                                                                  state->run,
                                                                                  state->initialized,
                                                                                  state->prepared);

    while (!state->exit)
    {
        pr_message("entry top audio streaming loop");

        g_mutex_lock(&state->lock);
        if(state->err)
        {
            state->run = FALSE;
            (void)audproc_service_change_audio_state(AUD_PROC_SRC_TYPE_ENT, AUDPROC_AS_STREAM_STOP, 0);
        }
        else
            (void)audproc_service_change_audio_state(AUD_PROC_SRC_TYPE_ENT, AUDPROC_AS_STREAM_SILENCE, 0);

        state->initialized      = FALSE;
        state->prepared         = TRUE;
        state->isReadWriteErr   = FALSE;

        /* alsa stop was terminated,
           the main thread which runs the audproc_alsa_stop function is woken up
        */
        g_cond_broadcast(&state->update);

        /* initialize thread for next run*/
        state->err = 0;
        state->apl_err = 0;
        state->rts_err = 0;
        state->retry = 0;

        /* wait release from audproc_alsa_start() function */
        while (!state->run && !state->exit)
        {
            pr_message("wait of the streaming start\n");
            g_cond_wait(&state->update, &state->lock);
        }

        g_mutex_unlock(&state->lock);

        /* audio thread terminated with call audproc_alsa_exit() ? */
        if (state->exit)
            continue;


        (void)audproc_service_change_audio_state(AUD_PROC_SRC_TYPE_ENT, AUDPROC_AS_STREAM_INIT, 0);

        //pr_debug("audio streaming prepared\n");
        state->audio_started = 0;

        /* run audio streaming */
        while (!state->err && state->run && !state->exit)
        {
            /* read audio frames from input stream devices*/
            if(state->audproc_can_read && !audproc_stream_buffer_is_frame_pending_avail(&state->PreProcess))//apl_fifo_has_frame_avail)
            {
                state->err = audproc_alsa_read_all(state, AUD_PROC_SRC_TYPE_ENT);
                if (state->err)
                {
                    state->isReadErr = TRUE;
                    goto rts_err; /*lint Info 801: prio2: Use of goto is deprecated*/
                }

            }

            if(!audproc_stream_buffer_is_frame_pending_avail(&state->PreProcess))//.apl_fifo_has_frame_avail)
                state->audproc_can_process = audproc_alsa_process_prepare(state);
            else
                state->audproc_can_process = TRUE;

            if (!state->audproc_can_process)
            {
                pr_message("audproc_alsa_process_prepare is pending\n");
                ETG_TRACE_USR3(("[audproc_alsa_ent_streaming_thread]:audproc_alsa_process_prepare is pending"));
                continue;
            }

            /* process audio frames */
            state->err = state->apl_err = audproc_apl_handler_process_audio_ent(state->aplinst, &state->PreProcess, &state->PostProcess);

            if (state->err)
            {
                pr_message("audproc_apl_handler_process_audio_ent has failed\n");
                ETG_TRACE_USR3(("[audproc_alsa_ent_streaming_thread]:audproc_alsa_process_prepare has failed"));
                continue;
            }

            state->audproc_can_write = audproc_alsa_write_prepare(state);
            if (!state->audproc_can_write)
            {
                pr_message("audproc_alsa_write_prepare is pending\n");
                ETG_TRACE_USR3(("[audproc_alsa_ent_streaming_thread]:audproc_alsa_write_prepare is pending "));
                continue;
            }


            /* write audio frames to output stream devices*/
            if(!audproc_alsa_silence_proc_act_is(state))
            {
                state->err = audproc_alsa_write_all(state, AUD_PROC_SRC_TYPE_ENT);
                if (state->err)
                {
                    state->isWriteErr = TRUE;
                    state->recov_success_write_count = 0;
                    goto rts_err; /*lint Info 801: prio2: Use of goto is deprecated*/
                }
            }

            if(state->isReadWriteErr)
            {
                if(!state->dosilence)
                {
                    (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_ENT
                                                             ,AUDPROC_AS_STREAM_RUNNING
                                                             ,1
                                                             ,state->retry);

                }
                else if(AUDPROC_AS_STREAM_WAIT_RECOVER == audproc_service_get_audio_state(AUD_PROC_SRC_TYPE_ENT))
                {
                    if(++state->recov_success_write_count == 1)
                    {
                        state->silence_wrt_cnt_after_recover = 0;
                    }

                    pr_message("current state is AUDPROC_AS_STREAM_WAIT_RECOVER, recov_success_write_count -> %d , silence_wrt_cnt_after_recover -> %d",state->recov_success_write_count, state->silence_wrt_cnt_after_recover);
                    if(    (state->recov_success_write_count > AUDPROC_MAX_SUCCESS_IO_AFTER_RECOVER)
                        && (state->silence_wrt_cnt_after_recover == 0))
                    {
                         (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_ENT
                                                                 ,AUDPROC_AS_STREAM_SILENCE_PRELOAD
                                                                 ,1
                                                                 ,AUDPROC_MAX_COUNT_SILENCE_AFTER_RECOVER);

                    }
                }
                else //state->dosilence
                {
                    ETG_TRACE_USR3(("[audproc_alsa_ent_streaming_thread]: handle isReadWriteErr, silence are inserted, curr_silence_cnt -> %d", state->silence_wrt_cnt_after_recover));
                    if(state->silence_wrt_cnt_after_recover > AUDPROC_MAX_COUNT_SILENCE_AFTER_RECOVER )
                    {
                        (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_ENT
                                                                 ,AUDPROC_AS_STREAM_RUNNING
                                                                 ,1
                                                                 ,state->retry);
                    }
                    else //!state->silence_wrt_cnt_after_recover <= AUDPROC_MAX_COUNT_SILENCE_AFTER_RECOVER
                    {
                        state->silence_wrt_cnt_after_recover++;
                        pr_message("silence_wrt_cnt_after_recover -> %d \n", state->silence_wrt_cnt_after_recover);
                        ETG_TRACE_USR3(("[audproc_alsa_ent_streaming_thread]: silence_wrt_cnt_after_recover -> %d", state->silence_wrt_cnt_after_recover));
                    }
                }
            }

            /* count audio streaming loop as long as the initialization state is not reported*/
            if(!state->initialized)
            {
                state->audio_started++;
                pr_message("%d successfull read/process/write sequence prior to change in running state\n", state->audio_started);
                ETG_TRACE_USR3(("[audproc_alsa_ent_streaming_thread]: %d successfull read/process/write sequence prior to change in running state ", state->audio_started));
            }

rts_err:
            /* insert retry, with err ENODATA (61) or EPIPE (32)  */
            if( state->err == -ENODATA || state->err == -EPIPE)
            {
                pr_debug("rts streaming failed with err(%d) and current retry count is (%d)\n", state->err, state->retry);

                state->last_err = state->err;

                if(state->isReadErr)
                    state->err = audproc_audio_rts_handler_recover(state, D_CONV_CAPTURE_DEVICE);
                else if(state->isWriteErr)
                    state->err = audproc_audio_rts_handler_recover(state, D_CONV_PLAYBACK_DEVICE);

                if(state->err)
                    pr_warning("rts recover failed with err(%d) and current retry count is (%d)\n", state->err, state->retry);


                /* state no audio streaming */
                state->audio_started = 0;

                if(!state->isReadWriteErr)
                {
                    if(    !state->audio_started
                        && !state->initialized)
                    {
                        (void)audproc_service_change_audio_state(AUD_PROC_SRC_TYPE_ENT, AUDPROC_AS_STREAM_WAIT_DATA, 1, state->last_err );
                    }
                    else
                    {
                        (void)audproc_service_change_audio_state(AUD_PROC_SRC_TYPE_ENT, AUDPROC_AS_STREAM_WAIT_RECOVER, 1, state->last_err );
                        //state->isReadWriteErr = TRUE;
                        pr_debug("wait streaming recovers\n");
                    }
                }
                else
                {
                    if(state->isReadErr)
                    {
                        //ETG_TRACE_USR3(("[audproc_alsa_ent_streaming_thread]: reinitialize silence_wrt_cnt after read error"));
                        state->silence_wrt_cnt_after_recover = 0; // new - 1; /*to be validated */
                    }
                }
                // in all other case including EPIPE, do nothing
                state->err = AUDPROC_ERR_OK;
                state->rts_err = AUDPROC_ERR_OK;
                state->retry++;
                state->isReadErr  = FALSE;
                state->isWriteErr = FALSE;
            }

            else if (state->err == -22)// -EINVAL) //-22
            {
                state->last_err = state->err;

                if(state->isReadErr)
                    (void)audproc_audio_rts_handler_recreate(state, FALSE, D_CONV_CAPTURE_DEVICE);
                else if(state->isWriteErr)
                    (void)audproc_audio_rts_handler_recreate(state, FALSE, D_CONV_PLAYBACK_DEVICE);

                pr_message("last %s operation returned with err -> %d\n", state->isReadErr ? "READ" : "WRITE", state->err);
                ETG_TRACE_USR3(("[audproc_alsa_ent_streaming_thread]: returned with err -> %d last %s operation", state->err, state->isReadErr ? "READ" : "WRITE"));


                state->err = AUDPROC_ERR_OK;
                state->isReadErr  = FALSE;
                state->isWriteErr = FALSE;
            }

            /*audio was started without failure, streaming is started  */
            if(    (state->audio_started > D_MIN_R_W_COUNT_BEFORE_SIGNAL_READ)
                && !state->initialized)
            {
                g_mutex_lock(&state->lock);
                state->initialized = TRUE;

                (void)audproc_service_change_audio_state(AUD_PROC_SRC_TYPE_ENT, AUDPROC_AS_STREAM_RUNNING, 0);

                /* signal send to wake up from call within "audproc_alsa_start()" function,
                   the return code is returned to object interface
                */
                g_cond_broadcast(&state->update);
                g_mutex_unlock(&state->lock);
            }


            if(    !state->audio_started
                && !state->initialized)
            {
                pr_message("wait for receive of the first audio frame : %d\n", state->retry);
                (void)audproc_service_change_audio_state(AUD_PROC_SRC_TYPE_ENT, AUDPROC_AS_STREAM_WAIT_DATA, 0);
            }



            /* request for log statistic, should be done from alsa_thread context,
               otherwise rts lib assert an abort.
               req_statistic flag is released within audproc_alsa_print_current_audio_state()
            */
            if(state->req_statistic)
            {
                audproc_alsa_print_current_audio_state(state);
            }

        }

        /* handle with error */
        if (state->err)
        {
            if (state->apl_err)
                audproc_critical(state->apl_err,"audio terminated by AUDPROC error");
            else
            {
                (void)audproc_service_change_audio_state(AUD_PROC_SRC_TYPE_ENT, AUDPROC_AS_STREAM_ERROR, 0);
                pr_critical("audio terminated by error %d",state->err);
            }
        }

        pr_message("the streaming was stopped");
    }


    /* terminate audio thread */
    (void)audproc_service_change_audio_state(AUD_PROC_SRC_TYPE_ENT, AUDPROC_AS_STREAM_TERMINATED, 0);


    pr_debug("thread was ended with run(%d), exit(%d), initialized(%d) and err(%d)\n",state->run,
                                                                                      state->exit ,
                                                                                      state->initialized,
                                                                                      state->err);
    pr_message("EXIT");

    return NULL;
}

/*******************************************************************************
*
* FUNCTION: audproc_alsa_info_streaming_thread
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gpointer audproc_alsa_info_streaming_thread(gpointer data)
{

    audproc_alsa_state*             state = (audproc_alsa_state*)NULL;


    pr_message("start audproc alsa streaming thread");

    /* if data non Null, than multi-instance interface is used */
    if(data)
    {
        state = (audproc_alsa_state*)data;
        state->aplStr = state->audproc_stream_data->audcfg.audproc_apl_inst_name;

        /* retrieve apl instance name */
        state->aplinst = audproc_apl_handler_get_instance(state->aplStr);

        if(!state->aplinst)
        {
            pr_critical("no APL instance available, processing cannot be continued\n");
            return NULL;
        }
    }
    else
    {
        pr_critical("data pointer is NULL?\n");
        return NULL;
    }

    /* set thread priority */
    audproc_alsa_set_scheduling_priority(state);

    pr_debug("instance state exit(%d), run(%d), initialized(%d), prepared(%d)\n", state->exit,
                                                                                  state->run,
                                                                                  state->initialized,
                                                                                  state->prepared);

    while (!state->exit)
    {
        pr_debug("entry top audio streaming loop");

        g_mutex_lock(&state->lock);
        if(state->err)
        {
            state->run = FALSE;
            (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_INFO
                                                     ,AUDPROC_AS_STREAM_STOP
                                                     ,0);
        }
        else
            (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_INFO
                                                     ,AUDPROC_AS_STREAM_SILENCE
                                                     ,0);

        state->initialized      = FALSE;
        state->prepared         = TRUE;
        state->isReadWriteErr   = FALSE;

        /* alsa stop was terminated,
           the main thread which runs the audproc_alsa_stop function is woken up
        */
        g_cond_broadcast(&state->update);

        /* initialize thread for next run*/
        state->err = 0;
        state->apl_err = 0;
        state->rts_err = 0;
        state->retry = 0;
        //(void)audproc_audio_rts_handler_clear_xrun(state);

        /* wait release from audproc_alsa_start() function */
        while (!state->run && !state->exit)
        {
            pr_message("audio streaming awaiting\n");
            g_cond_wait(&state->update, &state->lock);
        }

        g_mutex_unlock(&state->lock);

        /* audio thread terminated with call audproc_alsa_exit() ? */
        if (state->exit)
            continue;

        /* setup thread priority to real time
        audproc_alsa_set_scheduling_priority(state);
        audproc_alsa_set_audio_thread_state(TRUE, state);
        (void)audproc_audio_rts_handler_voice_reset(state);
        //audproc_audio_rts_handler_performance_test_init(50000);
        */

        (void)audproc_service_change_audio_state(AUD_PROC_SRC_TYPE_INFO
                                                ,AUDPROC_AS_STREAM_INIT
                                                ,0);

        pr_message("audio streaming prepared\n");
        state->audio_started = 0;

        /* run audio streaming */
        while (!state->err && state->run && !state->exit)
        {
            /* read audio frames from input stream devices*/
            if(state->audproc_can_read)
            {
                state->err = audproc_alsa_read_all(state, AUD_PROC_SRC_TYPE_INFO);
                if (state->err)
                {
                    state->isReadErr = TRUE;
                    goto rts_err; /*lint Info 801: prio2: Use of goto is deprecated*/
                }

            }

            /* new */
            state->audproc_can_process = audproc_alsa_process_prepare(state);
            if (!state->audproc_can_process)
                continue;


            if(state->isReadWriteErr)
            {
                (void)audproc_service_change_audio_state(AUD_PROC_SRC_TYPE_INFO
                                                        ,AUDPROC_AS_STREAM_RUNNING
                                                        ,1
                                                        ,state->retry);
                state->isReadWriteErr = FALSE;
                state->retry = 0;
            }

            /* process audio frames */
            state->err = state->apl_err = audproc_apl_handler_process_audio_info(state->aplinst, &state->PreProcess, &state->PostProcess);

            if (state->err)
                continue;

            /* new */
            state->audproc_can_write = audproc_alsa_write_prepare(state);
            if (!state->audproc_can_write)
                continue;


            /* write audio frames to output stream devices*/
            state->err = audproc_alsa_write_all(state, AUD_PROC_SRC_TYPE_INFO);
            if (state->err)
            {
                state->isWriteErr = TRUE;
                goto rts_err; /*lint Info 801: prio2: Use of goto is deprecated*/
            }


            if(state->isReadWriteErr)
            {
               (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_INFO
                                                        ,AUDPROC_AS_STREAM_RUNNING
                                                        ,1
                                                        ,state->retry);
                state->isReadWriteErr = FALSE;
                state->retry = 0;
            }

            /* count audio streaming loop as long as the initialization state is not reported*/
            if(!state->initialized)
                state->audio_started++;


rts_err:
            /* insert retry, with err ENODATA (61) or EPIPE (32)  */
            if( state->err == -ENODATA || state->err == -EPIPE)
            {
                pr_debug("rts streaming failed with err(%d) and current retry count is (%d)\n", state->err, state->retry);


                if(state->isReadErr)
                    state->err = audproc_audio_rts_handler_recover(state, D_CONV_CAPTURE_DEVICE);
                else if(state->isWriteErr)
                    state->err = audproc_audio_rts_handler_recover(state, D_CONV_PLAYBACK_DEVICE);

                if(state->err)
                    pr_warning("rts recover failed with err(%d) and current retry count is (%d)\n", state->err, state->retry);

                // in all other case including EPIPE, do nothing
                state->isReadErr  = FALSE;
                state->isWriteErr = FALSE;
                state->err = AUDPROC_ERR_OK;
                state->rts_err = AUDPROC_ERR_OK;
                state->retry++;

                /* state no audio streaming */
                state->audio_started = 0;

                if(!state->isReadWriteErr)
                {
                    if(    !state->audio_started
                        && !state->initialized)
                    {
                        (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_INFO
                                                                 ,AUDPROC_AS_STREAM_WAIT_DATA
                                                                 ,0);
                    }
                    else
                    {
                        (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_INFO
                                                                 ,AUDPROC_AS_STREAM_WAIT_RECOVER
                                                                 ,0);
                        state->isReadWriteErr = TRUE;
                        pr_message("wait streaming recovers\n");
                    }
                    //state->isReadWriteErr = TRUE;
                }
            }

            /*audio was started without failure, streaming is started  */
            if(    (state->audio_started > D_MIN_R_W_COUNT_BEFORE_SIGNAL_READ)
                && !state->initialized)
            {
                g_mutex_lock(&state->lock);
                state->initialized = TRUE;

                (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_INFO
                                                         ,AUDPROC_AS_STREAM_RUNNING
                                                         ,0);

                /* signal send to wake up from call within "audproc_alsa_start()" function,
                   the return code is returned to object interface
                */
                g_cond_broadcast(&state->update);
                g_mutex_unlock(&state->lock);
            }

            if(    !state->audio_started
                && !state->initialized)
            {
                pr_message("wait for receive of the first audio frame : %d\n", state->retry);
                (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_INFO
                                                         ,AUDPROC_AS_STREAM_WAIT_DATA
                                                         ,0);
            }

            /* request for log statistic, should be done from alsa_thread context,
               otherwise rts lib assert an abort.
               req_statistic flag is released within audproc_alsa_print_current_audio_state()
            */
            if(state->req_statistic)
            {
                audproc_alsa_print_current_audio_state(state);
            }

        }

        /* handle with error */
        if (state->err)
        {
            if (state->apl_err)
                audproc_critical(state->apl_err,"audio terminated by AUDPROC error");
            else
            {
                (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_INFO
                                                         ,AUDPROC_AS_STREAM_ERROR
                                                         ,0);
                pr_critical("audio terminated by error %d",state->err);
            }
        }

        if(state->initialized)
        {
            if(!state->audio_started)
            {
                pr_message("close after successfull streaming, variable audio_started is forced to 1\n");
                state->audio_started = 1;
            }
            pr_message("close after successfull streaming with audio_started(%d)\n", state->audio_started );
        }

        pr_message("audio streaming stopped");
    }


    /* terminate audio thread */
    (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_INFO
                                             ,AUDPROC_AS_STREAM_TERMINATED
                                             ,0);


    pr_debug("thread was ended with run(%d), exit(%d), initialized(%d) and err(%d)\n",state->run,
                                                                                      state->exit ,
                                                                                      state->initialized,
                                                                                      state->err);
    pr_message("exit audproc alsa streaming thread");

    return NULL;
}
/*******************************************************************************
*
* FUNCTION: audproc_alsa_voice_streaming_thread
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gpointer audproc_alsa_voice_streaming_thread(gpointer data)
{
    audproc_alsa_state*             state = (audproc_alsa_state*)NULL;


    pr_message("start audproc alsa streaming thread");


    /* if data non Null, than multi-instance interface is used */
    if(data)
    {
        state = (audproc_alsa_state*)data;
        const gchar* aplStr = state->audproc_stream_data->audcfg.audproc_apl_inst_name;

        /* retrieve apl instance name */
        state->aplinst = audproc_apl_handler_get_instance(aplStr);

        if(!state->aplinst)
        {
            pr_critical("no APL instance available, processing cannot be continued\n");
            return NULL;
        }
    }
    else
    {
        pr_critical("data pointer is NULL?\n");
        return NULL;
    }


    audproc_alsa_set_scheduling_priority(state); //--> this the problem root cause
    pr_debug("instance state exit(%d), run(%d), initialized(%d), prepared(%d)\n", state->exit, state->run, state->initialized, state->prepared);

    while (!state->exit)
    {
        pr_debug("entry top audio streaming loop");

        g_mutex_lock(&state->lock);
        if(state->err)
        {
            state->run = FALSE;
            (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_VOICE
                                                     ,AUDPROC_AS_STREAM_STOP
                                                     ,0);
        }
        else
            (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_VOICE
                                                     ,AUDPROC_AS_STREAM_SILENCE
                                                     ,0);

        state->initialized      = FALSE;
        state->prepared         = TRUE;
        state->isReadWriteErr   = FALSE;

        /* alsa stop was terminated,
           the main thread which runs the audproc_alsa_stop function is woken up
        */
        g_cond_broadcast(&state->update);

        /* initialize thread for next run*/
        state->err = 0;
        state->apl_err = 0;
        state->rts_err = 0;
        state->retry = 0;

        /* wait release from audproc_alsa_start() function */
        while (!state->run && !state->exit)
        {
            pr_message("audio streaming awaiting\n");
            g_cond_wait(&state->update, &state->lock);
        }

        g_mutex_unlock(&state->lock);

        /* audio thread terminated with call audproc_alsa_exit() ? */
        if (state->exit)
            continue;

        /*************************
         * start time out
         */

        (void)audproc_service_change_audio_state(AUD_PROC_SRC_TYPE_VOICE
                                                ,AUDPROC_AS_STREAM_INIT
                                                ,0);


        pr_message("audio streaming prepared\n");
        state->audio_started = 0;

        /* run audio streaming */
        while (!state->err && state->run && !state->exit)
        {
            /* read audio frames from input stream devices*/
            state->err = audproc_alsa_read_all(state, AUD_PROC_SRC_TYPE_VOICE);
            if (state->err)
            {
                state->isReadErr = TRUE;
                goto rts_err; /*lint Info 801: prio2: Use of goto is deprecated*/
            }

            /* process audio stream data only */
            state->err = state->apl_err = audproc_apl_handler_process_audio_voice(state->aplinst, &state->InAudio, &state->OutAudio); //state);

            if(state->apl_err == APL_ERR_NO_RESULT)
                state->err = 0;

            if(state->err)
                continue;

            /* write audio frames to output stream devices*/
            if(state->apl_err != APL_ERR_NO_RESULT)
                    state->err = audproc_alsa_write_all(state, AUD_PROC_SRC_TYPE_VOICE);

            if (state->err)
            {
                state->isWriteErr = TRUE;
                state->recov_success_write_count = 0;
                goto rts_err; /*lint Info 801: prio2: Use of goto is deprecated*/
            }

            if(state->isReadWriteErr)
            {
                ETG_TRACE_USR3(("[audproc_alsa_voice_streaming_thread]: state->isReadWriteErr is TRUE"));
                if(!state->dosilence)
                {
                    (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_VOICE
                                                             ,AUDPROC_AS_STREAM_RUNNING
                                                             ,1
                                                             ,state->retry);

                }

                else if(AUDPROC_AS_STREAM_WAIT_RECOVER == audproc_service_get_audio_state(AUD_PROC_SRC_TYPE_VOICE))
                {
                    ETG_TRACE_USR3(("[audproc_alsa_voice_streaming_thread]: current state is AUDPROC_AS_STREAM_WAIT_RECOVER, recov_success_write_count -> %d , silence_wrt_cnt_after_recover -> %d",state->recov_success_write_count, state->silence_wrt_cnt_after_recover));
                    state->recov_success_write_count++;
                    if(    (state->recov_success_write_count > AUDPROC_MAX_SUCCESS_IO_AFTER_RECOVER)
                        && (state->silence_wrt_cnt_after_recover == 0))
                    {
                         (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_VOICE
                                                                  ,AUDPROC_AS_STREAM_SILENCE_PRELOAD
                                                                  ,1
                                                                  ,AUDPROC_MAX_COUNT_SILENCE_AFTER_RECOVER);

                    }
                }
                else //state->dosilence
                {
                    ETG_TRACE_USR3(("[audproc_alsa_voice_streaming_thread]: handle isReadWriteErr, silence are inserted, curr_silence_cnt -> %d", state->silence_wrt_cnt_after_recover));
                    if(state->silence_wrt_cnt_after_recover > AUDPROC_MAX_COUNT_SILENCE_AFTER_RECOVER )
                    {
                        (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_VOICE
                                                                 ,AUDPROC_AS_STREAM_RUNNING
                                                                 ,1
                                                                 ,state->retry);
                    }
                    else //!state->silence_wrt_cnt_after_recover <= AUDPROC_MAX_COUNT_SILENCE_AFTER_RECOVER
                    {
                        state->silence_wrt_cnt_after_recover++;
                        pr_debug("silence_wrt_cnt_after_recover -> %d \n", state->silence_wrt_cnt_after_recover);
                        ETG_TRACE_USR3(("[audproc_alsa_voice_streaming_thread]: silence_wrt_cnt_after_recover -> %d", state->silence_wrt_cnt_after_recover));
                    }
                }
            }

            /* count audio streaming loop as long as the initialization state is not reported*/
            if(!state->initialized)
                state->audio_started++;

rts_err:
            /* insert retry, with err ENODATA (61) or EPIPE (32)  */
            if( state->err == -ENODATA || state->err == -EPIPE  )
            {
                pr_debug("rts streaming failed with err(%d) and current retry count is (%d)\n", state->err, state->retry);

                if(state->isReadErr == TRUE)
                    state->err = audproc_audio_rts_handler_recover(state, D_CONV_CAPTURE_DEVICE);

                else if(state->isWriteErr == TRUE)
                //if(state->isWriteErr == TRUE)
                    state->err = audproc_audio_rts_handler_recover(state, D_CONV_PLAYBACK_DEVICE);

                if(state->err)
                    pr_warning("rts creation failed with err(%d) and current retry count is (%d)\n", state->err, state->retry);

                // in all other case including EPIPE, do nothing
                #ifdef D_PREPROCESS_SILENCE_ENT_SRC_ONLY
                state->isReadErr  = FALSE;
                state->isWriteErr = FALSE;
                #endif
                state->err = AUDPROC_ERR_OK;
                state->rts_err = AUDPROC_ERR_OK;
                state->retry++;

                /* state no audio streaming */
                state->audio_started = 0;

                if(!state->isReadWriteErr)
                {
                    if(    !state->audio_started
                        && !state->initialized)
                    {
                        (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_VOICE
                                                                 ,AUDPROC_AS_STREAM_WAIT_DATA
                                                                 ,0);
                    }
                    else
                    {
                        (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_VOICE
                                                                 ,AUDPROC_AS_STREAM_WAIT_RECOVER
                                                                 ,0);


                        //state->isReadWriteErr = TRUE;
                        pr_message("wait streaming recovers\n");
                    }
                }
                else
                {
                    if(state->isReadErr)
                    {
                        //ETG_TRACE_USR3(("[audproc_alsa_voice_streaming_thread]: reinitialize silence_wrt_cnt after read error"));
                        state->silence_wrt_cnt_after_recover = 1; /*to be validated */
                    }
                }
                #ifndef D_PREPROCESS_SILENCE_ENT_SRC_ONLY
                state->isReadErr  = FALSE;
                state->isWriteErr = FALSE;
                #endif
            }

            /*audio was started without failure, streaming is started  */
            if(    (state->audio_started > D_MIN_R_W_COUNT_BEFORE_SIGNAL_READ)
                && !state->initialized)
            {
                g_mutex_lock(&state->lock);
                state->initialized = TRUE;

                (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_VOICE
                                                         ,AUDPROC_AS_STREAM_RUNNING
                                                         ,0);


                /* signal send to wake up from call within "audproc_alsa_start()" function,
                   the return code is returned to object interface
                */
                g_cond_broadcast(&state->update);
                g_mutex_unlock(&state->lock);
                pr_message("audio streaming running"); //pr_message("Starting audio");
            }

            if(    !state->audio_started
                && !state->initialized)
            {
                pr_message("wait for receive of the first audio frame : %d\n", state->retry);
                (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_VOICE
                                                         ,AUDPROC_AS_STREAM_WAIT_DATA
                                                         ,0);
            }
            /* request for log statistic, should be done from alsa_thread context,
               otherwise rts lib assert an abort.
               req_statistic flag is released within audproc_alsa_print_current_audio_state()
            */
            if(state->req_statistic)
            {
                audproc_alsa_print_current_audio_state(state);
            }
        }

        /* handle with error */
        if (state->err)
        {
            if (state->apl_err)
                audproc_critical(state->apl_err,"audio terminated by AUDPROC error");
            else
            {
                pr_critical("audio terminated by error %d",state->err);
            }
        }

        if(state->initialized)
        {
            if(!state->audio_started)
            {
                pr_message("close after successfull streaming, variable audio_started is forced to 1\n");
                state->audio_started = 1;
            }
            pr_message("close after successfull streaming with audio_started(%d)\n", state->audio_started );
        }

        pr_message("audio streaming stopped");
    }

    /* terminate audio thread */
    (void)audproc_service_change_audio_state( AUD_PROC_SRC_TYPE_VOICE
                                             ,AUDPROC_AS_STREAM_TERMINATED
                                             ,0);


    pr_debug("thread was ended with run(%d), exit(%d), initialized(%d) and err(%d)\n",state->run, state->exit , state->initialized, state->err);
    pr_message("exit audproc alsa streaming thread");

    return NULL;
}

/*******************************************************************************
*
* FUNCTION: audproc_alsa_create_alsa_thread_start
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int audproc_alsa_create_ent_streaming_thread(audproc_alsa_state* state)
{
    int err = AUDPROC_ERR_OK;
    GError *error;

    pr_message("ENTERED");

    if(!state)
        return AUDPROC_ERR_NULL_POINTER;


    pr_debug("current value state->exit(%d)\n", state->exit);

    /* create a new thread only if not still exist */
    if (!state->thread)
    {
        state->thread = g_thread_try_new ("audproc_alsa_ent_streaming_thread", audproc_alsa_ent_streaming_thread, (gpointer)state, &error);
        if (!state->thread)
        {
            pr_warning("fails to create audproc_alsa_ent_streaming_thread with error code(%d) and message (%s)\n",error->code, error->message );
            err = AUDPROC_ERR_FAILED;
        }
    }
    else
    {
        pr_warning("thread already created\n");
    }

    pr_message("EXIT");
    return err;
 }


/*******************************************************************************
*
* FUNCTION: audproc_alsa_create_voice_streaming_thread
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int audproc_alsa_create_voice_streaming_thread(audproc_alsa_state* state)
{
    int err = AUDPROC_ERR_OK;
    GError *error;

    pr_message("ENTERED");

    if(!state)
        return AUDPROC_ERR_NULL_POINTER;

    pr_debug("current value state->exit(%d)\n", state->exit);

    /* create a new thread only if not still exist */
    if (!state->thread)
    {
        state->thread = g_thread_try_new ("audproc_alsa_voice_streaming_thread", audproc_alsa_voice_streaming_thread, (gpointer)state, &error);
        if (!state->thread)
        {
            pr_warning("fails to create audproc_alsa_voice_streaming_thread with error code(%d) and message (%s)\n",error->code, error->message );
            err = AUDPROC_ERR_FAILED;
        }
    }
    else
    {
        pr_warning("thread already created\n");
    }

    pr_message("EXIT");
    return err;
 }

/*******************************************************************************
*
* FUNCTION: audproc_alsa_create_info_streaming_thread
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int audproc_alsa_create_info_streaming_thread(audproc_alsa_state* state)
{
    int err = AUDPROC_ERR_OK;
    GError *error;

    pr_message("ENTERED");

    if(!state)
        return AUDPROC_ERR_NULL_POINTER;

    pr_debug("current value state->exit(%d)\n", state->exit);

    /* create a new thread only if not still exist */
    if (!state->thread)
    {
        state->thread = g_thread_try_new ("audproc_alsa_info_streaming_thread", audproc_alsa_info_streaming_thread, (gpointer)state, &error);
        if (!state->thread)
        {
            pr_warning("fails to create audproc_alsa_info_streaming_thread with error code(%d) and message (%s)\n",error->code, error->message );
            err = AUDPROC_ERR_FAILED;
        }
    }
    else
    {
        pr_warning("thread already created\n");
    }

    pr_message("EXIT");
    return err;
 }
